// 
// x_maplist.c -- server maplist rotation/manipulation routines 
// 
// 
// 1/98 - L. Allan Campbell (Geist) 
// 
// INCLUDES ///////////////////////////////////////////////// 
#include "g_local.h" 
// FUNCTIONS //////////////////////////////////////////////// 
// 
// LoadMapList 
// 
// Opens the specified file and scans/loads the maplist names 
// from the file's [maplist] section. (list is terminated with 
// "###") 
// 
// Args: 
// ent - entity (client) to print diagnostic messages to. 
// filename - name of file containing maplist. 
// 
// Return: 0 = normal exit, maplist loaded 
// 1 = abnormal exit 
// 
int LoadMapList(edict_t *ent, char *filename) 
{ 
FILE *fp; 
int i=0; 
char szLineIn[80]; 
fp = OpenFile(ent, filename); 
if (fp) // opened successfully? 
{ 
// scan for [maplist] section 
do 
{ 
fscanf(fp, "%s", szLineIn); 
} while (!feof(fp) && (Q_stricmp(szLineIn, "[maplist]") != 0)); 
if (feof(fp)) 
{ 
// no [maplist] section 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
gi.cprintf (ent, PRINT_HIGH, "ERROR - No [maplist] section in \"%s\".\n", filename); 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
} 
else 
{ 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 

// read map names into array 
while ((!feof(fp)) && (i<MAX_MAPS)) 
{ 
fscanf(fp, "%s", szLineIn); 
if (Q_stricmp(szLineIn, "###") == 0) // terminator is "###" 
break; 
// TODO: check that maps exist before adding to list 
// (might be difficult to search a .pak file for these) 
strncpy(maplist.mapnames[i], szLineIn, MAX_MAPNAME_LEN); 
gi.cprintf(ent, PRINT_HIGH, "...%s\n", maplist.mapnames[i]); 
i++; 
} 
} 
CloseFile(ent, fp); 
if (i == 0) 
{ 
gi.cprintf (ent, PRINT_HIGH, "No maps listed in [maplist] section of %s\n", filename); 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
return 0; // abnormal exit -- no maps in file 
} 
gi.cprintf (ent, PRINT_HIGH, "%i map(s) loaded.\n", i); 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
maplist.nummaps = i; 
return 1; // normal exit 
} 

return 0; // abnormal exit -- couldn't open file 
} 
// 
// ClearMapList 
// 
// Clears/invalidates maplist. Might add more features in the future, 
// but resetting .nummaps to 0 will suffice for now. 
// 
// Args: 
// ent - entity (client) to print diagnostic messages to (future development). 
// 
// Return: (none) 
// 
void ClearMapList(edict_t *ent) 
{ 
maplist.nummaps = 0; 
} 
// 
// DisplayMaplistUsage 
// 
// Displays current command options for maplists. 
// 
// Args: 
// ent - entity (client) to display help screen (usage) to. 
// 
// Return: (none) 
// 
void DisplayMaplistUsage(edict_t *ent) 
{ 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
gi.cprintf (ent, PRINT_HIGH, "Maplist usage:\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST <filename> [<rot_flag>]\n"); 
gi.cprintf (ent, PRINT_HIGH, " <filename> - server ini file\n"); 
gi.cprintf (ent, PRINT_HIGH, " <rot_flag> - 0 = sequential (def)\n"); 
gi.cprintf (ent, PRINT_HIGH, " 1 = random\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST START to go to 1st map\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST NEXT to go to next map\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST to display current list\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST OFF to clear/disable\n"); 
gi.cprintf (ent, PRINT_HIGH, "/MAPLIST HELP for this screen\n"); 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
} 
// 
// Cmd_Maplist_f 
// 
// Main command line parsing function. Enables/parses/diables maplists. 
// 
// Args: 
// ent - entity (client) to display messages to, if necessary. 
// 
// Return: (none) 
// 
// TODO: change "client 0" privs to be for server only, if dedicated. 
// only allow other clients to view list and see HELP screen. 
// (waiting for point release for this feature) 
// 
void Cmd_Maplist_f (edict_t *ent) 
{ 
int i; // looping and temp variable 
char *filename; 
switch (gi.argc()) 
{ 
case 2: // various commands, or enable and assume rotationflag default 
if (Q_stricmp(gi.argv(1), "HELP") == 0) 
{ 
DisplayMaplistUsage(ent); 
break; 
} 
// only allow if client 0 
if (ent != &g_edicts[1]) 
{ 
gi.cprintf (ent, PRINT_HIGH, "MAPLIST options locked by server.\n"); 
break; 
} 
if (Q_stricmp(gi.argv(1), "START") == 0) 
{ 
if (maplist.nummaps > 0) // does a maplist exist? 
EndDMLevel(); 
else 
DisplayMaplistUsage(ent); 
break; 
} 
else if (Q_stricmp(gi.argv(1), "NEXT") == 0) 
{ 
if (maplist.nummaps > 0) // does a maplist exist? 
EndDMLevel(); 
else 
DisplayMaplistUsage(ent); 
break; 
} 
else if (Q_stricmp(gi.argv(1), "OFF") == 0) 
{ 
if (maplist.nummaps > 0) // does a maplist exist? 
{ 
ClearMapList(ent); 
fwflags->value = (int) fwflags->value & ~FW_MAP_LIST; 
//dmflags->value = (int) dmflags->value & ~DF_MAP_LIST; 
gi.cprintf (ent, PRINT_HIGH, "Maplist cleared/disabled.\n"); 
} 
else 
{ 
// maplist doesn't exist, so display usage 
DisplayMaplistUsage(ent); 
} 
break; 
} 
else 
maplist.rotationflag = 0; 

// no break here is intentional; supposed to fall though to case 3 
case 3: // enable maplist - all args explicitly stated on command line 
// only allow if client 0 
if (ent != &g_edicts[1]) 
{ 
gi.cprintf (ent, PRINT_HIGH, "MAPLIST options locked by server.\n"); 
break; 
} 
if (gi.argc() == 3) // this is required, because it can still = 2 
{ 
i = atoi(gi.argv(2)); 
if (Q_stricmp(gi.argv(1), "GOTO") == 0) 
{ 
// user trying to goto specified map # in list 
if ((i<1) || (i>maplist.nummaps)) 
DisplayMaplistUsage(ent); 
else 
{ 
ent = G_Spawn (); 
ent->classname = "target_changelevel"; 
ent->map = maplist.mapnames[i-1]; 
maplist.currentmap = i-1; 
BeginIntermission(ent); 
} 
break; 
} 
else 
{ 
// user trying to specify new maplist 
if ((i<0) || (i>=ML_ROTATE_NUM_CHOICES)) // check for valid rotationflag 
{ 
// outside acceptable values for rotationflag 
DisplayMaplistUsage(ent); 
break; 
} 
else 
{ 
maplist.rotationflag = atoi(gi.argv(2)); 
} 
} 
} 
filename = gi.argv(1); // get filename from command line 
if ((int) fwflags->value & FW_MAP_LIST)
//if ((int) dmflags->value & DF_MAP_LIST)
{ 
// tell user to cancel current maplist before starting new maplist 
gi.cprintf (ent, PRINT_HIGH, "You must disable current maplist first. (/CMD MAPLIST OFF)\n"); 
} 
else 
{ 
// load new maplist 
if (LoadMapList(ent, filename)) // return 1 = success 
{ 
fwflags->value = (int) fwflags->value | FW_MAP_LIST; 
//dmflags->value = (int) dmflags->value | DF_MAP_LIST;
gi.cprintf (ent, PRINT_HIGH, "Maplist created/enabled. You can now use START or NEXT.\n"); 
maplist.currentmap = -1; 
} 
} 
break; 
case 1: // display current maplist 
if (maplist.nummaps > 0) // does a maplist exist? 
{ 
gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
for (i=0; i<maplist.nummaps; i++) 
{ 
gi.cprintf (ent, PRINT_HIGH, "#%2d \"%s\"\n", i+1, maplist.mapnames[i]); 
} 
gi.cprintf (ent, PRINT_HIGH, "%i map(s) in list.\n", i); 
gi.cprintf (ent, PRINT_HIGH, "Rotation flag = %i ", maplist.rotationflag); 
switch (maplist.rotationflag) 
{ 
case ML_ROTATE_SEQ: 
gi.cprintf (ent, PRINT_HIGH, "\"sequential\"\n"); 
break; 
case ML_ROTATE_RANDOM: 
gi.cprintf (ent, PRINT_HIGH, "\"random\"\n"); 
break; 

default: 
gi.cprintf (ent, PRINT_HIGH, "(ERROR)\n"); 
} // end switch 
if (maplist.currentmap == -1) 
{ 
gi.cprintf (ent, PRINT_HIGH, "Current map = #-1 (not started)\n"); 
} 
else 
{ 
gi.cprintf (ent, PRINT_HIGH, "Current map = #%i \"%s\"\n", 
maplist.currentmap+1, maplist.mapnames[maplist.currentmap]); 
} 

gi.cprintf (ent, PRINT_HIGH, "-------------------------------------\n"); 
break; 
} 
// this is when the command is "/cmd maplist", but no maplist exists to display 
DisplayMaplistUsage(ent); 
break; 
default: 
DisplayMaplistUsage(ent); 
} // end switch 
}
