// s_flag.c

#include "g_local.h"
#include "s_flag.h"
#include "s_team.h"

int Flag_WhichIcon(int team)
{
	int i;
	char state;
	char icon[8];
	edict_t *e;
	gitem_t	*flag;

	// figure out what icon to display for team logos
	// three states:
	//   flag at base
	//   flag taken
	//   flag dropped

	state = 0;

	if (team == CTF_TEAM1)
	{
		e = G_Find(NULL, FOFS(classname), "item_flag_team1");
		flag = item_flag1;
	}
	else
	{
		e = G_Find(NULL, FOFS(classname), "item_flag_team2");
		flag = item_flag2;
	}

	if (e != NULL) {
		if (e->solid == SOLID_NOT) {
			state = 'd';

			// not at base
			// check if on player
			for (i = 1; i <= game.maxclients; i++)
				if (g_edicts[i].inuse &&
					g_edicts[i].client->pers.inventory[ITEM_INDEX(flag)]) {
					// enemy has it
					state = 't';
					break;
				}
		} else if (e->spawnflags & DROPPED_ITEM)
			state = 'd';
	}

	sprintf(icon, "i_ctf%i%c", team, state);

	return gi.imageindex (icon);
};

void Flag_InitFlags()
{
	if (!item_flag1)
		item_flag1 = FindItemByClassname("item_flag_team1");
	if (!item_flag2)
		item_flag2 = FindItemByClassname("item_flag_team2");
};

void Flag_Drop (edict_t *flag)
{
	gi.unlinkentity(flag);
	VectorCopy (flag->owner->s.origin, flag->s.origin);
	flag->s.modelindex = flag->owner->s.modelindex3;
	flag->style = FLAG_DROPPED;
	flag->owner = NULL;
	flag->movetype = MOVETYPE_TOSS;
	flag->solid = SOLID_TRIGGER;

	flag->timestamp = level.time + TEAM_FLAG_RETURN_TIME;
	VectorSet(flag->velocity, 0, 0, 300);
	gi.linkentity(flag);
};

void Flag_Think (edict_t *flag)
{
	flag->s.frame = FRAME_idle1 + ((flag->s.frame - FRAME_idle1) + 1) % 7;
	flag->nextthink = level.time + FRAMETIME;
};

void Flag_Reset(int ctf_team)
{
	char *c;
	edict_t *ent;

	switch (ctf_team) {
	case CTF_TEAM1:
		c = "item_flag_team1";
		break;
	case CTF_TEAM2:
		c = "item_flag_team2";
		break;
	default:
		return;
	}

	ent = NULL;
	while ((ent = G_Find (ent, FOFS(classname), c)) != NULL) {
		if (ent->spawnflags & DROPPED_ITEM)
			G_FreeEdict(ent);
		else {
			ent->svflags &= ~SVF_NOCLIENT;
			ent->solid = SOLID_TRIGGER;
			gi.linkentity(ent);
			ent->s.event = EV_ITEM_RESPAWN;
		}
	}
}

void Flag_Capture(edict_t *player, gitem_t *flag, int flag_team)
{
	int i;
	edict_t *cl_ent;
	int		amt;

	player->client->pers.inventory[ITEM_INDEX(flag)] = 0;

	//ctfgame.last_flag_capture = level.time;
	//ctfgame.last_capture_team = ctf_team;

	gi.sound (player, CHAN_RELIABLE+CHAN_NO_PHS_ADD+CHAN_VOICE, gi.soundindex("flag/capture.wav"), 1, ATTN_NONE, 0);
	
	player->s.modelindex3 = 0;
	
	// Broadcast notifications
	for (i=0 ; i<game.maxclients ; i++)
	{
		cl_ent = g_edicts + 1 + i;
		if (!cl_ent->inuse)
			continue;
		if (cl_ent->client->resp.ctf_team == player->client->resp.ctf_team)
			gi.cprintf(cl_ent, PRINT_HIGH, "%s captured the enemy flag!\n", (cl_ent == player ? "You" : "Your team"));
		else
			gi.cprintf(cl_ent, PRINT_HIGH, "Your flag was captured!\n");
	};

	// Award bonuses to players
	for (i=0 ; i<game.maxclients ; i++)
	{
		cl_ent = g_edicts + 1 + i;
		if (!cl_ent->inuse)
			continue;
		if (cl_ent->client->resp.ctf_team != player->client->resp.ctf_team)
			continue;
		if (cl_ent != player)
			amt = TEAM_TEAM_BONUS;
		else
			amt = TEAM_CAPTURE_BONUS;
		cl_ent->client->resp.score += amt;
		// Let's see if this player recently returned our flag
		if (cl_ent->client->resp.ctf_lastreturnedflag + TEAM_RETURN_FLAG_ASSIST_TIMEOUT > level.time)
		{
			gi.cprintf(cl_ent, PRINT_HIGH, "You get a bonus for returning your flag!\n");
			cl_ent->client->resp.score += TEAM_RETURN_FLAG_ASSIST_BONUS;
		};
	};

	Flag_Reset(flag_team);
};

static void Flag_DroppedTouch(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	//owner (who dropped us) can't touch for two secs
	if (other == ent->owner && 
		ent->nextthink - level.time > CTF_AUTO_FLAG_RETURN_TIMEOUT-2)
		return;

	Touch_Item (ent, other, plane, surf);
}

static void Flag_DroppedThink(edict_t *ent)
{
	ent->s.frame = FRAME_idle1 + ((ent->s.frame - FRAME_idle1) + 1) % 7;
	ent->nextthink = level.time + FRAMETIME;

	if (level.time < ent->timestamp)
		return;

	// auto return the flag
	// reset flag will remove ourselves
	if (strcmp(ent->classname, "item_flag_team1") == 0) {
		Flag_Reset(CTF_TEAM1);
		gi.bprintf(PRINT_HIGH, "The %s flag has returned!\n",
			Team_TeamName(CTF_TEAM1));
	} else if (strcmp(ent->classname, "item_flag_team2") == 0) {
		Flag_Reset(CTF_TEAM2);
		gi.bprintf(PRINT_HIGH, "The %s flag has returned!\n",
			Team_TeamName(CTF_TEAM2));
	}
}

void Flag_DeadDrop(edict_t *self)
{
	edict_t *dropped = NULL;

	if (!item_flag1 || !item_flag2)
		Flag_InitFlags();

	if (self->client->pers.inventory[ITEM_INDEX(item_flag1)]) {
		dropped = Drop_Item(self, item_flag1);
		self->client->pers.inventory[ITEM_INDEX(item_flag1)] = 0;
		gi.bprintf(PRINT_HIGH, "%s lost the %s flag!\n",
			self->client->pers.netname, Team_TeamName(CTF_TEAM1));
	} else if (self->client->pers.inventory[ITEM_INDEX(item_flag2)]) {
		dropped = Drop_Item(self, item_flag2);
		self->client->pers.inventory[ITEM_INDEX(item_flag2)] = 0;
		gi.bprintf(PRINT_HIGH, "%s lost the %s flag!\n",
			self->client->pers.netname, Team_TeamName(CTF_TEAM2));
	}

	if (dropped) {
		dropped->timestamp = level.time + CTF_AUTO_FLAG_RETURN_TIMEOUT;
		dropped->think = Flag_DroppedThink;
		dropped->nextthink = level.time + FRAMETIME;
		dropped->touch = Flag_DroppedTouch;
	}
}

/*
Function: Flag_PickupFlag
Purpose:  This function determines whether or not
          the entity is allowed to pick up the
		  flag
*/
qboolean Flag_PickupFlag(edict_t *self, edict_t *other)
{
	int i;
	int player_team, flag_team, enemy_team;
	gitem_t *flag_item, *enemy_flag_item;
	edict_t *cl_ent;

	player_team = other->client->resp.ctf_team;
	if (strcmp(self->classname, "item_flag_team1") == 0)
		flag_team = CTF_TEAM1;
	else
		flag_team = CTF_TEAM2;

	// same team, if the flag at base, check to he has the enemy flag
	if (player_team == CTF_TEAM1) {
		flag_item = item_flag1;
		enemy_flag_item = item_flag2;
		enemy_team = CTF_TEAM2;
	} else {
		flag_item = item_flag2;
		enemy_flag_item = item_flag1;
		enemy_team = CTF_TEAM1;
	}

	if (player_team == flag_team)
	{
		// This path is followed if we just touched our own flag
		if (!(self->spawnflags & DROPPED_ITEM))
			// the flag is at home base.  if the player has the enemy
			// flag, he's just won!
		{
			if (other->client->pers.inventory[ITEM_INDEX(enemy_flag_item)])
			{
				Flag_Capture(other, enemy_flag_item, enemy_team);
			}
			return false;
		}	
		// hey, its not home.  return it by teleporting it back
		gi.bprintf(PRINT_HIGH, "%s returned the %s flag!\n", 
			other->client->pers.netname, Team_TeamName(other->client->resp.ctf_team));
		other->client->resp.score += TEAM_RECOVERY_BONUS;
		other->client->resp.ctf_lastreturnedflag = level.time;
		// TODO: Replace this sound
		//gi.sound (ent, CHAN_RELIABLE+CHAN_NO_PHS_ADD+CHAN_VOICE, gi.soundindex("ctf/flagret.wav"), 1, ATTN_NONE, 0);
		//CTFResetFlag will remove this entity!  We must return false
		Flag_Reset(other->client->resp.ctf_team);
		return false;
	}

	other->client->pers.inventory[ITEM_INDEX(enemy_flag_item)] = 1;
	other->client->resp.ctf_flagsince = level.time;
	gi.sound (other, CHAN_RELIABLE+CHAN_NO_PHS_ADD+CHAN_VOICE, gi.soundindex("flag/taken.wav"), 1, ATTN_NONE, 0);

	// pick up the flag
	// if it's not a dropped flag, we just make it disappear
	// if it's dropped, it will be removed by the pickup caller
	if (!(self->spawnflags & DROPPED_ITEM)) {
		self->flags |= FL_RESPAWN;
		self->svflags |= SVF_NOCLIENT;
		self->solid = SOLID_NOT;
	}

	// Broadcast notifications
	for (i=0 ; i<game.maxclients ; i++)
	{
		cl_ent = g_edicts + 1 + i;
		if (!cl_ent->inuse)
			continue;
		if (cl_ent->client->resp.ctf_team == other->client->resp.ctf_team)
			gi.cprintf(cl_ent, PRINT_HIGH, "%s got the enemy flag!\n", (other == cl_ent ? "You" : other->client->pers.netname));
		else
			gi.cprintf(cl_ent, PRINT_HIGH, "%s has taken your flag!\n", other->client->pers.netname);
	};

	// Award bonus to player
	other->client->resp.score += TEAM_FLAG_BONUS;

	return true;
};

void Flag_Place (edict_t *self)
{
	float	*v;
	vec3_t	dest;
	trace_t	tr;

	if (self->model)
		gi.setmodel (self, self->model);
	else
		gi.setmodel (self, self->item->world_model);

	self->solid = SOLID_TRIGGER;
	self->movetype = MOVETYPE_TOSS;	

	VectorSet (self->mins, -16, -16, 0);
	VectorSet (self->maxs, 16, 16, 74);
	VectorClear(self->velocity);

	self->s.frame = FRAME_idle1;
	self->s.origin[2] += 6;
	self->think = Flag_Think;
	self->touch = Touch_Item;
	self->nextthink = level.time + 1 * FRAMETIME;
	self->style = FLAG_AT_BASE;
	self->flags = FL_TEAM_FLAG;
//	VectorCopy(self->s.origin, self->home_origin); // save for flag return

	v = tv(0,0,-128);
	VectorAdd (self->s.origin, v, dest);

	tr = gi.trace (self->s.origin, self->mins, self->maxs, dest, self, MASK_SOLID);
	if (tr.startsolid)
	{
		gi.dprintf ("Flag_Setup: %s startsolid at %s\n", self->classname, vtos(self->s.origin));
		G_FreeEdict (self);
		return;
	}

	VectorCopy (tr.endpos, self->s.origin);

};

void Flag_Spawn (edict_t *flag, int color) {

	gitem_t *item;
	
	item = FindItem( (color == CTF_TEAM1) ? "Red Flag" : "Blue Flag");

	PrecacheItem (item);

	flag->style = FLAG_AT_BASE;
	flag->item = item;
	flag->nextthink = level.time + 2 * FRAMETIME;    // items start after other solids
	flag->think = Flag_Place;
	flag->s.effects = item->world_model_flags;

}

void SP_item_flag_team1 (edict_t *ent)
{
	gi.dprintf("Spawning Red Flag\n");
	Flag_Spawn(ent, CTF_TEAM1);
}

void SP_item_flag_team2 (edict_t *ent)
{
	gi.dprintf("Spawning Blue Flag\n");
	Flag_Spawn(ent, CTF_TEAM2);
}

void Flag_CheckHurtCarrier(edict_t *targ, edict_t *attacker)
{
	gitem_t *flag_item;

	if (!targ->client || !attacker->client)
		return;

	if (targ->client->resp.ctf_team == CTF_TEAM1)
		flag_item = item_flag2;
	else
		flag_item = item_flag1;

	if (targ->client->pers.inventory[ITEM_INDEX(flag_item)] &&
		targ->client->resp.ctf_team != attacker->client->resp.ctf_team)
		attacker->client->resp.ctf_lasthurtcarrier = level.time;
}
