/*==========================================================================
 * 
 * Project : Quake2 Model loader contrib for quakeforge 
 * Author  : Terry 'Mongoose' Hendrix II
 * Website : http://www.westga.edu/~stu7440
 * Email   : stu7440@westga.edu
 * Object  : NA
 * Comments: This is the quake2 model loader module.
 *           I'm also adding a md2 -> mdl converter to this.
 *
 *           No endian checking.
 *
 *           See file COPYING for license details.
 *
 *           Based on viewmd2 (c) Michael K Vance, 1999  
 *                                briareos@lokigames.com
 *                                mkv102@psu.edu
 *
 *           Based on Meddle (c) Brian Martin 1996
 *                               brian@phyast.pitt.edu
 *
 *           This also uses qkspec and quake code and notes.
 *
 *-- History ---------------------------------------------------------- 
 *
 * 2000 03 03:
 * Mongoose - Created
 ==========================================================================*/

#define CONVERTER
#define TAG

#include "mddc.h"
#include "pcx.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>


BoundingBox_t *BoundingBox_New(const float xn, const float xx, const float yn,
		               const float yx, const float zn, const float zx)
{
   BoundingBox_t *bbox;

 
   bbox = (BoundingBox_t *)malloc(sizeof(BoundingBox_t));

   bbox->xMin = xn;
   bbox->xMax = xx;
   bbox->yMin = yn;
   bbox->yMax = yx;
   bbox->zMin = zn;
   bbox->zMax = zx;

   return bbox;
}

void BoundingBox_Update(BoundingBox_t *bbox, 
                 const float x, const float y, const float z)
{
    if (x < bbox->xMin) 
       bbox->xMin = x;
    else if ( x > bbox->xMax )
       bbox->xMax = x;

    if (y < bbox->yMin)
       bbox->yMin = y;
    else if (y > bbox->yMax)
       bbox->yMax = y;

    if (z < bbox->zMin)
       bbox->zMin = z;
    else if (z > bbox->zMax)
	bbox->zMax = z;
}

QuakeModel2_t* QuakeModel2_Import(char *filename)
{
   char skins[MD2_MAX_MD2SKINS][MD2_MAX_SKINNAME];
   int ident;
   int version;
   int skinWidth;
   int skinHeight;
   int frameSize;
   int numSkins;
   int numST;
   int numGLCmds;
   int ofsSkins;
   int ofsST;
   int ofsTris;
   int ofsFrames;
   int ofsGLCmds;
   int ofsEnd;
   int i, j;
   QuakeModel2_t *q2m = NULL;
   BoundingBox_t *bbox = NULL; 
   FILE *f;
   int numXYZ;
   int numTris;
   int numFrames;
   TexCoordsIndex_t *texCoords;
   TriangleIndex_t *tris;
   AliasFrame_t *frames;
   int *commands;
   int vertc = 0;
   

   f = fopen(filename, "rb");

   printf("QuakeModel2_Import\n |\n");

   if (!f)
   {
      printf(" |- File %s not found.\n", filename);
      return NULL;
   }
   else
      printf(" |- Importing %s as MD2.\n", filename);

   fread(&ident, sizeof(int), 1, f);
      
   if (ident != MD2_IDALIASHEADER) 
   {
      printf(" |- File is not in MD2 format.\n");
      return NULL;
   }

   fread(&version, sizeof(int), 1, f);
   
   if (version != MD2_ALIAS_VERSION) 
   {
      printf(" |- File is not in correct MD2 format.\n");
      return NULL;
   }

   fread(&skinWidth, sizeof(int), 1, f);
   fread(&skinHeight, sizeof(int), 1, f);
   fread(&frameSize, sizeof(int), 1, f);
   fread(&numSkins, sizeof(int), 1, f);

   printf(" |- Skin width is %i.\n", skinWidth);
   printf(" |- Skin height is %i.\n",skinHeight );
   printf(" |- Frame size is %i.\n", frameSize);
   printf(" |- Contains %i skins.\n", numSkins);

   if (numSkins > MD2_MAX_MD2SKINS) 
   {
      printf("MD2: numSkins > MD2_MAX_MD2SKINS\n");
      return NULL;
   }

   fread(&numXYZ, sizeof(int), 1, f);

   if (numXYZ > MD2_MAX_VERTS) 
   {
      printf("MD2: numXYZ > MD2_MAX_VERTS\n");
      return NULL;
   }

   fread(&numST, sizeof(int), 1, f);
   fread(&numTris, sizeof(int), 1, f);

   if (numTris > MD2_MAX_TRIANGLES) 
   {
      printf("MD2: numTris > MD2_MAX_TRIANGLES\n");
      return NULL;
   }

   fread(&numGLCmds, sizeof(int), 1, f);
   fread(&numFrames, sizeof(int), 1, f);

   if (numFrames > MD2_MAX_FRAMES) 
   {
      printf("MD2: numFrames > MD2_MAX_FRAMES\n");
      return NULL;
   }

   fread(&ofsSkins, sizeof(int), 1, f);
   fread(&ofsST, sizeof(int), 1, f);
   fread(&ofsTris, sizeof(int), 1, f);
   fread(&ofsFrames, sizeof(int), 1, f);
   fread(&ofsGLCmds, sizeof(int), 1, f);
   fread(&ofsEnd, sizeof(int), 1, f);

   /* John King (john.king@ucl.ac.uk): 
      This is where skin names are read.  I don't expose 
      the functionality, but it would be trivial to do so. */
   for (i = 0; i < numSkins; i++) 
   {
      fread(&(skins[i]), MD2_MAX_SKINNAME, 1, f);
   }

   printf(" |- Read %i skins.\n", i);   

   /* texCoords = new TexCoordsIndex[ numST ]; */
   texCoords = (TexCoordsIndex_t*)malloc(sizeof(TexCoordsIndex_t) * numST);

   /* Read texture coordinates in pixel values.*/
   for (i = 0; i < numST; i++) 
   {
      short s, t;

      fread(&s, sizeof(short), 1, f);
      fread(&t, sizeof(short), 1, f);

      texCoords[i].s = s;
      texCoords[i].t = t;
   }
   printf(" |- Read %i texCoords.\n", i);   

   /* tris = new TriangleIndex[ numTris ]; */
   tris = (TriangleIndex_t*)malloc(sizeof(TriangleIndex_t) * numTris);

   /* Read triangles. */
   for(i = 0; i < numTris; i++ ) 
   {
      short v0, v1, v2;
      short st0, st1, st2;

      fread(&v0, sizeof(short), 1, f);
      fread(&v1, sizeof(short), 1, f);
      fread(&v2, sizeof(short), 1, f);
      fread(&st0, sizeof(short), 1, f);
      fread(&st1, sizeof(short), 1, f);
      fread(&st2, sizeof(short), 1, f);

      tris[i].v[0] = v0;
      tris[i].v[1] = v1;
      tris[i].v[2] = v2;
      tris[i].st[0] = st0;
      tris[i].st[1] = st1;
      tris[i].st[2] = st2;
   }
   printf(" |- Read %i triangles.\n", i);   

   /* frames = new AliasFrame[ numFrames ]; */
   frames = (AliasFrame_t*)malloc(sizeof(AliasFrame_t) * numFrames);

   /* Read frames. */
   for(i = 0; i < numFrames; i++) 
   {
      float s0, s1, s2;
      float t0, t1, t2;


      fread(&s0, sizeof(float), 1, f);
      fread(&s1, sizeof(float), 1, f);
      fread(&s2, sizeof(float), 1, f);
      fread(&t0, sizeof(float), 1, f);
      fread(&t1, sizeof(float), 1, f);
      fread(&t2, sizeof(float), 1, f);

      frames[i].scale[0] = s0;
      frames[i].scale[1] = s1;
      frames[i].scale[2] = s2;
      frames[i].translate[0] = t0;
      frames[i].translate[1] = t1;
      frames[i].translate[2] = t2;

      fread(&frames[i].name, sizeof(char) * 16, 1, f);

      /* frames[i].verts = new VertexIndex[ numXYZ ]; */
      frames[i].verts = (VertexIndex_t*)malloc(sizeof(VertexIndex_t) *numXYZ);

      bbox = BoundingBox_New(0, 0, 0, 0, 0, 0);   

      for(j = 0; j < numXYZ; j++) 
      {
         unsigned char v;


         fread(&v, sizeof(unsigned char), 1, f);  /* Mongoose: Read 
                                                       in the packed vert */

         /* Mongoose: pos[i] = (packed[i] &scale[i])/ origin[i] */

	 frames[i].verts[j].x = ( ( v * frames[i].scale[0] ) + 
                                   frames[i].translate[0] );

         fread(&v, sizeof(unsigned char), 1, f);

	 frames[i].verts[j].y = ( ( v * frames[i].scale[1] ) + 
                                   frames[i].translate[1] );

         fread(&v, sizeof(unsigned char), 1, f);

	 frames[i].verts[j].z = ( ( v * frames[i].scale[2] ) + 
                                   frames[i].translate[2] );

         fread(&(frames[i].verts[j].normal), sizeof(unsigned char), 1, f);

	 BoundingBox_Update(bbox,
                            frames[i].verts[j].x, 
                            frames[i].verts[j].y, 
                            frames[i].verts[j].z);
	}
        vertc += j;
   }

   printf(" |- Read %i vertices in %i frames.\n", vertc, i);   

   if (i)
      printf(" |- Read %i vertices each frame.\n", vertc/i);   

   /* commands = new int[ numGLCmds ]; */
   commands = (int*)malloc(sizeof(int) * numGLCmds);

   for (i = 0; i < numGLCmds; i++ ) 
   {
      int cmd;


      fread(&cmd, sizeof(int), 1, f);
      commands[i] = cmd;
   }

   q2m = (QuakeModel2_t*)malloc(sizeof(QuakeModel2_t));

   q2m->skinW = skinWidth;
   q2m->skinH = skinHeight;
   q2m->numST = numST;

   q2m->numXYZ = numXYZ;
   q2m->numTris = numTris;
   q2m->numFrames = numFrames;
   q2m->texCoords = texCoords;
   q2m->tris = tris;
   q2m->frames = frames;
   q2m->commands = commands;
   q2m->bbox = bbox;

   printf(" |- Checksum would be{v %i, t %i, f %i}\n",
          numXYZ, numTris, numFrames);
   printf(" |- Done\n");
   return q2m;
}

void QuakeModel2_Delete(QuakeModel2_t *q2m)
{
   int i, j;

   
   if (!q2m)
      return;

   if (q2m->texCoords)
      free(q2m->texCoords);

   if (q2m->tris)
      free(q2m->tris);

   if (q2m->frames)
   {
      j = sizeof(q2m->frames) / sizeof(AliasFrame_t);

      for (i = 0; i < j; i++)
         if (q2m->frames[i].verts)
            free(q2m->frames[i].verts);
 
      free(q2m->frames);
   }

   if (q2m->commands)
      free(q2m->commands);

   free(q2m);
}

void QuakeModel2_getTexCoords(QuakeModel2_t *q2m,
                              short* s, short* t, int tri, int vert)
{
   int index = q2m->tris[tri].st[vert];
   *s = q2m->texCoords[index].s;
   *t = q2m->texCoords[index].t;
}

float QuakeModel2_getRadius(QuakeModel2_t *q2m)
{
    float radius;


    if (!q2m->bbox)
    {
       printf("QuakeModel2_getRadius: NULL bbox.\n");
       return 0.0;
    }

    radius = fabs(q2m->bbox->xMin);

    if (fabs(q2m->bbox->xMax) > radius) 
	radius = q2m->bbox->xMax;

    if (fabs(q2m->bbox->yMin) > radius)
	radius = q2m->bbox->yMin;

    if (fabs(q2m->bbox->yMax) > radius)
	radius = q2m->bbox->yMax;

    if (fabs(q2m->bbox->zMin) > radius)
	radius = q2m->bbox->zMin;

    if (fabs(q2m->bbox->zMax) > radius)
	radius = q2m->bbox->zMax;

    return radius;
}

/*=============================================================
 *  Build_Quake_Header_From_Md2              status 80%
 *=============================================================
 * Fill in MDL header 
 *
 * Fixme: Finish filling in feilds when 
 *        MD2 -> MDL mapping is known
 *
 *---Log-------------------------------------------------------
 *
 * 2000 03 05:
 * Mongoose - Created
 *
 *---Notes-----------------------------------------------------
 *
 * packedpos[i] = (pos[i] - origin[i]) / scale[i]
 *
 ==============================================================*/
void Build_Quake_Header_From_Md2(QuakeModel2_t *q2m, mdl_t *mdl)
{
   printf("Build_Quake_Header_From_Md2: Making a MDL header.");

   mdl->ident = MDL_IDPOLYGON;
   mdl->version = MDL_ALIAS_VERSION;
   mdl->boundingradius = QuakeModel2_getRadius(q2m);
   mdl->skinwidth = q2m->skinW;
   mdl->skinheight = q2m->skinH;

#define FEMALE_TEST
#ifdef FEMALE_TEST
   mdl->scale[0] = 0.1;
   mdl->scale[1] = 0.1;
   mdl->scale[2] = 0.1;
   mdl->scale_origin[0] = 0.0 - 16;
   mdl->scale_origin[1] = 0.0 - 16;
   mdl->scale_origin[2] = 0.0 + 8;
#else
   mdl->scale[0] = (q2m->bbox->xMax - q2m->bbox->xMin) / 255.9;
   mdl->scale[1] = (q2m->bbox->yMax - q2m->bbox->yMin) / 255.9;   
   mdl->scale[2] = (q2m->bbox->zMax - q2m->bbox->zMin) / 255.9;
   mdl->scale_origin[0] = q2m->bbox->xMin;
   mdl->scale_origin[1] = q2m->bbox->yMin;  
   mdl->scale_origin[2] = q2m->bbox->zMin;
#endif

   printf("\n |   |- scale {%f, %f, %f}\n", 
           mdl->scale[0], mdl->scale[1], mdl->scale[2]);
   printf(" |   |- offset {%f, %f, %f}\n |   |- ", 
           mdl->scale_origin[0], mdl->scale_origin[1], mdl->scale_origin[2]);


   /* Mongoose: These will be set later */
   mdl->numskins = 0;
   mdl->numverts = 0;
   mdl->numtris = 0; 
   mdl->numframes = 0; 
   mdl->flags = 0;

   /* Mongoose: Below here are unknowns with my guesses */
   mdl->eyeposition[0] = 0.0;     /* Eye position (useless?) */
   mdl->eyeposition[1] = 0.0;
   mdl->eyeposition[2] = q2m->bbox->zMax - 8.0;
   mdl->synctype = ST_SYNC;       /* See enum synctype_t */
   mdl->size = 0.0;               /* Average size of triangles */

   printf("  [done]\n");
}

/*=============================================================
 *  Build_Quake_TexCoord_From_Md2             status 90%
 *=============================================================
 * Build skin vertices
 *
 * Fixme: Need to kludge onseam info to allow for correct
 *        texture mapping!
 *
 *
 *---Log-------------------------------------------------------
 *
 * 2000 03 05:
 * Mongoose - Created
 *
 *---Notes-----------------------------------------------------
 * Skin vert data from qkspec:
 *
 * onseam is a flag, and if non zero it means that the vertex 
 * is on the boundary between the skin part that is applied on 
 * the front of the model, and the skin part that is applied on 
 * the back of the models (i.e. on the edge).
 *
 * If a vertex is onseam, but is part of a triangle that is on 
 * the back side of the model (facesfront is 0), then 
 * skinwidth/2 must be added to s so as to find the actual 
 * value of s.
 ==============================================================*/
void Build_Quake_TexCoord_From_Md2(QuakeModel2_t *q2m, QuakeModel_t *qm)
{
   int i, numverts;


   qm->header.numverts = q2m->numST;

   numverts = qm->header.numverts;

   qm->vertex = (stvert_t *)malloc(sizeof(stvert_t) * numverts);

   if (!qm->vertex)
   {
      qm->header.numverts = 0;
      return;
   }

   printf("Build_Quake_TexCoord_From_Md2: Converting %i 2d skin vertices.",
          numverts);

   for(i = 0; i < numverts; i++)
   {   
      qm->vertex[i].onseam = 0;               /* At present we have no data 
                                                 for gathering MD2 'onseam' */

      qm->vertex[i].s = q2m->texCoords[i].s;  /* X */
      qm->vertex[i].t = q2m->texCoords[i].t;  /* Y */
   }

   printf("  [done]\n");
}

/*=============================================================
 *  Build_Quake_TexCoord_From_Md2             status 90%
 *=============================================================
 * Build skin vertices
 *
 * Fixme: It seems MD2 ST triangles are needed for correct 
 *        texmapping, but how do we use them?
 *
 * Fixme: Can't be sure of facing of triangles, so this will
 *        only draw the front of a model with void on back?
 *
 *---Log-------------------------------------------------------
 *
 * 2000 03 05:
 * Mongoose - Created
 *
 *---Notes-----------------------------------------------------
 *
 * From qkspec:
 * 
 * An Alias Model is made of a set of triangle facets, with 
 * vertices at the boundaries. Triangles should all be valid 
 * triangles, not degenerates (like points or lines).
 *
 * Only vertices index are stored in triangles. the normal 
 * vector of the surface is reconstituted from the vertex position.
 *
 * The boolean facesfront indicates if the triangle is part of the 
 * front or the back skin. 1 means that it is on the front skin, 0 
 * means that it is on the back skin.
 *
 * When the triangle is on the back skin, then any skin vertex 
 * that is on the skin seam (as indicated by onseam=1) must have 
 * it's s coordinate increased by skinwidth/2.
 *
 * As a matter of fact, on the skin picture, the back skin is always 
 * situtated at the same level as the front skin, but moved by 
 * skinwidth/2 to the right (check this, with any model).
 *
 * The following code might make this easier to understand: 
 *
 *  for(j=0; j < numtris; j++)
 *  {
 *    for(i=0; i < 3 ; i++)
 *    { 
 *      vertex = triangles[j].vertices[i];
 *      s = vertices[vertex].s;
 *      t = vertices[vertex].t;
 *      if( (vertices[vertex].onseam) && (!triangle[j].facesfront))
 *      { 
 *        s += skinwidth / 2;
 *      }
 *      // use s and t as the coordinates of the vertex
 *    }
 *  }
 *
 ==============================================================*/
void Build_Quake_Triangles_From_Md2(QuakeModel2_t *q2m, QuakeModel_t *qm)
{
   int i, numtris;

   qm->header.numtris = q2m->numTris;

   numtris = qm->header.numtris;
  
   qm->triangle = (dtriangle_t *) malloc(sizeof(dtriangle_t) * numtris);

   printf("Build_Quake_Triangles_From_Md2: Building %i triangles.",
          numtris);

   for(i = 0; i < numtris; i++)
   {
      qm->triangle[i].vertindex[0] = q2m->tris[i].v[0];
      qm->triangle[i].vertindex[1] = q2m->tris[i].v[1];
      qm->triangle[i].vertindex[2] = q2m->tris[i].v[2];
      qm->triangle[i].facesfront = 1;  

      /* Mongoose: I guess make all triangles assume they face 
                   outwards for now, it's going to be hard to
                   figure out MD2 facing at this point */
   }

   printf("  [done]\n");
}

/*=============================================================
 *  Build_Quake_Skin                status 90%
 *=============================================================
 * Build skin from pcx file
 *
 * FIXME: Add pal conversion ( which is just busy work )
 *        Goto pcx.c and add if you like, just remap the index.
 *
 * FIXME: Only handles one skin, but this is just busy 
 *        work for multiskin
 *
 *---Log-------------------------------------------------------
 *
 * 2000 03 05:
 * Mongoose - Created
 *
 *---Notes-----------------------------------------------------
 *
 * I'm skipping trival coding for now, so I can finish faster.
 *
 ==============================================================*/
void Build_Quake_Skin(QuakeModel_t *model, char *pcxfilename, int q2_pal)
{
   FILE *f;
   short int w, h;


   printf("Build_Quake_Skin\n |- Using skin '%s'.\n", 
          pcxfilename);

   model->skin = (QuakeModel_Skin_t*)malloc(sizeof(QuakeModel_Skin_t));
   
   model->skin->type = 0;

   f = fopen(pcxfilename, "r");

   model->skin->bitmap = NULL;

   if (f)
      model->skin->bitmap = readpcx(f, NULL, &w, &h);

   if (model->skin->bitmap)
   {
      model->header.numskins = 1;

      printf(" |- Done\n");
   }
   else
      printf(" |- Failed\n");
}

/*=============================================================
 *  Build_Quake_Frames_From_Md2                status 90%
 *=============================================================
 * Build frames
 *
 * FIXME: Not sure if I need to support grouped frames
 *
 *---Log-------------------------------------------------------
 *
 * 2000 03 05:
 * Mongoose - Created
 *
 *---Notes-----------------------------------------------------
 *
 *
 ==============================================================*/
void Build_Quake_Frames_From_Md2(QuakeModel2_t *q2m, QuakeModel_t *qm)
{
   int i, j, k, hack;


   printf("Build_Quake_Frames_From_Md2: Constructing frames.\n");

   qm->header.numframes = q2m->numFrames;

   qm->frame = (QuakeModel_Frame_t *)
       malloc(sizeof(QuakeModel_Frame_t) * qm->header.numframes);

   if (!qm->frame)
      return;

   hack = 1;  /* Mongoose: Single frames, no grouping */
   //hack = 4;

   for(k = 0; k < qm->header.numframes/hack; k++)
   {
     //qm->frame[0].type = 1;   /* FIXME: made up number != 0 */

      if(qm->frame[k].type == 0)   /* if single frames */
      {
         strcpy(qm->frame[k].sframe.name, q2m->frames[k].name);

         printf(" |   |- Building frame[%d]: %s.\n", k + 1,
                qm->frame[k].sframe.name);

         qm->frame[k].sframe.bboxmin.v[0] = q2m->bbox->xMin;//FIXME
         qm->frame[k].sframe.bboxmin.v[1] = q2m->bbox->yMin;//FIXME
         qm->frame[k].sframe.bboxmin.v[2] = q2m->bbox->zMin;//FIXME
         qm->frame[k].sframe.bboxmin.lightnormalindex = 1;//FIXME

         qm->frame[k].sframe.bboxmax.v[0] = q2m->bbox->xMax;//FIXME
         qm->frame[k].sframe.bboxmax.v[1] = q2m->bbox->yMax;//FIXME
         qm->frame[k].sframe.bboxmax.v[2] = q2m->bbox->zMax;//FIXME
         qm->frame[k].sframe.bboxmax.lightnormalindex = 1;//FIXME

         qm->frame[k].sframe.tv = (trivertex_t*)
	        malloc(sizeof(trivertex_t) * qm->header.numverts);

         for(i = 0; i < qm->header.numverts; i++)
         {
	    /* Mongoose: Convert back to packed postions */

            qm->frame[k].sframe.tv[i].v[0] =
               (q2m->frames[k].verts[i].x - q2m->frames[k].translate[0]) /
		q2m->frames[k].scale[0];

            qm->frame[k].sframe.tv[i].v[1] =
               (q2m->frames[k].verts[i].y - q2m->frames[k].translate[1]) /
		q2m->frames[k].scale[1];

            qm->frame[k].sframe.tv[i].v[2] =
               (q2m->frames[k].verts[i].z - q2m->frames[k].translate[2]) /
		q2m->frames[k].scale[2];


            qm->frame[k].sframe.tv[i].lightnormalindex = 
                  q2m->frames[k].verts[i].normal;


            #ifdef HEAVY_DEBUG
            printf("{%i, %i, %i; %i}\n", 
                   qm->frame[k].sframe.tv[i].v[0],
                   qm->frame[k].sframe.tv[i].v[1],
                   qm->frame[k].sframe.tv[i].v[2],
                   qm->frame[k].sframe.tv[i].lightnormalindex);
            #endif
         }
      }
      else
      {
	 printf(" |   |\n");
         printf(" |   |- Building group[%d]: \n", k + 1);

         qm->frame[k].gframe.num_gsframes = hack;//FIXME
         
         qm->frame[k].gframe.bboxmin.v[0] = q2m->bbox->xMin;//FIXME
         qm->frame[k].gframe.bboxmin.v[1] = q2m->bbox->yMin;//FIXME
         qm->frame[k].gframe.bboxmin.v[2] = q2m->bbox->zMin;//FIXME
         qm->frame[k].gframe.bboxmin.lightnormalindex = 1;//FIXME

         qm->frame[k].gframe.bboxmax.v[0] = q2m->bbox->xMax;//FIXME
         qm->frame[k].gframe.bboxmax.v[1] = q2m->bbox->yMax;//FIXME
         qm->frame[k].gframe.bboxmax.v[2] = q2m->bbox->zMax;//FIXME
         qm->frame[k].gframe.bboxmax.lightnormalindex = 1;//FIXME

         qm->frame[k].gframe.gsframe_interval = (float*)
             malloc(sizeof(float)*qm->frame[k].gframe.num_gsframes);

         qm->frame[k].gframe.gsframe = (QuakeModel_MFrame_t*)
             malloc(sizeof(QuakeModel_MFrame_t)*
                    qm->frame[k].gframe.num_gsframes);


         for(j=0; j<qm->frame[k].gframe.num_gsframes; j++)
         {
	    qm->frame[k].gframe.gsframe_interval[j] = 0.1;//FIXME: guess
            //printf("%g :",qm->frame[k].gframe.gsframe_interval[j]);
         }

         for(j = 0; j < qm->frame[k].gframe.num_gsframes; j++)
         {
	    ////////FIXME
            qm->frame[k].gframe.gsframe[j].bboxmin.v[0] = q2m->bbox->xMin;
            qm->frame[k].gframe.gsframe[j].bboxmin.v[1] = q2m->bbox->yMin;
            qm->frame[k].gframe.gsframe[j].bboxmin.v[2] = q2m->bbox->zMin;
            qm->frame[k].gframe.gsframe[j].bboxmin.lightnormalindex = 1;

	    ////////FIXME
            qm->frame[k].gframe.gsframe[j].bboxmax.v[0] = q2m->bbox->xMax;
            qm->frame[k].gframe.gsframe[j].bboxmax.v[1] = q2m->bbox->yMax;
            qm->frame[k].gframe.gsframe[j].bboxmax.v[2] = q2m->bbox->zMax;
            qm->frame[k].gframe.gsframe[j].bboxmax.lightnormalindex = 1; 	    

 	    strcpy(qm->frame[k].gframe.gsframe[j].name, 
                   q2m->frames[j/*k*/].name);

            qm->frame[k].gframe.gsframe[j].tv = (trivertex_t*)
	         malloc(sizeof(trivertex_t) * qm->header.numverts);

            printf(" |   |  |- sub frame[%d]: %s\n",
                   j + 1, 
                   qm->frame[k].gframe.gsframe[j].name);

            for (i = 0; i < qm->header.numverts; i++)
            {
	       /* Mongoose: Convert back to packed postions */

               qm->frame[k].gframe.gsframe[j].tv[i].v[0] =
                  (q2m->frames[k].verts[i].x - q2m->frames[k].translate[0]) /
		   q2m->frames[k].scale[0];

               qm->frame[k].gframe.gsframe[j].tv[i].v[1] =
                  (q2m->frames[k].verts[i].y - q2m->frames[k].translate[1]) /
		   q2m->frames[k].scale[1];

               qm->frame[k].gframe.gsframe[j].tv[i].v[2] =
                  (q2m->frames[k].verts[i].z - q2m->frames[k].translate[2]) /
	       	   q2m->frames[k].scale[2];


               qm->frame[k].gframe.gsframe[j].tv[i].lightnormalindex = 
                     q2m->frames[k].verts[i].normal;

   
               #ifdef HEAVY_DEBUG
               printf("{%i, %i, %i; %i}\n", 
                      qm->frame[k].gframe.gsframe[j].tv[i].v[0],
                      qm->frame[k].gframe.gsframe[j].tv[i].v[1],
                      qm->frame[k].gframe.gsframe[j].tv[i].v[2],
                      qm->frame[k].gframe.gsframe[j].tv[i].lightnormalindex);
               #endif
            }
         }

         printf(" |   |  |- Done\n");
      }
   }

   printf(" |   |\n");
   printf(" |   |- Done\n");
}

QuakeModel_t *QuakeModel2_to_QuakeModel(QuakeModel2_t *q2m)
{
   QuakeModel_t *model;


   if (!q2m)
     return NULL;

   model = (QuakeModel_t *)malloc(sizeof(QuakeModel_t));

   printf("QuakeModel2_to_QuakeModel\n | \n");

   printf(" |- ");
   Build_Quake_Header_From_Md2(q2m, &model->header);

   printf(" |- ");
   Build_Quake_TexCoord_From_Md2(q2m, model);

   printf(" |- ");
   Build_Quake_Triangles_From_Md2(q2m, model);

   printf(" |- ");
   Build_Quake_Frames_From_Md2(q2m, model);


   printf(" |\n");
   printf(" |- Done\n");

   return model;
}

void QuakeModel_Delete(QuakeModel_t *model)
{
  int i, j, k;
   

   if (!model)
      return;

   if (model->triangle)
     free(model->triangle);

   if (model->vertex)
     free(model->vertex);

   if (model->skin->bitmap)
     free(model->skin->bitmap);

   if (model->skin)
     free(model->skin); 

   if (model->frame)
   {
     j = model->header.numframes;

      for (i = 0; i < j; i++)
      {
         for(k=0; j<model->frame[i].gframe.num_gsframes; k++)
	   if (model->frame[i].gframe.gsframe[k].tv)
             free(model->frame[i].gframe.gsframe[k].tv);

	 if (model->frame[i].sframe.tv)
            free(model->frame[i].sframe.tv);

	 if (model->frame[i].gframe.gsframe_interval)
	    free(model->frame[i].gframe.gsframe_interval);

         if (model->frame[i].gframe.gsframe)
	    free(model->frame[i].gframe.gsframe);
      }

      free(model->frame);
   }

   free(model);
}

void QuakeModel_Export(QuakeModel_t *model, char *filename)
{
   int i, j, k, ii, jj;
   FILE *out;

#ifdef TAG
   char tag[64];
   sprintf(tag, "[Converted MD2; MDDC by stu7440@westga.edu]");
#endif

   out = fopen(filename, "wb");

   printf("QuakeModel_Export\n |\n");

   if (!out)
   {  
      printf(" |- Couldn't write %s.\n", filename);
      return;
   }
   else
      printf(" |- Exporting %s as MDL.\n", filename);

   /* Mongoose: Start header */

   fwrite(&model->header.ident,4,1,out);
   fwrite(&model->header.version,4,1,out);
   fwrite(&model->header.scale[0],4,1,out);
   fwrite(&model->header.scale[1],4,1,out); 
   fwrite(&model->header.scale[2],4,1,out);
   fwrite(&model->header.scale_origin[0],4,1,out);
   fwrite(&model->header.scale_origin[1],4,1,out);
   fwrite(&model->header.scale_origin[2],4,1,out);
   fwrite(&model->header.boundingradius,4,1,out);
   fwrite(&model->header.eyeposition[0],4,1,out);           
   fwrite(&model->header.eyeposition[1],4,1,out);
   fwrite(&model->header.eyeposition[2],4,1,out);
   fwrite(&model->header.numskins,4,1,out);
   fwrite(&model->header.skinwidth,4,1,out);
   fwrite(&model->header.skinheight,4,1,out);
   fwrite(&model->header.numverts,4,1,out);
   fwrite(&model->header.numtris,4,1,out);
   fwrite(&model->header.numframes,1,4,out);
   fwrite(&model->header.synctype,1,4,out); 
   fwrite(&model->header.flags,4,1,out);     
   fwrite(&model->header.size,4,1,out);  
   
   /* Mongoose: End header */

   /* Mongoose: Skin info */
   for(k = 0; k < model->header.numskins; k++)
   {
      fwrite(&model->skin[k].type, 4, 1, out);

      if(model->skin[k].type == 0)
      {
#ifdef NOT_USING_MONGOOSE_IMAGE_BUFFER
         for(j = 0; j < model->header.skinheight; j++)
         {
	   for(i = 0; i < model->header.skinwidth; i++)
	   {
	      fputc(model->skin[k].bitmap[i+model->header.skinwidth*j], out);
	   }
         }
#else
	 for(i = 0; i < model->header.skinwidth*model->header.skinheight; i++)
            fputc(model->skin[k].bitmap[i], out);
#endif
      }
      else
      {
         fwrite(&model->skin[k].num_gskins, 4, 1, out);

         for(j = 0; j < model->skin[k].num_gskins; j++)
         {
            fwrite(&model->skin[k].gskin_interval[j], 4, 1, out);
         }

         for(j = 0; j < model->skin[k].num_gskins; j++)
         {
            for(jj = 0; jj < model->header.skinheight; jj++)
            {
               for(ii = 0; ii < model->header.skinwidth; ii++)
               {
                  fputc(model->skin[k].gskin[j].bitmap[ii+model->header.skinwidth*jj],
                        out);
               }
            }
         }
      }
   }
   printf(" |- Wrote %i skins.\n", k);

   /* Mongoose: Skin 2d Vertices */
   for(i = 0; i < model->header.numverts; i++)
   {
      fwrite(&model->vertex[i].onseam, 4, 1, out);
      fwrite(&model->vertex[i].s, 4, 1,out);
      fwrite(&model->vertex[i].t, 4, 1, out);
   }
   printf(" |- Wrote %i vertices.\n", i);

   /* Mongoose: Triangle definitions */
   for(i = 0; i < model->header.numtris; i++)
   {
      fwrite(&model->triangle[i].facesfront, 4, 1, out);
      fwrite(&model->triangle[i].vertindex[0], 4, 1, out);
      fwrite(&model->triangle[i].vertindex[1], 4, 1, out);
      fwrite(&model->triangle[i].vertindex[2], 4, 1, out);
   }
   printf(" |- Wrote %i triangles.\n", i);

   /* Mongoose: Frames */
   for(k = 0; k < model->header.numframes;  k++)
   {
     //model->frame[k].type = 1;  // FIXME: hack to force multi frames

      fwrite(&model->frame[k].type,4,1,out); 

      if (model->frame[k].type == 0)  
      {
         fwrite(&model->frame[k].sframe.bboxmin,1,4,out);
         fwrite(&model->frame[k].sframe.bboxmax,1,4,out);
         fwrite(model->frame[k].sframe.name,16,1,out);

         for(i=0;i<model->header.numverts;i++)
         {
            fwrite(model->frame[k].sframe.tv+i,1,4,out);
         }
      }
      else
      {
         fwrite(&model->frame[k].gframe.num_gsframes,1,4,out);
         fwrite(&model->frame[k].gframe.bboxmin,1,4,out);
         fwrite(&model->frame[k].gframe.bboxmax,1,4,out);
         for(j=0; j<model->frame[k].gframe.num_gsframes; j++)
         {
            fwrite(&model->frame[k].gframe.gsframe_interval[j],4,1,out);
         }

         for(j=0; j<model->frame[k].gframe.num_gsframes; j++)
         {
            fwrite(&model->frame[k].gframe.gsframe[j].bboxmin,1,4,out);
            fwrite(&model->frame[k].gframe.gsframe[j].bboxmax,1,4,out);
            fwrite(model->frame[k].gframe.gsframe[j].name,16,1,out);

            for(i=0;i< model->header.numverts;i++)
            {
               fwrite(model->frame[k].gframe.gsframe[j].tv+i,1,4,out);
            }
         }
      }
   }
   printf(" |- Wrote %i frames.\n", k);

#ifdef TAG
   fwrite(tag, 1, strlen(tag), out);
#endif

   fclose(out);
   printf(" |- Done\n");
}

#ifdef  CONVERTER
int main(int argc, char *argv[])
{
   QuakeModel2_t *q2m = NULL; 
   QuakeModel_t *qm = NULL; 
   int q2_pal = 0;


   if (argc > 4)
      q2_pal = atoi(argv[4]);

   if (argc > 2)
   {
      printf("\n");
      q2m = QuakeModel2_Import(argv[1]);
      printf("\n");

      if (q2m)
      {        
         printf("\n");
	 qm = QuakeModel2_to_QuakeModel(q2m);
         printf("\n");

         if (qm)
	 {
	    if (argc > 3)
            {
               printf(" ");
               Build_Quake_Skin(qm, argv[3], q2_pal);   
               printf(" ");
	    }

            printf("\n");
	    QuakeModel_Export(qm, argv[2]);
            printf("\n");
            QuakeModel_Delete(qm);
	 }

         QuakeModel2_Delete(q2m);
      }
   }
   else
   {
      printf("%s FILENAME.MD2 FILENAME.MDL\n", argv[0]);
      return 0;
   }

   return 0;
}
#endif
