#include "newproject.h"
#include "globals.h"

#define VERSION "1.9.2"

int main (int argc, char *argv[])
{
  char basepath[1024];
  int  localedit = 0; //edit the g_local.h file?
  int  publicedit = 0; //edit the bg_local.h file?
  int  overwrite = 0; //overwrite an existing mod?
  int  backup = 0; //backup mode?
  
  msg(va("NewMod %s by Mark \"WarZone\" Smeltzer", VERSION));
  msg("-----------------------------------------\n");

  //get and store the starting directory
  getexecutablepath(argv[0], basepath, 1024); //used to create the compile_<modname>.bat file
 
  if (!checkargs(argc, argv))
  {
    helpmsg();
  } else {
    char *current;
    int i = 4;
    
    if (strcmp("backup", argv[ARG_BACKUP]) == 0)
    {
      backup = 1;
      dobackup(); //kind of a hack to put this here... but oh well :P
      return 0;
    }

    for (i = 4; i < argc; i++) {
      current = argv[i];
      if (strcmp("localedit", current) == 0)
      {
        localedit = 1;
      } 
      else if (strcmp("publicedit", current) == 0)
      {
        publicedit = 1;
      }
      else if (strcmp("overwrite", current) == 0)
      {
        overwrite = 1;
      }
    }
    
    if (badsource()) {
      errormsg(ERROR_BAD_SOURCE);
    } else {

      if (!direxists(argv[ARG_Q3_PATH]))
      { //make sure the path to Q3A exists
        errormsg(ERROR_BAD_Q3_PATH);
      } 
      else if (subdirexists(argv[ARG_MOD_NAME], argv[ARG_Q3_PATH]) && !overwrite) 
      { //test to see if the mod already exists!
        errormsg(ERROR_BAD_MOD_NAME);
      }
      else
      { //start coping stuff over,,
        if (subdirexists(argv[ARG_MOD_NAME], argv[ARG_Q3_PATH]))
        {
          char in = '\0';
          msg(va("%s is about to be overwritten. Are you sure you want to do this? [Y / N]", argv[ARG_MOD_NAME]));
          
          while (in != 'N' && in != 'n' && in != 'Y' && in != 'y')
            in = getch();

          if (in == 'N' || in == 'n')
          {
            msg("Aborting..");
            return 0; //premature exit
          }
        }
        
        msg("Creating directories...");
        msg(va("  %s\\%s", argv[ARG_Q3_PATH], argv[ARG_MOD_NAME]));

        // only continue if the mod path is created successfully
        if (createsubdir(argv[ARG_MOD_NAME], argv[ARG_Q3_PATH]) || overwrite)
        {
          //create the main mod directory...
          msg(va("  %s", argv[ARG_MOD_PATH]));
          createdir(argv[ARG_MOD_PATH]);

          //create the sub directories
          msg(va("  %s\\game", argv[ARG_MOD_PATH]));
          createsubdir("game", argv[ARG_MOD_PATH]);
          msg(va("  %s\\cgame", argv[ARG_MOD_PATH]));
          createsubdir("cgame", argv[ARG_MOD_PATH]);
          msg(va("  %s\\ui", argv[ARG_MOD_PATH]));
          createsubdir("ui", argv[ARG_MOD_PATH]);

          msg("\n=============\n");

          //copy over the source files
          msg("Copying Project files..");
          copyfiles(copyPROJECTfiles, argv[ARG_MOD_PATH], "");
          msg("Copying Server files..");
          copyfiles(copyGAMEfiles, argv[ARG_MOD_PATH], "game");
          msg("Copying Client files..");
          copyfiles(copyCGAMEfiles, argv[ARG_MOD_PATH], "cgame");
          msg("Copying User Interface files..");
          copyfiles(copyUIfiles, argv[ARG_MOD_PATH], "ui");

          msg("\n=============\n");
          
          //create new q3asm script files
          msg("Creating new compilation scripts..");
          createq3asm(buildGAMEfiles, argv[ARG_MOD_PATH], argv[ARG_Q3_PATH], argv[ARG_MOD_NAME], "game", "qagame");
          createq3asm(buildCGAMEfiles, argv[ARG_MOD_PATH], argv[ARG_Q3_PATH], argv[ARG_MOD_NAME], "cgame", "cgame");
          createq3asm(buildUIfiles, argv[ARG_MOD_PATH], argv[ARG_Q3_PATH], argv[ARG_MOD_NAME], "ui", "ui");

          msg("\n=============\n");
          
          //create a playit.bat to load the mod
          msg("Creating \"playit.bat\"...");
          newstartbatch (argv[ARG_MOD_NAME], argv[ARG_MOD_PATH], argv[ARG_Q3_PATH]);

          msg("\n=============\n");

          //create a compile_<modname>.bat to compile the mod from any directory
          msg(va("Creating \"%s\\compile_%s.bat\"...", basepath, argv[ARG_MOD_NAME]));
          newcompilebatch (basepath, argv[ARG_MOD_NAME], argv[ARG_MOD_PATH]);

          msg("\n=============\n");

          if (localedit)
          {
            msg("Searching g_local.h for GAMEVERSION definition...");
            if (updatedefinition (va("%s\\game\\g_local.h", argv[ARG_MOD_PATH]), "GAMEVERSION", argv[ARG_MOD_NAME]) == 1)
            {
              msg("Updating g_local.h...");
              msg("Done.");
            } else {
              msg("Not found. You will need to manually edit this file.");
              getch();
            }

            msg("\n=============\n");
          }

          if (publicedit)
          {
            msg("Searching bg_public.h for GAME_VERSION definition...");
            if (updatedefinition (va("%s\\game\\bg_public.h", argv[ARG_MOD_PATH]), "GAME_VERSION", va("%s-1",argv[ARG_MOD_NAME])) == 1)
            {
              msg("Updating bg_public.h...");
              msg("Done.");
            } else {
              msg("Not found. You will need to manually edit this file.");
              getch();
            }

            msg("\n=============\n");
          }

          //all done!
          msg("Task complete");
        } else {
          errormsg(ERROR_UNKNOWN); //couldnt create the mod directory?
        }
      }
    }
  }
  
  return 0;
}

//help text that is displayed when improper input is given
void helpmsg (void)
{
  msg("Program syntax:");
  msg("newmod <modname> <sourcepath> <q3path> <options..>\n");
  msg("Example:");
  msg("newmod mymod c:\\mymod C:\\\"Program Files\"\\\"Quake III Arena\"\n");
  msg("This would create a new code set from the current source code in the");
  msg("path \"c:\\mymod\" and add a \"mymod\" directory to your Q3A directory.");
  msg("\nValid Options:");
  msg("overwrite  - allows NewMod to overwrite an existing mod. Not Recommended.");
  msg("localedit  - NewMod will edit the g_local.h file to set GAMEVERSION properly.");
  msg("   If NewMod doesn't make this edit, you must do so by hand. To run your mod,");
  msg("   Q3A requires that GAME_VERSION be set to your mod's \"modname\". Recommended.");
  msg("publicedit - NewMod will edit the bg_public.h file to set GAME_VERSION properly");
  msg("   Changing this variable is required by, and only by, mods which make");
  msg("   changes to the cgame, or ui modules. Not Recommended.");
  msg("\nTo make create a backup:");
  msg("newmod backup");
}

//this function checks the current directory structure for the required source files
//return 1 if invalid source
//return 0 if valid
int badsource (void)
{ //FIXME: a better test would be to check the build*.dat
  if (direxists("game") && 
      direxists("cgame") &&
      direxists("ui"))
    return 0;
  return 1;
}

//lets the user know if an error has occured.
void errormsg (int error)
{
  char *out;
  
  switch (error) {
  case ERROR_BAD_SOURCE:
    out = "The source code is invalid. Please install a fresh copy of the source code and try again.";
    break;
  case ERROR_BAD_MOD_NAME:
    out = "The mod name you have chosen already exists in your Q3A directory. Please choose a different name.";
    break;
  default:
    out = "Unknown error.";
  }

  msg("The following error was encountered:");
  msg(out);
}

/*========
Copies files listed in the filelist to modpath\dir
========*/
void copyfiles(char *filelist, char *path, char *dir)
{
  FILE *input;
  char filename[80];
  FILE *in;
  FILE *out;
  char buffer[512];

  input = fopen(filelist, "r");
  
  if (input == NULL)
  {
    msg(va("Error loading %s!", filelist));
    return;
  }

  while (fgets (filename, sizeof(char) * sizeof(filename), input) != NULL)
  {
    //remove the last character because its a line break
    filename[strlen(filename) - 1] = '\0';

    if (strlen(filename) < 2)
      continue; // a while name has to be at least 2 chars long, prevents crashes

    if (strlen(dir) > 0)
    {
      in = fopen(va("%s\\%s", dir, filename), "r");
      out = fopen(va("%s\\%s\\%s", path, dir, filename), "w+");
    } else {
      in = fopen(filename, "r");
      out = fopen(va("%s\\%s", path, filename), "w+");
    }
    if (in != NULL && out != NULL)
    {
      while (fgets (buffer, sizeof(char) * sizeof(buffer), in) != NULL)
        fputs(buffer, out);

      fclose(in);
      fclose(out);
      msg(va("  %s Copied", filename));
    } else {
      msg(va("  Error copying %s", filename));
    }
  }

  fclose (input);
}

/*========
Creates a dir.q3asm in modpath\dir file that will compile filelist and put the QVM in q3path\modname
========*/
void createq3asm(char *filelist, char *modpath, char *q3path, char *modname, char *dir, char *qvm)
{
  FILE *input;
  FILE *out;
  char buffer[512];

  input = fopen(filelist, "r");

  if (input == NULL)
  {
    msg(va("Error loading %s!", filelist));
    return;
  }

  out = fopen(va("%s\\%s\\%s.q3asm", modpath, dir, dir), "w+");
    
  fputs (va("-o \"%s\\%s\\vm\\%s\"\n", q3path, modname, qvm), out);
  /*==========
  Outputs: -o "q3path\modname\qvm" 
  ==========*/
  
  while (fgets (buffer, sizeof(char) * sizeof(buffer), input) != NULL)
    fputs(buffer, out);

  fclose (out);
  fclose (input);
}

/*========
Creates a playit.bat file that will load the new mod into Q3A
========*/
void newstartbatch (char *modname, char *modpath, char *q3path)
{
  FILE *out;

  out = fopen(va("%s\\playit.bat", modpath), "w+");
    
  fputs("@echo off\n", out);
  fputs(va("%c:\n", q3path[0]), out);
  fputs(va("cd %s\n", q3path), out);
  fputs(va("quake3.exe +set fs_game \"%s\"\n", modname), out);
    
  fclose (out);
}

//newcompilebatch (basepath, argv[ARG_MOD_NAME], argv[ARG_MOD_PATH]);
void newcompilebatch (char *path, char *modname, char *modpath)
{
  FILE *out;

  out = fopen(va("%s\\compile_%s.bat", path, modname), "w+");
    
  fputs("@echo off\n", out);
  fputs(va("compilemod %s compile.bat\n", modpath), out);
    
  fclose (out);
}

int checkargs (int num, char *args[])
{
  int i;

  for (i = 0; i < num; i++)
    strlwr(args[i]);

  if (num == ARG_BACKUP + 1 && strcmp("backup", args[ARG_BACKUP]) == 0)
    return 1;
  if (num >= 4)
    return 1;

  return 0;
}

void dobackup (void)
{
  char path[1024];
  char newpath[1024];
  int subdir;

  //make sure the current directory contains valid source code structure
  if (badsource()) {
    errormsg(ERROR_BAD_SOURCE);
    return;
  }
  
  //get and store the directory
  getbasepath(path, 1024);

  msg("Creating directories...");
  //check to see if the backups directory exists
  if (!subdirexists("backups", path))
  {
    FILE *fp;
    createsubdir("backups", path);
    fp = fopen("backups\\counter.dat", "w+");
    subdir = 1;
    fputs(va("%d\n", subdir), fp);
    fclose(fp);
  } else {
    FILE *fp;
    char buffer[512];

    fp = fopen("backups\\counter.dat", "r");
    if (fp == NULL)
    {
      fp = fopen("backups\\counter.dat", "w+");
      subdir = 1;
      fputs(va("%d\n", subdir), fp);
      fclose(fp);
    }
    else 
    {
      fgets(buffer, sizeof(buffer), fp);
      fclose(fp);
      subdir = atoi(buffer) + 1;

      fp = fopen("backups\\counter.dat", "w");
      fputs(va("%d\n", subdir), fp);
      fclose(fp);
    }
  }

  createdir(va("%s\\backups\\%d", path, subdir));

  memset(newpath, 0, sizeof(newpath));
  strcpy(newpath, va("%s\\backups\\%d", path, subdir));
  
  //create the sub directories
  msg(va("  backups\\%d\\game", subdir));
  createsubdir("game", newpath);
  msg(va("  backups\\%d\\cgame", subdir));
  createsubdir("cgame", newpath);
  msg(va("  backups\\%d\\ui", subdir));
  createsubdir("ui", newpath);

  msg("\n=============\n");

  //copy over the source files
  msg("Copying Project files..");
  copyfiles(copyPROJECTfiles, newpath, "");
  msg("Copying Server files..");
  copyfiles(copyGAMEfiles, newpath, "game");
  msg("Copying Client files..");
  copyfiles(copyCGAMEfiles, newpath, "cgame");
  msg("Copying User Interface files..");
  copyfiles(copyUIfiles, newpath, "ui");

  msg("\n=============\n");

  //all done!
  msg("Task complete");
}