/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "util.h"
#include "bsp.h"
#include "shader.h"
#include "render.h"
#include "tex.h"
#include "lightmap.h"
#include "mesh.h"
#include "skybox.h"
#include "md3.h"
#include "mapent.h"
#include "renderback.h"
#include "glinc.h"
#include <math.h>

/* The back-end of the rendering pipeline does the actual drawing.
 * All triangles which share a rendering state (shader) are pushed together
 * into the 'arrays' structure.  This includes verts, texcoords, and element
 * numbers.  The renderer is then 'flushed': the rendering state is set
 * and the triangles are drawn.  The arrays and rendering state is then
 * cleared for the next set of triangles.
 */

/* FIXME: The manner in which faces are "pushed" to the arrays is really
   absimal.  I'm sure it could be highly optimized. */
/* FIXME: It would be nice to have a consistent view of faces, meshes,
   mapents, etc. so we don't have to have a "push" function for each. */

#define TWOPI 6.28318530718
#define TURB_SCALE 0.2


/* Triangle arrays */
typedef struct
{
    int numverts;
    vec4_t *verts;
    colour_t *colour;
    texcoord_t *tex_st;
    texcoord_t *lm_st;
    int numelems;
    int *elems;
} arrays_t;

static void render_pushface(face_t *face);
static void render_pushmesh(mesh_t *mesh);
static void render_flush(int shader, int lmtex);
static double render_func_eval(uint_t func, float *args);
static int render_setstate(shaderpass_t *pass, uint_t lmtex);
static void render_clearstate(shaderpass_t *pass);
static void render_pushface_deformed(int shadernum, face_t *face);

static arrays_t arrays;

void
render_backend_init(void)
{
    /* FIXME: need to account for extra verts/elems in meshes */
    
    arrays.verts = (vec4_t*)malloc(r_numverts * sizeof(vec4_t));
    arrays.tex_st = (texcoord_t*)malloc(r_numverts * sizeof(texcoord_t));
    arrays.lm_st = (texcoord_t*)malloc(r_numverts * sizeof(texcoord_t));
    arrays.elems = (int*)malloc(r_numelems * sizeof(int));
    arrays.colour = (colour_t*)malloc(r_numverts * sizeof(colour_t));
}

void
render_backend_finalize(void)
{
    free(arrays.verts);
    free(arrays.tex_st);
    free(arrays.lm_st);
    free(arrays.elems);
    free(arrays.colour);
}

void
render_backend(facelist_t *facelist)
{
    int f, shader, lmtex;
    uint_t key;
    face_t *face;
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    arrays.numverts = arrays.numelems = 0;
    key = (uint_t)-1;
    for (f=0; f < facelist->numfaces; ++f)
    {
	face = &r_faces[facelist->faces[f].face];

	/* Look for faces that share rendering state */
	if (facelist->faces[f].sortkey != key)
	{
	    /* Flush the renderer and reset */
	    if (f) render_flush(shader, lmtex);
	    shader = face->shader;
	    lmtex = face->lm_texnum;
	    key = facelist->faces[f].sortkey;
	}

	/* Push the face to the triangle arrays */
	switch (face->facetype)
	{
	    case FACETYPE_NORMAL:
	    case FACETYPE_TRISURF:
		if (r_shaders[shader].flags & SHADER_DEFORMVERTS)
		    render_pushface_deformed(shader, face);
		else
		    render_pushface(face);
		break;
	    case FACETYPE_MESH:
		render_pushmesh(&r_meshes[facelist->faces[f].face]);
		break;
	    default:
		break;
	}
    }
    /* Final flush to clear queue */
    render_flush(shader, lmtex);    
}

void
render_backend_sky(int numsky, int *skylist)
{
    int s, i, shader;
    float skyheight;
    uint_t *elem;

    shader = r_faces[skylist[0]].shader;
    skyheight = r_shaders[shader].skyheight;
    arrays.numverts = arrays.numelems = 0;

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    /* Center skybox on camera to give the illusion of a larger space */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(r_eyepos[0], r_eyepos[1], r_eyepos[2]);
    glScalef(skyheight, skyheight, skyheight);

    /* FIXME: Need to cull skybox based on face list */
    for (s=0; s < 5; s++)
    {
	elem = r_skybox->elems;
	for (i=0; i < r_skybox->numelems; i++)
	{
	    arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
	}
	for (i=0; i < r_skybox->numpoints; i++)
	{
	    vec_copy(r_skybox->points[s][i], arrays.verts[arrays.numverts]);
	    arrays.verts[arrays.numverts][3] = 1.0f;
	    vec2_copy(r_skybox->tex_st[s][i], arrays.tex_st[arrays.numverts]);
	    arrays.numverts++;
	}
    }

    render_flush(shader, 0);

    /* Restore world space */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

void
render_backend_mapent(int mapent)
{
    int i, j, k;
    mapent_inst_t *inst = &g_mapent_inst[mapent];
    mapent_class_t *klass = &g_mapent_class[inst->klass];
    md3model_t *model;
    md3mesh_t *mesh;
    uint_t *elem;
    float funcargs[4];
    float bob;

    arrays.numverts = arrays.numelems = 0;

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    /* Calculate bob amount */
    funcargs[0] = funcargs[2] = 0.0f;
    funcargs[1] = klass->bobheight;
    funcargs[3] = inst->bobspeed;
    bob = (float)render_func_eval(SHADER_FUNC_SIN, funcargs);

    /* Translate to model origin + bob amount */
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glTranslatef(inst->origin[0], inst->origin[1], inst->origin[2] + bob);

    for (i=0; i < klass->numparts; i++)
    {
	model = &r_md3models[klass->parts[i].md3index];

	/* Scale and rotate part */
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glScalef(klass->parts[i].scale, klass->parts[i].scale,
		 klass->parts[i].scale); 
	glRotated(klass->parts[i].rotspeed * g_frametime, 0.0, 0.0, 1.0);
	    
	for (j = 0; j < model->nummeshes; j++)
	{
	    mesh = &model->meshes[j];
	    
	    elem = mesh->elems;
	    for (k = 0; k < mesh->numelems; k++)
	    {
		arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
	    }

	    for (k = 0; k < mesh->numverts; k++)
	    {
		vec_copy(mesh->points[k], arrays.verts[arrays.numverts]);
		arrays.verts[arrays.numverts][3] = 1.0f;
		vec2_copy(mesh->tex_st[k],  arrays.tex_st[arrays.numverts]);
		arrays.numverts++;
	    }

	    /* Draw it */
	    render_flush(mesh->shader, 0);
	}
	/* Restore unrotated state */
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
    }

    /* Restore world space */
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
}

static void
render_pushface(face_t *face)
{
    int i, *elem;
    vertex_t *vert;

    elem = &r_elems[face->firstelem];
    for (i = 0; i < face->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
    
    vert = &r_verts[face->firstvert];
    for (i = 0; i < face->numverts; ++i)
    {
	vec_copy(vert->v_point, arrays.verts[arrays.numverts]);
	arrays.verts[arrays.numverts][3] = 1.0f;
	vec2_copy(vert->tex_st, arrays.tex_st[arrays.numverts]);
	vec2_copy(vert->lm_st, arrays.lm_st[arrays.numverts]);	
	if (r_shaders[face->shader].flags & SHADER_NEEDCOLOURS)
	    colour_copy(vert->colour, arrays.colour[arrays.numverts]);
	vert++;
	arrays.numverts++;
    }	    
}

static void
render_pushface_deformed(int shadernum, face_t *face)
{
    /* Push the face, deforming each vertex as we go. */
    /* FIXME: Better to deform vertexes after pushing, but where
       does the normal info come from ? */
    /* Only wave deformation supported here */
    shader_t *shader = &r_shaders[shadernum];
    float args[4], startoff, off, wavesize, deflect;
    int i, *elem;
    vertex_t *vert;
    vec3_t v;

    /* Setup wave function */
    args[0] = shader->deformv_wavefunc.args[0];
    args[1] = shader->deformv_wavefunc.args[1];
    args[3] = shader->deformv_wavefunc.args[3];
    startoff = shader->deformv_wavefunc.args[2];
    wavesize = shader->deformv_wavesize;

    elem = &r_elems[face->firstelem];
    for (i = 0; i < face->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
        
    vert = &r_verts[face->firstvert];
    for (i = 0; i < face->numverts; ++i)
    {
	/* FIXME: this clearly isn't the way deform waves are applied to
	   world coordinates.  For now, it at least waves the banners :) */
	off = (vert->v_point[0] + vert->v_point[1] + vert->v_point[2]) /
	    wavesize;

	/* Evaluate wave function */
	args[2] = startoff + off;
	deflect = render_func_eval(shader->deformv_wavefunc.func, args);
	/* Deflect vertex along its normal vector by wave amount */
	vec_copy(vert->v_norm, v);
	vec_scale(v, deflect, v);
	vec_add(v, vert->v_point, v);

	/* Push it */
	vec_copy(v, arrays.verts[arrays.numverts]);
	arrays.verts[arrays.numverts][3] = 1.0f;
	vec2_copy(vert->tex_st, arrays.tex_st[arrays.numverts]);
	vec2_copy(vert->lm_st, arrays.lm_st[arrays.numverts]);	
	if (r_shaders[face->shader].flags & SHADER_NEEDCOLOURS)
	    colour_copy(vert->colour, arrays.colour[arrays.numverts]);
	vert++;
	arrays.numverts++;	
    }	    
}

static void
render_pushmesh(mesh_t *mesh)
{
    int u, v, i, *elem;

    elem = mesh->elems;
    for (i = 0; i < mesh->numelems; ++i)
    {
	arrays.elems[arrays.numelems++] = arrays.numverts + *elem++;
    }
    
    i = 0;    
    for (v = 0; v < mesh->size[1]; ++v)
    {
	for (u = 0; u < mesh->size[0]; ++u)
	{
	    vec_copy(mesh->points[i], arrays.verts[arrays.numverts]);
	    arrays.verts[arrays.numverts][3] = 1.0f;
	    vec2_copy(mesh->tex_st[i],  arrays.tex_st[arrays.numverts]);
	    vec2_copy(mesh->lm_st[i], arrays.lm_st[arrays.numverts]);
	    arrays.numverts++;
	    i++;
	}
    }
}

static void
render_stripmine(int numelems, int *elems)
{
    int toggle;
    uint_t a, b, elem;

    /* Vertexes are in tristrip order where possible.  If we can't lock
     * the vertex arrays (glLockArraysEXT), then it's better to send
     * tristrips instead of triangles (less transformations).
     * This function looks for and sends tristrips.
     */

    /* Tristrip order elems look like this:
     *  0 1 2 2 1 3 2 3 4 4 3 5 4 5 7 7 5 6  <-- elems
     *    b a a b b a b a a b b a b a a b b  <-- ab pattern
     *    \ 1 / \ 2 / \ 3 / \ 4 / \ 5 /      <-- baa/bba groups
     */
    
    elem = 0;
    while (elem < numelems)
    {
	toggle = 1;
	glBegin(GL_TRIANGLE_STRIP);
	
	glArrayElement(elems[elem++]);
	b = elems[elem++];
	glArrayElement(b);
	a = elems[elem++];
	glArrayElement(a);
	
	while (elem < numelems)
	{
	    if (a != elems[elem] || b != elems[elem+1])
		break;
	    
	    if (toggle)
	    {
		b = elems[elem+2];
		glArrayElement(b);
	    }
	    else
	    {
		a = elems[elem+2];
		glArrayElement(a);
	    }
	    elem += 3;
	    toggle = !toggle;
	}
	glEnd();
    }
}

static void
render_flush(int shadernum, int lmtex)
{
    int p;
    shader_t *shader = &r_shaders[shadernum];
    
    if (arrays.numverts == 0) return;

    /* Face culling */
    if (shader->flags & SHADER_NOCULL)
	glDisable(GL_CULL_FACE);
    else
	glEnable(GL_CULL_FACE);

    /* FIXME: if compiled vertex arrays supported, lock vertex array here */
    glVertexPointer(4, GL_FLOAT, 0, arrays.verts);
    
    if (shader->flags & SHADER_NEEDCOLOURS)
	glColorPointer(4, GL_UNSIGNED_BYTE, 0, arrays.colour);

    /* FIXME: Multitexturing, if supported...
     * Multitexturing can be handled by examining the number of passes
     * for this shader, and spreading them amongst available texture
     * units.  E.g. if there are 3 passes and 2 tex units, we do 2 at
     * once and then one -- glDrawElements() is called twice.
     */
    
    for (p=0; p < shader->numpasses; p++)
    {
	/* Set rendering state for this pass */
	if (!render_setstate(&shader->pass[p], lmtex))
	    continue;
	
#if 0
	glDrawElements(GL_TRIANGLES, arrays.numelems, GL_UNSIGNED_INT,
		       arrays.elems);
#else
	/* We don't have compiled vertex arrays (locking) so find tristrips */
	render_stripmine(arrays.numelems, arrays.elems);
#endif
	
	/* Clear certain rendering state variables */
	render_clearstate(&shader->pass[p]);
    }
    
    /* Clear arrays */
    arrays.numverts = arrays.numelems = 0; 
}

static double
render_func_eval(uint_t func, float *args)
{
    double x, y;

    /* Evaluate a number of time based periodic functions */
    /* y = args[0] + args[1] * func( (time + arg[3]) * arg[2] ) */
    
    x = (g_frametime + args[2]) * args[3];
    x -= floor(x);

    switch (func)
    {
	case SHADER_FUNC_SIN:
	    y = sin(x * TWOPI);
	    break;
	    
	case SHADER_FUNC_TRIANGLE:
	    if (x < 0.5)
		y = 2.0 * x - 1.0;
	    else
		y = -2.0 * x + 2.0;
	    break;
	    
	case SHADER_FUNC_SQUARE:
	    if (x < 0.5)
		y = 1.0;
	    else
		y = -1.0;
	    break;
	    
	case SHADER_FUNC_SAWTOOTH:
	    y = x;
	    break;
	    
	case SHADER_FUNC_INVERSESAWTOOTH:
	    y = 1.0 - x;
	    break;
    }

    return y * args[1] + args[0];
}

static int
render_setstate(shaderpass_t *pass, uint_t lmtex)
{
    if (pass->flags & SHADER_LIGHTMAP)
    {
	/* Select lightmap texture */
	glTexCoordPointer(2, GL_FLOAT, 0, arrays.lm_st);
	glBindTexture(GL_TEXTURE_2D, r_lightmaptex[lmtex]);
    }
    else if (pass->flags & SHADER_ANIMMAP)
    {
	uint_t texobj;
	int frame;

	/* Animation: get frame for current time */
	frame = (int)(g_frametime * pass->anim_fps) % pass->anim_numframes;
	
	glTexCoordPointer(2, GL_FLOAT, 0, arrays.tex_st);
	texobj = r_textures[pass->anim_frames[frame]];
	if (texobj < 0) return 0;
	glBindTexture(GL_TEXTURE_2D, texobj);
    }
    else
    {
	uint_t texobj;

	/* Set normal texture */
	glTexCoordPointer(2, GL_FLOAT, 0, arrays.tex_st);
	if (pass->texref < 0) return 0;
	texobj = r_textures[pass->texref];
	if (texobj < 0) return 0;
	glBindTexture(GL_TEXTURE_2D, texobj);
    }

    if (pass->flags & SHADER_BLEND)
    {
	glEnable(GL_BLEND);
	glBlendFunc(pass->blendsrc, pass->blenddst);
    }
    else
	glDisable(GL_BLEND);

    if (pass->flags & SHADER_ALPHAFUNC)
    {
	glEnable(GL_ALPHA_TEST);
	glAlphaFunc(pass->alphafunc, pass->alphafuncref);
    }
    else
	glDisable(GL_ALPHA_TEST);
    
    glDepthFunc(pass->depthfunc);
    if (pass->flags & SHADER_DEPTHWRITE)
	glDepthMask(GL_TRUE);
    else
	glDepthMask(GL_FALSE);

    if (pass->rgbgen == SHADER_GEN_IDENTITY)
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
    else if (pass->rgbgen == SHADER_GEN_WAVE)
    {
	float rgb = (float)render_func_eval(pass->rgbgen_func.func,
					    pass->rgbgen_func.args);
	glColor4f(rgb, rgb, rgb, 1.0f);
    }
    else if (pass->rgbgen == SHADER_GEN_VERTEX)
	/* FIXME: I don't think vertex colours are what is meant here */
	glEnableClientState(GL_COLOR_ARRAY);

    if (pass->flags & SHADER_TCMOD)
    {
	/* Save identity texture transform */
	glMatrixMode(GL_TEXTURE);
	glPushMatrix();

	/* Move center of texture to origin */
	glTranslatef(0.5f, 0.5f, 0.0f);
	
	/* FIXME: Is this the right order for these transforms ? */

	if (pass->tcmod & SHADER_TCMOD_ROTATE)
	    glRotated(pass->tcmod_rotate * g_frametime, 0.0, 0.0, 1.0);
	if (pass->tcmod & SHADER_TCMOD_SCALE)
	    glScalef(pass->tcmod_scale[0], pass->tcmod_scale[1], 1.0f);

	if (pass->tcmod & SHADER_TCMOD_TURB)
	{
	    /* Don't know what the exact transform is for turbulance, but
	       this seems to do something close: sin wave scaling in s
	       and t with a 90 degrees phase difference */
	    double x, y1, y2;
	    x = (g_frametime + pass->tcmod_turb[2]) * pass->tcmod_turb[3];
	    x -= floor(x);
	    y1 = sin(x * TWOPI) * pass->tcmod_turb[1] + pass->tcmod_turb[0];
	    y2 = sin((x+0.25) * TWOPI) * pass->tcmod_turb[1] +
		pass->tcmod_turb[0];
	    glScaled(1.0+y1*TURB_SCALE, 1.0+y2*TURB_SCALE, 1.0);
	}

	if (pass->tcmod & SHADER_TCMOD_STRETCH)
	{
	    double y = render_func_eval(pass->tcmod_stretch.func,
					pass->tcmod_stretch.args);
	    glScaled(1.0/y, 1.0/y, 1.0);
	}
	
	if (pass->tcmod & SHADER_TCMOD_SCROLL)
	    glTranslated(pass->tcmod_scroll[0] * g_frametime,
			 pass->tcmod_scroll[1] * g_frametime, 0.0);

	/* Replace center of texture */
	glTranslatef(-0.5f, -0.5f, 0.0f);
    }

    return 1;
}

static void
render_clearstate(shaderpass_t *pass)
{
    if (pass->flags & SHADER_TCMOD)
    {
	/* Revert to identity texture transform */
	glMatrixMode(GL_TEXTURE);
	glPopMatrix();
    }
    if (pass->rgbgen == SHADER_GEN_VERTEX)
	glDisableClientState(GL_COLOR_ARRAY);
}
