/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "util.h"
#include "bsp.h"
#include "shader.h"
#include "mapent.h"
#include "render.h"
#include "renderback.h"
#include "glinc.h"
#include <stdlib.h>
#include <math.h>

/* The front end of the rendering pipeline decides what to draw, and
 * the back end actually draws it.  These functions build a list of faces
 * to draw, sorts it by shader, and sends it to the back end (renderback.c)
 */

#define MAX_TRANSPARENT 1000

enum
{
    CLIP_RIGHT_BIT   = 1,
    CLIP_LEFT_BIT    = 1 << 1,
    CLIP_TOP_BIT     = 1 << 2,
    CLIP_BOTTOM_BIT  = 1 << 3,
    CLIP_FAR_BIT     = 1 << 4,
    CLIP_NEAR_BIT    = 1 << 5
};

static void render_walk_model(int n);
static int classify_point(vec3_t p, int plane_n);
static void render_walk_node(int n, int accept);
static void render_walk_leaf(int n, int accept);
static void render_walk_face(int n);
static void gen_clipmat(void);
static int cliptest_point(vec4_t p);
static int cliptest_bbox(bbox_t bbox);
static int cliptest_bboxf(bboxf_t bbox);
static int find_cluster(vec3_t pos);
static void sort_faces(void);

static mat4_t clipmat;        /* Matrix to go from worldspace to clipspace */
static facelist_t facelist;   /* Faces to be drawn */
static facelist_t translist;  /* Transparent faces to be drawn */
static int r_leafcount;       /* Counts up leafs walked for this scene */
static int *r_faceinc;        /* Flags faces as "included" in the facelist */
static int *skylist;          /* Sky faces hit by walk */
static int numsky;            /* Number of sky faces in list */
static float cos_fov;         /* Cosine of the field of view angle */

void
render_init(void)
{
    int i;
    
    facelist.faces = (rendface_t*)malloc(r_numfaces * sizeof(rendface_t));
    translist.faces = (rendface_t*)malloc(MAX_TRANSPARENT *
					  sizeof(rendface_t));
    r_faceinc = (int*)malloc(r_numfaces * sizeof(int));
    skylist = (int*)malloc(100 * sizeof(int));
    render_backend_init();

    /* Find cluster for each mapent */
    for (i=0; i < g_mapent_numinst; i++)
	g_mapent_inst[i].cluster = find_cluster(g_mapent_inst[i].origin);
}

void
render_finalize(void)
{
    free(facelist.faces);
    free(translist.faces);
    free(r_faceinc);
    free(skylist);
    render_backend_finalize();
}

void
render_scene(void)
{
    int i;
    
    r_leafcount = 0;

    /* Find eye cluster for PVS checks */
    if (!r_lockpvs)
	r_eyecluster = find_cluster(r_eyepos);

    /* Need to enable depth mask before clear */
    glDepthMask(GL_TRUE);
    
    /* FIXME: color buffer clear can be optionally removed */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    /* Set up camera */
    gluLookAt(r_eyepos[0], r_eyepos[1], r_eyepos[2],
	      r_eyepos[0]+r_eyedir[0], r_eyepos[1]+r_eyedir[1],
	      r_eyepos[2]+r_eyedir[2], 0.0, 0.0, 1.0);


#if 0
    /* Useful for exploring face culling effectiveness */
    if (r_lockpvs)
    {
	render_backend(&facelist);
	return;
    }
#endif    

    cos_fov = cos(r_eyefov/2.0f * DEG2RAD);

    /* Get clip coordinate transformation matrix */
    gen_clipmat();

    facelist.numfaces = translist.numfaces = 0;
    numsky = 0;
    /* Clear "included" faces list */
    memset(r_faceinc, 0, r_numfaces * sizeof(int));

    /* "Walk" the BSP tree to generate a list of faces to render */
    /* FIXME: include other models */
    render_walk_model(0);

    /* Sort the face list */
    sort_faces();

    /* FIXME: Reset depth buffer range based on max/min Z in facelist */
    
    /* Draw sky first */
    if (numsky)
	render_backend_sky(numsky, skylist);

    /* Draw normal faces */
    render_backend(&facelist);

    /* Draw visible mapents (md3 models) */
    for (i=0; i < g_mapent_numinst; i++)
    {	
	if (r_eyecluster < 0 ||
	    BSP_TESTVIS(r_eyecluster, g_mapent_inst[i].cluster))
	    render_backend_mapent(i);
    }
    
    /* Draw transparent faces last */
    if (translist.numfaces)
	render_backend(&translist);

#if 0
    /* Enable for speeds reporting (like r_speeds 1) */
    printf("faces: %d, leafs: %d\n", facelist.numfaces + translist.numfaces,
	   r_leafcount);
#endif    
}

static void
render_walk_model(int n)
{
    if (n == 0)
	render_walk_node(0, 0);
    else
	/* FIXME: models > 0 are just leafs ? */
	Error("Models > 0 not supported");
}

static int
classify_point(vec3_t p, int plane_n)
{
    /* Determine which side of plane p is on */
    plane_t *plane = &r_planes[plane_n];

    return (vec_dot(p, plane->vec) < plane->offset ? -1 : 1);
}

static void
render_walk_node(int n, int accept)
{
    node_t *node = &r_nodes[n];

    if (!accept)
    {
	/* Test the node bounding box for intersection with the view volume */
	int clipstate = cliptest_bbox(node->bbox);
	/* If this node is rejected, reject all sub-nodes */
	if (!clipstate) return;
	/* If this node is trivially accepted, accept all sub-nodes */
	if (clipstate == 2) accept = 1;
    }

    /* Classify eye wrt splitting plane */
    if (classify_point(r_eyepos, node->plane) > 0)
    {
	/* In front of plane: render front first, then back */
	if (node->children[0] < 0)
	    render_walk_leaf(-(node->children[0] + 1), accept);
	else
	    render_walk_node(node->children[0], accept);
	if (node->children[1] < 0)
	    render_walk_leaf(-(node->children[1] + 1), accept);
	else
	    render_walk_node(node->children[1], accept);
    }
    else
    {
	/* Behind plane: render back first, then front */
	if (node->children[1] < 0)
	    render_walk_leaf(-(node->children[1] + 1), accept);
	else
	    render_walk_node(node->children[1], accept);
	if (node->children[0] < 0)
	    render_walk_leaf(-(node->children[0] + 1), accept);
	else
	    render_walk_node(node->children[0], accept);	
    }
}

static void
render_walk_leaf(int n, int accept)
{
    leaf_t *leaf = &r_leafs[n];
    int i;

    /* Test visibility before bounding box */
    if (r_eyecluster >= 0)
    {
	if (! BSP_TESTVIS(r_eyecluster, leaf->cluster)) return;
    }
    
    if (!accept)
    {
	if (!cliptest_bbox(leaf->bbox)) return;
    }    

    r_leafcount++;
    
    for (i=0; i < leaf->numfaces; ++i)
    {
	render_walk_face(r_lfaces[i + leaf->firstface]);
    }
}

static void
render_walk_face(int n)
{
    face_t *face = &r_faces[n];

    /* Check if face is already included in the facelist */
    if (r_faceinc[n]) return;
    r_faceinc[n] = 1;

    if (face->facetype == FACETYPE_NORMAL)
    {
	/* Face plane culling */
	/* FIXME: This simple test is clearly not sufficient.
	   Q3A gets a lot more culling at this point. */
	if (vec_dot(face->v_norm, r_eyedir) > cos_fov)
	    return;
    }
    
    else if (face->facetype == FACETYPE_MESH)
    {
	/* Check bounding box for meshes */
	if (!cliptest_bboxf(face->bbox))
	    return;
    }

    /* Check for sky flag */
    if (r_shaders[face->shader].flags & SHADER_SKY)
    {
	/* Push to sky list */
	skylist[numsky++] = n;
    }

    /* Check for transparent */
    else if (r_shaders[face->shader].flags & SHADER_TRANSPARENT)
    {
	translist.faces[translist.numfaces].face = n;
	translist.faces[translist.numfaces++].sortkey = SORTKEY(face);
    }

    /* Normal face */
    else
    {
	/* Push face to facelist */
	facelist.faces[facelist.numfaces].face = n;
	facelist.faces[facelist.numfaces++].sortkey = SORTKEY(face);
    }
}

static void
gen_clipmat(void)
{
    mat4_t modelview[32];
    mat4_t proj[2];

    /* Get the modelview and projection matricies from GL */
    glGetFloatv(GL_MODELVIEW_MATRIX, (GLfloat*)modelview);
    glGetFloatv(GL_PROJECTION_MATRIX, (GLfloat*)proj);

    /* Multiply to get clip coordinate transformation */
    mat4_mmult(proj[0], modelview[0], clipmat);
}

static int
cliptest_point(vec4_t p)
{
    int mask = 0;
    const float cx = p[0];
    const float cy = p[1];
    const float cz = p[2];
    const float cw = p[3];      
    
    if (cx >  cw) mask |= CLIP_RIGHT_BIT;
    if (cx < -cw) mask |= CLIP_LEFT_BIT;
    if (cy >  cw) mask |= CLIP_TOP_BIT;
    if (cy < -cw) mask |= CLIP_BOTTOM_BIT;
    if (cz >  cw) mask |= CLIP_FAR_BIT;
    if (cz < -cw) mask |= CLIP_NEAR_BIT;

    return mask;
}

/* Test bounding box for intersection with view fustrum.
 * Return val:   0 = reject
 *               1 = accept
 *               2 = trivially accept (entirely in fustrum)
 */
static int
cliptest_bboxf(bboxf_t bv)
{
    static int corner_index[8][3] =
    {
	{0, 1, 2}, {3, 1, 2}, {3, 4, 2}, {0, 4, 2},
	{0, 1, 5}, {3, 1, 5}, {3, 4, 5}, {0, 4, 5}
    };

    vec4_t corner[8];
    int clipcode, clip_or, clip_and, clip_in;
    int i;

    /* Check if eye point is contained */
    if (r_eyepos[0] >= bv[0] && r_eyepos[0] <= bv[3] &&
	r_eyepos[1] >= bv[1] && r_eyepos[1] <= bv[4] &&
	r_eyepos[2] >= bv[2] && r_eyepos[2] <= bv[5])
	return 1;
    
    clip_in = clip_or = 0; clip_and = 0xff;
    for (i=0; i < 8; ++i)
    {
	corner[i][0] = bv[corner_index[i][0]];
	corner[i][1] = bv[corner_index[i][1]];
	corner[i][2] = bv[corner_index[i][2]];
	corner[i][3] = 1.0;

	mat4_vmult(clipmat, corner[i], corner[i]);
	clipcode = cliptest_point(corner[i]);
	clip_or |= clipcode;
	clip_and &= clipcode;
	if (!clipcode) clip_in = 1;
    }

    /* Check for trival acceptance/rejection */
    if (clip_and) return 0;
    if (!clip_or) return 2;
    if (clip_in) return 1;   /* At least one corner in view fustrum */

#if 0
    /* FIXME: need something better for this. */
    /* Maybe find maximum radius to each corner */
    {
	/* Normalize coordinates */
	vec3_t center, rad;
	float cw;

	cw = 1.0f/corner[0][3];
	vec_scale(corner[0], cw, corner[0]);
	corner[0][3] = 1.0;
	cw = 1.0f/corner[6][3];
	vec_scale(corner[6], cw, corner[6]);
	corner[6][3] = 1.0;

	/* Check for non-trivial acceptance */
	vec_avg(corner[0], corner[6], center);
	vec_sub(corner[0], center, rad);
	if (sqrt(vec_dot(center, center)) -
	    sqrt(vec_dot(rad, rad)) <= 1.41421356)
	    return 1;
    }
	
    return 0;
#endif
    return 1;
}

static int
cliptest_bbox(bbox_t bbox)
{
    bboxf_t bv;

    bv[0] = (float)bbox[0];
    bv[1] = (float)bbox[1];
    bv[2] = (float)bbox[2];
    bv[3] = (float)bbox[3];
    bv[4] = (float)bbox[4];
    bv[5] = (float)bbox[5];

    return cliptest_bboxf(bv);
}

static int
find_cluster(vec3_t pos)
{
    node_t *node;
    int cluster = -1;
    int leaf = -1;
    
    node = &r_nodes[0];

    /* Find the leaf/cluster containing the given position */
    
    while (1)
    {
	if (classify_point(pos, node->plane) > 0)
	{
	    if (node->children[0] < 0)
	    {
		leaf = -(node->children[0] + 1);
		break;
	    }
	    else
	    {
		node = &r_nodes[node->children[0]];
	    }
	}
	else
	{
	    if (node->children[1] < 0)
	    {
		leaf = -(node->children[1] + 1);
		break;
	    }
	    else
	    {
		node = &r_nodes[node->children[1]];
	    }
	}	    
    }

    if (leaf >= 0)
	cluster = r_leafs[leaf].cluster;
    return cluster;
}

static int
face_cmp(const void *a, const void *b)
{
    return ((rendface_t*)a)->sortkey - ((rendface_t*)b)->sortkey;
}

static void
sort_faces(void)
{
    /* FIXME: expand qsort code here to avoid function calls */
    qsort((void*)facelist.faces, facelist.numfaces, sizeof(rendface_t),
	  face_cmp);
}
