/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <stdio.h>
#include "util.h"
#include "pak.h"
#include "bsp.h"
#include "entity.h"

#define BSPHEADERID  (*(int*)"IBSP")
#define BSPVERSION 46

#define SHADERS_ADD 100  /* Room for additional shaders */

/* BSP lumps in the order they appear in the header */
enum
{
    ENTITIES,
    SHADERREFS,
    PLANES,
    NODES,
    LEAFS,
    LFACES,
    UNKNOWN_LUMP_7,
    MODELS,
    UNKNOWN_LUMP_9,
    UNKNOWN_LUMP_10,
    VERTS,
    ELEMS,
    UNKNOWN_LUMP_13,
    FACES,
    LIGHTMAPS,
    UNKNOWN_LUMP_16,
    VISIBILITY,
    NUM_LUMPS
};

static struct header
{
    int id, ver;
    struct { int fileofs, filelen; } lump[NUM_LUMPS];
} *bspheader;

static byte_t *bspdata;

static int readlump(int lump, void** mem, size_t elem);

#define READLUMP(lump,val) \
      r_num##val = readlump(lump, (void**)&r_##val, sizeof(*(r_##val)))

void
bsp_read(const char *fname)
{
    int len;
    
    if (!pak_open(fname))
	Error("Could not read bsp file %s", fname);
    len = pak_getlen();
    bspdata = (byte_t*)malloc(len);
    pak_read(bspdata, len, 1);
    pak_close();
    
    bspheader = (struct header*)bspdata;
    
    if (bspheader->id != BSPHEADERID)
	Error("Not a bsp file: %s", fname);
    if (bspheader->ver != BSPVERSION)
	Error("Bad bsp file version");

    /* Make additional room for shader refs to be added later */
    r_numshaders = bspheader->lump[SHADERREFS].filelen / sizeof(shaderref_t);
    r_shaderrefs = (shaderref_t*)malloc((r_numshaders + SHADERS_ADD) *
					sizeof(shaderref_t));
    memcpy(r_shaderrefs, bspdata + bspheader->lump[SHADERREFS].fileofs,
	   r_numshaders * sizeof(shaderref_t));
    r_addshaderstart = r_numshaders;
					    
    READLUMP(PLANES, planes);
    READLUMP(NODES, nodes);
    READLUMP(LEAFS, leafs);
    READLUMP(LFACES, lfaces);
    READLUMP(MODELS, models);
    READLUMP(VERTS, verts);
    READLUMP(ELEMS, elems);
    READLUMP(FACES, faces);
    r_lightmapsize = readlump(LIGHTMAPS, (void**)&r_lightmapdata, 1);
    (void)readlump(VISIBILITY, (void**)&r_visibility, 1);

    entity_parse(bspheader->lump[ENTITIES].filelen,
		 (char*)(bspdata + bspheader->lump[ENTITIES].fileofs));
    
    free(bspdata);
}

void
bsp_list(void)
{
    printf("Contents of BSP file:\n\n");
    printf("num entities     %d\n", g_numentities);
    printf("num models       %d\n", r_nummodels);
    printf("num shaders      %d\n", r_numshaders);
    printf("num planes       %d\n", r_numplanes);
    printf("num verts        %d\n", r_numverts);
    printf("num vertex elems %d\n", r_numelems);
    printf("num leafs        %d\n", r_numleafs);
    printf("num nodes        %d\n", r_numnodes);
    printf("num faces        %d\n", r_numfaces);    
    printf("num lfaces       %d\n", r_numlfaces);
    printf("vis. clusters    %d\n", r_visibility->numclusters);
    
};    

void
bsp_free(void)
{
    free(r_models);
    free(r_shaderrefs);
    free(r_verts);
    free(r_planes);
    free(r_leafs);
    free(r_nodes);
    free(r_faces);
    free(r_lfaces);
    free(r_elems);
    free(r_visibility);

    entity_free();
}

/* Add shader to the shaderref list (from md3 loads) */
int
bsp_addshaderref(const char *shadername)
{
    int i;
    static int calls = 0;

    /* Check for shader already in the list */
    for (i=0; i < r_numshaders; i++)
    {
	if (!strcmp(r_shaderrefs[i].name, shadername))
	    return i;
    }
    
    if (++calls > SHADERS_ADD)
	Error("Too many additional shaders");
        
    strcpy(r_shaderrefs[r_numshaders++].name, shadername);    
    return r_numshaders-1;
}

static int
readlump(int lump, void** mem, size_t elem)
{
    int len = bspheader->lump[lump].filelen;
    int num = len / elem;
    *mem = malloc(len);

    memcpy(*mem, bspdata + bspheader->lump[lump].fileofs, num * elem);
    return num;
}
