/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "util.h"
#include "pak.h"
#include "bsp.h"
#include "render.h"
#include "tex.h"
#include "lightmap.h"
#include "shader.h"
#include "mesh.h"
#include "skybox.h"
#include "entity.h"
#include "mapent.h"
#include "uicommon.h"
#include "glinc.h"
#include <math.h>
#include <stdio.h>

#define AZ_SCALE 0.3
#define EL_SCALE 0.3
#define MOVE_SPEED 400.0

static int ox, oy;
static int move = 0, strafe = 0;
static double fpstime;
static int fpsframes;
static int invert_mouse = 0;

static void
find_start_pos(void)
{
    /* Find the first spawn point in the entities list */
    int i;
    const char *cname;
    
    r_eye_el = r_eye_az = 0.0;
    r_eyepos[0] = r_eyepos[1] = r_eyepos[2] = 0.0;
    for (i = 0; i < g_numentities; i++)
    {
	cname = entity_value(i, "classname");
	if (!strcmp(cname, "info_player_deathmatch"))
	{
	    r_eye_az = entity_float(i, "angle");
	    entity_vec3(i, "origin", r_eyepos);
	    break;
	}
    }
}

static void
move_eye(double intervaltime)
{
    vec3_t delta;

    if (move)
    {
	vec_copy(r_eyedir, delta);
	vec_scale(delta, move * MOVE_SPEED * intervaltime, delta);
	vec_add(r_eyepos, delta, r_eyepos);
    }
    
    if (strafe)
    {
	float norm = sqrt(r_eyedir[0]*r_eyedir[0] + r_eyedir[1]*r_eyedir[1]);
	delta[0] = r_eyedir[1] / norm;
	delta[1] = -r_eyedir[0] / norm;
	delta[2] = 0.0f;
	vec_scale(delta, strafe * MOVE_SPEED * intervaltime, delta);
	vec_add(r_eyepos, delta, r_eyepos);	
    }
}

void
ui_init(int argc, char **argv)
{
    /* My GL driver (linux/TNT2) has problems with glEnableClientState(),
       but this seems to clear it up.  Go figure. */
    glFinish();
    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    g_frametime = 0;
    fpstime = 0;
    fpsframes = 0;

    /* Some defaults */
    r_eyefov = 90.0;
    r_lodbias = 1;
    r_gamma = 1.7;

    pak_openpak(argc > 2 ? argv[2] : "demoq3/pak0.pk3");
    bsp_read(argc > 1 ? argv[1] : "maps/q3dm7.bsp");

    mesh_create_all();
    skybox_create();
    mapent_loadall();
    shader_readall();
    render_init();
    lightmap_loadobjs();
    tex_loadobjs();

    find_start_pos();
    vec_point(r_eyedir, r_eye_az, r_eye_el);

    glDisable(GL_DITHER);
    glShadeModel(GL_SMOOTH);
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glColor3f(1.0, 1.0, 1.0);

    glEnable(GL_DEPTH_TEST);
    glCullFace(GL_FRONT);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    glViewport(0, 0, VIEWPORT_W, VIEWPORT_H);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(r_eyefov, (float)VIEWPORT_W/(float)VIEWPORT_H,
		   10.0, 3000.0);
    glMatrixMode(GL_MODELVIEW);
}

void
ui_display(double time)
{
    double intervaltime;

    intervaltime = time - g_frametime;
    g_frametime = time;

    /* FPS counter */
    fpsframes++;
    if (g_frametime - fpstime > 1.0)
    {
	printf("FPS: %f\n", fpsframes / (g_frametime - fpstime));
	fpstime = g_frametime;
	fpsframes = 0;
    }

    move_eye(intervaltime);
    
    render_scene();
    glFlush();
}

void
ui_mouse_down(int x, int y)
{
    ox = x; oy = y;
}

void
ui_mouse_motion(int x, int y)
{
    r_eye_az += -(x - ox) * AZ_SCALE;
    r_eye_el += -(y - oy) * EL_SCALE * (invert_mouse ? -1.0f : 1.0f);

    vec_point(r_eyedir, r_eye_az, r_eye_el);
    
    ox = x; oy = y;    
}

void
ui_key_down(unsigned char key)
{
    switch (key)
    {
	case 'w':
	    move = 1;
	    break;
	case 's':
	    move = -1;
	    break;
	case 'a':
	    strafe = -1;
	    break;
	case 'd':
	    strafe = 1;
	    break;
	case 'l':
	    r_lockpvs = !r_lockpvs;
	    if (r_lockpvs) printf("cluster: %d\n", r_eyecluster);
	    break;
	case 'i':
	    invert_mouse = !invert_mouse;
	    printf("invert_mouse = %d\n", invert_mouse);
	    break;
    }
}

void
ui_key_up(unsigned char key)
{
    switch (key)
    {
	case 'w':
	case 's':
	    move = 0;
	    break;

	case 'a':
	case 'd':
	    strafe = 0;
	    break;
    }
}

void
ui_finish(void)
{
    shader_freeall();
    mesh_free_all();
    skybox_free();
    mapent_freeall();
    lightmap_freeobjs();
    tex_freeobjs();
    render_finalize();
    pak_closepak();
}
