/* Aftershock 3D rendering engine
 * Copyright (C) 1999 Stephen C. Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "util.h"
#include "pak.h"
#include "unzip.h"
#include <string.h>

/*
 * These are facilities to manipulate pk3 files.  It uses Gilles Vollant's
 * unzip extensions to zlib (source included with distribution).
 */

static unzFile pakfile = NULL;

void
pak_openpak(const char *path)
{
    pakfile = unzOpen(path);
    if (!pakfile)
	Error("Failed to open pk3 file %s\n", path);
}

void
pak_closepak(void)
{
    if (pakfile)
	unzClose(pakfile);
    pakfile = NULL;
}

int
pak_open(const char *path)
{
   if (unzLocateFile(pakfile, path, 2) != UNZ_OK)
       return 0;
   if (unzOpenCurrentFile(pakfile) != UNZ_OK)
       return 0;
   return 1;
}

void
pak_close(void)
{
    unzCloseCurrentFile(pakfile);
}

int
pak_read(void *buf, uint_t size, uint_t nmemb)
{
    int len;

    len = unzReadCurrentFile(pakfile, buf, size * nmemb);
    return len / size;
}

uint_t
pak_getlen(void)
{
    unz_file_info info;

    if (unzGetCurrentFileInfo(pakfile, &info, NULL, 0, NULL, 0, NULL, 0)
	!= UNZ_OK)
	return 0;
    return info.uncompressed_size;
}

uint_t
pak_readfile(const char *path, uint_t bufsize, byte_t *buf)
{
    uint_t len;
    
    if (!pak_open(path))
	return 0;

    if ((len = pak_getlen()) != 0)
    {
	if (pak_read((void*)buf, 1, len) != len)
	    len = 0;
    }
    pak_close();    
    return len;
}

uint_t
pak_listshaders(uint_t bufsize, char *buf)
{
    uint_t num = 0, len;
    int status;
    char *end = buf + bufsize;

    status = unzGoToFirstFile(pakfile);
    while (status == UNZ_OK)
    {
	unzGetCurrentFileInfo(pakfile, NULL, buf, 64, NULL, 0, NULL, 0);
	len = strlen(buf);
	if (len >= 7 && strcmp(&buf[len-7], ".shader") == 0)
	{
	    num++;
	    buf += len + 1;
	    if (buf > end)
		Error("pak_listshaders(): buffer overrun");
	}
	status = unzGoToNextFile(pakfile);
    }

    return num;
}

int
pak_checkfile(const char *path)
{
    int status;

    status = unzLocateFile(pakfile, path, 2);
    return (status == UNZ_OK);
}
