/*
	This program patches and unpatches an fxt file.  It allows messages to be
   added, removed, and changed, with the exception of the very first message
   in the file.  The exception exists because I have not yet figured out
   completely the algorithm used to encrypt the first 8 characters in the
   file.  The program will generate an undo patch which will completely
   reverse any changes made by the original patch if required.

   The program accepts the following parameters:

   1. FXT Filename, including path and extension		(ALWAYS IN POSITION 1)
   2. Patch filename, including patch and extension		(-patch)
   3. Output filename, including path and extension		(-out)
   4. Output filename, including patch and extension	(-undo)
	  for the undo patch
   5. Verbose mode or quiet mode.				   		(-v || -q)
   6. Decrypt the file at the first argument.  If no	(-decrypt)
	  output file is specified then the output is
	  written to the original file.
   7. Encrypt the file at the first argument.  If no	(-encrypt)
	  output file is specified then the output is
	  written to the original file.
   8. Generate a patch from from two complete FXT files	(-generate)
	  The original FXT file is assumed to be argument 1
	  and the new FXT file is assumed to be argument 2.
	  The output file must be specified using the -out
	  switch.  An undo file may optionally be produced
	  using the -undo flag.

   If parameter 3 is not specified then the original FXT file is written to
   instead.
   If parameter 4 is not specified then no undo patch will be created.
   Parameter 5 will default to -q if not specified.

   Here are the command lines for a number of common operations:

   Decrypt an FXT file and write the decrypted version to english.txt
   fxtpatch english.fxt -out english.txt -decrypt

   Encrypt a TXT file and write the encrypted version to english.fxt
   fxtpatch english.txt -out english.fxt -encrypt

   Apply a patch file to english.fxt and generate an undo patch as well
   fxtpatch english.fxt -patch patch.txt -undo revert.txt

   Warnings :
   The encrypt and decrypt commands only use the -out parameter if specfied.  If
   a -out parameter is not specified then the output will be written to the
   original file.

   The program works in the following manner:

   1. The FXT file specified is loaded.  All entries are loaded from the file
	   and placed within a linked list of mesage structures.  This message
	   structure essentially has the id of the message, and then the message
	   itself.

   2. The patch file is loaded and placed in another linked list with the same
	   structure as above.

   3. The patch list is walked and each entry in the patch file has the
	   appropriate action performed on it.  If the message structure indicates
	   that the message is to be removed then the message is taken out of the
	   main FXT file linked list.  If a message ID is not found within the linked
	   list then that message is appended to the main FXT file linked list.  If
	   the message ID is found then the message within the main FXT linked list
	   will be changed accordingly.

   4.  The main linked list is then walked and the output file (and an undo
	   file if requested) is written.

   The format of the patch file should be as follows:

   [id 1]Message 1
   [id 2]Message 2

   If a message is to be removed then only the id of the message should be
   specfied.  The message text should be all on one line, terminated by a
   carriage return. e.g.
   [id 3]

Ver	Date	  	By	Description
---	-----------	--- ---------------------------------------------------------
0.0	18-Jan-1998	DML Started version 1.0 of the program.
0.0	15-Mar-1998	DML Finally got around to doing some more after taking a day
					off work to do it (well, actually just didn't go in on
					Sunday).
0.0	02-May-1998	DML	Hey, a whole weekend off.  This is my chance to finally
					complete this damn program.  Added a whole load of linked
					list handling functions
0.0	03-May-1998	DML	It nearly works.  Seems to garble the input files though.
					Maybe I shouldn't try and hold the entire file in memory.
					I'll look at this after I've got some sleep, but all the
					basic code is no in place.
1.0	04-May-1998	DML	Finished it.  Pretty much all pointers are far pointers now.
					It helps to make it all work.  Added the encrypt and decrypt
					capability as well as tuning all of the linked list handling
					functions.  Removed all of the obsolete code as well.  The
					input files are now loaded a line at a time rather than the
					whole thing in one go.  Helps to keep the memory usage down.
					Now all I have to do is write the documentation. :)
1.1	06-May-1998	DML	Added the -generate command line parameter to create patch
					files.  Encrypted and Decrypted files are now automatically
					detected.  The prevents multiple decryption/encryption
					problems.
1.2	08-May-1998	DML	Fixed a bug which actually caused the generated fxt files to
					fail to load.  Guess who never did their testing properly. :)
					It is now possible to encrypt and decrypt all language files.
					You can also add messages with IDs < 1001 now, and they will
					be added to the file properly.  Thanks very much to
					Michael Mendelsohn for finally giving me the rest of the
					puzzle for the encryption and decryption.

Notes:	If anyone has any comments on the coding style/algorythms used within
		this code I would like to hear them.  You can contact me at
			Darren@briarlea.softnet.co.uk
		I hope this patch works as well for you as it does for me. :)
*/
#include <stdio.h>
#include <alloc.h>
#include <stdlib.h>
#include <string.h>
#include "fxtpatch.h"

#define FXT_FILE	1
#define TXT_FILE	0
#define TRUE		1
#define FALSE		0
#define FILE_BUFFER_SIZE	1000
#define DEBUG		if (giDebug)

int		giDebug = 0;
char	acDecrypt[8] = {0x63,0xC6,0x8C,0x18,0x30,0x60,0xC0,0x80};

/*---------------------------------------------------------------------------*\
	Main Program
\*---------------------------------------------------------------------------*/
int main( int argc, char **argv )
{
	ST_LIST_HEADER	stFXTMessages;
	ST_LIST_HEADER	stPatchMessages;
	ST_LIST_HEADER	stUndoMessages;
	ST_PROGRAM_ARGS	stProgramArgs;

	//	Initialise all of the list headers
	InitListHeader( &stFXTMessages );
	InitListHeader( &stPatchMessages );
	InitListHeader( &stUndoMessages );

	//   Verify that the program has the correct number of parameters, and make
	//   sure that the program arguments structure is filled correctly
	ParseCommandLine( argc, argv, &stProgramArgs );

	if ( stProgramArgs.iDecrypt )
	{
		//	The user wishes to decrypt the encrypted file.  Make it so.
		if ( stProgramArgs.iVerboseMode )
			printf("Loading <%s> and decrypting it\n", argv[1]);
		LoadFile( argv[1], &stFXTMessages );

		DEBUG WalkList( &stFXTMessages );

		if ( stProgramArgs.iVerboseMode )
			printf("Writing decrypted file to <%s>\n", argv[stProgramArgs.iOutputFile]);
		WriteFile( argv[stProgramArgs.iOutputFile],
				   &stFXTMessages,
				   TXT_FILE );
	}
	else if ( stProgramArgs.iEncrypt )
	{
		//	The user wishes to encrypt the TXT file.  Make it happen.
		if ( stProgramArgs.iVerboseMode )
			printf("Loading <%s> and encrypting it\n", argv[1]);
		LoadFile( argv[1], &stFXTMessages );

		DEBUG WalkList( &stFXTMessages );

		if ( stProgramArgs.iVerboseMode )
			printf("Writing encrypted file to <%s>\n", argv[stProgramArgs.iOutputFile]);
		WriteFile( argv[stProgramArgs.iOutputFile],
				   &stFXTMessages,
				   FXT_FILE );
	}
	else if ( stProgramArgs.iGenerate )
	{
		//	The user wishes to generate a patch file from the FXT file, and,
		//	from the patch file passed in.  The file generated will be the patch
		//	file required to make the FXT file (parameter 1) into the patch file
		//	specified.  An undo file will optionally be produced.  The output
		//	file will be a text patch file.

		//	If no output file is specified then complain.
		if ( stProgramArgs.iOutputFile == 1 )
		{
			fprintf(stderr,"You have not specified an output file for the new patch.\n");
			exit( 1 );
		}
		//	If no patch file is specified then complain.
		if ( stProgramArgs.iPatchFile == 0 )
		{
			fprintf(stderr,"You must specify another FXT file in order to generate a patch file.\n");
			exit( 1 );
		}

		//	Load the original FXT file
		if ( stProgramArgs.iVerboseMode )
			printf("Loading FXT file <%s> and decrypting it\n", argv[1]);
		LoadFile( argv[1], &stFXTMessages );

		//  Load the patch file (another FXT file) into memory.
		if ( stProgramArgs.iVerboseMode )
			printf("Loading patch file into memory\n");
		LoadFile( argv[stProgramArgs.iPatchFile], &stPatchMessages );

		//  Apply the patch file to the linked list of FXT message structures
		if ( stProgramArgs.iVerboseMode )
			printf("Generating patch file from original and new FXT files\n");
		GeneratePatch( &stFXTMessages,
					   &stPatchMessages,
					   &stUndoMessages,
					   stProgramArgs.iVerboseMode );

		if ( stProgramArgs.iVerboseMode )
			printf("Patch generated.  Writing output files.\n");

		//  Write the newly encrypted FXT file to the output file specified, or,
		//	reopen the input FXT file for write.
		WriteFile( argv[stProgramArgs.iOutputFile],
				   &stPatchMessages,
				   TXT_FILE );

		//	Check to see if an undo patch file needs to be written.  If so,
		//	write it out
		if ( stProgramArgs.iUndoFile > 0 )
		{
			//	Write the undo patch file out to disk
			WriteFile( argv[stProgramArgs.iUndoFile],
					   &stUndoMessages,
					   TXT_FILE );
		}
	}
	else
	{
		if ( stProgramArgs.iPatchFile == 0 )
		{
			fprintf(stderr,"No patch file specified.  Use -patch <filename>\n");
			exit( 1 );
		}

		//	Load the FXT file
		if ( stProgramArgs.iVerboseMode )
			printf("Loading FXT file <%s> and decrypting it\n", argv[1]);
		LoadFile( argv[1], &stFXTMessages );

		//  Load the patch file into memory.  This method also ensures that the
		//	patch file is closed.
		if ( stProgramArgs.iVerboseMode )
			printf("Loading patch file into memory\n");
		LoadFile( argv[stProgramArgs.iPatchFile], &stPatchMessages );

		//  Apply the patch file to the linked list of FXT message structures
		if ( stProgramArgs.iVerboseMode )
			printf("Applying patch file to messages from original FXT file\n");

		ApplyPatch( &stFXTMessages,
					&stPatchMessages,
					&stUndoMessages,
					stProgramArgs.iVerboseMode );

		if ( stProgramArgs.iVerboseMode )
			printf("Patch applied to FXT file.  Writing output files.\n");

		//  Write the newly encrypted FXT file to the output file specified, or,
		//	reopen the input FXT file for write.
		WriteFile( argv[stProgramArgs.iOutputFile],
				   &stFXTMessages,
				   FXT_FILE );

		//	Check to see if an undo patch file needs to be written.  If so,
		//	write it out
		if ( stProgramArgs.iUndoFile > 0 )
		{
			//	Write the undo patch file out to disk
			WriteFile( argv[stProgramArgs.iUndoFile],
					   &stUndoMessages,
					   TXT_FILE );
		}
	}
	//	Free all of the linked lists in memory
	DestroyList( &stFXTMessages );
	DestroyList( &stPatchMessages );
	DestroyList( &stUndoMessages );

	//	Program finished.  Return success to the shell.
	return 0;
}

/*---------------------------------------------------------------------------*\
	This function parses the command line parameters specfied to the patch
	program and makes sure that they are all valid.  Any invalid parameters
	are ignored.
\*---------------------------------------------------------------------------*/
void ParseCommandLine( int iNumberOfArgs,
					   char **pszProgramArgs,
					   ST_PROGRAM_ARGS *pstProgramArgs )
{
	int iLoop = 0;

	printf("FXTPatch version 1.1 by Darren Latham.\n\n");

	//	Specify all of the default values for the structure
	pstProgramArgs->iVerboseMode = FALSE;	//	Verbose mode default to off
	pstProgramArgs->iOutputFile = 1;		//	Output file defaults to input FXT
	pstProgramArgs->iPatchFile = 0;			//	No patch file by default
	pstProgramArgs->iUndoFile = FALSE;		//	No undo file by default
	pstProgramArgs->iEncrypt = FALSE;		//	Default behaviour is to apply patch
	pstProgramArgs->iDecrypt = FALSE;		//	Default behaviour is to apply patch
	pstProgramArgs->iGenerate = FALSE;		//	Default is to not generate patch file

	//	If the number of command line arguments is incorrect then exit the program
	//   with a relevant message
	if ( iNumberOfArgs < 3 )
	{
		fprintf(stderr,"Incorrect usage.  Please use\n");
		fprintf(stderr,"FXTPATCH <fxt filename> -patch <patch filename> -out <output filename> [-undo <undo patch filename>] [<-v || -q>] [ENTER]\n");
		fprintf(stderr,"-v : verbose mode, -q : quiet mode (default)\n\n");
		fprintf(stderr,"-generate will generate patch files from two complete FXT files.\n");
		fprintf(stderr,"use : FXTPATCH <original FXT> -patch <new FXT file> -out <output file name> [-undo <undo filename>]  [<-v || -q>] [ENTER]\n");
		fprintf(stderr,"[] indicate optional parameters.\n");
		fprintf(stderr,"To encrypt/decrpyt FXT files use -encrypt and -decrypt flags.\n");
		fprintf(stderr,"esu : FXTPATCH <fxt filename> [-out <output file>] -encrypt || -decrypt.\n");
		fprintf(stderr,"Version 1.1, by Darren Latham\n\ne.g.\n");
		fprintf(stderr,"FXTPATCH ENGLISH.FXT CARS.TXT -undo UNDOCARS.TXT -v\n\n");
		exit( 1 );
	}

	//  Determine whether the program is in apply/remove and/or verbose or quiet
	//  mode.  This is cheating outrageously as multiple parameters are
	//  supported with all invalid parameters being ignored.
	iLoop = 2;

	while ( iLoop < iNumberOfArgs )
	{
		if ( strcmp( pszProgramArgs[iLoop], "-q" ) == 0 )
			pstProgramArgs->iVerboseMode = FALSE;

		if ( strcmp( pszProgramArgs[iLoop], "-v" ) == 0 )
			pstProgramArgs->iVerboseMode = TRUE;

		if ( strcmp( pszProgramArgs[iLoop], "-out" ) == 0 )
		{
			//	Output filename has been specified so move iLoop on by one in
			//	the list of parameters and store the position
			iLoop++;
			pstProgramArgs->iOutputFile = iLoop;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-patch" ) == 0 )
		{
			//	Patch filename has been specified so move iLoop on by one in
			//	the list of parameters and store the position
			iLoop++;
			pstProgramArgs->iPatchFile = iLoop;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-undo" ) == 0 )
		{
			//	Undo filename has been specified so move iLoop on by one in
			//	the list of parameters and store the position
			iLoop++;
			pstProgramArgs->iUndoFile = iLoop;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-debug" ) == 0 )
		{
			giDebug = 1;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-encrypt" ) == 0 )
		{
			pstProgramArgs->iEncrypt = TRUE;
			pstProgramArgs->iDecrypt = FALSE;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-decrypt" ) == 0 )
		{
			pstProgramArgs->iEncrypt = FALSE;
			pstProgramArgs->iDecrypt = TRUE;
		}

		if ( strcmp( pszProgramArgs[iLoop], "-generate" ) == 0 )
		{
			pstProgramArgs->iGenerate = TRUE;
		}

		//	Move iLoop to the next command line paramter
		iLoop++;
	}
}

/*---------------------------------------------------------------------------*\
	This function writes the area of memory passed to the function to the
	file specfied.
\*---------------------------------------------------------------------------*/
void WriteFile( char *szFilename, ST_LIST_HEADER *pstHeader, int iFXTFile )
{
	FILE	*fOut = NULL;
	int		iLoop;
	int		iDecrypt = 0;
	char	acBuffer[1000];

	//	If there are no items in the list passed to the function simply return
	if (pstHeader->iItems == 0)		return;

	//	Attempt to open the input file and report an error if it is not opened
	fOut = fopen( szFilename, "w" );
	if ( fOut == NULL )
	{
		fprintf(stderr,"Unable to open the file <%s>\n",szFilename);
		exit( 1 );
	}

	//	Write all of the messages held in the list of messages to a file.
	if ( iFXTFile )
	{
		//	For each message in the output file write it to a buffer,
		//	encrypt it, and, then write the buffer out to the output file.
		for ( pstHeader->pstCurrent = pstHeader->pstFirst;
				pstHeader->pstCurrent != NULL;
					pstHeader->pstCurrent = pstHeader->pstCurrent->pstNext )
		{
			//	Write the message to the buffer
			WriteMessageToBuffer( acBuffer,
								  pstHeader->pstCurrent->pstMessage->szID,
								  pstHeader->pstCurrent->pstMessage->szMessage );

			for (iLoop = 0; iLoop < strlen( acBuffer ); iLoop++)
			{
				if ( iDecrypt < 8 )
				{
					acBuffer[iLoop] += acDecrypt[iDecrypt];
					iDecrypt++;
				}

				acBuffer[iLoop]++;

			}

			//	Write the whole thing to the output file with a terminator and
			//	the opening and closing braces around the message ID.
			if ( pstHeader->pstCurrent->pstNext == NULL )
			{
				//	Never write the terminating zero for the last message as
				//	it will cause the file to fail to load.
				fprintf(fOut,"%s", acBuffer);
			}
			else
			{
				fprintf(fOut,"%s%c", acBuffer, 1);
			}
		}
	}
	else
	{
		//	Write unencrypted data to the output file
		for ( pstHeader->pstCurrent = pstHeader->pstFirst;
				pstHeader->pstCurrent != NULL;
					pstHeader->pstCurrent = pstHeader->pstCurrent->pstNext )
		{
			fprintf(fOut,"[%s]%s\n", pstHeader->pstCurrent->pstMessage->szID,
									 pstHeader->pstCurrent->pstMessage->szMessage);
		}
	}

	//	Close the file
	fclose( fOut );
}

/*---------------------------------------------------------------------------*\
	This function applies the patch linked list of messages to the main linked
	list of messages.  The patch is applied in the order the messages to patch
	are contained within the patch list.  The function works as follows:
	For each patch message in the patch list
		Find message in main list by match IDs.

		If message not found and the patch message length > 0 then
			Add the patch message to the end of the main list
			Add a zero length message to the Undo list with the same ID

		If Message found and patch message length = 0 then
			Add the message found to the patch list.
			Remove the message found from the main list

		If Message found and patch message length > 0 then
			Add the message found to the patch list
			Remove the message found from the main list
			Add the patch message to the main list
	End for

\*---------------------------------------------------------------------------*/
void ApplyPatch( ST_LIST_HEADER *pstMainList,
				 ST_LIST_HEADER *pstPatchList,
				 ST_LIST_HEADER *pstUndoList,
				 int iVerbose )
{
	ST_MESSAGE	*pstMessage;

	for ( pstPatchList->pstCurrent = pstPatchList->pstFirst;
			pstPatchList->pstCurrent != NULL;
				pstPatchList->pstCurrent = pstPatchList->pstCurrent->pstNext )
	{
		//	Locate any message in the main list which has a match ID as the
		//	one passed into the method.  This method sets the pstCurrent pointer
		//	of the header to the matching list item.
		FindMatchingMessage( pstMainList, pstPatchList->pstCurrent->pstMessage );

		if ( _fstrlen(pstPatchList->pstCurrent->pstMessage->szMessage) > 0 )
		{
			if ( pstMainList->pstCurrent == NULL )
			{
				if (iVerbose)
					printf("Adding message [%s].\n",
								pstPatchList->pstCurrent->pstMessage->szID );
				//	The message in the patch list was not found in the main list,
				//	and it contains valid text.  Add the message to the main list,
				//	and an inverse operation to the undo list.

				//	Create the undo message first
				pstMessage = CreateDeleteMessage(
										pstPatchList->pstCurrent->pstMessage );
				AppendMessageToList( pstUndoList, pstMessage );

				//	Now create the message to add to the main list and then add
				//	it order by its ID.  NOTE: Messages with zero length IDs are
				//	always added at the end.
				pstMessage = CreateDuplicateMessage(
										pstPatchList->pstCurrent->pstMessage );
				AddMessageInOrderToList( pstMainList, pstMessage );
			}
			else
			{
				//	The message has been specified in both the patch list and
				//	the main list.  If the messages are different then the
				//	message in the main list needs to be added to the undo
				//	list, and, the message in the patch list needs to be added
				//	to the message in the main list
				if ( _fstrcmp( pstMainList->pstCurrent->pstMessage->szMessage,
							 pstPatchList->pstCurrent->pstMessage->szMessage ) )
				{
					if (iVerbose)
						printf("Changing message [%s].\n",
								pstPatchList->pstCurrent->pstMessage->szID );
					//	Create the duplicate message from the main list first
					//	And add it to the undo list
					pstMessage = CreateDuplicateMessage(
										pstMainList->pstCurrent->pstMessage );
					AppendMessageToList( pstUndoList, pstMessage );

					//	Free the message from the main list and add the new one
					pstMessage = CreateDuplicateMessage(
										pstPatchList->pstCurrent->pstMessage );
					FreeMessage( pstMainList->pstCurrent->pstMessage );
					pstMainList->pstCurrent->pstMessage = pstMessage;
				}
			}
		}
		else
		{
			if ( pstMainList->pstCurrent != NULL )
			{
				//	We have found a message with a match ID, and the patch is to
				//  remove it.  Add the message from the main list to the undo
				//	list, and then remove the Item from the main list.
				if (iVerbose)
					printf("Removing message [%s].\n",
								pstMainList->pstCurrent->pstMessage->szID );
				pstMessage = CreateDuplicateMessage(
										pstMainList->pstCurrent->pstMessage );
				AppendMessageToList( pstUndoList, pstMessage );

				//	Remove the current item from the main list
				RemoveCurrentItemFromList( pstMainList );
			}
		}
	}
}

/*---------------------------------------------------------------------------*\
	This function initialises a list header.  It simply sets all pointers
	to NULL and sets the number of items in the list to 0
\*---------------------------------------------------------------------------*/
void InitListHeader( ST_LIST_HEADER *pstHeader )
{
	pstHeader->pstFirst = NULL;
	pstHeader->pstLast = NULL;
	pstHeader->pstCurrent = NULL;
	pstHeader->iItems = 0;
}

/*---------------------------------------------------------------------------*\
	This function appends a new message to the end of the linked list.
\*---------------------------------------------------------------------------*/
void AppendMessageToList(	ST_LIST_HEADER *pstHeader, ST_MESSAGE *pstMessage )
{
	ST_LIST_ITEM far *pstNewItem = NULL;

	//	Allocate space for the new list item.
	pstNewItem = CreateListItem();

	//	Assign the message to the new list item, and then set the next pointer
	//	to be null as we are adding the new list item to the end of the list.
	pstNewItem->pstMessage = pstMessage;
	pstNewItem->pstNext = NULL;

	//	Handle the special case where the list does not contain any items
	if ( pstHeader->pstFirst == NULL )
	{
		pstHeader->pstLast = pstNewItem;
		pstHeader->pstFirst = pstNewItem;
		pstHeader->pstCurrent = pstNewItem;
		pstHeader->iItems = 1;
		pstNewItem->pstPrevious = NULL;
	}
	else
	{
		//	Make sure that the original last item's next pointer points to the
		//	new item, and that the new items previous pointer points to the
		//	original last item.  Also increase the number of entries in the
		//	list by 1
		pstHeader->pstLast->pstNext = pstNewItem;
		pstNewItem->pstPrevious = pstHeader->pstLast;
		pstHeader->pstCurrent = pstNewItem;
		pstHeader->pstLast = pstNewItem;
		pstHeader->iItems++;
	}
}

/*---------------------------------------------------------------------------*\
	This function appends a new message to the end of the linked list.
\*---------------------------------------------------------------------------*/
void AddMessageInOrderToList( ST_LIST_HEADER *pstHeader, ST_MESSAGE *pstMessage )
{
	ST_LIST_ITEM far *pstNewItem = NULL;
	ST_LIST_ITEM far *pstItem = NULL;
	int	iNotFound = TRUE;

	//	Allocate space for the new list item.
	pstNewItem = CreateListItem();

	//	Assign the message to the new list item, and then set the next pointer
	//	to be null as we are adding the new list item to the end of the list.
	pstNewItem->pstMessage = pstMessage;
	pstNewItem->pstNext = NULL;

	//	Handle the special case where the list does not contain any items
	if ( pstHeader->pstFirst == NULL )
	{
		pstHeader->pstLast = pstNewItem;
		pstHeader->pstFirst = pstNewItem;
		pstHeader->pstCurrent = pstNewItem;
		pstHeader->iItems = 1;
		pstNewItem->pstPrevious = NULL;
	}
	else
	{
		//	Walk through the list comparing the new message ID with each
		//	message ID already within the list and locate the first ID which is
		//	>= than the ID of the new message.  A zero length ID is considered
		//	to be greater than everything else.
		pstItem = pstHeader->pstFirst;

		while ( iNotFound )
		{
			if (pstItem != NULL)
			{
				if ( CompareMessageIDs(pstMessage, pstItem->pstMessage) <= 0 )
				{
					iNotFound = FALSE;
				}
				else
				{
					pstItem = pstItem->pstNext;
				}
			}
			else
			{
				iNotFound = FALSE;
			}
		}

		if ( pstItem == NULL )
		{
			//	We need to append the new item to the end of the list as it has
			//	the greatest ID
			pstHeader->pstLast->pstNext = pstNewItem;
			pstNewItem->pstPrevious = pstHeader->pstLast;
			pstHeader->pstCurrent = pstNewItem;
			pstHeader->pstLast = pstNewItem;
			pstHeader->iItems++;
		}
		else if ( pstItem->pstPrevious == NULL )
		{
			//	We are going to insert a message at the front of the list.
			pstNewItem->pstNext = pstHeader->pstFirst;
			pstNewItem->pstPrevious = NULL;
			pstHeader->pstCurrent = pstNewItem;
			pstHeader->pstFirst->pstPrevious = pstNewItem;
			pstHeader->pstFirst = pstNewItem;
			pstHeader->iItems++;
		}
		else
		{
			//	pstItem points to the first element in the list which has an
			//	ID greater or equal to that of the message passed into the
			//	function.  This means we wish to insert the new item in the
			//	element immediately before the one pstItem points to.
			pstNewItem->pstNext = pstItem;
			pstNewItem->pstPrevious = pstItem->pstPrevious;
			pstItem->pstPrevious = pstNewItem;
			pstNewItem->pstPrevious->pstNext = pstNewItem;
			pstHeader->iItems++;
			pstHeader->pstCurrent = pstNewItem;
		}
	}
}

/*---------------------------------------------------------------------------*\
	This function removes a complete linked list of messages from memory, and
	re-initialises the list.
\*---------------------------------------------------------------------------*/
void DestroyList( ST_LIST_HEADER *pstHeader )
{
	pstHeader->pstCurrent = pstHeader->pstFirst;

	while ( pstHeader->pstCurrent != NULL )
	{
		//	Free the message completely from memory
		FreeMessage( pstHeader->pstCurrent->pstMessage );

		//	If the next item is NULL then we are at the end of the list so
		//	free the item now.  Otherwise, move to the next element and then
		//	free the previous Item.
		if ( pstHeader->pstCurrent->pstNext == NULL )
		{
			farfree( pstHeader->pstCurrent );
			pstHeader->pstCurrent = NULL;
		}
		else
		{
			pstHeader->pstCurrent = pstHeader->pstCurrent->pstNext;
			farfree( pstHeader->pstCurrent->pstPrevious );
		}
	}

	InitListHeader( pstHeader );
}

/*---------------------------------------------------------------------------*\
	This function creates a message which is the exact duplicate of the one
	passed into the function.
\*---------------------------------------------------------------------------*/
ST_MESSAGE far * CreateDuplicateMessage( ST_MESSAGE far *pstMessage )
{
	ST_MESSAGE	*pstNewMessage;

	pstNewMessage = CreateMessage();

	pstNewMessage->szID = _fstrdup( pstMessage->szID );
	pstNewMessage->szMessage = _fstrdup( pstMessage->szMessage );

	if ( (pstNewMessage->szID == NULL) ||
		 (pstNewMessage->szMessage == NULL) )
	{
		fprintf(stderr,"Unable to allocate memory for message information in CreateDuplicateMessage\n");
		exit( 1 );
	}

	return pstNewMessage;
}

/*---------------------------------------------------------------------------*\
	This function creates a delete message based upon the message passed to
	the function.  A delete message is one whoose message is simply a zero
	terminator (the message contains an ID);
\*---------------------------------------------------------------------------*/
ST_MESSAGE far * CreateDeleteMessage( ST_MESSAGE far *pstMessage )
{
	ST_MESSAGE far *pstDeleteMessage;

	pstDeleteMessage = CreateMessage();

	pstDeleteMessage->szID = _fstrdup( pstMessage->szID );
	pstDeleteMessage->szMessage = CreateEmptyString();

	if ( (pstDeleteMessage->szID == NULL) ||
		 (pstDeleteMessage->szMessage == NULL) )
	{
		fprintf(stderr,"Unable to allocate memory for message information in CreateDeleteMessage\n");
		exit( 1 );
	}

	return pstDeleteMessage;
}

/*---------------------------------------------------------------------------*\
	This function removes the item pointed to by pstCurrent in the linked
	list.  The current Item pointer is then set to NULL.
\*---------------------------------------------------------------------------*/
void RemoveCurrentItemFromList( ST_LIST_HEADER *pstHeader )
{
	if ( pstHeader == NULL )
	{
		DEBUG printf("NULL Header passed to RemoveCurrentItemFromList\n");
		return;
	}

	if ( pstHeader->pstCurrent == NULL )
	{
		DEBUG printf("Current Item is null in RemoveCurrentItemFromList\n");
		return;
	}

	//	There are two special cases which need to be taken into account
	//	The first is where the item being removed is the first item in
	//	the list, and the second is where the item being removed is the last
	//	item in the list.  Check these conditions first
	if ( pstHeader->pstCurrent->pstPrevious == NULL )
	{
		//	We are removing the first item in the list so ensure header pointer
		//	remains correct, and that next item has its previous pointer set to
		//	NULL
		pstHeader->pstFirst = pstHeader->pstCurrent->pstNext;
		pstHeader->pstFirst->pstPrevious = NULL;
	}
	else if ( pstHeader->pstCurrent->pstNext == NULL )
	{
		//	We are at the end of the list.  So drop the last list item ensuring
		//	all header pointers remain correct
		pstHeader->pstLast = pstHeader->pstCurrent->pstPrevious;
		pstHeader->pstLast->pstNext = NULL;
	}
	else
	{
		//	We are dropping a list element from somewhere in the middle.  Ensure
		//	all pointers are correctly assigned before removing the current item
		pstHeader->pstCurrent->pstPrevious->pstNext = pstHeader->pstCurrent->pstNext;
		pstHeader->pstCurrent->pstNext->pstPrevious = pstHeader->pstCurrent->pstPrevious;
	}

	//	Free all memory associated with the this item
	FreeMessage( pstHeader->pstCurrent->pstMessage );
	farfree( pstHeader->pstCurrent );

	pstHeader->iItems--;
	pstHeader->pstCurrent = NULL;
}

/*---------------------------------------------------------------------------*\
	This function frees any memory associated with a message, including the
	message structure itself.
\*---------------------------------------------------------------------------*/
void FreeMessage( ST_MESSAGE far *pstMessage )
{
	if (pstMessage != NULL )
	{
		if ( pstMessage->szID != NULL )
		{
			farfree( pstMessage->szID );
		}

		if ( pstMessage->szMessage != NULL )
		{
			farfree( pstMessage->szMessage );
		}

		farfree( pstMessage );
	}
}

/*---------------------------------------------------------------------------*\
	This function locates the message with the same ID as the message passed
	into the function within the list passed.
\*---------------------------------------------------------------------------*/
void FindMatchingMessage( ST_LIST_HEADER *pstMainList,
						  ST_MESSAGE *pstMessageToFind )
{
	//	Search each element for the message ID we are looking for
	for ( pstMainList->pstCurrent = pstMainList->pstFirst;
			pstMainList->pstCurrent != NULL;
				pstMainList->pstCurrent = pstMainList->pstCurrent->pstNext )
	{
		if ( _fstrcmp( pstMainList->pstCurrent->pstMessage->szID,
					   pstMessageToFind->szID ) == 0 )
		{
			return;
		}
	}

	//	Message not found so set the current pointer to NULL
	pstMainList->pstCurrent = NULL;
}

/*---------------------------------------------------------------------------*\
	This function gets called when in the hidden DEBUG mode to walk the list
	passed to the function printing out the list information.
\*---------------------------------------------------------------------------*/
void WalkList( ST_LIST_HEADER *pstHeader )
{
	ST_LIST_ITEM far *pstItem;

	printf("pstHeader = %p\npstFirst = %p\npstLast = %p\niItems = %d\n",
						pstHeader, pstHeader->pstFirst,
						pstHeader->pstLast, pstHeader->iItems);
	for ( pstItem = pstHeader->pstFirst;
			pstItem != NULL;
				pstItem = pstItem->pstNext )
	{
		printf("[%s]%s\n", pstItem->pstMessage->szID,
						   pstItem->pstMessage->szMessage );
	}
}

/*---------------------------------------------------------------------------*\
	This function loads a patch file or FXT file into a linked list of message
	structures.  The FXT or TXT file is loaded into memory and then parsed one
	message at a time.  The function automatically determines whether the file
	being loaded is encrypted or decrypted by examing the first byte of the
	file.
\*---------------------------------------------------------------------------*/
void LoadFile( char *szFilename,
			   ST_LIST_HEADER *pstHeader )
{
	FILE	*fIn;
	int		iFXTFile = FALSE;
	int		iPos = 0;
	int		iDecrypt = 0;
	char	cChr;
	char far *acBuffer;

	fIn = fopen( szFilename, "rb" );
	if ( fIn == NULL )
	{
		printf("Unable to locate file <%s>\n", szFilename );
		return;
	}

	//	Determine whether the file is an FXT file or not.  If it is an
	//	FXT file then the first byte will be 0xBF
	if ( fgetc( fIn ) == 0xBF )		iFXTFile = TRUE;

	//	Move to the start of the file
	fseek( fIn, 0L, SEEK_SET );

	acBuffer = (char far *)farmalloc( sizeof(char) * FILE_BUFFER_SIZE );

	if (acBuffer == NULL)
	{
		printf("Unable to allocate enough memory for buffer\n");
		fclose(fIn);
		return;
	}

	do
	{
		cChr = fgetc( fIn );

		if ( cChr != EOF )
		{
			if ( iFXTFile )
			{
				//	We are processing an FXT file.  Un-encode the first 8 chrs
				//	if we are processing the furst message else	simply subtract
				//	1 from each character in the file.
				if ( iDecrypt < 8 )
				{
					cChr -= acDecrypt[iDecrypt];
					iDecrypt++;
				}

				cChr--;
			}

			acBuffer[iPos] = cChr;
DEBUG
{
	if ( cChr == 0 )	cChr = '\n';
	printf("%c",cChr);
}
			//	If we have encountered a '\n' character then check the previous
			//	character to see if it is a new line character.  If so then
			//	also set it to 0 as we don't want it.
			//	NOTE : Assumes an 0x0D 0x0A ordering.  TRUE for DOS.  Can't
			//	remember for UNIX.  Probably not an issue though :)
			if ( acBuffer[iPos] == '\n' )
			{
				acBuffer[iPos] = 0;
				if ( acBuffer[iPos-1] == '\r' )		acBuffer[iPos-1] = 0;
			}
		}
		else
		{
			//	We have reached the end of the file so make sure it is zero
			//	terminated
			acBuffer[iPos] = 0;
		}

		//	If we are pointing at a zero terminator then it means we have read
		//	in a complete message.  The next step is to extract the message ID
		//	and message text from the buffer
		if (acBuffer[iPos] == 0)
		{
			iPos = 0;
			ExtractMessageIntoList( pstHeader, acBuffer );
		}
		else
		{
			iPos++;
		}
	}
	while( cChr != EOF );

	fclose(fIn);
	farfree( acBuffer );
}

/*---------------------------------------------------------------------------*\
	This function is reponsible for extracting the message text stored in the
	buffer passed to the function, and creating the message, and adding it to
	the list of messages within this file.
\*---------------------------------------------------------------------------*/
void ExtractMessageIntoList( ST_LIST_HEADER *pstHeader, char far *szMessage )
{
	int	iEndOfIDPos;
	ST_MESSAGE far *pstMessage;

	if ( szMessage[0] != '[' )
	{
		//	we are not at the start of a message so return.  This should remove
		//	any carriage returns from the patch file
		return;
	}
DEBUG	printf("Extracting <%s>\n", szMessage);
	pstMessage = CreateMessage();

	//	Locate the ']' and turn it into a zero terminator
	iEndOfIDPos = 0;
	while ( szMessage[iEndOfIDPos] != ']' )		iEndOfIDPos++;
	szMessage[iEndOfIDPos] = 0;

	//	Now, we need to convert the ID into a string.  If the end of the ID
	//	is at position 1 then we are dealing with a zero length ID.
	if ( iEndOfIDPos == 1 )
	{
		pstMessage->szID = CreateEmptyString();
	}
	else
	{
		pstMessage->szID = _fstrdup( &(szMessage[1]) );
	}

	//	Now to add the message text to the message structure
	if ( szMessage[iEndOfIDPos+1] == 0 )
	{
		//	We are working with an empty message
		pstMessage->szMessage = CreateEmptyString();
	}
	else
	{
		//	Add the message to the message structure
		pstMessage->szMessage = _fstrdup( &(szMessage[iEndOfIDPos+1]) );
	}

	AppendMessageToList( pstHeader, pstMessage );

}

/*---------------------------------------------------------------------------*\
	This function allocates space for the message structure used to contain
	the message information
\*---------------------------------------------------------------------------*/
ST_MESSAGE far * CreateMessage( void )
{
	ST_MESSAGE far * pstMessage;

	pstMessage = (ST_MESSAGE far *)farmalloc( sizeof(ST_MESSAGE) );

	//	If we do not have enough memory then blow up.
	if ( pstMessage == NULL )
	{
		fprintf(stderr,"Unable to allocate memory for MEssage structure in CreateMessage.\n");
		exit( 1 );
	}

	pstMessage->szID = NULL;
	pstMessage->szMessage = NULL;

	return pstMessage;
}

/*---------------------------------------------------------------------------*\
	This function allocates space for a list item
\*---------------------------------------------------------------------------*/
ST_LIST_ITEM far * CreateListItem( void )
{
	ST_LIST_ITEM far * pstItem;

	pstItem = (ST_LIST_ITEM far *)farmalloc( sizeof(ST_LIST_ITEM) );

	//	If we do not have enough memory then blow up.
	if ( pstItem == NULL )
	{
		fprintf(stderr,"Unable to allocate memory for ListItem structure in CreateListItem.\n");
		exit( 1 );
	}

	pstItem->pstNext = NULL;
	pstItem->pstPrevious = NULL;
	pstItem->pstMessage = NULL;

	return pstItem;
}

/*---------------------------------------------------------------------------*\
	This function creates a empty string.  It is simply a character with a
	zero in it.
\*---------------------------------------------------------------------------*/
char far * CreateEmptyString( void )
{
	char far *pcChr;

	pcChr = (char far *)farmalloc( sizeof(char) );

	if ( pcChr == NULL )
	{
		fprintf(stderr,"Unable to allocate single character in CreateEmptyString.\n");
		exit( 1 );
	}

	*pcChr = 0;

	return pcChr;
}

/*---------------------------------------------------------------------------*\
	This function compares two message IDs and returns the following values:
	Message 1 ID < Message 2 ID	= a negative value
	Message 1 ID > Message 2 ID = a positive value
	Message 1 ID = Message 2 ID = zero
	A zero length ID is considered to be the largest ID.
\*---------------------------------------------------------------------------*/
int CompareMessageIDs( ST_MESSAGE *pstMessage1, ST_MESSAGE *pstMessage2 )
{
	//	If message 1 and message 2 have zero length IDs then they are equal
	if ((_fstrlen(pstMessage1->szID) == 0) && (_fstrlen(pstMessage2->szID) == 0) )
		return 0;

	//	If message 1 has a zero length ID then it is > Message 2
	if ( _fstrlen(pstMessage1->szID) == 0 )
		return 1;

	//	If message 2 has a zero length ID then it is > Message 1
	if ( _fstrlen(pstMessage2->szID) == 0 )
		return -1;

	//	If none of the above then return the string comparison of the two
	//	message IDs as this gives similar results
	return _fstrcmp(pstMessage1->szID, pstMessage2->szID);
}

/*---------------------------------------------------------------------------*\
	This function generates a patch file from two complete message files.  The
	patch file produced will turn the original file into the new file.  The
	logic is as follows:

	For each message in the New list
		If the message is contained within the original list, and its text is
			the same in both messages it is removed from both lists.

		If the message is contained within the original list, and the texts
			are different, then the message is removed from the original list.

		If the message is not found within the original list it is added
			to the undo patch list.

	End for

	For each message left in the Original list
		Add a delete message to the patch list.
	End for

	At the end of this function, the Original list will contain garbage, the
	New list will contain the patch file we wish to write, and the undo list
	will contain an equal and opposite patch file to the one in the new list.
\*---------------------------------------------------------------------------*/
void GeneratePatch( ST_LIST_HEADER *pstOriginal,
					ST_LIST_HEADER *pstNew,
					ST_LIST_HEADER *pstUndo,
					int iVerbose )
{
	ST_MESSAGE	*pstMessage;

	pstNew->pstCurrent = pstNew->pstFirst;

	while( pstNew->pstCurrent != NULL )
	{
		//	Locate any message in the main list which has a match ID as the
		//	one passed into the method.  This method sets the pstCurrent pointer
		//	of the header to the matching list item.
		DEBUG printf("Locating message [%s]\n",pstNew->pstCurrent->pstMessage->szID);
		FindMatchingMessage( pstOriginal, pstNew->pstCurrent->pstMessage );

		if ( pstOriginal->pstCurrent != NULL )
		{
			//	Both the original and new message lists contain the message ID,
			//	but, are the messages actually different?
			if ( _fstrcmp( pstOriginal->pstCurrent->pstMessage->szMessage,
							pstNew->pstCurrent->pstMessage->szMessage ) == 0 )
			{
				ST_LIST_ITEM *pstCurrent;

				if ( iVerbose )
					printf("Removing message duplicated in both files.\n");

				//	The messages are the same so we should remove the message
				//	from both the original list and the new list without doing
				//	anything.  Store the current item from the New list's
				//	previous pointer so we can set pstCurrent after we have
				//	removed both current items
				if ( pstNew->pstCurrent->pstPrevious != NULL )
				{
					pstCurrent = pstNew->pstCurrent->pstNext;
					RemoveCurrentItemFromList( pstNew );
					RemoveCurrentItemFromList( pstOriginal );
					pstNew->pstCurrent = pstCurrent;
				}
				else
				{
					RemoveCurrentItemFromList( pstNew );
					RemoveCurrentItemFromList( pstOriginal );
					pstNew->pstCurrent = pstNew->pstFirst;
				}
			}
			else
			{
				if ( iVerbose )
					printf("Storing changed message.\n");

				//	The messages are different so we should remove the message
				//	from the original list and keep the message in the new list.
				//	Before we do that however, make sure that we create an undo
				//	message from the original.
				pstMessage = CreateDuplicateMessage( pstOriginal->pstCurrent->pstMessage );
				AppendMessageToList( pstUndo, pstMessage );
				RemoveCurrentItemFromList( pstOriginal );
				pstNew->pstCurrent = pstNew->pstCurrent->pstNext;
			}
		}
		else
		{
			//	The message in the new list was not found within the original
			//	list so keep it.  Create a delete message for the undo file.
			if ( iVerbose )
				printf("Archiving new message.\n");

			pstMessage = CreateDeleteMessage( pstNew->pstCurrent->pstMessage );
			AppendMessageToList( pstUndo, pstMessage );
			pstNew->pstCurrent = pstNew->pstCurrent->pstNext;
		}
	}

	//	Now, for every message left in the original list, we need to add a
	//	duplicate message to the undo list, and, a delete message to the new
	//	list.
	for ( pstOriginal->pstCurrent = pstOriginal->pstFirst;
			pstOriginal->pstCurrent != NULL;
				pstOriginal->pstCurrent = pstOriginal->pstCurrent->pstNext )
	{
		if ( iVerbose )
			printf("Removing unwanted messages from original file.\n");

		//	Create a duplicate message first and add it to the undo list
		pstMessage = CreateDuplicateMessage( pstOriginal->pstCurrent->pstMessage );
		AppendMessageToList( pstUndo, pstMessage );

		//	Create a delete message and add it to the New list.
		pstMessage = CreateDeleteMessage( pstOriginal->pstCurrent->pstMessage );
		AppendMessageToList( pstNew, pstMessage );
	}
}

/*---------------------------------------------------------------------------*\
	This function writes a complete message to a buffer passed as a near
	pointer.  The message includes [ and ] braces, and terminates the string
	with a zero
\*---------------------------------------------------------------------------*/
void WriteMessageToBuffer( char *acBuffer, char far *szID, char far *szMessage )
{
	int	iBufferPos = 0;
	int	iLoop;

	acBuffer[iBufferPos] = '[';
	iBufferPos++;

	for (iLoop = 0; iLoop < _fstrlen( szID ); iLoop++)
	{
		acBuffer[iBufferPos] = szID[iLoop];
		iBufferPos++;
	}

	acBuffer[iBufferPos] = ']';
	iBufferPos++;

	for (iLoop = 0; iLoop < _fstrlen( szMessage ); iLoop++)
	{
		acBuffer[iBufferPos] = szMessage[iLoop];
		iBufferPos++;
	}

	acBuffer[iBufferPos] = 0;
}
