/* $Id: widecomp.cpp,v 1.8 2000/11/23 18:42:39 umoeller Exp $ */

/***************************************************************************
 *                                                                         *
 *  Decompression routines for the class WIArchive                         *
 *                                                                         *
 *  This file Copyright (C) 1998-2000 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include "setup.h"              // added V0.9.2 (2000-03-15) [umoeller]

#ifdef WIN32
#include <limits.h>
#endif

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

/****************************************************************************
 * Decompress a previously compressed file
 *
 * Arguments - head: Pointer to file header
 *
 * Error handling is as follows:
 *  --  if a file read error occurs with the Archive file,
 *      the callback is called;
 *  --  in all other error situations, you get a return value
 *      != 0.
 *
 * Returns:
 *  0       everything OK
 *  > 1     output file handling error (e.g. disk full);
 *          this is the return value from ferror(). You should
 *          then check errno or (with VAC++) _doserrno to find
 *          out what happened.
 *  < 1     decompression error from zlib, which are:
 *               Z_DATA_ERROR
 *               Z_STREAM_ERROR
 *               Z_MEM_ERROR
 *               Z_BUF_ERROR
 *          See inflate() in zlib.h for details. This is most
 *          definitely the result of a corrupt archive.
 */
short WIArchive::Expand (WIFileHeader *head)
{
    bz_stream z;
    char *outbuf;
    int status;
    long readcount, l;
    unsigned long ul;

    // First of all, do some initialization
    head->crc = INIT_CRC;
    text = new char[BUFFERSIZE];
    outbuf = new char[BUFFERSIZE];
    z.bzalloc = 0;
    z.bzfree  = 0;
    z.opaque = NULL;
    z.bzalloc = NULL;
    z.bzfree = NULL;
    z.next_in = text;
    readcount = head->compsize;
    if (readcount < (long)BUFFERSIZE)
        l = readcount;
    else
        l = BUFFERSIZE;
    z.avail_in = fread (text, 1, l, _Archive);
    readcount -= l;

    // read error checking added
    if (z.avail_in == 0)
    {
        // check if it's really an error and not maybe EOF
        int ferr = ferror(_Archive);
        if (ferr != 0)
            if (_pfnCallback != NULL)
                (*_pfnCallback)(CBM_ERR_READ,
                                CBREC_CANCEL,   // allow only "Cancel"
                                head);
    }

    BZ2_bzDecompressInit (&z, 0, 0);

    // Let the decompression begin!
    z.next_out = outbuf;
    z.avail_out = BUFFERSIZE;
    for (;;)
    {
        if (_pfnCallback != NULL)
            (*_pfnCallback)(CBM_PERCENTAGE,
                            CalcPercentage (ftell(_File), head->origsize),
                            head);
        if (z.avail_in == 0)
             break;
        while (z.avail_in)
        {
            status = BZ2_bzDecompress (&z);

            if (status < 0)
                // < 0 means ZLIB error
                return (status);    // < 0

            for (ul = 0;
                 ul < BUFFERSIZE - z.avail_out;
                 ul++)
                UPDATE_CRC (text[ul]);

            // write error checking added
            if (BUFFERSIZE - z.avail_out)
            {
                size_t BytesWritten = fwrite (outbuf, 1, BUFFERSIZE - z.avail_out, _File);

                if (BytesWritten != (BUFFERSIZE - z.avail_out))
                {
                    // check if it's really an error and not maybe EOF
                    int ferr = ferror(_File);
                    if (ferr != 0)
                    {
                        // error occured:
                        // report error to caller
                        return (abs(ferr));        // > 0
                    }
                }
            }

            z.next_out = outbuf;
            z.avail_out = BUFFERSIZE;
        }
        if (status == BZ_STREAM_END)  break;
        z.next_in = text;
        if (readcount < (long)BUFFERSIZE)
            l = readcount;
        else
            l = BUFFERSIZE;
        z.avail_in = fread (text, 1, l, _Archive);
        readcount -= l;
    }

    // We're done.  Yee-hah...
    BZ2_bzDecompressEnd (&z);
    delete [] outbuf;
    delete [] text;
    return 0;
}

/****************************************************************************
 * Extract a stored file
 *
 * Arguments - head: Pointer to file header
 */
short WIArchive::Extract (WIFileHeader *head)
{
    unsigned long crc, count;
    unsigned long bytes, ul;

    count = head->compsize;
    crc = INIT_CRC;

    // ----- Extract the stored file
    if (_pfnCallback != NULL)
        (*_pfnCallback)(CBM_PERCENTAGE,
                        CalcPercentage (ftell(_File), head->origsize),
                        head);
    text = new char[BUFFERSIZE];
    while (count)
    {
        if (count >= BUFFERSIZE)
            bytes = BUFFERSIZE;
        else
            bytes = count;

        // read error checking added
        size_t BytesRead = fread (text, 1, bytes, _Archive);
        if (BytesRead == 0)
        {
            // check if it's really an error and not maybe EOF
            int ferr = ferror(_Archive);
            if (ferr != 0)
                if (_pfnCallback != NULL)
                    (*_pfnCallback)(CBM_ERR_READ,
                                    CBREC_CANCEL,   // allow only "Cancel"
                                    head);
        }

        for (ul = 0; ul < bytes; ul++)
            UPDATE_CRC (text[ul]);

        // write error checking added
        size_t BytesWritten = fwrite (text, 1, bytes, _File);
        if (BytesWritten != bytes)
        {
            // check if it's really an error and not maybe EOF
            int ferr = ferror(_File);
            if (ferr != 0)
                if (_pfnCallback != NULL)
                    (*_pfnCallback)(CBM_ERR_WRITE,
                                    CBREC_CANCEL,   // allow only "Cancel"
                                    head);
        }

        if (_pfnCallback != NULL)
            (*_pfnCallback)(CBM_PERCENTAGE,
                            CalcPercentage (ftell(_File), head->origsize),
                            head);
        count -= bytes;
    }
    delete [] text;

    // ----- Check if the CRC matched
    crc ^= INIT_CRC;
    if (crc == head->crc)
        return 1;
    else
        return 0;
}

