/* $Id: wiarchive.h,v 1.12 2000/11/23 18:42:27 umoeller Exp $ */

/***************************************************************************
 *                                                                         *
 *  Main class for WarpIN, WIArchive (header file)                         *
 *                                                                         *
 *  This file Copyright (C) 1998-2000 Jens Bckman, Ulrich Mller.           *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, in version 2 as it comes in the COPYING  *
 *  file of this distribution.                                             *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 ***************************************************************************/

#ifndef WIARCHIVE_INCLUDE
#define WIARCHIVE_INCLUDE

#include <stdio.h>
#include <list>
#include <time.h>               // needed for WIFileHeader
#include "bzlib.h"

#ifdef WIN32
	using namespace std;
#endif

// this is the maximum path length allowed on OS/2
#define MAXPATHLEN 256

// ----- Structures used in the archive -------------------------------------

/*
 *@@ WIFile:
 *      structure representing a file in an archive.
 */

struct WIFile
{
    char    *name;              // Name of file
    char    *extra;             // Possible extra information
    short   package;            // Which package it belongs to
};

/*
 *@@ WIArcHeader:
 *      structure representing the header of an archive.
 *      This is stored in binary in the archive, but also
 *      a member of WIArchive.
 */

struct WIArcHeader
{
    unsigned char  v1, v2, v3, v4;       // Archive verification
    short wi_revision_needed;   // Minimum WarpIN version required
    char  path[MAXPATHLEN];     // Installation path
    char  name_app[64];         // Application name
    char  name_dev[64];         // Delevoper name
    char  inet_address[128];    // Internet download address
    short rev_app;              // Application revision
    short os;                   // Operating system to install to
    short packs;                // Number of packages in archive
    short origscript;           // Size of installation script
    short compscript;           // Size of compressed script
    long  extended;             // Extended data

    ~WIArcHeader()
    {
        memset(this, 0, sizeof(*this));     // V0.9.6 (2000-11-23) [umoeller]
    }
};

/*
 *@@ WIPackHeader:
 *      structure representing a package in an archive.
 *      This is stored in binary in the archive.
 */

struct WIPackHeader
{
    short number;               // Package number
    short files;                // Number of files in package
    long  pos;                  // Position of first WIFileHeader for this package in archive
    long  origsize;             // Size of package (original)
    long  compsize;             // Size of package (compressed)
    char  name[32];             // Name of package
};

const unsigned short WIFH_MAGIC = 0xF012;  // any value for "magic" in file header

#pragma pack(1)                 // save space in archive

/*
 *@@ WIFileHeader:
 *      structure representing a file in an archive.
 *      This is stored in binary in the archive.
 *
 *      In the archive, a WIFileHeader comes directly
 *      before the actual file data. To extract files
 *      for a package, we must find the first WIFileHeader
 *      for a package (which is stored in WIPackHeader.pos)
 *      and get the data which comes directly afterwards.
 *
 *      The next WIFileHeader is then at after the file
 *      data, whose size is stored in the "compsize"
 *      member.
 */

struct WIFileHeader
{
    unsigned short magic;       // must be WIFH_MAGIC for security
    short checksum;             // header checksum
    short method;               // compression method used:
                                // -- 0: stored (use WIArchive::Extract)
                                // -- 1: compressed (use WIArchive::Expand)
    short package;              // which package it belongs to
    long  origsize;             // size of file (original)
    long  compsize;             // size of file (compressed)
    unsigned long crc;          // file CRC checksum
    char  name[MAXPATHLEN];     // filename (*UM#3)
    time_t lastwrite;           // file's last write date/time (req. time.h) (*UM#3)
    time_t creation;            // file's creation date/time (req. time.h) (*UM#3)
            // we don't need last access date/time
    char  extended;             // size of extended information (if any)
};

#pragma pack()                  // save space in archive

// ----- Callback function definition and related stuff ---------------------

// generic callback function prototype
// this has now three parameters:
//      short         mode   one of the CBM_* flags below
//      short         param  meaning depends on CBM_* value
//      WIFileHeader* pwifh  the file header of the file in question
typedef int (FNWICALLBACK)(short, short, WIFileHeader*);
typedef FNWICALLBACK *PFNWICALLBACK;

// ----- new #define's for the "mode" parameter of the callbacks
const int CBM_PERCENTAGE   = 1;
            // in this case, "param" contains the percentage of the current
            // file being processed, and pwifh the current file header.
            // The return value doesn't matter.
const int CBM_UPDATING     = 2;
            // this is for compression after a file has been added to the
            // archive and the archive is to be updated; looks better if we
            // have an additional message then.
            // In this case, "param" has the compression that was achieved,
            // and pwifh the current file header.
            // The return value doesn't matter.
const int CBM_NEXTFILE     = 3;
            // this comes just before the WIArchive class attempts to open
            // a new output file for writing. "param" is always 0, "pwifh"
            // points to the file header which is about to be opened.
            // This allows the front-end to do two things:
            // a)   update the "current file" display
            // b)   check for whether that file exists already:
            //      -- if not, we return CBRC_PROCEED;
            //      -- if it does exist, we ask the user for whether the file
            //         may be overwritten; if so, we try to delete it.
            //         Otherwise, we return CBRC_SKIP, and the back-end does
            //         not touch the file.
            // This new approach (Alpha #3) is required because sometimes
            // the target file which already exists is _locked_, e.g. if it's
            // a system DLL in use. We then need to do a DosReplaceModule,
            // which we cannot do in the back-end if we want to keep it
            // platform-independent. Sorry for the hassle.
const int CBM_ERR_MEMORY   = 4;
            // Somehow, there wasn't enough memory to do something.  Not even
            // in swap space?  Well...  We're here, anyway.
            // We should not return from the callback then, but only terminate
            // WarpIn.
const int CBM_ERR_WRITE    = 5;
            // error writing into output file; this is most probable when the
            // target disk is full.
            // "param" then has one of the CBREC_* values below, which identify
            // whether the error is recoverable. Depending on that value,
            // we may return one of the CBRC_* values.
const int CBM_ERR_READ     = 6;
            // error reading from input file; maybe the download is corrupt.
            // "param" then has one of the CBREC_* values below, which identify
            // whether the error is recoverable. Depending on that value,
            // we may return one of the CBRC_* values.
const int CBM_ERR_ZLIB = 7;
            // error somewhere in the zlib decompression part of WIArchive;
            // this usually signifies that the archive is broken. We then
            // terminate installation, since there's no way to recover.
            // "param" then contains one of the ZLIBERR_* flags below.

// ----- Error descriptions in case of CBM_ERR_ZLIB
// the following are from zlib.h
const int ZLIBERR_ERRNO           = -1;
            // unknown meaning
const int ZLIBERR_STREAM_ERROR    = -2;
            // inconsistent stream data, e.g. unexpected NULL
const int ZLIBERR_DATA_ERROR      = -3;
            // input stream does not match zlib format
const int ZLIBERR_MEM_ERROR       = -4;
            // not enough memory
const int ZLIBERR_BUF_ERROR       = -5;
            // not enough room in output buffer
const int ZLIBERR_VERSION_ERROR   = -6;
            // unknown meaning
// the following are added by WIArchive
const int ZLIBERR_NO_WIFH_MAGIC   = -99;
            // this occurs if the first "short" in WIFileHeader
            // does not match WIFH_MAGIC; this check is added
            // to avoid crashes in the decompression routines

// ----- Error recovery flags for callback function
// If mode == CBM_ERR_xxx, the callback gets one of the following flags
// in "param", signalling whether the error is recoverable or not.
// Depending on this flag, we may return one of the CBRC_* values.
// NOTE: We must never return CBRC_ABORT, which the WIArchive class cannot
// handle, but terminate the install thread ourselves instead.
const int CBREC_CANCEL              = 1;
            // error is not recoverable: show "Cancel" only
const int CBREC_RETRYCANCEL         = 2;
            // show "Cancel" and "Retry"
const int CBREC_ABORTRETRYIGNORE    = 3;
            // show "Abort", "Retry", "Ignore" (== skip file)

// ----- Return values for the callback function
// If mode == CBM_ERR_xxx, the callback must return one of these three.
const int CBRC_ABORT = 1;
const int CBRC_RETRY = 2;
const int CBRC_IGNORE = 3;

// If mode == CBM_NEXTFILE, return one of these.
const int CBRC_PROCEED = 4;
const int CBRC_SKIP = 5;

// In all other modes, the return value is not examined by the back-end.

// Note:
// If you offer a "Cancel" option in your error / "File exists" dialogs,
// it is your own responsibility to terminate the install process. The
// WIArchive class cannot handle a "Cancel" or "Abort" return value.
// So the safest way to do all this is have the WIArchive class run in
// a separate thread while unpacking the files, and terminate the thread
// simply. This is what happens in the WarpIn front-end.

// ----- Archive internal stuff ---------------------------------------------
#ifdef WIARCHIVE_INTERNAL
    const unsigned char WI_VERIFY1 = 0x77;
    const unsigned char WI_VERIFY2 = 0x04;
    const unsigned char WI_VERIFY3 = 0x02;
    const unsigned char WI_VERIFY4 = 0xBE;
    const unsigned short WI_OS_OS2 = 1;          // 1 - OS/2

    const unsigned long INIT_CRC = 0xFFFFFFFF;
    const unsigned long CRCPOLY = 0xEBD88320;
    #define UPDATE_CRC(c) \
        head->crc = _crctable[(head->crc ^ (c)) & 0xFF] ^ (head->crc >> CHAR_BIT)

    const unsigned long BUFFERSIZE = 0x6000;

    extern char *text;
    // extern unsigned long crctable[];
#endif

const short WIARCHIVE_REVISION = 3;             // changed (99-11-03) [umoeller]
const short WIARCHIVE_REVISION_NEEDED = 3;      // changed (99-11-03) [umoeller]

/*
 *@@ PFNFORALLFILESCALLBACK:
 *      callback prototype for WIArchive::for_all_files
 */

typedef int (FNFORALLFILESCALLBACK)(WIFileHeader*, unsigned long);
typedef FNFORALLFILESCALLBACK *PFNFORALLFILESCALLBACK;

const int WIERR_OUTDATED_ARCHIVE  = 1;              // added (99-11-03) [umoeller]
const int WIERR_INVALID_HEADER    = 2;              // added (99-11-03) [umoeller]
const int WIERR_BZDECOMPRESS      = 3;              // added V0.9.2 (2000-03-10) [umoeller]
const int WIERR_FILENOTFOUND      = 4;              // added V0.9.2 (2000-03-10) [umoeller]

// ----- WIArchive class definition -------------------------------------
class WIArchive
{
    protected:
        short CalcPercentage(long, long);
        FILE* OpenTemp(const char *, const char *);
        int ReadArcHeader(FILE *);             // ADD: Scan entire file for a valid archive
        void ReadFilenames(FILE *);
        void ReadPackHeaders(FILE *);
        void Update();
        void WriteArcHeader(FILE *);
        void WritePackHeaders(FILE *);
        void Compress(WIFileHeader *);
        void Store(WIFileHeader *);

        WIArcHeader     _ArcHeader;
        list<WIFile*>   _FileList;
        list<WIFile*>   _ToDoList;
        list<WIPackHeader*>
                        _PackList;
        FILE            *_Archive;
        FILE            *_OldArchive;
        bool            _fOpenForReadOnly;
        // int          UsingTemp;
        FILE            *_TempFile;
        char            *_Script;
        char            *_Extended;
        char            _ArchiveName[128];
        char            _TempArchiveName[128];
        PFNWICALLBACK   _pfnCallback;
        long            _ArcStart;
        unsigned long   _crctable[256];

    public:
        WIArchive();
        ~WIArchive();
        void add(const char *, const char *, short);
        void close();
        void closeAbort();
        const WIArcHeader *getArcHeader();
        list<WIFile*>* getFileList();
        list<WIPackHeader*>* getPackList();
        const char* getScript();
        int open(const char *, int = 0);
        void remove(const char *, short);
        void setArcHeader(const WIArcHeader &);
        void setPackage(short, const char *);
        void setCallbackFunc(PFNWICALLBACK);
        void setScript(const char *);
        short unpack(short);
        short forAllFiles(WIPackHeader*, PFNFORALLFILESCALLBACK, unsigned long);

		// made public for WinWpi (2001-03-07) [cbockemuehl]
        void MakeDirectories(const char *);
        short Expand(WIFileHeader *);
        short Extract(WIFileHeader *);
        FILE            *_File;             // single input/output file that we're
                                            // currently working on

        int             _lastError;             // added (99-11-03) [umoeller]
};

#endif
