/* $Id: wiarchive.cpp,v 1.17 2000/11/23 18:42:38 umoeller Exp $ */

/*
 *@@sourcefile wiarchive.cpp:
 *      this is the WarpIN back-end class, WIArchive.
 *      The class handles all the grunt work of managing a compressed archive,
 *      including pretty advanced file management. Just create a new instance
 *      of this class for each and every archive file that needs to be processed.                         *
 *
 *@@header "wiarchive\wiarchive.h"
 */

/*
 *
 *  This file Copyright (C) 1998-2000 Jens Bckman, Ulrich Mller.
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, in version 2 as it comes in the COPYING
 *  file of this distribution.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <io.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <sys\utime.h>

#ifdef WIN32
#include <limits.h>
#include <direct.h>
#define mkdir _mkdir
#endif

// the following is required for VAC++
#ifdef __IBMCPP__
#include <direct.h>
#endif

#include "setup.h"              // added V0.9.2 (2000-03-15) [umoeller]

#define WIARCHIVE_INTERNAL
#include "wiarchive.h"

/*
 *@@ WIArchive:
 *      the constructor.
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: now finally initializing all members...
 */

WIArchive::WIArchive()
{
    // _ArcHeader.extended = 0; // added v0.9.4 (2000-08-05) [tahola]

    _Archive = 0;
    _OldArchive = 0;
    _fOpenForReadOnly = 0;
    _File = 0;
    _TempFile = 0;
    _Script = 0;
    _Extended = 0;
    _ArchiveName[0] = 0;
    _TempArchiveName[0] = 0;

    _pfnCallback = 0;
    _ArcStart = 0;

    // set up the CRC table right now
    unsigned long i, j, r;
    // for (i = 0; i <= 256; i++)
    for (i = 0; i < 256; i++)       // V0.9.6 (2000-11-23) [umoeller]
    {
        r = i;
        for (j = 0; j < 8; j++)
            if (r & 1)
                r = (r >> 1) ^ CRCPOLY;
            else
                r >>= 1;

        _crctable[i] = r;
    }

    _lastError = 0;         // added (99-11-03) [umoeller]
}

/*
 *@@ ~WIArchive:
 *      destructor.
 */

WIArchive::~WIArchive()
{
    list<WIFile*>::iterator start, end;

    // clean up to-do list
    start = _ToDoList.begin();
    end = _ToDoList.end();
    for (; start != end; start++)
    {
        WIFile *pFileThis = *start;
        if (pFileThis->name != NULL)
            delete[] pFileThis->name;
        if (pFileThis->extra != NULL)
            delete[] pFileThis->extra;
        delete pFileThis;
    }

    // clean up files lists
    start = _FileList.begin();
    end = _FileList.end();
    for (; start != end; start++)
    {
        WIFile *pFileThis = *start;
        if (pFileThis->name != NULL)
            delete[] pFileThis->name;
        if (pFileThis->extra != NULL)
            delete[] pFileThis->extra;
        delete pFileThis;
    }

    // erase the package list too
    list<WIPackHeader*>::iterator pstart = _PackList.begin(),
                                  pend = _PackList.end();
    for (; pstart != pend; pstart++)
    {
        delete (WIPackHeader*)(*pstart);
    }

    if (_Script != NULL)
        delete[] _Script;
    if (_Extended != NULL)
        delete[] _Extended;
}

/*
 *@@ CalcPercentage:
 *      Calculates a percentage value using the formula:
 *
 +          (100a + (b/2)) / b
 *
 *      a and b are two long numbers to use during calculations.
 *      Returns the percentage calculated.
 */

short WIArchive::CalcPercentage(long a, long b)
{
    int i;

    for (i = 0; i < 2; i++)
        if (a <= LONG_MAX / 10)
            a *= 10;
        else
            b /= 10;
    if ((long) (a + (b >> 1)) < a)
    {
        a >>= 1;
        b >>= 1;
    }
    if (b == 0)
        return 0;
    return (short) ((a + (b >> 1)) / b);
}

/*
 *@@ MakeDirectories:
 *      for the given filename, this checks if all directories
 *      (path components, if given) exist. If they don't, they
 *      are created ("deep" makedir).
 *
 *      "filename" can, but does not have to be, fully qualified.
 */

void WIArchive::MakeDirectories(const char *filename)
{
    char path[MAXPATHLEN];
    char *cp, c;

    // only do anything if the filename seems to
    // include a path at all
    if (    (strlen(filename) > 3)
         && (   (strchr(filename, '\\') != 0)
             || (strchr(filename, '/') != 0)
            )
       )
    {
        // Copy the filename and prepare for some serious directory creation
        strcpy(path, filename);
        cp = path;

        // advance past the drive letter only if we have one
        if (*(cp+1) == ':')
            cp += 3;

        // Now, make the directories
        while (*cp != '\0')
        {
            if (*cp == '\\' || *cp == '/')
            {
                c = *cp;
                *cp = '\0';
                #if defined( __IBMCPP__ ) | defined( WIN32 )
                    mkdir(path);       // IBM VAC++ version
                #else
                    mkdir(path, 0);
                #endif
                *cp = c;
            }
            cp++;
        }
    }
}

/*
 *@@ OpenTemp:
 *      creates a temporary archive filename and opens it.
 *
 *      "mode" must be a fopen()-style mode string.
 *
 *      Returns a C-library file or NULL on errors.
 *
 *@@changed V0.9.6 (2000-11-23) [umoeller]: now also storing temp file in member data
 */

FILE* WIArchive::OpenTemp(const char *filename,  // in: archive file name
                          const char *mode)      // in: open mode
{
    // FILE*   file = 0;
    char    TempName[256];
    char    *cp;

    strcpy (TempName, filename);
    cp = strrchr (TempName, '\\');
    if (cp == NULL)
    {
        if (TempName[1] == ':')
            cp = &TempName[2];
        else
            cp = TempName;
    }
    else
        cp++;
    strcpy (cp, "_witemp_");
    _TempFile = fopen(TempName, mode);
    if (_TempFile != NULL)
    {
        // opening of the temporary file went fine, say that we use it
        strcpy(_TempArchiveName, TempName);
    }

    return (_TempFile);
}

/*
 *@@ ReadArcHeader:
 *      reads in the archive header from the specified archive file
 *      and does some integrity checks.
 *
 *      Returns 0 on errors.
 */

int WIArchive::ReadArcHeader(FILE *file)
{
    bz_stream z;
    long l;

    // Check if this is a valid archive
    for (l = 0; l < 200000; l++)
    {
        fseek (file, l, SEEK_SET);
        fread ((char *)&_ArcHeader,
               1,
               sizeof(WIArcHeader),
               file);

        if (    (_ArcHeader.v1 != WI_VERIFY1)
             || (_ArcHeader.v2 != WI_VERIFY2)
             || (_ArcHeader.v3 != WI_VERIFY3)
             || (_ArcHeader.v4 != WI_VERIFY4)
           )
            continue;

        if (_ArcHeader.wi_revision_needed > WIARCHIVE_REVISION)
        {
            _lastError = WIERR_OUTDATED_ARCHIVE;   // added (99-11-03) [umoeller]
            return 0;
        }
        l = 999999;  break;
    }

    if (l != 999999)
    {
        _lastError = WIERR_INVALID_HEADER;  // added (99-11-03) [umoeller]
        return 0;
    }

    // Well, if we have come this far, we have a perfectly healthy archive
    // --- Jens says; famous last words!! This can solidly hang WarpIN if
    //     we don't do more error checking... bzlib returns ERROR CODES,
    //     so we should check for these
    _ArcStart = l;
    if (_ArcHeader.compscript > 0)
    {
        char* pTempBuffer = new char[_ArcHeader.compscript];
        _Script = new char[_ArcHeader.origscript + 1];
        fread(pTempBuffer, 1, _ArcHeader.compscript, file);

        // ----- Decompress the installation script
        z.bzalloc = 0;
        z.bzfree  = 0;
        z.opaque = NULL;
        z.bzalloc = NULL;
        z.bzfree = NULL;
        z.next_in = pTempBuffer;
        z.avail_in = _ArcHeader.compscript;
        z.next_out = _Script;
        z.avail_out = _ArcHeader.origscript + 1;
        int irc = BZ2_bzDecompressInit(&z, 0, 0);
        if (irc != BZ_OK)       // V0.9.2 (2000-03-10) [umoeller]
        {
            _lastError = WIERR_BZDECOMPRESS;
            return (0);
        }

        do
        {
            irc = BZ2_bzDecompress(&z);
            if (irc != BZ_OK && irc != BZ_STREAM_END)
            {
                _lastError = WIERR_BZDECOMPRESS;
                return (0);
            }
        }
        while (irc != BZ_STREAM_END);

        BZ2_bzDecompressEnd(&z);
        _Script[_ArcHeader.origscript] = '\0';
    }

    if (_ArcHeader.extended > 0)
    {
        _Extended = new char[_ArcHeader.extended];
        fread(_Extended, 1, _ArcHeader.extended, file);
    }

    return 1;       // no error
}

/*
 *@@ ReadFilenames:
 *      reads all the file names which are stored in the
 *      specified archive.
 *
 *      This does not save the current position!
 */

void WIArchive::ReadFilenames(FILE *file)
{
    WIFileHeader fhead;
    unsigned long pos;
    int i;

    list<WIPackHeader*>::iterator pstart = _PackList.begin(),
                                  pend = _PackList.end ();
    for (; pstart != pend; pstart++)
    {
        WIPackHeader *pPackHeaderThis = *pstart;    // V0.9.3 (2000-05-11) [umoeller]

        i = pPackHeaderThis->files;
        pos = pPackHeaderThis->pos;
        while (i--)
        {
            fseek(file, pos, SEEK_SET);
            fread((char *)&fhead, 1, sizeof (WIFileHeader), file);
            WIFile *wifile = new WIFile;
            wifile->package = fhead.package;
            wifile->extra = NULL;
            wifile->name = new char[strlen(fhead.name) + 1];
            strcpy (wifile->name, fhead.name);
            _FileList.push_back(wifile);
            pos += sizeof (WIFileHeader) + fhead.compsize;
        }
    }
}

/*
 *@@ ReadPackHeaders:
 *      reads in the contents of all package headers in the specified
 *      archive. This reads from the current position!
 */

void WIArchive::ReadPackHeaders(FILE *file)
{
    WIPackHeader *p;
    int i;

    i = _ArcHeader.packs;
    while (i--)
    {
        p = new WIPackHeader;
        fread ((char*)p,
               1,
               sizeof(WIPackHeader),
               file);
        _PackList.push_back(p);
    }
}

/*
 *@@ Update:
 *      this updates the archive (which must be in "write" mode)
 *      with everything that must be updated: adds and removes files
 *      and such.
 *
 *      Preconditions:
 *
 *      -- The archive must have been opened in "write" mode.
 */

void WIArchive::Update()
{
    WIFileHeader filehead;
    unsigned short s;
    long pos, pos2, packpos;

    fseek(_Archive, 0, SEEK_SET);
    WriteArcHeader(_Archive);
    WritePackHeaders(_Archive);

    unsigned short sMax = 0;

    // find the highest package index in the archive
    list<WIPackHeader*>::iterator pstart = _PackList.begin(),
                                  pend = _PackList.end();
    WIPackHeader* pPckThis;
    for (; pstart != pend; pstart++)
    {
        pPckThis = *pstart;
        if (pPckThis->number > sMax)
            sMax = pPckThis->number;
    }
    // sMax now has the highest package index in use

    // check for every package there is
    _ArcHeader.packs = 0;
    for (s = 1;
         s <= sMax;
         s++)
    {
        bool fFound = 0;
        // see if we have a file on the to-do list which
        // uses the package with index s
        list<WIFile*>::iterator start = _ToDoList.begin(),
                                end = _ToDoList.end();
        for (; start != end; start++)
        {
            if ((**start).package == s)
            {
                fFound = 1;
                break;
            }
        }

        if (fFound == 0)
        {
            // no file on the to-do list is using the package with index s:
            // then check if maybe a file on the list
            // of existing files uses it
            start = _FileList.begin();
            end = _FileList.end();
            for (; start != end; start++)
                if ((**start).package == s)
                {
                    fFound = 1;
                    break;
                }
        }

        if (fFound == 0)
            // no file is using this package (neither on
            // to-do list nor on existing list):
            // go for next package then, we don't need this
            continue;   // for s...

        // yes, we had at least one file in this package
        // (either on to-do list or on existing list)
        _ArcHeader.packs++;
        pstart = _PackList.begin();
        pend = _PackList.end();
        for (; pstart != pend; pstart++)
        {
            if ((**pstart).number == s)
                break;
        }

        // **pstart now has the current WIPackHeader to work on
        WIPackHeader *pPackHeaderThis = *pstart;        // V0.9.3 (2000-05-11) [umoeller]

        packpos = ftell(_Archive);
        pPackHeaderThis->files = 0;
        pPackHeaderThis->origsize = pPackHeaderThis->compsize = 0;

        // Process the files on our to-do list
        start = _ToDoList.begin();
        end = _ToDoList.end();
        for (; start != end; start++)
        {
            WIFile *pFileThis = *start;    // V0.9.3 (2000-05-11) [umoeller]

            if (pFileThis->package != s)
                // incorrect pack number
                continue;

            pos = ftell(_Archive);      // remember current position

            // copy data, store file header and open the source file
            strcpy (filehead.name, pFileThis->name);
            filehead.package = pFileThis->package;
            _File = fopen(pFileThis->extra, "rb");
            fseek(_File, 0, SEEK_END);
            filehead.origsize = ftell(_File);
            filehead.extended = 0;
            filehead.magic = WIFH_MAGIC;

            // get file date/time
            struct stat StatBuf;
            fstat(fileno(_File), &StatBuf);     // gets file info
            // store last write date and creation date in file header
            filehead.lastwrite = StatBuf.st_mtime;
            filehead.creation  = StatBuf.st_ctime;

            // write file header
            fwrite((char*)&filehead,
                   1,
                   sizeof(WIFileHeader),
                   _Archive);

            // Compress this thingie!
            rewind(_File);      // go to beginning of source file
            Compress(&filehead);
            if (filehead.compsize > filehead.origsize)
            {
                // compressed version is larger than original one:
                // rewind then and write the file as-is
                fseek(_Archive,
                      pos + sizeof (WIFileHeader),
                      SEEK_SET);
                rewind(_File);
                Store(&filehead);
            }
            if (_pfnCallback != NULL)
                (*_pfnCallback)(CBM_UPDATING,
                                CalcPercentage (filehead.compsize, filehead.origsize),
                                &filehead);

            // remember position _after_ compressed file
            // (this is for the next file header)
            pos2 = ftell(_Archive);
            // go to beginning of _this_ file and rewrite the
            // updated file header (compressed size etc.)
            fseek(_Archive, pos, SEEK_SET);
            fwrite((char *)&filehead,
                   1,
                   sizeof(WIFileHeader),
                   _Archive);
            // and go back to _after_ this file for the next file header
            fseek(_Archive,
                  pos2,
                  SEEK_SET);
            pPackHeaderThis->files++;
            pPackHeaderThis->origsize += filehead.origsize;
            pPackHeaderThis->compsize += filehead.compsize;
        }

        // copy all the old files for package index s

        // set file pointer to what the package header has
        // as the beginning of files.  Only do this if we use a temp file
        // (i.e, updating an old archive)
        if (_TempFile)
        {
            fseek(_OldArchive,
                  pPackHeaderThis->pos,
                  SEEK_SET);
            for (;;)
            {
                pos = ftell(_OldArchive);
                // read the file name and see if it's in the list
                size_t stRead = fread((char*)&filehead,
                                      1,
                                      sizeof(WIFileHeader),
                                      _OldArchive);
                if (stRead < sizeof(WIFileHeader))
                    break;

                // search the list now
                start = _FileList.begin();
                end = _FileList.end();
                for (; start != end; start++)
                {
                    WIFile *pFileThis = *start;    // V0.9.3 (2000-05-11) [umoeller]

                    if (    (strcmp(filehead.name, pFileThis->name) == 0)
                        &&  (filehead.package == pFileThis->package)
                        &&  (filehead.package == s)
                       )
                    {
                        // this is the file, let's copy it
                        // 1)   copy file header we got above
                        fwrite((char*)&filehead,
                               1,
                               sizeof(WIFileHeader),
                               _Archive);

                        // 2)   copy data in chunks of BLOCKSIZE
                        unsigned char *text = (unsigned char *)malloc(BUFFERSIZE);
                        long lBytesToGo = filehead.compsize;

                        while (lBytesToGo > 0)
                        {
                            long lBytesThis = lBytesToGo;
                            if (lBytesThis > BUFFERSIZE)
                                lBytesThis = BUFFERSIZE;

                            fread(text,
                                  1,           // size of block
                                  lBytesThis,  // no. of blocks
                                  _OldArchive);     // source stream

                            if ((long)fwrite(text,
                                             1,      // size of block
                                             lBytesThis,  // no. of blocks
                                             _Archive)    // target stream
                                    < lBytesThis)
                                // error occured (probably disk full):
                                if (_pfnCallback != NULL)
                                    (*_pfnCallback)(CBM_ERR_WRITE,
                                                    CBREC_CANCEL,
                                                    &filehead);

                            lBytesToGo -= BUFFERSIZE;
                        }
                        free (text);

                        // update package header
                        pPackHeaderThis->files++;
                        pPackHeaderThis->origsize += filehead.origsize;
                        pPackHeaderThis->compsize += filehead.compsize;
                        break; // for (; start != end; start++)

                        // the source file pointer is now on the next file header
                    }
                }

                // if we didn't find the file, advance the pointer
                if (start == end)
                {
                    fseek(_OldArchive,
                          filehead.compsize,
                          SEEK_CUR);
                    break;
                }
            }
        }
        pPackHeaderThis->pos = packpos;
    } // end for (s = 1; ...

    // Now, write the correct headers to this archive
    fseek(_Archive, 0, SEEK_SET);
    WriteArcHeader(_Archive);
    WritePackHeaders(_Archive);
    fclose(_Archive);

    // Remove the old archive and replace it with the new one.
    if (_TempFile)
    {
        fclose(_OldArchive);
        unlink(_ArchiveName);
        rename(_TempArchiveName, _ArchiveName);
        _TempFile = 0;
    }
}

/*
 *@@ WriteArcHeader:
 *      writes out the current member archive header into the
 *      specified file.
 *
 *      Writes to the current position!
 */

void WIArchive::WriteArcHeader(FILE *file)
{
    bz_stream z;
    char *pTempBuffer = 0;

    // First of all, set the archive identifiation and such thingies
    _ArcHeader.v1 = WI_VERIFY1;
    _ArcHeader.v2 = WI_VERIFY2;
    _ArcHeader.v3 = WI_VERIFY3;
    _ArcHeader.v4 = WI_VERIFY4;
    _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION;
    _ArcHeader.os = WI_OS_OS2;
    if (_Script == NULL)
    {
        _ArcHeader.compscript = _ArcHeader.origscript = 0;
    }
    else
    {
        _ArcHeader.origscript = strlen(_Script) + 1;
        int ScriptBufSize = _ArcHeader.origscript + 100;
        pTempBuffer = new char[ScriptBufSize];

        // ----- Compress the installation script
        z.bzalloc = 0;
        z.bzfree  = 0;
        z.opaque = NULL;
        z.bzalloc = NULL;
        z.bzfree = NULL;
        z.next_in = _Script;
        z.avail_in = _ArcHeader.origscript;
        z.next_out = pTempBuffer;
        z.avail_out = ScriptBufSize;
        BZ2_bzCompressInit (&z,
                            1,          // blockSize100k
                            0,          // verbosity
                            60);        // workFactor
        BZ2_bzCompress (&z, BZ_FINISH);
        _ArcHeader.compscript = ScriptBufSize - z.avail_out;
        BZ2_bzCompressEnd (&z);
    }

    if (fwrite((char*)&_ArcHeader,
               1,
               sizeof(WIArcHeader),
               file)
        < sizeof(WIArcHeader))
    {
        if (_pfnCallback != NULL)
            (*_pfnCallback)(CBM_ERR_WRITE,
                            CBREC_CANCEL,
                            NULL);
    }
    if (_Script != NULL)
    {
        fwrite (pTempBuffer, 1, _ArcHeader.compscript, file);
        delete[] pTempBuffer;
    }
    if (_Extended != NULL)
        fwrite(_Extended, 1, _ArcHeader.extended, file);
}

/*
 *@@ WritePackHeaders:
 *      writes all package headers into the specified archive
 *      file.
 *
 *      Writes to the current position!
 */

void WIArchive::WritePackHeaders(FILE *file)
{
    list<WIPackHeader *>::iterator start, end;

    start = _PackList.begin();
    end = _PackList.end();
    for (; start != end; start++)
    {
        fwrite((char*)(*start),
               1,
               sizeof(WIPackHeader),
               file);
    }
}

/*
 *@@ add:
 *      public method for adding a file to the to-do list.
 */

void WIArchive::add(const char *filename,  // in: file to add
                    const char *arcname,   // in: name of file to store in archive
                    short package)         // in: package index to add to
{
    WIFile *f;

    f = new WIFile;
    f->extra = new char[strlen(filename) + 1];
    f->name = new char[strlen(arcname) + 1];
    strcpy(f->extra, filename);
    strcpy(f->name, arcname);
    f->package = package;
    _ToDoList.push_back(f);
}

/*
 *@@ close:
 *      closes the archive. This calls WIArchive::Update in turn.
 */

void WIArchive::close()
{
    if (!_fOpenForReadOnly)
        Update();
            // this will in turn call fclose
    else
        fclose(_Archive);
}

/*
 *@@ closeAbort:
 *      closes the archive without performing an archive
 *      update. Use this for signal handlers to quickly
 *      close open files.
 *
 *@@added V0.9.4 (2000-07-22) [umoeller]
 */

void WIArchive::closeAbort()
{
    fclose(_Archive);
    if (_TempFile)
    {
        fclose(_OldArchive);
        _TempFile = 0;
    }
}

/*
 *@@ getArcHeader:
 *      returns a const pointer to the member archive header.
 */

const WIArcHeader* WIArchive::getArcHeader()
{
    return &_ArcHeader;
}

/*
 *@@ getFileList:
 *      returns a pointer to the member files list.
 */

list<WIFile*>* WIArchive::getFileList()
{
    return &_FileList;
}

/*
 *@@ getPackList:
 *      returns a pointer to the member list of packages.
 */

list<WIPackHeader*>* WIArchive::getPackList()
{
    return &_PackList;
}

/*
 *@@ getScript:
 *      returns the decompressed install script or NULL if
 *      none was found.
 */

const char *WIArchive::getScript()
{
    return _Script;
}

/*
 *@@ open:
 *      opens the archive and reads the basic stuff in.
 *
 *      This calls WIArchive::ReadArcHeader, WIArchive::ReadPackHeaders,
 *      and WIArchive::ReadFilenames in turn.
 *
 *      If (mode == 0), the archive is opened in "read-only" mode.
 *
 *      If (mode == 1), the archive is opened in "read-write" mode.
 *      In that case, this method calls WIArchive::OpenTemp to create
 *      a temporary file.
 *
 *      Returns 0 on errors.
 */

int WIArchive::open(const char *filename,
                    int mode)
{
    _fOpenForReadOnly = (mode == 0);     // we need to remember
                                         // whether the file was read-only for close()

    strcpy(_ArchiveName, filename);
    _Archive = fopen(filename, "rb");
    if (_Archive != NULL)
    {
        // The file exists - check if it's a valid archive
        if (ReadArcHeader(_Archive) == 0)
            return 0;               // Apparently it wasn't valid
        else
        {
            // Read all the files and packages in this archive
            ReadPackHeaders(_Archive);
            ReadFilenames(_Archive);
        }
        // If we are allowed to r/w, then create a new archive
        if (mode == 1)
        {
            fclose(_Archive);
            _Archive = OpenTemp(filename, "w+b");
            if (_Archive == NULL)
                // OK, so we couldn't create a new archive. Run away!
                return 0;
            // UsingTemp = 1;
            _OldArchive = fopen(filename, "rb");
        }
    }
    else
    {
        // Are we permitted to create a new archive?
        if (mode == 0)
        {
            _lastError = WIERR_FILENOTFOUND;   // added V0.9.2 (2000-03-11) [umoeller]
            return 0;               // No, we're not
        }
        _Archive = fopen(filename, "w+b");
    }
    return 1;
}

/*
 *@@ remove:
 *      removes the specified file from the specified package.
 */

void WIArchive::remove(const char *filename,
                       short package)
{
    list<WIFile*>::iterator start, end;

    // First of all, check if the file exists in the to-do list
    start = _ToDoList.begin();
    end = _ToDoList.end();
    for (; start != end; start++)
    {
        WIFile* pFileThis = *start;
        if (    (strcmp(pFileThis->name, filename) == 0)
             && (pFileThis->package == package)
           )
        {
            if (pFileThis->name != NULL)
                delete[] pFileThis->name;
            if (pFileThis->extra != NULL)
                delete[] pFileThis->extra;
            delete pFileThis;
            return;
        }
    }

    // was not on to-do list:
    // OK, remove it from the file list...
    start = _FileList.begin();
    end = _FileList.end();
    for (; start != end; start++)
    {
        WIFile* pFileThis = *start;
        if (   (strcmp (pFileThis->name, filename) == 0)
            && (pFileThis->package == package)
           )
        {
            if (pFileThis->name != NULL)
                delete[] pFileThis->name;
            if (pFileThis->extra != NULL)
                delete[] pFileThis->extra;
            delete pFileThis;
            break;
        }
    }
}

/*
 *@@ setArcHeader:
 *      copies the specified archive header data
 *      to the member archive header and fixes
 *      the data, if necessary.
 *
 *      head.wi_revision_needed is always updated
 *      to the present build level.
 */

void WIArchive::setArcHeader(const WIArcHeader &head)
{
    _ArcHeader.wi_revision_needed = WIARCHIVE_REVISION_NEEDED;
    strcpy(_ArcHeader.inet_address, head.inet_address);
    strcpy(_ArcHeader.name_app, head.name_app);
    strcpy(_ArcHeader.name_dev, head.name_dev);
    strcpy(_ArcHeader.path, head.path);
    _ArcHeader.rev_app = head.rev_app;
    _ArcHeader.os = head.os;

    // fix archive filename; if it has a trailing "\",
    // remove that
    char *cp = _ArcHeader.path;
    if (*cp != 0)
    {
        while (*cp != '\0')
            cp++;
        cp--;
        if (*cp != '\\')
        {
            cp++;  *cp = '\\';
            cp++;  *cp = '\0';
        }
    }
}

/*
 *@@ setCallbackFunc:
 *      sets the WIArchive callback function, which is
 *      used for all kinds of notification to the caller
 *      code, such as progress display and errors.
 */

void WIArchive::setCallbackFunc(PFNWICALLBACK pfnCallbackNew)
{
    _pfnCallback = pfnCallbackNew;
}

/*
 *@@ setPackage:
 *      sets some basic information about a package.
 *
 *      If no package with the specified index exists
 *      yet, a new package is created. Otherwise, the
 *      existing package is updated.
 */

void WIArchive::setPackage(short package,
                           const char *name)
{
    list<WIPackHeader *>::iterator start, end;
    WIPackHeader *p;
    int i = 0;

    // See if we already have a package with this number
    start = _PackList.begin ();
    end = _PackList.end ();
    for (; start != end; start++)
        if ((**start).number == package)
        {
            p = *start;
            i++;
            break;
        }
    if (i == 0)
        p = new WIPackHeader;

    // Store this information
    strcpy(p->name, name);
    p->number = package;
    p->files = p->origsize = p->compsize = 0;
    if (i == 0)
        // new package:
        _PackList.push_back(p);
}

/*
 *@@ setScript:
 *      copies an installation script to the archive.
 *      If a script exists already, it is replaced.
 */

void WIArchive::setScript(const char *cp)
{
    int i = strlen(cp);
    if (_Script != NULL)
        // exists already:
        delete[] _Script;
    _Script = new char[i + 1];
    strcpy(_Script, cp);
}

/*
 *@@ unpack:
 *      unpacks the specified package to disk.
 *
 *      This calls WIArchive::MakeDirectories.
 *
 *      Preconditions:
 *
 *      -- The packages list must have been initialized.
 *
 *      Returns 0 on errors, e.g. if we ran out of disk space.
 */

short WIArchive::unpack(short package)
{
    short irc = 0;      // error

    list<WIPackHeader*>::iterator start, end;
    char filename[256];
    int fFound = 0;

    // Find the correct package information and start processing the info
    WIPackHeader *pPckHeader = 0;
    start = _PackList.begin ();
    end = _PackList.end ();
    for (; start != end; start++)
    {
        pPckHeader = *start;
        if (pPckHeader->number == package)
        {
            // found:
            fFound = 1;
            break;
        }
    }

    if (fFound) // V0.9.6 (2000-11-23) [umoeller]
    {
        // move file pointer to position in archive,
        // as specified in the package header
        fseek(_Archive, pPckHeader->pos, SEEK_SET);
        // get no. of files
        long files = pPckHeader->files;

        // The uncompress-all-files-we-can-find loop
        while (files--)
        {
            // read in file header
            WIFileHeader fhead;
            fread((char*)&fhead,
                  1,
                  sizeof(WIFileHeader),
                  _Archive);

            // check for magic bytes
            if (fhead.magic != WIFH_MAGIC)  // wiarchive.h
            {
                if (_pfnCallback != NULL)
                    // magic byte doesn't match:
                    // report decompression error
                    (*_pfnCallback)(CBM_ERR_ZLIB,
                                    -99,
                                    &fhead);
                        // and stop!
                return (0);
            }

            // else go on
            strcpy(filename, _ArcHeader.path);
            strcat(filename, fhead.name);
            MakeDirectories(filename);         // Make directories, if they don't exist

            int iStatus = 0;
                // error status, meaning:
                //  0       OK
                //  >0      file open/write error
                //  <0      decompression error (zlib)
            int file_exists_rc = CBRC_PROCEED;
                // default is still to overwrite the file,
                // if no callback exists

            // call the "next file" callback.
            if (_pfnCallback != NULL)
                // This allows the front-end to do two things:
                // a)   update the "current file" display
                // b)   check for whether that file exists already:
                //      -- if not, we return CBRC_PROCEED;
                //      -- if it does exist, we ask the user for whether the file
                //         may be overwritten; if so, we try to delete it.
                //         Otherwise, we return CBRC_SKIP, and the back-end does
                //         not touch the file.
                // This new approach (Alpha #3) is required because sometimes
                // the target file which already exists is _locked_, e.g. if it's
                // a system DLL in use. We then need to do a DosReplaceModule,
                // which we cannot do in the back-end if we want to keep it
                // platform-independent. Sorry for the hassle.
                file_exists_rc = ((*_pfnCallback)(CBM_NEXTFILE, 0, &fhead));

            // error checking added
            long int lCurPos = ftell(_Archive);
            if (file_exists_rc == CBRC_PROCEED)
            {
                // do-while loop in case the user presses
                // "Retry" on errors below
                // normally, we don't loop
                int iRepeat = 0;
                do {
                    iRepeat = 0;

                    // create the target file!
                    _File = fopen(filename, "wb");

                    if (_File)
                    {
                        // file successfully opened for writing:
                        // **** go!
                        switch (fhead.method)
                        {
                            case 0: // stored
                                iStatus = Extract(&fhead); break;
                            case 1: // compressed
                                iStatus = Expand(&fhead);  break;
                            default:
                                return 0;
                        }
                        fclose(_File);
                    }
                    else
                        // file could not be opened:
                        iStatus = 99;

                    // the following blocks added
                    if (iStatus == 0)
                    {
                        // no error occured: set file date/time
                        struct utimbuf utb;
                        // utimbuf is declared in sys\utime.h as follows (VAC++):
                        /*   struct utimbuf
                             {
                                time_t actime;          // access time
                                time_t modtime;         // modification time
                             }; */

                        utb.actime  = fhead.lastwrite;
                        utb.modtime = fhead.lastwrite;
                        utime(filename, &utb);
                    }
                    else
                    {
                        // iStatus is
                        //      99      if the file could not be opened above
                        //   other > 1  if Extract or Expand couldn't write to the file
                        //     < 1      if the zlib had a decompression error

                        // default: skip the file (see below)
                        file_exists_rc = CBRC_SKIP;

                        if (_pfnCallback != NULL)
                        {
                            if (iStatus < 0)
                            {
                                // decompression error:
                                (*_pfnCallback)(CBM_ERR_ZLIB,
                                                iStatus,
                                                &fhead);
                                    // and stop!
                            }

                            // else: error related to file handling
                            int irc = 0;
                            irc = (*_pfnCallback)(CBM_ERR_WRITE,
                                            CBREC_ABORTRETRYIGNORE, // allow all selections
                                            &fhead);
                            switch (irc)
                            {
                                // CBRC_ABORT is handled in the front-end
                                case CBRC_RETRY:    iRepeat = 1; break;
                                            // stay in the do-while loop
                                // if the user presses "Ignore", the default
                                // CBRC_SKIP will be used (above)
                            }
                        }
                    }
                } while (iRepeat);  // only if CBRC_RETRY is returned on errors
            }

            if (file_exists_rc == CBRC_SKIP)
                // this is the case if
                // a)   CBM_NEXTFILE returned CBRC_SKIP
                //      (if the target file existed)
                // b)   CBM_ERR_WRITE returned CBRC_SKIP
                //      (if an error occured writing the file)
            {
                // file is to be skipped: move the file pointer
                // past the file data, or we'll get garbage for
                // the next file header
                fseek(_Archive,
                      lCurPos + fhead.compsize,
                      SEEK_SET);
            }
        } // while (files--)

        irc = 1;        // success
    } // end if (fFound) // V0.9.6 (2000-11-23) [umoeller]

    return (irc);
}

/*
 *@@ forAllFiles:
 *      this calls pfnWICallback for all files which belong to the
 *      specified package. If you want to list all the files, you
 *      need to call this method for each package (by going thru
 *      the package list).
 *
 *      This method accesses the archive file on disk, because
 *      the class-internal list does not have the WIFileHeader's,
 *      which we therefore must read in here. For large archives,
 *      this function can take a bit of time.
 *
 *      The callback gets called with the following:
 *
 *      --   WIFileHeader* pwifh  the file header of the file in question
 *
 *      --   unsigned long ulUser the user parameter passed to this function;
 *                                you can use this for a pointer to some
 *                                internal data
 *
 *      This function returns the number of files which were found,
 *      or 0 if no files were found, or -1 if an error occured.
 */

short WIArchive::forAllFiles(WIPackHeader* pckHeader,   // in: package to list files for
                             PFNFORALLFILESCALLBACK pfncbForAllFiles,
                                              // in: callback to call for each file
                             unsigned long ulUser)
                                              // in: user parameter passed to callback
{
    short sFileCount = 0;

    if (pckHeader)
    {
        // found our package:
        fseek(_Archive, pckHeader->pos, SEEK_SET);

        long files = pckHeader->files;

        WIFileHeader fhead;

        // go thru all the files
        while (files--)
        {
            fread((char*)&fhead,
                  1,
                  sizeof(WIFileHeader),
                  _Archive);

            // check for magic bytes
            if (fhead.magic != WIFH_MAGIC)  // wiarchive.h
            {
                // stop!
                return (-1);
            }

			// remember file position
			// moved before callback call (2001-03-07) [cbockemuehl]
            long int lCurPos = ftell(_Archive);

            // else: OK, call callback
            (*pfncbForAllFiles)(&fhead, ulUser);

            // next file: skip data
            fseek(_Archive,
                  lCurPos + fhead.compsize,
                  SEEK_SET);
            sFileCount++;
        }
    }

    return (sFileCount);
}


