/*
 * layout.c - manipulating layouts
 */

#include <stdio.h>
#include <stdlib.h>

#include "nup.h"

/*
 * Default Layout
 */

static  LAYREC  defaultPort = {
    0,                                  /* layout number            */
    "GOTHIC HeiseiKakuGothic-W5-H-90",  /* as default font name     */
    TRUE,                               /* prefer printer font      */
    86,                                 /* char height/line height  */
    66,                                 /* lines in a page          */
    80,                                 /* columns in a page        */
    20,                                 /* left margin              */
    20,                                 /* right margin             */
    20,                                 /* top margin               */
    20,                                 /* bottom margin            */
    10,                                 /* horz. spacing            */
    10,                                 /* virt. spacing            */
    LAYOUT_ORDER_ROW,                   /* page order               */
    TRUE,                               /* draw page frame          */
    TRUE,                               /* draw center marker       */
    0                                   /* reserved                 */
} ;

static  LAYREC  defaultLand = {
    0,                                  /* layout number            */
    "GOTHIC HeiseiKakuGothic-W5-H-90",  /* as default font name     */
    TRUE,                               /* prefer printer font      */
    85,                                 /* char height/line height  */
    66,                                 /* lines in a page          */
    132,                                /* columns in a page        */
    20,                                 /* left margin              */
    20,                                 /* right margin             */
    20,                                 /* top margin               */
    20,                                 /* bottom margin            */
    10,                                 /* horz. spacing            */
    10,                                 /* virt. spacing            */
    LAYOUT_ORDER_COL,                   /* page order               */
    TRUE,                               /* draw page frame          */
    TRUE,                               /* draw center marker       */
    0                                   /* reserved                 */
} ;

/*
 * Table (array) of Layouts
 */

#define NLAYOUTS  6

static  int     defLayout = LAYOUT_PORT2 ;
static  LAYREC  tabLayout[NLAYOUTS]      ;

/*
 * layoutInit - Initialize Layout Management
 */

int     layoutInit(HAB hab)
{
    HINI    hini    ;
    LONG    i, len  ;
    UCHAR   key[32] ;
    BOOL    stat    ;
    LAYREC  layout  ;

    if ((hini = PrfOpenProfile(hab, ProfilePath)) == NULLHANDLE) {
        return -1 ;
    }

    /*
     * Listup Layout Records
     */
     
    for (i = 0 ; i < NLAYOUTS ; i++) {
        sprintf(key, "LAYOUT%02d", i) ;
	stat = PrfQueryProfileData(hini, 
	                ProgramName, key, (PVOID) &layout, &len) ;
        if (stat != FALSE && len == sizeof(LAYREC)) {
            memcpy(&tabLayout[i], &layout, sizeof(LAYREC)) ;
        } else if (i >= LAYOUT_PORT1 && i <= LAYOUT_PORT4) {
	    memcpy(&tabLayout[i], &defaultPort, sizeof(LAYREC)) ;
        } else if (i >= LAYOUT_LAND1 && i <= LAYOUT_LAND4) {
	    memcpy(&tabLayout[i], &defaultLand, sizeof(LAYREC)) ;
        }
	tabLayout[i].index = i ;
    }

    /*
     * get default selection
     */

    i = LAYOUT_PORT2 ;  /* as default layout */
    
    stat = PrfQueryProfileData(hini, 
                        ProgramName, "DEFLAYOUT", (PVOID) &i, &len) ;
    if (stat == TRUE && len == sizeof(LONG) && i >= 0 && i < NLAYOUTS) {
        defLayout = i ;
    }

    PrfCloseProfile(hini) ;

    return 0 ;
}

/*
 * layoutDone - finalize Layout Management
 */

void    layoutDone(HAB hab)
{
    HINI    hini    ;
    LONG    i, len  ;
    UCHAR   key[32] ;
    BOOL    stat    ;
    LAYREC  layout  ;
    
    if ((hini = PrfOpenProfile(hab, ProfilePath)) == NULLHANDLE) {
        return ;
    }

    /*
     * Save (Update) Layouts
     */
     
    for (i = 0 ; i < NLAYOUTS ; i++) {
        sprintf(key, "LAYOUT%02d", i) ;
	PrfWriteProfileData(hini, ProgramName, 
	            key, (PVOID) &tabLayout[i], sizeof(LAYREC)) ;
    }

    /*
     * Also save current layout selection
     */

    PrfWriteProfileData(hini, ProgramName,
                "DEFLAYOUT", (PVOID) &defLayout, sizeof(ULONG)) ;

    PrfCloseProfile(hini) ;
    return ;
}

/*
 * layoutInfo - refer to layout data
 */

LAYPTR  layoutInfo(int layout)
{
    if (layout < 0 || layout >= NLAYOUTS) {
        return NULL ;
    }
    return &tabLayout[defLayout = layout] ;
}

/*
 * layoutCur - current layout
 */

int     layoutCur(void)
{
    return defLayout ;
}

/*
 * layoutPrev, layoutNext - prev. next layout
 */

int     layoutPrev(int layout)
{
    if ((layout - 1) < 0) {
        return NLAYOUTS - 1 ;
    }
    return layout - 1 ;
}

int     layoutNext(int layout)
{
    if ((layout + 1) >= NLAYOUTS) {
        return 0 ;
    }
    return layout + 1 ;
}
