// Freeware APRS tcpip server.  7-21-96
//Version 1.1  1-7-1997
// Copyright 1996 by Dale A. Heatherington, WA4DSY
// Written exclusively for OS/2 Warp
//
// Compiled with IBM Visual Age C++
// IBM tcpip programmers tool kit required.
// Edited with Visual SlickEdit with tabs every 3 chars.

//Version 1.1 - Added UDP listen code and enabled logged on clients to send data into the system.

/*

FUNCTIONS PERFORMED:

This program gets data from a TNC connected to a serial port
and sends it to all clients who have connected to port 14579
from a tcpip network. Clients can use telnet to watch raw
TNC data or JavAPRS to view the data plotted on a map.

A history of TNC data going back 30 minutes is kept in
memory and delivered to each user when he connects.  This
history data is filtered to remove duplicates and certain
other unwanted information.

REMOTE CONTROL of the TNC
The server system operator can access the TNC remotely via
telnet.  Simply telnet to the server and, after the server
finishes sending the history file, enter an control-R (remote-control).
The user will be prompted for a user name and password.  If these
are valid the sysop will be able to type commands to the TNC.  Each
line is buffered before it is sent to the TNC and data from the TNC
is not transmitted until a CR is sent.  Unfortunately TNCs prompt
with "cmd:" without a CR so this prompt will not be visible.
When the sysop is finished with the TNC remote control session
he can exit with a control-Q.  The session is still connected
but the TNC is inacessable.  Type control-D to disconnect.

Note that the TNC data is still being transmitted to all other
clients which are connected during the remote control session
so don't enter data you don't want the world to see.

*/



/*	BUGS

	The bug which caused this to freeze up after 1 to 45 hours on line
   was triggered when a user connection was broken without a proper
	shutdown handshake.  This server kept sending data to the dead
	socket until it ran out of buffers and blocked the send() call.
	Since the blocked send() call was in the SendToAll() loop no
	other users could get any data and the Async queue started backing up.
	If no other memory allocation failures occured the server would
	start working again after minutes to hours depending on when tcpip finally
	released the dead socket.

	Using non-blocking mode for the user sockets and testing the
	return code for "would have blocked" and closing the offending socket
	fixes the problem.

	This code has run without any problems for over 2months now.
*/


/*-------------------------------------------------------------------------------*/
/* Here's some off the air data for reference */
//
//$GPRMC,013426,A,3405.3127,N,08422.4680,W,1.7,322.5,280796,003.5,W*73
/*
KD4GVX>APRS,WB4VNT-2,N4NEQ-2*,WIDE:@071425/Steve in Athens, Ga.
N4NEQ-9>APRS,RELAY*,WIDE:/211121z3354.00N/08418.04W/GGA/NUL/APRS/good GGA FIX/A=
000708
N4NEQ-9>APRS,RELAY,WIDE*:/211121z3354.00N/08418.04W/GGA/NUL/APRS/good GGA FIX/A=
000708
KE4FNU>APRS,KD4DLT-7,N4NEQ-2*,N4KMJ-7,WIDE:@311659/South Atlanta via v7.5a@Stock
bridge@DavesWorld
KC4ELV-2>APRS,KD4DLT-7,N4NEQ-2*,WIDE:@262026/APRS 7.4 on line.
KC4ELV-2>APRS,KD4DLT-7,N4NEQ-2,WIDE*:@262026/APRS 7.4 on line.
N4QEA>APRS,SEV,WIDE,WIDE*:@251654/John in Knoxville, TN.  U-2000
WD4JEM>APRS,N4NEQ-2,WIDE*:@170830/APRS 7.4 on line.
KD4DLT>APRS,KD4DLT-7,N4NEQ-2*,WIDE:@201632/APRS 7.6f on line.
N4NEQ-3>BEACON,WIDE,WIDE:!3515.46N/08347.70W#PHG4970/WIDE-RELAY Up High!!!
N4NEQ-3>BEACON,WIDE*,WIDE:!3515.46N/08347.70W#PHG4970/WIDE-RELAY Up High!!!
KD4DKW>APRS:@151615/APRS 7.6f on line.
KE4KQB>APRS,KD4DLT-7,WIDE*:@111950/APRS 7.6 on line.
WB4VNT>APRS,WB4VNT-2,N4NEQ-2*,WIDE:@272238/7.5a on line UGA rptr 147.000+
N4YTR>APRS,AB4KN-2*,WIDE:@111443zEd - ARES DEC National Weather Service, GA
N4YTR>APRS,AB4KN-2,WIDE*:@111443zEd - ARES DEC National Weather Service, GA
W6PNC>APRS,N4NEQ-2,WIDE:@272145/3358.60N/08417.84WyJohn in Dunwoody, GA
WA4DSY-9>APRS,WIDE:$GPRMC,014441,A,3405.3251,N,08422.5074,W,0.0,359.3,280796,003
.5,W*77
N6OAA>APRS,GATE,WIDE*:@280144z4425.56N/08513.11W/ "Mitch", Lake City, MI

-------------------------------------------------------------------------------------*/


#define INCL_DOS
#define INCL_DOSDEVIOCTL
#define INCL_DOSDATETIME
#define INCL_DOSNMPIPES
#define INCL_DOSQUEUES
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES

#include <os2.h>

#pragma pack(1)

#include <assert.h>
#include <process.h>
#include <conio.h>
#include <time.h>

#include <stdio.h>
#include <ctype.h>

#include <io.h>
#include <sys\stat.h>
#include <fcntl.h>
#include <share.h>

#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <idate.hpp>
#include <itime.hpp>

#include <fstream.h>
#include <iostream.h>
#include <strstrea.h>
#include <iomanip.h>

//tcpip header files
#define OS2
#include <netinet/in.h>
#include <sys/socket.h>
#include <netdb.h>
#include <utils.h>
#include <types.h>
#include <sys/ioctl.h>


// #define SERVERPORT 14579			/* APRS server port */

#define CMD_END_THREAD -1L
#define SRC_TNC	-2L
#define SRC_USER	-3L
#define SRC_INTERNAL -4L

#define MAXCLIENTS 30
#define BUFSIZE 1024
#define SBUFSIZE 1024
#define PASSWORDFILE "password.txt"
#define LOGFILE "AprsServ.log"
#define WELCOME "welcome.txt"
#define TNC_INIT  "init.tnc"
#define TNC_RESTORE "restore.tnc"
#define SAVE_HISTORY "history.txt"
#define TEST "WA4DSY>APRS,WIDE:!3405.31N/08422.46WyWA4DSY APRS Internet Server running on OS/2 Warp.\r\n"

//-------------------------------------------------------------------
#define	NONEp	0
#define ODDp	1
#define EVENp	2
#define MARKp	3
#define SPACEp	4


//---------------------------------------------------------------------
typedef
struct	ComStruct
		{  char	name[6];        // name of this port
			HEV	hevPacketRcvd;  // event semaphore  rx
			HEV	hevPacketSent;  // event semaphore  tx
			HFILE	hCOM;           // file handle for com port
			ULONG	baud;				 // baud rate
			BYTE	parity;         // NONEp,ODDp,EVENp,MARKp,SPACEp
			BYTE	stopbits;       // 1 or 2
			BYTE	databits;		 // 6,7,8
			BYTE	brk;				 // 0=no break, 1=break
		  } ;

typedef
struct History;

struct	History
			{	int	   ttl;     //Down counter for max age in minutes (time-to-live)
				int		count;	//Down counter for max entries for this call sign
				History 	*last;	//pointer to previous History structure
				History 	*next;	//pointer to next History structure
				char		*data;	//pointer to data char string
			};


void _Optlink ReadCom(void* vp) ;	  //Com port read thread
int SendFiletoTNC(HFILE hCom, char *szName);
int SendFiletoClient(int session, char *szName);
void PrintStats(ostream &os) ;
void printhex(char *cp, int n);
void RemoveDups(char *cp);



ComStruct		COMx;
const char		szServerID[] = "OS/2 APRS Server: ";
const char		szJAVAMSG[] = "SERVER>JAVA:javaMSG  :";
TID				tidReadCom;
HQUEUE			hqAsync;            //Async thread Queue handle
HEV				_hevStarted;
HMTX				hmtxLog;			//Mutual exclusion semi for WriteLog function
HMTX				hmtxSendFile;
HMTX				hmtxSend;		//protect socket send()  (May not be needed...)

int *sessions;   //points to array of active socket descriptors
BOOL ShutDownServer;
int ConnectedClients;
ULONG WatchDog, tickcount, TotalConnects, TotalChars, TotalLines, MaxConnects;

int ServerPort;
int ServerSocket;
HEV	NotSending;
ULONG ulNScnt = 0;

History *pHead, *pTail;
int ItemCount;
HMTX	hmtxHistory;

int ttlDefault = 30;	  			//Default time to live of a history item (30 minutes)
int CountDefault = 7;		   //Max of 7 instances of one call sign in history list

char passDefault[] = "aprs"	  ;
char userDefault[] = "user";

//---------------------------------------------------------------------
void CreateHistoryList()
{
	 pHead = NULL;
	 pTail = NULL;
	 ItemCount = 0;
}
//---------------------------------------------------------------------
//return TRUE if destination of packet matches "ref"
//This is for filtering out unwanted packets
BOOL CmpDest(const char *line, const char *ref)
{	BOOL rv = FALSE;
	char *cp = new char[strlen(ref)+3];
	
	strcpy(cp,">");
	strcat(cp,ref);
	strcat(cp,",");
	if (strstr(line,cp) !=	NULL) rv = TRUE;
	delete cp;
	return rv;
}
//---------------------------------------------------------------------
//Returns true if "call" matches first string in "s"
//"call" must be less than 31 chars long.
BOOL callsign(char *s, const char *call)
{
	char cp[17];
	if (strlen(call) > 14) return FALSE;
	strncpy(cp,call,16);
	strncat(cp,">",16);
	char *ss = strstr(s,cp);
	if (ss == s) return TRUE ;else return FALSE;

}
//---------------------------------------------------------------------
//Compares two packets and returns TRUE if the source call signs are equal
BOOL CompareSourceCalls(char *s1, char *s2)
{
	char call[12];
	strncpy(call,s2,10);
	char *eos = strchr(call,'>');
	if (eos != NULL) eos[0] = NULL; else return FALSE;
	return callsign(s1,call);
}

//---------------------------------------------------------------------
// This sets the time-to-live and max count values for each packet received.
// Only defaults are implimented but additional code can customize
// these values different items.
void GetMaxAgeAndCount(char *buf, int *MaxAge, int *MaxCount)
{

	*MaxAge = ttlDefault;
	*MaxCount = CountDefault;

	if (callsign(buf,"WA4DSY-9")) {  *MaxAge = 120; *MaxCount = 30; }
   if (callsign(buf,"KO4HD-9")) { *MaxAge = 5760; *MaxCount = 15; }
				
}

//---------------------------------------------------------------------
void AddHistoryItem(void *vp, int ttl, int maxcount)
{
	char *item = (char*)vp;

	if (strchr(item,'>') == NULL) return ;		//must have a ">" in the header
   if (CmpDest(item,"ID") == TRUE) return ;   //Don't save any ID packets
	if (CmpDest(item,"CQ") == TRUE) return;   //No CQs either
	if (strstr(item,":ack") != NULL) return;  //No ack packets
   if (strstr(item,":ACK") != NULL) return;  //No ack packets


   DosRequestMutexSem(hmtxHistory,SEM_INDEFINITE_WAIT);

  	History *hp = new History;
   int n = strlen(item)+1;
   hp->data = new char[n];
   strcpy(hp->data,item);	 //Copy packet into data structure
   hp->ttl = ttl;				 //set ttl at time-to-live
	hp->count = maxcount;	//set maximum number of items from same call sign
   	
	if (pTail == NULL)		 //Starting from empty list
		{
			pTail = hp;
			pHead = hp;
			hp->next = NULL;
			hp->last = NULL;
			ItemCount++;
		   DosReleaseMutexSem(hmtxHistory);
			return ;
		}

  	pTail->next = hp;			//List has at least one item in it.
	hp->last = pTail;
	hp->next = NULL;
	ItemCount++;
	pTail = hp;
	RemoveDups(hp->data);		  //Remove any previous duplicates and/or items with zero count field
	DosReleaseMutexSem(hmtxHistory);
	return;
}
//---------------------------------------------------------------------
//Don't call this from anywhere except DeleteOldItems().
void DeleteHistoryItem(History *hp)
{

  	History *li = hp->last;
	History *ni = hp->next;
	
	if (hp != NULL)
		{  if (li != NULL) li->next = ni;
			if (ni != NULL) ni->last = li;
			if (hp == pHead) pHead = ni;
			if (hp == pTail)  pTail  = li;

			if (hp->data != NULL) delete hp->data;	 //Delete the data
			delete hp;									  	 //Delete the history structure
			ItemCount--;
	   }
	
   return;

}

//-----------------------------------------------------------------------
//Call this once per minute
//This decrements the ttl field then deletes items when the ttl field is zero.
int DeleteOldItems(void)	
{	int delCount = 0;


	if (pHead == NULL) return 0;

	DosRequestMutexSem(hmtxHistory,SEM_INDEFINITE_WAIT);

	History *hp = pHead;
	History *pNext;

	do
		{	assert(hp != NULL);
		   hp->ttl--;
			pNext = hp->next;
			if (hp->ttl <= 0 )
					{   DeleteHistoryItem(hp);
						 delCount++;
					}
			if (pNext == NULL) break;
			hp = pNext;
		}while (TRUE);
	
 	DosReleaseMutexSem(hmtxHistory);
	return delCount;
}
	
//-------------------------------------------------------------------------
//Removes one duplicate packet from list starting at next to last one
//and working backwards. Must be called from MutexSem protected function.
//Also, if the call signs match, the items count field is decremented.
//If the count is zero the item is deleted.  This limits the maximum
//packets from one call sign that can be in the list.
void RemoveDups(char *cp)
{	int x;
	BOOL z;
	char *s1,*s2;
	if (pTail == pHead) return;

	History *hp = pTail->last;
	s1 = strchr(cp,':');								   //Find the first colon in reference string

	do
		{	x = strncmp(cp,hp->data,5);				//First check only first 5 chars for speed
			if (x == 0)										//If they match then check the rest...
				{
					s2 = strchr(hp->data,':');			//find the first colon in search string
					if ((s2 != NULL) && (s1 != NULL))
						{
						  x = strcmp(s1,s2);				//compare all after colon
						}else
							x = strcmp(cp,hp->data);	//if no colon just do a full string compare

				   if (CompareSourceCalls(cp,hp->data))	//If call signs match then decrement the count
						{	hp->count--;
							if (hp->count <= 0) z = TRUE; else z = FALSE;
						}
				}
			if ((x == 0) || (z == TRUE) || (hp->last == NULL)) break;  //If all match or end of data then break
			hp = hp->last;									//Go back in time through list
		}while (TRUE) ;

	if ((x == 0) || (z == TRUE)) DeleteHistoryItem(hp);	//Delete the duplicate or item with zero count

}

//--------------------------------------------------------------------------
//Send the history items to the user when he connects
int SendHistory(int session)
{
	
	int 	rc,n,count=0;
   static int SendingHistory=0;

	if (pHead == NULL) return 0;			//Nothing to send

   DosRequestMutexSem(hmtxHistory,SEM_INDEFINITE_WAIT);	 // grab mutex semaphore

	History *hp = pHead;											//Start at the beginning of list

	iovec *piov = new iovec[ItemCount];					   //create io vector table for writev()
	assert(piov != NULL);

 	for(count=0;count<ItemCount;count++)
		{	
		  	piov[count].iov_base = hp->data;					//Populate io vector table
			piov[count].iov_len  = strlen(hp->data);
		   if (hp->next == NULL) break;
	 		hp = hp->next;											//go to next item in list
			
		}

   DosRequestMutexSem(hmtxSend,SEM_INDEFINITE_WAIT);
	rc = writev(session,piov,ItemCount);				 //Send history data to user
	if (rc == -1)
		{ DosBeep(1000,200);
		  cerr <<  "writev() error in SendHistory()" << endl;
		  shutdown(session,2);
		  soclose(session);			//close socket if error
		}
	DosReleaseMutexSem(hmtxSend);
	delete piov;
	DosReleaseMutexSem(hmtxHistory);
  	return count;
}
//---------------------------------------------------------------------
//Save history list to disk
int SaveHistory(char *name)
{
	
	int icount = 0;
	if (pHead == NULL) return 0;
   DosRequestMutexSem(hmtxHistory,SEM_INDEFINITE_WAIT);

   ofstream hf(name, ios::binary);			  // Open the output file

	if (hf)						  //if no open errors go ahead and write data
		{
			History *hp = pHead;
	
			for(;;)
				{
					hf << hp->ttl << " " << hp->count << " " << hp->data ;  //write to file
					icount++;
					if (hp->next == NULL) break;
					hp = hp->next;											//go to next item in list
				}
         hf.close();
		}

	DosReleaseMutexSem(hmtxHistory);
   return icount;
}
//---------------------------------------------------------------------
int ReadHistory(char *name)
{
	int icount = 0;
	int ttl,count;
	char data[256];
	const char crlf[] = {0x0d, 0x0a, 0};
	char dummy;
	
   DosRequestMutexSem(hmtxHistory,SEM_INDEFINITE_WAIT);

   ifstream hf(name);			  	// Create ifstream instance "hf" and open the input file

   if (hf)						  		//if no open errors go ahead and read data
		{
			
			while(hf)
				{
					hf >> ttl >> count ;				// read ttl (time to live) and count
					hf.get();							// skip 1 space
				   hf.get(data,253,'\n');			// read the rest of the line as a char string
					strcat(data,crlf);				// Append a CR-LF on the end
					AddHistoryItem((void*)data,ttl,count);
					icount++;
				}
         hf.close();
		}




	
	DosReleaseMutexSem(hmtxHistory);
   return icount;
  }

	
//---------------------------------------------------------------------

//Sets various parameters on a COM port using the OS/2 DosDevIOCtl
//function.  For more info on this function see page 722 in
// Real World Programming for OS/2  2.1   (Sams (C) 1993 isbn 0-672-30300-0 )

LONG SetupComPort(ComStruct* COM)
 {
    APIRET rc;

    printf("handle=%d\n",COM->hCOM);

    ULONG baud = COM->baud;

    rc = DosCreateEventSem(NULL,&COM->hevPacketRcvd,0,FALSE);
    if (rc) return rc;

    rc = DosCreateEventSem(NULL,&COM->hevPacketSent,0,FALSE);
    if (rc) return rc;

    rc = DosDevIOCtl(COM->hCOM, IOCTL_ASYNC, ASYNC_SETBAUDRATE,
		(PVOID)&baud, sizeof(USHORT), NULL, NULL,0L, NULL);
    if (rc) return rc;

    LINECONTROL LineControl;
    LineControl.bDataBits	= COM->databits;// 8 data bits
    LineControl.bParity		= COM->parity;  // even parity
    LineControl.bStopBits	= COM->stopbits;// 1 stop bit
    LineControl.fTransBreak	= COM->brk;  	// no break

	 printf("DataBits=%d  Parity=%d  Stop=%d  Break=%d\n", (int)LineControl.bDataBits,
                           (int)LineControl.bParity,
                           (int)LineControl.bStopBits,
									(int)LineControl.fTransBreak);


    rc = DosDevIOCtl(COM->hCOM, IOCTL_ASYNC, ASYNC_SETLINECTRL,
		&LineControl, sizeof(LINECONTROL), NULL, NULL, 0L, NULL);

	 printf("Returned %d\n",rc);

    if (rc) return rc;
    ULONG	u,v;
    USHORT	usComError = 0;
    MODEMSTATUS	ModemStatus;
    ModemStatus.fbModemOn	= DTR_ON | RTS_ON;
    ModemStatus.fbModemOff	= 0;

    rc = DosDevIOCtl(COM->hCOM, IOCTL_ASYNC,ASYNC_SETMODEMCTRL,
		     &ModemStatus, sizeof(MODEMSTATUS),&u,
		     &usComError, sizeof(USHORT),&v);

    return 0;


 }


//--------------------------------------------------------------------
int AsyncOpen(char *szPort)
{
  APIRET	rc;
  ULONG		ulAction;
  HFILE	        hFile;
  char		ch;



  strcpy(COMx.name,szPort);
  COMx.baud 	= 9600;
  COMx.stopbits	= 0;
  COMx.parity 	= NONEp;
  COMx.databits = 8;
  COMx.brk	= 0;

  rc = DosOpen(COMx.name,&COMx.hCOM, &ulAction, 0L, FILE_NORMAL, FILE_OPEN,
		      OPEN_ACCESS_READWRITE | OPEN_SHARE_DENYNONE, 0L);

  if (rc)
      { cerr << "Error " << COMx.name << " COM port not available, RC=" << rc  << endl << flush;
			DosBeep(500,200);

	return -1;
      }
      else
      {
       if ((rc = SetupComPort(&COMx)) != 0)
	       {  cerr << "error in setting up COM port rc=" << rc << endl << flush;
				return -2;
	       }
      }


  ULONG ulPostct;
  tidReadCom = _beginthread(ReadCom,NULL,16384,(void*)&COMx);
  if (DosCreateEventSem(NULL,&_hevStarted,0,FALSE)) return -1;
  if (DosWaitEventSem(_hevStarted,500))
			{	cerr << "COM thread did not start.\n" << flush;
            return -3;
			}
  DosResetEventSem(_hevStarted,&ulPostct);


 return 0;

 }

//--------------------------------------------------------------------
 int AsyncClose(void)
 {

   DosClose(COMx.hCOM);     //close COM port

	return 0;
}

//-------------------------------------------------------------------
APIRET WriteCom(HFILE hCom, char *cp)
{	ULONG actual;
	
	ULONG len = strlen(cp);
	return DosWrite(hCom,cp,len,&actual);

}
//-------------------------------------------------------------------
//--------------------------------------------------------------------
// This is the COM port read thread

void _Optlink ReadCom(void* vp)
{
  USHORT i,j,n;
  APIRET  rc;
  unsigned char *buf= new unsigned char[BUFSIZE];
  unsigned char c;
  ComStruct* rxc = (ComStruct*)vp;
  ULONG	ulBytesRead;
  char *msgbuf;

  int count,errorcount = 0;



  DosPostEventSem(_hevStarted);

  i = 0;
  c = 0;

while (!ShutDownServer)
	{
		 do                        //sit here till end of line
         {							   //and read chars into buffer.
											//The DosRead() func will time out every 60 seconds
											//and return rc=0 with zero bytes read.

           rc = DosRead(rxc->hCOM, &c, 1, &ulBytesRead);

			
			  TotalChars += ulBytesRead ;

			
			  if (rc != 0)
				  {
	 					if (ShutDownServer)
							{	cerr << "Exiting Async com thread\n" << endl << flush;
						      delete buf;
							   _endthread();
							}
						errorcount++;
						cerr << "Error reading COM port. rc=" << rc
								<< "  Errorcount=" << errorcount
								<< endl
								<< flush;
						c = NULL;
						
						if (errorcount > 20)
							{	cerr << "Max consecutive COM error count exceeded, terminating thread.\n" << flush;
								delete buf;
								AsyncClose();
								_endthread();
							}
				  }

			  if ((ulBytesRead > 0) && (rc == 0))
				  {
						c = c & 0x7f;
						if (i < (BUFSIZE-3)) buf[i++] = c; else i = 0;
				  }else c = 0;
			
			

         }while (( c != 0x0a) && (c != 0x0d));

         WatchDog++;
         tickcount = 0;

	  		if (i > 3)
			{	TotalLines++;
			   errorcount = 0;
		      buf[i-1] = 0x0d;
				buf[i++] = 0x0a;
            buf[i++] = NULL;
				count = i;

				
				//printhex((char*)buf,i);

				if (buf != NULL)
					{
	
						rc = DosWriteQueue(hqAsync,SRC_TNC,count,buf,0); //*** buf must be freed by Queue reader ***.
                  if (rc)
							{	cerr << "DosWriteQueue error rc= " << rc << endl << flush ;
								if (buf != NULL) delete buf;
							}
						buf = new unsigned char[BUFSIZE];  //get a new buffer
						assert(buf != NULL);
					}else
						cerr	<<	"Failed to allocate memory for async queue message" << endl << flush;
			}

      i = 0;		//Reset buffer pointer
		c = 0;

	} // Loop until server is shut down

}
		

//---------------------------------------------------------------------
//---------------------------------------------------------------------
//Add a new user to the list of active sessions
BOOL AddSession(int s)
{	BOOL rv = FALSE;

   DosWaitEventSem(NotSending,SEM_INDEFINITE_WAIT);

	DosEnterCritSec();
	for(int i=0;i<MAXCLIENTS;i++) if (sessions[i] == -1) break;
	if(i < MAXCLIENTS)
		{	sessions[i] = s;
			ConnectedClients++;
			rv = TRUE;
		}
	DosExitCritSec();
	return rv;
}

//--------------------------------------------------------------------
//Remove a user
BOOL DeleteSession(int s)
{
	int i = 0;
	BOOL rv = FALSE;
	DosWaitEventSem(NotSending,SEM_INDEFINITE_WAIT);
   DosEnterCritSec();
	while ((sessions[i] != s) && (i < MAXCLIENTS)) i++;
	if (i < MAXCLIENTS) ;
		{	
	      sessions[i] = -1;
			ConnectedClients--;
			if ( ConnectedClients < 0) ConnectedClients = 0;
			rv = TRUE;
		}
	DosExitCritSec();
	return rv;

}
//---------------------------------------------------------------------
void CloseAllSessions(void)
{
	DosWaitEventSem(NotSending,SEM_INDEFINITE_WAIT);

	for(int i=0;i<MAXCLIENTS;i++)
		  if (sessions[i] != -1 )
			  {	shutdown(sessions[i],2);
					soclose(sessions[i]);
					sessions[i] = -1;
					
			  }
}
//---------------------------------------------------------------------
//Send data at "p" to all logged on clients listed in the sessions array.


void SendToAllClients( void* p, int n)
{
	char *cp = (char*)p;
	int ccount = 0;
	int rc;
	
	DosResetEventSem(NotSending,&ulNScnt);		//Let eveyone know we're sending now

	assert(p != NULL);
	assert(n < 512);

	for(int i=0;i<MAXCLIENTS;i++)
		if (sessions[i] != -1)
           {	DosRequestMutexSem(hmtxSend,SEM_INDEFINITE_WAIT);
					rc = send(sessions[i],cp,n,0);		 //Send data to all clients
			      if (rc == -1)
						{ DosBeep(1000,200);
					     psock_errno("SendToAllClients()");
                    shutdown(sessions[i],2);		//Close the offending socket
						  soclose(sessions[i]);				//
						  sessions[i] = -1;					//remove client from list
						}
					DosReleaseMutexSem(hmtxSend);
					ccount++;
					DosSleep(2);
				}
	DosPostEventSem(NotSending);					//Tell other threads we're finished here

	cout << "Sent " << setw(4) << n << " bytes to " << setw(3) << ccount << " clients"
			<<	endl
			<<	flush ;

	return;
}


//---------------------------------------------------------------------
// Pushes a character string into the server send queue.
APIRET BroadcastString(char *cp)
{

	int count = strlen(cp);
	char *msgbuf = new char[count];
	assert(msgbuf != NULL);
	strncpy(msgbuf,cp,count);
   APIRET rc = DosWriteQueue(hqAsync,SRC_INTERNAL,count,msgbuf,0); //DeQueue() deletes the *msgbuf
	return rc ;
}

//---------------------------------------------------------------------
//This is a thread.  It removes items from the Async queue and sends
//them to all the clients.
void _Optlink DeQueue(void* vp)
{
 	APIRET rc;
 	CHAR		priority ;
	REQUESTDATA	cmd;
	VOID		*vbuff;
	ULONG		bufflen;
	int MaxAge,MaxCount;


 while(!ShutDownServer)
  {
	do
	  {	rc = DosReadQueue( hqAsync,&cmd,&bufflen,&vbuff,0,DCWW_WAIT,&priority,(HEV)NULL);
	      if (rc != 0)
				{	cerr << "Read Queue error rc= " << rc << endl << flush;
				}

         if (cmd.ulData == CMD_END_THREAD)
			    { cerr << "Ending DeQueue\n" << flush;
					_endthread();
				 }

 	  }while(rc);

  if (vbuff != NULL)
	  {
	
	      if (cmd.ulData == SRC_TNC)
					{ GetMaxAgeAndCount((char*)vbuff,&MaxAge,&MaxCount);	//Set max ttl and count values
					  	AddHistoryItem(vbuff,MaxAge,MaxCount);   				//Put item in history list.
					}
			//if (cmd.ulData != SRC_USER)			//Comment out this line to enable user to user comm
				SendToAllClients(vbuff,bufflen);	// Send item out on internet

			delete vbuff;							// Important!  Free the memory at this pointer.
			vbuff = NULL;
	  }else
		  cerr << "Error in DeQueue: vbuff is NULL" << endl << flush;

  }




}

//----------------------------------------------------------------------
int WriteLog(char *cp)
{	FILE *f;
	time_t ltime;
	char szTime[40];
	char *p;
	APIRET rc;

	rc = DosRequestMutexSem(hmtxLog,SEM_INDEFINITE_WAIT);
	if (rc)
		{	cerr	<< "Request Mutex Semaphore error in WriteLog, rc= "  << rc << endl << flush;
		   return -1;
		}

	f = fopen(LOGFILE,"a+");
	if (f == NULL)
		{	cerr << "failed to open " << LOGFILE << endl;
		   return -1;
		}

	time(&ltime) ;                           //Time Stamp
	strncpy(szTime,ctime(&ltime),39);
	p = strchr(szTime,(int)'\n');
	if (p != NULL) p[0] = ' ';			 //convert new line to a space
	fprintf(f,"%s %s\n",szTime,cp);	 //Write log entry with time stamp

	fclose(f);
	DosReleaseMutexSem(hmtxLog);
	return 0;
}

//-----------------------------------------------------------------------
void printhex(char *cp, int n)
{
	for (int i=0;i<n;i++) printf("%02X ",cp[i]);
	printf("\n");

}
//-----------------------------------------------------------------------
int SendSessionStr(int session, char *s)
{
   DosRequestMutexSem(hmtxSend,SEM_INDEFINITE_WAIT);
   int rc = send(session,s,strlen(s),0);
   DosReleaseMutexSem(hmtxSend);
	return rc;
}
//-----------------------------------------------------------------------
//Reads password file and checks user names and passwords against
//the parameters passed in szUser and szPass.
//If the password file doesn't exist the default user and passwords are used.
//Password file syntax:
// # lines starting with '#' or '*' are comments and ignored
// joe_user joes_password
// user is first word on line, password is the second.
//
BOOL checkpass(char *fname, char *szUser, char *szPass)
{
	BOOL found=FALSE;
	char line[256];
	IString user,pass;
	IString refUser(szUser);
	IString refPass(szPass);
	//cout << szUser << "   " << szPass << endl;
	ifstream pwf(fname);

	if (pwf)
		{	
			 while ((pwf) && (!found))
				 {
                pwf.getline(line,253);								//get a line from file
					 if ((line[0] != '#') && (line[0] != '*'))   //No comments
						 {	 IString isLine(line,256);					//Make a string out of it
	  						 user = isLine.word(1);						//get 1st word to user
							 pass = isLine.word(2);						//get 2nd word to pass
                      if ((user.indexOf(refUser) > 0) && (pass.indexOf(refPass) > 0)) found = TRUE;
						 }

					
				}

			 pwf.close();

		}else		 //Use default values for user and password
			{	
				if ((refUser.indexOf(userDefault) > 0 ) && (refPass.indexOf(passDefault) > 0))
						found = TRUE;
			}

		return found;
}

//-----------------------------------------------------------------------

//An instance of this thread is created for each user who connects.

#define BASE 0
#define USER 1
#define PASS 2
#define REMOTE 3

void _Optlink TCPSessionThread(void *p)
{
	char *buf, *cp,lastch=0;
	int BufRemaining;
  	int session = (int)p;
	int BytesSent,BytesRead,dsize, i;
	struct sockaddr_in peer_adr;
	char szPeer[33], szError[256], szLog[256], infomsg[256];
	char c;
	int adr_size = sizeof(peer_adr);
	int n, rc,data;
	ULONG State = BASE;
	PTIB ptib;
	PPIB ppib;
	char szUser[16], szPass[16];

	if (session < 0) return;

	TotalConnects++;

	ITime starttime;
	szPeer[0] = NULL;

	if (getpeername(session, (struct sockaddr *)&peer_adr, &adr_size) == 0)
		{
			strncpy(szPeer,inet_ntoa(peer_adr.sin_addr),32);
		}

	strcpy(szError,"Max users exceeded, disconnecting\r\n");

  	cout << szPeer << " has connected" << endl << flush;

	data = 1;  //Set socket for non-blocking
	ioctl(session,FIONBIO,(char*)&data,sizeof(int));

	n = SendHistory(session);        //Send out the previous N minutes of APRS activity.
   cout << "sent " << n << " history items." << endl << flush;

  	SendFiletoClient(session,WELCOME);		 //Read Welcome message from file

	if (!AddSession(session))
		{ cerr << "Can't add client to session list - closing connection.";
		  rc = SendSessionStr(session,szError);
		  if (rc == -1) psock_errno("AddSession");
		  shutdown(session,2);
		  soclose(session);
		  WriteLog("Error, too many users");
		  return;
		}
	strcpy(szLog,szPeer);
	strcat(szLog," connected");
	WriteLog(szLog);

 	{ostrstream msg(infomsg,256);

		msg	<<	szJAVAMSG
				<<	szServerID
				<<	szPeer
				<< " connected. "
				<< ConnectedClients
				<< " users online.\r\n"
				<<	ends;

		BroadcastString(infomsg);

		msg.~ostrstream();
	}

	if (ConnectedClients > MaxConnects) MaxConnects= ConnectedClients;


	do
		{											 /*The logic here is that 1 char is fetched
														on each loop and tested for a CR or LF	before
														the buffer is processed.  If "i" is -1 then
														any socket error other that "would block" causes
														BytesRead and i to be set to zero which causes
														the socket to be closed and the thread terminated.
														If "i" is greater than 0 the character in "c" is
														put in the buffer.
													 */
											
			buf = new char[SBUFSIZE];		 //allocate a buffer
		   BytesRead = 0;						 //initialize byte counter
			do
			  {
					i = recv(session,&c,1,0);		//get 1 byte into c
					if (i == -1)
						{

							int err = sock_errno();
							if (err != SOCEWOULDBLOCK)
										{	BytesRead = 0;	  //exit on errors other than SOCEWOULDBLOCK
											i = 0;

										}
							DosSleep(25);  //Sleep 25 ms. Don't hog cpu while in loop awaiting data

						}

                if (ShutDownServer)
						 {	delete buf;
							_endthread();
						 }

					 if (i > 0)
						 {	
							if ( !((lastch == 0x0d) && (c == 0x0a)) )	  //reject LF after CR
									{	c = c & 0x7f;
										if (BytesRead < SBUFSIZE-3) buf[BytesRead] = c;
										BytesRead += i;
									}

							switch (c)
								{
									case 0x04:	i = 0;   		//Control-D disconnect
													BytesRead = 0;
													break;

									case 0x12:	DeleteSession(session);
													SendSessionStr(session,"\r\n220 Remote TNC control mode. Ctrl-Q to quit.\r\n503 User:");
													State = USER;    //  ^r Set for remote control of tnc
													i = 0;
													break;

									case 0x11:	if (State != BASE)
														{	if (State != REMOTE) AddSession(session);
															State = BASE;  // ^q Turn off remote
															SendSessionStr(session,"\r\n200 Exit remote mode successfull\r\n");
														}
													i = 0;
													break;
								};

     						 }else c = 0;


					
			  }while ((i != 0) && (c != 0x0d) && (c != 0x0a));
			
				
			if ((BytesRead > 0) && ( i > 0) && (c != 0x0a))		 // 1 or more bytes needed
				{																 // and discard line feeds

               i = BytesRead - 1;
					buf[i++] = 0x0d;
					buf[i++] = 0x0a;
					buf[i++] = 0;

				   //cout << szPeer << ": " << buf << flush;		 //Debug code
				   //printhex(buf,i);									 //

               if (State == REMOTE)
						{	buf[i-2] = NULL;							//No line feed required for TNC
							//printhex(buf,strlen(buf));			//debug code
							WriteCom(COMx.hCOM,buf);
							delete buf;
						}

					if (State == BASE)
						rc = DosWriteQueue(hqAsync,SRC_USER,i,buf,0);  //note: buf is deleted in DeQueue


					int j = i-3;

               if ((State == PASS) && (BytesRead > 1))
						{	strncpy(szPass,buf,15);
							szPass[j] = NULL;
							if (checkpass(PASSWORDFILE,szUser,szPass))
								{	State = REMOTE;
		                     SendSessionStr(session,"\r\n230 Login successful. Control Q to exit remote mode.\r\n");
								}
								else
									{	SendSessionStr(session,"\r\n550 Login failed\r\n");
										State == BASE;
									}

                     AddSession(session);
							delete buf;
						}

					if ((State == USER) && (BytesRead > 1))
						{ 	strncpy(szUser,buf,15);
							szUser[j] = NULL;
							State = PASS;
						   SendSessionStr(session,"\r\n331 Pass:");
							delete buf;
						}

					
					
				 }else
						{	if (buf) delete buf;
							buf = NULL;
						}



		} while (BytesRead != 0);		  // Loop back and get another line.

	 DeleteSession(session);

	 shutdown(session,2);
 	 if (soclose(session) < 0) cerr << "Socket close error\n" << flush;

	

	 cout << szPeer << " has disconnected"
		  <<	endl
		  << flush ;
	 strcpy(szLog,szPeer);
	 strcat(szLog," disconnected ");
	 ITime endtime;
	 long  iConnecttime = endtime.asSeconds() - starttime.asSeconds() ;
	 ITime connecttime(iConnecttime);
	 strcat(szLog,(char*)connecttime.asString("%H:%M:%S"))	;

	 WriteLog(szLog);

	 ostrstream msg(infomsg,256);

	msg	<< szJAVAMSG
			<<	szServerID
			<<	szPeer
			<< " disconnected. "
			<< ConnectedClients
			<< " users online.\r\n"
			<<	ends;

    BroadcastString(infomsg);			  //Tell everyone who disconected
	 _endthread();
}


//------------------------------------------------------------------------

//One instance of this thread is created at server startup.
//It listens on the APRS port number (14579) for clients
//wanting to connect.  Each connect request is assigned a
//new socket and a new instance of TCPSessionThread() is created.

void _Optlink TCPServerThread(void *p)
{

	int	s,i,session,rc;
	struct sockaddr_in server,client;
	char buf[SBUFSIZE];
	int optval;

	ShutDownServer = FALSE;

	s = socket(PF_INET,SOCK_STREAM,0);
	ServerSocket = s;                  //copy it to a global

	optval = 1;								  //Allow address reuse
	setsockopt(s,SOL_SOCKET,SO_REUSEADDR,(char*)&optval,sizeof(int));

	if (s == 0)
		{  psock_errno("TCPServerThread socket error");
			ShutDownServer = TRUE;
			return;
		}

	memset(&server,0,sizeof(server));
	memset(&client,0,sizeof(client));

	server.sin_family = AF_INET;
	server.sin_addr.s_addr = INADDR_ANY;
	server.sin_port = htons(ServerPort);
	if (bind(s,(struct sockaddr *)&server, sizeof(server)) <  0)
		{
		   psock_errno("TCPServerThread bind error");
			ShutDownServer = TRUE;
			return;
		}

	listen(s,2);	

	cout << "Server listening on port " << ServerPort << endl;

	for(;;)
		{
		   i = sizeof(client);
			session = accept(s,(struct sockaddr *)&client,&i);
			if (ShutDownServer)
				{	soclose(s);
			      if (session >= 0) soclose(session);
					cerr << "Ending TCP server thread\n" << flush;
					_endthread();
				}

			if (session < 0)
				{ psock_errno( "Error in accepting a connection");
				} else
              if( _beginthread(TCPSessionThread,NULL,24576,(void*)session) < 0)
					  {
	 					  cerr << "Error creating new client thread.\n" << flush;
						  shutdown(session,2);
						  rc = soclose(session);      // Close it if thread didn't start
						  if (rc < 0) psock_errno("Server Thread soclose()");
					  }

         memset(&client,0,sizeof(client));
		}

}
//----------------------------------------------------------------------
//This thread listens to a UDP port and sends all packets heard to all
//logged on clients.

void _Optlink UDPServerThread(void *p)
{

#define UDPSIZE 256
	int sockint,s,i,rc, namelen, client_address_size;
	struct sockaddr_in client, server;
	char *buf;
	int UDP_Port = ServerPort;


   /*
    * Create a datagram socket in the internet domain and use the
    * default protocol (UDP).
    */
   if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
   {
       psock_errno("Failed to create datagram socket");
       ShutDownServer = TRUE;
		 return;
   }

   /*
    *
    * Bind my name to this socket so that clients on the network can
    * send me messages. (This allows the operating system to demultiplex
    * messages and get them to the correct server)
    *
    * Set up the server name. The internet address is specified as the
    * wildcard INADDR_ANY so that the server can get messages from any
    * of the physical internet connections on this host. (Otherwise we
    * would limit the server to messages from only one network interface)
    */
   server.sin_family      = AF_INET;   	/* Server is in Internet Domain */
   server.sin_port        = htons(UDP_Port) ;/* 0 = Use any available port       */
   server.sin_addr.s_addr = INADDR_ANY;	/* Server's Internet Address    */

   if (bind(s, (struct sockaddr *)&server, sizeof(server)) < 0)
   {
       psock_errno("Datagram socket bind error");
       ShutDownServer = TRUE;
		 return;

   }



  for(;;)		  //Loop forever
  {
   buf = new char[UDPSIZE];
   client_address_size = sizeof(client);

   i = recvfrom(s, buf, UDPSIZE, 0, (struct sockaddr *) &client, &client_address_size) ; //Get client udp data

   cout 	<<	inet_ntoa(client.sin_addr)
			<<	": " << buf
			<<	endl;

   rc = DosWriteQueue(hqAsync,SRC_USER,i,buf,0);  //Send udp data to all users.  Note: buf is deleted in DeQueue

	
		

    if (ShutDownServer)
				{	soclose(s);
			      _endthread();
				}

	}

 }



//----------------------------------------------------------------------
int SendFiletoTNC(HFILE hCom, char *szName)
{	char Line[256];
	ifstream file(szName);
	if (!file)
		{	cerr << "Can't open " << szName << endl << flush;
			return -1 ;
		}
	Line[0] = 0x03;			//Control C to get TNC into command mode
	Line[1] = NULL;
	WriteCom(hCom,Line);
	DosSleep(250);

	do
		{
			file.getline(Line,256);	  //Read each line in file and send to TNC
         if (file.eof()) break;
			strncat(Line,"\r",256);

			if ((Line[0] != '*') && (Line[0] != '#')  )		 //Reject comments
					{	WriteCom(hCom,Line);						 //Send line to TNC
						DosSleep(250);
					}

		}while (!file.eof());

   file.close();
	return 0;

}
//----------------------------------------------------------------------
int SendFiletoClient(int session, char *szName)
{	char Line[256];
	APIRET rc;
	int n;
	char *cp;


	rc = DosRequestMutexSem(hmtxSendFile,5000);
	if (rc)
		{	cerr <<  "Request Mutex Sem error in SendFiletoClient() rc = " << rc << endl << flush;
	      return -1;
		}

	ifstream file(szName);
	if (!file)
		{	cerr << "Can't open " << szName << endl << flush;
			return -1  ;
		}

	do
		{
			file.getline(Line,256);	  //Read each line in file and send to client session
			if (file.eof()) break;
			if (strlen(Line) > 0)
					{	strncat(Line,"\r\n",256);
						n = strlen(Line);
                  DosRequestMutexSem(hmtxSend,SEM_INDEFINITE_WAIT);
						rc = send(session,Line,n,0);
						if (rc == -1)
							{  DosBeep(1000,200);
								psock_errno("SendFileToClient()");
								shutdown(session,2);
								soclose(session);				//close the socket if error happened
							}
						DosReleaseMutexSem(hmtxSend);
						DosSleep(2);

					}

		}while (!file.eof());

   file.close();

	DosReleaseMutexSem(hmtxSendFile);
	return 0;

}
//----------------------------------------------------------------------
//Type 's' to view this while the server is running.
void PrintStats(ostream &os)
{
	os		<< endl
			<< "Total characters = " << TotalChars << endl
			<< "Total Lines      = " << TotalLines << endl
			<< "Total connects   = " << TotalConnects << endl
			<< "Max Connects     = " << MaxConnects << endl
			<< "History Items    = " << ItemCount << endl
			<< flush;
}

//----------------------------------------------------------------------
int main(int argc, char *argv[])
{
	
	int i,rc;
	char debug;
   ULONG	priority = QUE_FIFO;
	char buf[1024];
	unsigned lastSec;

	ITime time;

	TotalConnects = TotalChars = TotalLines = MaxConnects = 0;

	
	IString isComPort("COM2");				 //Default to COM2 and tcpip port 14579
	IString isServerPort("14579");
	

	cout << "WA4DSY APRS Internet server ver. 1.1" << endl;
	if (argc > 1)
		{	isComPort = IString(argv[1]);			//get optional 1st arg to COM port variable
		}
		
	if (argc > 2)
		{	isServerPort = IString(argv[2]);	  //get optional 2nd arg to Server port variable
			if (isServerPort.isDigits() == FALSE)
				{	cerr << "Specified server port '" << isServerPort << "' is not a number.\b\n";
				  	return -1;
				}
		}

	ServerPort = isServerPort.asInt();			 //set server port number

   IString isComQueue(isServerPort);				//Create a Queue name based on the socket number
	isComQueue.insert("\\QUEUES\\Q");				//Default name will be \QUEUES\Q14579
	cout << "Queue name is: " << isComQueue << endl;

	CreateHistoryList();
	cout << "Read " << ReadHistory(SAVE_HISTORY) << " History items from file\n" ;

/* Initialize sockets */
   sock_init();

  	
   rc = DosCreateQueue(&hqAsync,priority,(char*)isComQueue);		//Queue for com port serial data
	rc = DosCreateEventSem(NULL,&NotSending,0,TRUE);      //Event semaphore for SentToAllClients()
   rc = DosCreateMutexSem(NULL,&hmtxLog,0,FALSE);			//Mutual Exclusion semaphone for WriteLog func.
	rc = DosOpenMutexSem(NULL,&hmtxLog);
   rc = DosCreateMutexSem(NULL,&hmtxSendFile,0,FALSE);	//Mutual Exclusion semaphone for SendFiletoClient().
	rc = DosOpenMutexSem(NULL,&hmtxSendFile);
	rc = DosCreateMutexSem(NULL,&hmtxHistory,0,FALSE);	//Mutual Exclusion semaphone for History linked list.
   rc = DosOpenMutexSem(NULL,&hmtxHistory);
   rc = DosCreateMutexSem(NULL,&hmtxSend,0,FALSE);	//Mutual Exclusion semaphone socket send.
   rc = DosOpenMutexSem(NULL,&hmtxSend);


	
	sessions = new int[MAXCLIENTS];

	if (sessions == NULL) { cerr << "Can't create sessions pointer\n" ; return -1;}

	for(i=0;i<MAXCLIENTS;i++) sessions[i] = -1;
	ConnectedClients = 0;

	TID tidTCPServerthread	=  _beginthread(TCPServerThread,NULL,32768,NULL);
	TID tidUDPServerthread =   _beginthread(UDPServerThread,NULL,32768,NULL);
	DosSleep(1000);
	if (ShutDownServer || (tidTCPServerthread < 0) || (tidUDPServerthread < 0))
		{
			delete sessions;
			if (tidTCPServerthread < 0)
				cerr 	<< "TCPServerthread failed to start"
						<<  endl;

         if (tidUDPServerthread < 0)
				cerr 	<< "UDPServerthread failed to start"
						<<  endl;

			return -1;
		}

	ShutDownServer = FALSE;
	
	cout 	<< "Opening serial port "
			<< isComPort
			<< endl;

	if ((rc = AsyncOpen((char*)isComPort)) != 0)
		{  cerr 	<< "Failed to open serial port "
					<< isComPort
					<< " rc= "
					<< rc
					<< endl;
					DosSleep(1500);
			return -1;
		}

  cout << "Setting up TNC\n" << flush;
  //char c = 0x03;
  //ULONG bw;
  //DosWrite(COMx.hCOM,&c,1,&bw);   		//Send a ^C to the TNC

  SendFiletoTNC(COMx.hCOM,TNC_INIT);	//Setup TNC from initialization file


  TID tidDeQueuethread = _beginthread(DeQueue,NULL,16384,NULL);
  WriteLog("Server Start");
  cout << "Server start\n" << flush;

  do
	{
	DosSleep(100);

	 lastSec = time.seconds();
    time = time.now();

	 /*
	 if (time.seconds() != lastSec)	  //send the test message for debugging
		 {
		    char *cp = new char[256];  // *cp deleted in DeQueue()
			 strncpy(cp,TEST,256);
          int count = strlen(cp);
          rc = DosWriteQueue(hqAsync,0L,count,cp,0);
		 }
  	 */

	if (++tickcount >= 1200)			//About 2 minutes
		{	if (WatchDog < 2)
					{	cerr 	<< "No serial port activity.\n"
								<< flush;
					  	DosBeep(1800,100);
						PrintStats(cout);
					
					}
          tickcount = 0;
			 WatchDog = 0;				//Serial port reader increments this with each rx line.
		}

	
	if ((time.seconds() == 0 ) && (lastSec != 0))		//do this every minute
			{
				cout << " Deleted " << DeleteOldItems() << " items" << endl << flush;
			}

					
   if (kbhit())
	      { char ch = getch();

					
					  switch(ch)
					  {
 						case '?': cout <<	 "q = quit\n"
											<<	 "s = print statistics\n"
											<<  endl;
									 break;

						case 'q': 	debug = 'q'; break;

						case 's':	PrintStats(cout); break;
						
						default:  debug = ch;
					  };

			}
			
	    }while (debug != 'q');

	cout << "Saved " << SaveHistory(SAVE_HISTORY) << "history items.\n" << flush ;
	cerr << "ServerShutdown\n" << flush;
	WriteLog("Server Shutdown");
   ShutDownServer = TRUE;
   SendFiletoTNC(COMx.hCOM,TNC_RESTORE);
	AsyncClose() ;
	//rc = soclose(ServerSocket);
	//if (rc) cerr << "Error closing Server Socket  rc = " << rc << endl << flush;

   char *ShutDown = new char[127];
	strcpy(ShutDown,szJAVAMSG);
   strcat(ShutDown,"The APRS server is shutting down for a while.  Bye.\r\n");
   rc = DosWriteQueue(hqAsync,SRC_INTERNAL,strlen(ShutDown),ShutDown,0);

  	DosSleep(2000);

	CloseAllSessions();
 	DosWriteQueue(hqAsync,CMD_END_THREAD,NULL,NULL,0);	//Tell the Queue reader to end it's thread
	DosSleep(1000);
   DosCloseQueue(hqAsync);
	DosCloseMutexSem(hmtxLog);
   delete sessions;

   return 0;


}
