/*
 * tables.c - build/access to tables
 */

#define INCL_PM
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tablet.h"
#include "resource.h"

/*
 * list of tables
 */

static  TABLPTR Tables = NULL ;

/*
 * support for creating tables
 *
 *      isTableTile      - TRUE if line is title
 *      cleanupLine      - suppress space/CR/LF from line
 *      createTableEntry - Process Title Line
 *      makeTableData    - Suppress Spacers
 *      appendTableData  - Process Data  Line
 */

#define isSpace(ch)     ((ch) <= 0x20 ? TRUE : FALSE)

static  BOOL    isTableTitle(PUCHAR str)
{
    if (str == NULL || isSpace(*str)) {
        return FALSE ;
    }
    return TRUE ;
}

static  PUCHAR  cleanupLine(PUCHAR data)
{
    PUCHAR  ptr ;

    if (data == NULL) {
        return NULL ;
    }

    /*
     * Skip proceeding spaces (and '|' mark)
     */

    for (ptr = data ; *ptr != '\0' && isSpace(*ptr) ; ptr++) ;
    if (*ptr == '|') ptr++ ;
    data = ptr ;

    /*
     * Suppress CR/LF
     */

    for (ptr = data ; *ptr != '\0' ; ptr++) {
        if (*ptr == '\r' || *ptr == '\n') {
            *ptr = '\0' ;
            break ;
        }
    }
    return data ;
} 

static  TABLPTR createTableEntry(PUCHAR buff)
{
    PUCHAR  title ;
    TABLPTR new, ptr ;

    if ((buff = cleanupLine(buff)) == NULL) {
        return NULL ;
    }
    if ((title = strdup(buff)) == NULL) {
        return NULL ;
    }

    if ((new = (TABLPTR) malloc(sizeof(TABLREC))) == NULL) {
        free(title) ;
        return NULL ;
    }
    new->prev = NULL  ;
    new->next = NULL  ;
    new->name = title ;
    new->line = NULL  ;

    if (Tables == NULL) {
        Tables = new ;
    } else {
        for (ptr = Tables ; ptr->next != NULL ; ptr = ptr->next) ;
        new->prev = ptr ;
        ptr->next = new ;
    }
    return new ;
}

static  PUCHAR  makeTableData(PUCHAR buff)
{
    PUCHAR  sp, tp ;
    UCHAR   temp[1024] ;

    for (sp = buff, tp = temp ; *sp != '\0' ; ) {
        if ((*sp >= 0x80 && *sp < 0xa0) || (*sp >= 0xe0 && *sp < 0xfc)) {
            *tp++ = *sp++ ;
            *tp++ = *sp++ ;
        } else if (*sp > 0x20) {
            *tp++ = *sp++ ;
        } else {
            sp++ ;
        }
    }
    *tp = '\0' ;

    if (temp[0] == '\0') {
        return NULL ;
    }
    return strdup(temp) ;
}

static  BOOL    appendTableData(TABLPTR tab, PUCHAR buff)
{
    PUCHAR  data ;
    LINEPTR new, ptr ;

    if (tab == NULL || buff == NULL) {
        return FALSE ;
    }
    if ((buff = cleanupLine(buff)) == NULL) {
        return FALSE ;
    }
    if ((data = makeTableData(buff)) == NULL) {
        return FALSE ;
    }

    if ((new = (LINEPTR) malloc(sizeof(LINEREC))) == NULL) {
        free(data)   ;
        return FALSE ;
    }
    new->next = NULL ;
    new->data = data ;

    if (tab->line == NULL) {
        tab->line = new ;
    } else {
        for (ptr = tab->line ; ptr->next != NULL ; ptr = ptr->next) ;
        ptr->next = new ;
    }
    return TRUE ;
}

/*
 * tableLoad - load table in specified file
 */

BOOL    tableLoad(PUCHAR name)
{
    FILE    *fp ;
    TABLPTR tab = NULL ;
    UCHAR   buff[1024] ;

    if ((fp = fopen(name, "r")) == NULL) {
        return FALSE ;
    }
    while (fgets(buff, 1024, fp) != NULL) {
        if (isTableTitle(buff)) {
            tab = createTableEntry(buff) ;
        } else if (tab != NULL) {
            appendTableData(tab, buff) ;
        }
    }
    fclose(fp) ;

    return TRUE ;
}

/*
 * tableDone - dispose tables
 */

void    tableDone(void)
{
    TABLPTR tabPtr, tabNxt ;
    LINEPTR linPtr, linNxt ;

    for (tabPtr = Tables ; tabPtr != NULL ; tabPtr = tabNxt) {
        tabNxt = tabPtr->next ;
        for (linPtr = tabPtr->line ; linPtr != NULL ; linPtr = linNxt) {
            linNxt = linPtr->next ;
            if (linPtr->data != NULL) free(linPtr->data) ;
            free(linPtr) ;
        }
        if (tabPtr->name != NULL) free(tabPtr->name) ;
        free(tabPtr) ;
    }
}

/*
 * access to named table
 */

TABLPTR tableName(PUCHAR name)
{
    TABLPTR tab ;

    if (name == NULL || *name == '\0') {
        return NULL ;
    }
    for (tab = Tables ; tab != NULL ; tab = tab->next) {
        if (strcmp(tab->name, name) == 0) {
            return tab ;
        }
    }
    return NULL ;
}

/*
 * traverse on table list
 */

TABLPTR tableOpen(void)
{
    return Tables ;
}

TABLPTR tablePrev(TABLPTR tab)
{
    if (tab == NULL) {
        return NULL ;
    }
    return tab->prev ;
}

TABLPTR tableNext(TABLPTR tab)
{
    if (tab == NULL) {
        return NULL ;
    }
    return tab->next ;
}

/*
 * select table with Dialog
 */

static  TABLPTR curTable = NULL ;
static  TABLPTR newTable = NULL ;

static  void    listupTable(HWND hwndLst)
{
    TABLPTR tp ;
    SHORT   id ;
    
    for (tp = tableOpen(), id = 0 ; tp != NULL ; tp = tableNext(tp), id++) {
        WinSendMsg(hwndLst, LM_INSERTITEM, 
	            MPFROM2SHORT(LIT_END, 0), MPFROMP(tp->name)) ;
        if (tp == curTable) {
	    WinSendMsg(hwndLst, LM_SELECTITEM, 
	            MPFROMSHORT(id), MPFROMSHORT(TRUE)) ;
	}
    }
}

static  void    selectTable(HWND hwndLst)
{
    TABLPTR tp ;
    SHORT   id ;
    
    id = WinQueryLboxSelectedItem(hwndLst) ;

    for (tp = tableOpen() ; tp != NULL ; tp = tableNext(tp), id--) {
        if (id == 0) {
	    newTable = tp ;
	    break ;
	}
    }
}

static MRESULT EXPENTRY selectDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    POINTL  pt  ;
    SWP     swp ;
    HWND    hwndLst = WinWindowFromID(hwnd, IDD_TABLIST) ;
    
    switch (msg) {
    
    case WM_INITDLG :
        WinQueryPointerPos(HWND_DESKTOP, &pt) ;
	WinQueryWindowPos(hwnd, &swp)   ;
	swp.x = pt.x - (swp.cx / 10)    ;
	swp.y = pt.y - (swp.cy * 4 / 5) ;
	WinSetWindowPos(hwnd, NULLHANDLE, swp.x, swp.y, 0, 0, SWP_MOVE) ;
	listupTable(hwndLst) ;
	break ;
	
    case WM_COMMAND :
        switch (SHORT1FROMMP(mp1)) {
	case DID_OK :
	    selectTable(hwndLst) ;
	    WinDismissDlg(hwnd, DID_OK) ;
	    break ;
        case DID_CANCEL :
	    WinDismissDlg(hwnd, DID_CANCEL) ;
	    break ;
	}
	return (MRESULT) 0 ;
	
    case WM_CONTROL :
        if (SHORT1FROMMP(mp1) != IDD_TABLIST) {
	    return (MRESULT) 0 ;
	}
	if (SHORT2FROMMP(mp1) != LN_ENTER) {
	    return (MRESULT) 0 ;
	}
	selectTable(hwndLst) ;
	WinDismissDlg(hwnd, DID_OK) ;
	return (MRESULT) 0 ;
    }
    return WinDefDlgProc(hwnd, msg, mp1, mp2) ;
}

TABLPTR tableSelect(HWND hwnd, TABLPTR cur)
{
    ULONG   result ;
    
    curTable = cur  ;
    newTable = NULL ;
    
    result = WinDlgBox(HWND_DESKTOP, hwnd, 
                selectDlgProc, NULLHANDLE, IDD_SELECT, NULL) ;
    if (result == DID_OK) {
        return newTable ;
    } else {
        return NULL ;
    }    
}
