#include <os2.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "Defines.h"
#include "Structures.h"
#include "SupportFunctions.h"
#include "DisplayFunctions.h"

void Usage(PUCHAR probString)
{
    printf("\nHPFSUtil v%u.%02u by Mike Ruskai <%s>\n\n", VER_MAJOR, VER_MINOR, EMAIL_ADDRESS);
    printf("This software is released to the public domain.  Use at your own risk.\n\n");
    printf("Usage: hpfsutil.exe <drive letter> <options>\n\n");
    printf("<drive letter> - letter of HPFS drive to process\n\n");
    printf("Options:\n\n");
    printf("/GD - report current dirty bit status\n");
    printf("/DT - toggle dirty bit\n");
    printf("/SD:<T/F> - set dirty status to True or False\n");
    printf("/FS - compare actual and reported free space\n");
    printf("/SU - show HPFS system space usage\n");
    printf("/BS - show bad sector count\n");
    printf("/SHSU - display contents of SuperBlock\n");
    printf("/SHSP - display contents of SpareBlock\n");
    printf("/SHSPFD - display SpareBlock with full spare DirBlk list\n");
    printf("/DS:<sect>[-<sect>] - dump sector(s) to file\n");
    printf("/FN:<filename> - filename to dump sectors to; default sectdata.dat\n");
    printf("/NL - don't use IBM linedraw characters\n");
    printf("\nArgument order is unimportant.\n");
    if (probString!=NULL)
        {
        printf("\n%s\n", probString);
        }

    return;
}

void ReportError(PFUNCRETURN pfr)
{
    BOOL printErrorLoc=TRUE;

    if (!strcmp(pfr->errorFunc, "badbitmapsexist"))
        {
        printf("\nAt least one bad bitmap sector exists on your drive.\n");
        printf("\nThis program does not yet know how to deal with that.\n");
        printErrorLoc=FALSE;
        }
    else if (!strcmp(pfr->errorFunc, "endpastend"))
        {
        printf("\nGiven end sector exceeds sectors on drive.\n");
        printErrorLoc=FALSE;
        }
    else if (!strcmp(pfr->errorFunc, "malloc"))
        {
        printf("\nThere was a memory allocation failure.\n");
        }
    else if (!strcmp(pfr->errorFunc, "nolockdrive"))
        {
        printf("\nThe program was unable to lock the drive.\n");
        printf("\nDosDevIOCtl() returned %u.\n", pfr->errorCode);
        printErrorLoc=FALSE;
        }
    else
        {
        printf("\nThe function %s() failed with error %u.\n", pfr->errorFunc, pfr->errorCode);
        }
    if (printErrorLoc)
        printf("\nThe error occurred inside the %s() function.\n", pfr->errorHome);

    return;
}

void ShowSuperBlock(PSUPERBLOCK sub, ULONG sectorSize, UCHAR driveLetter, BOOL noLineChars)
{
    UCHAR lineBuff[81], obuf[7], hbuf[9], dbuf[27], lbuf[33], tbuf[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;
    ULONG offset, i;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    PrintSPTopBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    memset(tbuf, 0, sizeof(tbuf));
    lineBuff[0]=vertBar;

    memset(lineBuff+1, ' ', 15);

    sprintf(tbuf, "HPFS SuperBlock for drive %c: - %s bytes/sector", driveLetter, iCodel(sectorSize));
    strcat(lineBuff, tbuf);

    memset(lineBuff+strlen(lineBuff), ' ', 78-strlen(lineBuff));

    lineBuff[78]=vertBar;
    printf("%s\n", lineBuff);

    PrintSPTMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(obuf, "Offset");
    strcpy(hbuf, "   Hex  ");
    strcpy(dbuf, " Formatted/Decimal Value  ");
    strcpy(lbuf, "         Value Info         ");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    PrintSPMiddleBorder(noLineChars);

    offset=0;

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->signature[0]);
    sprintf(dbuf, "%s", iCodel(sub->signature[0]));
    strcpy(lbuf, "64-bit SuperBlock signature");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->signature[0]);
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    memset(obuf, 0, sizeof(obuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->signature[1]);
    sprintf(dbuf, "%s", iCodel(sub->signature[1]));
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->signature[1]);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%02X", sub->version);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->version));
    strcpy(lbuf, "HPFS version");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->version);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%02X", sub->funcVersion);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->funcVersion));
    strcpy(lbuf, "Vol size (2:<=4GB; 3:>4GB)");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->funcVersion)+2;
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->rootFNode);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->rootFNode));
    strcpy(lbuf, "Root dir FNode location");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->rootFNode);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->sectorCount);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->sectorCount));
    strcpy(lbuf, "Total sector count");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->sectorCount);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->badSectorCount);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->badSectorCount));
    strcpy(lbuf, "Bad sector count");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->badSectorCount);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->bitmapList);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->bitmapList));
    strcpy(lbuf, "Free space bitmap list");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->bitmapList);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->spareBitmapList);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->spareBitmapList));
    strcpy(lbuf, "Spare bitmap list");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->spareBitmapList);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->badSectorList);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->badSectorList));
    strcpy(lbuf, "Bad sector list");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->badSectorList);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->spareBadSectorList);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->spareBadSectorList));
    strcpy(lbuf, "Spare bad sector list");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->spareBadSectorList);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->lastCheckDate);
    if (sub->lastCheckDate>0)
        sprintf(dbuf, "%s", StrLeft(ctime((time_t*)&sub->lastCheckDate), 24, tbuf));
    else
        strcpy(dbuf, "Never");
    strcpy(lbuf, "Date CHKDSK /F last run");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->lastCheckDate);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->lastOptimizedDate);
    if (sub->lastOptimizedDate>0)
        sprintf(dbuf, "%s", StrLeft(ctime((time_t*)&sub->lastOptimizedDate), 24, tbuf));
    else
        strcpy(dbuf, "Never");
    strcpy(lbuf, "Date last optimized");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->lastOptimizedDate);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->dirBandSectorCount);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->dirBandSectorCount));
    strcpy(lbuf, "Dir band sector count");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->dirBandSectorCount);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->dirBandStart);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->dirBandStart));
    strcpy(lbuf, "Directory band start");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->dirBandStart);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->dirBandEnd);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->dirBandEnd));
    strcpy(lbuf, "Directory band end");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->dirBandEnd);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->dirBandBitmap);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->dirBandBitmap));
    strcpy(lbuf, "Directory band bitmap");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(sub->dirBandBitmap)+32;
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", sub->securityBlock);
    sprintf(dbuf, "%s", iCodel((ULONG)sub->securityBlock));
    strcpy(lbuf, "First ACL block location");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);

    PrintSPBottomBorder(noLineChars);

    return;
}

void ShowSpareBlock(PSPAREBLOCK spb, ULONG sectorSize, UCHAR driveLetter, BOOL fullDBList, BOOL noLineChars)
{
    UCHAR lineBuff[81], obuf[7], hbuf[9], dbuf[27], lbuf[33], tbuf[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;
    ULONG offset, i;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    PrintSPTopBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    memset(tbuf, 0, sizeof(tbuf));
    lineBuff[0]=vertBar;

    memset(lineBuff+1, ' ', 15);

    sprintf(tbuf, "HPFS SpareBlock drive %c: - %s bytes/sector", driveLetter, iCodel(sectorSize));
    strcat(lineBuff, tbuf);

    memset(lineBuff+strlen(lineBuff), ' ', 78-strlen(lineBuff));

    lineBuff[78]=vertBar;
    printf("%s\n", lineBuff);

    PrintSPTMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(obuf, "Offset");
    strcpy(hbuf, "   Hex  ");
    strcpy(dbuf, " Formatted/Decimal Value  ");
    strcpy(lbuf, "         Value Info         ");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    PrintSPMiddleBorder(noLineChars);

    offset=0;

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->signature[0]);
    sprintf(dbuf, "%s", iCodel(spb->signature[0]));
    strcpy(lbuf, "64-bit SpareBlock signature");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->signature[0]);
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    memset(obuf, 0, sizeof(obuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->signature[1]);
    sprintf(dbuf, "%s", iCodel(spb->signature[1]));
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->signature[1]);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%02X", spb->partStatus);
    sprintf(dbuf, "%s", BitString(spb->partStatus));
    strcpy(lbuf, "Partition status byte");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->partStatus)+3;
    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(dbuf, "^ ^ ^ ^ ^ ^ ^ ^ ");
    strcpy(lbuf, "    0: Dirty flag");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(dbuf, "7 6 5 4 3 2 1 0 ");
    strcpy(lbuf, "    1: Spare DirBlk's used");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    2: Hotfix sectors used");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    3: Bad sectors exist");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    4: Bad bitmaps exist");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    5: Fast formatted");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    6: Unknown");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    memset(lbuf, 0, sizeof(lbuf));
    strcpy(lbuf, "    7: Written by old IFS");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->hotfixList);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->hotfixList));
    strcpy(lbuf, "Hotfix list location");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->hotfixList);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->hotfixUsed);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->hotfixUsed));
    strcpy(lbuf, "Hotfix sectors in use");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->hotfixUsed);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->hotfixTotal);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->hotfixTotal));
    strcpy(lbuf, "Total hotfix sectors");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->hotfixTotal);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->spareDirBlockCount);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->spareDirBlockCount));
    strcpy(lbuf, "Spare DirBlk count");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->spareDirBlockCount);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->spareDirBlockFree);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->spareDirBlockFree));
    strcpy(lbuf, "Free spare DirBlk's");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->spareDirBlockFree);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->codePageSector);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->codePageSector));
    strcpy(lbuf, "Code page list location");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->codePageSector);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->codePageCount);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->codePageCount));
    strcpy(lbuf, "Number of code pages");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->codePageCount);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->superBlockCRC);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->superBlockCRC));
    strcpy(lbuf, "HPFS SuperBlock 32-bit CRC");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->superBlockCRC);
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->spareBlockCRC);
    sprintf(dbuf, "%s", iCodel((ULONG)spb->spareBlockCRC));
    strcpy(lbuf, "HPFS SpareBlock 32-bit CRC");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->spareBlockCRC)+60;
    PrintSPMiddleBorder(noLineChars);

    memset(obuf, 0, sizeof(obuf));
    memset(hbuf, 0, sizeof(hbuf));
    memset(dbuf, 0, sizeof(dbuf));
    memset(lbuf, 0, sizeof(lbuf));
    sprintf(obuf, "%02X/%03u", offset, offset+1);
    sprintf(hbuf, "%08X", spb->firstSpareDirBlock[0]);
    sprintf(dbuf, "%s", iCodel(spb->firstSpareDirBlock[0]));
    strcpy(lbuf, "Spare DirBlk #1 location");
    PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
    offset+=sizeof(spb->firstSpareDirBlock[0]);
    if (fullDBList)
        {
        for (i=1; i<spb->spareDirBlockCount; i++)
            {
            memset(obuf, 0, sizeof(obuf));
            memset(hbuf, 0, sizeof(hbuf));
            memset(dbuf, 0, sizeof(dbuf));
            memset(lbuf, 0, sizeof(lbuf));
            sprintf(obuf, "%02X/%03u", offset, offset+1);
            sprintf(hbuf, "%08X", spb->firstSpareDirBlock[i]);
            sprintf(dbuf, "%s", iCodel(spb->firstSpareDirBlock[i]));
            sprintf(lbuf, "Spare DirBlk #%u location", i+1);
            PrintSPDataLine(obuf, hbuf, dbuf, lbuf, noLineChars);
            offset+=sizeof(spb->firstSpareDirBlock[i]);
            }
        }

    PrintSPBottomBorder(noLineChars);

    return;
}

void ShowSystemUsage(UCHAR driveLetter, PSYSTEMUSAGE su, BOOL noLineChars)
{
    ULONG i;
    UCHAR horzBar=196;
    UCHAR lineBuff[81];

    if (noLineChars)
        {
        horzBar='-';
        }

    printf("\n    Drive %c: - %s bytes - %s bytes/sector\n", driveLetter,
           fCodel((double)su->sectorSize*(double)su->totalSectors),
           iCodel(su->sectorSize));

    printf(" ");
    memset(lineBuff, 0, sizeof(lineBuff));

    memset(lineBuff, horzBar, 77);

    printf("%s", lineBuff);

    printf("\n%12s - Directory band sectors\n",
           fCodel((double)su->dirBandSectors*(double)su->sectorSize));
    printf("%12s - Free space bitmaps\n",
           fCodel((double)su->bitmapSectors*(double)su->sectorSize));
    printf("%12s - Hotfix sectors (not necessarily in use)\n",
           fCodel((double)su->hotfixSectors*(double)su->sectorSize));
    printf("%12s - Spare directory blocks (4 sectors per block)\n",
           fCodel((double)su->spareDirBlockSectors*(double)su->sectorSize));
    printf("%12s - Front structures (BootBlock, mini-FSD, etc.)\n",
           fCodel((double)su->frontStructures*(double)su->sectorSize));
    printf("%12s - Initial security ACL block\n",
           fCodel((double)su->securityBlockSectors*(double)su->sectorSize));
    printf("%12s - Free space bitmap list sectors (shows location of bitmaps)\n",
           fCodel((double)su->bitmapListSectors*(double)su->sectorSize));
    printf("%12s - Directory band bitmap sectors\n",
           fCodel((double)su->dirBandBitmapSectors*(double)su->sectorSize));
    printf("%12s - Bad sector list (even if there are none)\n",
           fCodel((double)su->badSectorListSectors*(double)su->sectorSize));
    printf("%12s - Hot fix list sectors (location pointers)\n",
           fCodel((double)su->hotfixListSectors*(double)su->sectorSize));
    printf("%12s - Root directory DirBlock\n",
           fCodel((double)su->rootDirBlock*(double)su->sectorSize));
    printf("%12s - Code page sectors\n",
           fCodel((double)su->codePageSectors*(double)su->sectorSize));
    printf("%12s - Root directory FNode\n",
           fCodel((double)su->rootFNode*(double)su->sectorSize));

    printf(" ");
    memset(lineBuff, 0, sizeof(lineBuff));

    memset(lineBuff, horzBar, 77);

    printf("%s", lineBuff);

    printf("\n");

    printf("%12s - Total system usage\n",
           fCodel((double)su->totalSystemSectors*(double)su->sectorSize));

    return;
}

void ShowFreeSpaceInfo(PFREESPACEINFO fsi, UCHAR driveLetter, BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81], secBuff[14], byteBuff[18];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    PrintFSTopBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=vertBar;
    strcat(lineBuff, "    Drive ");
    lineBuff[11]=driveLetter;
    strcat(lineBuff, ": ");
    lineBuff[14]=vertBar;
    strcat(lineBuff, " Free Sectors ");
    lineBuff[29]=vertBar;
    strcat(lineBuff, "      Free Bytes ");
    lineBuff[47]=vertBar;
    printf("%s\n", lineBuff);

    PrintFSMiddleBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=vertBar;
    strcat(lineBuff, "    Reported ");
    lineBuff[14]=vertBar;
    memset(secBuff, 0, sizeof(secBuff));
    sprintf(secBuff, "%13s", iCodel(fsi->reportedFreeSectors));
    strcat(lineBuff, secBuff);
    lineBuff[28]=' ';
    lineBuff[29]=vertBar;
    memset(byteBuff, 0, sizeof(byteBuff));
    sprintf(byteBuff, "%16s", fCodel((double)fsi->reportedFreeSectors*(double)fsi->sectorSize));
    strcat(lineBuff, byteBuff);
    lineBuff[46]=' ';
    lineBuff[47]=vertBar;
    printf("%s\n", lineBuff);

    PrintFSMiddleBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=vertBar;
    strcat(lineBuff, "      Actual ");
    lineBuff[14]=vertBar;
    memset(secBuff, 0, sizeof(secBuff));
    sprintf(secBuff, "%13s", iCodel(fsi->actualFreeSectors));
    strcat(lineBuff, secBuff);
    lineBuff[28]=' ';
    lineBuff[29]=vertBar;
    memset(byteBuff, 0, sizeof(byteBuff));
    sprintf(byteBuff, "%16s", fCodel((double)fsi->actualFreeSectors*(double)fsi->sectorSize));
    strcat(lineBuff, byteBuff);
    lineBuff[46]=' ';
    lineBuff[47]=vertBar;
    printf("%s\n", lineBuff);

    PrintFSMiddleBorder(noLineChars);

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=vertBar;
    strcat(lineBuff, " Discrepancy ");
    lineBuff[14]=vertBar;
    memset(secBuff, 0, sizeof(secBuff));
    sprintf(secBuff, "%13s", iCodel(fsi->actualFreeSectors-fsi->reportedFreeSectors));
    strcat(lineBuff, secBuff);
    lineBuff[28]=' ';
    lineBuff[29]=vertBar;
    memset(byteBuff, 0, sizeof(byteBuff));
    sprintf(byteBuff, "%16s", fCodel((double)fsi->actualFreeSectors*(double)fsi->sectorSize-
                                     (double)fsi->reportedFreeSectors*(double)fsi->sectorSize));
    strcat(lineBuff, byteBuff);
    lineBuff[46]=' ';
    lineBuff[47]=vertBar;
    printf("%s\n", lineBuff);

    PrintFSBottomBorder(noLineChars);

    return;
}

void PrintFSTopBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=topLeftCorner;

    memset(lineBuff+1, horzBar, 13);

    lineBuff[14]=teeBarTop;

    memset(lineBuff+15, horzBar, 14);

    lineBuff[29]=teeBarTop;

    memset(lineBuff+30, horzBar, 17);

    lineBuff[47]=topRightCorner;
    printf("\n%s\n", lineBuff);

    return;
}

void PrintFSMiddleBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=teeBarLeft;

    memset(lineBuff+1, horzBar, 13);

    lineBuff[14]=crossBar;

    memset(lineBuff+15, horzBar, 14);

    lineBuff[29]=crossBar;

    memset(lineBuff+30, horzBar, 17);

    lineBuff[47]=teeBarRight;
    printf("%s\n", lineBuff);

    return;
}

void PrintFSBottomBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=botLeftCorner;

    memset(lineBuff+1, horzBar, 13);

    lineBuff[14]=teeBarBot;

    memset(lineBuff+15, horzBar, 14);

    lineBuff[29]=teeBarBot;

    memset(lineBuff+30, horzBar, 17);

    lineBuff[47]=botRightCorner;
    printf("%s\n", lineBuff);

    return;
}

void PrintSPTopBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=topLeftCorner;

    memset(lineBuff+1, horzBar, 77);

    lineBuff[78]=topRightCorner;

    printf("\n%s\n", lineBuff);

    return;
}

void PrintSPTMiddleBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=teeBarLeft;

    memset(lineBuff+1, horzBar, 8);

    lineBuff[9]=teeBarTop;

    memset(lineBuff+10, horzBar, 10);

    lineBuff[20]=teeBarTop;

    memset(lineBuff+21, horzBar, 26);

    lineBuff[47]=teeBarTop;

    memset(lineBuff+48, horzBar, 30);

    lineBuff[78]=teeBarRight;
    printf("%s\n", lineBuff);

    return;
}

void PrintSPMiddleBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=teeBarLeft;

    memset(lineBuff+1, horzBar, 8);

    lineBuff[9]=crossBar;

    memset(lineBuff+10, horzBar, 10);

    lineBuff[20]=crossBar;

    memset(lineBuff+21, horzBar, 26);

    lineBuff[47]=crossBar;

    memset(lineBuff+48, horzBar, 30);

    lineBuff[78]=teeBarRight;
    printf("%s\n", lineBuff);

    return;
}

void PrintSPBottomBorder(BOOL noLineChars)
{
    ULONG i;
    UCHAR lineBuff[81];
    UCHAR vertBar=179, horzBar=196, topLeftCorner=218,
          botLeftCorner=192, topRightCorner=191,
          botRightCorner=217, teeBarTop=194, teeBarBot=193,
          teeBarRight=180, teeBarLeft=195, crossBar=197;

    if (noLineChars)
        {
        vertBar='|'; horzBar='-'; topLeftCorner='/';
        botLeftCorner='\\'; topRightCorner='\\';
        botRightCorner='/'; teeBarTop='-'; teeBarBot='-';
        teeBarRight='|'; teeBarLeft='|'; crossBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=botLeftCorner;

    memset(lineBuff+1, horzBar, 8);

    lineBuff[9]=teeBarBot;

    memset(lineBuff+10, horzBar, 10);

    lineBuff[20]=teeBarBot;

    memset(lineBuff+21, horzBar, 26);

    lineBuff[47]=teeBarBot;

    memset(lineBuff+48, horzBar, 30);

    lineBuff[78]=botRightCorner;
    printf("%s\n", lineBuff);

    return;
}

void PrintSPDataLine(PUCHAR off, PUCHAR hex, PUCHAR dec, PUCHAR label, BOOL noLineChars)
{
    UCHAR lineBuff[81], tbuf[81];
    UCHAR vertBar=179;

    if (noLineChars)
        {
        vertBar='|';
        }

    memset(lineBuff, 0, sizeof(lineBuff));
    lineBuff[0]=vertBar;
    lineBuff[1]=' ';
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(tbuf, "%6s", off);
    strcat(lineBuff, tbuf);
    lineBuff[8]=' ';
    lineBuff[9]=vertBar;
    lineBuff[10]=' ';
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(tbuf, "%8s", hex);
    strcat(lineBuff, tbuf);
    lineBuff[19]=' ';
    lineBuff[20]=vertBar;
    lineBuff[21]=' ';
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(tbuf, "%24s", dec);
    strcat(lineBuff, tbuf);
    lineBuff[46]=' ';
    lineBuff[47]=vertBar;
    lineBuff[48]=' ';
    memset(tbuf, 0, sizeof(tbuf));
    sprintf(tbuf, "%-30s", label);
    strcat(lineBuff, tbuf);
    lineBuff[77]=' ';
    lineBuff[78]=vertBar;
    printf("%s\n", lineBuff);

    return;
}
