/*
 * EqTree
 * Equal Tree
 *
 * Vergleicht zwei Verzeichnisbume und stellt im Zielverzeichnisbaum
 * den Zustand des Quellverzeichnisbaums her.
 * Gelschte Dateien werden bis zu einer einstellbaren Anzahl von Tagen
 * beibehalten.
 *
 * Autor: SG
 * Stand: 5.4.94
 */

#define VERVersionString "Version 2.5\n" \
			 "\t(c) 1993,94 ExperTeam GmbH\n" \
			 "\twritten by Stephan Gerle\n"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "treeload.h"
#include "copy.h"
#include "logfile.h"

char *MAProgramName;
char *MASourceDir = NULL;
char *MADestDir = NULL;
char *MALogFile = "EQTREE.LOG";

int MAMaxAge = 7; /* eine Woche */
int MASafety = 1; /* Sicherheitsabfrage */

int MAEqual(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest);
int MANewer(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest);
int MAOlder(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest);
int MASrcNew(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest);
int MADstNew(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest);

TRTree *MASourceTree = NULL;
TRTree *MADestTree = NULL;
TRFuncEntry MAFuncArray[] = {
	MAEqual, /* Quelle und Ziel sind identisch */
	MANewer, /* Quelle ist neuer als Ziel */
	MAOlder, /* Quelle ist lter als Ziel */
	MASrcNew,/* Quelle ist neu */
	MADstNew/* Destination ist neu */
};

int MAEqual(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest)
{
    /*LOItem("File is equal! Nothing done '%s'",Source->File);*/
    return 1;
}

int MANewer(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest)
{
    if (TRIsDir(Source))
	return 1;

    LOItem("Src is newer! Copying '%s' -> '%s'",Source->File,Dest->File);
    if (COCopy(Source->File,Dest->File)==0)
	return 1;
    ERMessage("Copy failed ***");
    return 0;
}

#include <time.h>
#include <sys\types.h>
#include <sys\stat.h>

static char *File2DateStr(const char *Filename)
{
    struct stat statbuf;

    if (stat((char *)Filename,&statbuf)==0)
	asctime(gmtime(&statbuf.st_atime));
    else
	return "Unknown time/date";
}

int MAOlder(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest)
{
    if (TRIsDir(Source))
	return 1;

    LOItem("Src is older! Should not occur! Nothing done! '%s %s' ***",Source->File,File2DateStr(Source->File));
    return 1;
}

int MASrcNew(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest)
{
#define CDBUFSIZE   2048

    char    DestFileBuf[CDBUFSIZE];

    if (TRIsDir(Source))
	return 1;

    if (TDest->BaseLen+strlen(Source->File)-TSource->BaseLen+1 > CDBUFSIZE)
    {
	ERMessage("Unable to copy '%s' because Buffer is to small",Source->File);
	return 0;
    }
    strcpy(DestFileBuf,TDest->Base);
    strcat(DestFileBuf,Source->File+TSource->BaseLen);

    LOItem("Src is new! Copying '%s' -> '%s'",Source->File,DestFileBuf);
    if (COCopy(Source->File,DestFileBuf)==0)
	return 1;
    ERMessage("Copy failed ***");
    return 0;
}

unsigned long Date2Days(FDATE date)
{
    return date.year * 365 + date.month *31 + date.day;
}

unsigned long Time2Secs(FTIME time)
{
    return time.hours * 3600l + time.minutes * 60 + time.twosecs * 2;
}


#include <direct.h>

static int MARemoveIfEmpty(char *Path)
{
    int ret;

    if (Path[1] == ':')
	_chdrive(toupper(*Path)-'A'+1);

    if (chdir(Path)==0)
    {
	chdir("\\");
	ret = rmdir(Path);
	if (ret == 0)
	    LOItem("Path '%s' removed because is empty",Path);
    }

    return 1;
}

int MADstNew(TRTree *TSource,TREntry *Source,TRTree *TDest,TREntry *Dest)
{
    FDATE date;
    struct tm *ltm;
    time_t    ttime = time(NULL);
    int 	DelIt;

    ltm = localtime(&ttime);
    date.year = ltm->tm_year - 80;
    date.month = ltm->tm_mon + 1;
    date.day = ltm->tm_mday;
    if (Date2Days(date) - MAMaxAge > Date2Days(Dest->Date))
	DelIt = 1;
    else
	DelIt = 0;

    if (TRIsDir(Dest))
    {
	MARemoveIfEmpty(Dest->File);
    }
    else
    {
	if (DelIt)
	{
	    LOItem("Src deleted! Deleting '%s'",Dest->File);
	    COMakeWritable(Dest->File);
	    if (remove(Dest->File)!=0)
		ERMessage("File '%s' could not be deleted ***",Dest->File);
	    if (strchr(Dest->File,'\\'))
		*strchr(Dest->File,'\\') = 0;
	    MARemoveIfEmpty(Dest->File);
	}
	else
	    LOItem("Src deleted! Dest not old enough! Nothing done! '%s' ***",Dest->File);
    }
    return 1;
}

void DoHelp(char *argv[])
{
    fprintf(stderr, "Usage: %s [Options] <SourceDir> <DestDir>\n"
		    "Options: -h this help message\n"
		    "         -l:<LogFileName> Change Logfile to ... Default is EQTREE.LOG\n"
		    "         -d:<Days> Change max age to ... Default is 7\n"
		    "         -q Quiet: Keine Sicherheitsabfragen\n"
		    "         -v: get version\n\n",
		    argv[0]);
}

int ParseOptions(int argc,char *argv[])
{
    int i;

    for (i = 1; i < argc; i++)
    {
	if (strcmp(argv[i],"-h")==0)
	{
	    DoHelp(argv);
	    return 0;
	}
	if (strncmp(argv[i],"-v",2)==0)
	{
	    fprintf(stderr,"%s: %s",argv[0],VERVersionString);
	    exit(1);
	}

	if (strncmp(argv[i],"-l:",3)==0)
	    MALogFile = argv[i]+3;

	if (strncmp(argv[i],"-d:",3)==0)
	    MAMaxAge = atoi(argv[i]+3);

	if (strcmp(argv[i],"-q")==0)
	    MASafety = 0;

	if (*argv[i] != '-')
	    if (MASourceDir == NULL)
		MASourceDir = argv[i];
	    else
		if (MADestDir == NULL)
		    MADestDir = argv[i];
		else
		{
		    fprintf(stderr,"%s: Illegal Commandline '%s'\n",
				    argv[0],argv[i]);
		    DoHelp(argv);
		    return 0;
		}
    }
    if ((MADestDir == NULL) || (MASourceDir == NULL))
    {
	fprintf(stderr,"%s: Kommandline missing Source- and/or Destination directory\n",argv[0]);
	DoHelp(argv);
	return 0;
    }
    return 1;
}

int main(int argc,char *argv[])
{
    fprintf(stdout,"\n");
    MAProgramName = argv[0];

    if (ParseOptions(argc,argv))
    {
	time_t	ltime;

	LOInit(MALogFile);

	time(&ltime);
	LOItem( "Started              = %s"
		"Maximum age          = %d days\n"
		"Logfile              = %s\n"
		"Sourcedirectory      = %s\n"
		"Destinationdirectory = %s\n"
		"Starting Process....",
	    asctime(localtime(&ltime)),
	    MAMaxAge,MALogFile,MASourceDir,MADestDir);

	if ((MASourceTree = TRLoadTree(MASourceDir))==NULL)
	    ERFatal("Source-Directory '%s' does not exist.",MASourceDir);

	if ((MADestTree = TRLoadTree(MADestDir))==NULL)
	    ERFatal("Destination-Directory '%s' does not exist.",MADestDir);

	if (MASafety)
	{
	    if ((labs(MASourceTree->EntryCount-MADestTree->EntryCount)/2) >
		(long)(MASourceTree->EntryCount+MADestTree->EntryCount)/20)
	    {
		int answer;

		fprintf(stderr,"%s: Source and Destination different?!?\n"
				"\tContinue (y/n)?",argv[0]);
		do {
		    answer = toupper(getchar());
		} while ((answer != 'Y') && (answer != 'N'));
		if (answer == 'N')
		    exit(1);
	    }
	}

	if (!TRCompare(MASourceTree,MADestTree,MAFuncArray))
	    ERFatal("Treecompare *not* successfully terminated");

	time(&ltime);
	ERMessage("Treecompare successfully terminated at %s",asctime(localtime(&ltime)));
	TRFree(MASourceTree);
	TRFree(MADestTree);
	return 0;
    }
    else
	exit(1);
}
