/* Copyright Trevor Hemsley. Portions of code by Daniela Engert and Bjorn Mork */
/* Released under GPL */

#define INCL_DOSEXCEPTIONS
#define INCL_KBD
#define INCL_DOS
#define INCL_PM
#define INCL_GPILCIDS
#include <OS2.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include "pmdskmon.h"

#pragma pack(1)

#define DSKSP_CAT_SMART                      0x80  /* SMART IOCTL category */

#define DSKSP_SMART_ONOFF                    0x20  /* turn SMART on or off */
#define DSKSP_SMART_AUTOSAVE_ONOFF  0x21  /* turn SMART autosave on or off */
#define DSKSP_SMART_SAVE                     0x22  /* force save of SMART data */
#define DSKSP_SMART_GETSTATUS                0x23  /* get SMART status (pass/fail) */
#define DSKSP_SMART_GET_ATTRIBUTES  0x24  /* get SMART attributes table */
#define DSKSP_SMART_GET_THRESHOLDS  0x25  /* get SMART thresholds table */

#define SMART_CMD_ON                       1        /* on value for related SMART functions */
#define SMART_CMD_OFF                      0        /* off value for related SMART functions */

#define DSKSP_CAT_GENERIC                    0x90  /* generic IOCTL category */
#define DSKSP_GEN_GET_COUNTERS               0x40  /* get general counter values table */
#define DSKSP_GET_UNIT_INFORMATION  0x41  /* get unit configuration and BM DMA c*/
#define DSKSP_GET_INQUIRY_DATA               0x42  /* get ATA/ATAPI inquiry data */

typedef struct _DSKSP_CommandParameters {
  BYTE byPhysicalUnit;                         /* physical unit number 0-n */
               /* 0 = 1st disk, 1 = 2nd disk, ...*/
               /* 0x80 = Pri/Mas, 0x81=Pri/Sla, 0x82=Sec/Mas,*/
} DSKSP_CommandParameters, *PDSKSP_CommandParameters;

/*
 * Parameters for SMART and generic commands
 */

/*
 * SMART Attribute table item
 */

typedef struct _S_Attribute
{
  BYTE         byAttribID;                    /* attribute ID number */
  USHORT      wFlags;                            /* flags */
  BYTE         byValue;                          /* attribute value */
  BYTE         byVendorSpecific[8];        /* vendor specific data */
} S_Attribute;

/*
 * SMART Attribute table structure
 */

typedef struct _DeviceAttributesData
{
  USHORT      wRevisionNumber;                /* revision number of attribute table */
  S_Attribute Attribute[30];                  /* attribute table */
  BYTE         byReserved[6];                 /* reserved bytes */
  USHORT      wSMART_Capability;           /* capabilities word */
  BYTE         byReserved2[16];               /* reserved bytes */
  BYTE         byVendorSpecific[125];      /* vendor specific data */
  BYTE         byCheckSum;                    /* checksum of data in this structure */
} DeviceAttributesData, NEAR *NPDeviceAttributesData, FAR *PDeviceAttributesData;

/*
 * SMART Device Threshold table item
 */

typedef struct _S_Threshold
{
  BYTE         byAttributeID;                 /* attribute ID number */
  BYTE         byValue;                          /* threshold value */
  BYTE         byReserved[10];                /* reserved bytes */
} S_Threshold;

/*
 * SMART Device Threshold table
 */

typedef struct _DeviceThresholdsData
{
  USHORT      wRevisionNumber;                /* table revision number */
  S_Threshold Threshold[30];                  /* threshold table */
  BYTE         byReserved[18];                /* reserved bytes */
  BYTE         VendorSpecific[131];        /* vendor specific data */
  BYTE         byCheckSum;                    /* checksum of data in this structure */
} DeviceThresholdsData, NEAR *NPDeviceThresholdsData, FAR *PDeviceThresholdsData;

/*
 * Unit Configuration and Counters
 */

typedef struct _UnitInformationData
{
  USHORT wRevisionNumber;                    /* structure revision number */
  USHORT wTaskFileBase;                         /* task file register base addr */
  USHORT wAlternateStatusAddress;    /* alternate status register addr */
  USHORT wIRQ;                               /* interrupt request level */
  USHORT wFlags;                             /* flags */
  BYTE         byPIO_Mode;                    /* PIO transfer mode programmed */
  BYTE         byDMA_Mode;                    /* DMA transfer mode programmed */

} UnitInformationData, *PUnitInformationData;

/*
 * Unit Information Flags Definitions
 */

#define UIF_VALID     0x8000               /* unit information valid */
#define UIF_TIMINGS_VALID   0x4000         /* timing information valid */
#define UIF_RUNNING_BMDMA   0x2000         /* running Bus Master DMA on unit */
#define UIF_RUNNING_DMA     0x1000         /* running slave DMA on unit */
#define UIF_SLAVE     0x0002               /* slave on channel */
#define UIF_ATAPI     0x0001               /* ATAPI device if 1, ATA otherwise */

typedef struct _DeviceCountersData
{
  USHORT      wRevisionNumber;                /* counter structure revision */
  ULONG       TotalReadOperations;         /* total read operations performed */
  ULONG       TotalWriteOperations;        /* total write operations performed */
  ULONG       TotalWriteErrors;            /* total write errors encountered */
  ULONG       TotalReadErrors;                /* total read errors encountered */
  ULONG       TotalSeekErrors;                /* total seek errors encountered */
  ULONG       TotalSectorsRead;            /* total number of sectors read */
  ULONG       TotalSectorsWritten;         /* total number of sectors written */

  ULONG       TotalBMReadOperations;       /* total bus master DMA read operations */
  ULONG       TotalBMWriteOperations;      /* total bus master DMA write operations */
  ULONG       ByteMisalignedBuffers;       /* total buffers on odd byte boundary */
  ULONG       TransfersAcross64K;          /* total buffers crossing a 64K page boundary */
  USHORT      TotalBMStatus;                  /* total bad busmaster status */
  USHORT      TotalBMErrors;                  /* total bad busmaster error */
  ULONG       TotalIRQsLost;                  /* total lost interrupts */
  USHORT      TotalDRQsLost;                  /* total lost data transfer requests */
  USHORT      TotalBusyErrors;                /* total device busy timeouts        */
  USHORT      TotalBMStatus2;                 /* total bad busmaster status */
  USHORT      TotalChipStatus;                /* total bad chip status */
  USHORT      ReadErrors[4];
  USHORT      WriteErrors[2];
  USHORT      SeekErrors[2];
} DeviceCountersData, *PDeviceCountersData;

/* Identify Data */

typedef struct _IDENTIFYDATA  *PIDENTIFYDATA;

typedef struct _IDENTIFYDATA
{
  USHORT GeneralConfig;                     /*  0 General configuration bits      */
  USHORT TotalCylinders;                 /*  1 Default Translation - Num cyl   */
  USHORT Reserved;                          /*  2 Reserved             */
  USHORT NumHeads;                          /*  3         - Num heads */
  USHORT NumUnformattedbpt;              /*  4 Unformatted Bytes   - Per track */
  USHORT NumUnformattedbps;              /*  5         - Per sector*/
  USHORT SectorsPerTrack;                /*  6 Default Translation - Sec/Trk   */
  USHORT NumBytesISG;                       /*  7 Byte Len - inter-sector gap     */
  USHORT NumBytesSync;                      /*  8        - sync field        */
  USHORT NumWordsVUS;                       /*  9 Len - Vendor Unique Info         */
  CHAR      SerialNum[20];                  /* 10 Serial number           */
  USHORT CtrlType;                          /* 20 Controller Type            */
  USHORT CtrlBufferSize;                 /* 21 Ctrl buffer size - Sectors      */
  USHORT NumECCBytes;                       /* 21 ECC bytes -  read/write long    */
  CHAR      FirmwareRN[8];                  /* 23 Firmware Revision          */
  CHAR      ModelNum[40];                   /* 27 Model number            */
  USHORT NumSectorsPerInt;               /* 47 Multiple Mode - Sec/Blk       */
  USHORT DoubleWordIO;                      /* 48 Double Word IO Flag  */
  USHORT IDECapabilities;                /* 49 Capability Flags Word   */
  USHORT Reserved2;                         /* 50           */
  USHORT PIOCycleTime;                      /* 51 Transfer Cycle Timing - PIO     */
  USHORT DMACycleTime;                      /* 52           - DMA     */
  USHORT AdditionalWordsValid;           /* 53 Additional Words valid  */
  USHORT LogNumCyl;                         /* 54 Current Translation - Num Cyl   */
  USHORT LogNumHeads;                       /* 55           Num Heads */
  USHORT LogSectorsPerTrack;             /* 56           Sec/Trk   */
  ULONG  LogTotalSectors;                /* 57           Total Sec */
  USHORT LogNumSectorsPerInt;            /* 59                */
  ULONG  LBATotalSectors;                /* 60 LBA Mode - Sectors         */
  USHORT DMASWordFlags;                     /* 62                */
  USHORT DMAMWordFlags;                     /* 63                */
  USHORT AdvancedPIOModes;               /* 64 Advanced PIO modes supported */
  USHORT MinMWDMACycleTime;              /* 65 Minimum multiWord DMA cycle time */
  USHORT RecMWDMACycleTime;              /* 66 Recommended MW DMA cycle time */
  USHORT MinPIOCycleTimeWOFC;            /* 67 Minimum PIO cycle time without IORDY */
  USHORT MinPIOCycleTime;                /* 68 Minimum PIO cycle time  */
  USHORT Reserved3[82-69];               /* 69        */
  USHORT CommandSetSupported[3]; /* 82          */
  USHORT CommandSetEnabled[3];           /* 85        */
  USHORT UltraDMAModes;                     /* 88 Ultra DMA Modes    */
  USHORT Reserved4[93-89];               /* 89        */
  USHORT HardwareTestResult;             /* 93 hardware test result*/
  USHORT Reserved5[127-94];              /* 94         */
  USHORT MediaStatusWord;                /* 127 media status Word  */
  USHORT Reserved6[256-128];             /*         */
}IDENTIFYDATA;

#define CMD_VERBOSE   0x01
#define CMD_COUNTERS  0x02
#define CMD_IDENTIFY  0x04
#define CMD_SMARTDATA 0x08

typedef struct _STATSDATA
   {
   char        Model[41];
   BYTE        Options;
   #define     UNIT_PRESENT              0x01
   #define     UNIT_HASSMART             0x02
   #define     UNIT_FAILEDSMART          0x04
   BYTE        Flags;
   USHORT      Port;
   USHORT      AltPort;
   USHORT      IRQ;
   ULONG       TotalReadOperations;         /* total read operations performed */
   ULONG       TotalWriteOperations;        /* total write operations performed */
   ULONG       TotalErrors;                 /* total read/write/seek errors encountered */
   ULONG       TotalSectorsRead;            /* total number of sectors read */
   ULONG       TotalSectorsWritten;         /* total number of sectors written */
   ULONG       CurrentReadOps;
   ULONG       CurrentWriteOps;
   ULONG       CurrentErrors;
   ULONG       CurrentKBsRead;
   ULONG       CurrentKBsWrite;
   } STATSDATA;

STATSDATA               sd[16];
char                    Model[16][40];
DSKSP_CommandParameters Parms;
ULONG                   PLen = 1, IDLen = 512, UnitInfoLen = sizeof(UnitInformationData), update = 5000;
IDENTIFYDATA            Id;
UnitInformationData     UnitInfo;
APIRET                  rc;
HFILE                   hDevice;
HAB                     hab;
HMQ                     hmq;         /* Message queue handle */
HWND                    hpopup;   /* popup menu handle */
HWND                    mainwin, hwndPopUp;
HWND hwndFrame1;
ERRORID                 errorid;
int                     stop = 0, NumUnits = 0, first = 0, resetini = 0, NumDisks = 0, logdiscovery = 0;
int                     options = OPT_TITLEBAR | OPT_SMART;   /* show titlebar, check SMART */
int                     SMARTfail = 0, oldSMARTfail = 0;
int                     CharWidth = 6;
int                     CharHeight = 15;
HACCEL                  haccel;
RGB2                    rgb2;
int                     SmartCheckCounter = 0;
int                     FloatWindowRunning = 0, idlecount = 0;
HMQ hmqf;

/* number of characters in parts of display */
#define SUBTITLELEN 11
#define COLUMNLEN   7
#define COLUMNWIDTH 8

int main (int argc, char *argv[])
   {
   ULONG ActionTaken, result;
   UCHAR Options = 0;
   int i, k;
   USHORT row, column;
   BYTE attr[2] = {0,7};

   hab = WinInitialize (0);                          /* PM initialisation */

   hmq = WinCreateMsgQueue (hab, 0);                 /* get a message queue */
   errorid = WinGetLastError(hab);                   /* check for errors */

   if (argc > 1)                                     /* if any argument passed to us */
      {
      for (i = 1; i < argc; i++)                     /* iterate through args */
         {
         if (strcmpi(argv[i],"-defaults") == 0)      /* if it's -defaults */
            {
            resetini = 1;                             /* Ignore the current saved data */
            }
         else
            {
            if (strcmpi(argv[i],"-logdiscovery") == 0)      /* if it's -logdiscovery */
               {
               logdiscovery = 1;                             /* be verbose in GetValidUnits() */
               }
            else
               {
               char tmp[100];                      /* someone made a typo */
               sprintf(tmp, "Unknown option \"%s\"\nI understand \"-defaults\" and \"-logdiscovery\","
                            "not much else.", argv[i]);
               WinMessageBox(HWND_DESKTOP,
                             HWND_DESKTOP,
                             tmp,
                             "Error!!",
                             1,
                             MB_OK | MB_ERROR | MB_MOVEABLE);
               return 1;
               }
            }
         }
      }

   rc = DosOpen ("\\DEV\\IBMS506$",
                 &hDevice,
                 &ActionTaken,
                 0,
                 FILE_SYSTEM,
                 OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_SHARE_DENYNONE | OPEN_FLAGS_NOINHERIT | OPEN_ACCESS_READONLY,
                 NULL);
   if (rc)
      {
      WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
                    "Could not open \\DEV\\IBMS506$. Are you sure you have DANIS506.ADD loaded?",
                    "DANIS506 check?", 0,
                    MB_ERROR | MB_OK | MB_MOVEABLE);
      return rc;
      }

   GetValidUnits();                     // test all units potentially present to map them

   result = WinDlgBox(HWND_DESKTOP,     // put up the dialog window
                      HWND_DESKTOP,
                      ClientWndProc,
                      0,
                      ID_PMDSKMON,
                      NULL);

   WinDestroyMsgQueue (hmq);            // ah well, it was nice knowing you
   WinTerminate (hab);                  // I'll be off now then

   DosClose (hDevice);

   return (rc);
   }

//***********************************************************
// GetValidUnits()
// Purpose: Check all 16 units potentially present to see
//          which ones are
// Returns: none
//***********************************************************
void GetValidUnits(void)
   {
   int i, j, k;
   BOOL LastBlank = FALSE;
   char temp[200];

   for (i = 0; i < 16; i++)
      {
      Parms.byPhysicalUnit = 0x80 + i;   // test unit 0x80+i
      if (logdiscovery)
         {
         sprintf(temp, "Testing unit %d%c (0x%02x)\n", i/2, i%2 ? 's' : 'm', Parms.byPhysicalUnit);
         Logit(temp);
         }
      memset (&Id, 0, 512);              // start with blank canvas
      rc = DosDevIOCtl(hDevice,
                       DSKSP_CAT_GENERIC,
                       DSKSP_GET_INQUIRY_DATA,
                       (PVOID)&Parms,
                       PLen,
                       &PLen,
                       (PVOID)&Id,
                       IDLen,
                       &IDLen);
      if (logdiscovery)
         {
         sprintf(temp, "Unit %d%c DosDevIOCtl rc = 0x%04x\n", i/2, i%2 ? 's' : 'm', rc);
         Logit(temp);
         }
      if (rc == 0xFF03)                  // something nasty happened, bail out
         break;

      if (rc == 0xFF02)                  // unit there?
         {
         sd[i].Options &= ~UNIT_PRESENT; // no
         if (logdiscovery)
            {
            sprintf(temp, "Unit %d%c is not present\n", i/2, i%2 ? 's' : 'm');
            Logit(temp);
            }
         continue;
         }
      else
         {
         sd[i].Options |= UNIT_PRESENT;  // yes
         NumUnits++;                     // count it
         if (logdiscovery)
            {
            sprintf(temp, "Unit %d%c is present\n", i/2, i%2 ? 's' : 'm');
            Logit(temp);
            }
         }

      k = 0;
      for (j = 0; j < 40; j++)           // copy unit model number
         {
         if (LastBlank == TRUE)
            if (Id.ModelNum[j ^ 1] == ' ')
               continue;
         sd[i].Model[k++] = Id.ModelNum[j ^ 1]; // in reverse byte order
         if (Id.ModelNum[j ^ 1] == ' ')
            LastBlank = TRUE;
         else
            LastBlank = FALSE;
         }
      if (k > 0)
         if (sd[i].Model[k-1] == ' ')
            k--;
      sd[i].Model[k] = '\0';            // cap end of string
      if (logdiscovery)
         {
         sprintf(temp, "Unit %d%c is [%s]\n", i/2, i%2 ? 's' : 'm', sd[i].Model);
         Logit(temp);
         }

      rc = DosDevIOCtl(hDevice,          // this could probably be removed...
                       DSKSP_CAT_GENERIC,
                       DSKSP_GET_UNIT_INFORMATION,
                       (PVOID)&Parms,
                       PLen,
                       &PLen,
                       (PVOID)&UnitInfo,
                       UnitInfoLen,
                       &UnitInfoLen);
      if (logdiscovery)
         {
         sprintf(temp, "Unit %d%c DosDevIOCtl rc2 = 0x%04x\n", i/2, i%2 ? 's' : 'm', rc);
         Logit(temp);
         }

      sd[i].Flags                = UnitInfo.wFlags;  // as the PM version of this program
      sd[i].Port                 = UnitInfo.wTaskFileBase; // doesn't display any of this
      sd[i].AltPort              = UnitInfo.wAlternateStatusAddress;
      sd[i].IRQ                  = UnitInfo.wIRQ;
      sd[i].TotalReadOperations  = 0;
      sd[i].TotalWriteOperations = 0;
      sd[i].TotalErrors          = 0;
      sd[i].TotalSectorsRead     = 0;
      sd[i].TotalSectorsWritten  = 0;
      }
   j = 0;
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI)) // if the unit is present and not ATAPI
         strcpy(Model[j++], sd[i].Model);
      }
   NumDisks = j;
   if (logdiscovery)
      {
      sprintf(temp, "Found %d valid unit%c\n", NumDisks, NumDisks == 1 ? ' ' : 's');
      Logit(temp);
      }
   }

//***********************************************************
// GetUnitStats()
// Purpose: Grab device counters from each device present
//          and work out difference from last time
// Returns: none but populates data structures
//***********************************************************
void GetUnitStats(void)
   {
   int validunitnum = 0, i, DoSmartCheck = 0;

   if (SmartCheckCounter == 0)
      DoSmartCheck = 1;
   else
      {
      if (SmartCheckCounter >= (3600/(update/1000))) // number of seconds in 1 hour divided by update interval
         SmartCheckCounter = -1;
      }
   SmartCheckCounter++;

   for (i = 0; i < 16; i++)    // go through whole lot
      {
      Parms.byPhysicalUnit = 0x80 + i;
      if ((options & OPT_SMART) && (DoSmartCheck != 0))    // if SMART checking enabled
         {
         if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI)) // if the unit is present and not ATAPI
            {
            ULONG value;
            ULONG DataLen = sizeof (value);

            rc = DosDevIOCtl(hDevice,
                             DSKSP_CAT_SMART,
                             DSKSP_SMART_GETSTATUS,
                             (PVOID)&Parms,
                             PLen,
                             &PLen,
                             (PVOID)&value,
                             DataLen,
                             &DataLen);
            if (rc == 0xFF03)                           // oh, not allowed
               sd[i].Options &= ~UNIT_HASSMART;         // doesn't support SMART
            else
               {
               sd[i].Options |= UNIT_HASSMART;          // does support SMART
               if (value)
                  {
                  sd[i].Options |= UNIT_FAILEDSMART;    // but has failed it

                  if ((options & OPT_SMARTLOG))
                     {
                     char temp[200];

                     sprintf(temp, "Unit %d%c [%s] failed SMART check\n", i/2, i%2 ? 's' : 'm', sd[i].Model);
                     Logit(temp);
                     }
                  }
               else
                  {
                  sd[i].Options &= ~UNIT_FAILEDSMART;   // or passed it

                  if ((options & OPT_SMARTLOG))
                     {
                     char temp[200];

                     sprintf(temp, "Unit %d%c [%s] passed SMART check\n", i/2, i%2 ? 's' : 'm', sd[i].Model);
                     Logit(temp);
                     }
                  }
               }
            }  // end if unit present & not ATAPI
         } // end if SMART checking enabled

      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI)) // if unit present
         {
         ULONG CountersLen = sizeof (DeviceCountersData);
         DeviceCountersData Counters;

         rc = DosDevIOCtl(hDevice,                            // grab counters for this device
                          DSKSP_CAT_GENERIC,
                          DSKSP_GEN_GET_COUNTERS,
                          (PVOID)&Parms,
                          PLen,
                          &PLen,
                          (PVOID)&Counters,
                          CountersLen,
                          &CountersLen);

         sd[i].CurrentReadOps  = (Counters.TotalReadOperations -   // subtract old from new values
                                  sd[i].TotalReadOperations) * 1000 /
                                  update;
         sd[i].CurrentWriteOps = (Counters.TotalWriteOperations -
                                  sd[i].TotalWriteOperations) * 1000  /
                                  update;
         sd[i].CurrentErrors   = ((Counters.TotalReadErrors  +
                                   Counters.TotalWriteErrors +
                                   Counters.TotalSeekErrors) -
                                  sd[i].TotalErrors) * 1000  /
                                  update;
         sd[i].CurrentKBsRead  = ((Counters.TotalSectorsRead    -
                                   sd[i].TotalSectorsRead) / 2)  * 1000 /
                                   update;
         sd[i].CurrentKBsWrite = ((Counters.TotalSectorsWritten    -
                                   sd[i].TotalSectorsWritten) / 2) * 1000 /
                                   update;

         sd[i].TotalReadOperations  = Counters.TotalReadOperations;  // make new values the old ones
         sd[i].TotalWriteOperations = Counters.TotalWriteOperations;
         sd[i].TotalErrors          = (Counters.TotalReadErrors  +
                                       Counters.TotalWriteErrors +
                                       Counters.TotalSeekErrors);
         sd[i].TotalSectorsRead     = Counters.TotalSectorsRead;
         sd[i].TotalSectorsWritten  = Counters.TotalSectorsWritten;
         }
      } // end for
   }    // end GetUnitStats()

void Logit(char *text)
   {
   FILE *fp;
   time_t ltime;
   struct tm *newtime;
   char dest[20];

   time(&ltime);
   newtime = localtime(&ltime);
   strftime(dest, sizeof(dest), "%Y-%m-%d %H:%M:%S", newtime);
   fp = fopen("dskmon.log", "a");
   fprintf(fp, "%s %s", dest, text);
   fclose(fp);
   }

//***********************************************************
// UpdateDisplay()
// Purpose: Formats current counters and outputs them to
//          user
// Returns: none
//***********************************************************
void UpdateDisplay(HWND hwnd)
   {
   int validunitnum = 0, i;
   char   msg[255];
   char figure1[20], figure2[20];

   memset(msg, ' ', 254); // start with a blank canvas
   validunitnum = 0;
   for (i = 0; i < 16; i++)  // Print device names for each one present (e.g. 0m, 0s)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 6d%s ", i/2, i%2 ? "s":"m");
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_DEV_NAME, msg); // ask the control to update itself

   validunitnum = 0;
// Reads
   for (i = 0; i < 16; i++) // same again for read stats
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf(figure1, "% 7u", sd[i].CurrentReadOps);
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_NO_READS, msg);   // update it

   validunitnum = 0;
// Writes
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf(figure1, "% 7u", sd[i].CurrentWriteOps);
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_NO_WRITES, msg);

   validunitnum = 0;
// KB/s read
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf(figure1, "% 7u", sd[i].CurrentKBsRead);
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_KB_READ, msg);

   validunitnum = 0;
// KB/s write
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf(figure1, "% 7u", sd[i].CurrentKBsWrite);
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_KB_WRITE, msg);

   validunitnum = 0;
// Errors
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         sprintf(figure1, "% 7u", sd[i].CurrentErrors);
         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }
   WinSetDlgItemText(hwnd, IDD_NO_ERRORS, msg);

   validunitnum = 0;
// SMART
   if (first == 1)
      {
      first ++;         // we do this because PM helpfully restores colours including
                        // the SMART status line. This gets it set back to blue.
      oldSMARTfail = 1;
      }
   else
      oldSMARTfail = SMARTfail;
   for (i = 0; i < 16; i++)
      {
      if ((sd[i].Options & UNIT_PRESENT) && !(sd[i].Flags & UIF_ATAPI))
         {
         if ((options & OPT_SMART))
            {
            if ((sd[i].Options & UNIT_HASSMART))
               {
               if ((sd[i].Options & UNIT_FAILEDSMART))
                  {
                  WinAlarm(HWND_DESKTOP, WA_ERROR);
                  SMARTfail = 1;
                  }
               else
                  {
                  strcpy(figure1, "OK");
                  }
               }
            else
               {
               strcpy(figure1, "n/a");
               }
            }
         else
            strcpy(figure1, "Off");

         sprintf((char*)&msg+((validunitnum)*COLUMNWIDTH), "% 7s ", figure1);
         validunitnum++;
         }
      }

   if ((options & OPT_SMART)) // if SMART checking enabled
      {
      if (SMARTfail)          // if something failed
         {
         if (SMARTfail != oldSMARTfail) // if it changed state since last time
            {
            rgb2.bRed = 255;            // set foreground red
            rgb2.bGreen = 0;
            rgb2.bBlue = 0;
            WinSetPresParam(WinWindowFromID(hwnd, IDD_SMART),
                            PP_FOREGROUNDCOLOR,
                            (ULONG)sizeof(RGB2),
                            (PVOID)&rgb2);
            }
         }
      else
         {
         if (SMARTfail != oldSMARTfail) // SMART OK but is it the same as last time
            {
            rgb2.bRed = 0;
            rgb2.bGreen = 0;
            rgb2.bBlue = 255;           // flip it back to blue if it changed from fail to OK
            WinSetPresParam(WinWindowFromID(hwnd, IDD_SMART),
                            PP_FOREGROUNDCOLOR,
                            (ULONG)sizeof(RGB2),
                            (PVOID)&rgb2);
            }
         }
      }
   WinSetDlgItemText(hwnd, IDD_SMART, msg);
   }

//***********************************************************
// ClientWndProc()
// Purpose: Handles all messages from the client window
//
// Returns: none
//***********************************************************
MRESULT EXPENTRY ClientWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
   {
   switch (msg)
      {
      case WM_INITDLG:              // Kick off at startup
         mainwin = hwnd;
         InitWindow(hwnd);
         return (MRESULT)0;

      case WM_COMMAND:               // menu item selected
         switch (SHORT1FROMMP(mp1))
            {
            case IDM_ONTOP:          // if "on top" changed
               options ^= OPT_ONTOP; // flip it on/off
               WinCheckMenuItem(hpopup, IDM_ONTOP, (options & OPT_ONTOP));
               return (MRESULT)0;

            case IDM_TITLEBAR:
               options ^= OPT_TITLEBAR;
               ResizeWindow(hwnd);
               ShowTitleBar(hwnd, (options & OPT_TITLEBAR), TRUE);
               return (MRESULT)0;

            case IDM_SETUP:
               Setup(hwnd);
               return (MRESULT)0;

            case IDM_ABOUT:
               WinDlgBox(HWND_DESKTOP, hwnd, AboutProc, 0, ID_ABOUT, NULL);
               return (MRESULT)0;

            case IDM_EXIT:
               WinSendMsg(hwnd, WM_CLOSE, MPVOID, MPVOID);
               return (MRESULT)0;
            }
         break;

      case WM_TIMER:
         GetUnitStats();
         if (first == 0)
            first++;
         else
            UpdateDisplay(hwnd);
         if (options & OPT_ONTOP)             /* is there a better way to do this? */
            WinSetWindowPos(hwnd, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER);
         break;

      case WM_BUTTON1DOWN:
         WinSendMsg(hwnd, WM_TRACKFRAME, MPFROMSHORT(TF_MOVE), MPVOID);
         break;

      case WM_BUTTON1UP:
         WinSendMsg(hwnd, WM_TRACKFRAME, MPVOID, MPVOID);
         break;

      case WM_BUTTON1DBLCLK:
         WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(IDM_TITLEBAR), MPVOID);
         WinSendMsg(hwnd, WM_TRACKFRAME, MPVOID, MPVOID);
         break;

      case WM_BUTTON2DOWN:
         WinPopupMenu(hwnd, hwnd, hpopup, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1), 0,
                     PU_HCONSTRAIN |
                     PU_VCONSTRAIN |
                     PU_MOUSEBUTTON1 |
                     PU_MOUSEBUTTON2 |
                     PU_KEYBOARD);
         return (MRESULT)0;
         break;

      case WM_PRESPARAMCHANGED:
         {
         HPS hps = WinGetPS(hwnd);
         FONTMETRICS fm;
         APIRET rc;

         rc = GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), &fm);
         if (rc)
            {
            if ((fm.fsType & FM_TYPE_FIXED))
               {
               CharWidth = fm.lAveCharWidth;
               CharHeight = fm.lMaxBaselineExt;
               WinReleasePS(hps);
               }
            else
               {
               WinReleasePS(hps);
               return (MRESULT)1;
               }
            }
         else
            {
            errorid = WinGetLastError(hab);
            WinReleasePS(hps);
            return (MRESULT)1;
            }
         ResizeWindow(hwnd);
         ShowTitleBar(hwnd, (options & OPT_TITLEBAR), TRUE);
         }
         break;

      case WM_CONTROLPOINTER:
         {
         POINTL ptl;
         SWP    swp;

         WinQueryPointerPos(HWND_DESKTOP, &ptl);          /* and mouse position */
         WinQueryWindowPos(mainwin, &swp);
         ptl.x -= swp.x;
         ptl.y -= swp.y;

         if ((ptl.x > (SUBTITLELEN * CharWidth)+20) && // if mouse to right of fixed text labels
             (ptl.x < swp.cx) &&
             (ptl.y > 0)      &&
             (ptl.y < swp.cy) )
            {
            if (FloatWindowRunning == FALSE)
               {
               FloatWindowRunning = TRUE;
               _beginthread(FloatWindow, NULL, 65536, NULL);
               }
            }
         }
         break;

      case WM_CLOSE:
         WinDestroyWindow(hpopup);
         WinDestroyAccelTable(haccel);
         WinStoreWindowPos(AppName, "WindowPos", hwnd);
         PrfWriteProfileData(HINI_USERPROFILE, AppName, "Options", &options, sizeof(options));
         PrfWriteProfileData(HINI_USERPROFILE, AppName, "Refresh", &update, sizeof(update));
         break;
      } // switch

   return WinDefDlgProc(hwnd,msg,mp1,mp2);
   }

//***********************************************************
// InitWindow()
// Purpose: Restores old window position/colours etc
//          Resizes window to current requirements
//          Kicks off timer to update us later
// Returns: none
//***********************************************************
void InitWindow(HWND hwnd)
  {
  ULONG x, y;

  if (resetini == 0)
     {
     if (WinRestoreWindowPos(AppName, "WindowPos", hwnd))
        {
        y = sizeof(x);
        if (PrfQueryProfileData(HINI_USERPROFILE, AppName, "Options", &x, &y))
           options = x;
        y = sizeof(x);
        if (PrfQueryProfileData(HINI_USERPROFILE, AppName, "Refresh", &x, &y))
           update = x;
        }
     }
  resetini = 1;
  haccel = WinLoadAccelTable(hab, 0, ID_PMDSKMON);
  WinSetAccelTable(hab, haccel, hwnd);
  ResizeWindow(hwnd);
  UpdateDisplay(hwnd);
  WinStartTimer(hab, hwnd, ID_PMDSKMON, update);
  ShowTitleBar(hwnd, (options & OPT_TITLEBAR), TRUE);
  hpopup = WinLoadMenu(HWND_OBJECT, NULLHANDLE, ID_PMDSKMENU);
  WinCheckMenuItem(hpopup, IDM_ONTOP, (options & OPT_ONTOP));
  }

//***********************************************************
// ResizeWindow()
// Purpose: Works out window positions required and moves
//          everything around
// Returns: none
//***********************************************************
void ResizeWindow(HWND hwnd)
  {
  SWP   swp;
  int ColumnWidth = CharWidth * 7, winwidth, winheight, extra;

  WinQueryWindowPos(hwnd, (PSWP)&swp); // get current pos/size
  winwidth = (SUBTITLELEN * CharWidth) + (NumUnits * (CharWidth * COLUMNLEN)); // window width is easy
  extra = WinQuerySysValue(HWND_DESKTOP, SV_CYDLGFRAME); // size of dialog border
  if ((options & OPT_TITLEBAR))         // set window height catering for titlebar or not
     {
     winheight = (8 * CharHeight) + 8 + WinQuerySysValue(HWND_DESKTOP, SV_CYTITLEBAR) + (extra *2);
     }
  else
     winheight = (8 * CharHeight) + 8 + (extra * 2);

  WinSetWindowPos(hwnd,        // resize main window
                  HWND_TOP,
                  swp.x,
                  swp.y,
                  winwidth,
                  winheight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW | SWP_ACTIVATE);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_DISK_GROUP), // resize groupbox inside window
                  HWND_TOP,
                  4,
                  2+extra,
                  winwidth-8,
                  (8 * CharHeight),
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE1), // resize text windows
                  HWND_TOP,
                  15,
                  (6 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_DEV_NAME), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (6 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE2), // resize text windows
                  HWND_TOP,
                  15,
                  (5 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_NO_READS), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (5 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE3), // resize text windows
                  HWND_TOP,
                  15,
                  (4 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_NO_WRITES), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (4 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE4), // resize text windows
                  HWND_TOP,
                  15,
                  (3 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_KB_READ), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (3 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE5), // resize text windows
                  HWND_TOP,
                  15,
                  (2 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_KB_WRITE), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (2 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE6), // resize text windows
                  HWND_TOP,
                  15,
                  (1 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_NO_ERRORS), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (1 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);

  WinSetWindowPos(WinWindowFromID(hwnd, IDD_TITLE7), // resize text windows
                  HWND_TOP,
                  15,
                  (0 * CharHeight)+8,
                  (SUBTITLELEN * CharWidth),
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  WinSetWindowPos(WinWindowFromID(hwnd, IDD_SMART), // resize text windows
                  HWND_TOP,
                  (SUBTITLELEN * CharWidth)+20,
                  (0 * CharHeight)+8,
                  (NumUnits * (COLUMNLEN * CharWidth))-30,
                  CharHeight,
                  SWP_SIZE | SWP_MOVE | SWP_SHOW);
  }

//***********************************************************
// Setup()
// Purpose: Displays setup dialog
// Returns: none
//***********************************************************
void Setup(HWND hwnd)
  {
  int sav_update, sav_options;

  sav_update = update;
  sav_options = options;
  if (WinDlgBox(HWND_DESKTOP, hwnd, SetupDlgProc, 0, ID_SETUP,NULL) == DID_OK)
     WinStartTimer(hab, hwnd, ID_PMDSKMON, update);
  else
     {
     update = sav_update;
     options = sav_options;
     }
  }

//***********************************************************
// ShowTitleBar()
// Purpose: Hides or displays the titlebar and related bits
//          Is a horrible hack :-(
// Returns: none
//***********************************************************
void ShowTitleBar(HWND hwnd, BOOL show, BOOL resize)
  {
  HWND t, v;
  SWP size;
  int winwidth, winheight, extra;

  WinCheckMenuItem(hpopup, IDM_TITLEBAR, show);
  winwidth = (SUBTITLELEN * CharWidth) + (NumUnits * (CharWidth * COLUMNLEN));
  extra = WinQuerySysValue(HWND_DESKTOP, SV_CYDLGFRAME);
  if (show > 0)
     {
     winheight = (8 * CharHeight) + 8 +
                 WinQuerySysValue(HWND_DESKTOP, SV_CYTITLEBAR) +
                 (extra*2);

     WinShowWindow(WinWindowFromID(hwnd, FID_TITLEBAR), TRUE); // hello/goodbye titlebar
     WinShowWindow(WinWindowFromID(hwnd, FID_SYSMENU), TRUE);
     WinShowWindow(WinWindowFromID(hwnd, FID_MINMAX), TRUE);
     }
  else
     {
     winheight = (8 * CharHeight) + 8 + (extra*2);

     WinShowWindow(WinWindowFromID(hwnd, FID_TITLEBAR), FALSE); // hello/goodbye titlebar
     WinShowWindow(WinWindowFromID(hwnd, FID_SYSMENU), FALSE);
     WinShowWindow(WinWindowFromID(hwnd, FID_MINMAX), FALSE);
     }

  if (resize)
     {
     WinSetWindowPos(hwnd, HWND_TOP, 0, 0, winwidth, winheight, SWP_SIZE | SWP_ACTIVATE | SWP_SHOW);
     }
  }

//***********************************************************
// SetupDlgProc()
// Purpose: Processes messages from the setup dialog
// Returns: none
//***********************************************************
MRESULT EXPENTRY SetupDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
  {
  char buf[64];

  switch (msg)
     {
     case WM_INITDLG:
        {
        WinSendDlgItemMsg(hwnd,
                          IDD_REFRESH,
                          SPBM_SETLIMITS,
                          (MPARAM)300,
                          (MPARAM)1);  // spin button limits 1 - 300 seconds
        WinSendDlgItemMsg(hwnd,
                          IDD_REFRESH,
                          SPBM_SETCURRENTVALUE,
                          (MPARAM)(update/1000),
                          (MPARAM)0);  // current setting

        if ((options & OPT_SMART))
           {
           WinCheckButton(hwnd, IDD_SMART_ON, TRUE);
           WinEnableWindow(WinWindowFromID(hwnd, IDD_SMART_LOG), TRUE);
           if ((options & OPT_SMARTLOG))
              WinCheckButton(hwnd, IDD_SMART_LOG, TRUE);
           else
              WinCheckButton(hwnd, IDD_SMART_LOG, FALSE);
           }
        else
           {
           WinCheckButton(hwnd, IDD_SMART_OFF, TRUE);
           WinEnableWindow(WinWindowFromID(hwnd, IDD_SMART_LOG), FALSE);
           }

        return (MRESULT)0;
        }

     case WM_CONTROL:
        switch (SHORT1FROMMP(mp1))
           {
           case IDD_SMART_OFF:
              WinEnableWindow(WinWindowFromID(hwnd, IDD_SMART_LOG), FALSE);
              break;
           case IDD_SMART_ON:
              WinEnableWindow(WinWindowFromID(hwnd, IDD_SMART_LOG), TRUE);
              break;
           }
           break;

     case WM_COMMAND:
        switch (COMMANDMSG(&msg)->cmd)
           {
           case DID_OK:           /* if OK button clicked */
              WinSendDlgItemMsg(hwnd,
                                IDD_REFRESH,
                                SPBM_QUERYVALUE,
                                (MPARAM)&update,
                                (MPARAM)0);
              update *= 1000;
              if (WinQueryButtonCheckstate(hwnd, IDD_SMART_ON))
                 options |= OPT_SMART;
              else
                 options &= (0xffffffff - OPT_SMART);

              if (WinQueryButtonCheckstate(hwnd, IDD_SMART_LOG))
                 options |= OPT_SMARTLOG;
              else
                 options &= (0xffffffff - OPT_SMARTLOG);
              break;

           case DID_CANCEL:
              break;
           }

     case WM_DESTROY:
        {
        break;
        }
     }

  return WinDefDlgProc(hwnd, msg, mp1, mp2);
  }

//***********************************************************
// AboutProc()
// Purpose: Processes messages from the About... dialog
// Returns: none
//***********************************************************
MRESULT EXPENTRY AboutProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
  {
  char buf[128];

  switch (msg)
     {
     case WM_INITDLG:
        {
        sprintf(buf, "Version %s %s %s", VERSION, COMPILEDATE, COMPILETIME);
        WinSetDlgItemText(hwnd, IDD_VERSION, buf);
        }
     }

  return WinDefDlgProc(hwnd, msg, mp1, mp2);
  }

//***********************************************************
// FloatWindow()
// Purpose: Thread to display floating disk name window
// Returns: none
//***********************************************************
void _Optlink FloatWindow(void* arg)
   {
   HAB hab;
   ULONG flCreateFlags = FCF_BORDER | FCF_NOBYTEALIGN;
   QMSG        qmsg;                                 /* Message data structure */
   APIRET rc;

   hab = WinInitialize (0);                          /* PM initialisation */
   hmqf = WinCreateMsgQueue (hab, 0);                 /* get a message queue */

   idlecount = 0;
   if (!WinRegisterClass(hab,
                        "FloatingHelp",
                        fnFloatHelp,      /*  message handler */
                        NULLHANDLE, /*  CS_SIZEREDRAW */
                        sizeof(ULONG)))
      Logit("WinRegisterClass failed\n");
   hwndFrame1 = WinCreateStdWindow(HWND_DESKTOP,
                                  WS_DISABLED | WS_VISIBLE,      /* Frame window style */
                                  &flCreateFlags,   /* window style */
                                  "FloatingHelp",   /* Class name */
                                  "Floating Help",  /* Window title */
                                  0,                /* Default client style */
                                  NULLHANDLE,       /* Resources in EXE file */
                                  IDD_FHELP,        /* Resource/Window ID */
                                  &hwndPopUp);      /* Window handle returned */
   if (!hwndFrame1)
      Logit("WinCreateStdWindow failed\n");

   WinEnableWindow(hwndFrame1, TRUE);
   WinEnableWindowUpdate (hwndFrame1, TRUE);
   WinShowWindow(hwndFrame1, TRUE);
   WinSetPresParam(hwndPopUp, PP_FONTNAMESIZE, 7, "8.Helv");

   while (WinGetMsg (hab, &qmsg, 0, 0, 0))        /* Start message loop */
      WinDispatchMsg (hab, &qmsg);

   if (WinIsWindow (hab, hwndFrame1))
      WinDestroyWindow (hwndFrame1);

   WinDestroyMsgQueue (hmq);            // ah well, it was nice knowing you
   WinTerminate (hab);                  // I'll be off now then
   FloatWindowRunning = FALSE;
   }

//***********************************************************
// fnFloatHelp()
// Purpose: Processes messages from the floating window
// Returns: none
//***********************************************************
MRESULT APIENTRY fnFloatHelp (HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
   {
   POINTL ptl;
   POINTL aptl[41];
   SWP swp;
   HPS hps;
   ULONG sdLen;

   switch (msg)
      {
      case WM_CREATE:
         WinQueryPointerPos(HWND_DESKTOP, &ptl);          /* and mouse position */
         WinQueryWindowPos(mainwin, &swp);
         ptl.x -= swp.x;
         ptl.y -= swp.y;
         if (ptl.x > (SUBTITLELEN * CharWidth)+20)
            {
            int curpos = ptl.x - ((SUBTITLELEN * CharWidth)+20);

            curpos /= CharWidth;
            curpos /= COLUMNWIDTH;
            if (curpos > NumDisks-1)
               return (MRESULT)0;
            swp.x += ptl.x;
            swp.y += ptl.y;
            swp.cy = CharHeight+2;
            sdLen = strlen(Model[curpos]);
            hps = WinGetPS(hwnd);
            GpiQueryCharStringPos(hps,
                                 0L,
                                 sdLen,
                                 Model[curpos],
                                 NULL,
                                 aptl);

            WinReleasePS(hps);
            swp.cx = aptl[sdLen].x+5;

            WinSetWindowPos(hwndFrame1,
                           HWND_TOP,
                           swp.x,
                           swp.y,
                           swp.cx,
                           swp.cy,
                           SWP_SIZE | SWP_MOVE | SWP_SHOW);
            WinStartTimer(hab, hwnd, IDD_FHELP, 1000);
            return (MRESULT)0;
            }
         break;

      case WM_TIMER:
         WinQueryPointerPos(HWND_DESKTOP, &ptl);          /* and mouse position */
         WinQueryWindowPos(mainwin, &swp);
         ptl.x -= swp.x;
         ptl.y -= swp.y;
         if ((ptl.x > (SUBTITLELEN * CharWidth)+20) &&
             (ptl.x < swp.cx) &&
             (ptl.y > 0) &&
             (ptl.y < swp.cy) )
            {
            int curpos = ptl.x - ((SUBTITLELEN * CharWidth)+20);

            curpos /= CharWidth;
            curpos /= COLUMNWIDTH;
            if (curpos > NumDisks-1)
               return (MRESULT)0;
            swp.x += ptl.x;
            swp.y += ptl.y;
            swp.cy = CharHeight+2;
            sdLen = strlen(Model[curpos]);
            hps = WinGetPS(hwnd);
            GpiQueryCharStringPos(hps,
                                 0L,
                                 sdLen,
                                 Model[curpos],
                                 NULL,
                                 aptl);

            WinReleasePS(hps);
            swp.cx = aptl[sdLen].x+5;

            WinSetWindowPos(hwndFrame1,
                           HWND_TOP,
                           swp.x,
                           swp.y,
                           swp.cx,
                           swp.cy,
                           SWP_SIZE | SWP_MOVE | SWP_SHOW);
            WinStartTimer(hab, hwnd, IDD_FHELP, 1000);
            WinInvalidateRect(hwnd, NULL, TRUE);
            return (MRESULT)0;
            }
         else
            {
            idlecount++;
            if (idlecount > 5)
               {
               WinPostMsg(hwnd, WM_QUIT, (MPARAM)0, (MPARAM)0);
               FloatWindowRunning = FALSE;
               }
            }
         break;

      case WM_PAINT:
         {
         HPS hps;
         RECTL rectHelpWindow;
         ULONG length;
         int curpos;

         WinQueryPointerPos(HWND_DESKTOP, &ptl);          /* and mouse position */
         WinQueryWindowPos(mainwin, &swp);
         ptl.x -= swp.x;
         ptl.y -= swp.y;
         if (ptl.x > (SUBTITLELEN * CharWidth)+20)
            {
            curpos = ptl.x - ((SUBTITLELEN * CharWidth)+20);
            curpos /= CharWidth;
            curpos /= COLUMNWIDTH;
            }
         hps = WinBeginPaint (hwnd,
                              NULLHANDLE,          /* Get a cache presentation space */
                              &rectHelpWindow);  /* Receive update rectangle */
         WinQueryWindowRect (hwnd, &rectHelpWindow);  /* Get window rectangle */
         WinFillRect (hps,
                      &rectHelpWindow,
                      SYSCLR_WINDOW);     /* System window color */
         ptl.x = 2;
         ptl.y = 2;
         GpiMove(hps, &ptl);

         length = strlen(Model[curpos]);
         GpiCharString(hps, length, Model[curpos]);
         WinEndPaint (hps);
         return (MRESULT)0;
         }
         break;
      }
   _flushall();
   return WinDefWindowProc(hwnd, msg, mp1, mp2);
   }

