/* $Id: smartmon.c,v 1.29 2005/04/10 09:14:19 root Exp $ */

#include <ctype.h>
#include <io.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define BUILDMODE 2
#include <csp_defs.h>
#include <csp_lib.h>

#define INCL_BASE
#include <os2.h>

#include "contrib/idedata.h"
#include "contrib/attrs.h"
#include "contrib/hddspd.h"

#include "contrib/syslog.h"

#include "uniftime.h"

/* Product version macro */

#define PROD_VER              "1.55"
#define M_VERSION PROD_VER " (" PROD_STRING ")"

/* Implementation-dependent constants */

#define HDC_ADD           "IBMS506$"    /* Name of HDC adapter device driver */
#define HDD_PIPE    "\\PIPE\\IBMHDD"    /* For monitoring */
#define SMART_FILE       "smart.dat"    /* S.M.A.R.T. persistent status */
#define SMREC_SIZE               576    /* HDDSPEED-compatible structure */

#define INVALID_TEMP            -274    /* Admittedly invalid value */

#define swap(a, b)  b^=a^=b^=a          /* Handy macro */
#define APRINTF(str) if(ansi) printf(str)

/* Local variables */

static char hdc_add[]=HDC_ADD;
static int raw_mode=0;
static int daemon=0;
static int smart_delay=900;
static int ansi;
static char *tempmark="C";
static int fahrenheit=0;

#define FLG_HIT              0x8000     /* Flag used in S.M.A.R.T. report (thus
                                           needs a description) */
#define ATTRIB_OK(a) (a!=194&&a!=231)   /* Attribute eligible for T.E.C.
                                           prediction */

/* Daemon flags */

#define DAEMON_PIPE          0x0001     /* Report to PIPE */
#define DAEMON_SYSLOG        0x0002     /* Report to SYSLOG */
#define DAEMON_PIPE_DETAIL (0x0004|DAEMON_PIPE) /* Report to PIPE in detail */
#define DAEMON_ONCE          0x8000     /* Do it once and exit */

/* Flags map */

struct
{
 unsigned short flg;
 char *flag;
 char *desc;
}
flags_map[]=
{
 {SF_CRITICAL, "*", "life critical"},
 {SF_ONLINE, "OC", "online collection"},
 {SF_PERFORMANCE, "PR", "performance related"},
 {SF_ERROR_RATE, "ER", "error rate"},
 {SF_EVENT_COUNT, "EC", "event count"},
 {SF_SELF_PRESERVE, "SP", "self-preserving"},
 {0, NULL, NULL}
};

/* Daemon's scope-of-interest */

struct unit_status
{
 int temperature;
 int realloc_sec;
 char model[41];
 char serial[21];
};

/* Simple checksum verification routine */

BYTE get_checksum(BYTE *start, BYTE *end)
{
 BYTE rc=0;

 while(start!=end)
  rc-=*start++;
 return(rc);
}

/* Fixes string data returned by the unit (the endia issue and termination
   symbol) */

static char *fix_devstr(char *dest, char *str, int sn)
{
 int i;
 char t;

#define qstr(c) ((unsigned char)(c)>=' '&&(unsigned char)(c)<=0xFE)
 memset(dest, 0, sn+1);
 memcpy(dest, str, sn);
 ltrim(dest);
 for(i=0; i<sn; i+=2)
 {
  t=dest[i+1];
  dest[i+1]=dest[i];
  dest[i]=t;
  if(i>0&&!qstr(dest[i]))
  {
   dest[i]='\0';
   break;
  }
  else if(!qstr(dest[i+1]))
  {
   dest[i+1]='\0';
   break;
  }
 }
 ltrim(dest);
 rtrim(dest);
 return(dest);
#undef qstr
}

/* Retrieves the S.M.A.R.T. data table */

static int get_smart(HFILE hf, DSKSP_CommandParameters *pdparams, struct smart_rec *prc,
                     struct smart_exec *pexec)
{
 ULONG ip_len, opa_len, opt_len, opi_len;
 IDENTIFYDATA id;
 DATETIME dt;
 APIRET rc;

 /* Obtain the attribute/threshold pairs */
 ip_len=sizeof(*pdparams);
 opa_len=sizeof(prc->dev_attrs);
 opt_len=sizeof(pexec->dev_thresholds);
 opi_len=sizeof(id);
 if((rc=DosDevIOCtl(hf, DSKSP_CAT_GENERIC, DSKSP_GET_INQUIRY_DATA, (PVOID)pdparams, ip_len, &ip_len, &id, opi_len, &opi_len))||
    (rc=DosDevIOCtl(hf, DSKSP_CAT_SMART, DSKSP_SMART_GET_ATTRIBUTES, (PVOID)pdparams, ip_len, &ip_len, &prc->dev_attrs, opa_len, &opa_len))||
    (rc=DosDevIOCtl(hf, DSKSP_CAT_SMART, DSKSP_SMART_GET_THRESHOLDS, (PVOID)pdparams, ip_len, &ip_len, &pexec->dev_thresholds, opt_len, &opt_len)))
  return(1);
 memmove(prc->model, id.ModelNum, 40);
 memmove(prc->serial_num, id.SerialNum, 20);
 memmove(pexec->firmware, id.FirmwareRN, 8);
 /* Write the current date/time into the S.M.A.R.T. monitor structure */
 DosGetDateTime(&dt);
 prc->start_date=get_tstamp(dt.year, dt.month, dt.day, dt.hours, dt.minutes, dt.seconds);
 return(0);
}

/* Converts the temperature readings if necessary */

static int convert_temp(int val)
{
 if(val<0||val>127)
  return(INVALID_TEMP);
 if(!fahrenheit)
  return(val);
 /* Round up when converting to Fahrenheit */
 return(((val*18+9)/10)+32);
}

/* Attempts to guess the drive family by model. Returns a mask that all
   attributes can be &'ed with. */

static unsigned short family_by_model(char *model)
{
 if(!memicmp(model, "WD", 2))
  return(FAMILY_WD);
 if(!memicmp(model, "ST", 2))
  return(FAMILY_CONNERGATE);
 if(!memicmp(model, "Quantum", 2))
  return(FAMILY_QUANTUM);
 if(!memicmp(model, "MAXTOR", 6))
  return(FAMILY_QUAXTOR);
 if(!memicmp(model, "IBM-", 4))
  return(FAMILY_IBM);
 if(!memicmp(model, "IC", 2))
  return(FAMILY_HGST);
 if(!memicmp(model, "FUJITSU", 7))
  return(FAMILY_FUJITSU);
 if(!memicmp(model, "SAMSUNG", 7))
  return(FAMILY_SAMSUNG);
 return(FAMILY_ALL);
}

/* Processes the attributes/values */

static int show_smart(HFILE hf, DSKSP_CommandParameters *pdparams,
                      struct unit_status *status)
{
 int rc=1;
 int i, j, k, attr, val, t, v, mv, ov, dec;
 char *p;
 char attr_desc[50];
 unsigned long cur_date, start_date, tec_date;
 unsigned long ntec_date=0x7FFFFFFF;
 int ntec_attr=0;
 char ntec_name[ATTR_NAME_MAX+1];
 char attr_raw[13], temp_val[32], realloc_val[32], fmt_val[80];
 char i_model[41], i_sn[21], i_fw[9];
 char c_flags[32], st_bar[11], st_cols;
 int temperature=INVALID_TEMP, realloc_sec=-1;
 struct smart_rec rec, ref_rec;
 int ref_rec_valid=0;
 struct smart_exec ex;
 PTIB ptib;
 PIB *ppib;
 FILE *stream;
 char sm_name[CCHMAXPATH];
 struct ts_exp texp;
 unsigned short family;

 if(get_smart(hf, pdparams, &rec, &ex))
 {
  printf("Failed to query S.M.A.R.T. values\n");
  return(1);
 }
 /* Visual part */
 fix_devstr(i_fw, ex.firmware, 8);
 fix_devstr(i_model, rec.model, 40);
 fix_devstr(i_sn, rec.serial_num, 20);
 for(i=0; i<40; i+=2)
  swap(rec.model[i], rec.model[i+1]);
 for(i=0; i<20; i+=2)
  swap(rec.serial_num[i], rec.serial_num[i+1]);
 if(status!=NULL)
 {
  strcpy(status->model, i_model);
  strcpy(status->serial, i_sn);
 }
 family=family_by_model(rec.model);
 if(!daemon)
  printf("%s, #%s, firmware v %s, ATTv%u, THRv%u\n", i_model, i_sn, i_fw, rec.dev_attrs.wRevisionNumber, ex.dev_thresholds.wRevisionNumber);
 if(get_checksum((BYTE *)&rec.dev_attrs, (BYTE *)&rec.dev_attrs.byCheckSum)!=rec.dev_attrs.byCheckSum)
  printf("WARNING: attribute table is inconsistent!\n");
 if(get_checksum((BYTE *)&ex.dev_thresholds, (BYTE *)&ex.dev_thresholds.byCheckSum)!=ex.dev_thresholds.byCheckSum)
  printf("WARNING: S.M.A.R.T. threshold table is inconsistent\n");
 /* Get the stored S.M.A.R.T. block into ref_rec */
 DosGetInfoBlocks(&ptib, &ppib);
 DosQueryModuleName(ppib->pib_hmte, CCHMAXPATH, sm_name);
 strcpy(((p=strrchr(sm_name, '\\'))==NULL)?sm_name:(p+1), SMART_FILE);
 if((stream=fopen(sm_name, "rb"))==NULL)
 {
  if((stream=fopen(sm_name, "wb"))!=NULL)
  {
   fwrite(&rec, 1, SMREC_SIZE, stream);
   fclose(stream);
  }
 }
 else
 {
  while(fread(&ref_rec, 1, SMREC_SIZE, stream)==SMREC_SIZE)
  {
   /* Models and serial #'s must match */
   if(!memcmp(&ref_rec, &rec, 60))
   {
    ref_rec_valid=1;
    if(!daemon)
     printf("T.E.C. prediction monitoring started on %02u/%02u/%04u, %02u:%02u:%02u\n",
            ts_day(ref_rec.start_date), ts_month(ref_rec.start_date), ts_year(ref_rec.start_date),
            ts_hour(ref_rec.start_date), ts_min(ref_rec.start_date), ts_sec(ref_rec.start_date));
    break;
   }
  }
  fclose(stream);
  /* Append a new S.M.A.R.T. record if none found. */
  if(!ref_rec_valid&&(stream=fopen(sm_name, "ab"))!=NULL)
  {
   fwrite(&rec, 1, SMREC_SIZE, stream);
   fclose(stream);
  }
 }
 if(!ref_rec_valid)
  ref_rec=rec;
 start_date=dos2unif(ref_rec.start_date);
 cur_date=dos2unif(rec.start_date);
 rc=0;
 /* Find the temperature readings */
 for(i=0; i<30; i++)
 {
  if(rec.dev_attrs.Attribute[i].byAttribID!=0)
  {
   /* Perform a lookup in the secondary table */
   for(j=0; j<30; j++)
    if(ex.dev_thresholds.Threshold[i].byAttributeID!=0)
     break;
   if(j<30)
   {
    attr=ex.dev_thresholds.Threshold[i].byAttributeID;
    val=(int)*(unsigned short *)rec.dev_attrs.Attribute[i].byRaw;
    if(attr==231||attr==194)
     temperature=convert_temp(val);
    else if(attr==5)
     realloc_sec=val;
   }
  }
 }
 /* The daemon shall bail out here */
 if(daemon)
 {
  if(status!=NULL)
  {
   status->temperature=temperature;
   status->realloc_sec=realloc_sec;
  }
  return(rc);
 }
 printf("\n");
 if(raw_mode)
 {
  printf("Attribute               Value  Threshold  Worst  Raw           Flags\n"
         "\n");
 }
 else
 {
  printf("  Attribute                 ID Threshold Value Indicator  1/Month   T.E.C.\n"
         "\n");
 }
 for(i=0; i<30; i++)
 {
  if(rec.dev_attrs.Attribute[i].byAttribID!=0)
  {
   /* Perform a lookup in the secondary table */
   for(j=0; j<30; j++)
    if(ex.dev_thresholds.Threshold[i].byAttributeID!=0)
     break;
   if(j<30)
   {
    /* The report follows ... */
    for(k=0; std_attrs[k].num!=0&&
             (std_attrs[k].num!=rec.dev_attrs.Attribute[i].byAttribID||
              (std_attrs[k].family&family)==0);
        k++)
     ;
    if(std_attrs[k].num!=0)
     strcpy(attr_desc, std_attrs[k].name);
    else
     sprintf(attr_desc, "Attribute %u", (unsigned int)rec.dev_attrs.Attribute[i].byAttribID);
    t=ex.dev_thresholds.Threshold[i].byValue;
    v=rec.dev_attrs.Attribute[i].byValue;
    ov=ref_rec.dev_attrs.Attribute[i].byValue;
    if(!ATTRIB_OK(rec.dev_attrs.Attribute[i].byAttribID)||ov<=v||cur_date<=start_date)
     tec_date=0;
    else
     tec_date=cur_date+(v-t)*(cur_date-start_date)/(ov-v);
    if(tec_date-start_date>8766000)     /* 1000 years ahead are out of prediction <g> */
     tec_date=0;
    else
    {
     if(tec_date<ntec_date)
     {
      ntec_date=tec_date;
      ntec_attr=i;
      strcpy(ntec_name, attr_desc);
     }
    }
    if(!raw_mode)
    {
     /* NORMAL MODE */
     /* Calculate the graphical progress bar */
     st_bar[10]='\0';
     if(v<=t)
      st_cols=0;
     else
     {
      if(t<100)
       mv=100;
      else if(t<200)
       mv=200;
      else if(t<255)
       mv=255;
      else
       mv=256;
      st_cols=(v-t)*10/(mv-t);
     }
     if(st_cols>10)
      st_cols=10;
     memset(st_bar, 32, 10);
     memset(st_bar, ansi?22:'=', st_cols);
     if(v>=ov||cur_date<=start_date)
      dec=0;
     else
      dec=(ov-v)*7305/(cur_date-start_date);
     if(!ATTRIB_OK(rec.dev_attrs.Attribute[i].byAttribID))
      dec=0.0;
     printf("%c %-25s %-3u    %3u    %3u  ", (rec.dev_attrs.Attribute[i].wFlags&SF_CRITICAL)?'*':' ',
            attr_desc,
            (unsigned int)rec.dev_attrs.Attribute[i].byAttribID,
            (int)t, (int)v);
     APRINTF("\x1B[1;31m"); printf("%c", st_bar[0]); APRINTF("\x1B[33m"); printf("%c%c", st_bar[1], st_bar[2]); APRINTF("\x1B[32m"); printf("%s", st_bar+3); APRINTF("\x1B[0m"); printf(" %3u.%u    ", dec/10, dec%10);
     if(v>=ov||cur_date<=start_date||!ATTRIB_OK(rec.dev_attrs.Attribute[i].byAttribID)||tec_date==0)
      printf("Unknown");
     else if(v<t)
     {
      APRINTF("\x1B[1;31m"); printf("  Yes  "); APRINTF("\x1B[0m");
      ntec_date=0;
      ntec_attr=i;
      strcpy(ntec_name, attr_desc);
     }
     else
     {
      exp_unif(&texp, tec_date);
      APRINTF("\x1B[1m"); printf("%02u/%04u", texp.month, texp.year); APRINTF("\x1B[0m");
     }
     printf("\n");
    }
    else
    {
     /* RAW MODE */
     c_flags[0]='\0';
     memset(attr_raw, 0, 13);
     for(k=0; k<6; k++)
      sprintf(attr_raw+k*2, "%02x", (int)rec.dev_attrs.Attribute[i].byRaw[5-k]);
     for(k=0; flags_map[k].flg!=0; k++)
     {
      if((flags_map[k].flg&~FLG_HIT)!=SF_CRITICAL&&(rec.dev_attrs.Attribute[i].wFlags&(flags_map[k].flg&~FLG_HIT)))
      {
       flags_map[k].flg|=FLG_HIT;
       strcat(c_flags, " ");
       strcat(c_flags, flags_map[k].flag);
      }
     }
     printf("%-25s %3u        %3u    %3u  %s %s\n",
             attr_desc,
             (int)rec.dev_attrs.Attribute[i].byValue,
             (int)ex.dev_thresholds.Threshold[i].byValue,
             (int)rec.dev_attrs.Attribute[i].byWorst,
             attr_raw,
             c_flags
            );
    }
   }
  }
 }
 /* Totals */
 sprintf(temp_val, (temperature!=INVALID_TEMP)?"%u%s":"N/A", temperature, tempmark);
 sprintf(realloc_val, (realloc_sec)>=0?"%u":"N/A", realloc_sec);
 printf("%s\n", raw_mode?
       "":
       "");
 sprintf(fmt_val, "Reallocated Sectors: %%-%us                Current Drive Temperature: %%s\n",
         raw_mode?7:8);
 printf(fmt_val, realloc_val, temp_val);
 if(raw_mode)
 {
  j=0;
  printf("Flags map:\n");
  for(i=0; flags_map[i].flg!=0; i++)
  {
   v=flags_map[i].flg;
   if(v&FLG_HIT)
   {
    printf("  %s: %-20s", flags_map[i].flag, flags_map[i].desc);
    printf((++j%3)?"":"\n");
   }
  }
  if(j)
   printf("\n");
 }
 if(ref_rec_valid)
 {
  if(ntec_date==0x7FFFFFFF)
   printf("T.E.C. not detected.\n");
  else if(ntec_date==0)
  {
   APRINTF("\x1B[1m"); printf("! T.E.C. detected with %sattribute %u"); printf("\n", rec.dev_attrs.Attribute[ntec_attr].wFlags&SF_CRITICAL?"\x1B[31mlife-critical\x1B[37m ":"", rec.dev_attrs.Attribute[ntec_attr].byAttribID); APRINTF("\x1B[0m"); printf("\n");
  }
  else
  {
   exp_unif(&texp, ntec_date);
   printf("Nearest prognosed T.E.C.: "); APRINTF("\x1B[1m"); printf("%02u/%02u/%04u, %s (%s)",
           texp.day, texp.month, texp.year, ntec_name,
           rec.dev_attrs.Attribute[ntec_attr].wFlags&SF_CRITICAL?"life-critical":"not critical");
   APRINTF("\x1B[0m"); printf("\n");
  }
 }
 return(rc);
}

/* S.M.A.R.T. executive part */

static int smart_proc(int unit)
{
 HFILE hf, hp;
 ULONG action;
 int rc;
 DSKSP_CommandParameters dparams;
 UnitInformationData udd;
 DATETIME dt;
 ULONG ip_len, op_len, pipe_written;
 ULONG smart_status;
 struct unit_status last_status;
 int chk_atapi=1;
 char tmp[256];
#ifdef USE_SYSLOG
 char *pt;
#endif

 if(rc=DosOpen(hdc_add, &hf, &action, 0L, FILE_NORMAL,
       OPEN_ACTION_OPEN_IF_EXISTS, OPEN_SHARE_DENYNONE, NULL))
 {
  printf("DosOpen failed with rc=%u\n", rc);
  return(rc);
 }
 dparams.byPhysicalUnit=(BYTE)unit+0x80;
 ip_len=sizeof(dparams);
 op_len=sizeof(udd);
 rc=DosDevIOCtl(hf, DSKSP_CAT_GENERIC, DSKSP_GET_UNIT_INFORMATION, (PVOID)&dparams, ip_len, &ip_len, &udd, op_len, &op_len);
 if(rc==0xFF02)
 {
  dparams.byPhysicalUnit=(BYTE)unit;
  ip_len=sizeof(dparams);
  op_len=sizeof(udd);
  rc=DosDevIOCtl(hf, DSKSP_CAT_GENERIC, DSKSP_GET_UNIT_INFORMATION, (PVOID)&dparams, ip_len, &ip_len, &udd, op_len, &op_len);
  chk_atapi=0;
 }
 if(rc==0xFF02)
  printf("No device at unit %u\n", unit);
 else if(rc==0xFF03)
  printf("Invalid unit number: %u\n", unit);
 else if(rc)
  printf("Failed to query parameters of unit %u, rc=0x%04x\n", unit, rc);
 if(rc)
  return(rc);
 if((udd.wFlags&UIF_ATAPI)&&chk_atapi)
 {
  printf("Cannot retrieve S.M.A.R.T. data for ATAPI devices\n");
  return(1);
 }
 ip_len=sizeof(dparams);
 /* Retrieve S.M.A.R.T. capabilities and overall prognosis */
 op_len=sizeof(smart_status);
 rc=DosDevIOCtl(hf, DSKSP_CAT_SMART, DSKSP_SMART_GETSTATUS, (PVOID)&dparams,
                ip_len, &ip_len, &smart_status, op_len, &op_len);
 if(rc)
 {
  if(rc==0xFF03)
   printf("S.M.A.R.T. is disabled or not supported\n");
  else
   printf("Failed to retrieve S.M.A.R.T. data with rc=%04x\n", rc);
 }
 else
 {
  if(smart_status)
   printf("S.M.A.R.T. reports drive failure!\n");
  if(daemon)
  {
   /* Reopen the pipe each time so we can permit multiple startup/shutdown
      cycles at remote end */
   while(1)
   {
    if(daemon&DAEMON_PIPE)
    {
     hp=0;
     do
     {
      rc=DosOpen(HDD_PIPE, &hp, &action, 0L, 0, FILE_OPEN, OPEN_ACCESS_WRITEONLY|
                 OPEN_SHARE_DENYWRITE, 0L);
      if(rc==ERROR_PATH_NOT_FOUND)
       printf("Remote pipe does not exist, operation failed.\n");
      if(rc==ERROR_PIPE_BUSY)
       DosWaitNPipe(HDD_PIPE, 100L);
     } while(rc==ERROR_PIPE_BUSY);
    }
    if(rc=show_smart(hf, &dparams, &last_status))
     sprintf(tmp, "ERROR. last value: %d%s", last_status.temperature, tempmark);
    else
    {
     if((daemon&DAEMON_PIPE_DETAIL)==DAEMON_PIPE_DETAIL)
      if(last_status.temperature!=INVALID_TEMP)
       sprintf(tmp, "%d\t%s %s\t%d%s\t%u", unit, last_status.model, last_status.serial,
                                            last_status.temperature, tempmark,last_status.realloc_sec);
      else
       sprintf(tmp, "%d\tN/A", unit);
     else
      if(last_status.temperature!=INVALID_TEMP)
       sprintf(tmp, "%d%s", last_status.temperature, tempmark);
      else
       sprintf(tmp, "N/A");
    }
    if(hp!=0)
    {
     DosWrite(hp, tmp, strlen(tmp), &pipe_written);
     DosClose(hp);
    }
    DosGetDateTime(&dt);
    printf("%02u/%02u/%04u %02u:%02u:%02u: T hdd=%s\n",
           dt.day, dt.month, dt.year, dt.hours, dt.minutes, dt.seconds, tmp);
#ifdef USE_SYSLOG
    if(daemon&DAEMON_SYSLOG)
    {
     pt=tmp+sprintf(tmp, "%s #%s: ", last_status.model, last_status.serial);
     if(last_status.temperature!=INVALID_TEMP)
      pt+=sprintf(pt, "%u%s", last_status.temperature, tempmark);
     if(last_status.realloc_sec>=0)
     {
      if(last_status.temperature!=INVALID_TEMP)
       pt+=sprintf(pt, ", ");
      pt+=sprintf(pt, "%u RB", last_status.realloc_sec);
     }
     openlog("SMARTMON", 0, LOG_INFO);
     syslog(LOG_INFO, "%s", tmp);
     closelog();
    }
#endif
    if(daemon&DAEMON_ONCE)
     break;
    DosSleep(1000*smart_delay);
   }
   /* NOTREACHED */
  }
  else
   rc=show_smart(hf, &dparams, NULL);
 }
 DosClose(hf);
 return(rc);
}

/* Main routine */

int main(int argc, char **argv)
{
 int rc=0;
 int unit=0;
 int i;

 /* Run-time sanity check */
 if(sizeof(struct smart_rec)!=SMREC_SIZE)
 {
  printf("Internal error at %s:%u\n", __FILE__, __LINE__);
  return(127);
 }
 ansi=isatty(1);
 /* Put our banner */
 APRINTF("\x1B[1;32m");
 printf("HDD S.M.A.R.T. monitor v " PROD_VER
        " ("
          PROD_OS ", " PROD_COMPILER
#ifdef USE_SYSLOG
          ", SYSLOG"
#endif
        ")\n");
 APRINTF("\x1B[0m");
 /* Analyze command line */
 for(i=1; i<argc; i++)
 {
  if(!stricmp(argv[i], "/RAW"))
   raw_mode=1;
  else if(!stricmp(argv[i], "/TM"))
   daemon|=DAEMON_PIPE;
  else if(!stricmp(argv[i], "/DETAIL"))
   daemon|=DAEMON_PIPE_DETAIL;
  else if(!stricmp(argv[i], "/F"))
  {
   fahrenheit=1;
   tempmark="F";
  }
#ifdef USE_SYSLOG
  else if(!stricmp(argv[i], "/SYSLOG"))
   daemon|=DAEMON_SYSLOG;
#endif
  else if(!stricmp(argv[i], "/ONCE"))
   daemon|=DAEMON_ONCE;
  else if(!memicmp(argv[i], "/DELAY:", 7))
  {
   smart_delay=atoi(argv[i]+7);
   if(smart_delay<1)
    smart_delay=1;
  }
  else if(isdigit(argv[i][1]))
   unit=atoi(argv[i]+1);
  else
  {
   APRINTF("\x1B[1;32m"); printf("Programmed by Andrew Belov. Built on " __DATE__ ", " __TIME__); APRINTF("\x1B[0m");
   printf("\n\n"
          "Parameters:\n"
          "     [/0]|/1|/2... = query secondary units (default: /0 = unit 0)\n"
          "              /RAW = extended output (delve into raw values)\n"
          "               /TM = attr. 194/231 temperature monitoring (to " HDD_PIPE ")\n"
          "           /DETAIL = adds more detail to /TM report\n"
#ifdef USE_SYSLOG          
          "           /SYSLOG = similar to /TM but reports to SYSLOG daemon\n"
#endif          
          "             /ONCE = modifier for single-time instead of daemon mode\n"
          "                /F = convert temperature readings to Fahrenheit\n"
          "        /DELAY:<n> = set query cycle to <n> seconds (defaults to 900)\n"
          "\n"
          "Examples:\n"
          "             SMARTMON /1 /TM = temperature monitoring mode for unit 1\n"
          "               SMARTMON /RAW = display raw values for unit 0\n"
#ifdef USE_SYSLOG          
          "SMARTMON /DELAY:1800 /SYSLOG = communicate to SYSLOG every 30 minutes\n"
#endif          
          "\n");
   return(1);
  }
 }
 printf("\n");
 smart_proc(unit);
 return(rc);
}
