/*
 * postscript.c
 * Copyright (C) 1999-2002 A.J. van Os; Released under GPL
 *
 * Description:
 * Functions to deal with the PostScript format
 *
 *================================================================
 * The function vImagePrologue is based on:
 * jpeg2ps - convert JPEG compressed images to PostScript Level 2
 * Copyright (C) 1994-99 Thomas Merz (tm@muc.de)
 *================================================================
 * The credit should go to him, but all the bugs are mine.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include "version.h"
#include "antiword.h"

/* The output must be in PostScript */
static BOOL		bUsePostScript = FALSE;
/* The character set */
static encoding_type	eEncoding = encoding_neutral;
/* The image level */
static image_level_enum	eImageLevel = level_default;
/* The output must use landscape orientation */
static BOOL		bUseLandscape = FALSE;
/* The height of a PostScript page (in DrawUnits) */
static long		lPageHeight = LONG_MAX;
/* Current time for a PS header */
static const char	*szCreationDate = NULL;
/* Current creator for a PS header */
static const char	*szCreator = NULL;
/* Current font information */
static draw_fontref	tFontRefCurr = (draw_fontref)-1;
static short		sFontsizeCurr = -1;
static int		iColorCurr = -1;
/* Current vertical position information */
static long		lYtopCurr = -1;
/* PostScript page counter */
static int		iPageCount = 0;
/* Image counter */
static int		iImageCount = 0;
/* Local representation of the non-breaking space */
static UCHAR		ucNbsp = 0;

static char iso_8859_1_data[] = { "\
/newcodes	% ISO-8859-1 character encodings\n\
[\n\
160/space 161/exclamdown 162/cent 163/sterling 164/currency\n\
165/yen 166/brokenbar 167/section 168/dieresis 169/copyright\n\
170/ordfeminine 171/guillemotleft 172/logicalnot 173/hyphen 174/registered\n\
175/macron 176/degree 177/plusminus 178/twosuperior 179/threesuperior\n\
180/acute 181/mu 182/paragraph 183/periodcentered 184/cedilla\n\
185/onesuperior 186/ordmasculine 187/guillemotright 188/onequarter\n\
189/onehalf 190/threequarters 191/questiondown 192/Agrave 193/Aacute\n\
194/Acircumflex 195/Atilde 196/Adieresis 197/Aring 198/AE 199/Ccedilla\n\
200/Egrave 201/Eacute 202/Ecircumflex 203/Edieresis 204/Igrave 205/Iacute\n\
206/Icircumflex 207/Idieresis 208/Eth 209/Ntilde 210/Ograve 211/Oacute\n\
212/Ocircumflex 213/Otilde 214/Odieresis 215/multiply 216/Oslash\n\
217/Ugrave 218/Uacute 219/Ucircumflex 220/Udieresis 221/Yacute 222/Thorn\n\
223/germandbls 224/agrave 225/aacute 226/acircumflex 227/atilde\n\
228/adieresis 229/aring 230/ae 231/ccedilla 232/egrave 233/eacute\n\
234/ecircumflex 235/edieresis 236/igrave 237/iacute 238/icircumflex\n\
239/idieresis 240/eth 241/ntilde 242/ograve 243/oacute 244/ocircumflex\n\
245/otilde 246/odieresis 247/divide 248/oslash 249/ugrave 250/uacute\n\
251/ucircumflex 252/udieresis 253/yacute 254/thorn 255/ydieresis\n\
] bind def\n\
\n\
/reencdict 12 dict def\n\
\n\
" };

static char iso_8859_2_data[] = { "\
/newcodes	% ISO-8859-2 character encodings\n\
[\n\
160/space 161/Aogonek 162/breve 163/Lslash 164/currency 165/Lcaron\n\
166/Sacute 167/section 168/dieresis 169/Scaron 170/Scommaaccent\n\
171/Tcaron 172/Zacute 173/hyphen 174/Zcaron 175/Zdotaccent 176/degree\n\
177/aogonek 178/ogonek 179/lslash 180/acute 181/lcaron 182/sacute\n\
183/caron 184/cedilla 185/scaron 186/scommaaccent 187/tcaron\n\
188/zacute 189/hungarumlaut 190/zcaron 191/zdotaccent 192/Racute\n\
193/Aacute 194/Acircumflex 195/Abreve 196/Adieresis 197/Lacute\n\
198/Cacute 199/Ccedilla 200/Ccaron 201/Eacute 202/Eogonek\n\
203/Edieresis 204/Ecaron 205/Iacute 206/Icircumflex 207/Dcaron\n\
208/Dcroat 209/Nacute 210/Ncaron 211/Oacute 212/Ocircumflex\n\
213/Ohungarumlaut 214/Odieresis 215/multiply 216/Rcaron 217/Uring\n\
218/Uacute 219/Uhungarumlaut 220/Udieresis 221/Yacute 222/Tcommaaccent\n\
223/germandbls 224/racute 225/aacute 226/acircumflex 227/abreve\n\
228/adieresis 229/lacute 230/cacute 231/ccedilla 232/ccaron 233/eacute\n\
234/eogonek 235/edieresis 236/ecaron 237/iacute 238/icircumflex\n\
239/dcaron 240/dcroat 241/nacute 242/ncaron 243/oacute 244/ocircumflex\n\
245/ohungarumlaut 246/odieresis 247/divide 248/rcaron 249/uring\n\
250/uacute 251/uhungarumlaut 252/udieresis 253/yacute 254/tcommaaccent\n\
255/dotaccent\n\
] bind def\n\
\n\
/reencdict 12 dict def\n\
\n\
" };

static char ibm_866_data[] = { "\
/newcodes       % IBM-866 character encodings\n\
[\n\
128/afii10017 129/afii10018 130/afii10019 131/afii10020 132/afii10021 133/afii10022\n\
134/afii10024 135/afii10025 136/afii10026 137/afii10027 138/afii10028 139/afii10029\n\
140/afii10030 141/afii10031 142/afii10032 143/afii10033 144/afii10034 145/afii10035\n\
146/afii10036 147/afii10037 148/afii10038 149/afii10039 150/afii10040 151/afii10041\n\
152/afii10042 153/afii10043 154/afii10044 155/afii10045 156/afii10046 157/afii10047\n\
158/afii10048 159/afii10049 160/afii10065 161/afii10066 162/afii10067 163/afii10068\n\
164/afii10069 165/afii10070 166/afii10072 167/afii10073 168/afii10074 169/afii10075\n\
170/afii10076 171/afii10077 172/afii10078 173/afii10079 174/afii10080 175/afii10081\n\
176/SF140000 177/SF150000 178/SF160000 179/SF110000 180/SF090000 181/SF190000\n\
182/SF200000 183/SF210000 184/SF220000 185/SF230000 186/SF240000 187/SF250000\n\
188/SF260000 189/SF270000 190/SF280000 191/SF030000 192/SF020000 193/SF070000\n\
194/SF060000 195/SF080000 196/SF100000 197/SF050000 198/SF360000 199/SF370000\n\
200/SF380000 201/SF390000 202/SF400000 203/SF410000 204/SF420000 205/SF430000\n\
206/SF440000 207/SF450000 208/SF460000 209/SF470000 210/SF480000 211/SF490000\n\
212/SF500000 213/SF510000 214/SF520000 215/SF530000 216/SF540000 217/SF040000\n\
218/SF010000 219/SF610000 220/SF570000 221/SF580000 222/SF590000 223/SF600000\n\
224/afii10082 225/afii10083 226/afii10084 227/afii10085 228/afii10086 229/afii10087\n\
230/afii10088 231/afii10089 232/afii10090 233/afii10091 234/afii10092 235/afii10093\n\
236/afii10094 237/afii10095 238/afii10096 239/afii10097 240/afii10023 241/afii10071\n\
242/afii10053 243/afii10101 244/afii10056 245/afii10104 246/afii10062 247/afii10110\n\
248/degree 249/productdot 250/periodcentered 251/radical\n\
252/afii61352 253/currency 254/filledbox 255/spacerequired\n\
] bind def\n\
\n\
/reencdict 12 dict def\n\
\n\
" };

static char iso_8859_x_func[] = { "\
% change fonts using ISO-8859-x characters\n\
/ChgFnt		% size psname natname => font\n\
{\n\
	dup FontDirectory exch known		% is re-encoded name known?\n\
        { exch pop }                            % yes, get rid of long name\n\
        { dup 3 1 roll ReEncode } ifelse        % no, re-encode it\n\
	findfont exch scalefont setfont\n\
} bind def\n\
\n\
/ReEncode\n\
{\n\
reencdict begin\n\
	/newname exch def\n\
	/basename exch def\n\
	/basedict basename findfont def\n\
	/newfont basedict maxlength dict def\n\
	basedict\n\
        { exch dup /FID ne\n\
                { dup /Encoding eq\n\
                        { exch dup length array copy newfont 3 1 roll put }\n\
                        { exch newfont 3 1 roll put } ifelse\n\
                }\n\
                { pop pop } ifelse\n\
        } forall\n\
	newfont /FontName newname put\n\
	newcodes aload pop newcodes length 2 idiv\n\
        { newfont /Encoding get 3 1 roll put } repeat\n\
	newname newfont definefont pop\n\
end\n\
} bind def\n\
\n\
" };

static char misc_func[] = { "\
% draw a line and show the string\n\
/LineShow	% string linewidth movement\n\
{\n\
	gsave\n\
		0 exch rmoveto\n\
		setlinewidth\n\
		dup\n\
		stringwidth pop\n\
		0 rlineto stroke\n\
	grestore\n\
	show\n\
} bind def\n\
\n\
% begin an EPS file (level 2 and up)\n\
/BeginEPSF\n\
{\n\
	/b4_Inc_state save def\n\
	/dict_count countdictstack def\n\
	/op_count count 1 sub def\n\
	userdict begin\n\
		/showpage { } def\n\
		0 setgray 0 setlinecap\n\
		1 setlinewidth 0 setlinejoin\n\
		10 setmiterlimit [ ] 0 setdash newpath\n\
		false setstrokeadjust false setoverprint\n\
} bind def\n\
\n\
% end an EPS file\n\
/EndEPSF {\n\
	count op_count sub { pop } repeat\n\
	countdictstack dict_count sub { end } repeat\n\
	b4_Inc_state restore\n\
} bind def\n\
\n\
" };


/*
 * vAddPageSetup - add the page setup
 */
static void
vAddPageSetup(FILE *pOutFile)
{
	if (bUseLandscape) {
		fprintf(pOutFile, "%%%%BeginPageSetup\n");
		fprintf(pOutFile, "90 rotate\n");
		fprintf(pOutFile, "0.00 %.2f translate\n",
					-dDrawUnits2Points(lPageHeight));
		fprintf(pOutFile, "%%%%EndPageSetup\n");
	}
} /* end of vAddPageSetup */

/*
 * vMove2NextPage - move to the start of the next page
 */
static void
vMove2NextPage(diagram_type *pDiag)
{
	fail(pDiag == NULL);
	fail(!bUsePostScript);

	fprintf(pDiag->pOutFile, "showpage\n");
	iPageCount++;
	fprintf(pDiag->pOutFile, "%%%%Page: %d %d\n", iPageCount, iPageCount);
	vAddPageSetup(pDiag->pOutFile);
	pDiag->lYtop = lPageHeight - PS_TOP_MARGIN;
	lYtopCurr = -1;
} /* end of vMove2NextPage */

/*
 * vMoveToPS - move to the given X,Y coordinates (Postscript)
 *
 * Move the current position of the given diagram to its X,Y coordinates,
 * start on a new page if needed
 */
static void
vMoveToPS(diagram_type *pDiag, long lLastVerticalMovement)
{
	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);

	if (pDiag->lYtop < PS_BOTTOM_MARGIN) {
		vMove2NextPage(pDiag);
		/* Repeat the last vertical movement on the new page */
		pDiag->lYtop -= lLastVerticalMovement;
	}
	fail(pDiag->lYtop < PS_BOTTOM_MARGIN);

	if (pDiag->lYtop != lYtopCurr) {
		fprintf(pDiag->pOutFile, "%.2f %.2f moveto\n",
			dDrawUnits2Points(pDiag->lXleft + PS_LEFT_MARGIN),
			dDrawUnits2Points(pDiag->lYtop));
		lYtopCurr = pDiag->lYtop;
	}
} /* end of vMoveToPS */

/*
 * vPrologue - set options and perform the PostScript initialization
 */
static void
vPrologue(FILE *pOutFile, const char *szTask, const char *szFilename)
{
	options_type	tOptions;
	const char	*szTmp;
	time_t	tTime;

	fail(pOutFile == NULL);
	fail(szTask == NULL || szTask[0] == '\0');

	vGetOptions(&tOptions);
	if (tOptions.iPageHeight == INT_MAX) {
		lPageHeight = LONG_MAX;
	} else {
		lPageHeight = lPoints2DrawUnits(tOptions.iPageHeight);
	}
	DBG_DEC(lPageHeight);
	bUsePostScript = tOptions.bUseOutlineFonts;
	bUseLandscape = tOptions.bUseLandscape;
	eEncoding = tOptions.eEncoding;
	eImageLevel = tOptions.eImageLevel;
	tFontRefCurr = (draw_fontref)-1;
	sFontsizeCurr = -1;
	iColorCurr = -1;
	lYtopCurr = -1;
	iImageCount = 0;

	if (!bUsePostScript) {
		return;
	}

	szCreator = szTask;

	fprintf(pOutFile, "%%!PS-Adobe-2.0\n");
	fprintf(pOutFile, "%%%%Title: %s\n", szBasename(szFilename));
	fprintf(pOutFile, "%%%%Creator: %s %s\n", szCreator, VERSIONSTRING);
	szTmp = getenv("LOGNAME");
	if (szTmp == NULL || szTmp[0] == '\0') {
		szTmp = getenv("USER");
		if (szTmp == NULL || szTmp[0] == '\0') {
			szTmp = "unknown";
		}
	}
	fprintf(pOutFile, "%%%%For: %.50s\n", szTmp);
	errno = 0;
	tTime = time(NULL);
	if (tTime == (time_t)-1 && errno != 0) {
		szCreationDate = NULL;
	} else {
		szCreationDate = ctime(&tTime);
	}
	if (szCreationDate == NULL || szCreationDate[0] == '\0') {
		szCreationDate = "unknown\n";
	}
	fprintf(pOutFile, "%%%%CreationDate: %s", szCreationDate);
	if (bUseLandscape) {
		fprintf(pOutFile, "%%%%Orientation: Landscape\n");
		fprintf(pOutFile, "%%%%BoundingBox: 0 0 %d %d\n",
				tOptions.iPageHeight, tOptions.iPageWidth);
	} else {
		fprintf(pOutFile, "%%%%Orientation: Portrait\n");
		fprintf(pOutFile, "%%%%BoundingBox: 0 0 %d %d\n",
				tOptions.iPageWidth, tOptions.iPageHeight);
	}
} /* end of vPrologue */

/*
 * vEpilogue - clean up after everything is done
 */
static void
vEpilogue(FILE *pFile)
{
	if (!bUsePostScript) {
		fprintf(pFile, "\n");
		return;
	}

	fprintf(pFile, "%%%%Trailer\n");
	fprintf(pFile, "%%%%Pages: %d\n", iPageCount);
	fprintf(pFile, "%%%%EOF\n");
	szCreationDate = NULL;
	szCreator = NULL;
} /* end of vEpilogue */

/*
 * vPrintPalette - print a postscript palette
 */
static void
vPrintPalette(FILE *pOutFile, const imagedata_type *pImg)
{
	int	iIndex;

	fail(pOutFile == NULL);
	fail(pImg == NULL);
	fail(pImg->iColorsUsed < 2);
	fail(pImg->iColorsUsed > 256);

	fprintf(pOutFile, "[ /Indexed\n");
	fprintf(pOutFile, "\t/Device%s %d\n",
		pImg->bColorImage ? "RGB" : "Gray", pImg->iColorsUsed - 1);
	fprintf(pOutFile, "<");
	for (iIndex = 0; iIndex < pImg->iColorsUsed; iIndex++) {
		fprintf(pOutFile, "%02x",
				(unsigned int)pImg->aucPalette[iIndex][0]);
		if (pImg->bColorImage) {
			fprintf(pOutFile, "%02x%02x",
				(unsigned int)pImg->aucPalette[iIndex][1],
				(unsigned int)pImg->aucPalette[iIndex][2]);
		}
		if (iIndex % 8 == 7) {
			fprintf(pOutFile, "\n");
		} else {
			fprintf(pOutFile, " ");
		}
	}
	fprintf(pOutFile, ">\n");
	fprintf(pOutFile, "] setcolorspace\n");
} /* end of vPrintPalette */

/*
 * vImagePrologue - perform the Encapsulated PostScript initialization
 */
void
vImagePrologue(diagram_type *pDiag, const imagedata_type *pImg)
{
	FILE	*pOutFile;

	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(pImg == NULL);

	if (!bUsePostScript) {
		return;
	}

	if (pImg->iVerSizeScaled <= 0 || pImg->iHorSizeScaled <= 0) {
		return;
	}

	fail(szCreationDate == NULL);
	fail(szCreator == NULL);
	fail(eImageLevel == level_no_images);

	iImageCount++;

	DBG_DEC_C(pDiag->lXleft != 0, pDiag->lXleft);

	pDiag->lYtop -= lPoints2DrawUnits(pImg->iVerSizeScaled);
	vMoveToPS(pDiag, lPoints2DrawUnits(pImg->iVerSizeScaled));

	pOutFile = pDiag->pOutFile;

	fprintf(pOutFile, "BeginEPSF\n");
	fprintf(pOutFile, "%%%%BeginDocument: image%03d.eps\n", iImageCount);
	fprintf(pOutFile, "%%!PS-Adobe-2.0 EPSF-2.0\n");
	fprintf(pOutFile, "%%%%Creator: %s %s\n", szCreator, VERSIONSTRING);
	fprintf(pOutFile, "%%%%Title: Image %03d\n", iImageCount);
	fprintf(pOutFile, "%%%%CreationDate: %s", szCreationDate);
	fprintf(pOutFile, "%%%%BoundingBox: 0 0 %d %d\n",
				pImg->iHorSizeScaled, pImg->iVerSizeScaled);
	fprintf(pOutFile, "%%%%DocumentData: Clean7Bit\n");
	fprintf(pOutFile, "%%%%LanguageLevel: 2\n");
	fprintf(pOutFile, "%%%%EndComments\n");
	fprintf(pOutFile, "%%%%BeginProlog\n");
	fprintf(pOutFile, "%%%%EndProlog\n");
	fprintf(pOutFile, "%%%%Page: 1 1\n");

	fprintf(pOutFile, "save\n");

	switch (pImg->eImageType) {
	case imagetype_is_jpeg:
		fprintf(pOutFile, "/Data1 currentfile ");
		fprintf(pOutFile, "/ASCII85Decode filter def\n");
		fprintf(pOutFile, "/Data Data1 << ");
		fprintf(pOutFile, ">> /DCTDecode filter def\n");
		switch (pImg->iComponents) {
		case 1:
			fprintf(pOutFile, "/DeviceGray setcolorspace\n");
			break;
		case 3:
			fprintf(pOutFile, "/DeviceRGB setcolorspace\n");
			break;
		case 4:
			fprintf(pOutFile, "/DeviceCMYK setcolorspace\n");
			break;
		default:
			DBG_DEC(pImg->iComponents);
			break;
		}
		break;
	case imagetype_is_png:
		if (eImageLevel == level_gs_special) {
			fprintf(pOutFile,
			"/Data2 currentfile /ASCII85Decode filter def\n");
			fprintf(pOutFile,
			"/Data1 Data2 << >> /FlateDecode filter def\n");
			fprintf(pOutFile, "/Data Data1 <<\n");
			fprintf(pOutFile, "\t/Colors %d\n", pImg->iComponents);
			fprintf(pOutFile, "\t/BitsPerComponent %d\n",
						pImg->iBitsPerComponent);
			fprintf(pOutFile, "\t/Columns %d\n", pImg->iWidth);
			fprintf(pOutFile,
				">> /PNGPredictorDecode filter def\n");
		} else {
			fprintf(pOutFile,
			"/Data1 currentfile /ASCII85Decode filter def\n");
			fprintf(pOutFile,
			"/Data Data1 << >> /FlateDecode filter def\n");
		}
		if (pImg->iComponents == 3) {
			fprintf(pOutFile, "/DeviceRGB setcolorspace\n");
		} else if (pImg->iColorsUsed > 0) {
			vPrintPalette(pOutFile, pImg);
		} else {
			fprintf(pOutFile, "/DeviceGray setcolorspace\n");
		}
		break;
	case imagetype_is_dib:
		fprintf(pOutFile, "/Data currentfile ");
		fprintf(pOutFile, "/ASCII85Decode filter def\n");
		if (pImg->iBitsPerComponent <= 8) {
			vPrintPalette(pOutFile, pImg);
		} else {
			fprintf(pOutFile, "/DeviceRGB setcolorspace\n");
		}
		break;
	default:
		fprintf(pOutFile, "/Data currentfile ");
		fprintf(pOutFile, "/ASCIIHexDecode filter def\n");
		fprintf(pOutFile, "/Device%s setcolorspace\n",
			pImg->bColorImage ? "RGB" : "Gray");
		break;
	}

	/* Translate to lower left corner of image */
	fprintf(pOutFile, "%.2f %.2f translate\n",
			dDrawUnits2Points(pDiag->lXleft + PS_LEFT_MARGIN),
			dDrawUnits2Points(pDiag->lYtop));

	fprintf(pOutFile, "%d %d scale\n",
				pImg->iHorSizeScaled, pImg->iVerSizeScaled);

	fprintf(pOutFile, "{ <<\n");
	fprintf(pOutFile, "\t/ImageType 1\n");
	fprintf(pOutFile, "\t/Width %d\n", pImg->iWidth);
	fprintf(pOutFile, "\t/Height %d\n", pImg->iHeight);
	if (pImg->eImageType == imagetype_is_dib) {
		/* Scanning from left to right and bottom to top */
		fprintf(pOutFile, "\t/ImageMatrix [ %d 0 0 %d 0 0 ]\n",
			pImg->iWidth, pImg->iHeight);
	} else {
		/* Scanning from left to right and top to bottom */
		fprintf(pOutFile, "\t/ImageMatrix [ %d 0 0 %d 0 %d ]\n",
			pImg->iWidth, -pImg->iHeight, pImg->iHeight);
	}
	fprintf(pOutFile, "\t/DataSource Data\n");

	switch (pImg->eImageType) {
	case imagetype_is_jpeg:
		fprintf(pOutFile, "\t/BitsPerComponent 8\n");
		switch (pImg->iComponents) {
		case 1:
			fprintf(pOutFile, "\t/Decode [0 1]\n");
			break;
		case 3:
			fprintf(pOutFile, "\t/Decode [0 1 0 1 0 1]\n");
			break;
		case 4:
			if (pImg->bAdobe) {
				/*
				 * Adobe-conforming CMYK file
				 * applying workaround for color inversion
				 */
				fprintf(pOutFile,
					"\t/Decode [1 0 1 0 1 0 1 0]\n");
			} else {
				fprintf(pOutFile,
					"\t/Decode [0 1 0 1 0 1 0 1]\n");
			}
			break;
		default:
			DBG_DEC(pImg->iComponents);
			break;
		}
		break;
	case imagetype_is_png:
		if (pImg->iComponents == 3) {
			fprintf(pOutFile, "\t/BitsPerComponent 8\n");
			fprintf(pOutFile, "\t/Decode [0 1 0 1 0 1]\n");
		} else if (pImg->iColorsUsed > 0) {
			fail(pImg->iBitsPerComponent > 8);
			fprintf(pOutFile, "\t/BitsPerComponent %d\n",
					pImg->iBitsPerComponent);
			fprintf(pOutFile, "\t/Decode [0 %d]\n",
					(1 << pImg->iBitsPerComponent) - 1);
		} else {
			fprintf(pOutFile, "\t/BitsPerComponent 8\n");
			fprintf(pOutFile, "\t/Decode [0 1]\n");
		}
		break;
	case imagetype_is_dib:
		fprintf(pOutFile, "\t/BitsPerComponent 8\n");
		if (pImg->iBitsPerComponent <= 8) {
			fprintf(pOutFile, "\t/Decode [0 255]\n");
		} else {
			fprintf(pOutFile, "\t/Decode [0 1 0 1 0 1]\n");
		}
		break;
	default:
		fprintf(pOutFile, "\t/BitsPerComponent 8\n");
		if (pImg->bColorImage) {
			fprintf(pOutFile, "\t/Decode [0 1 0 1 0 1]\n");
		} else {
			fprintf(pOutFile, "\t/Decode [0 1]\n");
		}
		break;
	}

	fprintf(pOutFile, "  >> image\n");
	fprintf(pOutFile, "  Data closefile\n");
	fprintf(pOutFile, "  showpage\n");
	fprintf(pOutFile, "  restore\n");
	fprintf(pOutFile, "} exec\n");
} /* end of vImagePrologue */

/*
 * vImageEpilogue - clean up after Encapsulated PostScript
 */
void
vImageEpilogue(diagram_type *pDiag)
{
	FILE	*pOutFile;

	if (!bUsePostScript) {
		return;
	}

	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);

	pOutFile = pDiag->pOutFile;

	fprintf(pOutFile, "%%%%EOF\n");
	fprintf(pOutFile, "%%%%EndDocument\n");
	fprintf(pOutFile, "EndEPSF\n");

	pDiag->lXleft = 0;
} /* end of vImageEpilogue */

/*
 * bAddDummyImage - add a dummy image
 *
 * return TRUE when successful, otherwise FALSE
 */
BOOL
bAddDummyImage(diagram_type *pDiag, const imagedata_type *pImg)
{
	FILE	*pOutFile;

	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(pImg == NULL);

	if (!bUsePostScript) {
		return FALSE;
	}

	if (pImg->iVerSizeScaled <= 0 || pImg->iHorSizeScaled <= 0) {
		return FALSE;
	}

	iImageCount++;

	DBG_DEC_C(pDiag->lXleft != 0, pDiag->lXleft);

	pDiag->lYtop -= lPoints2DrawUnits(pImg->iVerSizeScaled);
	vMoveToPS(pDiag, lPoints2DrawUnits(pImg->iVerSizeScaled));

	pOutFile = pDiag->pOutFile;

	fprintf(pOutFile, "gsave %% Image %03d\n", iImageCount);
	fprintf(pOutFile, "\tnewpath\n");
	fprintf(pOutFile, "\t%.2f %.2f moveto\n",
			dDrawUnits2Points(pDiag->lXleft + PS_LEFT_MARGIN),
			dDrawUnits2Points(pDiag->lYtop));
	fprintf(pOutFile, "\t1.0 setlinewidth\n");
	fprintf(pOutFile, "\t0.3 setgray\n");
	fprintf(pOutFile, "\t0 %d rlineto\n", pImg->iVerSizeScaled);
	fprintf(pOutFile, "\t%d 0 rlineto\n", pImg->iHorSizeScaled);
	fprintf(pOutFile, "\t0 %d rlineto\n", -pImg->iVerSizeScaled);
	fprintf(pOutFile, "\tclosepath\n");
	fprintf(pOutFile, "\tstroke\n");
	fprintf(pOutFile, "grestore\n");

	pDiag->lXleft = 0;

	return TRUE;
} /* end of bAddDummyImage */

/*
 * pCreateDiagram - create and initialize a diagram
 *
 * remark: does not return if the diagram can't be created
 */
diagram_type *
pCreateDiagram(const char *szTask, const char *szFilename)
{
	diagram_type	*pDiag;

	fail(szTask == NULL || szTask[0] == '\0');
	DBG_MSG("pCreateDiagram");

	/* Get the necessary memory */
	pDiag = xmalloc(sizeof(diagram_type));
	/* Initialization */
	pDiag->pOutFile = stdout;
	vPrologue(pDiag->pOutFile, szTask, szFilename);
	iPageCount = 0;
	pDiag->lXleft = 0;
	if (bUsePostScript) {
		pDiag->lYtop = lPageHeight - PS_TOP_MARGIN;
	} else {
		pDiag->lYtop = 0;
	}
	/* Return success */
	return pDiag;
} /* end of pCreateDiagram */

/*
 * vDestroyDiagram - remove a diagram by freeing the memory it uses
 */
void
vDestroyDiagram(diagram_type *pDiag)
{
	DBG_MSG("vDestroyDiagram");

	fail(pDiag == NULL);

	if (pDiag == NULL) {
		return;
	}
	if (bUsePostScript && pDiag->lYtop < lPageHeight - PS_TOP_MARGIN) {
		fprintf(pDiag->pOutFile, "showpage\n");
	}
	vEpilogue(pDiag->pOutFile);
	pDiag = xfree(pDiag);
} /* end of vDestroyDiagram */

/*
 * vAddFonts2Diagram - add the list of fonts and complete the prologue
 */
void
vAddFonts2Diagram(diagram_type *pDiag)
{
	FILE	*pOutFile;
	const font_table_type *pTmp, *pTmp2;
	int	iLineLen;
	BOOL	bFound;

	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);

	if (!bUsePostScript) {
		return;
	}

	pOutFile = pDiag->pOutFile;
	iLineLen = fprintf(pOutFile, "%%%%DocumentFonts:");

	if (tGetFontTableLength() == 0) {
		iLineLen += fprintf(pOutFile, " Courier");
	} else {
		pTmp = NULL;
		while ((pTmp = pGetNextFontTableRecord(pTmp)) != NULL) {
			/* Print the document fonts */
			bFound = FALSE;
			pTmp2 = NULL;
			while ((pTmp2 = pGetNextFontTableRecord(pTmp2))
					!= NULL && pTmp2 < pTmp) {
				bFound = STREQ(pTmp2->szOurFontname,
						pTmp->szOurFontname);
				if (bFound) {
					break;
				}
			}
			if (bFound) {
				continue;
			}
			if (iLineLen + (int)strlen(pTmp->szOurFontname) > 78) {
				fprintf(pOutFile, "\n%%%%+");
				iLineLen = 3;
			}
			iLineLen += fprintf(pOutFile,
					" %s", pTmp->szOurFontname);
		}
	}
	fprintf(pOutFile, "\n");
	fprintf(pOutFile, "%%%%Pages: (atend)\n");
	fprintf(pOutFile, "%%%%EndComments\n");
	fprintf(pOutFile, "%%%%BeginProlog\n");

	switch (eEncoding) {
	case encoding_iso_8859_1:
		fprintf(pOutFile, "%s\n%s", iso_8859_1_data, iso_8859_x_func);
		break;
	case encoding_iso_8859_2:
		fprintf(pOutFile, "%s\n%s", iso_8859_2_data, iso_8859_x_func);
		break;
        case encoding_ibm_866:
                fprintf(pOutFile, "%s\n%s", ibm_866_data, iso_8859_x_func);
                break;
        default:
		DBG_DEC(eEncoding);
		break;
	}

	/* The rest of the functions */
	fprintf(pOutFile, "%s", misc_func);
	fprintf(pOutFile, "%%%%EndProlog\n");
	iPageCount = 1;
	fprintf(pDiag->pOutFile, "%%%%Page: %d %d\n", iPageCount, iPageCount);
	vAddPageSetup(pDiag->pOutFile);
} /* end of vAddFonts2Diagram */

/*
 * vPrintPS - print a PostScript string
 */
static void
vPrintPS(FILE *pFile, const char *szString, int iStringLength,
		UCHAR ucFontstyle)
{
	const UCHAR	*ucBytes;
	int		iCount;

	fail(szString == NULL || iStringLength < 0);

	if (szString == NULL || szString[0] == '\0' || iStringLength <= 0) {
		return;
	}

	ucBytes = (UCHAR *)szString;
	(void)putc('(', pFile);
	for (iCount = 0; iCount < iStringLength ; iCount++) {
		switch (ucBytes[iCount]) {
		case '(':
		case ')':
		case '\\':
			(void)putc('\\', pFile);
			(void)putc(szString[iCount], pFile);
			break;
		default:
                        if ((int)ucBytes[iCount] < 0x20){
                                DBG_HEX(ucBytes[iCount]);
				(void)putc(' ', pFile);
                        } else if ((int)ucBytes[iCount] >= 0x80) {
                                fprintf(pFile, "%c", ucBytes[iCount]);
			} else {
				(void)putc(szString[iCount], pFile);
			}
			break;
		}
	}
	fprintf(pFile, ") ");
	if ((bIsStrike(ucFontstyle) || bIsMarkDel(ucFontstyle)) &&
			sFontsizeCurr > 0) {
		fprintf(pFile, "%.2f %.2f LineShow\n",
			sFontsizeCurr * 0.02, sFontsizeCurr * 0.12);
	} else if (bIsUnderline(ucFontstyle) && sFontsizeCurr > 0) {
		fprintf(pFile, "%.2f %.2f LineShow\n",
			sFontsizeCurr * 0.02, sFontsizeCurr * -0.06);
	} else {
		fprintf(pFile, "show\n");
	}
} /* end of vPrintPS */

/*
 * vPrintTXT - print a Text string
 */
static void
vPrintTXT(FILE *pFile, const char *szString, int iStringLength)
{
	const UCHAR	*ucBytes;
	int		iCount;

	fail(szString == NULL || iStringLength < 0);

	if (szString == NULL || szString[0] == '\0' || iStringLength <= 0) {
		return;
	}

	if (eEncoding == encoding_utf8) {
		fprintf(pFile, "%.*s", iStringLength, szString);
		return;
	}

	if (ucNbsp == 0) {
		ucNbsp = ucGetNbspValue();
		DBG_HEX_C(ucNbsp != 0xa0, ucNbsp);
	}

	ucBytes = (UCHAR *)szString;
	for (iCount = 0; iCount < iStringLength ; iCount++) {
		if (ucBytes[iCount] == ucNbsp) {
			(void)putc(' ', pFile);
		} else {
			(void)putc(szString[iCount], pFile);
		}
	}
} /* end of vPrintTXT */

/*
 * vSetColor - move to the given color
 */
static void
vSetColor(FILE *pFile, int iColor)
{
	ULONG	ulTmp, ulRed, ulGreen, ulBlue;

	ulTmp = ulColor2Color(iColor);
	ulRed   = (ulTmp & 0x0000ff00) >> 8;
	ulGreen = (ulTmp & 0x00ff0000) >> 16;
	ulBlue  = (ulTmp & 0xff000000) >> 24;
	fprintf(pFile, "%.3f %.3f %.3f setrgbcolor\n",
				ulRed / 255.0, ulGreen / 255.0, ulBlue / 255.0);
} /* end of vSetColor */

/*
 * vMoveToTXT - move to the given X,Y coordinates (Text)
 *
 * Move the current position of the given diagram to its X,Y coordinates,
 * start on a new page if needed
 */
static void
vMoveToTXT(diagram_type *pDiag)
{
	int	iCount, iNbr;

	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);

	if (pDiag->lYtop != lYtopCurr) {
		iNbr = iDrawUnits2Char(pDiag->lXleft);
		for (iCount = 0; iCount < iNbr; iCount++) {
			(void)putc(FILLER_CHAR, pDiag->pOutFile);
		}
		lYtopCurr = pDiag->lYtop;
	}
} /* end of vMoveToTXT */

/*
 * vMove2NextLine - move to the next line
 */
void
vMove2NextLine(diagram_type *pDiag, draw_fontref tFontRef, short sFontsize)
{
	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(sFontsize < MIN_FONT_SIZE || sFontsize > MAX_FONT_SIZE);

	pDiag->lYtop -= lComputeLeading(sFontsize);
	if (!bUsePostScript) {
		(void)fprintf(pDiag->pOutFile, "\n");
	}
} /* end of vMove2NextLine */

/*
 * vSubstring2Diagram - put a sub string into a diagram
 */
void
vSubstring2Diagram(diagram_type *pDiag,
	char *szString, int iStringLength, long lStringWidth,
	int iColor, UCHAR ucFontstyle, draw_fontref tFontRef,
	short sFontsize, short sMaxFontsize)
{
	const char	*szOurFontname;

	fail(pDiag == NULL || szString == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(pDiag->lXleft < 0);
	fail(iStringLength < 0);
	fail((size_t)iStringLength != strlen(szString));
	fail(sFontsize < MIN_FONT_SIZE || sFontsize > MAX_FONT_SIZE);
	fail(sMaxFontsize < MIN_FONT_SIZE || sMaxFontsize > MAX_FONT_SIZE);
	fail(sFontsize > sMaxFontsize);

	if (szString[0] == '\0' || iStringLength <= 0) {
		return;
	}

	if (bUsePostScript) {
		if (tFontRef != tFontRefCurr || sFontsize != sFontsizeCurr) {
			szOurFontname = szGetFontname(tFontRef);
			fail(szOurFontname == NULL);
			fprintf(pDiag->pOutFile,
				"%.1f /%s /%s-ISO-8859-x ChgFnt\n",
				(double)sFontsize / 2.0,
				szOurFontname, szOurFontname);
			tFontRefCurr = tFontRef;
			sFontsizeCurr = sFontsize;
		}
		if (iColor != iColorCurr) {
			vSetColor(pDiag->pOutFile, iColor);
			iColorCurr = iColor;
		}
		vMoveToPS(pDiag, lComputeLeading(sMaxFontsize));
		vPrintPS(pDiag->pOutFile, szString, iStringLength, ucFontstyle);
	} else {
		vMoveToTXT(pDiag);
		vPrintTXT(pDiag->pOutFile, szString, iStringLength);
	}
	pDiag->lXleft += lStringWidth;
} /* end of vSubstring2Diagram */

/*
 * Create an start of paragraph by moving the y-top mark
 */
void
vStartOfParagraph2Diagram(diagram_type *pDiag,
	draw_fontref tFontRef, short sFontsize, long lBeforeIndentation)
{
	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(sFontsize < MIN_FONT_SIZE || sFontsize > MAX_FONT_SIZE);
	fail(lBeforeIndentation < 0);

	if (bUsePostScript) {
		pDiag->lXleft = 0;
		pDiag->lYtop -= lMilliPoints2DrawUnits(lBeforeIndentation);
	} else {
		if (lBeforeIndentation >= lTwips2MilliPoints(HEADING_GAP)) {
			/* A large gap is replaced by an empty line */
			vMove2NextLine(pDiag, tFontRef, sFontsize);
		}
	}
} /* end of vStartOfParagraph2Diagram */

/*
 * Create an end of paragraph by moving the y-top mark
 */
void
vEndOfParagraph2Diagram(diagram_type *pDiag,
	draw_fontref tFontRef, short sFontsize, long lAfterIndentation)
{
	fail(pDiag == NULL);
	fail(pDiag->pOutFile == NULL);
	fail(sFontsize < MIN_FONT_SIZE || sFontsize > MAX_FONT_SIZE);
	fail(lAfterIndentation < 0);

	if (pDiag->lXleft > 0) {
		/* To the start of the line */
		vMove2NextLine(pDiag, tFontRef, sFontsize);
	}

	if (bUsePostScript) {
		pDiag->lXleft = 0;
		pDiag->lYtop -= lMilliPoints2DrawUnits(lAfterIndentation);
	} else {
		if (lAfterIndentation >= lTwips2MilliPoints(HEADING_GAP)) {
			/* A large gap is replaced by an empty line */
			vMove2NextLine(pDiag, tFontRef, sFontsize);
		}
	}
} /* end of vEndOfParagraph2Diagram */

/*
 * Create an end of page
 */
void
vEndOfPage2Diagram(diagram_type *pDiag,
	draw_fontref tFontRef, short sFontsize, long lAfterIndentation)
{
	if (bUsePostScript) {
		vMove2NextPage(pDiag);
	} else {
		vEndOfParagraph2Diagram(pDiag,
					tFontRef,
					sFontsize,
					lAfterIndentation);
	}
} /* end of vEndOfPage2Diagram */
