/*
 * fontlist.c
 * Copyright (C) 1998-2001 A.J. van Os; Released under GPL
 *
 * Description:
 * Build, read and destroy a list of Word font information
 */

#include <stdlib.h>
#include <stddef.h>
#include "antiword.h"


/* Variables needed to write the Font Information List */
static font_desc_type	*pAnchor = NULL;
static font_desc_type	*pFontLast = NULL;


/*
 * vDestroyFontInfoList - destroy the Font Information List
 */
void
vDestroyFontInfoList(void)
{
	font_desc_type	*pCurr, *pNext;

	DBG_MSG("vDestroyFontInfoList");

	/* Free the Font Information List */
	pCurr = pAnchor;
	while (pCurr != NULL) {
		pNext = pCurr->pNext;
		pCurr = xfree(pCurr);
		pCurr = pNext;
	}
	pAnchor = NULL;
	/* Reset all control variables */
	pFontLast = NULL;
} /* end of vDestroyFontInfoList */

/*
 * vCorrectFontValues - correct font values that Antiword can't use
 */
void
vCorrectFontValues(font_block_type *pFontBlock)
{
	int	iRealSize;
	UCHAR	ucRealStyle;

	iRealSize = pFontBlock->sFontsize;
	ucRealStyle = pFontBlock->ucFontstyle;
	if (bIsSmallCapitals(pFontBlock->ucFontstyle)) {
		/* Small capitals become normal capitals in a smaller font */
		iRealSize = (iRealSize * 8 + 5) / 10;
		ucRealStyle &= ~FONT_SMALL_CAPITALS;
		ucRealStyle |= FONT_CAPITALS;
	}

	if (iRealSize < MIN_FONT_SIZE) {
		DBG_DEC(iRealSize);
		iRealSize = MIN_FONT_SIZE;
	} else if (iRealSize > MAX_FONT_SIZE) {
		DBG_DEC(iRealSize);
		iRealSize = MAX_FONT_SIZE;
	}

	pFontBlock->sFontsize = (short)iRealSize;
	if (pFontBlock->ucFontcolor == 8) {
		/* White text to light gray text */
		pFontBlock->ucFontcolor = 16;
	}
	pFontBlock->ucFontstyle = ucRealStyle;
} /* end of vCorrectFontValues */

/*
 * vAdd2FontInfoList - Add an element to the Font Information List
 */
void
vAdd2FontInfoList(const font_block_type *pFontBlock)
{
	font_desc_type	*pListMember;

	fail(pFontBlock == NULL);

	NO_DBG_MSG("bAdd2FontInfoList");

	if (pFontBlock->ulFileOffset == FC_INVALID) {
		/*
		 * This offset is really past the end of the file,
		 * so don't waste any memory by storing it.
		 */
		return;
	}

	NO_DBG_HEX(pFontBlock->ulFileOffset);
	NO_DBG_DEC_C(pFontBlock->ucFontnumber != 0,
					pFontBlock->ucFontnumber);
	NO_DBG_DEC_C(pFontBlock->sFontsize != DEFAULT_FONT_SIZE,
					pFontBlock->sFontsize);
	NO_DBG_DEC_C(pFontBlock->ucFontcolor != 0,
					pFontBlock->ucFontcolor);
	NO_DBG_HEX_C(pFontBlock->ucFontstyle != 0x00,
					pFontBlock->ucFontstyle);

	if (pFontLast != NULL &&
	    pFontLast->tInfo.ulFileOffset == pFontBlock->ulFileOffset) {
		/*
		 * If two consecutive fonts share the same
		 * offset, remember only the last font
		 */
		fail(pFontLast->pNext != NULL);
		pFontLast->tInfo = *pFontBlock;
		return;
	}

	/* Create list member */
	pListMember = xmalloc(sizeof(font_desc_type));
	/* Fill the list member */
	pListMember->tInfo = *pFontBlock;
	pListMember->pNext = NULL;
	/* Correct the values where needed */
	vCorrectFontValues(&pListMember->tInfo);
	/* Add the new member to the list */
	if (pAnchor == NULL) {
		pAnchor = pListMember;
	} else {
		fail(pFontLast == NULL);
		pFontLast->pNext = pListMember;
	}
	pFontLast = pListMember;
} /* end of vAdd2FontInfoList */

/*
 * Get the record that follows the given recored in the Font Information List
 */
const font_block_type *
pGetNextFontInfoListItem(const font_block_type *pCurr)
{
	const font_desc_type	*pRecord;
	size_t	tOffset;

	if (pCurr == NULL) {
		if (pAnchor == NULL) {
			/* There are no records */
			return NULL;
		}
		/* The first record is the only one without a predecessor */
		return &pAnchor->tInfo;
	}
	tOffset = offsetof(font_desc_type, tInfo);
	/* Many casts to prevent alignment warnings */
	pRecord = (font_desc_type *)(void *)((char *)pCurr - tOffset);
	fail(pCurr != &pRecord->tInfo);
	if (pRecord->pNext == NULL) {
		/* The last record has no successor */
		return NULL;
	}
	return &pRecord->pNext->tInfo;
} /* end of pGetNextFontInfoListItem */
