/*
 * findtext.c
 * Copyright (C) 1998-2001 A.J. van Os; Released under GPL
 *
 * Description:
 * Find the blocks that contain the text of MS Word files
 */

#include <stdio.h>
#include <stdlib.h>
#include "antiword.h"


/*
 * bAddTextBlocks - Add the blocks to the text block list
 *
 * Returns TRUE when successful, FALSE if not
 */
BOOL
bAddTextBlocks(ULONG ulCharPosFirst, ULONG ulTotalLength,
	BOOL bUsesUnicode, USHORT usPropMod,
	ULONG ulStartBlock, const ULONG *aulBBD, size_t tBBDLen)
{
	text_block_type	tTextBlock;
	ULONG	ulCharPos, ulOffset, ulIndex;
	long	lToGo;

	fail(ulTotalLength > (ULONG)LONG_MAX / 2);
	fail(ulStartBlock > MAX_BLOCKNUMBER && ulStartBlock != END_OF_CHAIN);
	fail(aulBBD == NULL);

	NO_DBG_HEX(ulCharPosFirst);
	NO_DBG_DEC(ulTotalLength);

	if (bUsesUnicode) {
		/* One character equals two bytes */
		NO_DBG_MSG("Uses Unicode");
		lToGo = (long)ulTotalLength * 2;
	} else {
		/* One character equals one byte */
		NO_DBG_MSG("Uses ASCII");
		lToGo = (long)ulTotalLength;
	}

	ulCharPos = ulCharPosFirst;
	ulOffset = ulCharPosFirst;
	for (ulIndex = ulStartBlock;
	     ulIndex != END_OF_CHAIN && lToGo > 0;
	     ulIndex = aulBBD[ulIndex]) {
		if (ulIndex >= (ULONG)tBBDLen) {
			DBG_DEC(ulIndex);
			DBG_DEC(tBBDLen);
			werr(1, "The Big Block Depot is damaged");
		}
		if (ulOffset >= BIG_BLOCK_SIZE) {
			ulOffset -= BIG_BLOCK_SIZE;
			continue;
		}
		tTextBlock.ulFileOffset =
			(ulIndex + 1) * BIG_BLOCK_SIZE + ulOffset;
		tTextBlock.ulCharPos = ulCharPos;
		tTextBlock.ulLength = min(BIG_BLOCK_SIZE - ulOffset, lToGo);
		tTextBlock.bUsesUnicode = bUsesUnicode;
		tTextBlock.usPropMod = usPropMod;
		ulOffset = 0;
		if (!bAdd2TextBlockList(&tTextBlock)) {
			DBG_HEX(tTextBlock.ulFileOffset);
			DBG_HEX(tTextBlock.ulCharPos);
			DBG_DEC(tTextBlock.ulLength);
			DBG_DEC(tTextBlock.bUsesUnicode);
			DBG_DEC(tTextBlock.usPropMod);
			return FALSE;
		}
		ulCharPos += tTextBlock.ulLength;
		lToGo -= tTextBlock.ulLength;
	}
	DBG_DEC_C(lToGo != 0, lToGo);
	return lToGo == 0;
} /* end of bAddTextBlocks */

/*
 * bGet6DocumentText - make a list of the text blocks of Word 6/7 files
 *
 * Code for "fast saved" files.
 *
 * Returns TRUE when successful, FALSE if not
 */
BOOL
bGet6DocumentText(FILE *pFile, BOOL bUsesUnicode, ULONG ulStartBlock,
	const ULONG *aulBBD, size_t tBBDLen, const UCHAR *aucHeader)
{
	UCHAR	*aucBuffer;
	ULONG	ulBeginTextInfo, ulTextOffset, ulTotLength;
	size_t	tTextInfoLen, tOff;
	int	iIndex, iType, iLen, iPieces;
	USHORT	usPropMod;

	DBG_MSG("bGet6DocumentText");

	fail(pFile == NULL);
	fail(aulBBD == NULL);
	fail(aucHeader == NULL);

	ulBeginTextInfo = ulGetLong(0x160, aucHeader);	/* fcClx */
	DBG_HEX(ulBeginTextInfo);
	tTextInfoLen = (size_t)ulGetLong(0x164, aucHeader);	/* lcbClx */
	DBG_DEC(tTextInfoLen);

	aucBuffer = xmalloc(tTextInfoLen);
	if (!bReadBuffer(pFile, ulStartBlock,
			aulBBD, tBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, ulBeginTextInfo, tTextInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return FALSE;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tTextInfoLen);

	tOff = 0;
	while (tOff < tTextInfoLen) {
		iType = (int)ucGetByte(tOff, aucBuffer);
		tOff++;
		if (iType == 0) {
			DBG_FIXME();
			tOff++;
			continue;
		}
		if (iType == 1) {
			iLen = (int)usGetWord(tOff, aucBuffer);
			vAdd2PropModList(aucBuffer + tOff);
			tOff += iLen + 2;
			continue;
		}
		if (iType != 2) {
			werr(0, "Unknown type of 'fastsaved' format");
			aucBuffer = xfree(aucBuffer);
			return FALSE;
		}
		/* Type 2 */
		iLen = (int)usGetWord(tOff, aucBuffer);
		NO_DBG_DEC(iLen);
		tOff += 4;
		iPieces = (iLen - 4) / 12;
		DBG_DEC(iPieces);
		for (iIndex = 0; iIndex < iPieces; iIndex++) {
			ulTextOffset = ulGetLong(
				tOff + (iPieces + 1) * 4 + iIndex * 8 + 2,
				aucBuffer);
			usPropMod = usGetWord(
				tOff + (iPieces + 1) * 4 + iIndex * 8 + 6,
				aucBuffer);
			ulTotLength = ulGetLong(tOff + (iIndex + 1) * 4,
						aucBuffer) -
					ulGetLong(tOff + iIndex * 4,
						aucBuffer);
			NO_DBG_HEX_C(usPropMod != 0, usPropMod);
			if (!bAddTextBlocks(ulTextOffset, ulTotLength,
					bUsesUnicode, usPropMod,
					ulStartBlock,
					aulBBD, tBBDLen)) {
				aucBuffer = xfree(aucBuffer);
				return FALSE;
			}
		}
		break;
	}
	aucBuffer = xfree(aucBuffer);
	return TRUE;
} /* end of bGet6DocumentText */

/*
 * bGet8DocumentText - make a list of the text blocks of Word 8/97 files
 *
 * Returns TRUE when successful, FALSE if not
 */
BOOL
bGet8DocumentText(FILE *pFile, const pps_info_type *pPPS,
	const ULONG *aulBBD, size_t tBBDLen,
	const ULONG *aulSBD, size_t tSBDLen,
	const UCHAR *aucHeader)
{
	const ULONG	*aulBlockDepot;
	UCHAR	*aucBuffer;
	ULONG	ulTextOffset, ulBeginTextInfo;
	ULONG	ulTotLength, ulLen;
	ULONG	ulTableStartBlock, ulTableSize;
	long	lIndex, lPieces;
	size_t	tTextInfoLen, tBlockDepotLen, tBlockSize, tOff;
	int	iType, iLen;
	BOOL	bUsesUnicode;
	USHORT	usDocStatus, usPropMod;

	DBG_MSG("bGet8DocumentText");

	fail(pFile == NULL || pPPS == NULL);
	fail(aulBBD == NULL || aulSBD == NULL);
	fail(aucHeader == NULL);

  	ulBeginTextInfo = ulGetLong(0x1a2, aucHeader);	/* fcClx */
	DBG_HEX(ulBeginTextInfo);
	tTextInfoLen = (size_t)ulGetLong(0x1a6, aucHeader);	/* lcbClx */
	DBG_DEC(tTextInfoLen);

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		ulTableStartBlock = pPPS->t1Table.ulSB;
		ulTableSize = pPPS->t1Table.ulSize;
	} else {
		ulTableStartBlock = pPPS->t0Table.ulSB;
		ulTableSize = pPPS->t0Table.ulSize;
	}
	DBG_DEC(ulTableStartBlock);
	if (ulTableStartBlock == 0) {
		DBG_DEC(ulTableStartBlock);
		return FALSE;
	}
	DBG_HEX(ulTableSize);
	if (ulTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		aulBlockDepot = aulSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		aulBlockDepot = aulBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tTextInfoLen);
	if (!bReadBuffer(pFile, ulTableStartBlock,
			aulBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, ulBeginTextInfo, tTextInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return FALSE;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tTextInfoLen);

	tOff = 0;
	while (tOff < tTextInfoLen) {
		iType = (int)ucGetByte(tOff, aucBuffer);
		tOff++;
		if (iType == 0) {
			DBG_FIXME();
			tOff++;
			continue;
		}
		if (iType == 1) {
			iLen = (int)usGetWord(tOff, aucBuffer);
			vAdd2PropModList(aucBuffer + tOff);
			tOff += iLen + 2;
			continue;
		}
		if (iType != 2) {
			werr(0, "Unknown type of 'fastsaved' format");
			aucBuffer = xfree(aucBuffer);
			return FALSE;
		}
		/* Type 2 */
		ulLen = ulGetLong(tOff, aucBuffer);
		if (ulLen < 4) {
			DBG_DEC(ulLen);
			return FALSE;
		}
		tOff += 4;
		lPieces = (long)((ulLen - 4) / 12);
		DBG_DEC(lPieces);
		for (lIndex = 0; lIndex < lPieces; lIndex++) {
			ulTextOffset = ulGetLong(
				tOff + (lPieces + 1) * 4 + lIndex * 8 + 2,
				aucBuffer);
			usPropMod = usGetWord(
				tOff + (lPieces + 1) * 4 + lIndex * 8 + 6,
				aucBuffer);
			ulTotLength = ulGetLong(tOff + (lIndex + 1) * 4,
						aucBuffer) -
					ulGetLong(tOff + lIndex * 4,
						aucBuffer);
			if ((ulTextOffset & BIT(30)) == 0) {
				bUsesUnicode = TRUE;
			} else {
				bUsesUnicode = FALSE;
				ulTextOffset &= ~BIT(30);
				ulTextOffset /= 2;
			}
			NO_DBG_HEX_C(usPropMod != 0, usPropMod);
			if (!bAddTextBlocks(ulTextOffset, ulTotLength,
					bUsesUnicode, usPropMod,
					pPPS->tWordDocument.ulSB,
					aulBBD, tBBDLen)) {
				aucBuffer = xfree(aucBuffer);
				return FALSE;
			}
		}
		break;
	}
	aucBuffer = xfree(aucBuffer);
	return TRUE;
} /* end of bGet8DocumentText */
