/* Rexx program to add the entries of the BackAgain 2000 Scheduler *\
   Folder to the scheduler.  In other words:  Activate the entries
   in the folder after system start.

   The script just accesses the Scheduler Folder object. This does
   everything needed. The rest is for installation/de-installation

   Installation: Run the program with the parameter "Install":
                 The program copies itself and it's icon file into
                 the Back Again 2000 program directory, creates an
                 program object in the the Back Again 2000 folder,
                 and a shadow object in the Startup folder.

   De-Installation: Run the program with the parameter "Uninstall":
                    The program removes all objects and deletes
                    it's files.

   Author: Michael Warmuth <michael@warmuth.at>

\* (c) Michael Warmuth - EDV-Dienstleistungen 2001                 */


   /* Configuration */
   ba2k_sched_id      = '<BA2KObjSched_V21>'
   ba2k_sched_setting = 'MENUITEMSELECTED=0;'
   ba2k_folder_id     = '<BA2K_SV>'
   ba2k_ini_app       = 'BackAgain2000_V21'
   ba2k_ini_key       = 'InstPath'
   ba2k_name          = 'Back Again/2000'
   my_id              = '<BA2KObjSchedulerAdd>'
   startup_id         = '<WP_START>'

   /* Load the Rexx Utility functions */
   if RxFuncQuery('SysLoadFuncs') then do
      call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
      call SysLoadFuncs
   end /* if */

   /* Check parameter to decide if we have to install ourself */
   if arg() > 0 then do
      parse upper value arg(1) with parm

      select
         /* Install myself */
         when parm = 'INSTALL' then do
            call Self_Install
         end /* when */
         /* Uninstall myself: Heeeeeelpppp I must die! */
         when parm = 'UNINSTALL' then do
            call Self_Uninstall
         end /* when */
      /* Show help */
      otherwise
         do i = 1 by 1 until right(next_line,2) = '*/'
            next_line = sourceline(i)
            say next_line
         end /* do */
         retc = 0
         signal Exit_Now
      end /* select */
   end /* if */

   /* Just access the object with a command doing nothing */
   say 'Initializing' ba2k_name 'Scheduler...'
   retc = SysSetObjectData(ba2k_sched_id,ba2k_sched_setting)
   if retc then do
      say '   ' || ba2k_name 'Scheduler initialized.'
      retc = 0
   end /* if */
   else do
      say 'Could not find' ba2k_name 'Scheduler. Aborting...'
      retc = 1
      signal Exit_Now
   end /* else */

   /* Exit as we have done all we had to do */
   Exit_Now:
   say '...exiting.'
   exit retc




/* Function to install ourself */
Self_Install:
   /* Install our files */
   say 'Installing' ba2k_name 'Scheduler Initialization...'

   /* Get our own file location */
   parse source . . my_file
   my_icon = left(my_file,lastpos('.',my_file)) || 'ico'
   my_path = strip(left(my_file,lastpos('\',my_file)),'t','\')

   /* Get BA2K installation dir from user ini */
   parse value SysIni('USER',ba2k_ini_app,ba2k_ini_key) with,
                                 ba2k_dir '00'x .
   if left(ba2k_dir,6) = 'ERROR:' then do
      say ba2k_name 'not installed. Aborting...'
      retc = 1
      signal Exit_Now
   end /* if */
   else do
      say '   Found' ba2k_name 'installation in',
          ba2k_dir || '.'
   end /* else */

   /* Copy ourself and our icon file to BA2K dir */
   '@copy "' || my_file || '" "' || ba2k_dir ||,
                                 '" 1> nul 2> nul'
   if rc \= 0 then do
      say 'Could not install (copy) myself. Aborting...'
      retc = 1
      signal Exit_Now
   end /* if */
   else do
      say '   Installed myself to' ba2k_dir || '.'
   end /* else */

   '@copy "' || my_icon || '" "' || ba2k_dir ||,
                                 '" 1> nul 2> nul'
   if rc \= 0 then do
      say 'Could not install (copy) my icon. Aborting...'
      retc = 1
      signal Exit_Now
   end /* if */
   else do
      say '   Installed my icon to' ba2k_dir || '.'
   end /* else */


   /* Create our program objects */

   /* Our new paths */
   my_new_file = ba2k_dir || '\' || filespec('n',my_file)
   my_new_icon = ba2k_dir || '\' || filespec('n',my_icon)

   /* Destroy program object from previous installation */
   if SysDestroyObject(my_id) then do
      say '   Destroyed program object from previous',
          'installation.'
   end /* if */

   /* Create my program object in BA2K folder */
   if SysCreateObject('WPProgram',,
                     ba2k_name || '^Schduler Initialization',,
                     ba2k_folder_id,,
                     'ICONFILE=' || my_new_icon ||,
                     ';EXENAME=' || my_new_file ||,
                     ';PROGTYPE=WINDOWABLEVIO;MINIMIZED=YES' ||,
                     ';NOAUTOCLOSE=NO' ||,
                     ';OBJECTID=' || my_id || ';',,
                     'U') then do
      say '   Program object created in' ba2k_name 'folder.'
   end /* if */
   else do
      say 'Could not create program object. Aborting...'
      retc = 1
      signal Exit_Now
   end /* else */

   /* Create a shadow of my program object in Startup folder */
   if SysCreateShadow(my_id,startup_id) then do
      say '   Shadow of program object created in Startup folder.'
   end /* if */
   else do
      say 'Could not create shadow in Startup folder. Aborting...'
   end /* else */


   /* Delete ourself and our icon file from current location */
   if translate(my_path) \= translate(ba2k_dir) then do
      call SysFileDelete my_file
      call SysFileDelete my_icon
      say '   Deleted myself and my icon file from' my_path || '.'
   end /* if */


   /* Tell the user how great we are */
   say '...installation completed successfully.',
       'I will make a test run:'

   return 0




/* Function to do suizide */
Self_Uninstall:
   say 'You ordered me to do suizide...'

   /* Destroy program object */
   if SysDestroyObject(my_id) then do
      say '   Destroyed my program object.'
   end /* if */
   else do
      say 'Could not destroy my program object. Aborting...'
      retc = 1
      signal Exit_Now
   end /* else */


   /* Get our own file location */
   parse source . . my_file
   my_icon = left(my_file,lastpos('.',my_file)) || 'ico'
   my_path = strip(left(my_file,lastpos('\',my_file)),'t','\')

   /* Delete ourself and our icon file from current location */
   call SysFileDelete my_file
   call SysFileDelete my_icon
   say '   Deleted myself and my icon file from' my_path || '.'

   /* Say good bye to it all */
   say 'Suizide completed. Good bye and have a nice life...'
   say '                                                   ..'
   say '                                                     .'
   say '                                                      .'
   say '                                                      .'
   say '                                                      .'
