/*
 * Copyright 2009 Vasilkin Andrew <digi@os2.snc.ru>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice,
 *       this list of conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *
 *    3. The name of the author may not be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

//#define INCL_LONGLONG
#include <ndfs.h>
#define NDPL_LARGEFILES
#include <ndextpl2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h> 
#include <lqlib.h>
#include <debug.h>

#define LQ_PIPE_NAME		"\\PIPE\\LQ"
#define EA_UNIT_ALLOC		"LQ.BYTEALLOC"

#define RESFL_CHANGED		1
#define RESFL_DIRTY		2
#define RESFL_DEAD		8

#define OPEN_EXISTING	OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS
#define OPEN_REPLACE	OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS
#define OPEN_CREATE	OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS

#define FLUSH_TIMEOUT		10000
//#define FIND_BUF_MAX_ITEMS	24
#define FIND_BUF_MAX_ITEMS	1

typedef struct _SCANDIR	*PSCANDIR;
typedef struct _SCANDIR {
  HDIR			hDir;
  ULONG			ulCount;
  UCHAR			ucFindPath[CCHMAXPATH];
  FILEFINDBUF3L		sFindBuf[FIND_BUF_MAX_ITEMS];
} SCANDIR;

typedef struct _LQRES
{
  struct _LQRES		*psNext;
  struct _LQRES		**ppsSelf;

  PSZ			pszPath;
  ULONG			cbPath;

  ULONG			ulFileSystemId;
  ULONG			ulSectorUnit;
  USHORT		ulByteSector;

  LLONG			llByteLimit;
  LLONG			llByteAlloc;

  ULONG			ulMountCount;
  ULONG			ulFlags;
  HMTX			hmtxLock;
  ULONG			ulChangeTimestamp;

  LONG			lOpenFilesCount;
  LONG			lConnectionsCount;
} LQRES, *PLQRES;

typedef struct _CONNECTION {
  PLQRES		psRes;
} CONNECTION, *PCONNECTION;

const NDPROPERTYINFO asProperties[] =
{
  { ND_PROP_STRING, 0, "PATH",		""   },
  { ND_PROP_STRING, 0, "LIMIT",		""   },
  { ND_PROP_STRING, 0, NULL,		NULL },
};

const char *NdpTypes[] = { "LQ", NULL };
const NDPROPERTYINFO *NdpPropertiesInfo[] = { asProperties };
const unsigned long NdpAttribute = NDPA_REUSE_FILE_CONNECTIONS;

static PLUGINHELPERTABLE2	*ph = NULL;
static HMTX			hmtxResList;
static PLQRES			psResList = NULL;
static PLQRES			*ppsResLast = &psResList;
static TID			tidFlush;
static HEV			hevStop;
static HEV			hevScan;
static HMUX			hmuxThreadEvents;
static LQPIPE			sLQPipe;

VOID _System FlushThread(ULONG ulParam);

static BOOL _lqGetFlag(PLQRES psRes, ULONG ulFlag)
{
  APIRET	rc;
  BOOL		fResult;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
    fResult = FALSE;
  }
  else
  {
    fResult = (psRes->ulFlags & ulFlag) != 0;

    DosReleaseMutexSem( psRes->hmtxLock );
  }

  return fResult;
}

static PCHAR _lqGetLocalPath(PLQRES psRes, PCHAR pcBuf, PSZ pszPath)
{
  APIRET	rc;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
  }
  else
  {
    sprintf( pcBuf, "%s\\%s", psRes->pszPath, pszPath );
    DosReleaseMutexSem( psRes->hmtxLock );
  }

  return pcBuf;
}

static LLONG _lqGetBytesFree(PLQRES psRes)
{
  LLONG		llByteFree;
  APIRET	rc;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
    llByteFree = 0;
  }
  else
  {
    llByteFree = psRes->llByteLimit - psRes->llByteAlloc;

    DosReleaseMutexSem( psRes->hmtxLock );
  }

  return llByteFree;
}

static VOID _lqIncBytesAlloc(PLQRES psRes, LLONG llByteDelta)
{
  APIRET	rc;

  if ( llByteDelta == 0 )
    return;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
  }
  else
  {
    psRes->llByteAlloc += llByteDelta;
    if ( psRes->llByteAlloc < 0 )
      psRes->llByteAlloc = 0;

    if ( (psRes->ulFlags & RESFL_CHANGED) == 0 )
    {
      rc = lqlibEASetLLong( psRes->pszPath, EA_UNIT_ALLOC, (LLONG)(-1) );

      if ( rc != NO_ERROR )
        debug( "Cannot set EA (rc=%u): \"dirty\" flag for %s", rc,
               psRes->pszPath );

      psRes->ulFlags |= RESFL_CHANGED;
    }

    DosQuerySysInfo( QSV_MS_COUNT, QSV_MS_COUNT, &psRes->ulChangeTimestamp,
                     sizeof(ULONG) );

    DosReleaseMutexSem( psRes->hmtxLock );
  }
}

static int _lqOpenFile(PLQRES psRes, PSZ pszName, PHFILE phFile,
                       LONGLONG llSize, ULONG ulAttribute,
                       ULONG ulOpenFlags, ULONG ulOpenMode)
{
  APIRET	rc;
  CHAR		acBuf[CCHMAXPATH];
  ULONG		ulAction;
  FILESTATUS3L	sStat;
  LLONG		llFileByteAllocOld;
  LLONG		llByteFree;
  BOOL		fWriteMode;

  _lqGetLocalPath( psRes, &acBuf, pszName );

  fWriteMode = ( ulOpenMode & (OPEN_ACCESS_WRITEONLY | OPEN_ACCESS_READWRITE) ) != 0;

  if (
       fWriteMode ||
       ( ( ulOpenFlags &
           (OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS) ) != 0 )
     )
  {
    if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
    {
//dbg      debug( "Resource %s is \"dirty\"", psRes->pszPath );
      return ERROR_ACCESS_DENIED;
    }

    llByteFree = _lqGetBytesFree( psRes );

    rc = DosQueryPathInfo( &acBuf, FIL_STANDARDL, &sStat,
                           sizeof(FILESTATUS3L) );

    if ( rc != NO_ERROR )
    {
      if ( rc == ERROR_FILE_NOT_FOUND )
      {
        if ( llByteFree <= 0 )
        {
//dbg          debug( "ERROR_DISK_FULL @ %s", pszName );
          return ERROR_DISK_FULL;
        }
        llFileByteAllocOld = 0;
      }
      else
        return rc;
    }
    else
      llFileByteAllocOld = *(PLLONG)(&sStat.cbFileAlloc);

    if ( *(PLLONG)&llSize > llByteFree )
      *(PLLONG)&llSize = llByteFree > 0 ? llByteFree : 0;
  }

  rc = DosOpenL( &acBuf, phFile, &ulAction, llSize, ulAttribute,
                 ulOpenFlags, ulOpenMode, NULL );

  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosOpenL(%s), rc=%d", &acBuf, rc );
  }
  else
  {
//dbg    debug( "File %s opened, handle=%u%s", &acBuf, *phFile,
//dbg      fWriteMode ? " (WRITE)" : "" );

    if ( fWriteMode )
    {
      rc = DosQueryFileInfo( *phFile, FIL_STANDARDL, &sStat,
                             sizeof(FILESTATUS3L) );

      if ( rc != NO_ERROR )
      {
//dbg        debug( "DosQueryFileInfo(), rc=%d", rc );
        DosClose( *phFile );
        return rc;
      }

      _lqIncBytesAlloc( psRes,
                        *(PLLONG)&sStat.cbFileAlloc - llFileByteAllocOld );
    }

    psRes->lOpenFilesCount++;
  }

  return rc;
}

static VOID _lqStoreByteAlloc(PLQRES psRes)
{
  APIRET	rc;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
  }
  else
  {
    rc = lqlibEASetLLong( psRes->pszPath, EA_UNIT_ALLOC, psRes->llByteAlloc );
    if ( rc != NO_ERROR )
      debug( "Cannot set EA (rc=%d): %s", rc, psRes->pszPath );

    psRes->ulFlags &= ~RESFL_CHANGED;

    DosReleaseMutexSem( psRes->hmtxLock );
  }
}

static APIRET _lqCalcByteAlloc(PSZ pszPath, PLQRES psRes, BOOL fStart)
{
  PSCANDIR		psScanDir;
  APIRET		rc, rc2;
  PFILEFINDBUF3L	psFindItem;

  if ( (psRes->ulFlags & RESFL_DEAD) != 0 )
    return ERROR_INTERRUPT;

  psScanDir = malloc( sizeof(SCANDIR) );
  if ( psScanDir == NULL )
  {
    debug( "Not enought memory" );
    return rc;
  }
/*
  rc = DosAllocMem( (PVOID *)&psScanDir, sizeof(SCANDIR), fPERM | PAG_COMMIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosAllocMem(), rc=%u", rc );
    return rc;
  }
*/

  if ( fStart )
    psRes->llByteAlloc = 0;

  psScanDir->ulCount	= FIND_BUF_MAX_ITEMS;
  psScanDir->hDir	= HDIR_CREATE;

  sprintf( psScanDir->ucFindPath, "%s\\*", pszPath );
  rc = DosFindFirst( psScanDir->ucFindPath, &psScanDir->hDir,
                      FILE_ARCHIVED | FILE_DIRECTORY | FILE_SYSTEM |
                      FILE_HIDDEN | FILE_READONLY,
                      &psScanDir->sFindBuf,
                      FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L),
                      &psScanDir->ulCount, FIL_STANDARDL);

  while( rc == NO_ERROR )
  {
    for( psFindItem = &psScanDir->sFindBuf; ;
         psFindItem = (PFILEFINDBUF3L) ( ((PCHAR)psFindItem) + psFindItem->oNextEntryOffset )
       )
    {
      if ( (psFindItem->attrFile & FILE_DIRECTORY) )
      {
        if ( *(PUSHORT)&psFindItem->achName != 0x002E 
             && ( strcmp( psFindItem->achName, ".." ) != 0 ) )
        {
          sprintf( psScanDir->ucFindPath, "%s\\%s", pszPath,
                   psFindItem->achName );

          rc = _lqCalcByteAlloc( psScanDir->ucFindPath, psRes, FALSE );
          if ( rc != NO_ERROR )
            break;
        }
      }
      else
        psRes->llByteAlloc += *(PLLONG)&psFindItem->cbFileAlloc;

      if ( psFindItem->oNextEntryOffset == 0 )
        break;
    }

    if ( rc != NO_ERROR )
      break;

    psScanDir->ulCount = FIND_BUF_MAX_ITEMS;

    rc = DosFindNext( psScanDir->hDir, &psScanDir->sFindBuf,
                      FIND_BUF_MAX_ITEMS * sizeof(FILEFINDBUF3L),
                      &psScanDir->ulCount );
  }

  rc2 = DosFindClose( psScanDir->hDir );   
  if ( rc2 != NO_ERROR )
  {
    debug( "DosFindClose(), rc=%u", rc2 );
  }
  
  free( psScanDir );
/*
  rc2 = DosFreeMem( psScanDir );
  if ( rc2 != NO_ERROR )
  {
    debug( "DosFreeMem(), rc=%u", rc2 );
  }
*/

  return rc == ERROR_NO_MORE_FILES ? NO_ERROR : rc;
}

static VOID _lqAddRes(PLQRES psRes)
{
  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT );

  psRes->psNext = NULL;
  psRes->ppsSelf = ppsResLast;
  *ppsResLast = psRes;
  ppsResLast = &psRes->psNext;

  DosReleaseMutexSem( hmtxResList );
}

static VOID _lqRemoveRes(PLQRES psRes)
{
  psRes->ulFlags |= RESFL_DEAD;
  DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );

  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT );

  *psRes->ppsSelf = psRes->psNext;
  if ( psRes->psNext != NULL )
  {
    psRes->psNext->ppsSelf = psRes->ppsSelf;
    psRes->psNext = NULL;
  }
  else
    ppsResLast = psRes->ppsSelf;

  DosReleaseMutexSem( hmtxResList );
}

static PLQRES _lqFindRes(PSZ pszPath)
{
  PLQRES	psRes;

  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT );

  for( psRes = psResList; psRes != NULL; psRes = psRes->psNext )
  {
    if ( stricmp( psRes->pszPath, pszPath ) == 0 )
      break;
  }

  DosReleaseMutexSem( hmtxResList );

  return psRes;
}


BOOL fnPipeWriteStat(HPIPE hPipe, PCHAR pcBuf, BOOL fCompact)
{
  PLQRES	psRes;
  PSZ		pszFormat;
  ULONG		ulLength;

  if ( fCompact )
  {
    pszFormat = "\"%s\" %lld %lld %lld %s %s %d %d";
  }
  else
  {
    pszFormat =
      "Path: %s\n"\
      "Limit: %lld\n"\	
      "Allocated: %lld\n"\
      "Free: %lld\n"\
      "Flag \"changed\": %s\n"\
      "Flag \"dirty\": %s\n"\
      "Open files: %d\n"\
      "Connections: %d";
  }

  DosRequestMutexSem( hmtxResList, SEM_INDEFINITE_WAIT );

  if ( psResList == NULL )
  {
    DosReleaseMutexSem( hmtxResList );
    return FALSE;
  }

  for( psRes = psResList; psRes != NULL; psRes = psRes->psNext )
  {
    DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );

    ulLength =
      sprintf( pcBuf, pszFormat,
        psRes->pszPath, psRes->llByteLimit, psRes->llByteAlloc,
        psRes->llByteLimit - psRes->llByteAlloc,
        (psRes->ulFlags & RESFL_CHANGED) != 0 ? "ON" : "OFF",
        (psRes->ulFlags & RESFL_DIRTY) != 0 ? "ON" : "OFF",
        psRes->lOpenFilesCount, psRes->lConnectionsCount
      );

    DosReleaseMutexSem( psRes->hmtxLock );

    if ( !fCompact || ( psRes->psNext == NULL ) )
      strcpy( &pcBuf[ulLength], "\n" );

    lqlibPipeWriteMessage( hPipe, pcBuf );
  }

  DosReleaseMutexSem( hmtxResList );

  return TRUE;
}


VOID _System FlushThread(ULONG ulParam)
{
  APIRET	rc;
  PLQRES	psRes;
  BOOL		fFound;
  ULONG		ulNow;
  ULONG		ulEvSemId;

  while( TRUE )
  {
    rc = DosWaitMuxWaitSem( hmuxThreadEvents, 1000, &ulEvSemId );
    if ( rc == NO_ERROR  )
    {
      if ( ulEvSemId == 0 )
      {
        debug( "Stop event" );
        break;
      }

      DosResetEventSem( hevScan, &ulEvSemId );
    }
    else if ( rc != ERROR_TIMEOUT )
    {
      debug( "DosWaitMuxWaitSem(), rc=%u", rc );
      break;
    }

    DosQuerySysInfo( QSV_MS_COUNT, QSV_MS_COUNT, &ulNow, sizeof(ULONG) );

    fFound = FALSE;

    rc = DosRequestMutexSem( hmtxResList, 100 );
    if ( rc == ERROR_TIMEOUT  )
    {
      debug( "Resources list locking timed out" );
      continue;
    }

    for( psRes = psResList; psRes != NULL; psRes = psRes->psNext )
    {
      rc = DosRequestMutexSem( psRes->hmtxLock, SEM_IMMEDIATE_RETURN );

      if ( rc == NO_ERROR )
      {
        if ( (psRes->ulFlags & RESFL_DIRTY) != 0 )
        {
          fFound = TRUE;
          break;
        }

        if ( ( (psRes->ulFlags & RESFL_CHANGED) != 0 ) &&
             ( (ulNow - psRes->ulChangeTimestamp) > FLUSH_TIMEOUT ) )
        {
//dbg          debug( "Resource %s is \"dirty\" - store value, clear flag",
//dbg            psRes->pszPath );
          _lqStoreByteAlloc( psRes );
        }

        DosReleaseMutexSem( psRes->hmtxLock );
      }
      else
      {
        if ( rc != ERROR_TIMEOUT  )
          debug( "DosRequestMutexSem(), rc=%u", rc );
      }
    }

    DosReleaseMutexSem( hmtxResList );

    if ( fFound )
    {
//dbg      debug( "Calc allocated space for %s", psRes->pszPath );
      _lqCalcByteAlloc( psRes->pszPath, psRes, TRUE );
      _lqStoreByteAlloc( psRes );
      psRes->ulFlags &= ~RESFL_DIRTY;
      DosReleaseMutexSem( psRes->hmtxLock );
    }
  }
}


int APIENTRY NdpPluginLoad(PLUGINHELPERTABLE2 *pPHT2)
{
  SEMRECORD	asSemList[2];
  APIRET	rc;

  DebugInit( DEBUG_FILE );
  ph = pPHT2;

  if ( ph->cb < sizeof(PLUGINHELPERTABLE2) )
  {
    debug( "failed" );
    return ERROR_INVALID_FUNCTION;
  }

  DosCreateMutexSem( NULL, &hmtxResList, 0, FALSE );
  DosCreateEventSem( NULL, &hevStop, 0, FALSE );
  DosCreateEventSem( NULL, &hevScan, 0, FALSE );

  asSemList[0].hsemCur = (HSEM)hevStop;
  asSemList[0].ulUser = 0;
  asSemList[1].hsemCur = (HSEM)hevScan;
  asSemList[1].ulUser = 1;
  DosCreateMuxWaitSem( NULL, &hmuxThreadEvents, 2, &asSemList, DCMW_WAIT_ANY );

  rc = DosCreateThread( &tidFlush, &FlushThread, 0,
                        CREATE_READY | STACK_SPARSE, 2*65536 );
  if ( rc != NO_ERROR )
  {
    debug( "DosCreateThread(), rc=%u", rc );
  }

  rc = lqlibPipeServInit( &sLQPipe, LQ_PIPE_NAME, fnPipeWriteStat );

  if ( rc != NO_ERROR )
    debug( "lqlibPipeServInit(), rc=%u", rc );

  return NO_ERROR;
}

int APIENTRY NdpPluginFree()
{
  debug( "shutdown..." );

  if ( psResList != NULL )
    debug( "Not all of resources was unmounted!" );

  DosPostEventSem( hevStop );
  DosWaitThread( &tidFlush, DCWW_WAIT );

  lqlibPipeServDone( &sLQPipe );
    
  DosCloseMuxWaitSem( hmuxThreadEvents );
  DosCloseEventSem( hevStop );
  DosCloseEventSem( hevScan );
  DosCloseMutexSem( hmtxResList );
  debug( "Done" );
  DebugDone();

  return NO_ERROR;
}

int APIENTRY NdpMountResource(HRESOURCE *presource, int type,
                              NDPROPERTYHANDLE properties)
{
  char const	*pcPath;
  ULONG		cbPath;
  char const	*pcLimit;
  ULONG		cbLimit;
  LLONG		llLimit;
  ULONG		ulDiskNum;
  FILESTATUS3L	sStat;
  APIRET	rc;
  PLQRES	psRes;
  ULONG		ulByteUnit;
  FSALLOCATE	sFSAlloc;

  ph->fsphQueryStringProperty( properties, "path", &pcPath, &cbPath );
  ph->fsphQueryStringProperty( properties, "limit", &pcLimit, &cbLimit );
  
  ph->fsphString2longlong( pcLimit, NULL, &llLimit, 10 );
  llLimit *= 1024;

  psRes = _lqFindRes( (PSZ)pcPath );
  if ( psRes != NULL )
  {
    debug( "Path \"%s\" already in use, mount it @ %p", pcPath, psRes );
  }
  else
  {
//dbg    debug( "Mount path: \"%s\", limit: %lld", pcPath, llLimit );

    if ( ( cbPath < 4 ) || ( pcPath == NULL ) )
    {
      debug( "Invalid parameter: local path too short" );
      return ERROR_INVALID_PARAMETER;
    }

    ulDiskNum = pcPath[0];
    ulDiskNum = ulDiskNum >= 'a' ? ulDiskNum - 'a' : ulDiskNum - 'A';

    if ( ulDiskNum > ('z'-'a') || pcPath[1] != ':' )
    {
      debug( "Invalid parameter: invalid path" );
      return ERROR_INVALID_PARAMETER;
    }

    rc = DosQueryPathInfo( pcPath, FIL_STANDARDL,
                           &sStat, sizeof(FILESTATUS3L) );
    if ( ( rc != NO_ERROR ) || ( (sStat.attrFile & FILE_DIRECTORY) == 0 ) )
    {
      debug( "DosQueryPathInfo(%s), rc=%d, FILE_DIRECTORY = %d",
             pcPath, rc, sStat.attrFile & FILE_DIRECTORY != 0 );
      return ERROR_INVALID_PARAMETER;
    }

    psRes = calloc( 1, sizeof(LQRES) );
    if ( psRes == NULL )
    {
      debug( "Not enought memory" );
      return ERROR_NOT_ENOUGH_MEMORY;
    }

    psRes->pszPath = malloc( cbPath + 1 );
    if ( psRes->pszPath == NULL )
    {
      debug( "Not enought memory" );
      free( psRes );
      return ERROR_NOT_ENOUGH_MEMORY;
    }
    strcpy( psRes->pszPath, pcPath );
    psRes->cbPath = cbPath;

    rc = DosQueryFSInfo( ulDiskNum + 1, FSIL_ALLOC, &sFSAlloc,
                         sizeof(FSALLOCATE) );

    if ( rc != NO_ERROR )
    {
      debug( "DosQueryFSInfo(), rc=%d. Fake values are used.", rc );
      psRes->ulFileSystemId	= (ULONG)psRes;
      psRes->ulSectorUnit	= 4;
      psRes->ulByteSector	= 512;
    }
    else
    {
      psRes->ulFileSystemId	= sFSAlloc.idFileSystem;
      psRes->ulSectorUnit	= sFSAlloc.cSectorUnit;
      psRes->ulByteSector	= sFSAlloc.cbSector;
    }

    psRes->llByteAlloc = -1;
    rc = lqlibEAQueryLLong( (PSZ)pcPath, EA_UNIT_ALLOC, &psRes->llByteAlloc );

    if ( rc != NO_ERROR )
    {
      debug( "lqlibEAQueryLLong(), rc=%d", rc );
      return rc;
    }

    if ( psRes->llByteAlloc <= 0 )
    {
      if ( psRes->llByteAlloc < 0 )
      {
//dbg        debug( "Resource is \"dirty\"" );
        psRes->ulFlags |= RESFL_DIRTY;
        DosPostEventSem( hevScan );
      }
    }
    else
    {
//dbg      debug( "Allocated bytes counter loaded from EA: %lld",
//dbg             psRes->llByteAlloc );
    }

    DosCreateMutexSem( NULL, &psRes->hmtxLock, 0, FALSE );

//dbg    debug( "New resource %s @ %p", pcPath, psRes );
    _lqAddRes( psRes );
  }

  ulByteUnit = psRes->ulSectorUnit * psRes->ulByteSector;
  psRes->llByteLimit = llLimit < ulByteUnit ? ulByteUnit : llLimit;

  psRes->ulMountCount++;
  *(PLQRES *)presource = psRes;

  return NO_ERROR;
}

int APIENTRY NdpFreeResource(HRESOURCE resource)
{
  PLQRES	psRes = (PLQRES)resource;

  psRes->ulMountCount--;

  if ( psRes->ulMountCount == 0 )
  {
//dbg    debug( "Remove resource %s @ %p", psRes->pszPath, psRes );

    if ( _lqGetFlag( psRes, RESFL_CHANGED ) )
      _lqStoreByteAlloc( psRes );

    _lqRemoveRes( psRes );
    DosCloseMutexSem( psRes->hmtxLock );
    free( psRes->pszPath );
    free( psRes );
  }

  return NO_ERROR;
}

int APIENTRY NdpCreateConnection(HRESOURCE resource, HCONNECTION *pconn)
{
  PLQRES	psRes = (PLQRES)resource;
  PCONNECTION	psConnection;

  psConnection = (PCONNECTION)malloc( sizeof( CONNECTION ) );
  psConnection->psRes = psRes;
  psRes->lConnectionsCount++;

  *(PCONNECTION *)pconn = psConnection;

  return NO_ERROR;
}

int APIENTRY NdpFreeConnection(HCONNECTION conn)
{
  PCONNECTION	psConnection = (PCONNECTION)conn;

  if ( psConnection != NULL )
  {
    psConnection->psRes->lConnectionsCount--;
    free( psConnection );
  }

  return NO_ERROR;
}

int APIENTRY NdpRsrcCompare(HRESOURCE resource, HRESOURCE resource2)
{
  PLQRES	psRes = (PLQRES)resource;
  PLQRES	psRes2 = (PLQRES)resource2;
  BOOL		fEqual;

  fEqual = ( stricmp( psRes->pszPath, psRes2->pszPath ) == 0 );

  return fEqual ? ND_RSRC_EQUAL : ND_RSRC_DIFFERENT;
}

int APIENTRY NdpRsrcUpdate(HRESOURCE resource, HRESOURCE resource2)
{
//  PLQRES	psRes = (PLQRES)resource;

//dbg  debug( "Update resource %s @ %p", psRes->pszPath, psRes );

  return NO_ERROR;
}

int APIENTRY NdpRsrcQueryInfo(HRESOURCE resource, ULONG *pulFlags, void *pdata,
                              ULONG insize, ULONG *poutlen)
{
  PLQRES	psRes = (PLQRES)resource;
  int		rc;
  FILESTATUS3L	sStat;

  rc = DosQueryPathInfo( psRes->pszPath, FIL_STANDARDL,
                         &sStat, sizeof(FILESTATUS3L) );

  if ( rc == NO_ERROR )
  {
    *poutlen = psRes->cbPath + 1;

    if ( *poutlen > insize )
    {
      debug( "Not enought buffer space" );
      rc = ERROR_BUFFER_OVERFLOW;
    }
    else
    {
      memcpy( pdata, psRes->pszPath, *poutlen );
      rc = NO_ERROR;
    }

//    if ( (sStat.attrFile & FILE_DIRECTORY) != 0 )
      *pulFlags = ND_RSRC_AVAILABLE;
  }
  else
  {
//dbg    debug( "Resource '%s' is not available", psRes->pszPath );
  }

  return rc;
}

int APIENTRY NdpRsrcQueryFSAttach(HRESOURCE resource, void *pdata, ULONG insize,
                                  ULONG *poutlen)
{
  ULONG		ulFlags;

  return NdpRsrcQueryInfo( resource, &ulFlags, pdata, insize, poutlen );
}

int APIENTRY NdpRsrcQueryFSAllocate(HRESOURCE resource, NDFSALLOCATE *pfsa)
{
  PLQRES	psRes = (PLQRES)resource;
  ULONG		ulByteUnit = psRes->ulSectorUnit * psRes->ulByteSector;
  ULONG		ulUnitAlloc = psRes->llByteAlloc / ulByteUnit;

  pfsa->cSectorUnit		= psRes->ulSectorUnit;
  pfsa->cUnit 			= psRes->llByteLimit / ulByteUnit;
  pfsa->cUnitAvail		= ( pfsa->cUnit > ulUnitAlloc ) ?
                                    ( pfsa->cUnit - ulUnitAlloc ) : 0;
  pfsa->cbSector		= psRes->ulByteSector;

  return NO_ERROR;
}


int APIENTRY NdpRefresh(HCONNECTION conn, char *path, int tree)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  APIRET	rc;

  rc = DosRequestMutexSem( psRes->hmtxLock, SEM_INDEFINITE_WAIT );
  if ( rc != NO_ERROR )
  {
    debug( "DosRequestMutexSem(), rc = %u", rc );
  }
  else
  {
    if ( (psRes->ulFlags & RESFL_DIRTY) != 0 )
    {
//dbg      debug( "resource %s already \"drity\"", psRes->pszPath );
    }
    else
    {
//dbg      debug( "start refresh (%s,%d): %s", path, tree, psRes->pszPath );
      psRes->ulFlags |= RESFL_DIRTY;
      DosPostEventSem( hevScan );
    }

    DosReleaseMutexSem( psRes->hmtxLock );
  }

  return NO_ERROR;
}

int APIENTRY NdpQueryPathInfo(HCONNECTION conn, void *plist, char *szPath)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  APIRET	rc;
  CHAR		acBuf[CCHMAXPATH];
  FILESTATUS3L	sStat;
  PCHAR		pcParentPath = "";
  ULONG		ulLength = 0;
  NDPATHELEMENT	*psPathElement = ph->fsphNameElem( 0 );

  if ( !ph->fsphIsLastElem( 0, NULL ) )
  {
    ulLength = strlen( szPath ) - psPathElement->length - 1;
    pcParentPath = (PCHAR)alloca( ulLength + 1 + 1 );

    memcpy( pcParentPath, szPath, ulLength );
    pcParentPath[ulLength] = '\0';
//dbg    debug( "parent path: %s", pcParentPath );
  }

  if ( ph->fsphStrChr( psPathElement->name, '*' ) ||
       ph->fsphStrChr( psPathElement->name, '?' ) )
  {
    if ( pcParentPath[0] == '\0' )
    {
      rc = ERROR_FILE_NOT_FOUND;
    }
    else
    {
      rc = DosQueryPathInfo( _lqGetLocalPath( psRes, &acBuf, pcParentPath ),
                             FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );

      if ( ( rc != NO_ERROR ) || ( (sStat.attrFile & FILE_DIRECTORY) == 0 ) )
      {
        rc = ERROR_PATH_NOT_FOUND;
      }
      else
      {
        rc = ERROR_FILE_NOT_FOUND;
      }
    }
  }
  else
  {
    rc = DosQueryPathInfo( _lqGetLocalPath( psRes, &acBuf, szPath ),
                           FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );

    if ( rc == NO_ERROR )
    {
      ph->fsphAddFile32L( plist, &sStat, psPathElement->name,
                          psPathElement->length, NULL, 0, 0 );
    }
    else
    {
      if ( pcParentPath[0] == '\0' )
      {
        rc = ERROR_FILE_NOT_FOUND;
      }
      else
      {
        rc = DosQueryPathInfo( _lqGetLocalPath( psRes, &acBuf, pcParentPath ),
                               FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );

        if ( ( rc != NO_ERROR ) || ( (sStat.attrFile & FILE_DIRECTORY) == 0 ) )
        {
          rc = ERROR_PATH_NOT_FOUND;
        }
        else
        {
          rc = ERROR_FILE_NOT_FOUND;
        }
      }
    }
  }

  if ( rc == ERROR_PATH_NOT_FOUND )
  {
//dbg    debug( "ERROR_PATH_NOT_FOUND @ %s", szPath );
  }
  else if ( rc == ERROR_FILE_NOT_FOUND )
  {
//dbg    debug( "ERROR_FILE_NOT_FOUND @ %s", szPath );
  }
  else if ( rc != NO_ERROR )
  {
//dbg    debug( "rc=%u @ %s", rc, szPath );
  }

  return rc;
}

int APIENTRY NdpDeletePathInfo(HRESOURCE resource, NDFILEINFO *pfi)
{
  return NO_ERROR;
}

int APIENTRY NdpDiscardResourceData(HRESOURCE resource, NDDATABUF *pdatabuf)
{
  return NO_ERROR;
}

int APIENTRY NdpSetPathInfo(HCONNECTION conn, NDFILEINFO *pfi, char *szPathName)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acBuf[CCHMAXPATH];
  APIRET	rc;

  rc = DosSetPathInfo( _lqGetLocalPath( psRes, &acBuf, szPathName ),
                       FIL_STANDARDL, &pfi->stat,
                       sizeof(FILESTATUS3L), DSPI_WRTTHRU );
  if ( rc != NO_ERROR )
  {
    debug( "DosSetPathInfo(%s), rc = %u", &acBuf, rc );
  }

  return rc;
}

int APIENTRY NdpSetCurrentDir(HCONNECTION conn, NDFILEINFO *pfi, char *szPath)
{
//dbg  debug( "Enter" );
  return NO_ERROR;
}

int APIENTRY NdpCopy(HCONNECTION conn, NDFILEINFO *pfiDst, char *szDst,
                     NDFILEINFO *pfiSrc, char *szSrc, ULONG ulOption)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  FILESTATUS3L	sStat;
  APIRET	rc;
  CHAR		acDst[CCHMAXPATH];
  CHAR		acSrc[CCHMAXPATH];

  if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psRes->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  _lqGetLocalPath( psRes, &acSrc, szSrc );

  rc = DosQueryPathInfo( &acSrc, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryPathInfo(%s), rc=%d", &acSrc, rc );
    return rc;
  }

  if ( *(PLLONG)&sStat.cbFileAlloc > _lqGetBytesFree( psRes ) )
    return ERROR_DISK_FULL;

  rc = DosCopy( &acSrc, _lqGetLocalPath( psRes, &acDst, szDst ), ulOption );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosCopy(%s,%s,0x%x), rc = %u", &acSrc, &acDst, ulOption, rc );
  }
  else
  {
    DosQueryPathInfo( acDst, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
    _lqIncBytesAlloc( psRes, *(PLLONG)&sStat.cbFileAlloc );
  }

  return rc;
}

int APIENTRY NdpCopy2(HCONNECTION conn, HRESOURCE resDst, NDFILEINFO *pfiDst,
                      char *szDst, NDFILEINFO *pfiSrc, char *szSrc,
                      ULONG ulOption)
{
  PLQRES	psResSrc = ((PCONNECTION)conn)->psRes;
  PLQRES	psResDst = (PLQRES)resDst;
  FILESTATUS3L	sStat;
  APIRET	rc;
  CHAR		acDst[CCHMAXPATH];
  CHAR		acSrc[CCHMAXPATH];

  if ( _lqGetFlag( psResDst, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psResDst->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  _lqGetLocalPath( psResSrc, &acSrc, szSrc );

  rc = DosQueryPathInfo( &acSrc, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryPathInfo(%s), rc=%d", &acSrc, rc );
    return rc;
  }

  if ( *(PLLONG)&sStat.cbFileAlloc > _lqGetBytesFree( psResDst ) )
  {
//dbg    debug( "ERROR_DISK_FULL @ %s", szDst );
    return ERROR_DISK_FULL;
  }

  rc = DosCopy( &acSrc, _lqGetLocalPath( psResDst, &acDst, szDst ), ulOption );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosCopy(%s,%s,0x%x), rc = %u", &acSrc, &acDst, ulOption, rc );
  }
  else
  {
    DosQueryPathInfo( acDst, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
    _lqIncBytesAlloc( psResDst, *(PLLONG)&sStat.cbFileAlloc );
  }

  return rc;
}

int APIENTRY NdpForceDelete(HCONNECTION conn, NDFILEINFO *pfi, char *szFile)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  FILESTATUS3L	sStat;
  UCHAR		acBuf[CCHMAXPATH];
  APIRET	rc;

  if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psRes->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  _lqGetLocalPath( psRes, &acBuf, szFile );

  rc = DosQueryPathInfo( &acBuf, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryPathInfo(%s), rc=%d", &acBuf, rc );
    return ERROR_ACCESS_DENIED;
  }

  rc = DosForceDelete( &acBuf );

  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosForceDelete(%s), rc=%d", &acBuf, rc );
  }
  else
  {
//dbg    debug( "Deleted: %s, (free %lld bytes)", &acBuf,
//dbg           *(PLLONG)&sStat.cbFileAlloc );
    _lqIncBytesAlloc( psRes, -(*(PLLONG)&sStat.cbFileAlloc) );
  }

  return rc;
}

int APIENTRY NdpCreateDir(HCONNECTION conn, NDFILEINFO *pfiparent,
                          char *szDirName, FEALIST *pFEAList)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  UCHAR		acBuf[CCHMAXPATH];
  APIRET	rc;

  rc = DosCreateDir( _lqGetLocalPath( psRes, &acBuf, szDirName ), NULL );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosCreateDir(%s), rc=%d", &acBuf, rc );
  }

  return rc;
}

int APIENTRY NdpFindStart(HCONNECTION conn, void *plist, NDFILEINFO *pfiparent,
                          char *szPath, ULONG ulAttribute)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acBuf[CCHMAXPATH];
  HDIR		hDir = HDIR_CREATE;
  FILEFINDBUF3L	sFind;
  ULONG		ulFileNames;
  APIRET	rc;

  _lqGetLocalPath( psRes, &acBuf, szPath );
  ulFileNames = 1;
  rc = DosFindFirst( &acBuf, &hDir,
                     FILE_ARCHIVED | FILE_DIRECTORY |
                     FILE_SYSTEM | FILE_HIDDEN | FILE_READONLY,
                     &sFind, sizeof(FILEFINDBUF3L),
                     &ulFileNames, FIL_STANDARDL );

  while ( rc == NO_ERROR )
  {
    if ( ph->fsphAttrMatch( ulAttribute, sFind.attrFile ) )
      ph->fsphAddFile32L( plist, (FILESTATUS3L *)&sFind.fdateCreation,
                          sFind.achName, sFind.cchName, NULL, 0, 0 );

    ulFileNames = 1;
    rc = DosFindNext( hDir, &sFind, sizeof(FILEFINDBUF3L), &ulFileNames );
  }

  DosFindClose( hDir );

  if ( rc == ERROR_NO_MORE_FILES )
    return NO_ERROR;

//dbg  debug( "%s, rc=%u", &acBuf, rc );

  return rc;
}

int APIENTRY NdpMove(HCONNECTION conn, NDFILEINFO *pfiDst, char *szDst,
                     NDFILEINFO *pfiSrc, char *szSrc)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acDst[CCHMAXPATH];
  CHAR		acSrc[CCHMAXPATH];
  APIRET	rc;

  rc = DosMove( _lqGetLocalPath( psRes, &acSrc, szSrc ),
                _lqGetLocalPath( psRes, &acDst, szDst ) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosMove(%s,%s), rc=%u", &acSrc, &acDst, rc );
  }

  return rc;
}

int APIENTRY NdpMove2(HCONNECTION conn, HRESOURCE resDst, NDFILEINFO *pfiDst,
                      char *szDst, NDFILEINFO *pfiSrc, char *szSrc)
{
  PLQRES	psResSrc = ((PCONNECTION)conn)->psRes;
  PLQRES	psResDst = (PLQRES)resDst;
  FILESTATUS3L	sStat;
  CHAR		acDst[CCHMAXPATH];
  CHAR		acSrc[CCHMAXPATH];
  APIRET	rc;

  if ( _lqGetFlag( psResSrc, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psResSrc->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  if ( _lqGetFlag( psResDst, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psResDst->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  _lqGetLocalPath( psResSrc, &acSrc, szSrc );

  rc = DosQueryPathInfo( &acSrc, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryPathInfo(%s), rc=%d", &acSrc, rc );
    return rc;
  }

  if ( *(PLLONG)&sStat.cbFileAlloc > _lqGetBytesFree( psResDst ) )
  {
//dbg    debug( "ERROR_DISK_FULL @ %s", szDst );
    return ERROR_DISK_FULL;
  }

  rc = DosMove( &acSrc, _lqGetLocalPath( psResDst, &acDst, szDst ) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosMove(%s,%s), rc=%d", &acSrc, &acDst, rc );
  }
  else
  {
    _lqIncBytesAlloc( psResSrc, -(*(PLLONG)&sStat.cbFileAlloc) );
    DosQueryPathInfo( &acDst, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
    _lqIncBytesAlloc( psResDst, *(PLLONG)&sStat.cbFileAlloc );
  }

  return rc;
}

int APIENTRY NdpChangeCase(HCONNECTION conn, char *szDst, NDFILEINFO *pfiSrc,
                           char *szSrc, char *szNewName, ULONG ulNameLen)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acDst[CCHMAXPATH];
  CHAR		acSrc[CCHMAXPATH];
  APIRET	rc;

  if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psRes->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  rc = DosMove( _lqGetLocalPath( psRes, &acSrc, szSrc ),
                _lqGetLocalPath( psRes, &acDst, szDst ) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosMove(%s,%s), rc=%u", &acSrc, &acDst, rc );
  }

  return rc;
}

int APIENTRY NdpRename(HCONNECTION conn, char *szDst, NDFILEINFO *pfiSrc,
                       char *szSrc, char *szNewName, ULONG ulNameLen)
{
  return NdpChangeCase( conn, szDst, pfiSrc, szSrc, szNewName, ulNameLen );
}

int APIENTRY NdpOpenExisting(HCONNECTION conn, NDFILEINFO *pfi,
                             NDFILEHANDLE *phandle, char *szFileName,
                             ULONG ulOpenMode, USHORT *pfNeedEA)
{
  APIRET	rc;
  LONGLONG	llSize = { 0 };

  rc = _lqOpenFile( ((PCONNECTION)conn)->psRes, szFileName, (PHFILE)phandle, llSize,
                    FILE_NORMAL, OPEN_EXISTING, ulOpenMode & 0xFFFF );

  return rc;
}

int APIENTRY NdpOpenReplaceL(HCONNECTION conn, NDFILEINFO *pfi,
                             NDFILEHANDLE *phandle, char *szFileName,
                             LONGLONG llSize, ULONG ulOpenMode,
                             ULONG ulAttribute, FEALIST *pFEAList)
{
  APIRET	rc;

  rc = _lqOpenFile( ((PCONNECTION)conn)->psRes, szFileName, (PHFILE)phandle, llSize,
                    ulAttribute, OPEN_REPLACE, ulOpenMode & 0xFFFF );

  return rc;
}

int APIENTRY NdpOpenCreateL(HCONNECTION conn, NDFILEINFO *pfiparent,
                            NDFILEHANDLE *phandle, char *szFileName,
                            LONGLONG llSize, ULONG ulOpenMode,
                            ULONG ulAttribute, FEALIST *pFEAList)
{
  APIRET	rc;

  rc = _lqOpenFile( ((PCONNECTION)conn)->psRes, szFileName, (PHFILE)phandle, llSize,
                    ulAttribute, OPEN_CREATE, ulOpenMode & 0xFFFF );

  return rc;
}

int APIENTRY NdpDeleteDir(HCONNECTION conn, NDFILEINFO *pfi, char *szDir)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acBuf[CCHMAXPATH];
  APIRET	rc;

  rc = DosDeleteDir( _lqGetLocalPath( psRes, &acBuf, szDir ) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosDeleteDir(%s), rc=%u", &acBuf, rc );
  }

  return rc;
}

int APIENTRY NdpSetFileAttribute(HCONNECTION conn, NDFILEINFO *pfi,
                                 char *szFileName, USHORT usAttr)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  CHAR		acBuf[CCHMAXPATH];
  HFILE		hFile;
  ULONG		ulAction;
  APIRET	rc;
  LONGLONG	llSize = {0};
  FILESTATUS3L	sStat;

  rc = DosOpenL( _lqGetLocalPath( psRes, &acBuf, szFileName ),
                 &hFile, &ulAction, llSize, FILE_NORMAL,
                 OPEN_EXISTING, OPEN_FLAGS_FAIL_ON_ERROR | OPEN_SHARE_DENYNONE
                 | OPEN_ACCESS_WRITEONLY, NULL );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosOpenL(%s), rc=%d", &acBuf, rc );
  }
  else
  {
    rc = DosQueryFileInfo( hFile, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
    if ( rc != NO_ERROR )
    {
//dbg      debug( "DosQueryFileInfo(), rc=%d", rc );
    }
    else
    {
      sStat.attrFile = usAttr;
      rc = DosSetFileInfo( hFile, FIL_STANDARDL, &sStat, sizeof(FILESTATUS3L) );
      if ( rc != NO_ERROR )
      {
//dbg        debug( "DosSetFileInfo(), rc=%d", rc );
      }
    }
    DosClose( hFile );
  }

  return rc;
}

int APIENTRY NdpFlush(HRESOURCE resource)
{
  PLQRES	psRes = (PLQRES)resource;

  if ( (psRes->ulFlags & RESFL_CHANGED) == 0 )
    return NO_ERROR;

  _lqStoreByteAlloc( psRes );

  return NO_ERROR;
}

//#pragma off(unreferenced)
int APIENTRY NdpIOCTL(int type, HRESOURCE resource, char *path, int function,
                       void *in, ULONG insize, PULONG poutlen)
//#pragma on(unreferenced)
{
//  return ERROR_NOT_SUPPORTED;
//dbg  debug( "type: %d, path: %s, function: %s, in: %s", type, path, function, in );
  return NO_ERROR;
}

int APIENTRY NdpFileQueryInfo(HCONNECTION conn, NDFILEHANDLE handle,
                              void *plist)
{
  APIRET	rc;
  FILESTATUS3L	sStat;

  rc = DosQueryFileInfo( (HFILE)handle, FIL_STANDARDL, &sStat,
                         sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryFileInfo(%u), rc=%u", handle, rc );
  }
  else
    rc = ph->fsphAddFile32L (plist, &sStat, NULL, 0, NULL, 0, 0);

  return rc;
}

int APIENTRY NdpFileEAQuery(HCONNECTION conn, NDFILEHANDLE handle, GEALIST *pGEAList, FEALIST *pFEAList)
{
  return ERROR_NOT_SUPPORTED;
}

int APIENTRY NdpFileEASet(HCONNECTION conn, NDFILEHANDLE handle, FEALIST *pFEAList)
{
  return ERROR_NOT_SUPPORTED;
}

int APIENTRY NdpFileSetInfo(HCONNECTION conn, NDFILEHANDLE handle, NDFILEINFO *pfi)
{
  APIRET	rc;

  rc = DosSetFileInfo( (HFILE)handle, FIL_STANDARDL, &pfi->stat,
                       sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosSetFileInfo(%u), rc=%u", handle, rc );
  }

  return rc;
}

int APIENTRY NdpFileSetFilePtrL(HCONNECTION conn, NDFILEHANDLE handle,
                                LONGLONG llOffset, ULONG ulMethod,
                                LONGLONG *pllActual)
{
  APIRET	rc;

  rc = DosSetFilePtrL( (HFILE)handle, llOffset, ulMethod, pllActual );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosSetFilePtrL(%u), rc=%u", handle, rc );
  }
  else
  {
//dbg    debug( "The pointer moved: %lld bytes from the %s, file handle=%u",
//dbg           /**(PLLONG)&*/llOffset,
//dbg           ulMethod == FILE_BEGIN ? "beginning of the file" :
//dbg             ulMethod == FILE_CURRENT ? "current location" : "end of the file",
//dbg           handle );
  }

  return rc;
}

int APIENTRY NdpFileClose(HCONNECTION conn, NDFILEHANDLE handle)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  APIRET	rc;

  rc = DosClose( (HFILE)handle );
  if ( rc != NO_ERROR )
  {
    debug( "DosClose(%u), rc=%u", handle, rc );
  }
  else
  {
//dbg    debug( "File closed, handle=%u", handle );
    psRes->lOpenFilesCount--;
  }

  return rc;
}

int APIENTRY NdpFileCommit(HCONNECTION conn, NDFILEHANDLE handle)
{
  return NO_ERROR;
}

int APIENTRY NdpFileNewSizeL(HCONNECTION conn, NDFILEHANDLE handle,
                             LONGLONG llLen)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  FILESTATUS3L	sStat;
  APIRET	rc;
  LLONG		llFileByteAllocOld;
  LLONG		llByteFree;

  if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psRes->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  rc = DosQueryFileInfo( (HFILE)handle, FIL_STANDARDL, &sStat,
                         sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryFileInfo(), rc=%d", rc );
    return rc;
  }
  llFileByteAllocOld = *(PLLONG)&sStat.cbFileAlloc;

  if ( *(PLLONG)&llLen > llFileByteAllocOld )
  {
    llByteFree = _lqGetBytesFree( psRes );

    if ( llByteFree <= 0 )
    {
//dbg      debug( "ERROR_DISK_FULL @ handle=%u", handle );
      return ERROR_DISK_FULL;
    }

    if ( llByteFree < ( *(PLLONG)&llLen - *(PLLONG)&sStat.cbFile ) )
      *(PLLONG)&llLen = ( *(PLLONG)&sStat.cbFile + llByteFree );
  }

  rc = DosSetFileSizeL( (HFILE)handle, llLen );

  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosSetFileSizeL(%u,%lld), rc=%u", handle, llLen, rc );
  }
  else
  {
    rc = DosQueryFileInfo( (HFILE)handle, FIL_STANDARDL, &sStat,
                           sizeof(FILESTATUS3L) );

    if ( rc != NO_ERROR )
    {
//dbg      debug( "DosQueryFileInfo(%u), rc=%d", handle, rc );
      return NO_ERROR;
    }

    _lqIncBytesAlloc( psRes,
                      *(PLLONG)&sStat.cbFileAlloc - llFileByteAllocOld );
  }

  return rc;
}

int APIENTRY NdpFileRead(HCONNECTION conn, NDFILEHANDLE handle, void *pBuffer,
                         ULONG ulRead, ULONG *pulActual)
{
  APIRET	rc;

  rc = DosRead( (HFILE)handle, pBuffer, ulRead, pulActual );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosRead(%u), rc=%d", handle, rc );
  }

  return rc;
}

int APIENTRY NdpFileWrite(HCONNECTION conn, NDFILEHANDLE handle, void *pBuffer,
                          ULONG ulWrite, ULONG *pulActual)
{
  PLQRES	psRes = ((PCONNECTION)conn)->psRes;
  FILESTATUS3L	sStat;
  APIRET	rc;
  LLONG		llByteFree;
  LLONG		llFileByteAlloc;
  LONGLONG	llFileByteSizeOld;

  if ( _lqGetFlag( psRes, RESFL_DIRTY ) )
  {
//dbg    debug( "Resource (%s) is \"dirty\"", psRes->pszPath );
    return ERROR_ACCESS_DENIED;
  }

  llByteFree = _lqGetBytesFree( psRes );
  if ( llByteFree <= 0 )
  {
//dbg    debug( "ERROR_DISK_FULL @ handle=%u", handle );
    return ERROR_DISK_FULL;
  }

  if ( llByteFree < ulWrite )
  {
//dbg    debug( "%u bytes requested to write, reduce to %llu bytes",
//dbg           ulWrite, llByteFree );
    ulWrite = llByteFree;
  }

  rc = DosQueryFileInfo( (HFILE)handle, FIL_STANDARDL, &sStat,
                         sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosQueryFileInfo(%u), rc=%d", handle, rc );
    return rc;
  }

  llFileByteAlloc = *(PLLONG)&sStat.cbFileAlloc;
  llFileByteSizeOld = sStat.cbFile;

  rc = DosWrite( (HFILE)handle, pBuffer, ulWrite, pulActual );

  if ( rc != NO_ERROR )
  {
//dbg    debug( "DosWrite(%u), rc=%d", handle, rc );
    return rc;
  }

  rc = DosQueryFileInfo( (HFILE)handle, FIL_STANDARDL, &sStat,
                         sizeof(FILESTATUS3L) );
  if ( rc != NO_ERROR )
  {
    debug( "DosQueryFileInfo(%u), rc=%d", handle, rc );
    return rc;
  }

  llFileByteAlloc = *(PLLONG)&sStat.cbFileAlloc - llFileByteAlloc;

  if ( llFileByteAlloc > llByteFree )
  {
//dbg    debug( "New allocated size of file out of limit - roll back (%lld bytes) @ handle=%u",
//dbg           llFileByteSizeOld, handle );
    DosSetFileSizeL( (HFILE)handle, llFileByteSizeOld );
    return ERROR_DISK_FULL;
  }

  _lqIncBytesAlloc( psRes, llFileByteAlloc );

  return rc;
}

int APIENTRY NdpEAQuery(HCONNECTION conn, GEALIST *pGEAList, NDFILEINFO *pfi, FEALIST *pFEAList)
{
  return ERROR_NOT_SUPPORTED;
}

int APIENTRY NdpEASet(HCONNECTION conn, FEALIST *pFEAList, NDFILEINFO *pfi)
{
  return ERROR_NOT_SUPPORTED;
}

int APIENTRY NdpEASize(HCONNECTION conn, NDFILEINFO *pfi, ULONG *pulEASize)
{
  *pulEASize = 0;
  return ERROR_NOT_SUPPORTED;
}

int APIENTRY NdpFileEASize(HCONNECTION conn, NDFILEHANDLE handle, ULONG *pulEASize)
{
  return ERROR_NOT_SUPPORTED;
}
