/* EXEC procedure that will limit the simultaneous users of
   a resource.
   
   4 arguments (seperated by a comma) can be included in the 
   Option: EXEC  line in ATTRIBS.CFG:
        family  : the set of resources to which the max_users applies
        max_users : the maximum number of users
        failfile : relative to srehttp2 directory. If html, some substitutions
                   are done
        retryafter: a retry-after (this many seconds) header is added to the response

   *  If family is not specified, then the action (the portion of the
      selector before the ?) is used as the family.

   *  If max_users is not specified, then a value of 1 is used.

   * if failfail not specified, a generic "server is busy" response is returned

   * if retryafter is not specified, then no retry-after header is included
  
   *  Note that "family", whether it is explicitily defined or not, is NOT 
      host specific. That is, the set of users of a "family" of resources (and a
      family can consist of just one resource) is not sub-divided into what
      host the client sent a request to.
  
  Examples (of entries in ATTRIBS.CFG)

     Option: EXEC max_users.rex
     Option: EXEC max_users.rex family1
     Option: EXEC max_users.rex familyN , 6 , failfam.htm ,30
     Option: EXEC !max_u   familyN  4 

  Note: As the last example suggests ...
          if you intend to use MAX_USER.REX frequently, you might want
          to use the LOAD_EXEC option (in PRELOADS.CFG) to "load" this
          code into macrospace.  If you do this, you'll need to use the
          !PROC_NAME syntax in your EXEC options (in ATTRIBS.CFG).
          See ADV_OPTS.HTM for more details on how to preload EXEC procedures.

                ==========================================================
*/

/* SREhttp/2 passes the following arguments */
parse arg anarg,source,request,seluse2,servername,host_nickname,datadir
id_info=sre_request_info()
own_id=sre_request_info('OWN',,id_info)

parse upper var anarg afamily ',' maxu ',' failfile ',' retryafter .
if maxu='' then maxu=1
if datatype(maxu)<>'NUM' then do
   rcode=sre_error_response(500,,,'MAX_USER.REX: max-users is not a number: 'maxu,id_info)
   return rcode
end


if retryafter<>"" then do
  if datatype(retryafter)<>'NUM' then do
     rcode=sre_error_response(500,,,'MAX_USER.REX: retryafter is not a number: 'retryafter,id_info)
     return rcode
  end
  parse var request . . cprot .
  parse var cprot . '/' cprot
  if cprot<1.1 then retryafter=''
end

if afamily='' then do
    parse upper var seluse2 afamily  '?' .
end
afamily=strip(afamily)


if words(afamily)>1 then do
   rcode=sre_error_response(500,,,'MAX_USER.REX: improper family-name: 'afamily,id_info)
   return rcode
end


failfile0=failfile
if failfile<>'' then do
   fdir=sreh2_value('CFG_DIR',,host_nickname)
   usef=sreh2_queryfile(fdir,strip(failfile))
   if usef='' then do
       fdir2=sre_value('H2_CFG_DIR',,'SRE')
       usef=sreh2_queryfile(fdir2,strip(failfile))
   end
   if usef='' then do   
      rcode=sre_error_response(500,,,'MAX_USER.REX: missing failure file: '||failfile,id_info)
      return rcode
   end
   failfile=sre_read_file(usef,2,2)  /* if empty, use generic response */
end


/* See if this family is defined in SRE_CACHE */

fp=sre_cache('MAX_USER','READ',afamily,,,own_id)
if abbrev(translate(strip(fp)),'ERROR')=1 then do  /* try creating it */
   fp=sre_cache('MAX_USER','CREATE',,,,own_id)
   if abbrev(translate(strip(fp)),'ERROR')=1 then do  
      rcode=sre_error_response(500,,,'MAX_USER.REX: unable to create cache entry: 'fp,id_info)
      return rcode
   end
end

/* if empty, no users. */
thisreq=sre_request_info('REQ',,id_info) 


if words(fp) < maxu then do
    fp=fp||' '||thisreq   /* add this request number to busy list*/
    fp=sre_cache('MAX_USER','WRITE',afamily,fp,,0)  /* don't wait for answer */
    return ''
end 
else do         /* maybe busy. But first clean out completed requests */
   reqlist=sre_request_info('!LIST',,id_info)
   reqlist=translate(reqlist,' ',',')
   newfp=''             /* remove reqnums in fp that are not in rlist */
   do mm=1 to words(fp)
      afp=strip(translate(word(fp,mm)))
      if wordpos(afp,reqlist)>0 then newfp=newfp||' '||afp
   end 
end 
/* check against this more correct list */
fp=newfp

if  words(fp) < maxu  then do
    fp=fp||' '||thisreq   /* add this request number to busy list*/
    fp=sre_cache('MAX_USER','WRITE',afamily,fp,,0)  /* don't wait for answer */
    return ''            /* '' means "okay" */
end 

/* if here, still too many. Give up (don't bother writing cleaned up results)
    Use failfile (if specified), or use generic response  */

if failfile='' then do

  if retryafter='' then do      /* no retry-after header */
    rcode=sre_error_response(503,,,  ,
      'Sorry. At this moment, too many clients are using '||afamily||'.<p>Try again in a few minutes.',id_info)
    return rcode
  end
  else do
     foo=sre_command('HEADER Retry-After: '||retryafter,,id_info)
    rcode=sre_error_response(503,,,  ,
      'Sorry. At this moment, too many clients are using '||afamily|| ,
        '.<p>Your browser will automatically try again in '||retryafter||' seconds ... or you can try again in a few minutes.',id_info)
    return rcode
  end
end
else do
  amessage=''
  ahd=''
  parse var request . sel0 .
  if retryafter<>"" then do  
      amessage='Your browser will automatically try again in '||retryafter||' seconds.'
      ahd='Retry-After: '||retryafter
  end 
  mtype=get_mimetype(failfile0,host_nickname)
  if translate(mtype)='TEXT/HTML' then do  /*  html, do substitutions */
      failfile=sre_html_replace(failfile, ,
                       '#URL '||sel0 , ,
                       '#SERVER '||servername ,,
                       '#MESSAGE '||amessage,,
                       '#LINK '||'<a href="'||strip(sel0)||'">Try again?</a> ')
  end   
  rcode=sre_command('Var type '||mtype||' nocache ',failfile,id_info,'HTTP/1.1 503 Service Unavailable ',ahd)
  return rcode
end


/* ----------------------------------------------------------------------- */
/* Return the media type of a file, based on its extension.     */
/* Check some common ones, then call sreh2_get_mimetype */
get_mimetype:procedure
parse arg aa,ahost

  ipp=lastpos('.',aa)
  if ipp=0 then return 'application/octet-stream'  /* the default */

/* Get the extension; this assumes filenames have at least one '.' */
  fext=strip(translate(substr(aa,ipp+1)))

  cmtype=sreh2_value('CHECK_MIMETYPE_FILE',,ahost)

  if cmtype=0 then do   /* check "common" types first */
     if fext='HTM' | fext='HTML' | fext='SHT' | fext='SHTML' then return 'text/html'
     if fext='GIF' then return 'image/gif'
     if fext='JPG' | fext='JPEG' then return 'image/jpeg'
     if fext='TXT' | fext='TEXT' then return 'text/plain'
  end
  aa=sreh2_get_mimetype(fext,ahost,cmtype)
  return aa



