/*
    Interactive script to display the current state of IMAPD.

*/


/* Uncomment the following line for debug information. */
/* debugFile = "imapd-info.dbg" */


/* ************************************************************* */
/*                        Initialization                         */
/* ************************************************************* */

/* Set trap conditions. */
signal on Error
signal on Failure name Error
signal on Halt
signal on Syntax name Error

call RxFuncAdd "SysLoadFuncs", "RexxUtil", "SysLoadFuncs"
call SysLoadFuncs
parse value SysTextScreenSize() with scrHeight scrWidth

/* You can comment out variable socketName to work through the named pipe
   instead of a local socket. */
socketName     = "imapd"
pipeName       = "imapd"
usePipe        = 0
socketHandle   = -1
scrLineCnt     = 0
scrPrintCancel = 0
CRLF           = x2c("0D") || x2c("0A")
ESC            = x2c( "1B" )
global         = "pipeName socketName usePipe socketHandle debugFile " ,
                 "scrHeight scrWidth scrLineCnt scrPrintCancel CRLF ESC"


/* ************************************************************* */

parse value readHomeDirectoriesAndConnections( 0 ) with homeCnt connCnt

do forever

  /* Clear screen and print title. */
  call newScreen "Main menu"

  /* Some useful information. */
  say "Open home directories: " || homeCnt
  say "Established connections: " || connCnt
  say

  /* Display the menu and wait for user selection. */
  menu.0 = 2
  menu.1 = "Open home directories and connections."
  menu.2 = "Used disk space and quotas."
  userSelect = userItemSelect( "menu" )

  /* Execute selected item. */
  select
    when userSelect = "1" then
      parse value readHomeDirectoriesAndConnections( 1 ) with homeCnt connCnt

    when userSelect = "2" then
      call menuUsedDiskSpaceAndQuotas

    otherwise /* <ESC> pressed */
      leave
  end /* select */
end

/* Shutdown and exit. */
call SysCls
call imapdClose
EXIT

/* ************************************************************* */


/* readHomeDirectoriesAndConnections( print )
   print is 1 - print information.

   Returns the number of open home directories and the total number of
   connections. Values are separated by space.
*/
readHomeDirectoriesAndConnections: procedure expose (global)
  /* Request to IMAPD: QUERYFS
     Answer: +OK IMAP file system
             INBOX check period: NNN sec.
             Home domain1\user1: sessions[, dirty]
             Home domain1\user2: sessions[, dirty]
             Home domainN\userN: sessions[, dirty]
             .
  */
  if imapdTransact( "QUERYFS", "body" ) = "" then
    return "0 0"

  print = arg( 1 )

  if print then
    call printStart

  homeCnt = 0
  connCnt = 0

  if print then
  do
    call sortBody
    call printLine( left( "Home directory", 40, " " ) || "| Connections" )
    call printLine( copies( "-", 53 ) )
  end

  do i = 1 to body.0
    if left( body.i, 5 ) \= "Home " then
      iterate

    parse var body.i "Home " dir ": " conn ", " dirty
    if dom = "" then
      iterate

    homeCnt = homeCnt + 1
    connCnt = connCnt + conn

    if print then
    do
      if dirty \= "" then
        dirty = " (changed)"

      conn = format( conn, 5 )

      if \printLine( left( dir" ", 40, "." ) || "." || conn || dirty ) then
        leave
    end
  end

  if print then
    call printEnd

  return homeCnt || " " || connCnt


/* menuUsedDiskSpaceAndQuotas()

   The implementation of the submenu "Used disk space and quotas".
*/
menuUsedDiskSpaceAndQuotas: procedure expose (global)
  sort = 0
  sortCaption.0 = "percentage of used space"
  sortCaption.1 = "used space"

  do forever
    /* Request to IMAPD: QUERYSTORAGE
       Answer: +OK storage: D:\MailRoot\Path
               MailRoot: used/total
               Domain domain1: used/total
               User user11: usedInbox,usedImap/total
               User user12: usedInbox,usedImap/total (non-blocked)
               Domain domain2: used/total
               User user21: usedInbox,usedImap/total
               User user22: usedInbox,usedImap/total (non-blocked)
               .

       Where: used          - used [bytes],
              total         - limit [bytes] or "unlimited",
              usedInbox     - used for INBOX,
              usedImap      - used for other folders,
              (non-blocked) - E-Mail is accepted even if the limit is exceeded.
    */
    rc = imapdTransact( "QUERYSTORAGE", "body" )
    if rc = "" then
      return

    /* Display the menu screen and wait for the user selection. */

    call newScreen "Used disk space and quotas"

    parse var rc "storage: " storage
    if storage \= "" then
      say "Root directory: " || storage

    parse var body.1 "MailRoot: " usedStr
    if usedStr \= "" then
      say "Used " || usedSpaceStr( usedStr ) || CRLF

    menu.0 = 3
    menu.1 = "Domains"
    menu.2 = "Home directories"
    menu.3 = "Sort by: " || sortCaption.sort
    userSelect = userItemSelect( "menu" )

    select
      when userSelect = ESC then /* <ESC> pressed - return to previous menu. */
        leave

      when userSelect = "3" then /* Change sort type and display menu again. */
      do
        sort = (sort + 1) // 2
        iterate
      end

      otherwise
        nop
    end  /* select */

    /* Create: body. - list of indexes to sort, data. - lines to output */

    domain = ""
    j = 0
    do i = 2 to body.0
      parse var body.i objType " " objName ": " usedStr

      select
        when userSelect = "1" then         /* "1. Domains" selected */
        do
          if objType \= "Domain" then
            iterate
        end

        when userSelect = "2" then        /* "2. Home directories" selected */
        do
          if objType = "Domain" then
          do
            /* Store current domain name to add to usernames. */
            domain = objName
            iterate
          end

          if objType \= "User" then
            iterate

          objName = domain || "\" || objName
        end
      end

      used = usedSpaceStr( usedStr )

      select
        when sort = 0 then       /* Sort by percentage of used space */
        do
          parse var used sortId "%" line
          sortId = strip( sortId )
        end

        when sort = 1 then       /* Sort by used space (in bytes) */
        do
          parse var usedStr sortId "/" line
          parse var sortId sortId "," usedImap
          if usedImap \= "" then
            sortId = sortId + usedImap
        end
      end

      j = j + 1
      /* Second string in body. is index for data. */
      body.j = right( sortId, 13, "0" ) || " " || j
      /* data. is list of strings to output */
      data.j = left( objName" ", 38, "." ) || " " || used
      
    end  /* do i = 2 to body.0 */
    body.0 = j
    data.0 = j

    /* Sort strings in body. */
    call sortBody

    /* Get indexes from sorted body. and print strings from data. */
    call printStart
    do i = body.0 to 1 by -1
      parse var body.i idx " " j
      if \printLine( data.j ) then
        leave
    end
    call printEnd
  end

  /* Return to the previous menu. */
  return



/* ************************************************************* */
/*                          Utilites                             */
/* ************************************************************* */

/* usedSpaceStr( line )
   Parses QUERYSTORAGE answer in format N1,N2/Total to human readable string.
*/
usedSpaceStr: procedure
  parse arg used "/" total " " nb

  parse var used usedInbox "," usedImap
  if usedImap \= "" then
    used = usedInbox + usedImap

  if total = "unlimited" then
  do
    pr = 0
    str = ", unlimited"
  end
  else do
    pr = format( used * 100 / total,,0 )
    str = " from " || bytesStr( total )
  end

  if nb = "(non-blocked)" then
    str = str || ", not blockable"

  return format( pr, 3 ) || "% : " || bytesStr( used ) || str

bytesFromStr: procedure
  val = arg( 1 )
  return " from " || bytesStr( val )


/* bytesStr( bytes )
   Returns a short string representation of value bytes.
*/
bytesStr: procedure
  val = arg( 1 )

  if val < 1024 then
  do
    str = val
    eu = "b"
  end
  else do
    if val >= (1024 * 1024 * 1024 * 1024) then
    do
      val = val % (1024 * 1024 * 1024)
      eu = "Tb"
    end
    else if val >= (1024 * 1024 * 1024) then
    do
      val = val % (1024 * 1024)
      eu = "Gb"
    end
    else if val >= (1024 * 1024) then
    do
      val = val % 1024
      eu = "Mb"
    end
    else
      eu = "Kb"

    d_rem = val // 1024
    d_quot = val % 1024

    if val > 102400 | d_rem == 0 then
      /*  NNN / NN / N  */
      str = d_quot
    else
    do
      /*  N.NN / NN.N   */
      if val < 1024 then
        str = format( val / 1024, , 2 )
      else
        str = format( val / 1024, , 1 )
    end
  end

  return str || " " || eu


/* sortBody()
   Sorts the values in body.N variable.
   Variable body.0 must contain the number of records.
*/
sortBody: procedure expose body.
  if symbol( "body.0" ) \= "VAR" then
    return

  if body.0 \= 0 then
    call _qsort 1, body.0

  return

/* Sort function from "REXX Tips & Tricks" book. */
_qsort: procedure expose body.

  arg lf, re

  if re - lf < 9 then
    do lf = lf to re - 1

      m = lf

      do j = lf + 1 to re
        if body.j << body.m then                                   /* v2.80 */
          m = j
      end /* j = lf +1 to re */

      xchg = body.m;
      body.m = body.lf;
      body.lf = xchg

    end /* lf = lf to re -1 */
  else do
    i = lf
    j = re
    k = (lf + re) % 2
    t = body.k

    do until i > j

      do while body.i << t                                    /* v2.80 */
        i = i + 1
      end /* while body.i << t */

      do while body.j >> t                                    /* v2.80 */
        j = j - 1
      end /* while body.j >> t */

      if i <= j then
      do
        xchg = body.i
        body.i = body.j
        body.j = xchg
        i = i + 1
        j = j - 1
      end /* if i <= j then */

    end /* until i > j */

    call _qsort lf, j
    call _qsort i, re
  end /* else */

  return


/* newScreen( title )
   Clears the screen and prints the title.
*/
newScreen: procedure expose (global)
  title = arg( 1 )
  call SysCls

  titleLen = length( title )
  spaceStr = copies( " ", ( scrWidth - titleLen ) % 2 )
  say spaceStr || title
  say spaceStr || copies( "-", titleLen )
  say
  return


/* userItemSelect( stem )

   Prints a numbered list of items from the variable stem.1 .. stem.N and waits
   for the user's choice. Variable stem.0 must contain the number of records
   and should be less than or equal to 9.
   Returns the selected item number (1..stem.0) or ESC ("1B"x).
*/
userItemSelect:
  _uis.__items = arg( 1 )
  _uis.__N = value( _uis.__items || ".0" )

  do _uis.__i = 1 to _uis.__N
    say "  " || _uis.__i || ". " || value( _uis.__items || "." || _uis.__i ) 
  end
  say ""

  call charout , "Select item (1-" || _uis.__N || ") or press <ESC> to quit: "
  do forever
    _uis.__key = SysGetKey( "NOECHO" )
    if _uis.__key = ESC | (_uis.__key >= 1 & _uis.__key <= _uis.__N) then
      leave
  end
  return _uis.__key


/* printStart()
   printLine( string )
   printEnd()

   Functions for displaying large lists of strings.
   After filling all the lines of the screen, user will be prompted to continue
   the output or stop.
*/

printStart: procedure expose (global)
  scrLineCnt = 0
  scrPrintCancel = 0
  call SysCls
  return

printLine: procedure expose (global)
  scrLineCnt = scrLineCnt + 1

  say arg( 1 )

  if scrLineCnt < (scrHeight - 1) then
    return 1

  call charout , "--- Press ESC to stop or other key to continue. ---"
  scrLineCnt = 0
  key = SysGetKey( "NOECHO" )

  scrPrintCancel = ( key = ESC )
  if scrPrintCancel then
    return 0

  call charout , x2c("0D") || copies( " ", 45 ) || x2c("0D")
  return 1

printEnd: procedure expose (global)
  if scrPrintCancel then
    scrPrintCancel = 0
  else do
    call charout , "--- Press any key to return to the menu. ---"
    call SysGetKey "NOECHO"
    call SysCls
  end
  return


/* ************************************************************* */
/*          IMAPD control interface universal routines           */
/* ************************************************************* */

/* imapdOpen()

   Open pipe or socket.
   Returns 0 if an error occurred or 1 if successful.
*/

_imapdOpenSock: procedure expose (global)
  if symbol( "socketName" ) \= "VAR" then
    return 0

  if RxFuncQuery( "rxsfOpen" ) = 1 then
  do
    call RxFuncAdd "rxsfLoadFuncs", "rxsf", "rxsfLoadFuncs"
    call rxsfLoadFuncs

    if RxFuncQuery( "rxsfOpen" ) = 1 then
    do
      call debug "Error loading library RXSF.DLL"
      return 0
    end
  end

  socketHandle = rxsfOpen( socketName )
  if socketHandle = -1 then
  do
    call debug "Socket " || socketName || " open error"
    return 0
  end

  call debug "Socket " || socketName || " is open"
  return 1

_imapdOpenPipe: procedure expose (global)
  if symbol( "pipeName" ) \= "VAR" then
    return 0

  /* Expand the pipe name according to the system requirements. */
  if translate( left( pipeName, 6 ) ) \= "\PIPE\" then
    pipeName = "\PIPE\" || pipeName

  rc = stream( pipeName, "c", "open" )
  if left( rc, 6 ) = "READY:" then
  do
    call debug "Pipe " || pipeName || " is open"
    return 1
  end

  if left( rc, 9 ) = "NOTREADY:" then
  do
    rc = substr( rc, 10 )
    select
      when rc = 231 then       /* ERROR_PIPE_BUSY */
        rc = rc || ", pipe is busy"
   
      when rc = 3 then         /* ERROR_PATH_NOT_FOUND  */
        rc = rc || ", pipe does not exist"
    end
  end
  call debug "Pipe " || pipeName || " open error: " || rc

  return 0

imapdOpen: procedure expose (global)
  if _imapdOpenSock() then
  do
    usePipe = 0
    return 1
  end

  if _imapdOpenPipe() then
  do
    usePipe = 1
    return 1
  end

  return 0


/* imapdClose()

   Closes the socket or pipe opened by function sfOpen().
*/
imapdClose: procedure expose (global)
  if usePipe then
  do
    call debug "close pipe"
    call stream pipeName, "c", "close"
  end
  else if socketHandle \= -1 then
  do
    call debug "close socket"
    call rxsfClose socketHandle
    socketHandle = -1
  end
  return


/* imapdRequest( request[, stem] )

   Sends a request and receives a response.
   If the stem is specified, the response body will be read into it.
   The connection must be previously opened by the function imapdOpen().
   Returns the IMAPD response like: "+OK: details" or "-ERR: details".
*/
imapdRequest:
  _ir.__req = arg( 1 )
  _ir.__bodyStem = arg( 2 )

  call debug "Request: " || _ir.__req

  if usePipe then
  do

    /* Send a request through the pipe. */
    if lineout( pipeName, _ir.__req ) \= 0 then
    do
      call log "Error writing to the pipe " || pipeName
      return "ERROR:Error writing to the pipe"
    end

    /* Get a response from IMAPD. */
    _ir.__res = linein( pipeName )

    if _ir.__bodyStem \= "" then
    do
      _ir.__cnt = 0
      if word( _ir.__res, 1 ) = "+OK" then
      do
        do forever
          line = linein( pipeName )
          if line = "." then
            leave;

          _ir.__cnt = _ir.__cnt + 1
          call value _ir.__bodyStem || "." || _ir.__cnt, line
        end
      end
      call value _ir.__bodyStem || ".0", _ir.__cnt
    end

  end   /* if usePipe */
  else if _ir.__bodyStem = "" then
    _ir.__res = rxsfRequest( socketHandle, _ir.__req )
  else
    _ir.__res = rxsfRequest( socketHandle, _ir.__req, _ir.__bodyStem )

  call debug "Answer: " || _ir.__res
  return _ir.__res


/* imapdTransact( request[, stem] )

   Opens connection to IMAPD, sends a request and receives a response,
   closes the connection.
   If the stem is specified, the response body will be read into it.
   Returns an empty string if an error occurred and the response details if
   successful.
*/
imapdTransact:
  if \imapdOpen() then
    call die "IMAPD interface open error. Is IMAPD runned?"

  rc = imapdRequest( arg(1), arg(2) )
  call imapdClose

  parse var rc resp details
  if resp = "+OK" then
    return details

  say "Error. Server answer: " || details
  return ""


/* ************************************************************* */

Error:
  parse source . . cmdFile
  say "---"
  say "Signal " || condition( "C" ) || " in " || cmdFile
  say "  Source line " || SIGL || ": " || sourceline( SIGL )

  haveRC = symbol("RC") = "VAR"

  if condition( "D" ) \= '' then
    say "  Description: " || condition( "D" )
  if ( condition( "C" ) = 'SYNTAX' ) & haveRC then
    say "  Error (" || RC || "): " || errortext( RC )
  else if haveRC then
    say "  Error " || RC

  exit 0

/* die( message )
   Prints on the screen and writes a message to the debug file. Then quits.
 */
die: procedure
  str = "ERROR " || arg( 1 )
  say str
  call debug str
  exit 1

/* debug( message )
   Writes a message to the debug file.
*/
debug: procedure expose debugFile
  if symbol( "debugFile" ) \= "VAR" | debugFile = "" then
    return

  if stream( debugFile, "c", "open write" ) \= "READY:" then
  do
    say "Cannot open/create debug file " || debugFile
    return
  end

  call lineout debugFile, date( "S" ) || " " || time() || " " || arg( 1 )
  call stream debugFile, "c", "close"
  return
