/* RepairSystemDumpFile - repair corrupted system trap dump file
   All offsets and numeric values are implicitly 0-relative hex
   All index (ndx) values are implicitly 1-relative decimal

   FIXME to support large files
   FIXME to allow _pgpPageDir on command line
   FIXME to guess loader

   Copyright (c) 2014-2020 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of
   the GNU General Public License Version 3 or newer.
   The GPL Software License can be found in gnugpl3.txt or at
   http://www.gnu.org/licenses/licenses.html#GPL

   2014-07-21 SHL Baseline
   2015-12-30 SHL Time to finish this
   2015-12-31 SHL Time to finish this
   2016-01-02 SHL Avoid numerics errors
   2016-01-03 SHL More W4 support
   2016-02-14 SHL More error reporting
   2016-02-16 SHL Add more Sys2...() error reporting
   2016-02-16 SHL Rework G.!Indexes.!LowData math
   2017-08-10 SHL Switch to Globals
   2017-08-10 SHL Support AOSLDR
   2017-09-27 SHL Sync with templates
   2017-09-27 SHL Update some comments and messages
   2020-01-15 SHL Sync with templates
   2020-01-15 SHL Rewrite GDT selector correctly for debug kernels
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.6'

Globals = 'gAosLdr gArgList. gCmdName gDbgKrnl gDumpFile gEnv',
	  'gHandle gKrnlType gHeader gIndexes. gOffs.',
	  'gRxUtilExLoaded gSegs. gUseRxUtilEx gValues. gVersion'

call Initialize

Main:

  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  numeric digits 15			/* Need to support 32-bit numerics */

  gRxUtilExLoaded = 0

  do argNum = 1 to gArgList.0
    dumpFile = gArgList.argNum
    gDumpFile = dumpFile
    call DoOneFile
  end

  exit

/* end main */

/*=== DoOneFile() Process gDumpFile ===*/

DoOneFile: procedure expose (Globals)

  say
  say 'Checking' gDumpFile
  call ReadDumpHeader
  call GetKrnlType

  krnlType = gKrnlType

  changes = RepairHeader()

  say
  if changes = 0 then
    say 'No changes required'
  else do
    say 'Need to modify' changes 'header items'
    call AskYNQ 'Rewrite dump file header'
    if RESULT >= 2 then exit
    if RESULT = 0 then do
      say
      call RewriteDumpHeader
      say 'Modified' changes 'header items'
    end
  end

  return

/* end DoOneFile */

/*=== GetDWord(hexoffset) Return hex dword ===*/

GetDWord: procedure expose (Globals)

  parse arg hexoffset
  ndx = x2d(hexoffset)
  word = substr(gHeader, ndx + 1, 4)
  word = reverse(word)
  hexword = c2x(word)
  return hexword

/* end GetDWord */

/*=== GetKrnlType() Extract kernel type from buffer and set globals ===*/

GetKrnlType: procedure expose (Globals)

  revNdx = pos('Internal revision', gHeader)

  if revNdx = 0 then
    call Die 'Cannot find Internal revision in 1st 4MB bytes of' gDumpFile

  i = pos('00'x, gHeader, revNdx)

  if i = 0 then
    call Die 'Cannot find Internal revision string terminator 1st 4MB bytes of' gDumpFile

  header = substr(gHeader, revNdx, i - revNdx)
  parse var header . . revision
  revision = strip(revision)

  say 'Internal revision header found at offset' d2x(revNdx - 1) '(' || revNdx - 1 || ')'
  say 'Kernel revision is' revision

  gDbgKrnl = pos('BUGBITS:', gHeader) > 0

  /* gIndexes. are file offsets
     gOffs. are LOWDATA offsets or expected values for dump header items
     gOffs. are derived from .sym files
     gValues . are expected values for dump header items
     gValues . are derived from code
   */

  /* 2017-08-10 SHL FIXME to verify AOSLDR adjustments */
  select
  when pos('_SMP', revision) > 0 & gDbgKrnl then do
    /* SMP debug kernel */
    gKrnlType = 'SMP'
    gIndexes.!MPData = x2d('a000') + 512	/* a000 is mem offset, 512 is file offset */
    if gAosLdr then
      gSegs.!LowData = '0500'
    else
      gSegs.!LowData = '0b00'
    decHeaderPage = 10			/* Header somewhere in LOWDATA:a000 page */
    gOffs.!CurProcPID = 'ac4'
    gOffs.!TaskData = '2e66'
    gOffs.!SysSemDataTable = '89a2'
    gOffs.!papTCBPtrs = '2de0'
    gOffs.!FirstPacket = '3d58'
    gOffs.!LastPacket = '66a8'
    gValues.!ras_krnl_type = '2002000C'	/* ras_krnl_type */
  end
  when pos('_SMP', revision) > 0 then do
    /* SMP retail kernel */
    gKrnlType = 'SMP'
    gIndexes.!MPData = x2d('a000') + 512	/* a000 is mem offset, 512 is file offset */
    if gAosLdr then
      gSegs.!LowData = '0500'
    else
      gSegs.!LowData = '0b00'
    decHeaderPage = 10			/* Header somewhere in LOWDATA:a000 page */
    gOffs.!TaskData = '2e66'
    gOffs.!CurProcPID = 'ac4'
    gOffs.!SysSemDataTable = '8892'
    gOffs.!papTCBPtrs = '2de0'
    gOffs.!FirstPacket = '3cd4'
    gOffs.!LastPacket = '6624'
    gValues.!ras_krnl_type = '2000000C'
  end
  when pos('_W4', revision) > 0 & gDbgKrnl then do
    /* W4 debug kernel */
    gKrnlType = 'W4'
    if gAosLdr then
      call 'FIXME to support W4 debug kernel with AOSLDR'
    gSegs.!LowData = '0a00'
    decHeaderPage = 7			/* Header somewhere in LOWDATA:7000 page */
    gOffs.!CurProcPID = '910'
    gOffs.!TaskData = '0e82'
    gOffs.!SysSemDataTable = '5bab'
    gOffs.!FirstPacket = '1d68'
    gOffs.!LastPacket = '46b8'
    gOffs.!papTCBPtrs = 'dfc'
    gValues.!ras_krnl_type = '0002000C'
  end
  when pos('_W4', revision) > 0 then do
    /* W4 retail kernel */
    gKrnlType = 'W4'
    if gAosLdr then
      call 'FIXME to support W4 debug kernel with AOSLDR'
    gSegs.!LowData = '0a00'
    decHeaderPage = 7			/* Header somewhere in LOWDATA:7000 page */
    gOffs.!CurProcPID = '910'
    gOffs.!TaskData = '0e82'
    gOffs.!SysSemDataTable = '5a9b'
    gOffs.!FirstPacket = '1ce4'
    gOffs.!LastPacket = '4634'
    gOffs.!papTCBPtrs = 'dfc'
    gValues.!ras_krnl_type = '0000000C'
  end
  when pos('_UNI', revision) > 0 > 0 & gDbgKrnl then do
    gKrnlType = 'UNI'
    call Die 'FIXME to support' gKrnlType 'kernel'
  end
  when pos('_UNI', revision) > 0 then do
    gKrnlType = 'UNI'
    call Die 'FIXME to support' gKrnlType 'kernel'
  end
  otherwise
    Die 'Cannot determine kernel type from' revision
  end

  /* Calculate 0 relative location of LOWDATA in file
     Internal revision string is Header + 2 relative to LOWDATA
     Header is in either a000 or 7000 page depending on kernel
     -1 converts to 0 relative
     - 512 accounts for file header
   */
  gIndexes.!LowData = ((revNdx - 1 - 512) % 4096 * 4096) - decHeaderPage * 4096 + 512
  say 'LOWDATA segment starts at file offset' d2x(gIndexes.!LowData) '(' || gIndexes.!LowData || ')'

  say 'krnlType is' gKrnlType
  say 'Debug kernel is' gDbgKrnl

  return

/* end GetKrnlType */

/*=== GetWord(hexoffset, segname) Get 16-bit word from dump header or LOWDATA segment ===*/

GetWord: procedure expose (Globals)
  parse arg hexoffset, segName
  ndx = x2d(hexoffset)

  select
  when segName = 'mpdata' then
    ndx = ndx + gIndexes.!MPData
  when segName = 'lowdata' then
    ndx = ndx + gIndexes.!LowData
  when segName = '' then
    nop					/* 0 relative */
  otherwise
    call Die 'Segment name' segName 'unexpected'
  end

  word = substr(gHeader, ndx + 1, 2)	/* REXX is 1 relative */
  word = reverse(word)			/* Data is low-endian */
  word = c2x(word)
  return word

/* end GetWord */

/*=== ReadDumpHeader() Read 1st 4MB of dump file into header buffer ===*/

ReadDumpHeader: procedure expose (Globals)

  mbytes = 4
  bytes = mbytes * 1024 * 1024

  drop ErrCondition

  signal off NotReady
  call stream gDumpFile, 'C', 'OPEN READ'
  signal on NotReady name Error

  if RESULT == 'READY:' then
    gUseRxUtilEx = 0			/* Use REXX IO */
  else do
    /* Check if need to use large file IO */
    err = RESULT
    size = stream(gDumpFile, 'C', 'QUERY SIZE')
    if size \= 1 & size < 2 * 1024 ** 3 then
      call Die 'Cannot open' gDumpFile '- file may be locked (' || err || ')'
    else do
      /* Assume must be > 2GB REXX if says just 1 byte and cannot open */
      if \ gRxUtilExLoaded then do
	say 'Loading RxUtilEx - need large file support'
	call LoadRxUtilEx
	gRxUtilExLoaded = 1
      end
      gHandle = Sys2Open(gDumpFile, 'O', 'R')
      if gHandle == '' then
	call Die 'Cannot open' gDumpFile '- file may be locked - SYS2ERR' SYS2ERR
      gUseRxUtilEx = 1
    end
  end

  if \ gUseRxUtilEx then do
    call on NotReady name CatchError
    gHeader = charin(gDumpFile, 1, bytes)
    signal on NotReady name Error
    call stream gDumpFile, 'C', 'CLOSE'
    if symbol('ErrCondition') == 'VAR' then
      call Die 'Cannot read' gDumpFile '- file may be corrupted or truncated (' || ErrCondition || ')'
  end
  else do
    gHeader = Sys2Read(gHandle, bytes)
    err = SYS2ERR
    if \ Sys2Close(gHandle) then
      call Die 'Cannot close' gDumpFile '- SYS2ERR' SYS2ERR
  end

  read = length(gHeader)
  if read \= bytes then do
    if \ gUseRxUtilEx then
      call Die 'Cannot read' gDumpFile '- file may be corrupted or truncated - expected' mbytes 'MBytes, read' bytes 'bytes'
    else
      call Die 'Cannot read' gDumpFile '- file may be corrupted or truncated - expected' mbytes 'MBytes, read' bytes 'bytes - SYS2ERR' err
  end

  return

/* end ReadDumpHeader */

/*=== RepairDWord(hexoffset, newhexdword) Repair one dword ===*/

RepairDWord: procedure expose (Globals)

  parse arg hexoffset, newhexdword, desc
  oldhexdword = GetDWord(hexoffset)
  if oldhexdword == translate(newhexdword) then
    changes = 0
  else do
    say 'Changing' hexoffset 'from' oldhexdword 'to' newhexdword '(' || desc || ')'
    call PutDWord hexoffset, newhexdword
    changes = 1
  end
  return changes

/* end RepairDWord */

/*=== RepairHeader() Repair dump file header ===*/

RepairHeader: procedure expose (Globals)

  changes = 0

  /* curProcPID 2e (446) - 0..MaxThreads - 1 (CurProcPID) */

  if gKrnlType == 'SMP' then
    segName ='mpdata'
  else
    segName ='lowdata'

  changes = changes + RepairWord('426', gSegs.!LowData, 'low_data_seg')

  changes = changes + RepairWord('428', gSegs.!LowData, 'high_data_seg')

  changes = changes + RepairWord('42a', gSegs.!LowData, 'seg ras_stda_addr')

  maxThreads = GetWord('434')
  say 'MaxThreads is' maxThreads

  curPid = GetWord(gOffs.!CurProcPID, segName)
  say 'CurProcPID is' curPid

  if curPid \= 0 then
    changes = changes + RepairWord('446', curPid, 'CurProcPID')

  /* TaskData (30) @0448: 0b00:2e66 */
  changes = changes + RepairSegOff('448', gSegs.!LowData, gOffs.!TaskData, 'TaskData')

  pkt = GetWord(gOffs.!FirstPacket, 'lowdata')
  changes = changes + RepairWord('44c', pkt, 'FirstPacket')

  pkt = GetWord(gOffs.!LastPacket, 'lowdata')
  changes = changes + RepairWord('44e', pkt, 'LastPacket')

  /* SysSemDataTable (38) @0450: 0b00:8892 */
  changes = changes + RepairSegOff('450', gSegs.!LowData, gOffs.!SysSemDataTable, 'SysSemDataTable')

  /* GDT_Seg (3c) @0454: 0128 */
  if gDbgKrnl then
    changes = changes + RepairWord('454', '0138', 'GDT_Seg')	/* Thanks dixie */
  else
    changes = changes + RepairWord('454', '0128', 'GDT_Seg')

  /* Buffers_Seg (3e) @0456: 00a8 */
  changes = changes + RepairWord('456', '00a8', 'Buffers_Seg')

  /* papTCBPtrs (40) @0458: 0460:0000 */
  off = gOffs.!papTCBPtrs
  seg = d2x(x2d(off) + 2)

  off = GetWord(off, 'lowdata')
  seg = GetWord(seg, 'lowdata')
  changes = changes + RepairSegOff('458', seg, off, 'papTCBPtrs')

  /* ssSeg            @045c: ???? */
  /* espOffset        @045e: ???????? */
  ss = GetWord('45c')
  esp = GetDWord('45e')
  /* SMP
     90        GDT_DFSTACK                Trap 8 stack selector
     1530      GDT_PTDA                   PTDA/TCB/TSD selector - kernel SS
     1558      GDT_INTSTACK32
     15e8      GDT_INTSTACK               Interrupt stack alias - SMP kernel

     W4
     30        GDT_PTDA                   PTDA/TCB/TSD selector
     90        GDT_DFSTACK                Trap 8 stack selector
     e8        GDT_INTSTACK               Interrupt stack alias - Warp4 kernel
     1558      GDT_INTSTACK32
  */
  if gKrnlType == 'SMP' then do
    ssok = ss == '0090' | ss == '1530' | ss == '1558' | ss == translate('15e8')
    ssnew = '15e8'
  end
  else do
    ssok = ss == '0090' | ss == '0030' | ss == '1558' | ss == translate('00e8')
    ssnew = '00e8'
  end

  if \ ssok then do
    was = changes
    changes = changes + RepairWord('45c', ssnew, 'ss')
    changes = changes + RepairDWord('45e', '0000ff8a', 'esp')
    if changes \= was then do
      s = strip(ssnew,, "'")
      say 'ss:esp forced to ' s || ':ff8a and is probably incorrect'
    end
  end

  /* _signature      db      "OFF"   ; 4e (466) - 0046464f */
  changes = changes + RepairDWord('466', '0046464f', '_signature')

  /* ras_krnl_type (5a)       @0472: 0e020000 */
  /* 0b00:000086ea  2000000c */
  changes = changes + RepairDWord('472', gValues.!ras_krnl_type, 'ras_krnl_type')

  /* _pgPhysCnt 52 (46a) - _pgPhysMax + 1 */

  hexphyscnt = GetDWord('46a')		/* _pgPhysCnt pages */
  physcnt = x2d(hexphyscnt)
  hexdiskend = GetDWord('4')		/* disk_end */
  diskpages = ((x2d(hexdiskend) + 1) % 4096)
  hexdiskpages = d2x(diskpages, 8)

  if physcnt \= diskpages then do
    say '_pgPhysCnt is' hexphyscnt 'pages'
    say 'disk_end is' hexdiskend 'which is' hexdiskpages 'pages'
    changes = changes + RepairDWord('46a', hexdiskpages, '_pgPhysCnt')
  end

  if 0 then do
  if changes > -1 then do
    pages = x2d(physcnt)
    /* 1MB = 256 pages, 512MB = 131072 4KB pages, 4GB = 1048576 4KB pages */
    if right(physcnt, 1) \== '0' | pages < 131072 | pages > 1048576 then
      say '_pgPhysCnt (' || physcnt || ') probably corrupted - check with dd _pgPhysMax l1'
  end
  end

  return changes

/* end RepairHeader */

/*=== RepairSegOff(hexoffset, newhexseg, newhexoffset) Repair seg:offset ===*/

RepairSegOff: procedure expose (Globals)

  parse arg hexoffset, newhexseg, newhexoff, desc
  hexoffset2 = d2x(x2d(hexoffset) + 2)
  /* seg:off order is non-standard */
  oldhexseg = GetWord(hexoffset)
  oldhexoff = GetWord(hexoffset2)
  if oldhexseg == translate(newhexseg) & oldhexoff == translate(newhexoff) then
    changes = 0
  else do
    say 'Changing' hexoffset 'from' oldhexseg':'oldhexoff 'to' newhexseg':'newhexoff '(' || desc || ')'
    call PutWord hexoffset, newhexseg
    call PutWord hexoffset2, newhexoff
    changes = 1
  end
  return changes

/* end RepairSegOff */

/*=== RepairWord(hexoffset, newhexword) Repair one word ===*/

RepairWord: procedure expose (Globals)

  parse arg hexoffset, newhexword, desc
  oldhexword = GetWord(hexoffset)
  if oldhexword == translate(newhexword) then
    changes = 0
  else do
    say 'Changing' hexoffset 'from' oldhexword 'to' newhexword '(' || desc || ')'
    call PutWord hexoffset, newhexword
    changes = 1
  end
  return changes

/* end RepairWord */

/*=== RewriteDumpHeader() Rewrite 1st 2048 (0x800) bytes of dump file from header buffer ===*/

RewriteDumpHeader: procedure expose (Globals)

  if \ gUseRxUtilEx then do
    drop ErrCondition
    call on NotReady name CatchError
    call charout gDumpFile, left(gHeader, 2048), 1
    signal on NotReady name Error
    call stream gDumpFile, 'C', 'CLOSE'
    if symbol('ErrCondition') == 'VAR' then
      call Die 'Cannot rewrite' gDumpFile '- file may be read-only (' || ErrCondition || ')'
  end
  else do
    gHandle = Sys2Open(gDumpFile, 'O', 'RW')
    if gHandle == '' then
      call Die 'Cannot open' gDumpFile '- file may be locked - SYS2ERR' SYS2ERR
    bytes = Sys2Write(gHandle, left(gHeader, 2048))
    err = SYS2ERR
    if \ Sys2Close(gHandle) then
      call Die 'Cannot close' gDumpFile '- SYS2ERR' SYS2ERR
    if bytes \= 2048 then
      call Die 'Cannot rewrite' gDumpFile '- file may be read-only - wrote' bytes 'bytes of 2048 bytes - SYS2ERR' err
  end

  return

/* end RewriteDumpHeader */

/*=== PutDWord(hexoffset, hexdword) Replace dword at offset in header buffer ===*/

PutDWord: procedure expose (Globals)

  parse arg hexoffset, hexdword
  ndx = x2d(hexoffset)

  if length(hexdword) \= 8 then
    call Die hexdword 'is not 4 bytes'

  word = reverse(x2c(hexdword))
  gHeader = overlay(word, gHeader, ndx + 1)
  return

/* end PutDWord */

/*=== PutWord(hexoffset, hexword) Replace word at offset in header buffer ===*/

PutWord: procedure expose (Globals)

  parse arg hexoffset, hexword
  ndx = x2d(hexoffset)

  if length(hexword) \= 4 then
    call Die hexword 'is not 2 bytes'

  word = reverse(x2c(hexword))		/* To Intel format */
  gHeader = overlay(word, gHeader, ndx + 1)
  return

/* end PutWord */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  /* Preset defaults */
  gArgList.0 = 0			/* Reset arg count */
  gAosLdr = 1

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg

  select
  when curSw == 'a' then
    gAosLdr = 1
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'i' then
    gAosLdr = 0
  when curSw == 'V' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg
  if \ IsFile(curArg) then
    call ScanArgsUsage 'Cannot find' curArg
  i = gArgList.0 + 1
  gArgList.i = curArg
  gArgList.0 = i

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)
  if gArgList.0 = 0 then do
    dumpFile = 'dumpdata.001'
    if \ IsFile(dumpFile) then
      call ScanArgsUsage 'System dump file name required'
    gArgList.1 = dumpFile
    gArgList.0 = 1
  end
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Repair system trap dump file.'
  say
  say 'Usage:' gCmdName '[-a] [-h] [-i] [-V] [-?] dumpfile...'
  say
  say '  -a           Assume Arca Noae AOSLDR, this is the default'
  say '  -h -?        Display this message'
  say '  -i           Assume IBM OS2LDR, defaults to AOSLDR'
  say '  -v           Enable verbose output'
  say '  -V           Display version number and quit'
  say
  say '  dumpfile     System dump file name, defaults to dumpdata.001'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-a] [-h] [-i] [-V] [-?] dumpfile...'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== AskYNQ(prompt, skip) returns 0=Yes, 1=No, 2=Quit ===*/

AskYNQ: procedure
  parse arg msg, skip

  /* In case window title has odd characters */
  signal off Error
  '@if "%_DOS%" == "OS2" activate "%_WINTITLE"'	/* Take focus if running 4OS2 */
  signal on Error

  /* If line skip requested */
  if skip \= '' & skip \= 0 then
    call lineout 'STDERR', ''

  if msg == '' then
    msg = 'Continue'
  call charout 'STDERR', msg '(y/n/q) ? '
  do forever
    key = translate(SysGetKey('NOECHO'))
    if key == 'Y' | key == 'N' then do
      call lineout 'STDERR', key
      if key == 'Y' then
	ynq = 0
      else
	ynq = 1
      leave
    end
    if key == 'Q' | c2x(key) == '1B' then do
      call lineout 'STDERR', ''
      ynq = 2
      leave
    end
  end /* forever */
  return ynq

/* end AskYNQ */

/*=== CatchError() Catch condition; return gErrCondition ===*/

CatchError:
  gErrCondition = condition('C')
  return
/* end CatchError */

/*=== IsFile(file) return true if arg is file and file exists ===*/

IsFile: procedure expose (Globals)
  parse arg file
  if file == '' then
    yes = 0
  else do
    /* '.' returns files in '.' - so avoid false positives */
    call SysFileTree file, 'files', 'F'
    if RESULT \= 0 then
      call Die 'SysFileTree' file 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = file \== '.' & files.0 \= 0
  end
  return yes

/* end IsFile */

/*=== LoadRxUtilEx() Load Alex's RxUtilEx functions ===*/

LoadRxUtilEx:
  if RxFuncQuery('Sys2LoadFuncs') then do
    call RxFuncAdd 'Sys2LoadFuncs', 'RXUTILEX', 'Sys2LoadFuncs'
    if RESULT then
      call Die 'Cannot load Sys2LoadFuncs'
    call Sys2LoadFuncs
  end
  return

/* end LoadRxUtilEx */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \= '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/* The end */
