/* 
 * tclOS2Notify.c --
 *
 *	This file contains OS/2-specific procedures for the notifier,
 *	which is the lowest-level part of the Tcl event loop.  This file
 *	works together with ../generic/tclNotify.c.
 *
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 * Copyright (c) 1996-1998 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "tclOS2Int.h"

/*
 * The information below is used to provide read, write, and
 * exception masks to select during calls to Tcl_DoOneEvent.
 */

static fd_set checkMasks[3];
                                /* This array is used to build up the masks
                                 * to be used in the next call to select.
                                 * Bits are set in response to calls to
                                 * Tcl_WatchFile. */
static fd_set readyMasks[3];
                                /* This array reflects the readable/writable
                                 * conditions that were found to exist by the
                                 * last call to select. */
static int numFdBits;           /* Number of valid bits in checkMasks
                                 * (one more than highest fd for which
                                 * Tcl_WatchFile has been called). */

/*
 * Static routines in this file:
 */

static int      MaskEmpty _ANSI_ARGS_((long *maskPtr));

/*
 * The following variable is a backdoor for use by Tk.  It is set when
 * Tk needs to process events on the Tcl event queue without reentering
 * the system event loop.  Tk uses it to flush the Tcl event queue.
 */

static int ignoreEvents = 0;

/*
 *----------------------------------------------------------------------
 *
 * TclOS2FlushEvents --
 *
 *	This function is a special purpose hack to allow Tk to
 *	process queued Window events during a recursive event loop
 *	without looking for new events on the system event queue.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Services any pending Tcl events and calls idle handlers.
 *
 *----------------------------------------------------------------------
 */

void
TclOS2FlushEvents()
{
    ignoreEvents = 1;
    while (Tcl_DoOneEvent(TCL_DONT_WAIT|TCL_WINDOW_EVENTS|TCL_IDLE_EVENTS)) {
    }
    ignoreEvents = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_WatchFile --
 *
 *	Arrange for Tcl_DoOneEvent to include this file in the masks
 *	for the next call to select.  This procedure is invoked by
 *	event sources, which are in turn invoked by Tcl_DoOneEvent
 *	before it invokes select.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	
 *	The notifier will generate a file event when the I/O channel
 *	given by fd next becomes ready in the way indicated by mask.
 *	If fd is already registered then the old mask will be replaced
 *	with the new one.  Once the event is sent, the notifier will
 *	not send any more events about the fd until the next call to
 *	Tcl_NotifyFile. 
 *
 *----------------------------------------------------------------------
 */

void
Tcl_WatchFile(file, mask)
    Tcl_File file;		/* Opaque identifier for a stream. */
    int mask;			/* OR'ed combination of TCL_READABLE,
        			 * TCL_WRITABLE, and TCL_EXCEPTION:
        			 * indicates conditions to wait for
        			 * in select. */
{
    int fd, type;

    fd = (int) Tcl_GetFileInfo(file, &type);

    if (type != TCL_OS2_FILE && type != TCL_OS2_SOCKET) {
        panic("Tcl_WatchFile: unexpected file type");
    }

    if (fd >= FD_SETSIZE) {
        panic("Tcl_WatchFile can't handle file id %d", fd);
    }

    if (mask & TCL_READABLE) {
        FD_SET(fd, &checkMasks[0]);
    }
    if (mask & TCL_WRITABLE) {
        FD_SET(fd, &checkMasks[1]);
    }
    if (mask & TCL_EXCEPTION) {
        FD_SET(fd, &checkMasks[2]);
    }
    if (numFdBits <= fd) {
        numFdBits = fd+1;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_FileReady --
 *
 *	Indicates what conditions (readable, writable, etc.) were
 *	present on a file the last time the notifier invoked select.
 *	This procedure is typically invoked by event sources to see
 *	if they should queue events.
 *
 * Results:
 *	The return value is 0 if none of the conditions specified by mask
 *	was true for fd the last time the system checked.  If any of the
 *	conditions were true, then the return value is a mask of those
 *	that were true.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_FileReady(file, mask)
    Tcl_File file;	/* File handle for a stream. */
    int mask;			/* OR'ed combination of TCL_READABLE,
        			 * TCL_WRITABLE, and TCL_EXCEPTION:
        			 * indicates conditions caller cares about. */
{
    int result = 0, type, fd;

    fd = (int) Tcl_GetFileInfo(file, &type);
    if (type != TCL_OS2_FILE && type != TCL_OS2_SOCKET) {
        panic("Tcl_FileReady: unexpected file type");
    }

    if ((mask & TCL_READABLE) && (FD_ISSET(fd, &readyMasks[0]))) {
        result |= TCL_READABLE;
    }
    if ((mask & TCL_WRITABLE) && (FD_ISSET(fd, &readyMasks[1]))) {
        result |= TCL_WRITABLE;
    }
    if ((mask & TCL_EXCEPTION) && (FD_ISSET(fd, &readyMasks[2]))) {
        result |= TCL_EXCEPTION;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_WaitForEvent --
 *
 *	This procedure does the lowest level wait for events in a
 *	platform-specific manner.  It uses information provided by
 *	previous calls to Tcl_WatchFile, plus the timePtr argument,
 *	to determine what to wait for and how long to wait.
 *
 * Results:
 *	The return value is normally TCL_OK.  However, if there are
 *	no events to wait for (e.g. no files and no timers) so that
 *	the procedure would block forever, then it returns TCL_ERROR.
 *
 * Side effects:
 *	May put the process to sleep for a while, depending on timePtr.
 *	When this procedure returns, an event of interest to the application
 *	has probably, but not necessarily, occurred.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_WaitForEvent(timePtr)
    Tcl_Time *timePtr;		/* Specifies the maximum amount of time
        			 * that this procedure should block before
        			 * returning.  The time is given as an
        			 * interval, not an absolute wakeup time.
        			 * NULL means block forever. */
{
    QMSG msg;
    int foundEvent = 1;
    struct timeval timeout, *timeoutPtr;
    int numFound;

    /*
     * If we are ignoring events from the system, just return immediately.
     */

    if (ignoreEvents) {
        return TCL_OK;
    }

    /*
     * Set up the asynchronous select handlers for any sockets we
     * are watching.
     */

    /*
     * Look for an event, setting a timer so we don't block forever.
     * Return TCL_ERROR if this is a command-line tcl application.
     */

    memcpy((VOID *) readyMasks, (VOID *) checkMasks, 3*sizeof(fd_set));
    if (usePm) {
        if (timePtr != NULL) {
            ULONG ms;
    	    ms = timePtr->sec * 1000;
    	    ms += timePtr->usec / 1000;
        
    	    if (ms > 0) {
    	        ULONG timerHandle = WinStartTimer(hab, NULLHANDLE, 1, ms);
    	        WinGetMsg(hab, &msg, NULLHANDLE, 0, 0);
    	        WinStopTimer(hab, NULLHANDLE, timerHandle);
    	    } else {
        
    	        /*
    	         * If the timeout is too small, we just poll.
    	         */
    	        foundEvent = WinPeekMsg(hab, &msg, NULLHANDLE, 0, 0, PM_REMOVE);
    	    }
        } else {
    	    WinGetMsg(hab, &msg, NULLHANDLE, 0, 0);
        }
        
        /*
         * Dispatch the message, if we found one.  If we are exiting, be
         * sure to inform Tcl so we can clean up properly.
         */
        
        if (foundEvent) {
            if (msg.msg == WM_QUIT) {
                Tcl_Exit(0);
            }
            WinDispatchMsg(hab, &msg);
        }
        /*
         * We've had any timeout already, so if there's nothing to look for
         * just return TCL_OK
         */
        if ((numFdBits == 0) || (MaskEmpty((long *) readyMasks))) {
            return TCL_OK;
        }
    }

    if (timePtr == NULL) {
        if ((numFdBits == 0) || (MaskEmpty((long *) readyMasks))) {
            return TCL_ERROR;
        }
        timeoutPtr = NULL;
    } else {
        timeoutPtr = &timeout;
        timeout.tv_sec = timePtr->sec;
        timeout.tv_usec = timePtr->usec;
    }
    numFound = select(numFdBits, &readyMasks[0], &readyMasks[1], &readyMasks[2],
                      timeoutPtr);

    /*
     * Some systems don't clear the masks after an error, so
     * we have to do it here.
     */

    if (numFound == -1) {
        FD_ZERO(&readyMasks[0]);
        FD_ZERO(&readyMasks[1]);
        FD_ZERO(&readyMasks[2]);
    }

    /*
     * Reset the check masks in preparation for the next call to
     * select.
     */

    numFdBits = 0;
    FD_ZERO(&checkMasks[0]);
    FD_ZERO(&checkMasks[1]);
    FD_ZERO(&checkMasks[2]);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_Sleep --
 *
 *	Delay execution for the specified number of milliseconds.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Time passes.
 *
 *----------------------------------------------------------------------
 */

void
Tcl_Sleep(ms)
    int ms;			/* Number of milliseconds to sleep. */
{
    DosSleep(ms);
}

/*
 *----------------------------------------------------------------------
 *
 * MaskEmpty --
 *
 *      Returns nonzero if mask is empty (has no bits set).
 *
 * Results:
 *      Nonzero if the mask is empty, zero otherwise.
 *
 * Side effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

static int
MaskEmpty(maskPtr)
    long *maskPtr;
{
    long *runPtr, *tailPtr;
    int found, sz;

    sz = 3 * sizeof(fd_set);
    for (runPtr = maskPtr, tailPtr = maskPtr + sz, found = 0;
             runPtr < tailPtr;
             runPtr++) {
        if (*runPtr != 0) {
            found = 1;
            break;
        }
    }
    return !found;
}
