/* 
 * tclOS2Pipe.c -- This file implements the OS/2-specific pipeline exec
 *                 functions.
 *      
 *
 * Copyright (c) 1996 Sun Microsystems, Inc.
 * Copyright (c) 1996-2001 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#include "tclOS2Int.h"

#define HF_STDIN  0
#define HF_STDOUT 1
#define HF_STDERR 2

/*
 * The following variable is used to tell whether this module has been
 * initialized.
 */

static int initialized = 0;

/*
 * The following defines identify the various types of applications that
 * run under OS/2.  There is special case code for the various types.
 */

#define APPL_NONE       0
#define APPL_DOS        1
#define APPL_WIN3X      2
#define APPL_WIN32      3
#define APPL_OS2WIN     4
#define APPL_OS2FS      5
#define APPL_OS2PM      6
#define APPL_OS2CMD     7

/*
 * The following constants and structures are used to encapsulate the state
 * of various types of files used in a pipeline.
 */

#define OS2_TMPFILE	1           /* OS/2 emulated temporary file. */
#define OS2_FILE	2           /* Basic OS/2 file. */
#define OS2_FIRST_TYPE	OS2_TMPFILE
#define OS2_LAST_TYPE	OS2_FILE

/*
 * This structure encapsulates the common state associated with all file
 * types used in a pipeline.
 */

typedef struct OS2File {
    int type;                   /* One of the file types defined above. */
    HFILE handle;               /* Open file handle. */
} OS2File;

/*
 * The following structure is used to keep track of temporary files
 * and delete the disk file when the open handle is closed.
 * The type field will be OS2_TMPFILE.
 */

typedef struct TmpFile {
    OS2File file;               /* Common part. */
    char *name;                 /* Name of temp file. */
} TmpFile;

/*
 * State flags used in the PipeInfo structure below.
 */

#define PIPE_PENDING    (1<<0)  /* Message is pending in the queue. */
#define PIPE_ASYNC      (1<<1)  /* Channel is non-blocking. */

/*
 * This structure describes per-instance data for a pipe based channel.
 */

typedef struct PipeInfo {
    Tcl_Channel channel;        /* Pointer to channel structure. */
    int validMask;              /* OR'ed combination of TCL_READABLE,
                                 * TCL_WRITABLE, or TCL_EXCEPTION: indicates
                                 * which operations are valid on the file. */
    int watchMask;              /* OR'ed combination of TCL_READABLE,
                                 * TCL_WRITABLE, or TCL_EXCEPTION: indicates
                                 * which events should be reported. */
    int flags;                  /* State flags, see above for a list. */
    TclFile readFile;           /* Output from pipe. */
    TclFile writeFile;          /* Input from pipe. */
    TclFile errorFile;          /* Error output from pipe. */
    int numPids;                /* Number of processes attached to pipe. */
    Tcl_Pid *pidPtr;            /* Pids of attached processes. */
    struct PipeInfo *nextPtr;   /* Pointer to next registered pipe. */
} PipeInfo;

/*
 * The following pointer refers to the head of the list of pipes
 * that are being watched for file events.
 */

static PipeInfo *firstPipePtr;

/*
 * The following structure is what is added to the Tcl event queue when
 * pipe events are generated.
 */

typedef struct PipeEvent {
    Tcl_Event header;           /* Information that is standard for
                                 * all events. */
    PipeInfo *infoPtr;          /* Pointer to pipe info structure.  Note
                                 * that we still have to verify that the
                                 * pipe exists before dereferencing this
                                 * pointer. */
} PipeEvent;

/*
 * Declarations for functions used only in this file.
 */
static int      ApplicationType(Tcl_Interp *interp, const char *fileName,
                    char *fullName);
static TclFile  MakeFile(HFILE handle);
static int      PipeBlockModeProc(ClientData instanceData, int mode);
static void     PipeCheckProc _ANSI_ARGS_((ClientData clientData,
                    int flags));
static int      PipeCloseProc(ClientData instanceData, Tcl_Interp *interp);
static int      PipeEventProc(Tcl_Event *evPtr, int flags);
static void     PipeExitHandler(ClientData clientData);
static int      PipeGetHandleProc(ClientData instanceData, int direction,
                    ClientData *handlePtr);
static void     PipeInit(void);
static int      PipeInputProc(ClientData instanceData, char *buf, int toRead,
                    int *errorCode);
static int      PipeOutputProc(ClientData instanceData, char *buf, int toWrite,
                    int *errorCode);
static void     PipeWatchProc(ClientData instanceData, int mask);
static void     PipeSetupProc _ANSI_ARGS_((ClientData clientData,
                    int flags));
static void     TempFileCleanup _ANSI_ARGS_((ClientData clientData));

/*
 * This structure describes the channel type structure for command pipe
 * based IO.
 */

static Tcl_ChannelType pipeChannelType = {
    "pipe",                     /* Type name. */
    PipeBlockModeProc,          /* Set blocking or non-blocking mode.*/
    PipeCloseProc,              /* Close proc. */
    PipeInputProc,              /* Input proc. */
    PipeOutputProc,             /* Output proc. */
    NULL,                       /* Seek proc. */
    NULL,                       /* Set option proc. */
    NULL,                       /* Get option proc. */
    PipeWatchProc,              /* Set up notifier to watch the channel. */
    PipeGetHandleProc,          /* Get an OS handle from channel. */
};
/*
 *----------------------------------------------------------------------
 *
 * PipeInit --
 *
 *      This function initializes the static variables for this file.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Creates a new event source.
 *
 *----------------------------------------------------------------------
 */

static void
PipeInit()
{
#ifdef VERBOSE
    printf("PipeInit\n");
    fflush(stdout);
#endif

    initialized = 1;
    firstPipePtr = NULL;
    Tcl_CreateEventSource(PipeSetupProc, PipeCheckProc, NULL);
    Tcl_CreateExitHandler(PipeExitHandler, NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * PipeExitHandler --
 *
 *      This function is called to cleanup the pipe module before
 *      Tcl is unloaded.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Removes the pipe event source.
 *
 *----------------------------------------------------------------------
 */

static void
PipeExitHandler(clientData)
    ClientData clientData;      /* Old window proc */
{
#ifdef VERBOSE
    printf("PipeExitHandler\n");
    fflush(stdout);
#endif

    Tcl_DeleteEventSource(PipeSetupProc, PipeCheckProc, NULL);
    initialized = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeSetupProc --
 *
 *      This procedure is invoked before Tcl_DoOneEvent blocks waiting
 *      for an event.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Adjusts the block time if needed.
 *
 *----------------------------------------------------------------------
 */

void
PipeSetupProc(data, flags)
    ClientData data;            /* Not used. */
    int flags;                  /* Event flags as passed to Tcl_DoOneEvent. */
{
    PipeInfo *infoPtr;
    Tcl_Time blockTime = { 0, 0 };

#ifdef VERBOSE
    printf("PipeSetupProc\n");
    fflush(stdout);
#endif

    if (!(flags & TCL_FILE_EVENTS)) {
        return;
    }

    /*
     * Check to see if there is a watched pipe.  If so, poll.
     */

    for (infoPtr = firstPipePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (infoPtr->watchMask) {
            Tcl_SetMaxBlockTime(&blockTime);
            break;
        }
    }
    return;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeCheckProc --
 *
 *      This procedure is called by Tcl_DoOneEvent to check the pipe
 *      event source for events.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      May queue an event.
 *
 *----------------------------------------------------------------------
 */

static void
PipeCheckProc(data, flags)
    ClientData data;            /* Not used. */
    int flags;                  /* Event flags as passed to Tcl_DoOneEvent. */
{
    PipeInfo *infoPtr;
    PipeEvent *evPtr;

#ifdef VERBOSE
    printf("PipeSetupProc\n");
    fflush(stdout);
#endif

    if (!(flags & TCL_FILE_EVENTS)) {
        return;
    }

    /*
     * Queue events for any watched pipes that don't already have events
     * queued.
     */

    for (infoPtr = firstPipePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (infoPtr->watchMask && !(infoPtr->flags & PIPE_PENDING)) {
            infoPtr->flags |= PIPE_PENDING;
            evPtr = (PipeEvent *) ckalloc(sizeof(PipeEvent));
            evPtr->header.proc = PipeEventProc;
            evPtr->infoPtr = infoPtr;
            Tcl_QueueEvent((Tcl_Event *) evPtr, TCL_QUEUE_TAIL);
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MakeFile --
 *
 *      This function constructs a new TclFile from a given data and
 *      type value.
 *
 * Results:
 *      Returns a newly allocated OS2File as a TclFile.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static TclFile
MakeFile(handle)
    HFILE handle;	/* Type-specific data. */
{
    OS2File *filePtr;

#ifdef VERBOSE
    printf("MakeFile handle [%d]\n", handle);
    fflush(stdout);
#endif

    filePtr = (OS2File *) ckalloc(sizeof(OS2File));
    if (filePtr != (OS2File *)NULL) {
        filePtr->type = OS2_FILE;
        filePtr->handle = handle;
    }

    return (TclFile)filePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpMakeFile --
 *
 *      Make a TclFile from a channel.
 *
 * Results:
 *      Returns a new TclFile or NULL on failure.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

TclFile
TclpMakeFile(channel, direction)
    Tcl_Channel channel;        /* Channel to get file from. */
    int direction;              /* Either TCL_READABLE or TCL_WRITABLE. */
{
    HFILE handle;

#ifdef VERBOSE
    printf("TclpMakeFile\n");
    fflush(stdout);
#endif

    if (Tcl_GetChannelHandle(channel, direction,
            (ClientData *) &handle) == TCL_OK) {
        return MakeFile(handle);
    } else {
        return (TclFile) NULL;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCreateTempFile --
 *
 *      This function opens a unique file with the property that it
 *      will be deleted when its file handle is closed.  The temporary
 *      file is created in the system temporary directory.
 *
 * Results:
 *      Returns a valid TclFile, or NULL on failure.
 *
 * Side effects:
 *      Creates a new temporary file.
 *
 *----------------------------------------------------------------------
 */

TclFile
TclpCreateTempFile(contents, namePtr)
    char *contents;             /* String to write into temp file, or NULL. */
    Tcl_DString *namePtr;       /* If non-NULL, pointer to initialized
                                 * DString that is filled with the name of
                                 * the temp file that was created. */
{
    char *name;
    HFILE handle;
    ULONG action, length, result;
    TmpFile *tmpFilePtr;

#ifdef VERBOSE
    printf("TclpCreateTempFile\n");
    fflush(stdout);
#endif


    /*
     * tempnam allocates space for name with *malloc*
     * use free() when deleting the file
     * First argument is directory that is used when the directory in the
     * TMP environment variable doesn't exist or the variable isn't set.
     * Make sure we can always create a temp file => c:\ .
     */
    name = tempnam("C:\\", "Tcl");
    if (name == (char *)NULL) return NULL;

    tmpFilePtr = (TmpFile *) ckalloc(sizeof(TmpFile));
    /*
     * See if we can get memory for later before creating the file to minimize
     * system interaction
     */
    if (tmpFilePtr == (TmpFile *)NULL) {
        /* We couldn't allocate memory, so free the tempname memory and abort */
        free((char *)name);
	return NULL;
    }

    rc = DosOpen(name, &handle, &action, 0, FILE_NORMAL,
                 OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS,
                 OPEN_SHARE_DENYNONE | OPEN_ACCESS_READWRITE, NULL);

#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("TclpCreateTempFile: DosOpen [%s] handle [%d] rc [%d]\n",
           name, handle, rc);
    fflush(stdout);
#endif
    if (rc != NO_ERROR) {
        goto error;
    }

    /*
     * Write the file out, doing line translations on the way.
     */

    if (contents != NULL) {
        char *p;

        for (p = contents; *p != '\0'; p++) {
            if (*p == '\n') {
                length = p - contents;
                if (length > 0) {
                    rc = DosWrite(handle, (PVOID)contents, length, &result);
#ifdef VERBOSE
                    printf("DosWrite handle %d [%s] returned [%d]\n",
                           handle, contents, rc);
    fflush(stdout);
#endif
                    if (rc != NO_ERROR) {
                        goto error;
                    }
                }
                if (DosWrite(handle, "\r\n", 2, &result) != NO_ERROR) {
                    goto error;
                }
                contents = p+1;
            }
        }
        length = p - contents;
        if (length > 0) {
            rc = DosWrite(handle, (PVOID)contents, length, &result);
#ifdef VERBOSE
            printf("DosWrite handle %d [%s] returned [%d]\n",
                   handle, contents, rc);
    fflush(stdout);
#endif
            if (rc != NO_ERROR) {
                goto error;
            }
        }
    }

    rc = DosSetFilePtr(handle, 0, FILE_BEGIN, &result);
#ifdef VERBOSE
    printf("TclpCreateTempFile: DosSetFilePtr BEGIN handle [%d] returns %d\n",
           handle, rc);
    fflush(stdout);
#endif
    if (rc != NO_ERROR) {
        goto error;
    }

    if (namePtr != NULL) {
        Tcl_DStringAppend(namePtr, name, -1);
    }

    /*
     * A file won't be deleted when it is closed, so we have to do it ourselves.
     */

    tmpFilePtr->file.type = OS2_TMPFILE;
    tmpFilePtr->file.handle = handle;
    tmpFilePtr->name = name;
    /* Queue undeleted files for removal on exiting Tcl */
    Tcl_CreateExitHandler(TempFileCleanup, (ClientData)tmpFilePtr);
    return (TclFile)tmpFilePtr;

  error:
    TclOS2ConvertError(rc);
    rc = DosClose(handle);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles--;
#endif
    rc = DosDelete(name);
#ifdef VERBOSE
    printf("DosDelete [%s] (was handle [%d] returns %d\n", name, handle, rc);
    fflush(stdout);
#endif
    ckfree((char *)tmpFilePtr);
    /* NB! EMX has allocated name with malloc, use free! */
    free((char *)name);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpOpenFile --
 *
 *      This function opens files for use in a pipeline.
 *
 * Results:
 *      Returns a newly allocated TclFile structure containing the
 *      file handle.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

TclFile
TclpOpenFile(path, mode)
    char *path;
    int mode;
{
    HFILE handle;
    ULONG accessMode, createMode, flags, exist, result;

#ifdef VERBOSE
    printf("TclpOpenFile\n");
    fflush(stdout);
#endif

    /*
     * Map the access bits to the OS/2 access mode.
     */

    switch (mode & (O_RDONLY | O_WRONLY | O_RDWR)) {
        case O_RDONLY:
           accessMode = OPEN_ACCESS_READONLY;
           break;
       case O_WRONLY:
           accessMode = OPEN_ACCESS_WRITEONLY;
           break;
       case O_RDWR:
           accessMode = OPEN_ACCESS_READWRITE;
           break;
       default:
           TclOS2ConvertError(ERROR_INVALID_FUNCTION);
           return NULL;
    }

    /*
     * Map the creation flags to the OS/2 open mode.
     */

    switch (mode & (O_CREAT | O_EXCL | O_TRUNC)) {
        case (O_CREAT | O_EXCL | O_TRUNC):
            createMode = OPEN_ACTION_CREATE_IF_NEW |
                         OPEN_ACTION_FAIL_IF_EXISTS;
            break;
        case (O_CREAT | O_EXCL):
            createMode = OPEN_ACTION_CREATE_IF_NEW |
                         OPEN_ACTION_FAIL_IF_EXISTS;
            break;
        case (O_CREAT | O_TRUNC):
            createMode = OPEN_ACTION_CREATE_IF_NEW |
                         OPEN_ACTION_REPLACE_IF_EXISTS;
            break;
        case O_CREAT:
            createMode = OPEN_ACTION_CREATE_IF_NEW |
                         OPEN_ACTION_OPEN_IF_EXISTS;
            break;
        case O_TRUNC:
        case (O_TRUNC | O_EXCL):
            createMode = OPEN_ACTION_FAIL_IF_NEW |
                         OPEN_ACTION_REPLACE_IF_EXISTS;
            break;
        default:
            createMode = OPEN_ACTION_FAIL_IF_NEW |
                         OPEN_ACTION_OPEN_IF_EXISTS;
    }

    /*
     * If the file is not being created, use the existing file attributes.
     */

    flags = 0;
    if (!(mode & O_CREAT)) {
        FILESTATUS3 infoBuf;

        rc = DosQueryPathInfo(path, FIL_STANDARD, &infoBuf, sizeof(infoBuf));
        if (rc == NO_ERROR) {
            flags = infoBuf.attrFile;
        } else {
            flags = 0;
        }
    }


    /*
     * Set up the attributes so this file is not inherited by child processes.
     */

    accessMode |= OPEN_FLAGS_NOINHERIT;

    /*
     * Set up the file sharing mode.  We want to allow simultaneous access.
     */

    accessMode |= OPEN_SHARE_DENYNONE;

    /*
     * Now we get to create the file.
     */

    rc = DosOpen(path, &handle, &exist, 0, flags, createMode, accessMode,
                 (PEAOP2)NULL);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("TclOpenFile: DosOpen [%s] returns [%d]\n", path, rc);
    fflush(stdout);
#endif
    if (rc != NO_ERROR) {
        ULONG err = 0;

        switch (rc) {
            case ERROR_FILE_NOT_FOUND:
            case ERROR_PATH_NOT_FOUND:
                err = ERROR_FILE_NOT_FOUND;
                break;
            case ERROR_ACCESS_DENIED:
            case ERROR_INVALID_ACCESS:
            case ERROR_SHARING_VIOLATION:
            case ERROR_CANNOT_MAKE:
                err = (mode & O_CREAT) ? ERROR_FILE_EXISTS
                                       : ERROR_FILE_NOT_FOUND;
                break;
        }
        TclOS2ConvertError(err);
        return NULL;
    }

    /*
     * Seek to the end of file if we are writing.
     */

    if (mode & O_WRONLY) {
        rc = DosSetFilePtr(handle, 0, FILE_END, &result);
    }

    return MakeFile(handle);
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCreatePipe --
 *
 *      Creates an anonymous pipe.
 *
 * Results:
 *      Returns 1 on success, 0 on failure. 
 *
 * Side effects:
 *      Creates a pipe.
 *
 *----------------------------------------------------------------------
 */

int
TclpCreatePipe(readPipe, writePipe)
    TclFile *readPipe;         /* Location to store file handle for
                                * read side of pipe. */
    TclFile *writePipe;        /* Location to store file handle for
                                * write side of pipe. */
{
    HFILE readHandle, writeHandle;

#ifdef VERBOSE
    printf("TclpCreatePipe\n");
    fflush(stdout);
#endif

    /*
     * Using 1024 makes for processes hanging around until their output was
     * read, which doesn't always happen (eg. the "defs" file in the test
     * suite). The Control Program Reference gives 4096 in an example, which
     * "happens" to be the page size of the Intel x86.
     */
    rc = DosCreatePipe(&readHandle, &writeHandle, 4096);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles += 2;
    printf("DosCreatePipe returned [%d], read [%d], write [%d]\n",
           rc, readHandle, writeHandle);
    fflush(stdout);
#endif

    if (rc == NO_ERROR) {
        *readPipe = MakeFile(readHandle);
        *writePipe = MakeFile(writeHandle);
        return 1;
    }

    TclOS2ConvertError(rc);
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCloseFile --
 *
 *      Closes a pipeline file handle.  These handles are created by
 *      TclpOpenFile, TclpCreatePipe, or TclpMakeFile.
 *
 * Results:
 *      0 on success, -1 on failure.
 *
 * Side effects:
 *      The file is closed and deallocated.
 *
 *----------------------------------------------------------------------
 */

int
TclpCloseFile(file)
    TclFile file;       /* The file to close. */
{
    OS2File *filePtr = (OS2File *) file;

#ifdef VERBOSE
    printf("TclpCloseFile [%d] type %d\n", filePtr->handle, filePtr->type);
    fflush(stdout);
#endif
    if (filePtr->type < OS2_FIRST_TYPE || filePtr->type > OS2_LAST_TYPE) {
        panic("Tcl_CloseFile: unexpected file type");
    }

    rc = DosClose(filePtr->handle);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles--;
    printf("TclpCloseFile: DosClose [%d] returns %d\n", filePtr->handle, rc);
    fflush(stdout);
#endif
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        ckfree((char *) filePtr);
        return -1;
    }

    ckfree((char *) filePtr);
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCreateProcess --
 *
 *      Create a child process that has the specified files as its
 *      standard input, output, and error.  The child process runs
 *      asynchronously and runs with the same environment variables
 *      as the creating process.
 *
 *      The complete OS/2 search path is searched to find the specified
 *      executable.  If an executable by the given name is not found,
 *      automatically tries appending ".com", ".exe", and ".bat" to the
 *      executable name.
 *
 * Results:
 *      The return value is TCL_ERROR and an error message is left in
 *      interp->result if there was a problem creating the child
 *      process.  Otherwise, the return value is TCL_OK and *pidPtr is
 *      filled with the process id of the child process.
 *
 * Side effects:
 *      A process is created.
 *
 *----------------------------------------------------------------------
 */

int
TclpCreateProcess(interp, argc, argv, inputFile, outputFile, errorFile,
        pidPtr)
    Tcl_Interp *interp;         /* Interpreter in which to leave errors that
                                 * occurred when creating the child process.
                                 * Error messages from the child process
                                 * itself are sent to errorFile. */
    int argc;                   /* Number of arguments in following array. */
    char **argv;                /* Array of argument strings.  argv[0]
                                 * contains the name of the executable
                                 * converted to native format (using the
                                 * Tcl_TranslateFileName call).  Additional
                                 * arguments have not been converted. */
    TclFile inputFile;          /* If non-NULL, gives the file to use as
                                 * input for the child process.  If inputFile
                                 * file is not readable or is NULL, the child
                                 * will receive no standard input. */
    TclFile outputFile;         /* If non-NULL, gives the file that
                                 * receives output from the child process.  If
                                 * outputFile file is not writeable or is
                                 * NULL, output from the child will be
                                 * discarded. */
    TclFile errorFile;          /* If non-NULL, gives the file that
                                 * receives errors from the child process.  If
                                 * errorFile file is not writeable or is NULL,
                                 * errors from the child will be discarded.
                                 * errorFile may be the same as outputFile. */
    Tcl_Pid *pidPtr;            /* If this procedure is successful, pidPtr
                                 * is filled with the process id of the child
                                 * process. */
{
    int result, applType, nextArg, count, mode;
    HFILE inputHandle, outputHandle, errorHandle;
    HFILE stdIn = HF_STDIN, stdOut = HF_STDOUT, stdErr = HF_STDERR;
    HFILE orgIn = NEW_HANDLE, orgOut = NEW_HANDLE, orgErr = NEW_HANDLE;
    BOOL stdinChanged, stdoutChanged, stderrChanged;
    char execPath[MAX_PATH];
    char *originalName = NULL;
    OS2File *filePtr;
    ULONG action;
    char *arguments[256];

#ifdef VERBOSE
    int remember[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    printf("TclpCreateProcess\n");
    fflush(stdout);
#endif

    if (!initialized) {
        PipeInit();
    }

    applType = ApplicationType(interp, argv[0], execPath);
    if (applType != APPL_NONE) {
#ifdef VERBOSE
        printf("argv[0] %s, execPath %s\n", argv[0], execPath);
        fflush(stdout);
#endif
        originalName = argv[0];
        argv[0] = execPath;
    }

    result = TCL_ERROR;

#ifdef VERBOSE
    printf("1 opened files %d\n", openedFiles);;
    fflush(stdout);
#endif
    /* Backup original stdin, stdout, stderr by Dup-ing to new handle */
    rc = DosDupHandle(stdIn, &orgIn);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("2 opened files %d\n", openedFiles);;
    printf("DosDupHandle stdIn %d returns %d orgIn %d\n", stdIn, rc, orgIn);
    fflush(stdout);
#endif
    rc = DosDupHandle(stdOut, &orgOut);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("3 opened files %d\n", openedFiles);;
    printf("DosDupHandle stdOut %d returns %d, orgOut %d\n", stdOut, rc,
           orgOut);
    fflush(stdout);
#endif
    rc = DosDupHandle(stdErr, &orgErr);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("4 opened files %d\n", openedFiles);;
    printf("DosDupHandle stdErr %d returns %d orgErr %d\n", stdErr, rc,
           orgErr);
    fflush(stdout);
#endif

    /*
     * We have to check the type of each file, since we cannot duplicate
     * some file types.
     */

    inputHandle = NEW_HANDLE;
    if (inputFile != NULL) {
        filePtr = (OS2File *)inputFile;
        if (filePtr->type >= OS2_FIRST_TYPE || filePtr->type <= OS2_LAST_TYPE) {
            inputHandle = filePtr->handle;
        }
    }
    outputHandle = NEW_HANDLE;
    if (outputFile != NULL) {
        filePtr = (OS2File *)outputFile;
        if (filePtr->type >= OS2_FIRST_TYPE || filePtr->type <= OS2_LAST_TYPE) {
            outputHandle = filePtr->handle;
        }
    }
    errorHandle = NEW_HANDLE;
    if (errorFile != NULL) {
        filePtr = (OS2File *)errorFile;
        if (filePtr->type >= OS2_FIRST_TYPE || filePtr->type <= OS2_LAST_TYPE) {
            errorHandle = filePtr->handle;
        }
    }
#ifdef VERBOSE
    printf("    inputHandle [%d]\n", inputHandle);
    printf("    outputHandle [%d]\n", outputHandle);
    printf("    errorHandle [%d]\n", errorHandle);
    fflush(stdout);
#endif

    /*
     * Duplicate all the handles which will be passed off as stdin, stdout
     * and stderr of the child process. The duplicate handles are set to
     * be inheritable, so the child process can use them.
     */

    stdinChanged = stdoutChanged = stderrChanged = FALSE;
    if (inputHandle == NEW_HANDLE) {
        /*
         * If handle was not set, open NUL as input.
         */
#ifdef VERBOSE
        printf("Opening NUL as input\n");
        fflush(stdout);
#endif
        rc = DosOpen((PSZ)"NUL", &inputHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, NULL);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("5 opened files %d\n", openedFiles);;
    fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as input handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("1 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
    }
    if (inputHandle != stdIn) {
        /* Duplicate to standard input handle */
        rc = DosDupHandle(inputHandle, &stdIn);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("6 opened files %d\n", openedFiles);;
        printf("DosDupHandle inputHandle [%d] returned [%d], handle [%d]\n",
               inputHandle, rc, stdIn);
        fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate input handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("2 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
        stdinChanged = TRUE;
    }

    if (outputHandle == NEW_HANDLE) {
        /*
         * If handle was not set, open NUL as output.
         */
#ifdef VERBOSE
        printf("Opening NUL as output\n");
        fflush(stdout);
#endif
        rc = DosOpen((PSZ)"NUL", &outputHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_WRITEONLY, NULL);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("7 opened files %d\n", openedFiles);;
    fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as output handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("3 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
    }
    if (outputHandle != stdOut) {
        /* Duplicate to standard output handle */
        rc = DosDupHandle(outputHandle, &stdOut);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("8 opened files %d\n", openedFiles);;
        printf("DosDupHandle outputHandle [%d] returned [%d], handle [%d]\n",
               outputHandle, rc, stdOut);
        fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate output handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("4 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
        stdoutChanged = TRUE;
    }

    if (errorHandle == NEW_HANDLE) {
        /*
         * If handle was not set, open NUL as output.
         */
#ifdef VERBOSE
        printf("Opening NUL as error\n");
        fflush(stdout);
#endif
        rc = DosOpen((PSZ)"NUL", &errorHandle, &action, 0, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW,
                     OPEN_SHARE_DENYNONE | OPEN_ACCESS_WRITEONLY, NULL);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("9 opened files %d\n", openedFiles);;
    fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't open NUL as error handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("5 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
    }
    if (errorHandle != stdErr) {
        /* Duplicate to standard error handle */
        rc = DosDupHandle(errorHandle, &stdErr);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles++;
        printf("10 opened files %d\n", openedFiles);;
        printf("DosDupHandle errorHandle [%d] returned [%d], handle [%d]\n",
               errorHandle, rc, stdErr);
        fflush(stdout);
#endif
        if (rc != NO_ERROR) {
            TclOS2ConvertError(rc);
            Tcl_AppendResult(interp, "couldn't duplicate error handle: ",
                    Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("6 goto end\n");;
    fflush(stdout);
#endif
            goto end;
        }
        stderrChanged = TRUE;
    }

    /*
     * EMX's spawnv handles all the nitty-gritty DosStartSession stuff (like
     * session/no-session, building environment and arguments with/without
     * quoting etc.) for us, so we'll just use that and keep ourselves to
     * it's arguments like P_WAIT, P_PM, ....
     */

    /*
     * Run DOS (incl. .bat) and .cmd files via cmd.exe.
     */

    mode = P_SESSION;
    nextArg = 0;

    switch (applType) {
    case APPL_NONE:
    case APPL_DOS:
    case APPL_OS2CMD:
    case APPL_WIN3X:
        arguments[0] = "cmd.exe";
        arguments[1] = "/c";
        nextArg = 2;
        mode |= P_DEFAULT | P_MINIMIZE | P_BACKGROUND;
        break;
    case APPL_OS2WIN:
        mode |= P_DEFAULT | P_MINIMIZE | P_BACKGROUND;
        break;
    case APPL_OS2FS:
        mode |= P_FULLSCREEN | P_BACKGROUND;
        break;
    case APPL_OS2PM:
        if (usePm) {
            mode = P_PM | P_BACKGROUND;
        } else {
            mode |= P_PM | P_BACKGROUND;
        }
        break;
    default:
        mode |= P_DEFAULT | P_MINIMIZE | P_BACKGROUND;
    }
    for (count = 0; count < argc && nextArg < 256; count++) {
        arguments[nextArg] = argv[count];
        nextArg++;
    }
    arguments[nextArg] = '\0';

    *pidPtr = (Tcl_Pid) spawnv(mode,
                               arguments[0], arguments);
    if (*pidPtr == (Tcl_Pid) -1) {
        TclOS2ConvertError(rc);
        Tcl_AppendResult(interp, "couldn't execute \"", originalName,
                "\": ", Tcl_PosixError(interp), (char *) NULL);
#ifdef VERBOSE
            printf("1 goto end\n");;
    fflush(stdout);
#endif
        goto end;
    }
#ifdef VERBOSE
    printf("spawned pid %d\n", *pidPtr);
    fflush(stdout);
#endif

    result = TCL_OK;


    end:

    /* Restore original stdin, stdout, stderr by Dup-ing from new handle */
    stdIn = HF_STDIN; stdOut = HF_STDOUT; stdErr = HF_STDERR;
#ifdef VERBOSE
    remember[0] = stdinChanged;
    remember[1] = stdoutChanged;
    remember[2] = stderrChanged;
#endif
    if (stdinChanged) {
        rc = DosClose(stdIn);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles--;
        printf("11 opened files %d\n", openedFiles);;
        remember[3] = rc;
    fflush(stdout);
#endif
        rc = DosDupHandle(orgIn, &stdIn);
#ifdef VERBOSE
        remember[4] = rc;
#endif
    }
    rc = DosClose(orgIn);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles--;
    printf("12 opened files %d\n", openedFiles);;
    remember[5] = rc;
    fflush(stdout);
#endif

    if (stdoutChanged) {
        rc = DosClose(stdOut);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles--;
        printf("13 opened files %d\n", openedFiles);;
        remember[6] = rc;
    fflush(stdout);
#endif
        rc = DosDupHandle(orgOut, &stdOut);
#ifdef VERBOSE
        remember[7] = rc;
#endif
    }
    rc = DosClose(orgOut);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles--;
    printf("14 opened files %d\n", openedFiles);;
    remember[8] = rc;
    printf("stdinChanged %d, stdoutChanged %d, stderrChanged %d\n",
           remember[0], remember[1], remember[2]);
    if (remember[0]) {
        printf("DosClose \"new\" stdIn [%d] returned [%d]\n", stdIn,
               remember[3]);
        printf("DosDupHandle orgIn [%d] returned [%d]\n", orgIn,
               remember[4]);
        printf("DosClose orgIn [%d] returned [%d]\n", orgIn,
               remember[5]);
    }
    if (remember[1]) {
        printf("DosClose \"new\" stdOut [%d] returned [%d]\n", stdOut,
               remember[6]);
        printf("DosDupHandle orgOut [%d] returned [%d]\n", orgOut,
               remember[7]);
        printf("DosClose orgOut [%d] returned [%d]\n", orgOut,
               remember[8]);
    }
    fflush(stdout);
#endif

    if (stderrChanged) {
        rc = DosClose(stdErr);
#ifdef VERBOSE
        if (rc == NO_ERROR) openedFiles--;
        printf("15 opened files %d\n", openedFiles);;
        printf("DosClose \"new\" stdErr [%d] returned [%d]\n", stdErr, rc);
        fflush(stdout);
#endif
        rc = DosDupHandle(orgErr, &stdErr);
#ifdef VERBOSE
        printf("DosDupHandle orgErr [%d] returned [%d]\n", orgErr, rc);
        fflush(stdout);
#endif
    }
    rc = DosClose(orgErr);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles--;
    printf("16 opened files %d\n", openedFiles);;
    fflush(stdout);
#endif

    return result;
}

/*
 *--------------------------------------------------------------------
 *
 * ApplicationType --
 *
 *      Search for the specified program and identify if it refers to a DOS,
 *      Windows 3.x, OS/2 Windowable, OS/2 Full-Screen, OS/2 PM program.
 *      Used to determine how to invoke a program (if it can even be invoked).
 * Results:
 *      The return value is one of APPL_DOS, APPL_WIN3X, or APPL_WIN32
 *      if the filename referred to the corresponding application type.
 *      If the file name could not be found or did not refer to any known
 *      application type, APPL_NONE is returned and an error message is
 *      left in interp.  .bat files are identified as APPL_DOS.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
ApplicationType(interp, originalName, fullPath)
    Tcl_Interp *interp;         /* Interp, for error message. */
    const char *originalName;   /* Name of the application to find. */
    char fullPath[MAX_PATH];    /* Filled with complete path to
                                 * application. */
{
    int applType, i;
    char *ext;
    static char *extensions[] = {"", ".cmd", ".exe", ".bat", ".com", NULL};
    char tmpPath[MAX_PATH];
    FILESTATUS3 filestat;
    ULONG flags;

#ifdef VERBOSE
    printf("ApplicationType\n");
    fflush(stdout);
#endif

    applType = APPL_NONE;
    for (i = 0; extensions[i] != NULL; i++) {
        strncpy((char *)tmpPath, originalName, MAX_PATH - 5);
#ifdef VERBOSE
        printf("after strncpy, tmpPath %s, originalName %s\n", tmpPath,
                originalName);
        fflush(stdout);
#endif
        strcat(tmpPath, extensions[i]);
#ifdef VERBOSE
        printf("after strcat, tmpPath %s, extensions[%d] [%s]\n", tmpPath,
                i, extensions[i]);
        fflush(stdout);
#endif

        if (tmpPath[1] != ':' && tmpPath[0] != '\\') {
            rc = DosSearchPath(SEARCH_ENVIRONMENT | SEARCH_CUR_DIRECTORY |
                               SEARCH_IGNORENETERRS, "PATH", tmpPath,
                               (PBYTE)fullPath, MAXPATH);
            if (rc != NO_ERROR) {
#ifdef VERBOSE
                printf("DosSearchPath %s ERROR %d\n", tmpPath, rc);
                fflush(stdout);
#endif
                continue;
            }
#ifdef VERBOSE
            printf("DosSearchPath %s OK (%s)\n", tmpPath, fullPath);
            fflush(stdout);
#endif
        } else {
            strcpy(fullPath, tmpPath);
        }

        /*
         * Ignore matches on directories or data files, return if identified
         * a known type.
         */

        rc = DosQueryPathInfo(fullPath, FIL_STANDARD, &filestat,
                              sizeof(FILESTATUS3));
        if (rc != NO_ERROR || filestat.attrFile & FILE_DIRECTORY) {
            continue;
        }

        rc = DosQueryAppType(fullPath, &flags);
        if (rc != NO_ERROR) {
            continue;
        } 

        if ((flags & FAPPTYP_DLL) || (flags & FAPPTYP_PHYSDRV) ||
            (flags & FAPPTYP_VIRTDRV) || (flags & FAPPTYP_PROTDLL)) {
            /* No executable */
            continue;
        } 

        if (flags & FAPPTYP_NOTSPEC) {

            /* Not a recognized type, try to see if it's a .cmd or .bat */
            ext = strrchr(fullPath, '.');
            if ((ext != NULL) && (stricmp(ext, ".cmd") == 0)) {
                applType = APPL_OS2CMD;
                break;
            }
            if ((ext != NULL) && (stricmp(ext, ".bat") == 0)) {
                applType = APPL_DOS;
                break;
            }
            /* Still not recognized, might be a Win32 PE-executable */
            applType = APPL_NONE;
            break;
        }

        /*
         * NB! Some bozo defined FAPPTYP_WINDOWAPI as 0x0003 instead of 0x0004,
         * thereby causing it to have bits in common with both
         * FAPPTYP_NOTWINDOWCOMPAT (0x0001) and FAPPTYP_WINDOWCOMPAT (0x0002),
         * which means that for any OS/2 app, you get "PM app" as answer if
         * you don't take extra measures.
         * This is found in EMX 0.9c, 0.9b *AND* in IBM's own Visual Age C++
         * 3.0, so I must assume Eberhard Mattes was forced to follow the
         * drunk that defined these defines in the LX format....
         */
        if (flags & FAPPTYP_NOTWINDOWCOMPAT) {
            applType = APPL_OS2FS;
        }
        if (flags & FAPPTYP_WINDOWCOMPAT) {
            applType = APPL_OS2WIN;
        }
        /*
         * This won't work:
        if (flags & FAPPTYP_WINDOWAPI) {
            applType = APPL_OS2PM;
        }
         * Modified version:
         */
        if ((flags & FAPPTYP_NOTWINDOWCOMPAT)
            && (flags & FAPPTYP_WINDOWCOMPAT)) {
            applType = APPL_OS2PM;
        }
        if (flags & FAPPTYP_DOS) {
            applType = APPL_DOS;
        }
        if ((flags & FAPPTYP_WINDOWSREAL) || (flags & FAPPTYP_WINDOWSPROT)
            || (flags & FAPPTYP_WINDOWSPROT31)) {
            applType = APPL_WIN3X;
        }

        break;
    }

    if (applType == APPL_NONE) {
#ifdef VERBOSE
        printf("ApplicationType: APPL_NONE\n");
    fflush(stdout);
#endif
        TclOS2ConvertError(rc);
        Tcl_AppendResult(interp, "couldn't execute \"", originalName,
                "\": ", Tcl_PosixError(interp), (char *) NULL);
        return APPL_NONE;
    }
    return applType;
}

/*
 *----------------------------------------------------------------------
 *
 * TclpCreateCommandChannel --
 *
 *      This function is called by Tcl_OpenCommandChannel to perform
 *      the platform specific channel initialization for a command
 *      channel.
 *
 * Results:
 *      Returns a new channel or NULL on failure.
 *
 * Side effects:
 *      Allocates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
TclpCreateCommandChannel(readFile, writeFile, errorFile, numPids, pidPtr)
    TclFile readFile;           /* If non-null, gives the file for reading. */
    TclFile writeFile;          /* If non-null, gives the file for writing. */
    TclFile errorFile;          /* If non-null, gives the file where errors
                                 * can be read. */
    int numPids;                /* The number of pids in the pid array. */
    Tcl_Pid *pidPtr;            /* An array of process identifiers. */
{
    char channelName[20];
    int channelId;
    PipeInfo *infoPtr = (PipeInfo *) ckalloc((unsigned) sizeof(PipeInfo));

#ifdef VERBOSE
    printf("TclpCreateCommandChannel\n");
    fflush(stdout);
#endif

    if (!initialized) {
        PipeInit();
    }

    infoPtr->watchMask = 0;
    infoPtr->flags = 0;
    infoPtr->readFile = readFile;
    infoPtr->writeFile = writeFile;
    infoPtr->errorFile = errorFile;
    infoPtr->numPids = numPids;
    infoPtr->pidPtr = pidPtr;

    /*
     * Use one of the fds associated with the channel as the
     * channel id.
     */

    if (readFile) {
        channelId = (int) ((OS2File*)readFile)->handle;
    } else if (writeFile) {
        channelId = (int) ((OS2File*)writeFile)->handle;
    } else if (errorFile) {
        channelId = (int) ((OS2File*)errorFile)->handle;
    } else {
        channelId = 0;
    }

    infoPtr->validMask = 0;
    if (readFile != NULL) {
        infoPtr->validMask |= TCL_READABLE;
    }
    if (writeFile != NULL) {
        infoPtr->validMask |= TCL_WRITABLE;
    }

    /*
     * For backward compatibility with previous versions of Tcl, we
     * use "file%d" as the base name for pipes even though it would
     * be more natural to use "pipe%d".
     */

    sprintf(channelName, "file%d", channelId);
    infoPtr->channel = Tcl_CreateChannel(&pipeChannelType, channelName,
            (ClientData) infoPtr, infoPtr->validMask);

    /*
     * Pipes have AUTO translation mode on OS/2 and ^Z eof char, which
     * means that a ^Z will be appended to them at close. This is needed
     * for programs that expect a ^Z at EOF.
     */

    Tcl_SetChannelOption((Tcl_Interp *) NULL, infoPtr->channel,
            "-translation", "auto");
    Tcl_SetChannelOption((Tcl_Interp *) NULL, infoPtr->channel,
            "-eofchar", "\032 {}");
    return infoPtr->channel;
}

/*
 *----------------------------------------------------------------------
 *
 * TclGetAndDetachPids --
 *
 *      Stores a list of the command PIDs for a command channel in
 *      interp->result.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Modifies interp->result.
 *
 *----------------------------------------------------------------------
 */

void
TclGetAndDetachPids(interp, chan)
    Tcl_Interp *interp;
    Tcl_Channel chan;
{
    PipeInfo *pipePtr;
    Tcl_ChannelType *chanTypePtr;
    int i;
    char buf[20];

#ifdef VERBOSE
    printf("TclGetAndDetachPids\n");
    fflush(stdout);
#endif

    /*
     * Punt if the channel is not a command channel.
     */

    chanTypePtr = Tcl_GetChannelType(chan);
    if (chanTypePtr != &pipeChannelType) {
        return;
    }

    pipePtr = (PipeInfo *) Tcl_GetChannelInstanceData(chan);
    for (i = 0; i < pipePtr->numPids; i++) {
        sprintf(buf, "%lu", (unsigned long) pipePtr->pidPtr[i]);
        Tcl_AppendElement(interp, buf);
        Tcl_DetachPids(1, &(pipePtr->pidPtr[i]));
    }
    if (pipePtr->numPids > 0) {
        ckfree((char *) pipePtr->pidPtr);
        pipePtr->numPids = 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PipeBlockModeProc --
 *
 *      Set blocking or non-blocking mode on channel.
 *
 * Results:
 *      0 if successful, errno when failed.
 *
 * Side effects:
 *      Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

static int
PipeBlockModeProc(instanceData, mode)
    ClientData instanceData;    /* Instance data for channel. */
    int mode;                   /* TCL_MODE_BLOCKING or
                                 * TCL_MODE_NONBLOCKING. */
{
    PipeInfo *infoPtr = (PipeInfo *) instanceData;

#ifdef VERBOSE
    printf("PipeBlockModeProc\n");
    fflush(stdout);
#endif

    /*
     * Unnamed pipes on OS/2 can not be switched between blocking and
     * nonblocking, hence we have to emulate the behavior. This is done in
     * the input function by checking against a bit in the state. We set or
     * unset the bit here to cause the input function to emulate the correct
     * behavior.
     */

    if (mode == TCL_MODE_NONBLOCKING) {
        infoPtr->flags |= PIPE_ASYNC;
    } else {
        infoPtr->flags &= ~(PIPE_ASYNC);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeCloseProc --
 *
 *      Closes a pipe based IO channel.
 *
 * Results:
 *      0 on success, errno otherwise.
 *
 * Side effects:
 *      Closes the physical channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeCloseProc(instanceData, interp)
    ClientData instanceData;    /* Pointer to PipeInfo structure. */
    Tcl_Interp *interp;         /* For error reporting. */
{
    PipeInfo *pipePtr = (PipeInfo *) instanceData;
    Tcl_Channel errChan;
    int errorCode, result;
    PipeInfo *infoPtr, **nextPtrPtr;

#ifdef VERBOSE
    printf("PipeCloseProc\n");
    fflush(stdout);
#endif

    /*
     * Remove the file from the list of watched files.
     */

    for (nextPtrPtr = &firstPipePtr, infoPtr = *nextPtrPtr; infoPtr != NULL;
            nextPtrPtr = &infoPtr->nextPtr, infoPtr = *nextPtrPtr) {
        if (infoPtr == (PipeInfo *)pipePtr) {
            *nextPtrPtr = infoPtr->nextPtr;
            break;
        }
    }

    errorCode = 0;
    if (pipePtr->readFile != NULL) {
#ifdef VERBOSE
        printf("PipeCloseProc closing readFile\n");
        fflush(stdout);
#endif
        if (TclpCloseFile(pipePtr->readFile) != 0) {
            errorCode = errno;
        }
    }
    if (pipePtr->writeFile != NULL) {
#ifdef VERBOSE
        printf("PipeCloseProc closing writeFile\n");
        fflush(stdout);
#endif
        if (TclpCloseFile(pipePtr->writeFile) != 0) {
            if (errorCode == 0) {
                errorCode = errno;
            }
        }
    }

    /*
     * Wrap the error file into a channel and give it to the cleanup
     * routine.
     */

    if (pipePtr->errorFile) {
        OS2File *filePtr;
        filePtr = (OS2File *)pipePtr->errorFile;
        errChan = Tcl_MakeFileChannel((ClientData) filePtr->handle,
                TCL_READABLE);
/*
        TclpCloseFile(pipePtr->errorFile);
*/
    } else {
        errChan = NULL;
    }
    result = TclCleanupChildren(interp, pipePtr->numPids, pipePtr->pidPtr,
            errChan);
    if (pipePtr->numPids > 0) {
        ckfree((char *) pipePtr->pidPtr);
    }
    ckfree((char*) pipePtr);

    if (errorCode == 0) {
        return result;
    }
    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeInputProc --
 *
 *      Reads input from the IO channel into the buffer given. Returns
 *      count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *      A count of how many bytes were read is returned and an error
 *      indication is returned in an output argument.
 *
 * Side effects:
 *      Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeInputProc(instanceData, buf, bufSize, errorCode)
    ClientData instanceData;            /* Pipe state. */
    char *buf;                          /* Where to store data read. */
    int bufSize;                        /* How much space is available
                                         * in the buffer? */
    int *errorCode;                     /* Where to store error code. */
{
    PipeInfo *infoPtr = (PipeInfo *) instanceData;
    OS2File *filePtr = (OS2File*) infoPtr->readFile;
    ULONG bytesRead;
#if 0
    BYTE output[200];
    AVAILDATA avail;
    ULONG state;
#endif

#ifdef VERBOSE
    printf("PipeInputProc\n");
    fflush(stdout);
#endif

    *errorCode = 0;

    /*
     * Pipes will block until the requested number of bytes has been
     * read.  To avoid blocking unnecessarily, we look ahead and only
     * read as much as is available.
     */

#if 0

    rc = DosPeekNPipe(filePtr->handle, &output, bufSize, &bytesRead, &avail,
                      &state);
#ifdef VERBOSE
    printf("DosPeekNPipe %d returns %d\n", filePtr->handle, rc);
    fflush(stdout);
#endif
    if (rc == NO_ERROR) {
        if ((bytesRead != 0) && ((USHORT) bufSize > avail.cbpipe)) {
            bufSize = (int) bytesRead;
        } else if ((bytesRead == 0) && (infoPtr->flags & PIPE_ASYNC)) {
            errno = *errorCode = EAGAIN;
            return -1;
        } else if ((bytesRead == 0) && !(infoPtr->flags & PIPE_ASYNC)) {
            bufSize = 1;
        }
    } else {
        goto error;
    }

#endif

    /*
     * Note that we will block on reads from a console buffer until a
     * full line has been entered.  The only way I know of to get
     * around this is to write a console driver.  We should probably
     * do this at some point, but for now, we just block.
     */

    rc = DosRead(filePtr->handle, (PVOID) buf, (ULONG) bufSize, &bytesRead);
#ifdef VERBOSE
    { int i;
    printf("DosRead handle [%d] returns [%d], bytes read [%d]\n",
           filePtr->handle, rc, bytesRead);
    fflush(stdout);
    }
#endif
    if (rc != NO_ERROR) {
        goto error;
    }

    return bytesRead;

error:
    TclOS2ConvertError(rc);
    if (errno == EPIPE) {
        return 0;
    }
    *errorCode = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeOutputProc --
 *
 *      Writes the given output on the IO channel. Returns count of how
 *      many characters were actually written, and an error indication.
 *
 * Results:
 *      A count of how many characters were written is returned and an
 *      error indication is returned in an output argument.
 *
 * Side effects:
 *      Writes output on the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
PipeOutputProc(instanceData, buf, toWrite, errorCode)
    ClientData instanceData;            /* Pipe state. */
    char *buf;                          /* The data buffer. */
    int toWrite;                        /* How many bytes to write? */
    int *errorCode;                     /* Where to store error code. */
{
    PipeInfo *infoPtr = (PipeInfo *) instanceData;
    OS2File *filePtr = (OS2File *) infoPtr->writeFile;
    ULONG bytesWritten;

#ifdef VERBOSE
    printf("PipeOutputProc\n");
    fflush(stdout);
#endif

    *errorCode = 0;
    rc = DosWrite(filePtr->handle, (PVOID) buf, (ULONG) toWrite, &bytesWritten);
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        if (errno == EPIPE) {
            return 0;
        }
        *errorCode = errno;
        return -1;
    }
    return bytesWritten;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeEventProc --
 *
 *      This function is invoked by Tcl_ServiceEvent when a file event
 *      reaches the front of the event queue.  This procedure invokes
 *      Tcl_NotifyChannel on the pipe.
 *
 * Results:
 *      Returns 1 if the event was handled, meaning it should be removed
 *      from the queue.  Returns 0 if the event was not handled, meaning
 *      it should stay on the queue.  The only time the event isn't
 *      handled is if the TCL_FILE_EVENTS flag bit isn't set.
 *
 * Side effects:
 *      Whatever the notifier callback does.
 *
 *----------------------------------------------------------------------
 */

static int
PipeEventProc(evPtr, flags)
    Tcl_Event *evPtr;           /* Event to service. */
    int flags;                  /* Flags that indicate what events to
                                 * handle, such as TCL_FILE_EVENTS. */
{
    PipeEvent *pipeEvPtr = (PipeEvent *)evPtr;
    PipeInfo *infoPtr;
    OS2File *filePtr;
    int mask;
#if 0
    ULONG count;
    BYTE output[200];
    AVAILDATA avail;
    ULONG state;
#endif

#ifdef VERBOSE
    printf("PipeEventProc\n");
    fflush(stdout);
#endif

    if (!(flags & TCL_FILE_EVENTS)) {
        return 0;
    }

    /*
     * Search through the list of watched pipes for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that pipes can be deleted while the
     * event is in the queue.
     */

    for (infoPtr = firstPipePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (pipeEvPtr->infoPtr == infoPtr) {
            infoPtr->flags &= ~(PIPE_PENDING);
            break;
        }
    }

    /*
     * Remove stale events.
     */

    if (!infoPtr) {
        return 1;
    }

    /*
     * Check to see if the pipe is readable.
     * Note that if we can't tell if a pipe is writable, we always report it
     * as being writable.
     */

    filePtr = (OS2File*) ((PipeInfo*)infoPtr)->readFile;
    mask = TCL_WRITABLE|TCL_READABLE;

#if 0

    rc = DosPeekNPipe(filePtr->handle, &output, sizeof(output), &count,
                      &avail, &state);
#ifdef VERBOSE
    printf("DosPeekNPipe %d returns %d\n", filePtr->handle, rc);
    fflush(stdout);
#endif
    if (rc == NO_ERROR) {
        if (avail.cbpipe != 0) {
            mask |= TCL_READABLE;
        }
    } else {

        /*
         * If the pipe has been closed by the other side, then
         * mark the pipe as readable, but not writable.
         */

        if (state == NP_STATE_CLOSING) {
            mask = TCL_READABLE;
        }
    }

#endif

    /*
     * Inform the channel of the events.
     */

    Tcl_NotifyChannel(infoPtr->channel, infoPtr->watchMask & mask);
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * PipeWatchProc --
 *
 *      Called by the notifier to set up to watch for events on this
 *      channel.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
PipeWatchProc(instanceData, mask)
    ClientData instanceData;            /* Pipe state. */
    int mask;                           /* What events to watch for; OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    PipeInfo **nextPtrPtr, *ptr;
    PipeInfo *infoPtr = (PipeInfo *) instanceData;
    int oldMask = infoPtr->watchMask;

#ifdef VERBOSE
    printf("PipeWatchProc\n");
    fflush(stdout);
#endif

    /*
     * For now, we just send a message to ourselves so we can poll the
     * channel for readable events.
     */

    infoPtr->watchMask = mask & infoPtr->validMask;
    if (infoPtr->watchMask) {
        Tcl_Time blockTime = { 0, 0 };
        if (!oldMask) {
            infoPtr->nextPtr = firstPipePtr;
            firstPipePtr = infoPtr;
        }
        Tcl_SetMaxBlockTime(&blockTime);
    } else {
        if (oldMask) {
            /*
             * Remove the pipe from the list of watched pipes.
             */

            for (nextPtrPtr = &firstPipePtr, ptr = *nextPtrPtr;
                 ptr != NULL;
                 nextPtrPtr = &ptr->nextPtr, ptr = *nextPtrPtr) {
                if (infoPtr == ptr) {
                    *nextPtrPtr = ptr->nextPtr;
                    break;
                }
            }
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * PipeGetHandleProc --
 *
 *      Called from Tcl_GetChannelHandle to retrieve OS handles from
 *      inside a command pipeline based channel.
 *
 * Results:
 *      Returns TCL_OK with the fd in handlePtr, or TCL_ERROR if
 *      there is no handle for the specified direction.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int
PipeGetHandleProc(instanceData, direction, handlePtr)
    ClientData instanceData;    /* The pipe state. */
    int direction;              /* TCL_READABLE or TCL_WRITABLE */
    ClientData *handlePtr;      /* Where to store the handle.  */
{
    PipeInfo *infoPtr = (PipeInfo *) instanceData;
    OS2File *filePtr;

#ifdef VERBOSE
    printf("PipeGetHandleProc\n");
    fflush(stdout);
#endif

    if (direction == TCL_READABLE && infoPtr->readFile) {
        filePtr = (OS2File*) infoPtr->readFile;
        *handlePtr = (ClientData) filePtr->handle;
        return TCL_OK;
    }
    if (direction == TCL_WRITABLE && infoPtr->writeFile) {
        filePtr = (OS2File*) infoPtr->writeFile;
        *handlePtr = (ClientData) filePtr->handle;
        return TCL_OK;
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_WaitPid --
 *
 *      Emulates the waitpid system call.
 *
 * Results:
 *      Returns 0 if the process is still alive, -1 on an error, or
 *      the pid on a clean close.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Pid
Tcl_WaitPid(pid, statPtr, options)
    Tcl_Pid pid;
    int *statPtr;
    int options;
{
    ULONG flags;

#ifdef VERBOSE
    printf("Tcl_WaitPid\n");
    fflush(stdout);
#endif

    if (!initialized) {
        PipeInit();
    }

    if (options & WNOHANG) {
        flags = DCWW_NOWAIT;
    } else {
        flags = DCWW_WAIT;
    }

#ifdef VERBOSE
    printf("Waiting for PID %d (%s)", pid,
           options & WNOHANG ? "WNOHANG" : "WAIT");
    fflush(stdout);
#endif
    rc = waitpid((int)pid, statPtr, options);
#ifdef VERBOSE
    printf(", returns %d (*statPtr %x) %s %d\n", rc, *statPtr,
           WIFEXITED(*statPtr) ? "WIFEXITED" :
           (WIFSIGNALED(*statPtr) ? "WIFSIGNALED" :
            (WIFSTOPPED(*statPtr) ? "WIFSTOPPED" : "unknown")),
           WIFEXITED(*statPtr) ? WEXITSTATUS(*statPtr) :
           (WIFSIGNALED(*statPtr) ? WTERMSIG(*statPtr) :
            (WIFSTOPPED(*statPtr) ? WSTOPSIG(*statPtr) : 0)));
    fflush(stdout);
#endif
    return (Tcl_Pid)rc;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_PidObjCmd --
 *
 *      This procedure is invoked to process the "pid" Tcl command.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *----------------------------------------------------------------------
 */

        /* ARGSUSED */
int
Tcl_PidObjCmd(dummy, interp, objc, objv)
    ClientData dummy;           /* Not used. */
    Tcl_Interp *interp;         /* Current interpreter. */
    int objc;                   /* Number of arguments. */
    Tcl_Obj *CONST *objv;       /* Argument strings. */
{
    Tcl_Channel chan;
    Tcl_ChannelType *chanTypePtr;
    PipeInfo *pipePtr;
    int i;
    Tcl_Obj *resultPtr;
    char buf[20];

#ifdef VERBOSE
    printf("Tcl_PidObjCmd\n");
    fflush(stdout);
#endif

    if (objc > 2) {
        Tcl_WrongNumArgs(interp, 1, objv, "?channelId?");
        return TCL_ERROR;
    }
    if (objc == 1) {
        resultPtr = Tcl_GetObjResult(interp);
        sprintf(buf, "%lu", (unsigned long) getpid());
        Tcl_SetStringObj(resultPtr, buf, -1);
    } else {
        chan = Tcl_GetChannel(interp, Tcl_GetStringFromObj(objv[1], NULL),
                NULL);
        if (chan == (Tcl_Channel) NULL) {
            return TCL_ERROR;
        }
        chanTypePtr = Tcl_GetChannelType(chan);
        if (chanTypePtr != &pipeChannelType) {
            return TCL_OK;
        }

        pipePtr = (PipeInfo *) Tcl_GetChannelInstanceData(chan);
        resultPtr = Tcl_GetObjResult(interp);
        for (i = 0; i < pipePtr->numPids; i++) {
            sprintf(buf, "%lu", (unsigned long)pipePtr->pidPtr[i]);
            Tcl_ListObjAppendElement(/*interp*/ NULL, resultPtr,
                    Tcl_NewStringObj(buf, -1));
        }
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TempFileCleanup
 *
 *      This procedure is a Tcl_ExitProc used to clean up the left-over
 *      temporary files made by TclpCreateTempFile (IF they still exist).
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Closes/deletes files in, and deallocates storage used by list.
 *
 *----------------------------------------------------------------------
 */

static void
TempFileCleanup(clientData)
    ClientData clientData;      /* Address of TmpFile structure */
{
    TmpFile *deleteFile = (TmpFile *)clientData;
#ifdef VERBOSE
    printf("TempFileCleanup %x [%s] (was handle [%d])\n", clientData,
           ((TmpFile*)deleteFile)->name, deleteFile->file.handle);
    fflush(stdout);
#endif
    rc = DosDelete((PSZ)((TmpFile*)deleteFile)->name);
#ifdef VERBOSE
    if (rc != NO_ERROR) {
        printf("    DosDelete ERROR %d\n", rc);
    } else {
        printf("    DosDelete OK\n");
    }
    fflush(stdout);
#endif
    /* Watch it! name was *malloc*ed by tempnam, so don't use ckfree */
    if (deleteFile->name != NULL) {
        free((char *)deleteFile->name);
    }
    ckfree((char *)deleteFile);
}
