/* 
 * tclOS2Chan.c
 *
 *	Common channel driver for OS/2 channels based on files, command
 *	pipes and TCP sockets (EMX).
 *
 * Copyright (c) 1995-1997 Sun Microsystems, Inc.
 * Copyright (c) 1999-2001 Illya Vaes
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tclUnixChan.c 1.203 97/06/20 13:03:18
 */


/*
 * Serial ports are manipulated via DosDevIOCtl, category 1h ASYNC (RS232-C)
 * Other categories: 03h (Video), 04h (Keyboard), 05h (Parallel Port),
 * 07h (Mouse), 08h (logical Disk), 09h (Physical Disk), 0Ah (Character Device),
 * 0Bh (General Device), 0Ch (Advanced Power Management), 80h (Screen Control,
 * OEMHLP, Adapter Presence-Check Services, Resource Manager and CD-ROM Drive
 * and Disc) and 81h (Touch Device-Dependent Driver).
 * Summary of Category 01h IOCtl Commands (symbol for category IOCTL_ASYNC):
 * Func  Description                                Symbol
 *  14h  Reserved
 *  34h  Reserved
 *  41h  Set Bit Rate                                 ASYNC_SETBAUDRATE
 *  42h  Set Line Characteristics (stop/parity/data)  ASYNC_SETLINECTRL
 *  43h  Extended Set Bit Rate                        ASYNC_EXTSETBAUDRATE
 *  44h  Transmit Byte Immediate                      ASYNC_TRANSMITIMM
 *  45h  Set Break OFF                                ASYNC_SETBREAKOFF
 *  46h  Set Modem Control Signals                    ASYNC_SETMODEMCTRL
 *  47h  Behave as if XOFF received (Stop transmit)   ASYNC_STOPTRANSMIT
 *  48h  Behave as if XON received (Start transmit)   ASYNC_STARTTRANSMIT
 *  49h  Reserved                                        
 *  4Bh  Set Break ON                                 ASYNC_SETBREAKON
 *  53h  Set Device Control Block (DCB) Parameters    ASYNC_SETDCBINFO
 *  54h  Set Enhanced Mode Parameters                 ASYNC_SETENHANCEDMODEPARMS
 *  61h  Query Current Bit Rate                       ASYNC_GETBAUDRATE
 *  62h  Query Line Characteristics                   ASYNC_GETLINECTRL
 *  63h  Extended Query Bit Rate                      ASYNC_EXTGETBAUDRATE
 *  64h  Query COM Status                             ASYNC_GETCOMMSTATUS
 *  65h  Query Transmit Data Status                   ASYNC_GETLINESTATUS
 *  66h  Query Modem Control Output Signals           ASYNC_GETMODEMOUTPUT
 *  67h  Query Current Modem Input Signals            ASYNC_GETMODEMINPUT
 *  68h  Query Nr of Characters in Receive Queue      ASYNC_GETINQUECOUNT
 *  69h  Query Nr of Characters in Transmit Queue     ASYNC_GETOUTQUECOUNT
 *  6Dh  Query COM Error                              ASYNC_GETCOMMERROR
 *  72h  Query COM Event Information                  ASYNC_GETCOMMEVENT
 *  73h  Query Device Control Block (DCB) Parms       ASYNC_GETDCBINFO
 *  74h  Query Enhanced Mode Parameters               ASYNC_GETENHANCEDMODEPARMS
 *
 * To get the DosDevIOCtl declarations, we need to define INCL_DOSDEVIOCTL
 * before including os2.h, ie. before including tclOS2Int.h.
 */

#define INCL_DOSDEVIOCTL
#include	"tclOS2Int.h"
#undef INCL_DOSDEVIOCTL

/*
 * This is the size of the channel name for File based channels
 */

#define CHANNEL_NAME_SIZE       64
static char channelName[CHANNEL_NAME_SIZE+1];

/*
 * The following variable is used to tell whether this module has been
 * initialized.
 */

static int initialized = 0;

/*
 * State flags used in the info structures below.
 */

#define FILE_PENDING    (1<<0)  /* Message is pending in the queue. */
#define FILE_ASYNC      (1<<1)  /* Channel is non-blocking. */
#define FILE_APPEND     (1<<2)  /* File is in append mode. */

/*
 * The following structure contains per-instance data for a file based channel
 */

typedef struct FileInfo {
    Tcl_Channel channel;        /* Pointer to channel structure. */
    int validMask;              /* OR'ed combination of TCL_READABLE,
                                 * TCL_WRITABLE, or TCL_EXCEPTION: indicates
                                 * which operations are valid on the file. */
    int watchMask;              /* OR'ed combination of TCL_READABLE,
                                 * TCL_WRITABLE, or TCL_EXCEPTION: indicates
                                 * which events should be reported. */
    int flags;                  /* State flags, see above for a list. */
    HFILE handle;               /* Input/output file. */
    struct FileInfo *nextPtr;   /* Pointer to next registered file. */
} FileInfo;

/*
 * List of all file channels currently open.
 */

static FileInfo *firstFilePtr = NULL;

/*
 * The following structure is what is added to the Tcl event queue when
 * file events are generated.
 */

typedef struct FileEvent {
    Tcl_Event header;           /* Information that is standard for
                                 * all events. */
    FileInfo *infoPtr;          /* Pointer to file info structure.  Note
                                 * that we still have to verify that the
                                 * file exists before dereferencing this
                                 * pointer. */
} FileEvent;

/*
 * Static routines for this file:
 */

static int              ComGetOptionProc _ANSI_ARGS_((ClientData instanceData,
                            Tcl_Interp *interp, char *optionName,
                            Tcl_DString *dsPtr));
static int              ComInputProc _ANSI_ARGS_((ClientData instanceData,
                            char *buf, int toRead, int *errorCode));
static int              ComSetOptionProc _ANSI_ARGS_((ClientData instanceData,
                            Tcl_Interp *interp, char *optionName,
                            char *value));
static int		FileBlockModeProc _ANSI_ARGS_((
    			    ClientData instanceData, int mode));
static void             FileChannelExitHandler _ANSI_ARGS_((
                            ClientData clientData));
static void             FileCheckProc _ANSI_ARGS_((ClientData clientData,
                            int flags));
static int		FileCloseProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp));
static int              FileEventProc _ANSI_ARGS_((Tcl_Event *evPtr,
                            int flags));
static int		FileGetHandleProc _ANSI_ARGS_((ClientData instanceData,
		            int direction, ClientData *handlePtr));
static void             FileInit _ANSI_ARGS_((void));
static int		FileInputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toRead, int *errorCode));
static int		FileOutputProc _ANSI_ARGS_((
			    ClientData instanceData, char *buf, int toWrite,
                            int *errorCode));
static int		FileSeekProc _ANSI_ARGS_((ClientData instanceData,
			    long offset, int mode, int *errorCode));
static void             FileSetupProc _ANSI_ARGS_((ClientData clientData,
                            int flags));
static void		FileWatchProc _ANSI_ARGS_((ClientData instanceData,
		            int mask));

/*
 * This structure describes the channel type structure for file based IO:
 */

static Tcl_ChannelType fileChannelType = {
    "file",				/* Type name. */
    FileBlockModeProc,			/* Set blocking/nonblocking mode.*/
    FileCloseProc,			/* Close proc. */
    FileInputProc,			/* Input proc. */
    FileOutputProc,			/* Output proc. */
    FileSeekProc,			/* Seek proc. */
    NULL,				/* Set option proc. */
    NULL,				/* Get option proc. */
    FileWatchProc,			/* Initialize notifier. */
    FileGetHandleProc,			/* Get OS handles out of channel. */
};

static Tcl_ChannelType comChannelType = {
    "com",				/* Type name. */
    FileBlockModeProc,			/* Set blocking/nonblocking mode.*/
    FileCloseProc,			/* Close proc. */
    ComInputProc,			/* Input proc. */
    FileOutputProc,			/* Output proc. */
    NULL,				/* Seek proc. */
    ComSetOptionProc,			/* Set option proc. */
    ComGetOptionProc,			/* Get option proc. */
    FileWatchProc,			/* Initialize notifier. */
    FileGetHandleProc,			/* Get OS handles out of channel. */
};

/*
 *----------------------------------------------------------------------
 *
 * ComInputProc --
 *
 *      Reads input from the IO channel into the buffer given. Returns
 *      count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *      A count of how many bytes were read is returned and an error
 *      indication is returned in an output argument.
 *
 * Side effects:
 *      Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
ComInputProc(instanceData, buf, bufSize, errorCode)
    ClientData instanceData;    /* File state. */
    char *buf;                  /* Where to store data read. */
    int bufSize;                /* How much space is available
                                 * in the buffer? */
    int *errorCode;             /* Where to store error code. */
{
    FileInfo *infoPtr;
    ULONG bytesRead;
    ULONG len;
    USHORT comError;
    RXQUEUE inQueue;

    *errorCode = 0;
    infoPtr = (FileInfo *) instanceData;

    /* Retrieve and clear COM device error information */
    len = sizeof(comError);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_GETCOMMERROR,	/* Get COM error */
                     NULL,			/* No parameter packet */
                     0L,			/* Max size of parameter list */
                     NULL,			/* Size of parameter packet */
                     (PULONG)&comError,		/* Points at com error */
                     sizeof(comError),		/* Max size of data packet */
                     &len);			/* Size of data packet */
    if (rc == NO_ERROR) {
#ifdef VERBOSE
        printf("Com DosDevIOCtl ASYNC_GETCOMMERROR OK: %x\n", comError);
        fflush(stdout);
#endif
        if ((comError & 0xf) != 0) {
            *errorCode = EIO;
            return -1;
        }
        /* Determine nr. of characters in receive queue */
        len = sizeof(inQueue);
        inQueue.cch = 0;
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_GETINQUECOUNT,	/* Get in-queue count */
                         NULL,			/* No parameter packet */
                         0L,			/* Max size of parameter list */
                         NULL,			/* Size of parameter packet */
                         (PULONG)&inQueue,	/* Points at queue structure */
                         sizeof(inQueue),	/* Max size of data packet */
                         &len);			/* Size of data packet */
        if (inQueue.cch != 0) {
            if ((ULONG) bufSize > inQueue.cch) {
                bufSize = inQueue.cch;
            }
        } else {
            if (infoPtr->flags & FILE_ASYNC) {
                errno = *errorCode = EAGAIN;
                return -1;
            } else {
                bufSize = 1;
            }
        }
#ifdef VERBOSE
    } else {
        printf("Com DosDevIOCtl ASYNC_GETCOMMERROR ERROR %d\n", rc);
        fflush(stdout);
#endif
    }

    rc = DosRead(infoPtr->handle, (PVOID) buf, (ULONG) bufSize, &bytesRead);
#ifdef VERBOSE
    printf("ComInputProc DosRead handle [%x] returns [%d], bytes read [%d]\n",
           infoPtr->handle, rc, bytesRead);
    fflush(stdout);
#endif
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        *errorCode = errno;
        return -1;
    }

    return bytesRead;
}

/*
 *----------------------------------------------------------------------
 *
 * ComSetOptionProc --
 *
 *      Sets an option on a channel.
 *
 * Results:
 *      A standard Tcl result. Also sets interp->result on error if
 *      interp is not NULL.
 *
 * Side effects:
 *      May modify an option on a device.
 *
 *----------------------------------------------------------------------
 */

static int
ComSetOptionProc(instanceData, interp, optionName, value)
    ClientData instanceData;    /* File state. */
    Tcl_Interp *interp;         /* For error reporting - can be NULL. */
    char *optionName;           /* Which option to set? */
    char *value;                /* New value for option. */
{
    FileInfo *infoPtr;
    LINECONTROL line;
    ULONG parmLen;
    struct {
        ULONG lSpeed;
        UCHAR fraction;
    } extSpeed;
    USHORT speed;
    int len, i, end;
    char stop[3] = {'1', '\0', '\0'};
    static char *bad = "bad value for -mode";

    infoPtr = (FileInfo *) instanceData;

    len = strlen(optionName);
    if ((len <= 1) || (strncmp(optionName, "-mode", len) != 0)) {
        return Tcl_BadChannelOption(interp, optionName, "mode");
    }

#ifdef VERBOSE
    /* Get current state of com */
    parmLen = sizeof(line);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_GETLINECTRL,		/* Get line control */
                     NULL,			/* No parameter packet */
                     0L,			/* Max size of parameter list */
                     NULL,			/* Size of parameter packet */
                     (PULONG)&line,		/* Points at line structure */
                     sizeof(line),		/* Max size of data packet */
                     &parmLen			/* Size of data packet */
                    );
    if (rc != NO_ERROR) {
        if (interp) {
            Tcl_AppendResult(interp, "can't get comm line state", NULL);
        }
        return TCL_ERROR;
    }

    /* Get current baud rate */
    parmLen = sizeof(speed);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_GETBAUDRATE,		/* Get baud rate */
                     NULL,			/* No parameter packet */
                     0L,			/* Max size of parameter list */
                     NULL,			/* Size of parameter packet */
                     (PULONG)&speed,		/* Points at speed variable */
                     sizeof(speed),		/* Max size of data packet */
                     &parmLen			/* Size of data packet */
                    );
    if (rc != NO_ERROR) {
        if (interp) {
            Tcl_AppendResult(interp, "can't get comm baud rate", NULL);
        }
        return TCL_ERROR;
    }

    printf("Com Current comm stats h %x: %d,%d,%d,%d (baud,parity,data,stop)\n",
           infoPtr->handle,speed, line.bParity, line.bDataBits, line.bStopBits);
    fflush(stdout);
#endif

    /* Parse mode argument. It is of the form baud,parity,data,stop */
    i = sscanf(value, "%ld,%c,%d,%s%n", &(extSpeed.lSpeed), &(line.bParity),
               (int *)&(line.bDataBits), stop, &end);

    /* If not succesful return error */
    if ((i != 4) || (value[end] != '\0')) {
        if (interp) {
            Tcl_AppendResult(interp, bad, ": should be baud,parity,data,stop",
                             NULL);
        }
        return TCL_ERROR;
    }
    if (strchr("noems", line.bParity) == NULL) {
        if (interp != NULL) {
            Tcl_AppendResult(interp, bad, " parity: should be n, o, e, m, or s",
                             NULL);
        }
        return TCL_ERROR;
    }
    switch(line.bParity) {
        case 'n': line.bParity = 0; break;
        case 'o': line.bParity = 1; break;
        case 'e': line.bParity = 2; break;
        case 'm': line.bParity = 3; break;
        case 's': line.bParity = 4; break;
        default:  line.bParity = 2; break;
    }

    if ((line.bDataBits < 0x05) || (line.bDataBits > 0x08)) {
        if (interp != NULL) {
            Tcl_AppendResult(interp, bad, " data: should be 5, 6, 7, or 8",
                             NULL);
        }
        return TCL_ERROR;
    }

    /*
     * We get a warning if we uncomment the following '< 0' because the
     * bStopBits member is declared as BYTE = *unsigned* char.
     */
    if (/*(line.bStopBits < 0) ||*/ (line.bStopBits > 2)) {
        if (interp != NULL) {
            Tcl_AppendResult(interp, bad, " stop: should be 1, 1.5 or 2", NULL);
        }
        return TCL_ERROR;
    }
    if (stop[0] == '2') {
        line.bStopBits = 2;
    } else {
        if (stop[1] == '.' && stop[2] == '5') {
            line.bStopBits = 1;
        } else {
            /* 1 stopbit, OS/2 default */
            line.bStopBits = 0;
        }
    }

    /* Everything OK, set baud rate to new value */
    if (extSpeed.lSpeed > 19200) {
        /* We need to use EXTSETBAURATE for higher baud rates than 19200 */
        parmLen = sizeof(extSpeed);
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_SETBAUDRATE,	/* Set baud rate */
                         (PULONG)&extSpeed,	/* Points at speed variable */
                         sizeof(extSpeed),	/* Max size of param packet */
                         &parmLen,		/* Size of param packet */
                         NULL,			/* No data packet */
                         0L,			/* Max size of data list */
                         NULL			/* Size of data packet */
                        );
    } else {
        speed = (USHORT) extSpeed.lSpeed;
        parmLen = sizeof(speed);
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_SETBAUDRATE,	/* Set baud rate */
                         (PULONG)&speed,	/* Points at speed variable */
                         sizeof(speed),		/* Max size of param packet */
                         &parmLen,		/* Size of param packet */
                         NULL,			/* No data packet */
                         0L,			/* Max size of data list */
                         NULL			/* Size of data packet */
                        );
    }
    if (rc != NO_ERROR) {
#ifdef VERBOSE
        printf("Com DosDevIOCtl (EXT)SETBAUDRATE handle %x ERROR %d\n",
               infoPtr->handle, rc);
        fflush(stdout);
#endif
        if (interp) {
            Tcl_AppendResult(interp, "can't set comm baud rate", NULL);
        }
        return TCL_ERROR;
    }

    /* Everything OK, set line control to new values */
    parmLen =  3 * sizeof(BYTE);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_SETLINECTRL,		/* Set line control */
                     (PULONG)&line,		/* Points at line structure */
                     sizeof(line),		/* Max size of parm packet */
                     &parmLen,			/* Size of parm packet */
                     NULL,			/* No data packet */
                     0L,			/* Max size of data list */
                     NULL			/* Size of data packet */
                    );

    if (rc == NO_ERROR) {
        return TCL_OK;
    } else {
        if (interp) {
            Tcl_AppendResult(interp, "can't set comm line control",
                             NULL);
        }
        return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComGetOptionProc --
 *
 *      Gets a mode associated with an IO channel. If the optionName arg
 *      is non NULL, retrieves the value of that option. If the optionName
 *      arg is NULL, retrieves a list of alternating option names and
 *      values for the given channel.
 *
 * Results:
 *      A standard Tcl result. Also sets the supplied DString to the
 *      string value of the option(s) returned.
 *
 * Side effects:
 *      The string returned by this function is in static storage and
 *      may be reused at any time subsequent to the call.
 *
 *----------------------------------------------------------------------
 */

static int
ComGetOptionProc(instanceData, interp, optionName, dsPtr)
    ClientData instanceData;    /* File state. */
    Tcl_Interp *interp;          /* For error reporting - can be NULL. */
    char *optionName;           /* Option to get. */
    Tcl_DString *dsPtr;         /* Where to store value(s). */
{
    FileInfo *infoPtr;
    LINECONTROL line;
    ULONG parmLen;
    struct {
        ULONG lSpeed;
        UCHAR fraction;
        ULONG minRate;
        UCHAR minFraction;
        ULONG maxRate;
        UCHAR maxFraction;
    } extSpeed;
    USHORT speed;
    int len;
    char parity;
    char *stop;
    char buf[32];

    infoPtr = (FileInfo *) instanceData;

    if (optionName == NULL) {
        Tcl_DStringAppendElement(dsPtr, "-mode");
        len = 0;
    } else {
        len = strlen(optionName);
    }

    if ((len != 0) &&
            ((len <= 1) || (strncmp(optionName, "-mode", len) != 0))) {
        return Tcl_BadChannelOption(interp, optionName, "mode");
    }

    /* Get line control */
    parmLen = sizeof(line);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_GETLINECTRL,		/* Get line control */
                     NULL,			/* No parameter packet */
                     0L,			/* Max size of parameter list */
                     NULL,			/* Size of parameter packet */
                     (PULONG)&line,		/* Points at line structure */
                     sizeof(line),		/* Max size of data packet */
                     &parmLen			/* Size of data packet */
                    );
    if (rc != NO_ERROR) {
        if (interp) {
            Tcl_AppendResult(interp, "can't get comm line state", NULL);
        }
        return TCL_ERROR;
    }

    /* Get baud rate */
    parmLen = sizeof(speed);
    rc = DosDevIOCtl(infoPtr->handle,		/* Device handle */
                     IOCTL_ASYNC,		/* Serial-device control */
                     ASYNC_GETBAUDRATE,		/* Get baud rate */
                     NULL,			/* No parameter packet */
                     0L,			/* Max size of parameter list */
                     NULL,			/* Size of parameter packet */
                     (PULONG)&speed,		/* Points at speed variable */
                     sizeof(speed),		/* Max size of data packet */
                     &parmLen			/* Size of data packet */
                    );
    if (rc != NO_ERROR) {
        if (interp) {
            Tcl_AppendResult(interp, "can't get comm baud rate", NULL);
        }
        return TCL_ERROR;
    }
    /* If we got 1200 baud, we could be having a case of a baud rate > 19200 */
    extSpeed.lSpeed = speed;
    if (speed == 1200) {
        parmLen = sizeof(extSpeed);
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_EXTGETBAUDRATE,	/* Get baud rate */
                         NULL,			/* No parameter packet */
                         0L,			/* Max size of parameter list */
                         NULL,			/* Size of parameter packet */
                         (PULONG)&extSpeed,	/* Points at speed variable */
                         sizeof(extSpeed),	/* Max size of data packet */
                         &parmLen		/* Size of data packet */
                        );
        if (rc != NO_ERROR) {
            extSpeed.lSpeed = speed;
#ifdef VERBOSE
            printf("Com DosDevIOCtl ERROR %d, resetting speed to 1200\n", rc);
            fflush(stdout);
#endif
        }
    }

    parity = 'e';	/* OS/2 initial value; Windows version says 'n' */
    if (line.bParity < 5) {
        parity = "noems"[line.bParity];
    }

    stop = (line.bStopBits == 2) ? "2" :
            (line.bStopBits == 1) ? "1.5" : "1"; /* 1 is OS/2 initial value */

    sprintf(buf, "%d,%c,%d,%s", speed, parity, line.bDataBits, stop);
    Tcl_DStringAppendElement(dsPtr, buf);
 
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FileBlockModeProc --
 *
 *	Helper procedure to set blocking and nonblocking modes on a
 *	file based channel. Invoked by generic IO level code.
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
FileBlockModeProc(instanceData, mode)
    ClientData instanceData;		/* File state. */
    int mode;				/* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;

    /*
     * Files on OS/2 can not be switched between blocking and nonblocking,
     * hence we have to emulate the behavior. This is done in the input
     * function by checking against a bit in the state. We set or unset the
     * bit here to cause the input function to emulate the correct behavior.
     */

    if (mode == TCL_MODE_NONBLOCKING) {
        infoPtr->flags |= FILE_ASYNC;
    } else {
        infoPtr->flags &= ~(FILE_ASYNC);
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FileChannelExitHandler --
 *
 *      This function is called to cleanup the channel driver before
 *      Tcl is unloaded.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Destroys the communication window.
 *
 *----------------------------------------------------------------------
 */

static void
FileChannelExitHandler(clientData)
    ClientData clientData;      /* Old window proc */
{
    Tcl_DeleteEventSource(FileSetupProc, FileCheckProc, NULL);
    initialized = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FileCheckProc --
 *
 *      This procedure is called by Tcl_DoOneEvent to check the file
 *      event source for events.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      May queue an event.
 *
 *----------------------------------------------------------------------
 */

static void
FileCheckProc(data, flags)
    ClientData data;            /* Not used. */
    int flags;                  /* Event flags as passed to Tcl_DoOneEvent. */
{
    FileEvent *evPtr;
    FileInfo *infoPtr;

    if (!(flags & TCL_FILE_EVENTS)) {
        return;
    }

    /*
     * Queue events for any ready files that don't already have events
     * queued (caused by persistent states that won't generate WinSock
     * events).
     */

    for (infoPtr = firstFilePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (infoPtr->watchMask && !(infoPtr->flags & FILE_PENDING)) {
            infoPtr->flags |= FILE_PENDING;
            evPtr = (FileEvent *) ckalloc(sizeof(FileEvent));
            evPtr->header.proc = FileEventProc;
            evPtr->infoPtr = infoPtr;
            Tcl_QueueEvent((Tcl_Event *) evPtr, TCL_QUEUE_TAIL);
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FileCloseProc --
 *
 *	This procedure is called from the generic IO level to perform
 *	channel-type-specific cleanup when a file based channel is closed.
 *
 * Results:
 *	0 if successful, errno if failed.
 *
 * Side effects:
 *	Closes the device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileCloseProc(instanceData, interp)
    ClientData instanceData;	/* File state. */
    Tcl_Interp *interp;		/* For error reporting - unused. */
{
    FileInfo *fileInfoPtr = (FileInfo *) instanceData;
    FileInfo **nextPtrPtr;
    int errorCode = 0;

    /*
     * Remove the file from the watch list.
     */

    FileWatchProc(instanceData, 0);

    rc = DosClose(fileInfoPtr->handle);
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        errorCode = errno;
    }
#ifdef VERBOSE
      else {
        openedFiles--;
    }
#endif
    for (nextPtrPtr = &firstFilePtr; (*nextPtrPtr) != NULL;
         nextPtrPtr = &((*nextPtrPtr)->nextPtr)) {
        if ((*nextPtrPtr) == fileInfoPtr) {
            (*nextPtrPtr) = fileInfoPtr->nextPtr;
            break;
        }
    }
    ckfree((char *)fileInfoPtr);
    return errorCode;
}

/*----------------------------------------------------------------------
 *
 * FileEventProc --
 *
 *      This function is invoked by Tcl_ServiceEvent when a file event
 *      reaches the front of the event queue.  This procedure invokes
 *      Tcl_NotifyChannel on the file.
 *
 * Results:
 *      Returns 1 if the event was handled, meaning it should be removed
 *      from the queue.  Returns 0 if the event was not handled, meaning
 *      it should stay on the queue.  The only time the event isn't
 *      handled is if the TCL_FILE_EVENTS flag bit isn't set.
 *
 * Side effects:
 *      Whatever the notifier callback does.
 *
 *----------------------------------------------------------------------
 */

static int
FileEventProc(evPtr, flags)
    Tcl_Event *evPtr;           /* Event to service. */
    int flags;                  /* Flags that indicate what events to
                                 * handle, such as TCL_FILE_EVENTS. */
{
    FileEvent *fileEvPtr = (FileEvent *)evPtr;
    FileInfo *infoPtr;

    if (!(flags & TCL_FILE_EVENTS)) {
        return 0;
    }

    /*
     * Search through the list of watched files for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that files can be deleted while the
     * event is in the queue.
     */

    for (infoPtr = firstFilePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (fileEvPtr->infoPtr == infoPtr) {
            infoPtr->flags &= ~(FILE_PENDING);
            Tcl_NotifyChannel(infoPtr->channel, infoPtr->watchMask);
            break;
        }
    }
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * FileGetHandleProc --
 *
 *	Called from Tcl_GetChannelFile to retrieve OS handles from
 *	a file based channel.
 *
 * Results:
 *	Returns TCL_OK with the handle in handlePtr, or TCL_ERROR if
 *	there is no handle for the specified direction. 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
FileGetHandleProc(instanceData, direction, handlePtr)
    ClientData instanceData;	/* The file state. */
    int direction;		/* TCL_READABLE or TCL_WRITABLE */
    ClientData *handlePtr;	/* Where to store the handle.  */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;

    if (direction & infoPtr->validMask) {
        *handlePtr = (ClientData) infoPtr->handle;
        return TCL_OK;
    } else {
        return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FileInit --
 *
 *      This function creates the window used to simulate file events.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Creates a new window and creates an exit handler.
 *
 *----------------------------------------------------------------------
 */

static void
FileInit()
{
    initialized = 1;
    firstFilePtr = NULL;
    Tcl_CreateEventSource(FileSetupProc, FileCheckProc, NULL);
    Tcl_CreateExitHandler(FileChannelExitHandler, NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * FileInputProc --
 *
 *	This procedure is invoked from the generic IO level to read
 *	input from a file based channel.
 *
 * Results:
 *	The number of bytes read is returned or -1 on error. An output
 *	argument contains a POSIX error code if an error occurs, or zero.
 *
 * Side effects:
 *	Reads input from the input device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileInputProc(instanceData, buf, toRead, errorCodePtr)
    ClientData instanceData;		/* File state. */
    char *buf;				/* Where to store data read. */
    int toRead;				/* How much space is available
                                         * in the buffer? */
    int *errorCodePtr;			/* Where to store error code. */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;
    ULONG bytesRead;			/* How many bytes were actually
                                         * read from the input device? */

    *errorCodePtr = 0;
    
    /*
     * Note that we will block on reads from a console buffer until a
     * full line has been entered.  The only way I know of to get
     * around this is to write a console driver.  We should probably
     * do this at some point, but for now, we just block.  The same
     * problem exists for files being read over the network.
     */

#ifdef VERBOSE
    printf("FileInputProc to read %d handle %x\n", toRead, infoPtr->handle);
    fflush(stdout);
#endif
    rc = DosRead(infoPtr->handle, (PVOID) buf, (ULONG) toRead, &bytesRead);
#ifdef VERBOSE
    printf("FileInputProc DosRead handle %x returns %d, bytes read [%d]\n",
           infoPtr->handle, rc, bytesRead);
    fflush(stdout);
#endif
    if (rc == NO_ERROR) {
        return bytesRead;
    }

    TclOS2ConvertError(rc);
    *errorCodePtr = errno;
    if (errno == EPIPE) {
        return 0;
    }
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * FileOutputProc--
 *
 *	This procedure is invoked from the generic IO level to write
 *	output to a file channel.
 *
 * Results:
 *	The number of bytes written is returned or -1 on error. An
 *	output argument	contains a POSIX error code if an error occurred,
 *	or zero.
 *
 * Side effects:
 *	Writes output on the output device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileOutputProc(instanceData, buf, toWrite, errorCodePtr)
    ClientData instanceData;		/* File state. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCodePtr;			/* Where to store error code. */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;
    ULONG bytesWritten;
    ULONG newPos;

    /*
     * If we are writing to a file that was opened with O_APPEND, we need to
     * seek to the end of the file before writing the current buffer.
     */

    if (infoPtr->flags & FILE_APPEND) {
        DosSetFilePtr(infoPtr->handle, 0, FILE_END, &newPos);
    }

    rc = DosWrite(infoPtr->handle, (PVOID) buf, (ULONG) toWrite, &bytesWritten);
    if (rc != NO_ERROR) {
        TclOS2ConvertError(rc);
        if (errno == EPIPE) {
            return 0;
        }
        *errorCodePtr = errno;
        return -1;
    }
    DosResetBuffer(infoPtr->handle);
    return bytesWritten;
}

/*
 *----------------------------------------------------------------------
 *
 * FileSeekProc --
 *
 *	This procedure is called by the generic IO level to move the
 *	access point in a file based channel.
 *
 * Results:
 *	-1 if failed, the new position if successful. An output
 *	argument contains the POSIX error code if an error occurred,
 *	or zero.
 *
 * Side effects:
 *	Moves the location at which the channel will be accessed in
 *	future operations.
 *
 *----------------------------------------------------------------------
 */

static int
FileSeekProc(instanceData, offset, mode, errorCodePtr)
    ClientData instanceData;			/* File state. */
    long offset;				/* Offset to seek to. */
    int mode;					/* Relative to where
                                                 * should we seek? Can be
                                                 * one of SEEK_START,
                                                 * SEEK_SET or SEEK_END. */
    int *errorCodePtr;				/* To store error code. */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;
    ULONG moveMethod;
    ULONG newPos;

    *errorCodePtr = 0;
    if (mode == SEEK_SET) {
        moveMethod = FILE_BEGIN;
    } else if (mode == SEEK_CUR) {
        moveMethod = FILE_CURRENT;
    } else {
        moveMethod = FILE_END;
    }

    rc = DosSetFilePtr(infoPtr->handle, offset, moveMethod, &newPos);
    if (rc != NO_ERROR) {
#ifdef VERBOSE
        printf("FileSeekProc: DosSetFilePtr handle [%x] ERROR %d\n",
               infoPtr->handle, rc);
        fflush(stdout);
#endif
        return -1;
    }
#ifdef VERBOSE
    printf("FileSeekProc: DosSetFilePtr handle [%x] newPos [%d] OK\n",
           infoPtr->handle, newPos);
    fflush(stdout);
#endif
    return newPos;
}

/*
 *----------------------------------------------------------------------
 *
 * FileSetupProc --
 *
 *      This procedure is invoked before Tcl_DoOneEvent blocks waiting
 *      for an event.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Adjusts the block time if needed.
 *
 *----------------------------------------------------------------------
 */

void
FileSetupProc(data, flags)
    ClientData data;            /* Not used. */
    int flags;                  /* Event flags as passed to Tcl_DoOneEvent. */
{
    FileInfo *infoPtr;
    Tcl_Time blockTime = { 0, 0 };

    if (!(flags & TCL_FILE_EVENTS)) {
        return;
    }

    /*
     * Check to see if there is a ready file.  If so, poll.
     */

    for (infoPtr = firstFilePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (infoPtr->watchMask) {
            Tcl_SetMaxBlockTime(&blockTime);
            break;
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FileWatchProc --
 *
 *      Called by the notifier to set up to watch for events on this
 *      channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static void
FileWatchProc(instanceData, mask)
    ClientData instanceData;		/* The file state. */
    int mask;				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    FileInfo *infoPtr = (FileInfo *) instanceData;
    Tcl_Time blockTime = { 0, 0 };

    /*
     * Since the file is always ready for events, we set the block time
     * to zero so we will poll.
     */

    infoPtr->watchMask = mask & infoPtr->validMask;
    if (infoPtr->watchMask) {
        Tcl_SetMaxBlockTime(&blockTime);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TclpOpenFileChannel --
 *
 *	Open an file based channel on OS/2 systems.
 *
 * Results:
 *	The new channel or NULL. If NULL, the output argument
 *	errorCodePtr is set to a POSIX error and an error message is
 *	left in interp->result if interp is not NULL.
 *
 * Side effects:
 *	May open the channel and may cause creation of a file on the
 *	file system.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
TclpOpenFileChannel(interp, fileName, modeString, permissions)
    Tcl_Interp *interp;			/* Interpreter for error reporting;
                                         * can be NULL. */
    char *fileName;			/* Name of file to open. */
    char *modeString;			/* A list of POSIX open modes or
                                         * a string such as "rw". */
    int permissions;			/* If the open involves creating a
                                         * file, with what modes to create
                                         * it? */
{
    FileInfo *infoPtr;
    int seekFlag, mode, channelPermissions = 0;
    HFILE handle;
    ULONG accessMode = 0, createMode, flags, exist;
    BOOL readonly = FALSE;
    char *nativeName;
    Tcl_DString buffer;
    Tcl_ChannelType *channelTypePtr;

    if (!initialized) {
        FileInit();
    }

    mode = TclGetOpenMode(interp, modeString, &seekFlag);
    if (mode == -1) {
        return NULL;
    }

    nativeName = Tcl_TranslateFileName(interp, fileName, &buffer);
    if (nativeName == NULL) {
        return NULL;
    }

    /*
     * Hack for compatibility with Windows-oriented scripts: Windows uses
     * eg. "COM1:" for the first serial port, while OS/2 uses the reserved
     * name "COM1" (without ':'). Map the first to the latter.
     * If people have more than 4 comports they can sure make their script
     * have a special case for OS/2.
     */
    if ((nativeName[0] == 'C' || nativeName[0] == 'c') && 
        (stricmp(nativeName, "COM1:")== 0 || stricmp(nativeName, "COM2:")== 0 ||
         stricmp(nativeName, "COM3:")== 0 || stricmp(nativeName, "COM4:")== 0 ||
         stricmp(nativeName, "COM5:")== 0 || stricmp(nativeName, "COM6:")== 0 ||
         stricmp(nativeName, "COM7:")== 0 || stricmp(nativeName, "COM8:")== 0 ||
         stricmp(nativeName, "COM9:")== 0)
       ) {
#ifdef VERBOSE
        printf("Mapping Windows comport %s to OS/2's ", nativeName);
        fflush(stdout);
#endif
        nativeName[4] = '\0';
#ifdef VERBOSE
        printf("%s\n", nativeName);
        fflush(stdout);
#endif
    }

    switch (mode & (O_RDONLY | O_WRONLY | O_RDWR)) {
        case O_RDONLY:
            accessMode = OPEN_ACCESS_READONLY;
            readonly = TRUE; /* Needed because O_A_R is 0 */
            channelPermissions = TCL_READABLE;
            break;
        case O_WRONLY:
            accessMode = OPEN_ACCESS_WRITEONLY;
            channelPermissions = TCL_WRITABLE;
            break;
        case O_RDWR:
            accessMode = OPEN_ACCESS_READWRITE;
            channelPermissions = (TCL_READABLE | TCL_WRITABLE);
            break;
        default:
            panic("TclpOpenFileChannel: invalid mode value");
            break;
    }

    /*
     * Map the creation flags to the OS/2 open mode.
     */

    switch (mode & (O_CREAT | O_EXCL | O_TRUNC)) {
        case (O_CREAT | O_EXCL | O_TRUNC):
            createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS;
            break;
        case (O_CREAT | O_EXCL):
            createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS;
            break;
        case (O_CREAT | O_TRUNC):
            createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS;
            break;
        case O_CREAT:
            createMode = OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS;
            break;
        case O_TRUNC:
        case (O_TRUNC | O_EXCL):
            createMode = OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS;
            break;
        default:
            createMode = OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS;
            break;
    }

    /*
     * If the file is being created, get the file attributes from the
     * permissions argument, else use the existing file attributes.
     */

    if (mode & O_CREAT) {
        if (permissions & S_IWRITE) {
            flags = FILE_NORMAL;
        } else {
            flags = FILE_READONLY;
        }
    } else {
        FILESTATUS3 infoBuf;

        if (DosQueryPathInfo(nativeName, FIL_STANDARD, &infoBuf,
	                     sizeof(infoBuf)) == NO_ERROR) {
            flags = infoBuf.attrFile;
        } else {
            flags = 0;
        }
    }

    /*
     * Set up the file sharing mode.  We want to allow simultaneous access.
     */

    accessMode |= OPEN_SHARE_DENYNONE;

    /*
     * Now we get to create the file.
     */

    rc = DosOpen(nativeName, &handle, &exist, 0, flags, createMode,
                  accessMode, (PEAOP2)NULL);
#ifdef VERBOSE
    if (rc == NO_ERROR) openedFiles++;
    printf("DosOpen [%s]: handle [%x], rc [%d] (create [%x] access [%x])\n",
           nativeName, handle, rc, createMode, accessMode);
    fflush(stdout);
#endif

    if (rc != NO_ERROR) {
        ULONG err = ERROR_SIGNAL_REFUSED;

        switch (rc) {
            case ERROR_FILE_NOT_FOUND:
            case ERROR_PATH_NOT_FOUND:
                err = ERROR_FILE_NOT_FOUND;
                break;
            case ERROR_ACCESS_DENIED:
            case ERROR_INVALID_ACCESS:
            case ERROR_SHARING_VIOLATION:
            case ERROR_CANNOT_MAKE:
            case ERROR_OPEN_FAILED:
                err = (mode & O_CREAT) ? ERROR_FILE_EXISTS
                                       : ERROR_FILE_NOT_FOUND;
                break;
        }
        TclOS2ConvertError(err);
        if (interp != (Tcl_Interp *) NULL) {
            Tcl_AppendResult(interp, "couldn't open \"", fileName, "\": ",
                    Tcl_PosixError(interp), (char *) NULL);
        }
        Tcl_DStringFree(&buffer);
        return NULL;
    }

    infoPtr = (FileInfo *) ckalloc((unsigned) sizeof(FileInfo));
    infoPtr->nextPtr = firstFilePtr;
    firstFilePtr = infoPtr;
    infoPtr->validMask = channelPermissions;
    infoPtr->watchMask = 0;
    infoPtr->flags = (mode & O_APPEND) ? FILE_APPEND : 0;
    infoPtr->handle = handle;

    /*
     * We can't distinguish between a normal file and a com-port (reserved
     * file names COM1 through COM4, so just look at the name.
     */
    if ((nativeName[0] == 'C' || nativeName[0] == 'c') && 
        (stricmp(nativeName, "COM1") == 0 || stricmp(nativeName, "COM2") == 0 ||
         stricmp(nativeName, "COM3") == 0 || stricmp(nativeName, "COM4") == 0 ||
         stricmp(nativeName, "COM5") == 0 || stricmp(nativeName, "COM6") == 0 ||
         stricmp(nativeName, "COM7") == 0 || stricmp(nativeName, "COM8") == 0 ||
         stricmp(nativeName, "COM9") == 0)
       ) {
        /*
         * This is a com port.  Reopen it with the correct modes.
         */
        DCBINFO dcb;
        ULONG parmLen;

#ifdef VERBOSE
        printf("handle 0x%x (%s) is a COM port\n", handle, nativeName);
        fflush(stdout);
#endif
        DosClose(handle);
        rc = DosOpen(nativeName, &handle, &exist, 0, flags,
                     OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                     accessMode, (PEAOP2)NULL);
        if (rc != NO_ERROR) {
            TclOS2ConvertError(ERROR_FILE_NOT_FOUND);
            if (interp != (Tcl_Interp *) NULL) {
                Tcl_AppendResult(interp, "couldn't open \"", fileName, "\": ",
                        Tcl_PosixError(interp), (char *) NULL);
            }
            Tcl_DStringFree(&buffer);
            return NULL;
        }

        /*
         * Initialize the com port.
         * GETDCBINFO should be used before SETDCBINFO to set presently
         * reserved fields correctly when the application is not aware of them.
         */

        parmLen = sizeof(dcb);
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_GETDCBINFO,	/* Get device control block */
                         NULL,			/* No parameter packet */
                         0L,			/* Max size of parameter list */
                         NULL,			/* Size of parameter packet */
                         (PULONG)&dcb,		/* Points at dcb variable */
                         sizeof(dcb),		/* Max size of data packet */
                         &parmLen		/* Size of data packet */
                        );
        if (rc != NO_ERROR) {
            if (interp) {
                Tcl_AppendResult(interp, "can't get comm device info", NULL);
            }
            Tcl_DStringFree(&buffer);
            return NULL;
        }

        /*
         * The Windows port set the following:
         * Monitoring of events: character received and put in input buffer.
         * Set (request) input and output buffers to 4096 bytes.
         * Terminate all outstanding read and write operations and clear any
         * input and output buffer.
         * Read operation should return immediately with the characters that
         * have already been received, even if no characters have been received.
         * No timeout for write.
         */
#ifdef VERBOSE
        printf("previous fbTimeout %x read %d write %d\n", dcb.fbTimeout,
               dcb.usReadTimeout, dcb.usWriteTimeout);
        fflush(stdout);
#endif
#define W_INFINITE	0x01
#define R_NORMAL	0x02
#define R_WAITFOR_S	0x04
#define R_NOWAIT	0x06
#define EHB_DISABLE	0x08
#define EHB_ENABLE	0x10
#define EHB_AUTO	0x18
#define R_TRIGGER_4	0x20
#define R_TRIGGER_8	0x40
#define R_TRIGGER_14	0x60
#define T_BUF_LC_16	0x80
        dcb.fbTimeout &= W_INFINITE | R_NOWAIT;

        parmLen = sizeof(dcb);
        rc = DosDevIOCtl(infoPtr->handle,	/* Device handle */
                         IOCTL_ASYNC,		/* Serial-device control */
                         ASYNC_SETDCBINFO,	/* Set device control block */
                         (PULONG)&dcb,		/* Points at dcb variable */
                         sizeof(dcb),		/* Max size of param packet */
                         &parmLen,		/* Size of param packet */
                         NULL,			/* No data packet */
                         0L,			/* Max size of data list */
                         NULL			/* Size of data packet */
                        );
        if (rc != NO_ERROR) {
            if (interp) {
                Tcl_AppendResult(interp, "can't initialize comm", NULL);
            }
            Tcl_DStringFree(&buffer);
            return NULL;
        }

        channelTypePtr = &comChannelType;
    } else {
#ifdef VERBOSE
        printf("handle 0x%x (%s) not a COM port\n", handle, nativeName);
        fflush(stdout);
#endif
        channelTypePtr = &fileChannelType;
    }
    Tcl_DStringFree(&buffer);

    sprintf(channelName, "file%d", (int) handle);

    infoPtr->channel = Tcl_CreateChannel(channelTypePtr, channelName,
            (ClientData) infoPtr, channelPermissions);

    if (seekFlag) {
        if (Tcl_Seek(infoPtr->channel, 0, SEEK_END) < 0) {
            if (interp != (Tcl_Interp *) NULL) {
                Tcl_AppendResult(interp, "could not seek to end of file on \"",
                        channelName, "\": ", Tcl_PosixError(interp),
                        (char *) NULL);
            }
            Tcl_Close(NULL, infoPtr->channel);
            return NULL;
        }
    }

    /*
     * Files have default translation of AUTO and ^Z eof char, which
     * means that a ^Z will be appended to them at close.
     */

    if (channelTypePtr == &comChannelType) {
        /*
         * Gotcha.  Most modems need a "\r" at the end of the command
         * sequence.  If you just send "at\n", the modem will not respond
         * with "OK" because it never got a "\r" to actually invoke the
         * command.  So, by default, newlines are translated to "\r\n" on
         * output to avoid "bug" reports that the serial port isn't working.
         */

        if (Tcl_SetChannelOption(interp, infoPtr->channel, "-translation",
                "auto crlf") != TCL_OK) {
            Tcl_Close(NULL, infoPtr->channel);
            return NULL;
        }
    } else {
        Tcl_SetChannelOption(NULL, infoPtr->channel, "-translation", "auto");
    }
    Tcl_SetChannelOption(NULL, infoPtr->channel, "-eofchar", "\032 {}");
    return infoPtr->channel;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_MakeFileChannel --
 *
 *	Makes a Tcl_Channel from an existing OS level file handle.
 *
 * Results:
 *	The Tcl_Channel created around the preexisting OS level file handle.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_MakeFileChannel(handle, mode)
    ClientData handle;		/* OS level handle. */
    int mode;			/* ORed combination of TCL_READABLE and
                                 * TCL_WRITABLE to indicate file mode. */
{
    char channelName[20];
    FileInfo *infoPtr;

    if (!initialized) {
        FileInit();
    }

    if (mode == 0) {
        return NULL;
    }

    sprintf(channelName, "file%d", (int) handle);

    /*
     * Look to see if a channel with this handle and the same mode already
     * exists. If the handle is used, but the mode doesn't match, return NULL.
     */
    
    for (infoPtr = firstFilePtr; infoPtr != NULL; infoPtr = infoPtr->nextPtr) {
        if (infoPtr->handle == (HFILE) handle) {
            return (mode == infoPtr->validMask) ? infoPtr->channel : NULL;
        }
    }

    infoPtr = (FileInfo *) ckalloc((unsigned) sizeof(FileInfo));
    infoPtr->nextPtr = firstFilePtr;
    firstFilePtr = infoPtr;
    infoPtr->validMask = mode;
    infoPtr->watchMask = 0;
    infoPtr->flags = 0;
    infoPtr->handle = (HFILE) handle;
    infoPtr->channel = Tcl_CreateChannel(&fileChannelType, channelName,
            (ClientData) infoPtr, mode);

    /*
     * OS/2 files have AUTO translation mode and ^Z eof char on input.
     */

    Tcl_SetChannelOption(NULL, infoPtr->channel, "-translation", "auto");
    Tcl_SetChannelOption(NULL, infoPtr->channel, "-eofchar", "\032 {}");
    return infoPtr->channel;
}

/*
 *----------------------------------------------------------------------
 *
 * TclGetDefaultStdChannel --
 *
 *	Creates channels for standard input, standard output or standard
 *	error output if they do not already exist.
 *
 * Results:
 *	Returns the specified default standard channel, or NULL.
 *
 * Side effects:
 *	May cause the creation of a standard channel and the underlying
 *	file.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
TclGetDefaultStdChannel(type)
    int type;			/* One of TCL_STDIN, TCL_STDOUT, TCL_STDERR. */
{
    Tcl_Channel channel = NULL;
    HFILE handle = (HFILE)0L;
    int mode = 0;
    char *bufMode = NULL;

    switch (type) {
        case TCL_STDIN:
            handle = HF_STDIN;
            mode = TCL_READABLE;
            bufMode = "line";
            break;
        case TCL_STDOUT:
            handle = HF_STDOUT;
            mode = TCL_WRITABLE;
            bufMode = "line";
            break;
        case TCL_STDERR:
            handle = HF_STDERR;
            mode = TCL_WRITABLE;
            bufMode = "none";
            break;
        default:
            panic("TclGetDefaultStdChannel: Unexpected channel type");
            break;
    }

    channel = Tcl_MakeFileChannel((ClientData)handle, mode);

    /*
     * Set up the normal channel options for stdio handles.
     */

    if ((Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-translation",
            "auto") == TCL_ERROR)
            || (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel, "-eofchar",
                    "\032 {}") == TCL_ERROR)
            || (Tcl_SetChannelOption((Tcl_Interp *) NULL, channel,
                    "-buffering", bufMode) == TCL_ERROR)) {
        Tcl_Close((Tcl_Interp *) NULL, channel);
        return (Tcl_Channel) NULL;
    }
    return channel;
}

/*
 *----------------------------------------------------------------------
 *
 * TclOS2WaitForFile --
 *
 *	This procedure waits synchronously for a file to become readable
 *	or writable, with an optional timeout.
 *
 * Results:
 *	The return value is an OR'ed combination of TCL_READABLE,
 *	TCL_WRITABLE, and TCL_EXCEPTION, indicating the conditions
 *	that are present on file at the time of the return.  This
 *	procedure will not return until either "timeout" milliseconds
 *	have elapsed or at least one of the conditions given by mask
 *	has occurred for file (a return value of 0 means that a timeout
 *	occurred).  No normal events will be serviced during the
 *	execution of this procedure.
 *
 * Side effects:
 *	Time passes.
 *
 *----------------------------------------------------------------------
 */

int
TclOS2WaitForFile(handle, mask, timeout)
    HFILE handle;		/* Handle for file on which to wait. */
    int mask;			/* What to wait for: OR'ed combination of
				 * TCL_READABLE, TCL_WRITABLE, and
				 * TCL_EXCEPTION. */
    int timeout;		/* Maximum amount of time to wait for one
				 * of the conditions in mask to occur, in
				 * milliseconds.  A value of 0 means don't
				 * wait at all, and a value of -1 means
				 * wait forever. */
{
    Tcl_Time abortTime, now;
    struct timeval blockTime, *timeoutPtr;
    int numFound, result = 0;
    static fd_set readyMasks[3];
				/* This array reflects the readable/writable
				 * conditions that were found to exist by the
				 * last call to select. */

    /*
     * If there is a non-zero finite timeout, compute the time when
     * we give up.
     */

    if (timeout > 0) {
	TclpGetTime(&now);
	abortTime.sec = now.sec + timeout/1000;
	abortTime.usec = now.usec + (timeout%1000)*1000;
	if (abortTime.usec >= 1000000) {
	    abortTime.usec -= 1000000;
	    abortTime.sec += 1;
	}
	timeoutPtr = &blockTime;
    } else if (timeout == 0) {
	timeoutPtr = &blockTime;
	blockTime.tv_sec = 0;
	blockTime.tv_usec = 0;
    } else {
	timeoutPtr = NULL;
    }

    /*
     * Initialize the ready masks and compute the mask offsets.
     */

    if ((int)handle >= FD_SETSIZE) {
	panic("TclWaitForFile can't handle file id %d", (int)handle);
    }
    FD_ZERO(&readyMasks[0]);
    FD_ZERO(&readyMasks[1]);
    FD_ZERO(&readyMasks[2]);
    
    /*
     * Loop in a mini-event loop of our own, waiting for either the
     * file to become ready or a timeout to occur.
     */

    while (1) {
	if (timeout > 0) {
	    blockTime.tv_sec = abortTime.sec - now.sec;
	    blockTime.tv_usec = abortTime.usec - now.usec;
	    if (blockTime.tv_usec < 0) {
		blockTime.tv_sec -= 1;
		blockTime.tv_usec += 1000000;
	    }
	    if (blockTime.tv_sec < 0) {
		blockTime.tv_sec = 0;
		blockTime.tv_usec = 0;
	    }
	}
	
	/*
	 * Set the appropriate bit in the ready masks for the handle.
	 */

	if (mask & TCL_READABLE) {
	    FD_SET((int)handle, &readyMasks[0]);
	}
	if (mask & TCL_WRITABLE) {
	    FD_SET((int)handle, &readyMasks[1]);
	}
	if (mask & TCL_EXCEPTION) {
	    FD_SET((int)handle, &readyMasks[2]);
	}

	/*
	 * Wait for the event or a timeout.
	 */

	numFound = select(((int)handle)+1, &readyMasks[0], &readyMasks[1],
	                  &readyMasks[2], timeoutPtr);
	if (numFound == 1) {
	    if (FD_ISSET((int)handle, &readyMasks[0])) {
		result |= TCL_READABLE;
	    }
	    if (FD_ISSET((int)handle, &readyMasks[1])) {
		result |= TCL_WRITABLE;
	    }
	    if (FD_ISSET((int)handle, &readyMasks[2])) {
		result |= TCL_EXCEPTION;
	    }
	    result &= mask;
	    if (result) {
		break;
	    }
	}
	if (timeout == 0) {
	    break;
	}

	/*
	 * The select returned early, so we need to recompute the timeout.
	 */

	TclpGetTime(&now);
	if ((abortTime.sec < now.sec)
		|| ((abortTime.sec == now.sec)
		&& (abortTime.usec <= now.usec))) {
	    break;
	}
    }
    return result;
}
