/* DirEA.cmd
 * List a directory, showing .SUBJECT and .COMMENTS EAs for each
 * file. Also shows .KEYPHRASES and displays in short or long
 * formats.
 *
 * Use the command "dirEA /?" to see usage instructions.
 *
 * Author:  Bill Schindler
 *
 * -------------------------- History --------------------------
 * 12-Jan-1994	wfs  Created.
 *
 * 05-Feb-1994	wfs  Check for directory in file spec argument
 *		     Add option for showing .COMMENTS EA
 *		     Add usage subroutine
 *
 * 02-Jun-1994	wfs  ==Version 1.10 begins==
 *		     Fixed problem displaying multi-value	
 *		     comments.					
 *		     Added routines and fixed code in		
 *		     preparation for displaying additional EA	
 *		     values.					
 *		     Did some code clean up and simplified some
 *		     processing.
 *		     Add -K flag and display key words. 					
 *		     Changed long form to display subject,	
 *		     comments, & keywords.			
 *
 * 05-Jun-1994	wfs  Added simple filtering for files with a		
 *		     specific key word. 				
 *
 * 09-Jun-1994	wfs  Fixed a problem with sizing the subject		
 *		     string. The display routines were			
 *		     truncating the subject about 10 characters 	
 *		     too early. 					
 *		     Allow multiple file specifications on the		
 *		     command line.					
 *		     Allow command line flags to appear anywhere
 *		     on the command line.			
 * -------------------------------------------------------------
 */

IF LoadRexxUtil() THEN
  EXIT

/*
 * Process any command line arguments
 */
PARSE ARG filespec

i = 0
doComments = 0
useLongForm = 0
doKeywords = 0
doSearch = 0
searchFor = ''
doSubdirs = ''
specs. = ''
specs.0 = 0

DO WHILE filespec >< ''
  PARSE VAR filespec wrk filespec
  IF Left(wrk, 1) = '/' | Left(wrk, 1) = '-' THEN
    DO /* process command line switches */
    sep = Left(wrk, 1)	       /* save the / or - */
    pgmArg = SubStr(wrk, 2)    /* strip leading / or - */
    pgmArg = Translate(pgmArg) /* uppercase the arg */

    SELECT
      /* show .COMMENTS */
      WHEN pgmArg = 'C' THEN
	doComments = 1
      /* do subdirectories */
      WHEN pgmArg = 'S' THEN
	doSubdirs = 'S' /* used with SysFileTree */
      /* long format */
      WHEN pgmArg = 'L' THEN
	useLongForm = 1
      /* show .KEYPHRASES */
      WHEN pgmArg = 'K' THEN
	DO
	doKeywords = 1
	doSearch = 1
	END
      /* search for matching .KEYPHRASES */
      WHEN Left(pgmArg, 2) = 'K:' THEN
	DO
	IF doKeywords THEN
	  doSearch = 1
	ELSE
	  DO
	  doSearch = -1
	  doKeywords = 1
	  END
	PARSE VAR pgmArg . ':' searchFor
	END
      /* show usage and exit */
      /* variations on /help are also allowed for users
       * that don't have a clue */
      WHEN pgmArg = '?' | Abbrev('HELP', pgmArg, 1) THEN
	DO
	CALL ShowUsage
	EXIT
	END
      /* oops... */
      OTHERWISE
	SAY
	SAY "Invalid command line argument '" || sep || ,
	    pgmArg || "'"
	EXIT
    END
    END
  ELSE /* not a switch, must be a file spec */
    DO
    i = i + 1 /* count file specs */
    specs.0 = i
    IF Left(wrk, 1) = '"' THEN
      PARSE VALUE wrk filespec WITH '"' wrk '"' filespec
    specs.i = wrk
    END
END

IF useLongForm THEN
  DO /* force comments and keywords for long form */
  doComments = 1
  doKeywords = 1
  END

IF specs.0 = 0 THEN
  DO /* no filespecs, so default to '*.*' */
  specs.0 = 1
  specs.1 = '*.*'
  END
ELSE /* check filespecs, and add '*.*' where needed */
  DO i = 1 TO specs.0
    /* Assumption alert! Can't tell files from dirs without
     * a trailing '\', so we assume it's a file if we don't
     * find a trailing '\'! */
    IF Right(specs.i, 1) = '\' THEN
      specs.i = specs.i || '*.*'
  END

/*
 * Read the directory
 */
files. = ''
files.0 = 0
wrk. = ''
DO i = 1 TO specs.0
  CALL SysFileTree specs.i, 'wrk.', 'F' || doSubdirs, '***-*'
						    /* ASDHR */
  k = files.0 + 1 /* next available space in files. */
  files.0 = files.0 + wrk.0
  DO j = 1 TO wrk.0 /* collect files for each filespec */
    files.k = wrk.j
    k = k + 1
  END
END

IF files.0 > 1 THEN
  DO
  CALL CharOut , 'Total of' files.0 'files found for'
  delim = ''
  DO i = 1 TO specs.0
    CALL CharOut , delim "'"specs.i"'"
    delim = ','
  END
  SAY
  END
SAY

IF files.0 = 0 THEN /* no matching files? */
  DO
  SAY 'No files found for "'filespec'"'
  EXIT
  END

/*
 * Collect and parse the EAs for each file found
 */
longestFN = 4	 /* longest filename */
longestType = 4  /* longest EA type  */
longestSubj = 7  /* longest subject  */
filename. = ''

/* extended attribute types */
eat. = ''
eat.ascii    = 'FFFD'x
eat.asn1     = 'FFDD'x
eat.binary   = 'FFFE'x
eat.bitmap   = 'FFFB'x
eat.ea	     = 'FFEE'x
eat.icon     = 'FFF9'x
eat.metafile = 'FFFA'x
eat.mvmt     = 'FFDF'x
eat.mvst     = 'FFDE'x

CALL GetEAData /* collect EA stuff for all files */

/*
 * Display the files and associated EAs
 */
IF \ useLongForm THEN /* display the column headings */
  SAY ' 'Center('File', longestFN) '|' ,
      Center('Subject', longestSubj)
PARSE VALUE SysTextScreenSize() WITH rows cols

lastSubdir = ''
extraDirs = 0
matching = 0

DO i = 1 to files.0
  IF searchFor >< '' THEN
    DO
    IF Pos(searchFor, files.i.keywords) = 0 THEN
      ITERATE
    matching = matching + 1
    IF doSearch = -1 THEN
      files.i.keywords = ''
    END
  IF  doComments THEN
    CALL PageIt

  IF doSubdirs = 'S' THEN
    IF lastSubdir >< FileSpec('PATH', filename.i) THEN
      DO /* show new directory name whenever it changes */
      lastSubdir = FileSpec('PATH', filename.i)
      IF useLongForm THEN
	DO
	SAY
	extraDirs = extraDirs + 1
	IF \ doComments THEN /* page the output? */
	  CALL PageIt
	END
      SAY 'Directory:  ' || FileSpec('DRIVE', filename.i) ||,
	  lastSubdir
      extraDirs = extraDirs + 1
      IF \ doComments THEN /* page the output if no comments */
	CALL PageIt
      END
  /*
   * Build a single line with the file name and subject
   */
  IF useLongForm THEN
    DO /* long form puts subject on a separate line from file */
    SAY Left(files.i, 28) || '  ' ||,
	FileSpec('name', filename.i)
    IF files.i.subject >< '' THEN
      SAY '  Subject: ' || files.i.subject
    END
  ELSE /* not using long form */
    DO
    fn = Left(FileSpec('Name', filename.i), longestFN)
    subj = Left(files.i.subject, longestSubj)
    SAY ' ' || fn '|' subj
    END
  IF files.i.keywords >< '' THEN
    SAY '  Key words: ' || files.i.keywords
  IF files.i.comment >< '' THEN
    DO
    SAY; SAY files.i.comment; SAY
    END
END

/* Show how many files matched a key word */
IF searchFor >< '' THEN
  DO
  SAY; SAY matching 'files matched key word "'searchFor'".'
  END

EXIT

/* -------------------------------------------------------------
 * G e t E A D a t a
 * Get and format EAs for each file. Command line switches
 * control which EAs are collected.
 */
GetEAData:
  DO i = 1 to files.0
    /*
     * Get .SUBJECT EA
     */
    PARSE VAR files.i 38 filename.i
    /* track longest filename for column width */
    longestFN = Max(longestFN,,
	Length(FileSpec('Name', filename.i)))
    IF GetAndFormatEA(filename.i, '.SUBJECT') THEN
      DO
      subjLen = ea.1.length
      subj = ea.1.data
      END
    ELSE
      subj = ''
    files.i.subject = subj
    longestSubj = Max(longestSubj, Length(subj))
    /*
     * Get .COMMENTS EA
     */
    IF doComments THEN
      DO
      cmt = ''
      IF GetAndFormatEA(filename.i, '.COMMENTS') THEN
	DO j = 1 TO ea.0
	  IF Right(cmt, 1) = '0D'x THEN
	    cmt = cmt || '0A'x
	  cmt = cmt'  'Strip(ea.j.data, 'L')
	END
	files.i.comment = cmt
      END
    /*
     * Get .KEYPHRASES EA
     */
    IF doKeywords THEN
      DO /* collect the key phrases */
      kw = ''
      IF GetAndFormatEA(filename.i, '.KEYPHRASES') THEN
	DO j = 1 TO ea.0
	  kw = kw ea.j.data
	END
      files.i.keywords = Strip(kw, 'L')
      END
  END
  /* track longest subject for column size */
  longestSubj = Min(longestSubj, 80 - (longestFN + 6))
  RETURN

/* -------------------------------------------------------------
 * G e t A n d F o r m a t E A
 * Function to get an EA and parse it. The EA is broken into its
 * components and stored in the ea. compound variable.
 */
GetAndFormatEA: PROCEDURE EXPOSE eat. ea.
  fileName = Arg(1) /* file to use */
  eaName = Arg(2)   /* which EA to get and format */
  ea.0 = 0
  IF SysGetEA(fileName, eaName, stuff) = 0 ,
  & stuff >< '' THEN
    DO /* format the EA */
    CALL FormatEA stuff, 1
    RETURN 1
    END
  RETURN 0

/* -------------------------------------------------------------
 * F o r m a t E A
 * Recursive function to parse EAs.
 */
FormatEA: PROCEDURE EXPOSE eat. ea.
  stuff = Arg(1) /* the stuff to parse */
  count = Arg(2) /* current count of EAs */
  PARSE VAR stuff eat 3 .
  eat = Reverse(eat) /* convert little-endian to big-endian */
  SELECT
    /* EAT_ASCII */
    WHEN eat == eat.ascii THEN
      DO
      ea.0 = count
      PARSE VAR stuff 3 leng 5	 /* get the length */
      leng = C2D(Reverse(leng))  /* change endian */
      PARSE VAR stuff 5 ea.count.data +(leng) stuff
      ea.count.length = leng
      ea.count.type = 'ASCII'
      END
    /* EAT_MVMT */
    WHEN eat == eat.mvmt THEN
      DO
      PARSE VAR stuff 5 leng 7 stuff /* get the length */
      leng = C2D(Reverse(leng))      /* change endian */
      count = count - 1
      DO i = 1 TO leng	/* do each sub-type */
	stuff = FormatEA(stuff, count + i)
      END
      END
    /* EAT_MVST */
    WHEN eat == eat.mvst THEN
      DO
      PARSE VAR stuff 5 leng 7 stuff /* get the length */
      leng = C2D(Reverse(leng))      /* change endian */
      count = count - 1
      DO i = 1 TO leng
	stuff = FormatEA(stuff, count + i)
	stuff = Reverse(eat.ascii) || stuff
      END
      END
    /* Unrecognized EA type */
    OTHERWISE
      ea.0 = count
      ea.count.type = '???'
      ea.count.data = 'Unknown EA' C2X(eat)
      ea.count.length = 15
  END
  RETURN stuff

/* --------------------------------------------------------------
 * P a g e I t
 */
PageIt:
  /*
   * Handle displaying more than one screen of data. Since we're
   * not calculating the length of a .COMMENTS EA, we don't
   * bother paging the output when the 'C' option is turned on.
   */
  IF ((i + extraDirs) // (rows - 1)) = 0 THEN DO
    CALL CharOut 'CON:', '--- more ---'
    CALL SysGetKey 'NOECHO' /* toss whatever key is pressed */
    SAY  /* start on the next line */
    END
  RETURN

/* -------------------------------------------------------------
 * S h o w U s a g e
 */
ShowUsage: PROCEDURE
  SAY
  SAY "Lists comments and subject EAs for one or more files"
  SAY "  dirEA Version 1.10."
  SAY
  SAY "  Usage:  dirEA [options] [filespec [filespec ...]]"
  SAY
  SAY "  Options:"
  SAY "    -?        Displays this usage screen"
  SAY "    -C        Display both subject and comments"
  SAY "    -L        Produce a long format listing"
  SAY "              (includes subject, key words, and comments)"
  SAY "    -S        Processes pathname and its subdirectories"
  SAY "    -K        Display key words"
  SAY "    -K:word   Only display files with matching key word"
  SAY
  SAY "  The filespec parameter defaults to '*.*'"
  RETURN

/* -------------------------------------------------------------
 * L o a d R e x x U t i l
 */
LoadRexxUtil: PROCEDURE
  IF RxFuncQuery('SysLoadFuncs') THEN
    DO
    IF RxFuncAdd('SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs')
      THEN DO
      SAY "Error: Couldn't load RexxUtil library."
      RETURN 1
      END
    CALL SysLoadFuncs
    END
  RETURN 0
