/**************************************************************************/
/* yaGetKey.c - yet another GetKey                                        */
/*                                                                        */
/*========================================================================*/
/* Copyright (c) 2015 VicTor Smirnoff                   iamvic@rambler.ru */
/*                                                                        */
/* Permission is hereby granted, free  of charge, to any person obtaining */
/* a  copy  of  this  software   and   associated   documentation   files */
/* (the  "Software"),  to  deal  in  the  Software  without  restriction, */
/* including without limitation  the rights  to use, copy, modify, merge, */
/* publish, distribute, sublicense,  and/or  sell copies of the Software, */
/* and  to permit persons  to whom  the Software  is furnished  to do so, */
/* subject to the following conditions:                                   */
/*                                                                        */
/* The above  copyright  notice  and  this  permission  notice  shall  be */
/* included in all copies or substantial portions of the Software.        */
/*                                                                        */
/* THE SOFTWARE  IS PROVIDED  "AS IS",  WITHOUT  WARRANTY  OF  ANY  KIND, */
/* EXPRESS OR IMPLIED,  INCLUDING  BUT  NOT LIMITED  TO THE WARRANTIES OF */
/* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. */
/* IN NO EVENT SHALL  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE  FOR ANY */
/* CLAIM, DAMAGES OR OTHER LIABILITY,  WHETHER IN AN ACTION  OF CONTRACT, */
/* TORT  OR  OTHERWISE,  ARISING  FROM,  OUT  OF  OR  IN  CONNECTION WITH */
/* THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.             */
/*------------------------------------------------------------------------*/
#define YA_PRODUCT "YAGETKEY"
#define YA_VERSION "0.07"
#define YA_ABOUT   "Yet Another GetKey v0.07; Copyright (c) 2015 VicTor Smirnoff (see MIT License)"

#define INCL_KBD
#define INCL_REXXSAA
#define INCL_DOSPROCESS
#define INCL_DOSFILEMGR
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#define INCL_DOSERRORS

#include <os2.h>
#include <rexxsaa.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

/*********************************************************************/
/* Some Names used by stems                                          */
/*********************************************************************/

#define YA_NAME_ABOUT      "0ABOUT"
#define YA_NAME_SCANCODE   "0SCANCODE"

/*********************************************************************/
/* Special Variable Name and Values                                  */
/*********************************************************************/

#define  YA_NAME_KBDERRNO    "KBDERRNO"       // special variable name
#define  YA_NOERROR                    "NO_ERROR"
#define  YA_ERROR_KBD_FOCUS_REQUIRED   "ERROR_KBD_FOCUS_REQUIRED"
#define  YA_ERROR_KBD_KEYBOARD_BUSY    "ERROR_KBD_KEYBOARD_BUSY"
#define  YA_ERROR_KBD_DETACHED         "ERROR_KBD_DETACHED"
#define  YA_ERROR_KBD_UNKNOWN          "ERROR_KBD_CODE_%d"

/*********************************************************************/
/*  Definitions used by KBDKEYINFO.fsState                           */
/*********************************************************************/
#define  YA_ON                   "1"
#define  YA_OFF                  "0"
#define  YA_DOWN                 "1"
#define  YA_UP                   "0"

#define  YA_NAME_SYSREQ_DOWN          "0SYSREQ_DOWN"
#define  YA_NAME_CAPS_LOCK_DOWN       "0CAPS_LOCK_DOWN"
#define  YA_NAME_NUM_LOCK_DOWN        "0NUM_LOCK_DOWN"
#define  YA_NAME_SCROLL_LOCK_DOWN     "0SCROLL_LOCK_DOWN"
#define  YA_NAME_RIGHT_ALT_DOWN       "0RIGHT_ALT_DOWN"
#define  YA_NAME_RIGHT_CTRL_DOWN      "0RIGHT_CTRL_DOWN"
#define  YA_NAME_LEFT_ALT_DOWN        "0LEFT_ALT_DOWN"
#define  YA_NAME_LEFT_CTRL_DOWN       "0LEFT_CTRL_DOWN"
#define  YA_NAME_INSERT_ON            "0INSERT_ON"
#define  YA_NAME_CAPS_LOCK_ON         "0CAPS_LOCK_ON"
#define  YA_NAME_NUM_LOCK_ON          "0NUM_LOCK_ON"
#define  YA_NAME_SCROLL_LOCK_ON       "0SCROLL_LOCK_ON"
#define  YA_NAME_EITHER_ALT_DOWN      "0EITHER_ALT_DOWN"
#define  YA_NAME_EITHER_CTRL_DOWN     "0EITHER_CTRL_DOWN"
#define  YA_NAME_LEFT_SHIFT_DOWN      "0LEFT_SHIFT_DOWN"
#define  YA_NAME_RIGHT_SHIFT_DOWN     "0RIGHT_SHIFT_DOWN"

/*********************************************************************/
/*  Various definitions used by various functions.                   */
/*********************************************************************/

#define  YA_MAX_DIGITS   9        // maximum digits in numeric arg
#define  YA_MAX          256      // temporary buffer length
#define  YA_SLEEP        50L      // sleep interval in msec
#define  YA_RXSHV_BAD    (RXSHV_LVAR | RXSHV_BADN | RXSHV_MEMFL | RXSHV_BADF)

/*********************************************************************/
/* Numeric Return calls                                              */
/*********************************************************************/

#define  YA_INVALID_ROUTINE    40   // Raise Rexx error
#define  YA_VALID_ROUTINE       0   // Successful completion

/*********************************************************************/
/* Some useful macros                                                */
/*********************************************************************/

#define BUILDRXSTRING(t, s) { \
  strcpy((t)->strptr,(s)); \
  (t)->strlength = strlen((s)); \
}


RexxFunctionHandler yaGetKeyLoad;
RexxFunctionHandler yaGetKey;
RexxFunctionHandler yaGetKeyVer;
RexxFunctionHandler yaGetKeyDrop;

static PSZ  yaFncTable[] =
{
"yaGetKeyLoad",     // should be first in list
"yaGetKey",
"yaGetKeyVer",
"yaGetKeyDrop"      // should be last in list
};


/* key = yaGetKeyLoad( ) */
ULONG yaGetKeyLoad( PUCHAR name,        // name of the function
                     ULONG argC,        // number of arguments
                  RXSTRING argV[],      // list of argument strings
                       PSZ queueName,   // current queue name
                 PRXSTRING retStr       // returned result string
)
{
   APIRET     rc;
   INT        iSize, i = 0;

   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;

   if (argC > 0L)
      return YA_INVALID_ROUTINE;

   iSize = sizeof(yaFncTable)/sizeof(PSZ);
   for (i = 1; i < iSize; i++)
   {
      rc = RexxRegisterFunctionDll(yaFncTable[i],
                                   YA_PRODUCT,
                                   yaFncTable[i]);
   }
   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;
   return YA_VALID_ROUTINE;
}



/* key = yaGetKeyDrop( ) */
ULONG yaGetKeyDrop( PUCHAR name,        // name of the function
                     ULONG argC,        // number of arguments
                  RXSTRING argV[],      // list of argument strings
                       PSZ queueName,   // current queue name
                 PRXSTRING retStr       // returned result string
)
{
   APIRET     rc;
   INT        iSize, i = 0;

   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;

   if (argC > 0L)
      return YA_INVALID_ROUTINE;

   iSize = sizeof(yaFncTable)/sizeof(PSZ);
   for (i = 0; i < iSize; i++)
   {
      rc = RexxDeregisterFunction(yaFncTable[i]);
   }
   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;
   return YA_VALID_ROUTINE;
}



/* key = yaGetKey( [interval[, stem]] ) */
ULONG yaGetKey( PUCHAR name,        // name of the function
                 ULONG argC,        // number of arguments
              RXSTRING argV[],      // list of argument strings
                   PSZ queueName,   // current queue name
             PRXSTRING retStr       // returned result string
)
{
   HFILE      hf;
   SHIFTSTATE ss;
   ULONG      ulAction, ulLength;
   KBDKEYINFO keyInfo;
   APIRET     rc, rcTemp;
   SHVBLOCK   blockVar;
   CHAR       nameStem[YA_MAX];
   ULONG      lenNameStem = 0;
   CHAR       nameVar[YA_MAX];
   ULONG      lenNameVar = 0;
   CHAR       valueVar[YA_MAX];
   ULONG      lenValueVar = 0;
   ULONG      ioWait = IO_WAIT;
   clock_t    waitSec = 0, startTime = 0;
   UINT       fsState = 0x0001;
   UINT       bfsState = 0;
   UINT       isFirst = 0;

   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;

   if (argC > 2L)
      return YA_INVALID_ROUTINE;

   switch(argC)
   {
   case 0L:
      ioWait = IO_WAIT;
      break;
   case 1L:
   case 2L:
      if (RXVALIDSTRING(argV[0]))
      {
         if (argV[0].strlength > YA_MAX_DIGITS)
            return YA_INVALID_ROUTINE;
         if (strspn(argV[0].strptr, "0123456789") != argV[0].strlength)
            return YA_INVALID_ROUTINE;
         waitSec=atoi(argV[0].strptr);
      } else
         waitSec = 0L;
      if (waitSec == 0L)
         ioWait = IO_WAIT;
      else
         ioWait = IO_NOWAIT;
      if (argC == 1L)
         break;
// case 2L:  first parameter already checked
      if (!RXVALIDSTRING(argV[1]))
         return YA_INVALID_ROUTINE;
      if (argV[1].strlength > YA_MAX/2)
         return YA_INVALID_ROUTINE;
      strcpy(nameStem, argV[1].strptr);
      lenNameStem = strlen(nameStem);
      strupr(nameStem);
      if (nameStem[lenNameStem-1] != '.')
      {
         nameStem[lenNameStem++] = '.';
         nameStem[lenNameStem] = 0x00;
      }
      break;
   default:
      return YA_INVALID_ROUTINE;
   }

   blockVar.shvcode = RXSHV_SET;
   blockVar.shvret  = (UCHAR)0;
   blockVar.shvnext = (PSHVBLOCK)0;
   strcpy(nameVar, YA_NAME_KBDERRNO);
   lenNameVar = strlen(nameVar);
   MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
   blockVar.shvnamelen = lenNameVar;

   keyInfo.bNlsShift = (UCHAR)0;
   if (ioWait == IO_NOWAIT)
   {
      isFirst = 0;
      bfsState = 0;
      ss.fsState = 0;
      startTime = clock();
      do
      {
         rc = KbdCharIn(&keyInfo, ioWait, (HKBD)0);
         if ( (rc == 0) && (keyInfo.fbStatus == 0x00) )
         {
            if (lenNameStem > 0L)
            {
               rc = DosOpen("KBD$",
                            &hf,
                            &ulAction,
                            0L,
                            0,
                            FILE_OPEN,
                            OPEN_ACCESS_READONLY|OPEN_SHARE_DENYNONE,
                            0);
               if (rc == 0)
               {
                  ulAction = 0;
                  ulLength = sizeof(ss);
                  rcTemp = DosDevIOCtl(hf,
                                       IOCTL_KEYBOARD,
                                       KBD_GETSHIFTSTATE,
                                       0,
                                       0,
                                       &ulAction,
                                       &ss,
                                       sizeof(ss),
                                       &ulLength);
                  rc = DosClose(hf);
                  rc = rcTemp;
                  if (rc == 0)
                  {
                     if (isFirst == 0)
                     {
                        bfsState = ss.fsState;
                        isFirst = 1;
                     }
                     keyInfo.chScan = 0x00;
                     keyInfo.fsState = ss.fsState;
                     if (bfsState == ss.fsState)
                        rc = DosSleep(YA_SLEEP);
                  }
               }
            }

// DosSleep returns rc>0, when Ctrl-Break pressed
            else
               rc = DosSleep(YA_SLEEP);
         }
      } while( (rc == 0) && 
               (keyInfo.fbStatus == 0x00) && 
               (bfsState == ss.fsState) &&
               ((clock()-startTime)/CLOCKS_PER_SEC < waitSec) );
      bfsState = ss.fsState;
   } else
   {
      rc = KbdCharIn(&keyInfo, ioWait, (HKBD)0);
   }
   if (rc != 0)
   {
      switch(rc)
      {
      case ERROR_KBD_FOCUS_REQUIRED:
         strcpy(valueVar, YA_ERROR_KBD_FOCUS_REQUIRED);
         break;
      case ERROR_KBD_KEYBOARD_BUSY:
         strcpy(valueVar, YA_ERROR_KBD_KEYBOARD_BUSY);
         break;
      case ERROR_KBD_DETACHED:
         strcpy(valueVar, YA_ERROR_KBD_DETACHED);
         break;
      default:
         sprintf(valueVar, YA_ERROR_KBD_UNKNOWN, rc);
         break;
      }
      lenValueVar = strlen(valueVar);
      MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
      blockVar.shvvaluelen = lenValueVar;
      rc = RexxVariablePool(&blockVar);
      if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
         return YA_INVALID_ROUTINE;

      retStr->strptr[0] = 0x00;
      retStr->strlength = 0;
      return YA_VALID_ROUTINE;
   }

   valueVar[0] = 0x00;   // strcpy(valueVar, YA_NOERROR);
   lenValueVar = strlen(valueVar);
   MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
   blockVar.shvvaluelen = lenValueVar;
   rc = RexxVariablePool(&blockVar);
   if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
      return YA_INVALID_ROUTINE;

   if (lenNameStem > 0L)
   {
      blockVar.shvcode = RXSHV_SET;
      blockVar.shvret  = (UCHAR)0;
      blockVar.shvnext = (PSHVBLOCK)0;
      sprintf(nameVar, "%s%s", nameStem, YA_NAME_SCANCODE);
      lenNameVar = strlen(nameVar);
      MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
      blockVar.shvnamelen = lenNameVar;
      valueVar[0] = keyInfo.chScan;
      valueVar[1] = 0x00;
      if (keyInfo.fbStatus == 0x00)
         lenValueVar = 0;
      else
         lenValueVar = 1;
      MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
      blockVar.shvvaluelen = lenValueVar;
      rc = RexxVariablePool(&blockVar);
      if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
         return YA_INVALID_ROUTINE;

      do
      {
         blockVar.shvcode = RXSHV_SET;
         blockVar.shvret  = (UCHAR)0;
         blockVar.shvnext = (PSHVBLOCK)0;

         strcpy(nameVar, nameStem);
         lenNameVar = strlen(nameVar);

         switch (fsState)
         {
         case SYSREQ:
            strcpy(nameVar+lenNameStem, YA_NAME_SYSREQ_DOWN);
            if (keyInfo.fsState & SYSREQ)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case CAPSLOCK:
            strcpy(nameVar+lenNameStem, YA_NAME_CAPS_LOCK_DOWN);
            if (keyInfo.fsState & CAPSLOCK)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case NUMLOCK:
            strcpy(nameVar+lenNameStem, YA_NAME_NUM_LOCK_DOWN);
            if (keyInfo.fsState & NUMLOCK)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case SCROLLLOCK:
            strcpy(nameVar+lenNameStem, YA_NAME_SCROLL_LOCK_DOWN);
            if (keyInfo.fsState & SCROLLLOCK)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case RIGHTALT:
            strcpy(nameVar+lenNameStem, YA_NAME_RIGHT_ALT_DOWN);
            if (keyInfo.fsState & RIGHTALT)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case RIGHTCONTROL:
            strcpy(nameVar+lenNameStem, YA_NAME_RIGHT_CTRL_DOWN);
            if (keyInfo.fsState & RIGHTCONTROL)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case LEFTALT:
            strcpy(nameVar+lenNameStem, YA_NAME_LEFT_ALT_DOWN);
            if (keyInfo.fsState & LEFTALT)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case LEFTCONTROL:
            strcpy(nameVar+lenNameStem, YA_NAME_LEFT_CTRL_DOWN);
            if (keyInfo.fsState & LEFTCONTROL)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case INSERT_ON:
            strcpy(nameVar+lenNameStem, YA_NAME_INSERT_ON);
            if (keyInfo.fsState & INSERT_ON)
               strcpy(valueVar, YA_ON);
            else
               strcpy(valueVar, YA_OFF);
            break;

         case CAPSLOCK_ON:
            strcpy(nameVar+lenNameStem, YA_NAME_CAPS_LOCK_ON);
            if (keyInfo.fsState & CAPSLOCK_ON)
               strcpy(valueVar, YA_ON);
            else
               strcpy(valueVar, YA_OFF);
            break;

         case NUMLOCK_ON:
            strcpy(nameVar+lenNameStem, YA_NAME_NUM_LOCK_ON);
            if (keyInfo.fsState & NUMLOCK_ON)
               strcpy(valueVar, YA_ON);
            else
               strcpy(valueVar, YA_OFF);
            break;

         case SCROLLLOCK_ON:
            strcpy(nameVar+lenNameStem, YA_NAME_SCROLL_LOCK_ON);
            if (keyInfo.fsState & SCROLLLOCK_ON)
               strcpy(valueVar, YA_ON);
            else
               strcpy(valueVar, YA_OFF);
            break;

         case ALT:
            strcpy(nameVar+lenNameStem, YA_NAME_EITHER_ALT_DOWN);
            if (keyInfo.fsState & ALT)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case CONTROL:
            strcpy(nameVar+lenNameStem, YA_NAME_EITHER_CTRL_DOWN);
            if (keyInfo.fsState & CONTROL)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case LEFTSHIFT:
            strcpy(nameVar+lenNameStem, YA_NAME_LEFT_SHIFT_DOWN);
            if (keyInfo.fsState & LEFTSHIFT)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         case RIGHTSHIFT:
            strcpy(nameVar+lenNameStem, YA_NAME_RIGHT_SHIFT_DOWN);
            if (keyInfo.fsState & RIGHTSHIFT)
               strcpy(valueVar, YA_DOWN);
            else
               strcpy(valueVar, YA_UP);
            break;

         default:
            lenNameVar = 0L;
            break;
         }

         if (lenNameVar > 0L)
         {
            lenNameVar = strlen(nameVar);
            MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
            blockVar.shvnamelen = lenNameVar;
            lenValueVar = strlen(valueVar);
            MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
            blockVar.shvvaluelen = lenValueVar;
            rc = RexxVariablePool(&blockVar);
            if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
               return YA_INVALID_ROUTINE;
         }

         fsState = fsState << 1;
      } while (fsState != 0x0000);
   }

   if (keyInfo.fbStatus & 0x40)   // final character?
   {
      retStr->strptr[0] = keyInfo.chChar;
      retStr->strptr[1] = 0x00;
      retStr->strlength = 1;

/* with scancode reversing for Slash or Enter keys on pinpad */
/* incompatible with SysGetKey
      if (keyInfo.fbStatus & 0x02)   // scancode?
      {
         if (keyInfo.chScan == 0xE0)  // Slash or Enter keys on pinpad?
         {
            retStr->strptr[0] = keyInfo.chScan;
            retStr->strptr[1] = keyInfo.chChar;
         } else
         {
            retStr->strptr[0] = keyInfo.chChar;
            retStr->strptr[1] = keyInfo.chScan;
         }
         retStr->strptr[2] = 0x00;
         retStr->strlength = 2;
      }
*/
/* without scancode reversing for Slash or Enter keys on pinpad */
/* compatible with SysGetKey */
      if ((keyInfo.fbStatus & 0x02) && (keyInfo.chScan != 0xE0))
      {
         retStr->strptr[0] = keyInfo.chChar;
         retStr->strptr[1] = keyInfo.chScan;
         retStr->strptr[2] = 0x00;
         retStr->strlength = 2;
      }

      return YA_VALID_ROUTINE;
   }

   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;
   return YA_VALID_ROUTINE;
}



/* ver = yaGetKeyVer( [stem] ) */
ULONG yaGetKeyVer( PUCHAR name,        // name of the function
                    ULONG argC,        // number of arguments
                 RXSTRING argV[],      // list of argument strings
                      PSZ queueName,   // current queue name
                PRXSTRING retStr       // returned result string
)
{
   APIRET     rc;
   SHVBLOCK   blockVar;
   CHAR       nameStem[YA_MAX];
   ULONG      lenNameStem = 0;
   CHAR       nameVar[YA_MAX];
   ULONG      lenNameVar = 0;
   CHAR       valueVar[YA_MAX];
   ULONG      lenValueVar = 0;
   INT        iSize, i = 0;

   retStr->strptr[0] = 0x00;
   retStr->strlength = 0;

   if (argC > 1L)
      return YA_INVALID_ROUTINE;

   switch(argC)
   {
   case 0L:
      break;
   case 1L:
      if (!RXVALIDSTRING(argV[0]))
         return YA_INVALID_ROUTINE;
      if (argV[0].strlength > YA_MAX/2)
         return YA_INVALID_ROUTINE;
      strcpy(nameStem, argV[0].strptr);
      lenNameStem = strlen(nameStem);
      strupr(nameStem);
      if (nameStem[lenNameStem-1] != '.')
      {
         nameStem[lenNameStem++] = '.';
         nameStem[lenNameStem] = 0x00;
      }
      break;
   default:
      return YA_INVALID_ROUTINE;
   }

   if (lenNameStem > 0L)
   {
      blockVar.shvcode = RXSHV_SET;
      blockVar.shvret  = (UCHAR)0;
      blockVar.shvnext = (PSHVBLOCK)0;
      sprintf(nameVar, "%s%s", nameStem, YA_NAME_ABOUT);
      lenNameVar = strlen(nameVar);
      MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
      blockVar.shvnamelen = lenNameVar;
      strcpy(valueVar, YA_ABOUT);
      lenValueVar = strlen(valueVar);
      MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
      blockVar.shvvaluelen = lenValueVar;
      rc = RexxVariablePool(&blockVar);
      if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
         return YA_INVALID_ROUTINE;

      iSize = sizeof(yaFncTable)/sizeof(PSZ);
      for (i = 0; i < iSize; i++)
      {
         blockVar.shvcode = RXSHV_SET;
         blockVar.shvret  = (UCHAR)0;
         blockVar.shvnext = (PSHVBLOCK)0;
         sprintf(nameVar, "%s%d", nameStem, i+1);
         lenNameVar = strlen(nameVar);
         MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
         blockVar.shvnamelen = lenNameVar;
         strcpy(valueVar, yaFncTable[i]);
         lenValueVar = strlen(valueVar);
         MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
         blockVar.shvvaluelen = lenValueVar;
         rc = RexxVariablePool(&blockVar);
         if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
            return YA_INVALID_ROUTINE;
      }
      blockVar.shvcode = RXSHV_SET;
      blockVar.shvret  = (UCHAR)0;
      blockVar.shvnext = (PSHVBLOCK)0;
      sprintf(nameVar, "%s%d", nameStem, 0);
      lenNameVar = strlen(nameVar);
      MAKERXSTRING(blockVar.shvname, nameVar, lenNameVar);
      blockVar.shvnamelen = lenNameVar;
      sprintf(valueVar, "%d", iSize);
      lenValueVar = strlen(valueVar);
      MAKERXSTRING(blockVar.shvvalue, valueVar, lenValueVar);
      blockVar.shvvaluelen = lenValueVar;
      rc = RexxVariablePool(&blockVar);
      if ((rc == RXSHV_NOAVL) || (blockVar.shvret & YA_RXSHV_BAD))
         return YA_INVALID_ROUTINE;
   }

   strcpy(valueVar, YA_VERSION);
   BUILDRXSTRING(retStr, valueVar);

   return YA_VALID_ROUTINE;
}


/*------------------------------------------------------------------------*/
/* End of file yaGetKey.c                                                 */
/**************************************************************************/

