/*
 * exits.c -- Shows how to control a REXX program using system exits.
 *            The REXX interpreter will call back into your application
 *            at certain points during the program execution.  
 *
 *   Notes about return values: an exit returns one of three values,
 *   either RXEXIT_HANDLED to say that the exit handled the condition,
 *   RXEXIT_NOT_HANDLED to let the interpreter handle it (i.e., as if
 *   the exit hadn't been called), or RXEXIT_RAISE_ERROR to raise
 *   a REXX error (error no. 48, "Failure in system service" -- there is
 *   no way to specify another error).
 *
 *   Every exit is passed a different data structure to hold the
 *   exit-specific information.  These structures are all defined
 *   in REXXSAA.H and are described online in the REXX Reference that
 *   comes with the OS/2 Toolkit.
 */

#define INCL_REXXSAA

#include <os2.h>
#include <rexxsaa.h>

#include <stdio.h>
#include <ctype.h>

#include "exits.h"

/*
 * Define alternate structures... CSet and Watcom C handle bitfields
 * differently, so to avoid the problem we just redefine the structures
 * to use unsigned chars....
 */

typedef _Packed struct {        /* fnc */
   unsigned char     rxfnc_flags ;     /* function flags             */
   PUCHAR            rxfnc_name;       /* Pointer to function name.  */
   USHORT            rxfnc_namel;      /* Length of function name.   */
   PUCHAR            rxfnc_que;        /* Current queue name.        */
   USHORT            rxfnc_quel;       /* Length of queue name.      */
   USHORT            rxfnc_argc;       /* Number of args in list.    */
   PRXSTRING         rxfnc_argv;       /* Pointer to argument list.  */
   RXSTRING          rxfnc_retc;       /* Return value.              */
}  alternate_RXFNCCAL_PARM;

typedef _Packed struct {        /* rx */
   unsigned char     rxcmd_flags;      /* error/failure flags        */
   PUCHAR            rxcmd_address;    /* Pointer to address name.   */
   USHORT            rxcmd_addressl;   /* Length of address name.    */
   PUCHAR            rxcmd_dll;        /* dll name for command.      */
   USHORT            rxcmd_dll_len;    /* Length of dll name.        */
   RXSTRING          rxcmd_command;    /* The command string.        */
   RXSTRING          rxcmd_retc;       /* Pointer to return buffer   */
}  alternate_RXCMDHST_PARM;

/*
 * Define the table of exits... we'll pass this table to RexxStart
 * when the macro is started... the name you use must be the name
 * you registered the exit with.  (See below.)
 *
 * You don't have to use different functions for each exit, because
 * the exit number is passed as an argument, but I find it easier
 * to understand what is going on if I use a separate function for
 * each exit.  It's up to you, though.
 */

RXSYSEXIT ExitList[] = {
    { "InitExit", RXINI },
    { "TermExit", RXTER },
    { "FuncExit", RXFNC },
    { "CmdExit",  RXCMD },
    { NULL, RXENDLST }
};

/*
 * InitExit -- This exit will be called just before the REXX program
 *             is about to start execution.  You can use this exit
 *             to "prestuff" the variable pool or do some other
 *             initialization.
 */
  
RexxExitHandler InitExit;

LONG InitExit( LONG exitnum, LONG subfunction, PEXIT parm )
  {
    exitnum = exitnum;
    subfunction = subfunction;
    parm = parm;

    printf( "*** InitExit: program is about to start\n" );

    return( RXEXIT_HANDLED );
  }

/*
 * TermExit -- This exit will be called just after the REXX program
 *             stops.  The variable pool is still valid, so you
 *             can fetch values from the variable pool.
 */
  
RexxExitHandler TermExit;

LONG TermExit( LONG exitnum, LONG subfunction, PEXIT parm )
  {
    exitnum = exitnum;
    subfunction = subfunction;
    parm = parm;

    printf( "*** TermExit: program has ended\n" );

    return( RXEXIT_HANDLED );
  }

/*
 * FuncExit -- This exit will be called whenever an external function
 *             needs to be called.  It passes all the information
 *             necessary to call the function, so you can intercept
 *             the function call and handle it yourself if you want.
 *
 *             You can use this exit as an alternative to registering
 *             external functions.  Note, however, that as with all
 *             exits, the call to RexxStart() must specify that this
 *             exit is to be called!
 */
  
RexxExitHandler FuncExit;

LONG FuncExit( LONG exitnum, LONG subfunction, PEXIT parm )
  {
    alternate_RXFNCCAL_PARM *fparm;

    exitnum = exitnum;
    subfunction = subfunction;

    fparm = (PVOID) parm;

    printf( "*** FuncExit: function '%s' being called\n", fparm->rxfnc_name );

    return( RXEXIT_NOT_HANDLED );
  }

/*
 * CmdExit -- This exit will be called whenever a subcommand handler
 *            needs to be called.  It passes different information,
 *            but it's the same idea as the FuncExit above.
 */
  
RexxExitHandler CmdExit;

LONG CmdExit( LONG exitnum, LONG subfunction, PEXIT parm )
  {
    alternate_RXCMDHST_PARM *cparm;

    exitnum = exitnum;
    subfunction = subfunction;

    cparm = (PVOID) parm;

    printf( "*** CmdExit: passing string '%s' to host '%s'\n",
            cparm->rxcmd_command.strptr, cparm->rxcmd_address );

    return( RXEXIT_NOT_HANDLED );
  }

/*
 * Exit registration... before starting any REXX macros, we have
 * to register our subcommand exits with the interpreter.  We just
 * keep a table of functions and register each one in sequence.
 *
 * Remember -- even after you register the exits, you still have to
 * pass a list of exits to RexxStart(), otherwise your exits won't
 * be called!
 */

typedef struct {
    PSZ name;
    PFN function;
} rxexit_entry, *rxexit_entry_p;

static rxexit_entry REXXExits[] =
  {
    { "InitExit",  (PFN) InitExit },
    { "TermExit",  (PFN) TermExit },
    { "FuncExit",  (PFN) FuncExit },
    { "CmdExit",   (PFN) CmdExit },
    { NULL,    NULL  }
  };

/*
 * RegisterREXXExits -- Register our REXX exit(s).  We use the
 *                      RexxRegisterExitExe function to do the registration.
 *
 *  NOTE: RexxRegisterExitExe takes three parms, one is the name of
 *        the handler to register (the name should be all uppercase),
 *        the second is the address of the function to call, and the
 *        third is the address of an 8-byte block of user-defined
 *        data to store with the handler.  The function does not need
 *        to be exported.  After registration the function is only
 *        available to the process that registered it.
 *
 *        In contrast, RexxRegisterExitDll requires that the function
 *        be exported from a DLL and is then made available to all REXX
 *        programs in any process, subject to the caveats above about
 *        declaring the exits when calling RexxStart.
 */

BOOL RegisterREXXExits( void )
  {
    rxexit_entry_p ptr;

    for( ptr = REXXExits; ptr->name != NULL; ++ptr ){
        RexxRegisterExitExe( ptr->name, ptr->function, NULL );
    }

    return( TRUE );
  }

/*
 * DeregisterREXXExits -- Deregister the exits.  Always good manners.
 */

void DeregisterREXXExits( void )
  {
    rxexit_entry_p ptr;

    for( ptr = REXXExits; ptr->name != NULL; ++ptr ){
        RexxDeregisterExit( ptr->name, NULL );
    }
  }
