IMPLEMENTATION MODULE FieldEditor;

        (********************************************************)
        (*                                                      *)
        (*              Screen editing utilities                *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        9 November 1997                 *)
        (*  Status:             Seems to be OK                  *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT
    (* type *)  CARD8, ADDRESS;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM Windows IMPORT
    (* type *)  Window,
    (* proc *)  WriteString, EditString;

FROM NumericIO IMPORT
    (* proc *)  WriteHexByte, EditHexByte, WriteRJCard, EditCardinal;

FROM RealIO IMPORT
    (* proc *)  WriteReal, EditReal;

(************************************************************************)

TYPE
    BytePtr = POINTER TO CARD8;
    CardPtr = POINTER TO CARDINAL;
    RealPtr = POINTER TO REAL;
    (*
    <* M2EXTENSIONS + *>
    StringPtr = POINTER TO ARRAY OF CHAR;
    <* M2EXTENSIONS - *>
    *)
    StringPtr = POINTER TO ARRAY [0..511] OF CHAR;

    FieldType =  POINTER TO RECORD
                                FieldWriter: WriteProc;
                                FieldEditor: EditProc;
                            END (*RECORD*);

(************************************************************************)
(*              SCREEN OUTPUT FOR THE PREDEFINED TYPES                  *)
(************************************************************************)

PROCEDURE WriteCardinalField (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: CardPtr;

    BEGIN
        address := p;
        WriteRJCard (w, address^, width);
    END WriteCardinalField;

(************************************************************************)

PROCEDURE WriteByteField (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: BytePtr;

    BEGIN
        address := p;
        WriteHexByte (w, address^);
    END WriteByteField;

(************************************************************************)

PROCEDURE WriteRealField (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: RealPtr;

    BEGIN
        address := p;
        WriteReal (w, address^, width);
    END WriteRealField;

(************************************************************************)

PROCEDURE WriteStringField (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: StringPtr;

    BEGIN
        address := p;
        WriteString (w, address^);
    END WriteStringField;

(************************************************************************)
(*                  EDITORS FOR THE PREDEFINED TYPES                    *)
(************************************************************************)

PROCEDURE EditByteField (w: Window;  VAR (*INOUT*) p: ADDRESS;  width: CARDINAL);

    VAR address: BytePtr;

    BEGIN
        address := p;
        EditHexByte (w, address^);
    END EditByteField;

(************************************************************************)

PROCEDURE EditCardinalField (w: Window;  VAR (*INOUT*) p: ADDRESS;  width: CARDINAL);

    VAR address: CardPtr;

    BEGIN
        address := p;
        EditCardinal (w, address^, width);
    END EditCardinalField;

(************************************************************************)

PROCEDURE EditRealField (w: Window;  VAR (*INOUT*) p: ADDRESS;  width: CARDINAL);

    VAR address: RealPtr;

    BEGIN
        address := p;
        EditReal (w, address^, width);
    END EditRealField;

(************************************************************************)

PROCEDURE EditStringField (w: Window;  VAR (*INOUT*) p: ADDRESS;  width: CARDINAL);

    VAR address: StringPtr;

    BEGIN
        address := p;
        EditString (w, address^, width);
    END EditStringField;

(************************************************************************)
(*                         COMPARING TYPES                              *)
(************************************************************************)

PROCEDURE SameType (t1, t2: FieldType): BOOLEAN;

    (* Returns TRUE iff t1 = t2.        *)

    BEGIN
        RETURN t1 = t2;
    END SameType;

(************************************************************************)
(*                      DEFINING A NEW TYPE                             *)
(************************************************************************)

PROCEDURE DefineFieldType (Writer: WriteProc;  Editor: EditProc): FieldType;

    (* Introduces a new field type into the system.  Writer is a        *)
    (* user-supplied procedure to write a variable of the new type.     *)
    (* Editor is the user-supplied procedure for editing a variable of  *)
    (* that type.                                                       *)

    VAR result: FieldType;

    BEGIN
        NEW (result);
        WITH result^ DO
            FieldWriter := Writer;
            FieldEditor := Editor;
        END (*WITH*);
        RETURN result;
    END DefineFieldType;

(************************************************************************)

PROCEDURE DiscardFieldType (type: FieldType);

    (* A notification from the user that this type will not be used     *)
    (* again (unless it is redefined by another call to procedure       *)
    (* DefineFieldType).  Use of this procedure is optional, but is     *)
    (* recommended for the sake of "clean" memory management.           *)

    BEGIN
        DISPOSE (type);
    END DiscardFieldType;

(************************************************************************)
(*                          SCREEN OUTPUT                               *)
(************************************************************************)

PROCEDURE WriteField (w: Window;  address: ADDRESS;  type: FieldType;
                                                        width: CARDINAL);

    (* Writes address^ on the screen at the current cursor position in  *)
    (* window w.  The width parameter specifies how many character      *)
    (* positions to use.  Use width=0 for variable-width fields for     *)
    (* which the write procedure for that type must work out the width. *)

    BEGIN
        WITH type^ DO
            FieldWriter (w, address, width);
        END (*WITH*);
    END WriteField;

(************************************************************************)
(*                           THE EDITOR                                 *)
(************************************************************************)

PROCEDURE EditField (w: Window;  VAR (*INOUT*) address: ADDRESS;
                                type: FieldType;  width: CARDINAL);

    (* Edits the variable at the given address, and of the given type,  *)
    (* at the current cursor position in window w.  The width parameter *)
    (* specifies how many character positions are to be used on the     *)
    (* screen.  Set width=0 for variable-width fields where the editor  *)
    (* must determine the width.  We leave this procedure on seeing a   *)
    (* keyboard character which does not belong to us.  The cursor is   *)
    (* left just beyond the last character of the field as it is        *)
    (* displayed.  The terminating keystroke is returned to the         *)
    (* keyboard driver so that it can still be read by the caller.      *)

    BEGIN
        WITH type^ DO
            FieldEditor (w, address, width);
        END (*WITH*);
    END EditField;

(************************************************************************)
(*                      SETTING UP THE PREDEFINED TYPES                 *)
(************************************************************************)

BEGIN
    Byte := DefineFieldType (WriteByteField, EditByteField);
    Cardinal := DefineFieldType (WriteCardinalField, EditCardinalField);
    Real := DefineFieldType (WriteRealField, EditRealField);
    String := DefineFieldType (WriteStringField, EditStringField);
FINALLY
    DiscardFieldType (Byte);
    DiscardFieldType (Cardinal);  DiscardFieldType (Real);
    DiscardFieldType (String);
END FieldEditor.

