(**************************************************************************)
(*                                                                        *)
(*  Program to work out module dependencies in a Modula-2 program.        *)
(*  Copyright (C) 2019   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE Files;

        (********************************************************)
        (*                                                      *)
        (*                  File operations                     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            13 January 2000                 *)
        (*  Last edited:        24 May 2019                     *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*     WARNING: This module contains some code that     *)
        (*     is specific for XDS Modula-2 on the OS/2         *)
        (*     operating system.  Adjust as appropriate when    *)
        (*     porting this to another system.                  *)
        (*                                                      *)
        (********************************************************)

IMPORT Strings, OS2, IOChan, IOConsts, ChanConsts, SeqFile, TextIO;

FROM STextIO IMPORT
    (* proc *)  WriteString, WriteLn;

FROM Scanner IMPORT
    (* proc *)  StripSpaces;

FROM WildCard IMPORT
    (* proc *)  WildMatch;

FROM Storage IMPORT
    (* proc *)  ALLOCATE;

(************************************************************************)

CONST
    (* NameSize gives the maximum number of characters from the module  *)
    (* name used to form the corresponding file name.  If the value is  *)
    (* zero then there is no limit.  For XDS, set the value to 8 if     *)
    (* you're generating DOS-compatible 8.3 file names, and to 0        *)
    (* otherwise.                                                       *)

    NameSize = 0;

    Nul = CHR(0);
    TableSize = 2048;

TYPE
    PathString = ARRAY [0..2047] OF CHAR;

    PathPtr = POINTER TO RECORD
                  next: PathPtr;
                  this: FilenameString;
              END (*RECORD*);

VAR
    (* Flag to permit returning .SYM and .OBJ results in the case where *)
    (* a source file is unavailable.                                    *)

    BinaryOK: BOOLEAN;

    (* The number of entries in Table. *)

    TCount: CARDINAL;

    (* Redirection table. *)

    Table: ARRAY [0..TableSize-1] OF
               RECORD
                   mask: PathString;
                   result: PathPtr;
               END (*RECORD*);

(************************************************************************)
(*               FINDING A FILE NAME IN THE PATH TABLE                  *)
(************************************************************************)

PROCEDURE LocateFile (VAR (*IN*) name: FilenameString;
                      VAR (*OUT*) fullname: FilenameString): BOOLEAN;

    (* Given a relative file name, expands it out to a fully qualified  *)
    (* file name, using the search paths as specified by the            *)
    (* redirection list.  The function result is TRUE iff the file was  *)
    (* found.  If not found, fullname is set to the null string.        *)

    VAR success: BOOLEAN;
        ppath: PathPtr;
        k: CARDINAL;

    BEGIN
        k := 0;  ppath := NIL;
        LOOP
            IF k >= TCount THEN
                EXIT (*LOOP*);
            END (*IF*);
            IF WildMatch (name, Table[k].mask) THEN
                ppath := Table[k].result;
                EXIT (*LOOP*);
            END (*IF*);
            INC (k);
        END (*LOOP*);
        success := FALSE;
        WHILE (NOT success) AND (ppath <> NIL) DO
            success := OS2.DosSearchPath( 0, ppath^.this, name,
                                     fullname, SIZE(fullname) ) = 0;
            ppath := ppath^.next;
        END (*WHILE*);
        IF NOT success THEN
            fullname[0] := Nul;
        END (*IF*);
        RETURN success;
    END LocateFile;

(************************************************************************)
(*               MAPPING FROM MODULE NAME TO FILE NAME                  *)
(************************************************************************)

PROCEDURE LocateModule (modulename: FilenameString;
                        definition: BOOLEAN;
                        VAR (*OUT*) filename: FilenameString;
                        VAR (*OUT*) IsSource: BOOLEAN);

    (* Given a module name, works out the full file name.  If parameter *)
    (* 'definition' is true then we are looking for a definition module.*)
    (* On return IsSource=TRUE iff this is a source file rather than    *)
    (* a binary file.                                                   *)

    VAR partname: FilenameString;

    BEGIN
        IF (NameSize > 0) AND (NameSize <= MAX(FilenameIndex)) THEN
            modulename[NameSize] := Nul;
        END (*IF*);
        IsSource := FALSE;
        Strings.Assign (modulename, partname);
        IF definition THEN
            Strings.Append (".DEF", partname);
            IF LocateFile (partname, filename) THEN
                IsSource := TRUE;
            ELSIF BinaryOK THEN
                Strings.Assign (modulename, partname);
                Strings.Append (".SYM", partname);
                IF LocateFile (partname, filename) THEN
                END (*IF*);
            END (*IF*);
        ELSE
            Strings.Append (".MOD", partname);
            IF LocateFile (partname, filename) THEN
                IsSource := TRUE;
            ELSIF BinaryOK THEN
                Strings.Assign (modulename, partname);
                Strings.Append (".OBJ", partname);
                IF LocateFile (partname, filename) THEN
                END (*IF*);
            END (*IF*);
        END (*IF*);
    END LocateModule;

(************************************************************************)
(*                         INITIALISATION                               *)
(************************************************************************)

PROCEDURE FixRelativeName (VAR (*IN*) BaseDir: FilenameString;
                           VAR (*INOUT*) name: FilenameString);

    (* If name is a relative name, makes it relative to BaseDir. *)

    VAR temp: FilenameString;

    BEGIN
        IF (name[0] <> '/') AND (name[0] <> '\') AND (name[1] <> ':') THEN
            temp := name;
            name := BaseDir;
            Strings.Append ('\', name);
            Strings.Append (temp, name);
        END (*IF*);
    END FixRelativeName;

(************************************************************************)

PROCEDURE SetTableEntry (k: CARDINAL;  BaseDir: FilenameString;
                                       pathstring: PathString);

    (* Fills in the result field for Table[k]. *)

    VAR path: FilenameString;
        previous, current: PathPtr;
        found: BOOLEAN;  pos: CARDINAL;

    BEGIN
        Table[k].result := NIL;  previous := NIL;
        StripSpaces (pathstring);
        WHILE pathstring[0] <> Nul DO
            Strings.Assign (pathstring, path);
            Strings.FindNext (';', pathstring, 0, found, pos);
            IF found THEN
                path[pos] := Nul;
                StripSpaces (path);
                Strings.Delete (pathstring, 0, pos+1);
                StripSpaces (pathstring);
            ELSE
                pathstring[0] := Nul;
            END (*IF*);
            FixRelativeName (BaseDir, path);
            NEW (current);
            current^.next := NIL;
            current^.this := path;
            IF previous = NIL THEN
                Table[k].result := current;
            ELSE
                previous^.next := current;
            END (*IF*);
            previous := current;
        END (*WHILE*);
    END SetTableEntry;

(************************************************************************)

PROCEDURE CreatePathTable (AllowBinary: BOOLEAN;  StartDirectory: FilenameString);

    (* Initialises the search path information.  The StartDirectory     *)
    (* parameter gives the directory in which the redirection file is   *)
    (* located.  If there is no redirection file, then this parameter   *)
    (* gives the directory where all source files should be found.      *)

    (* If AllowBinary is TRUE then the LocateModule will return .SYM    *)
    (* and .OBJ files in the case where the appropriate .DEF or .MOD    *)
    (* file could not be found.                                         *)

    CONST RedirectionFileName = "XC.RED";

    VAR cid: IOChan.ChanId;  res: ChanConsts.OpenResults;
        buffer: ARRAY [0..4095] OF CHAR;
        pathstring: PathString;
        found: BOOLEAN;  pos: CARDINAL;

    BEGIN
        BinaryOK := AllowBinary;
        TCount := 0;
        Strings.Assign (StartDirectory, buffer);
        Strings.Append ('\', buffer);
        Strings.Append (RedirectionFileName, buffer);
        SeqFile.OpenRead (cid, buffer, SeqFile.text, res);
        IF res = ChanConsts.opened THEN
            LOOP
                IF TCount >= TableSize THEN
                    WriteString ("Redirection table is full");
                    WriteLn;
                    EXIT (*LOOP*);
                END (*IF*);
                TextIO.ReadRestLine (cid, buffer);
                IF IOChan.ReadResult(cid) <> IOConsts.allRight THEN
                    EXIT (*LOOP*);
                END (*IF*);
                TextIO.SkipLine (cid);
                Strings.FindNext ('=', buffer, 0, found, pos);
                IF found THEN
                    Strings.Assign (buffer, pathstring);
                    Strings.Delete (pathstring, 0, pos+1);
                    buffer[pos] := Nul;
                    StripSpaces (buffer);
                    Strings.Assign (buffer, Table[TCount].mask);
                    SetTableEntry (TCount, StartDirectory, pathstring);
                    INC (TCount);
                END (*IF*);
            END (*LOOP*);
            SeqFile.Close (cid);
        ELSE
            Table[0].mask := '*';
            SetTableEntry (0, StartDirectory, '.');
            TCount := 0;
        END (*IF*);

    END CreatePathTable;

(************************************************************************)

BEGIN
    BinaryOK := FALSE;
END Files.

