#define INCL_WIN
#include <os2.h>
#include <pmstddlg.h>
#include <irect.hpp>
#include <iexcept.hpp>
#include <ireslib.hpp>
#include <ipoint.hpp>
#include <istring.hpp>
#include <iexcept.hpp>
#include <ievtdata.hpp>
#include <iapp.hpp>
#include "kvalset.hpp"

// style flags
const KValueSet::Style KValueSet::bitmapItems(VS_BITMAP);
const KValueSet::Style KValueSet::iconItems(VS_ICON);
const KValueSet::Style KValueSet::textItems(VS_TEXT);
const KValueSet::Style KValueSet::rgbItems(VS_RGB);
const KValueSet::Style KValueSet::colorIndexItems(VS_COLORINDEX);
const KValueSet::Style KValueSet::controlBorder(VS_BORDER);
const KValueSet::Style KValueSet::itemBorder(VS_ITEMBORDER);
const KValueSet::Style KValueSet::rightToLeft(VS_RIGHTTOLEFT);
const KValueSet::Style KValueSet::scaleGraphics(VS_SCALEBITMAPS);
const KValueSet::Style KValueSet::ownerDraw(VS_OWNERDRAW);
const KValueSet::Style KValueSet::classDefaultStyle(KValueSet::textItems | IWindow::visible);

KValueSet::Style KValueSet::defaultStyle ( )
{
  return classDefaultStyle;
}

void  KValueSet::setDefaultStyle( const Style& style )
{
   // not implemented yet
}

KValueSet::KValueSet ( unsigned long        windowId,
                       IWindow*             pwndParent,
                       IWindow*             pwndOwner,
                       const IRectangle&    initial,
                       const IPair&         dimension,
                       const    Style&      style)
  : usRowCount(dimension.coord1())
  , usColumnCount(dimension.coord2())
{

   // create a value set control data structure to pass dimensions
   VSCDATA vscData;
   vscData.cbSize = sizeof(vscData);
   vscData.usRowCount = dimension.coord1();
   vscData.usColumnCount = dimension.coord2();

   // create the control using a call to IWindow::create
   IWindowHandle whValueSet =
     create(windowId, 
            0,
            style.asUnsignedLong(),
            WC_VALUESET, 
            pwndParent->handle(),
            (pwndOwner==0) ? IWindowHandle(0) : pwndOwner->handle(),
            initial, 
            &vscData, 
            0);
   if (whValueSet == 0)
   {
      // indicates an error so throw an exception
      ITHROWGUIERROR("IWindow::create");
   }
   
   // tell IWindow to start handling events for the window
   startHandlingEventsFor(whValueSet);
}

KValueSet::KValueSet ( unsigned long        windowId,
                       IWindow*             pwndParent )
{
   setAutoDestroyWindow(false);
   startHandlingEventsFor(windowId, pwndParent);
}

KValueSet::KValueSet ( const IWindowHandle& whValueSet)
{
   setAutoDestroyWindow(false);
   startHandlingEventsFor(whValueSet);
}

KValueSet::~KValueSet ( )
{
}

KValueSet& KValueSet::setItem ( const IPair &location, 
                                const IResourceId& id )
{

  IASSERTPARM(location > IPair(0,0) && location <= IPair(usRowCount,usColumnCount));

  try
  {
    IBitmapHandle hdl = id.resourceLibrary().loadBitmap(id);
    setItem(location,hdl);
  } 
  catch(...)
  {
    try
    {
      IPointerHandle hdl = id.resourceLibrary().loadIcon(id);
      setItem(location,hdl);
    }
    catch(...)
    {
      IString str = id.resourceLibrary().loadString(id);
      setItem(location,str);
    }
  }
  return *this;
}

KValueSet& KValueSet::setItem ( const IPair &item, 
                                const IBitmapHandle&  hBitmap )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  setItemType(item, bitmap);

  IEventResult result =
           handle().sendEvent(VM_SETITEM,
                     IEventData(item.coord1(), item.coord2()),
                     IEventData(hBitmap));
  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEM");

  return *this;
}

KValueSet& KValueSet::setItem ( const IPair &item, 
                                const IPointerHandle& hIcon )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  setItemType(item, icon);

  IEventResult result =
           handle().sendEvent(VM_SETITEM,
                              IEventData(item.coord1(), item.coord2()),
                              IEventData(hIcon));
  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEM");
  return *this;
}

KValueSet& KValueSet::setItem ( const IPair &item, 
                                const IColor&         color )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  setItemType(item, rgb);

  IEventResult result =
           handle().sendEvent(VM_SETITEM,
                              IEventData(item.coord1(), item.coord2()),
                              IEventData((unsigned long)color.asRGBLong()));
  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEM");
  return *this;
}

KValueSet& KValueSet::setItem ( const IPair &item, 
                                IColor::Color index )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  setItemType(item, colorIndex);

  IEventResult result =
           handle().sendEvent(VM_SETITEM,
                              IEventData(item.coord1(), item.coord2()),
                              IEventData(IColor(index).value()));
  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEM");
  return *this;
}

KValueSet& KValueSet::setItem ( const IPair &item, 
                                const char*           str )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  setItemType(item, text);

  IEventResult result =
           handle().sendEvent(VM_SETITEM,
                              IEventData(item.coord1(), item.coord2()),
                              IEventData((char *)str));
  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEM");

  return *this;
}

KValueSet& KValueSet::setItemBmp ( const IPair &item, const unsigned long bmpId )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  HPS hpsPresentationSpace = WinGetPS (handle());

  HMODULE ResModule = ICurrentApplication::current().userResourceLibrary().handle();

  HBITMAP hBmp = GpiLoadBitmap (hpsPresentationSpace, ResModule, bmpId, 0L, 0L);

  setItemType(item, bitmap);

  if (WinSendMsg (handle(), VM_SETITEM, MPFROM2SHORT(item.coord1(), item.coord2()), MPFROMLONG(hBmp)) == FALSE)
    ITHROWGUIERROR("VM_SETITEM");

  return *this;
}

KValueSet& KValueSet::setItemIco ( const IPair &item, const unsigned long icoId )
{
  IASSERTPARM(item > IPair(0,0) & item <= IPair(usRowCount, usColumnCount));

  HMODULE ResModule = ICurrentApplication::current().userResourceLibrary().handle();

  HPOINTER hIcon = WinLoadPointer (HWND_DESKTOP, ResModule, icoId);

  setItemType(item, icon);

  if (WinSendMsg (handle(), VM_SETITEM, MPFROM2SHORT(item.coord1(), item.coord2()), MPFROMLONG(hIcon)) == FALSE)
    ITHROWGUIERROR("VM_SETITEM");

  return *this;
}

KValueSet& KValueSet::setItemType ( const IPair &item, 
                                    ItemType itemType )
{
  unsigned short usType;

  switch(itemType)
  {
    case text:
       usType = VIA_TEXT;
       break;

    case icon:
       usType = VIA_ICON;
       break;

    case bitmap:
       usType = VIA_BITMAP;
       break;

    case rgb:
       usType = VIA_RGB;
       break;

    case colorIndex:
       usType = VIA_COLORINDEX;
       break;
  }

  IEventResult result = 
           handle().sendEvent(VM_SETITEMATTR,
                              IEventData(item.coord1(), item.coord2()),
                              IEventData(usType | VIA_DRAGGABLE, true));

  if (!result.asUnsignedLong())
    ITHROWGUIERROR("VM_SETITEMATTR");
  return *this;
}

KValueSet::Index KValueSet::itemToIndex(const IPair &item)
{
  IEventData data(item.coord1(), item.coord2());
  return data;
}

IPair KValueSet::indexToItem(KValueSet::Index index)
{
  IEventData data(index);
  return(IPair(data.highNumber(), data.lowNumber()));
}

IPair KValueSet::selectedItem()
{
  IEventResult result = handle().sendEvent(VM_QUERYSELECTEDITEM);
  return(IPair(result.highNumber(), result.lowNumber()));
}

KValueSet::Index KValueSet::selectedIndex()
{
  return itemToIndex(selectedItem());
}

KValueSet& KValueSet::selectItem(const IPair &item)
{
  IEventResult result = handle().sendEvent(VM_SELECTITEM, 
                            IEventParameter1(item.coord1(), item.coord2()));
  if (result == 0)
      ITHROWGUIERROR("KValueSet::selectItem");

  return *this;
}

KValueSet& KValueSet::selectIndex(KValueSet::Index index)
{
  selectItem(indexToItem(index));
  return *this;
}
