/*  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Dodatkowe kontrolki panelu
 * (c) 1999 Wojciech Gazda
 *
 * ctlmisc.c
 *
 * $Author: Wojciech_Gazda $
 * $Date: 1999/04/18 22:54:59 $
 * $Name:  $
 * $RCSfile: status.c $
 * $Revision: 2.2 $
 *
 */

#define INCL_DOSDEV

// Deklaracje OS/2
#define INCL_DOSMISC
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#define INCL_GPIREGIONS
#define INCL_GPILCIDS
#define INCL_WINSYS
#define INCL_WINSTDDRAG
#define INCL_WINWINDOWMGR
#define INCL_WINMESSAGEMGR
#include <os2.h>

// Funkcje biblioteczne
#include <malloc.h>
#include <string.h>

// Deklaracje lokalne
#define  __INTERNAL_USE__
#include "asuintl.h"
#include "status.h"


// Deklaracje staych
// Indeksy kolejnych kolorw okna WC_STATUS
#define STA_FRAMEHILITE       0    // PP_BORDERLIGHTCOLOR        Owietlenie ramki
#define STA_FRAMEDARK         1    // PP_BORDERDARKCOLOR         Cie ramki
#define STA_ACTIVEBKGND       2    // PP_BACKGROUNDCOLOR         Kolor ta aktywnej ramki
#define STA_INACTVBKGND       3    // PP_DISABLEDBACKGROUNDCOLOR Kolor ta zablokowanej ramki
#define STA_ACTIVETEXT        4    // PP_FOREGROUNDCOLOR         Kolor tekstu aktywnej ramki
#define STA_INACTVTEXT        5    // PP_DISABLEDFOREGROUNDCOLOR Kolor tekstu nieaktywnej ramki
#define STA_FRAMEHILITE2      6    // PP_BORDER2LIGHTCOLOR       Kolor podwietlenia wewntrznego marginesu
#define STA_FRAMEDARK2        7    // PP_BORDER2DARKCOLOR        Kolor zacienienia wewntrznego marginesu

// Liczba kolorw
#define STA_MAXCOLOR          8

// Rozmiary okna stylu SS_WINSTYLE (stae rozmiary)
#define CX_WINSTYLE           12   // Szeroko okna z atrybutem SS_WINSTYLE
#define CY_WINSTYLE           12   // Wysoko okna z atrybutem SS_WINSTYLE



// Typy danych
// Struktura kontrolna okna WC_STATUS
typedef struct
{ FONTMETRICS fm;                  // Parametry biecej czcionki
  ULONG ulHilite;                  // Aktualny stan okna
  ULONG ulWinstyle;                // Styl okna
  ULONG ulColor[STA_MAXCOLOR];     // Kolory elementw okna
  ULONG ulTextWidth;               // Dugo tekstu w pikselach
  BOOL  bForceRedraw;              // Znacznik wymuszenia rysowania ta dla SS_OWNERDRAW
  PSZ   szWintext;                 // Tekst wywietlany w oknie
} STATUS;



// Zmienne globalne
// Lista parametrw prezentacji uytych w obrbie okna + lista odpowiadajcych im kolorw
static PPTEMPLATE PresParams[STA_MAXCOLOR] =
  {{ PP_BORDERLIGHTCOLOR,        0L,    // Owietlenie ramki
     0,                          0xFFFFFF,
   },
   { PP_BORDERDARKCOLOR,         0L,    // Cie ramki
     0,                          0x808080,
   },
   { PP_BACKGROUNDCOLOR,         0L,    // Kolor ta aktywnej ramki
     SYSCLR_ENTRYFIELD,          0xFFFFFF,
   },
   { PP_DISABLEDBACKGROUNDCOLOR, 0L,    // Kolor ta zablokowanej ramki
     SYSCLR_ENTRYFIELD,          0xFFFFFF,
   },
   { PP_FOREGROUNDCOLOR,         0L,    // Kolor tekstu aktywnej ramki
     SYSCLR_WINDOWTEXT,          0x000000,
   },
   { PP_DISABLEDFOREGROUNDCOLOR, 0L,    // Kolor tekstu nieaktywnej ramki
     0,                          0x808080,
   },
   { PP_BORDER2LIGHTCOLOR,       0L,    // Kolor podwietlenia wewntrznego marginesu
     0,                          0xCCCCCC,
   },
   { PP_BORDER2DARKCOLOR,        0L,    // Kolor zacienienia wewntrznego marginesu
     0,                          0x000000,
   },
  };



// Wersja kontrolki
static CTLINFO ctlStatusVersion =
{ sizeof(CTLINFO),
  NULL,
  2, 20
};



// Prototypy funkcji
LONG ASUAAPI CtlStatusInitialize(HAB hab);
MRESULT EXPENTRY WinStatusProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

// Prototypy funkcji lokalnych
static MRESULT EXPENTRY WinStatusProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
static VOID    StatAutoscale(HWND hwnd, SWP *swp);
static VOID    StatCalcInterior(RECTL *inter, ULONG winstyle);
static VOID    StatDrawFrame(HPS hps, RECTL *size, ULONG winstyle);
static VOID    StatDrawText(HWND hwnd, HPS hps, RECTL *size, STATUS *stat);
static VOID    StatDrawWinstyle(HPS hps, RECTL *size, STATUS *stat);
static LONG    StatLoadWndParams(HWND hwnd, WNDPARAMS *wpr);
static MRESULT StatProcessMouse(HWND hwnd);
static VOID    StatRedraw(HWND hwnd);
static VOID    StatSetPresParam(HWND hwnd, LONG ppm);
static VOID    StatSetText(HWND hwnd, WNDPARAMS *wpr);
static VOID    StatStoreFont(HWND hwnd, STATUS *stat);
static VOID    StatStoreWindowText(HWND hwnd, STATUS *stat, PSZ text);
static LONG    StatUserCommand(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);




// Przygotowanie okna WC_STATUS do pracy
//
// Parametry:
//   hab       - [parametr] uchwyt anchor block
//
// Powrt:
//   0    - poprawne wykonanie funkcji
//   1    - bd rejestracji klasy okna
//
LONG ASUAAPI CtlStatusInitialize(HAB hab)
{ ULONG winstyle;   // Domylny styl okna
  LONG  rc;         // Kody bdw zwracane przez funkcje systemowe

  // Ustalenie domylnego stylu okna WC_STATUS
  winstyle = CS_SIZEREDRAW | CS_HITTEST;
  // Rejestracja nowej klasy okna - samoskalujce si okno statusowe
  rc = WinRegisterClass(hab, WC_STATUS, WinStatusProc, winstyle, sizeof(VOID *));
  if(!rc) return(1);

  return(0);
}





/*******************/
/* Funkcje lokalne */
/*******************/

// Procedura rysujca i kontrolujca okno statusowe WC_STATUS
//
MRESULT EXPENTRY WinStatusProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{ STATUS  *stat;         // Adres struktur kontrolnych okna

  // Wykonywanie komunikatw
  switch(msg)
  { case WM_ADJUSTWINDOWPOS:  // Automatyczne skalowanie okna
      StatAutoscale(hwnd, (SWP *)PVOIDFROMMP(mp1));
      break;


    case WM_CREATE:           // Tworzenie okna
      { // Struktura inicjujca przekazywana przez system
        CREATESTRUCT *pcr;

        // Alokacja pamici dla struktury kontrolnej okna
        stat = malloc(sizeof(STATUS));
        if(stat == NULL) return(MRFROMLONG(TRUE));
        memset(stat, 0, sizeof(STATUS));

        // Odczyt adresu struktury inicjujcej
        pcr = (CREATESTRUCT *)PVOIDFROMMP(mp2);
        // Zaadowanie presentation parameters ze struktury CREATESTRUCT
        PpmStorePresParams(hwnd, (PRESPARAMS *)pcr->pPresParams, PresParams, STA_MAXCOLOR);
        // Zapamitanie kolorw pobranych z presentation parameters
        PpmQueryPresColors(hwnd, STA_MAXCOLOR, PresParams, (LONG *)stat->ulColor);
        // Zapamitanie czcionki pobranej z presentation parameters
        StatStoreFont(hwnd, stat);
        // Zapamitanie tekstu
        StatStoreWindowText(hwnd, stat, pcr->pszText);

        // Zapamitanie stylu okna
        stat->ulWinstyle = pcr->flStyle;
        // Standardowo, okno jest podwietlone
        stat->ulHilite = TRUE;

        // Zapamitanie wsknika do struktury kontrolnej
        WinSetWindowPtr(hwnd, 0L, stat);
        // Poprawne wykonanie inicjacji
        return(MRFROMLONG(FALSE));
      }
      break;


    case WM_DESTROY:          // Zamykanie okna
      { // Odczyt adresu struktur kontrolnych
        stat = WinQueryWindowPtr(hwnd, 0L);
        // Zwolnienie pamici
        if(stat->szWintext != NULL) free(stat->szWintext);
        if(stat != NULL)            free(stat);
        return(0);
      }
      break;


    case WM_ENABLE:           // Zmiana stanu okna - wymuszenie przerysowania
      WinInvalidateRect(hwnd, NULL, FALSE);
      break;


    case WM_HITTEST:               // STerowanie przepywem komunikatw myszy
      return(StatProcessMouse(hwnd));

    case WM_NULL:                  // Odczyt wersji kontrolki
      AsuQueryCtlInfo(hwnd, msg, mp1, &ctlStatusVersion);
      break;

    case WM_PAINT:                 // Przerysowanie okna
      StatRedraw(hwnd);
      break;


    case WM_PRESPARAMCHANGED:      // Obsuga WinSetPesParam
      StatSetPresParam(hwnd, LONGFROMMP(mp1));
      break;


    case WM_QUERYWINDOWPARAMS:     // Odczyt parametrw okna
      return(MRFROMLONG(StatLoadWndParams(hwnd, (WNDPARAMS *)PVOIDFROMMP(mp1))));


    case WM_SETWINDOWPARAMS:       // Obsuga WinSetWindowText
      StatSetText(hwnd, (WNDPARAMS *)PVOIDFROMMP(mp1));
      break;


    case WM_SYSCOLORCHANGE:        // Zmianie ulegy kolory systemowe
      // Zmianie ulegy kolory
      { STATUS *stat;     // Adres struktury kontrolnej okna

        // Odczyt adresu struktur kontrolnych
        stat = WinQueryWindowPtr(hwnd, 0L);
        if(stat == NULL) break;
        // Uaktualnienie kolorw
        PpmQueryPresColors(hwnd, STA_MAXCOLOR, PresParams, (LONG *)stat->ulColor);
      }
      break;
  }

  // Interpretacja komunikatw sterujcych
  if((msg >= STAT_FIRST_MSG) && (msg <= STAT_LAST_MSG))
    return(MRFROMLONG(StatUserCommand(hwnd, msg, mp1, mp2)));

  // Przekazanie nieprzetworzonych komunikatw do standardowej procedury okna
  return(WinDefWindowProc(hwnd, msg, mp1, mp2));
}





// Automatyczne skalowanie okna - dostosowaie do wymiarw tekstu
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   swp       - [parametr/rezultat] modyfikowana struktura, zawierajca poprawione wsprzdne
//
static VOID StatAutoscale(HWND hwnd, SWP *swp)
{ STATUS *stat;     // Adres struktury kontrolnej okna
  LONG    border;   // Szeroko obramowania

  // Odczyt adresu struktur kontrolnych
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return;

  // Sprawdzenie czy okno jest stylu SS_WINSTYLE
  if(stat->ulWinstyle & SS_WINSTYLE)
  { // Ustalenie staych rozmiarw
    swp->cx = CX_WINSTYLE;
    swp->cy = CY_WINSTYLE;
  }
  else
  { // Okrelenie szerokoci obramowania
    if(stat->ulWinstyle & SS_THINFRAME) border = 1;
    else                                border = 2;

    // Sprawdzenie czy mona skalowa w poziomie
    if(stat->ulWinstyle & SS_XAUTOSCALE)
    { // Uwzgldnienie dugoci tekstu
      swp->cx = stat->ulTextWidth + (stat->fm.lMaxCharInc >> 1);
      // Uwzglnienie ramki
      if(stat->ulWinstyle & SS_LEFTFRAME)  swp->cx += border;
      if(stat->ulWinstyle & SS_RIGHTFRAME) swp->cx += border;
    }

    // Sprawdzenie czy mona skalowa w pionie
    if(stat->ulWinstyle & SS_YAUTOSCALE)
    { // Uwzgldnienie wysokoci tekstu
      swp->cy = stat->fm.lMaxBaselineExt;
      // Uwzgldnienie ramki
      if(stat->ulWinstyle & SS_TOPFRAME)    swp->cy += border;
      if(stat->ulWinstyle & SS_BOTTOMFRAME) swp->cy += border;
    }
  }
}





// Funkcja oblicza rozmiar i pooenie obszaru wewntrz obramowania
//
// Parametry:
//   inter     - [parametr/rezultat] wsprzdne obszaru rdowego i wynikowego
//   winstyle  - [parametr] styl okna
//
static VOID StatCalcInterior(RECTL *inter, ULONG winstyle)
{ LONG border;      // Grubo ramki w pikselach

  // Okrelenie gruboci ramki
  if(winstyle & SS_THINFRAME) border = 1;
  else                        border = 2;

  // Odjcie gruboci lewej ramki
  if(winstyle & SS_LEFTFRAME)
    inter->xLeft += border;

  // Odjcie gruboci prawej ramki
  if(winstyle & SS_RIGHTFRAME)
    inter->xRight -= border;

  // Odjcie gruboci grnej ramki
  if(winstyle & SS_TOPFRAME)
    inter->yTop -= border;

  // Odjcie gruboci dolnejramki
  if(winstyle & SS_BOTTOMFRAME)
    inter->yBottom += border;
}





// Rysowanie obramowania okna, zalene od jego stylu
//
// Parametry:
//   hps       - [parametr] uchwyt presentation space
//   wsize     - [parametr] rozmiary okna
//   winstyle  - [parametr] styl okna
//
static VOID StatDrawFrame(HPS hps, RECTL *size, ULONG winstyle)
{ POINTL pos;       // Wsprzdne oglnego przeznaczenia
  RECTL  inter;     // Granice obszaru wewntrz obramowania
  RECTL  wsize;     // Rozmiary okna

  // Inicjacja zmiennych
  wsize = *size;
  inter = *size;

  // Zewntrzne obramowanie
  GpiSetColor(hps, (winstyle & SS_THINFRAME ? STA_FRAMEDARK2 : STA_FRAMEHILITE));
  // Prawa krawd
  pos.x = wsize.xRight - 1;
  if(winstyle & SS_RIGHTFRAME)
  { pos.y = wsize.yTop - 1;
    GpiMove(hps, &pos);
    pos.y = wsize.yBottom;
    GpiLine(hps, &pos);
    inter.xRight --;
  }
  else
  { pos.y = wsize.yBottom;
    GpiMove(hps, &pos);
  }

  // Dolna krawd
  pos.x = wsize.xLeft;
  if(winstyle & SS_BOTTOMFRAME)
  { GpiLine(hps, &pos);
    inter.yBottom ++;
  }
  else GpiMove(hps, &pos);

  // Lewa krawd
  GpiSetColor(hps, (winstyle & SS_THINFRAME ? STA_FRAMEDARK2 : STA_FRAMEDARK));
  pos.y = wsize.yTop - 1;
  if(winstyle & SS_LEFTFRAME)
  { GpiLine(hps, &pos);
    inter.xLeft ++;
  }
  else GpiMove(hps, &pos);

  // Grna krawd
  pos.x = wsize.xRight - 1;
  if(winstyle & SS_TOPFRAME)
  { GpiLine(hps, &pos);
    inter.yTop --;
  }


  // Sprawdzenie czy mona rysowa wewntrzne obramowanie
  if(!(winstyle & SS_THINFRAME))
    if((inter.xRight > inter.xLeft) && (inter.yTop > inter.yBottom))
    { // Zapamitanie nowego rozmiaru okna
      wsize = inter;

      // Wewntrzne obramowanie
      GpiSetColor(hps, (ulWarpVer > 30 ? STA_FRAMEDARK2 : STA_FRAMEHILITE));
      // Grna krawd
      pos.x = wsize.xRight - 1;
      pos.y = wsize.yTop - 1;
      if(winstyle & SS_TOPFRAME)
      { GpiMove(hps, &pos);
        pos.x = wsize.xLeft;
        GpiLine(hps, &pos);
      }
      else
      { pos.x = wsize.xLeft;
        GpiMove(hps, &pos);
      }

      // Lewa krawd
      pos.y = wsize.yBottom;
      if(winstyle & SS_LEFTFRAME)
        GpiLine(hps, &pos);
      GpiMove(hps, &pos);

      // Dolna krawd
      GpiSetColor(hps, (ulWarpVer > 30 ? STA_FRAMEHILITE2 : STA_FRAMEDARK));
      pos.x = wsize.xRight - 1;
      if(winstyle & SS_BOTTOMFRAME)
        GpiLine(hps, &pos);
      else
        GpiMove(hps, &pos);

      // Prawa krawd
      pos.y = wsize.yTop - 1;
      if(winstyle & SS_RIGHTFRAME)
        GpiLine(hps, &pos);

      // Zaokrglenie ramki (tylko Warp3)
      if(ulWarpVer < 40)
      { GpiSetColor(hps, STA_ACTIVEBKGND);
        if((winstyle & SS_TOPFRAME) && (winstyle & SS_RIGHTFRAME))
        { pos.x = wsize.xRight; pos.y = wsize.yTop;
          GpiSetPel(hps, &pos);
        }
        if((winstyle & SS_LEFTFRAME) && (winstyle & SS_BOTTOMFRAME))
        { pos.x = wsize.xLeft - 1; pos.y = wsize.yBottom - 1;
          GpiSetPel(hps, &pos);
        }
      }
    }
}





// Wywietlenie tekstu wewntrz okna
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   hps       - [parametr] uchwyt przestrzeni roboczej
//   size      - [parametr] obszar w obrbie ktrego ma by rysowany tekst
//   stat      - [parametr] wskanik do struktury kontrolnej okna
//
static VOID StatDrawText(HWND hwnd, HPS hps, RECTL *size, STATUS *stat)
{ AREABUNDLE abn;             // Atrybuty wypenie
  CHARBUNDLE cbn;             // Atrybuty tekstu
  POINTL pos;                 // wsprzdne oglnego przeznaczenia
  POINTL txt[TXTBOX_COUNT];   // Rzmiary acucha tekstowego
  RECTL  box;                 // Obszar w ktrym odbywa si rysowanie
  LONG   textw;               // Szeroko tekstu

  // Sprawdzenie czy mona rysowa tekst wewntrz okna
  if((size->xRight <= size->xLeft) || (size->yTop <= size->yBottom)) return;
  box = *size;

  // Ustalenie kolorw ta i tekstu w zalenoci od stanu okna
  if((WinIsWindowEnabled(hwnd) == TRUE) && (stat->ulHilite == TRUE))
  { // Okno jest aktywne
    GpiSetColor(hps, STA_ACTIVETEXT);
    // Kolor ta tekstu
    cbn.lBackColor = STA_ACTIVEBKGND;
    GpiSetAttrs(hps, PRIM_CHAR, CBB_BACK_COLOR, 0L, &cbn);
    // Kolor wypenienia
    abn.lBackColor = STA_ACTIVEBKGND;
    GpiSetAttrs(hps, PRIM_AREA, ABB_BACK_COLOR, 0L, &abn);
  }
  else
  { // Okno jest nieaktywne
    GpiSetColor(hps, STA_INACTVTEXT);
    // Kolor ta tekstu
    cbn.lBackColor = STA_INACTVBKGND;
    GpiSetAttrs(hps, PRIM_CHAR, CBB_BACK_COLOR, 0L, &cbn);
    // Kolor wypenienia
    abn.lBackColor = STA_INACTVBKGND;
    GpiSetAttrs(hps, PRIM_AREA, ABB_BACK_COLOR, 0L, &abn);
  }

  // Odczyt rozmiarw tekstu
  GpiQueryTextBox(hps, strlen(stat->szWintext), stat->szWintext, TXTBOX_COUNT, txt);
  // Sprawdzenie czy tekst nie jest odwrcony
  if(txt[TXTBOX_BOTTOMLEFT].x > txt[TXTBOX_BOTTOMRIGHT].x)
  { // Odwrcenie tekstu do waciwej pozycji
    cbn.ptlAngle.x = -1;
    cbn.ptlAngle.y = 0;
    GpiSetAttrs(hps, PRIM_CHAR, CBB_ANGLE , 0L, &cbn);
  }

  if(stat->szWintext != NULL)
  { // Wywietlenie tekstu
    // Obliczenie pocztkowej pozycji pionowej
    pos.y  = (box.yTop + box.yBottom - stat->fm.lMaxBaselineExt) >> 1;
    pos.y += stat->fm.lMaxDescender;

    // Obliczenie pocztkowej pozycji poziomej
    switch(stat->ulWinstyle & 0x60)
    { case SS_LEFT:
        pos.x = box.xLeft + (stat->fm.lMaxCharInc >> 2); break;

      case SS_CENTER:
        pos.x = (box.xRight + box.xLeft - stat->ulTextWidth) >> 1; break;

      case SS_RIGHT:
        pos.x = box.xRight - (stat->fm.lMaxCharInc >> 2) - stat->ulTextWidth; break;

      case SS_AUTOSHIFT:
        // Obliczenie cakowitej szerokoci tekstu
        textw = stat->ulTextWidth + ((stat->fm.lMaxCharInc & 0xFFFFFFFC) >> 1);
        if(textw < (box.xRight - box.xLeft))
          pos.x = box.xLeft + (stat->fm.lMaxCharInc >> 2);
        else
          pos.x = box.xRight - (stat->fm.lMaxCharInc >> 2) - stat->ulTextWidth;
    }

    // Korekta wsprzdnych obszaru ograniczajcego
    box.xRight --; box.yTop --;

    if((stat->ulWinstyle & SS_OWNERDRAW) && (stat->bForceRedraw == TRUE))
    { // Wywietlenie tekstu bez ta
      GpiCharStringPosAt(hps, &pos, &box, CHS_CLIP,
                         strlen(stat->szWintext), stat->szWintext, NULL);
    }
    else
    { // Wywietlenie tekstu z tem
      GpiCharStringPosAt(hps, &pos, &box, CHS_CLIP | CHS_OPAQUE,
                         strlen(stat->szWintext), stat->szWintext, NULL);
    }

  }
  else
    if(!(stat->ulWinstyle & SS_OWNERDRAW) || (stat->bForceRedraw == FALSE))
    { // Wykasowanie ta
      pos.x = box.xLeft;  pos.y = box.yBottom;
      GpiMove(hps, &pos);
      pos.x = box.xRight; pos.y = box.yTop;
      GpiBox(hps, DRO_FILL, &pos, 0, 0);
    }
}





// Rysowanie przycisku skalowania, wywietlanego zwykle w prawym dolnym
// rogu okien.
//
// Parametry:
//   hps       - [parametr] uchwyt presentation space
//   size      - [parametr] wymiary okna
//   stat      - [parametr] wskanik do struktur kontrolnych okna
//
static VOID StatDrawWinstyle(HPS hps, RECTL *size, STATUS *stat)
{ POINTL xpos;      // Wsprzdne przy dolnej krawdzi
  POINTL ypos;      // Wsprzdne przy prawej krawdzi
  LONG   i;         // Zmienne pomocnicze

  // Rysowanie ta w przypadku, gdy okno jest otwarte ze stylem SS_OWNERDRAW
  // i to nie zostao przerysowane podczas obsugi WM_DRAWITEM
  if((stat->bForceRedraw == FALSE) || !(stat->ulWinstyle & SS_OWNERDRAW))
  { GpiSetColor(hps, STA_ACTIVEBKGND);
    GpiMove(hps, (POINTL *)size);
    GpiBox(hps, DRO_FILL, ((POINTL *)size) + 1, 0, 0);
  }

  // Rysowanie cieniowanych belek
  GpiSetColor(hps, STA_FRAMEDARK);
  // Wsprzdne pocztkowe
  xpos.x = size->xLeft + 1;
  xpos.y = size->yBottom;
  ypos.x = size->xRight - 1;
  ypos.y = size->yTop - 2;
  for(i = 0; i < 3; ++i)
  { GpiMove(hps, &xpos); GpiLine(hps, &ypos);
    xpos.x ++; ypos.y --;
    GpiMove(hps, &xpos); GpiLine(hps, &ypos);
    xpos.x += 3; ypos.y -= 3;
  }

  // Rysowanie owietlenia belek
  GpiSetColor(hps, STA_FRAMEHILITE);
  // Wsprzdne pocztkowe
  xpos.x = size->xLeft;
  xpos.y = size->yBottom;
  ypos.x = size->xRight- 1;
  ypos.y = size->yTop - 1;
  for(i = 0; i < 3; ++i)
  { GpiMove(hps, &xpos); GpiLine(hps, &ypos);
    xpos.x += 4; ypos.y -= 4;
  }
}





// Funkcja zapewnia dziaanie funkcjom:
// WinQueryWindowText, WinQueryWindowTextLength,
// wykonanym z uchwytem okna WC_STATUS
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   wpr       - [parametr/rezultat] zwracane parametry okna
//
// Powrt:
//   0    - FALSE bd odczytu
//   1    - TRUE  parametr umiesczono w strukturze WNDPARAMS
//
static LONG StatLoadWndParams(HWND hwnd, WNDPARAMS *wpr)
{ STATUS *stat;     // Wskanik do struktur kontrolnych okna statusowego
  LONG    rc;       // Kod bdu zwracany przez funkcje

  // Odczyt wskanika do struktur kontrolnych okna statusowego
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return(FALSE);
  rc = 0;

  // Wybr odpowiedniego parametru do odczytu
  if(wpr->fsStatus & WPM_CCHTEXT)
  { // Odczyt dugoci tekstu
    // Sprawdzenie czy jest tekst
    if(stat->szWintext == NULL)
      wpr->cchText = 0;
    else wpr->cchText = strlen(stat->szWintext);
    rc = 1;
  }

  if(wpr->fsStatus & WPM_TEXT)
  { // Odczyt tekstu
    if(stat->szWintext != NULL)
    { strcpy(wpr->pszText, stat->szWintext);
      rc = 1;
    }
    else rc = 0;
  }

  // Dane class specific niedostpne, bo ich nie ma
  return(rc);
}





// Funkcja przetwarza komunikat WM_HITTEST, dziki czemu komunikaty myszy
// mog by przekazywane do procedury waciciela, wtedy gdy
// uyto stylu SS_MOUSETRANSPARENT.
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna statusu
//
// Powrt:
//   HT_ERROR       - okno jest zablokowane (disabled)
//   HT_TRANSPARENT - uyto stylu SS_MOUSETRANSPARENT
//   HT_NORMAL      - w przeciwnym wypadku
//
static MRESULT StatProcessMouse(HWND hwnd)
{ STATUS   *stat;        // Wsjkanik do struktur kontrolnych okna statusowego

  // Sprawdzenie czy okno jest zablokowane
  if(WinIsWindowEnabled(hwnd) == FALSE)
    return(MRFROMLONG(HT_ERROR));

  // Odczyt wskanika do struktur kontrolnych okna statusowego
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return(MRFROMLONG(HT_NORMAL));

  // Sprawdzenie czy okno ma by przezroczyste dla myszy
  if(stat->ulWinstyle & SS_MOUSETRANSPARENT)
    return(MRFROMLONG(HT_TRANSPARENT));
  return(MRFROMLONG(HT_NORMAL));
}




// Przerysowanie okna statusu
//
// Parametry:
//   hwnd      - [parametr] uchwyt przerysowywanego okna
//
static VOID StatRedraw(HWND hwnd)
{ OWNERBACK bkgnd;       // Struktura umoliwiajca wacicielowi przerysowanie ta
  STATUS   *stat;        // Wsjkanik do struktur kontrolnych okna statusowego
  RECTL     wsize;       // Aktualne rozmiary okna
  RECTL     inter;       // Obszar uyteczny wewntrz obramowania
  HWND      owner;       // Uchwyt waciciela
  LONG      winid;       // Identyfikator okna hwnd
  HPS       hps;         // Uchwyt presentation space okna


  // Odczyt wskanika do struktur kontrolnych okna statusowego
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return;

  // Odczyt wymiarw okna, inicjacja zmiennych pomocniczych
  WinQueryWindowRect(hwnd, &wsize);
  // Obliczenie pozycji obszaru wewntrz obramowania
  inter = wsize;
  StatCalcInterior(&inter, stat->ulWinstyle);

  // Rozpoczcie przerysowywania
  hps = WinBeginPaint(hwnd, NULLHANDLE, NULL);
  // Tworzenie logicznej tablicy kolorw
  GpiCreateLogColorTable(hps, 0L, LCOLF_CONSECRGB, 0, STA_MAXCOLOR, (LONG *)stat->ulColor);

  // Wysanie komunikatu do waciciela, celem przerysowania ta
  if(stat->ulWinstyle & SS_OWNERDRAW)
  { // Odczyt uchwytu waciciela
    owner = WinQueryWindow(hwnd, QW_OWNER);
    if(owner != NULLHANDLE)
    { // Odczyt identyfikatora
      winid = WinQueryWindowUShort(hwnd, QWS_ID);

      // Inicjacja struktury sterujcej przerysowywaniem
      bkgnd.hwnd       = hwnd;
      bkgnd.hps        = hps;
      bkgnd.bound      = inter;
      bkgnd.refpoint.x = 0;
      bkgnd.refpoint.y = 0;
      // Wysanie komunikatu z daniem przerysowania ta
      stat->bForceRedraw = (BOOL)WinSendMsg(owner, WM_DRAWITEM, MPFROMSHORT(winid), MPFROMP(&bkgnd));
    }
  }

  if(stat->ulWinstyle & SS_WINSTYLE)
  { // Rysowanie przycisku skalowania okna
    StatDrawWinstyle(hps, &wsize, stat);
  }
  else
  { // Rysowanie fragmentw obramowania (zalenie od stylu okna)
    StatDrawFrame(hps, &wsize, stat->ulWinstyle);
    // Rysowanie tekstu (zalenie od stylu okna)
    StatDrawText(hwnd, hps, &inter, stat);
  }

  // Zakoczenie przerysowywania
  WinEndPaint(hps);
}





// Aktualizacja wybranego parametru prezentacji
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   ppm       - [parametr] identyfikator parametru
//
static VOID  StatSetPresParam(HWND hwnd, LONG ppm)
{ STATUS *stat;     // Wskanik do struktur kontrolnych okna
  RECTL   wsize;    // Aktualny rozmiar okna

  // Odczyt adresu struktur kontrolnych okna
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return;

  // Sprawdzenie czy zmianie ulega czcionka
  // lub kilka parametrw na raz - ppm = 0
  if((ppm == PP_FONTNAMESIZE) || (ppm == 0))
  { // Auaktualnienie struktur okna
    StatStoreFont(hwnd, stat);

    // Wymuszenie przeskalowania lub przerysowania
    if(stat->ulWinstyle & (SS_XAUTOSCALE | SS_YAUTOSCALE))
    { // Odczyt aktualnych rozmiarw okna
      WinQueryWindowRect(hwnd, &wsize);
      // Wymuszenie skalowania po zmianie czcionki
      WinSetWindowPos(hwnd, HWND_TOP, 0, 0,
                      wsize.xRight - wsize.xLeft,
                      wsize.yTop   - wsize.yBottom,
                      SWP_SIZE);
    }
    else
      // Wymuszenie przerysowania
      WinInvalidateRect(hwnd, NULL, FALSE);
  }

  // Sprawdzenie czy zmianie ulegy kolory
  // lub kilka parametrw na raz - ppm = 0
  if((ppm != PP_FONTNAMESIZE) || (ppm == 0))
  { // Zmianie ulegy kolory
    PpmQueryPresColors(hwnd, STA_MAXCOLOR, PresParams, (LONG *)stat->ulColor);
    // Wymuszenie przerysowania
    WinInvalidateRect(hwnd, NULL, FALSE);
  }

  // Sprawdzenie czy komunikat WM_PRESPARAMCHANGED zosta wysany
  // przez inny wtek - systemow palet czcionek lub kolorw
  if((ppm == PP_FONTNAMESIZE) || (ppm == PP_BACKGROUNDCOLOR) || (ppm == PP_FOREGROUNDCOLOR))
  { HWND owner;     // Uchwyt okna waciciela
    LONG winid;     // Identyfikator biecego okna
    HAB  hab;       // Uchwyt anchor block

    // Odczyt uchwytu anchor block
    hab = WinQueryAnchorBlock(hwnd);
    // Sprawdzenie czy zmiana parametru jest inicjowana przez inny wtek
    if(WinInSendMsg(hab) == TRUE)
    { // Sprawdzenie czy parametr istnieje w obrbie okna, czy te jest dziedziczony
      if(WinQueryPresParam(hwnd, ppm, 0, NULL, sizeof(ULONG), &winid, QPF_NOINHERIT))
      { // Odczyt uchwytu waciciela
        owner = WinQueryWindow(hwnd, QW_OWNER);
        // Odczyt identyfikatora okna
        winid = WinQueryWindowUShort(hwnd, QWS_ID);
        // Wstawienie potwierdzenia do waciciela okna
        if(owner != NULLHANDLE)
          WinPostMsg(owner, WM_CONTROL,
                     MPFROM2SHORT(winid, SMN_PRESPARAMCHANGED),
                     MPFROMLONG(ppm));

      }
    }
  }
}





// Ustawienie nowego tekstu wewntrz okna
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   text      - [parametr] zapisywany tekst
//
static VOID StatSetText(HWND hwnd, WNDPARAMS *wpr)
{ STATUS *stat;
  RECTL   wsize;

  // Odczyt adresu struktur kontrolnych okna
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return;
  // Odczyt aktualnych rozmiarw okna
  WinQueryWindowRect(hwnd, &wsize);

  if(wpr->fsStatus & WPM_TEXT)
  { // Zapamitanie tekstu
    StatStoreWindowText(hwnd, stat, wpr->pszText);

    // Przeskalowanie okna w trybie AUTOSCALE
    if(stat->ulWinstyle & (SS_XAUTOSCALE | SS_YAUTOSCALE))
    { WinSetWindowPos(hwnd, HWND_TOP, 0, 0,
                      wsize.xRight - wsize.xLeft,
                      wsize.yTop   - wsize.yBottom,
                      SWP_SIZE);
    }
    // Przerysowanie okna
    WinInvalidateRect(hwnd, NULL, FALSE);
  }
}





// Zapamitanie nazwy i parametrw czcionki odczytanej
// z presentation parameters, obliczenie i zapamitanie
// struktury FONTMETRICS
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   stat      - [parametr/rezultat] wskanik do struktur kontrolnych okna
//
static VOID StatStoreFont(HWND hwnd, STATUS *stat)
{ POINTL txt[TXTBOX_COUNT];   // Rozmiary tekstu
  HPS    hps;                 // Uchwyt pomocniczej presentation space
  LONG   rc;                  // Kody bdw zwracane przez funkcje systemowe

  // Utworzenie pomocniczej presentation space
  hps = WinGetPS(hwnd);
  // Odczyt parametrw domylnej czcionki okna
  GpiQueryFontMetrics(hps, sizeof(FONTMETRICS), &stat->fm);

  // Pomiar dugoci tekstu (jeli istnieje)
  if(stat->szWintext != NULL)
  { GpiQueryTextBox(hps, strlen(stat->szWintext), stat->szWintext, TXTBOX_COUNT, txt);
    rc = txt[TXTBOX_BOTTOMRIGHT].x - txt[TXTBOX_BOTTOMLEFT].x;
    if(rc < 0) rc = -rc;
    stat->ulTextWidth = rc;
  }

  // Zwolnienie pomocniczej presentation space
  hps = WinReleasePS(hps);
}





// Zapamitanie tekstu w wewntrznych strukturach okna, odczyt
// dugoci tekstu w pikselach, dla czcionki uytej w obrbie okna
//
// Parametry:
//   hwnd      - [parametr] uchwyt okna
//   stat      - [parametr/rezultat] struktury kontrolne okna
//   text      - [parametr] wskanik do zapamietywanego tekstu
//
static VOID StatStoreWindowText(HWND hwnd, STATUS *stat, PSZ text)
{ POINTL txt[TXTBOX_COUNT];   // Rozmiary tekstu odczytywane przez GpiQueryTextBox
  HPS hps;                    // Uchwyt pomocniczej presentation space
  PSZ tmp;                    // Wskanik pomocniczy


  // Zapamitanie tekstu wywietlanego w oknie
  if(text != NULL)
  { if(stat->szWintext == NULL)
      stat->szWintext = malloc(strlen(text) + 1);
    else
    { tmp = realloc(stat->szWintext, strlen(text) + 1);
      if(tmp == NULL)
      { free(stat->szWintext);
        stat->szWintext = NULL;
      }
      else stat->szWintext = tmp;
    }
    if(stat->szWintext != NULL)
      strcpy(stat->szWintext, text);
  }
  else
   if(stat->szWintext != NULL)
   { free(stat->szWintext);
     stat->szWintext = NULL;
   }

  // Obliczenie dugoci tekstu wyraonej w pikselach
  hps = WinGetPS(hwnd);

  // Obliczenie i zapamitanie dugoci tekstu w pikselach
  GpiQueryTextBox(hps, strlen(stat->szWintext), stat->szWintext, TXTBOX_COUNT, txt);
  stat->ulTextWidth = txt[TXTBOX_BOTTOMRIGHT].x - txt[TXTBOX_BOTTOMLEFT].x;

  // Zwolnienie pomocniczej presentation space
  WinReleasePS(hps);
}





// Interpretacja komunikatw sterujcych oknem
//
static LONG StatUserCommand(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{ STATUS *stat;     // Wskanik do struktur kontrolnych okna
  LONG border;      // Grubo obramowania
  LONG rc;          // Kod zwracany przez funkcj

  // Odczyt adresu struktur kontrolnych okna
  stat = WinQueryWindowPtr(hwnd, 0L);
  if(stat == NULL) return(0);

  rc = 0;
  switch(msg)
  { case SM_QUERYMAXCHARINC:   // Odczyt max. szerokoci znaku - lAveCharWidth
      return(stat->fm.lMaxCharInc);

    case SM_QUERYTEXTHEIGHT:   // Odczyt wysokoci tekstu - lMaxBaselineExt
      return(stat->fm.lMaxBaselineExt);

    case SM_QUERYTEXTWIDTH:    // Odczyt szerokoci tekstu
      return(stat->ulTextWidth);

    case SM_QUERYHEIGHT:       // Odczyt wysokoci okna
      if(stat->ulWinstyle & SS_WINSTYLE)
      { rc = CY_WINSTYLE;
      }
      else
      { rc = stat->fm.lMaxBaselineExt;
        border = stat->ulWinstyle & SS_THINFRAME ? 1 : 2;
        if(stat->ulWinstyle & SS_TOPFRAME)    rc += border;
        if(stat->ulWinstyle & SS_BOTTOMFRAME) rc += border;
      }
      break;

    case SM_QUERYHILITE:       // Odczyt aktywnoci okna
      return(stat->ulHilite);

    case SM_QUERYWIDTH:        // Odczyt szerokoci okna
      if(stat->ulWinstyle & SS_WINSTYLE)
      { rc = CX_WINSTYLE;
      }
      else
      { rc = stat->ulTextWidth + ((stat->fm.lMaxCharInc & 0xFFFFFFFC) >> 1);
        border = stat->ulWinstyle & SS_THINFRAME ? 1 : 2;
        if(stat->ulWinstyle & SS_LEFTFRAME)  rc += border;
        if(stat->ulWinstyle & SS_RIGHTFRAME) rc += border;
      }
      break;

    case SM_SETHILITE:         // Aktywacja / dezaktywacja okna
      if(LONGFROMMP(mp1) == TRUE)
      { // danie aktywacji okna
        if(stat->ulHilite == TRUE) return(0);
        // Aktywacja okna
        stat->ulHilite = TRUE;
        WinInvalidateRect(hwnd, NULL, FALSE);
      }
      else
      { // danie dezaktywacji okna
        if(stat->ulHilite == FALSE) return(0);
        // Dezaktywacja okna
        stat->ulHilite = FALSE;
        WinInvalidateRect(hwnd, NULL, FALSE);
      }
      return(0);
  }

  return(rc);
}

/*
 * $Log: status.c $
 * Revision 2.2  1999/04/18 22:54:59  Wojciech_Gazda
 * Nieudany (na razie) styl SS_WINSTYLE, poprawka komunikatu SM_QUERYWIDTH
 *
 * Revision 2.1  1999/03/20 20:09:09  Wojciech_Gazda
 * Face-lifting - dostosowanie wygldu do OS/2 Warp 3
 *
 * Revision 1.9  1999/02/28 22:16:49  Wojciech_Gazda
 * Dodano styl SS_MOUSETRANSPARENT i reakcj na zmian kolorw systemowych
 *
 * Revision 1.8  1999/02/26 23:14:54  Wojciech_Gazda
 * Dodanie komunikatw aktywujcych i dezaktywujcych okno
 *
 * Revision 1.7  1999/02/21 15:35:03  Wojciech_Gazda
 * Usunicie rcznej obsugi czcionek - system robi to sam
 *
 * Revision 1.6  1999/02/21 15:30:00  Wojciech_Gazda
 * Potwierdzenie SMN_PRESPARAMCHANGED generowane tylko po upuszczeniu
 * czego z palety systemowej, wstawiane a nie wysyane do kolejki waciciela
 *
 * Revision 1.5  1999/02/20 19:44:30  Wojciech_Gazda
 * Zmiana struktury OWNERBACK
 *
 * Revision 1.4  1999/02/11 12:41:53  Wojciech_Gazda
 * Dodanie stylu SS_AUTOSHIFT
 *
 * Revision 1.3  1999/02/03 22:24:18  Wojciech_Gazda
 * Obsuga presentation parameters - wyjazd do osobnego pliku
 * dodano obsug WinQueryWindowText i WinQueryWindowTextLength
 *
 * Revision 1.2  1999/02/03 20:33:08  Wojciech_Gazda
 * Poprawiono SM_QUERYWIDTH - zwraca co trzeba
 *
 * Revision 1.1  1999/01/31 15:11:55  Wojciech_Gazda
 * Initial revision
 *
 */
