#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include <limits.h>
#include <string.h>
#include <sqlcli.h>
#include <sqlcli1.h>
#include <sqlext.h>

#include "const.h"

#define	SmallSize	32767
#define	HugeSize	(1024 * 1024)

static struct {
	SQLINTEGER	length;
	SQLCHAR		buffer[HugeSize];
} buffer;
static SQLSMALLINT	len1, len2;
#define buf1		(buffer.buffer + 0)
#define buf2		(buffer.buffer + SmallSize)

static int		db2cli_debug;

typedef struct {
	STRLEN	len;
	char *	ptr;
} SQLString, SQLLargeString;

typedef struct {
	int	code;
	char *	text;
} TypeMap;

static SQLString null = { 0, "" };

static void
initialize(void)
{
	char *cp = getenv("DB2CLI_DEBUG");
	if (cp)
		db2cli_debug = atoi(cp);
}

static TypeMap cTypes[] = {
	{ SQL_C_CHAR, "SQL_C_CHAR" },
	{ SQL_C_LONG, "SQL_C_LONG" },
	{ SQL_C_SHORT, "SQL_C_SHORT" },
	{ SQL_C_FLOAT, "SQL_C_FLOAT" },
	{ SQL_C_DOUBLE, "SQL_C_DOUBLE" },
	{ SQL_C_DATE, "SQL_C_DATE" },
	{ SQL_C_TIME, "SQL_C_TIME" },
	{ SQL_C_TIMESTAMP, "SQL_C_TIMESTAMP" },
	{ SQL_C_BINARY, "SQL_C_BINARY" },
	{ SQL_C_BIT, "SQL_C_BIT" },
	{ SQL_C_TINYINT, "SQL_C_TINYINT" },
	{ SQL_C_DBCHAR, "SQL_C_DBCHAR" },
	{ SQL_C_SLONG, "SQL_C_SLONG" },
	{ SQL_C_SSHORT, "SQL_C_SSHORT" },
	{ SQL_C_STINYINT, "SQL_C_STINYINT" },
	{ SQL_C_ULONG, "SQL_C_ULONG" },
	{ SQL_C_USHORT, "SQL_C_USHORT" },
	{ SQL_C_UTINYINT, "SQL_C_UTINYINT" },
	{ SQL_C_BLOB_LOCATOR, "SQL_C_BLOB_LOCATOR" },
	{ SQL_C_CLOB_LOCATOR, "SQL_C_CLOB_LOCATOR" },
	{ SQL_C_DBCLOB_LOCATOR, "SQL_C_DBCLOB_LOCATOR" },
	{ 0, 0 }
};

static TypeMap sqlTypes[] = {
	{ SQL_CHAR, "SQL_CHAR" },
	{ SQL_NUMERIC, "SQL_NUMERIC" },
	{ SQL_DECIMAL, "SQL_DECIMAL" },
	{ SQL_INTEGER, "SQL_INTEGER" },
	{ SQL_SMALLINT, "SQL_SMALLINT" },
	{ SQL_FLOAT, "SQL_FLOAT" },
	{ SQL_REAL, "SQL_REAL" },
	{ SQL_DOUBLE, "SQL_DOUBLE" },
	{ SQL_DATE, "SQL_DATE" },
	{ SQL_TIME, "SQL_TIME" },
	{ SQL_TIMESTAMP, "SQL_TIMESTAMP" },
	{ SQL_VARCHAR, "SQL_VARCHAR" },
	{ SQL_LONGVARCHAR, "SQL_LONGVARCHAR" },
	{ SQL_BINARY, "SQL_BINARY" },
	{ SQL_VARBINARY, "SQL_VARBINARY" },
	{ SQL_LONGVARBINARY, "SQL_LONGVARBINARY" },
	{ SQL_BIGINT, "SQL_BIGINT" },
	{ SQL_TINYINT, "SQL_TINYINT" },
	{ SQL_BIT, "SQL_BIT" },
	{ SQL_GRAPHIC, "SQL_GRAPHIC" },
	{ SQL_VARGRAPHIC, "SQL_VARGRAPHIC" },
	{ SQL_LONGVARGRAPHIC, "SQL_LONGVARGRAPHIC" },
	{ SQL_BLOB, "SQL_BLOB" },
	{ SQL_CLOB, "SQL_CLOB" },
	{ SQL_DBCLOB, "SQL_DBCLOB" },
	{ 0, 0 }
};

static char *
mapCode(int code, TypeMap *p)
{
	static char s[10];
	for (; p->text; ++p)
		if (p->code == code)
			return p->text;
	sprintf(s, "%d", code);
	return s;
}

static __inline__ SQLString
getstr(SV *sv, int max)
{
	SQLString r;
	r.ptr = SvPV(sv, r.len);
	if (r.len > max)
		r.len = max;
	return r;
}


static void
dump_var(int sqlType, int prec, void *value)
{
	fprintf(stderr, "\tvalue=%p:", value);
	switch (sqlType) {
	case SQL_INTEGER:
	    fprintf(stderr, "%ld\n", *(long *)value);
	    break;
	case SQL_DOUBLE:
	    fprintf(stderr, "%g\n", *(double *)value);
	    break;
	default:
	  { int x;
	    char *cp = (char *)value;
	    fputc('\'', stderr);
	    for (x = 0; x < prec; ++x)
		if (cp[x] == '\'')
		    fputs("\\'", stderr);
		else if (isprint(cp[x]))
		    fputc(cp[x], stderr);
		else
		    fprintf(stderr, "\\x%02X", (unsigned char)cp[x]);
	    fputs("'\n", stderr);
	  }
	}
}

static void *
bindParams(SQLHSTMT hstmt, SV **args, int nargs, SQLRETURN *retval, int direct)
{
	int i;
	SQLSMALLINT nparams;
	struct Parm {
		union {
			long	ival;
			double	nval;
		}		val;
		SQLINTEGER	len;
	} *a;

	if (direct) {
	    nparams = nargs;
	    *retval = SQLFreeStmt(hstmt, SQL_RESET_PARAMS);
	} else
	    *retval = SQLNumParams(hstmt, &nparams);
	if (*retval != SQL_SUCCESS)
	    return 0;
	if (nparams)
	    a = malloc(nparams * sizeof(struct Parm));
	else
	    a = 0;
	for (i = 0; i < nparams; ++i) {
	    struct Parm *ap = &a[i];
	    SQLSMALLINT cType = SQL_C_CHAR;
	    SQLSMALLINT sqlType = SQL_CHAR;
	    SQLUINTEGER prec = 1;
	    SQLPOINTER  value = "";
	    ap->len = SQL_NULL_DATA;
	    if (i < nargs) {
		SV *sv = args[i];
		if (SvOK(sv)) {
		    if (SvIOKp(sv)) {
			ap->val.ival = SvIV(sv);
			ap->len = sizeof(long);
			cType = SQL_C_LONG;
			sqlType = SQL_INTEGER;
			value = &ap->val.ival;
		    } else if (SvNOKp(sv)) {
			ap->val.nval = SvNV(sv);
			ap->len = sizeof(double);
			cType = SQL_C_DOUBLE;
			sqlType = SQL_DOUBLE;
			value = &ap->val.nval;
		    } else if (SvPOKp(sv)) {
			value = SvPV(sv,prec);
			ap->len = prec;
		    }
		}
	    }
	    *retval = SQLBindParameter(hstmt, 1+i, SQL_PARAM_INPUT, cType, sqlType,
				prec, 0, value, prec, &ap->len);
	    if (db2cli_debug)
		fprintf(stderr, "bind #%d cType=%s sqlType=%s prec=%d len=%d retval=%d\n",
			1+i, mapCode(cType,cTypes), mapCode(sqlType,sqlTypes), prec, ap->len, *retval);
	    if (db2cli_debug >= 2)
		dump_var(sqlType, prec, value);
	    if (*retval < 0) {
		free(a);
		return 0;
	    }
	}
	return a;
}

static void *
bindParams2(SQLHSTMT hstmt, SV **args, int nargs, SQLRETURN *retval, int direct)
{
	dXSARGS;
	int i, k;
	SQLSMALLINT nparams;
	struct Parm {
		union {
			long	ival;
			double	nval;
		}		val;
		SQLINTEGER	len;
	} *a;
		
	if (direct) {
	    nparams = nargs / 3;
	    *retval = SQLFreeStmt(hstmt, SQL_RESET_PARAMS);
	} else
	    *retval = SQLNumParams(hstmt, &nparams);
	if (*retval != SQL_SUCCESS)
	    return 0;
	if (nparams)
	    a = malloc(nparams * sizeof(struct Parm));
	else
	    a = 0;
	for (i = k = 0; i < nparams; ++i, k += 3) {
	    struct Parm *ap = &a[i];
	    SQLSMALLINT cType = (k+0 < nargs) ? SvIV(args[k+0]) : SQL_C_CHAR;
	    SQLSMALLINT sqlType = (k+1 < nargs) ? SvIV(args[k+1]) : SQL_CHAR;
	    SQLUINTEGER prec = 1;
	    SQLPOINTER  value = "";
	    SV *sv = (k+2 < nargs) ? args[k+2] : &sv_undef;
	    ap->len = SQL_NULL_DATA;
	    if (SvOK(sv)) {
		if (SvIOKp(sv)) {
		    ap->val.ival = SvIV(sv);
		    ap->len = sizeof(long);
		    cType = SQL_C_LONG;
		    value = &ap->val.ival;
		} else if (SvNOKp(sv)) {
		    ap->val.nval = SvNV(sv);
		    ap->len = sizeof(double);
		    cType = SQL_C_DOUBLE;
		    value = &ap->val.nval;
		} else if (SvPOKp(sv)) {
		    value = SvPV(sv,prec);
		    ap->len = prec;
		}
	    }
	    *retval = SQLBindParameter(hstmt, 1+i, SQL_PARAM_INPUT, cType, sqlType,
				prec, 0, value, prec, &ap->len);
	    if (db2cli_debug)
		fprintf(stderr, "bind #%d cType=%s sqlType=%s prec=%d len=%d retval=%d\n",
			1+i, mapCode(cType,cTypes), mapCode(sqlType,sqlTypes), prec, ap->len, *retval);
	    if (db2cli_debug >= 2)
		dump_var(sqlType, prec, value);
	    if (*retval < 0) {
		free(a);
		return 0;
	    }
	}
	return a;
}

static int
not_here(s)
char *s;
{
	croak("%s not implemented on this architecture", s);
	return -1;
}

static long
constant(name, arg)
char *name;
int arg;
{
	const struct CliConst *p = in_word_set(name, strlen(name));
	if (p) {
		errno = 0;
		return p->val;
	}
	errno = strncmp(name, "SQL_", 4) ? EINVAL : ENOENT;
	return 0;
}


MODULE = DB2CLI		PACKAGE = DB2CLI

BOOT:	initialize();

double
constant(name,arg)
	char *		name
	int		arg

char *
cTypeName(cType)
	int		cType
    CODE:
	RETVAL = mapCode(cType, cTypes);
    OUTPUT:
	RETVAL

char *
sqlTypeName(sqlType)
	int		sqlType
    CODE:
	RETVAL = mapCode(sqlType, sqlTypes);
    OUTPUT:
	RETVAL

SQLRETURN
SQLAllocConnect(henv,hdbcOut)
	SQLHENV		henv
	SQLHDBC		&hdbcOut = NO_INIT
    OUTPUT:
	hdbcOut

SQLRETURN
SQLAllocEnv(henvOut)
	SQLHENV		&henvOut = NO_INIT
    CODE:
	if ((RETVAL = SQLAllocEnv(&henvOut)) == SQL_SUCCESS)
		SQLSetEnvAttr(henvOut, SQL_ATTR_OUTPUT_NTS, SQL_FALSE, 0);
    OUTPUT:
	RETVAL
	henvOut

SQLRETURN
SQLAllocStmt(hdbc,hstmtOut)
	SQLHDBC		hdbc
	SQLHSTMT	&hstmtOut = NO_INIT
    OUTPUT:
	hstmtOut

SQLRETURN
SQLCancel(hstmt)
	SQLHSTMT	hstmt

SQLRETURN
SQLColAttributes(hstmt,col,descType,textDescOut,numDescOut)
	SQLHSTMT	hstmt
	SQLUSMALLINT	col
	SQLUSMALLINT	descType
	char *		&textDescOut = NO_INIT
	SQLINTEGER	&numDescOut = NO_INIT
    CODE:
	RETVAL = SQLColAttributes(hstmt,col,descType,
		buf1,SmallSize,&len1,
		&numDescOut);
	if (RETVAL >= 0)
		sv_setpvn(ST(3), buf1, len1);
    OUTPUT:
	RETVAL

SQLRETURN
SQLColumnPrivileges(hstmt,catalog,schema,table,column)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
	SQLString	column
    CODE:
	RETVAL = SQLColumnPrivileges(hstmt,
			catalog.ptr,	catalog.len,
			schema.ptr,	schema.len,
			table.ptr,	table.len,
			column.ptr,	column.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLColumns(hstmt,catalog,schema,table,column)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
	SQLString	column
    CODE:
	RETVAL = SQLColumns(hstmt,
			catalog.ptr,	catalog.len,
			schema.ptr,	schema.len,
			table.ptr,	table.len,
			column.ptr,	column.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLConnect(hdbc,dataset,userId=null,authentication=null)
	SQLHDBC		hdbc
	SQLString	dataset
	SQLString	userId
	SQLString	authentication
    CODE:
	RETVAL = SQLConnect(hdbc,
			dataset.ptr,		dataset.len,
			userId.ptr,		userId.len,
			authentication.ptr,	authentication.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLDataSources(henv,direction,datasetOut,descriptionOut)
	SQLHENV		henv
	SQLUSMALLINT	direction
	char *		&datasetOut = NO_INIT
	char *		&descriptionOut = NO_INIT
    CODE:
	RETVAL = SQLDataSources(henv,direction,buf1,SmallSize,&len1,
					       buf2,SmallSize,&len2);
	if (RETVAL >= 0) {
		sv_setpvn(ST(2), buf1, len1);
		sv_setpvn(ST(3), buf2, len2);
	}
    OUTPUT:
	RETVAL

SQLRETURN
SQLDescribeCol(hstmt,col,colNameOut,sqlTypeOut,precOut,scaleOut,nullableOut)
	SQLHSTMT	hstmt
	SQLUSMALLINT	col
	char *		&colNameOut = NO_INIT
	SQLSMALLINT	&sqlTypeOut = NO_INIT
	SQLUINTEGER	&precOut = NO_INIT
	SQLSMALLINT	&scaleOut = NO_INIT
	SQLSMALLINT	&nullableOut = NO_INIT
    CODE:
	RETVAL = SQLDescribeCol(hstmt,col,buf1,SmallSize,&len1,
				&sqlTypeOut,&precOut,&scaleOut,&nullableOut);
	if (RETVAL >= 0)
		sv_setpvn(ST(2), buf1, len1);
    OUTPUT:
	RETVAL
	sqlTypeOut
	precOut
	scaleOut
	nullableOut

SQLRETURN
SQLDisconnect(hdbc)
	SQLHDBC		hdbc

SQLRETURN
SQLDriverConnect(hdbc,hwindow,connStrIn,connStrOut,completion)
	SQLHDBC		hdbc
	long		hwindow
	SQLString	connStrIn
	char *		&connStrOut = NO_INIT
	SQLUSMALLINT	completion
    CODE:
	RETVAL = SQLDriverConnect(hdbc,hwindow,
			connStrIn.ptr,connStrIn.len,
			buf1,SmallSize,&len1,
			completion);
	if (RETVAL >= 0)
		sv_setpvn(ST(3), buf1, len1);
    OUTPUT:
	RETVAL

SQLRETURN
SQLError(henv,hdbc,hstmt,sqlStateOut,nativeErrorOut,errorMsgOut)
	SQLHENV		henv
	SQLHDBC		hdbc
	SQLHSTMT	hstmt
	char *		&sqlStateOut = NO_INIT
	SQLINTEGER	&nativeErrorOut = NO_INIT
	char *		&errorMsgOut = NO_INIT
    CODE:
	RETVAL = SQLError(henv,hdbc,hstmt, buf1, &nativeErrorOut,
			buf2,SmallSize,&len2);
	if (RETVAL >= 0) {
		sv_setpvn(ST(3), buf1, 5);
		sv_setpvn(ST(5), buf2, len2);
	}
    OUTPUT:
	RETVAL
	nativeErrorOut

SQLRETURN
SQLExecDirect(hstmt,sqlStr,...)
	SQLHSTMT	hstmt
	SQLLargeString	sqlStr
    CODE:
    {	void *a = bindParams(hstmt, &ST(2), items-2, &RETVAL, 1);
	if (RETVAL >= 0)
		RETVAL = SQLExecDirect(hstmt, sqlStr.ptr,sqlStr.len);
	if (a)
		free(a);
    }
    OUTPUT:
	RETVAL

SQLRETURN
SQLExecDirect2(hstmt,sqlStr,...)
	SQLHSTMT	hstmt
	SQLLargeString	sqlStr
    CODE:
    {	void *a = bindParams2(hstmt, &ST(2), items-2, &RETVAL, 1);
	if (RETVAL >= 0)
	    RETVAL = SQLExecDirect(hstmt, sqlStr.ptr,sqlStr.len);
	if (a)
	    free(a);
    }
    OUTPUT:
	RETVAL

SQLRETURN
SQLExecute(hstmt,...)
	SQLHSTMT	hstmt
    CODE:
    {	void *a = bindParams(hstmt, &ST(1), items-1, &RETVAL, 0);
	if (RETVAL >= 0)
	    RETVAL = SQLExecute(hstmt);
	if (a)
	    free(a);
    }
    OUTPUT:
	RETVAL

SQLRETURN
SQLExecute2(hstmt,...)
	SQLHSTMT	hstmt
    CODE:
    {	void *a = bindParams2(hstmt, &ST(1), items-1, &RETVAL, 0);
	if (RETVAL >= 0)
	    RETVAL = SQLExecute(hstmt);
	if (a)
	    free(a);
    }
    OUTPUT:
	RETVAL

SQLRETURN
SQLFetch(hstmt,...)
	SQLHSTMT	hstmt
    CODE:
	RETVAL = SQLFetch(hstmt);
	if (RETVAL == SQL_SUCCESS || RETVAL == SQL_SUCCESS_WITH_INFO) {
		int i;
		for (i = 1; i < items; ++i) {
			SQLRETURN rc = SQLGetData(hstmt, i, SQL_C_CHAR,
				buffer.buffer, HugeSize, &buffer.length);
			if (rc < 0) {
				RETVAL = rc;
				break;
			}
			if (RETVAL == SQL_SUCCESS)
				RETVAL = rc;
			if (buffer.length == SQL_NULL_DATA)
				sv_setsv(ST(i), &sv_undef);
			else
				sv_setpvn(ST(i), buffer.buffer, buffer.length);
		}
	}
    OUTPUT:
	RETVAL

SQLRETURN
SQLForeignKeys(hstmt,primCatalog,primSchema,primTable,forgCatalog,forgSchema,forgTable)
	SQLHSTMT	hstmt
	SQLString	primCatalog
	SQLString	primSchema
	SQLString	primTable
	SQLString	forgCatalog
	SQLString	forgSchema
	SQLString	forgTable
    CODE:
	RETVAL = SQLForeignKeys(hstmt,
			primCatalog.ptr,	primCatalog.len,
			primSchema.ptr,		primSchema.len,
			primTable.ptr,		primTable.len,
			forgCatalog.ptr,	forgCatalog.len,
			forgSchema.ptr,		forgSchema.len,
			forgTable.ptr,		forgTable.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLFreeConnect(hdbc)
	SQLHDBC		hdbc

SQLRETURN
SQLFreeEnv(henv)
	SQLHENV		henv

SQLRETURN
SQLFreeStmt(hstmt,option=SQL_DROP)
	SQLHSTMT	hstmt
	SQLUSMALLINT	option

SQLRETURN
SQLGetConnectOptionLong(hdbc,option,param)
	SQLHDBC		hdbc
	SQLUSMALLINT	option
	SQLUINTEGER	&param = NO_INIT
    CODE:
	RETVAL = SQLGetConnectOption(hdbc,option, buffer.buffer);
	if (RETVAL >= 0)
		sv_setiv(ST(2), *(SQLUINTEGER *)&buffer.buffer);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetConnectOptionStr(hdbc,option,param)
	SQLHDBC		hdbc
	SQLUSMALLINT	option
	char *		&param = NO_INIT
    CODE:
	RETVAL = SQLGetConnectOption(hdbc,option, buffer.buffer);
	if (RETVAL >= 0)
		sv_setpv(ST(2), buffer.buffer);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetCursorName(hstmt,cursorOut)
	SQLHSTMT	hstmt
	char *		&cursorOut = NO_INIT
    CODE:
	RETVAL = SQLGetCursorName(hstmt, buf1,SmallSize,&len1);
	if (RETVAL >= 0)
		sv_setpvn(ST(1), buf1, len1);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetData(hstmt,col,cType,valueOut)
	SQLHSTMT	hstmt
	SQLUSMALLINT	col
	SQLSMALLINT	cType
	char *		&valueOut = NO_INIT
    CODE:
	buffer.length = SQL_NULL_DATA;
	RETVAL = SQLGetData(hstmt, col, cType, buffer.buffer,HugeSize,&buffer.length);
	if (RETVAL >= 0) {
		if (buffer.length == SQL_NULL_DATA)
			sv_setsv(ST(3), &sv_undef);
		else switch (cType) {
		case SQL_C_SHORT:
			sv_setiv(ST(3), *(SQLSMALLINT *)buffer.buffer);
			break;
		case SQL_C_LONG:
			sv_setiv(ST(3), *(SQLINTEGER *)buffer.buffer);
			break;
		case SQL_C_FLOAT:
			sv_setnv(ST(3), *(SQLREAL *)buffer.buffer);
			break;
		case SQL_C_DOUBLE:
			sv_setnv(ST(3), *(SQLDOUBLE *)buffer.buffer);
			break;
		default:
			sv_setpvn(ST(3), buffer.buffer, buffer.length);
			break;
		}
	}
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetEnvAttrLong(henv,attribute,value)
	SQLHENV		henv
	SQLINTEGER	attribute
	SQLUINTEGER	&value = NO_INIT
    CODE:
	RETVAL = SQLGetEnvAttr(henv,attribute, &value,0,0);
    OUTPUT:
	RETVAL
	value

SQLRETURN
SQLGetEnvAttrStr(henv,attribute,value)
	SQLHENV		henv
	SQLINTEGER	attribute
	char *		&value = NO_INIT
    CODE:
	buffer.length = 0;
	RETVAL = SQLGetEnvAttr(henv,attribute, buffer.buffer,HugeSize,&buffer.length);
	if (RETVAL >= 0)
		sv_setpvn(ST(2), buffer.buffer, buffer.length);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetFunctions(hdbc,function,existsOut)
	SQLHDBC		hdbc
	SQLUSMALLINT	function
	SQLUSMALLINT	&existsOut = NO_INIT
    OUTPUT:
	existsOut

SQLRETURN
SQLGetInfoShort(hdbc,infoType,infoOut)
	SQLHDBC		hdbc
	SQLUSMALLINT	infoType
	SQLUSMALLINT	&infoOut = 0;
    CODE:
	RETVAL = SQLGetInfo(hdbc, infoType, &infoOut,sizeof infoOut,&len1);
    OUTPUT:
	RETVAL
	infoOut

SQLRETURN
SQLGetInfoLong(hdbc,infoType,infoOut)
	SQLHDBC		hdbc
	SQLUSMALLINT	infoType
	SQLUINTEGER	&infoOut = 0;
    CODE:
	RETVAL = SQLGetInfo(hdbc, infoType, &infoOut,sizeof infoOut,&len1);
    OUTPUT:
	RETVAL
	infoOut

SQLRETURN
SQLGetInfoStr(hdbc,infoType,infoOut)
	SQLHDBC		hdbc
	SQLUSMALLINT	infoType
	char *		&infoOut = NO_INIT
    CODE:
	RETVAL = SQLGetInfo(hdbc, infoType, buf1,SmallSize,&len1);
	if (RETVAL >= 0)
		sv_setpvn(ST(2), buf1, len1);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetLength(hstmt,locType,locator,lengthOut,indicatorOut)
	SQLHSTMT	hstmt
	SQLSMALLINT	locType
	SQLINTEGER	locator
	SQLINTEGER	&lengthOut = NO_INIT
	SQLINTEGER	&indicatorOut = NO_INIT
    OUTPUT:
	lengthOut
	indicatorOut

SQLRETURN
SQLGetPosition(hstmt,locType,locator,searchLoc,searchLit,position,locatedAtOut,indicatorOut)
	SQLHSTMT	hstmt
	SQLSMALLINT	locType
	SQLINTEGER	locator
	SQLINTEGER	searchLoc
	SQLLargeString	searchLit
	SQLINTEGER	position
	SQLINTEGER	&locatedAtOut = NO_INIT
	SQLINTEGER	&indicatorOut = NO_INIT
    CODE:
	RETVAL = SQLGetPosition(hstmt,locType,locator,searchLoc,
			searchLit.ptr,searchLit.len,
			position,&locatedAtOut,&indicatorOut);
    OUTPUT:
	RETVAL
	locatedAtOut
	indicatorOut

SQLRETURN
SQLGetSQLCA(henv,hdbc,hstmt,sqlcaOut)
	SQLHENV		henv
	SQLHDBC		hdbc
	SQLHSTMT	hstmt
	char *		&sqlcaOut = NO_INIT
    CODE:
    {	struct sqlca ca;
	RETVAL = SQLGetSQLCA(henv,hdbc,hstmt, &ca);
	if (RETVAL >= 0)
		sv_setpvn(ST(3), (char *)&ca, sizeof ca);
    }
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetStmtOptionLong(hstmt,option,param)
	SQLHSTMT	hstmt
	SQLUSMALLINT	option
	SQLUINTEGER	&param = NO_INIT
    CODE:
	RETVAL = SQLGetStmtOption(hstmt, option, buffer.buffer);
	if (RETVAL >= 0)
		sv_setiv(ST(2), *(SQLUINTEGER *)buffer.buffer);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetStmtOptionStr(hstmt,option,param)
	SQLHSTMT	hstmt
	SQLUSMALLINT	option
	char *		param = NO_INIT
    CODE:
	RETVAL = SQLGetStmtOption(hstmt, option, buffer.buffer);
	if (RETVAL >= 0)
		sv_setpv(ST(2), buffer.buffer);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetSubString(hstmt,locType,locator,position,length,targetCType,valueOut,indicatorOut)
	SQLHSTMT	hstmt
	SQLSMALLINT	locType
	SQLINTEGER	locator
	SQLUINTEGER	position
	SQLUINTEGER	length
	SQLSMALLINT	targetCType
	char *		&valueOut = NO_INIT
	SQLINTEGER	&indicatorOut = NO_INIT
    CODE:
	RETVAL = SQLGetSubString(hstmt,locType,locator,position,length,targetCType,
				buffer.buffer,HugeSize,&buffer.length,&indicatorOut);
	if (RETVAL >= 0)
		sv_setpvn(ST(6), buffer.buffer, buffer.length);
    OUTPUT:
	RETVAL

SQLRETURN
SQLGetTypeInfo(hstmt,sqlType)
	SQLHSTMT	hstmt
	SQLSMALLINT	sqlType

SQLRETURN
SQLMoreResults(hstmt)
	SQLHSTMT	hstmt

SQLRETURN
SQLNativeSql(hdbc,stringIn,stringOut)
	SQLHDBC		hdbc
	SQLLargeString	stringIn
	char *		&stringOut = NO_INIT
    CODE:
	RETVAL = SQLNativeSql(hdbc, stringIn.ptr,stringIn.len,
				buffer.buffer,HugeSize,&buffer.length);
	if (RETVAL >= 0)
		sv_setpvn(ST(2), buffer.buffer, buffer.length);
    OUTPUT:
	RETVAL

SQLRETURN
SQLNumParams(hstmt,paramOut)
	SQLHSTMT	hstmt
	SQLSMALLINT	&paramOut = NO_INIT
    OUTPUT:
	paramOut

SQLRETURN
SQLNumResultCols(hstmt,paramOut)
	SQLHSTMT	hstmt
	SQLSMALLINT	&paramOut = NO_INIT
    OUTPUT:
	paramOut

SQLRETURN
SQLPrepare(hstmt,sqlString)
	SQLHSTMT	hstmt
	SQLLargeString	sqlString
    CODE:
	RETVAL = SQLPrepare(hstmt, sqlString.ptr,sqlString.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLPrimaryKeys(hstmt,catalog,schema,table)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
    CODE:
	RETVAL = SQLPrimaryKeys(hstmt,
			catalog.ptr,	catalog.len,
			schema.ptr,	schema.len,
			table.ptr,	table.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLProcedureColumns(hstmt,catalog,schema,procedure,column)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	procedure
	SQLString	column
    CODE:
	RETVAL = SQLProcedureColumns(hstmt,
			catalog.ptr,  	catalog.len,
			schema.ptr,	schema.len,
			procedure.ptr,	procedure.len,
			column.ptr,	column.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLProcedures(hstmt,catalog,schema,procedure)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	procedure
    CODE:
	RETVAL = SQLProcedures(hstmt,
			catalog.ptr,  	catalog.len,
			schema.ptr,	schema.len,
			procedure.ptr,	procedure.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLRowCount(hstmt,rowsOut)
	SQLHSTMT	hstmt
	SQLINTEGER	&rowsOut = NO_INIT

SQLRETURN
SQLSetColAttributes(hstmt,col,name,sqlType,prec,scale,nullable)
	SQLHSTMT	hstmt
	SQLUSMALLINT	col
	SQLString	name
	SQLSMALLINT	sqlType
	SQLUINTEGER	prec
	SQLSMALLINT	scale
	SQLSMALLINT	nullable
    CODE:
	RETVAL = SQLSetColAttributes(hstmt,col, name.ptr,name.len,
				sqlType,prec,scale,nullable);
    OUTPUT:
	RETVAL

SQLRETURN
SQLSetConnection(hdbc)
	SQLHDBC		hdbc

SQLRETURN
SQLSetConnectOption(hdbc,option,param)
	SQLHDBC		hdbc
	SQLUSMALLINT	option
	SV *		param
    CODE:
	if (SvPOKp(param))
		RETVAL = SQLSetConnectOption(hdbc, option, (SQLUINTEGER)SvPV(param,na));
	else
		RETVAL = SQLSetConnectOption(hdbc, option, SvIV(param));
    OUTPUT:
	RETVAL

SQLRETURN
SQLSetCursorName(hstmt,cursor)
	SQLHSTMT	hstmt
	SQLString	cursor
    CODE:
	RETVAL = SQLSetCursorName(hstmt, cursor.ptr,cursor.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLSetEnvAttr(henv,attribute,value)
	SQLHENV		henv
	SQLINTEGER	attribute
	SV *		value
    CODE:
	if (SvPOKp(value)) {
		STRLEN len;
		char *ptr = SvPV(value,len);
		RETVAL = SQLSetEnvAttr(henv, attribute, ptr,len);
	} else
		RETVAL = SQLSetEnvAttr(henv, attribute, (SQLPOINTER)SvIV(value),0);
    OUTPUT:
	RETVAL

SQLRETURN
SQLSetStmtOption(hstmt,option,param)
	SQLHSTMT	hstmt
	SQLUSMALLINT	option
	SV *		param
    CODE:
	if (SvPOKp(param))
		RETVAL = SQLSetStmtOption(hstmt, option, (SQLUINTEGER)SvPV(param,na));
	else
		RETVAL = SQLSetStmtOption(hstmt, option, SvIV(param));
    OUTPUT:
	RETVAL

SQLRETURN
SQLSpecialColumns(hstmt,colType,catalog,schema,table,scope,nullable)
	SQLHSTMT	hstmt
	SQLUSMALLINT	colType
	SQLString	catalog
	SQLString	schema
	SQLString	table
	SQLUSMALLINT	scope
	SQLUSMALLINT	nullable
    CODE:
	RETVAL = SQLSpecialColumns(hstmt,colType,
				catalog.ptr,	catalog.len,
				schema.ptr,	schema.len,
				table.ptr,	table.len,
				scope, nullable);
    OUTPUT:
	RETVAL

SQLRETURN
SQLStatistics(hstmt,catalog,schema,table,unique,accuracy)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
	SQLUSMALLINT	unique
	SQLUSMALLINT	accuracy
    CODE:
	RETVAL = SQLStatistics(hstmt,
				catalog.ptr,	catalog.len,
				schema.ptr,	schema.len,
				table.ptr,	table.len,
				unique,accuracy);
    OUTPUT:
	RETVAL

SQLRETURN
SQLTablePrivileges(hstmt,catalog,schema,table)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
    CODE:
	RETVAL = SQLTablePrivileges(hstmt,
				catalog.ptr,	catalog.len,
				schema.ptr,	schema.len,
				table.ptr,	table.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLTables(hstmt,catalog,schema,table,type)
	SQLHSTMT	hstmt
	SQLString	catalog
	SQLString	schema
	SQLString	table
	SQLString	type
    CODE:
	RETVAL = SQLTables(hstmt,catalog.ptr,	catalog.len,
				 schema.ptr,	schema.len,
				 table.ptr,	table.len,
				 type.ptr,	type.len);
    OUTPUT:
	RETVAL

SQLRETURN
SQLTransact(henv,hdbc,type=SQL_COMMIT)
	SQLHENV		henv
	SQLHDBC		hdbc
	SQLUSMALLINT	type

