module Turing;

/*
	A Turing machine interpreter and programming environment.

	Run the program using the "No Console" option (Application options).

	To generate an application for this program the memory of the Clean
	0.8 application should be set to at least 2000K.
	The linker needs an additional 900K of free memory inside or outside
	the Clean 0.8 application.
	To launch the generated application 700K of free memory is needed
	outside the Clean 0.8 application.
*/

import	StdClass;
import StdInt, StdString, StdBool, StdFile, deltaSystem;
import deltaFileSelect, deltaEventIO, deltaIOSystem, deltaMenu, deltaWindow, deltaTimer;

import tm, showtm, tmdialog, tmfile, Help;


     
	
	WdCorner	:== (0, 0);
	TWdCorner	:== (100, 210);
	WindowMin	:== (50, 50);
	WindowInit	:== (500, 265);
	TapeWdMin	:== (50, 50);
	TapeWdInit	:== (400, 60);
	WindowPSize	:== ((0,0), (MaxX, 265));
	TapeWdPSize	:== ((0,0), (MaxX, 92));
	
	Horbar :== ScrollBar (Thumb 0) (Scroll 24);
	Verbar :== ScrollBar (Thumb 0) (Scroll 8);

	Speed1	:== TicksPerSecond / 3;
	Speed2	:== TicksPerSecond / 6;
	Speed3	:== TicksPerSecond / 12;
	Speed4	:== TicksPerSecond / 20;
	Speed5	:== 0;


    

Start	:: * World -> * World;
Start world =  CloseEvents events` (closefiles files`` world``);
		 where {
		 (ts,nm,dl,files``,sd)=: tm;
	     (tm,events`)         =: StartIO [about, menu, window, timer] (NewTuring files`) [] events;
		 (files,world``)      =: openfiles world`;
		 (events,world`)      =: OpenEvents world;

		 about	=: DialogSystem [about_dialog];
		 (about_dialog,files`)=: MakeAboutDialog "Turing" HelpFile files Help;

		 menu	=: MenuSystem [file, machine];
		 file	=: PullDownMenu FileMenuId "File" Able
					[ MenuItem NewItemId  "New"        (Key 'N') Able   DoNew,
					  MenuItem OpenItemId "Open..."    (Key 'O') Able   DoOpen,
					  MenuItem SaveItemId "Save"       (Key 'S') Unable DoSave,
					  MenuItem SvAsItemId "Save As..." NoKey     Able   DoSaveAs,
		 	          MenuSeparator,
					  MenuItem HelpItemId "Help..."    (Key 'H') Able   Help,
		 	          MenuSeparator,
		 	          MenuItem QuitItemId "Quit"       (Key 'Q') Able   DoQuit
		 	        ];
		 machine=: PullDownMenu MachineMenuId "Machine" Able
		 	        [ MenuItem StepItemId  "Step"      (Key 'T') Able   DoStep,
		 	          MenuItem RunItemId   "Run"       (Key 'R') Able   DoRun,
		 	          MenuItem HaltItemId  "Continue"  (Key '.') Unable DoContinue,
		 	          MenuSeparator,
		 	          SubMenuItem DelayItemId "Speed" Able [delay]
		 	        ];
		 delay=:   MenuRadioItems NormId
		            [ MenuRadioItem VerSId "Very Slow" (Key '1') Able (SetDelay Speed1),
		              MenuRadioItem SlowId "Slow"      (Key '2') Able (SetDelay Speed2),
		              MenuRadioItem NormId "Normal"    (Key '3') Able (SetDelay Speed3),
		 	          MenuRadioItem FastId "Fast"      (Key '4') Able (SetDelay Speed4),
		              MenuRadioItem VerFId "Very Fast" (Key '5') Able (SetDelay Speed5)
		 	        ];

		 window	=: WindowSystem [trswd, tapewd];
		 trswd	=: ScrollWindow WindowID WdCorner "Turing Machine" Horbar Verbar
							   WindowPSize WindowMin WindowInit UpdateWindow
							   [Mouse Able EditTransitions, GoAway DoQuit];
		 tapewd	=: ScrollWindow TapeWdID TWdCorner "Tape" Horbar Verbar
							   TapeWdPSize TapeWdMin TapeWdInit UpdateTapeWd
							   [Mouse Able EditTape, GoAway DoQuit];
								      
		 timer	=: TimerSystem [Timer TimerID Unable Speed3 TimerStep];
		 };

NewTuring	:: Files -> Tm;
NewTuring files =  ((([],("",0),""),0,None),"",Speed3,files,True);

/*	Open a new empty Turing machine.
*/

DoNew	:: Tm IOTm -> TmIOState;
DoNew tmst=:(tm, nm, del, dsk, saved) io
	| saved = 	MakeNewTuring del dsk io;
// RWS	| sure = 	MakeNewTuring del dsk io1;
	| sure = 	MakeNewTuring del` dsk` io1;
	= 	(tmst1,io1);
		where {
// RWS
		(_, _, del`, dsk`, _)=: tmst1;
		(sure,tmst1,io1)=: SaveBeforeClose "opening a new Turing machine" tmst io;
		};

MakeNewTuring	:: Int Disk IOTm -> TmIOState;
MakeNewTuring del dsk io
	= 	(((([], ("",0), ""), 0, None), "", del, dsk, True), newio);
		where {
		newio=: ChangeIOState [DrawInWindow TapeWdID [ShowTape ("",0)],
		   					  DrawInWindow WindowID [ShowTransitions [] ""],
		   					  ChangeWindowTitle WindowID "Turing Machine",
		   					  DisableMenuItems [SaveItemId, HaltItemId] ] io;
		};

/*	Save the Turing machine.
*/

DoSave	:: Tm IOTm -> TmIOState;
DoSave ((turing,trn,com),name,del,disk,saved) io
		| success =  (((turing,trn,com),name,del,newd,True),io`);
		=  (tm`,alert);
		   where {
		   (success, newd)=: WriteTuringToFile turing name disk;
		   io`  =: DisableMenuItems [SaveItemId] io;
		   tm`  =: ((turing,trn,com),name,del,newd,saved);
		   alert=: Alert "The Turing machine has not been saved."
		                "The file could not be opened." io;
		   };

DoSaveAs	:: Tm IOTm -> TmIOState;
DoSaveAs tm=:(tmst=:(turing=:(trs,tape,state),trn,com),name,del,disk,saved) io
		| not result =  ((tmst, name,del,disk`,saved), io`);
		|  RemovePath fname  == HelpFile =  ((tmst, name,del,disk`,saved), alert1);
		| success =  ((tmst,fname,del,newd ,True ), io```);
		=  ((tmst, name,del,newd ,saved), alert2);
		   where {
		   (result,fname,disk`,io`)=: SelectOutputFile "Save T.M. As:" (RemovePath name) disk io;
		   (success, newd)	       =: WriteTuringToFile turing fname disk`;
		   io``                    =: ChangeWindowTitle WindowID (RemovePath fname) io`;
		   io```                   =: DisableMenuItems [SaveItemId] io``;
		   alert1                  =: Alert "The Turing machine cannot be saved to"
		   							       (("the help file \'" +++ HelpFile) +++ "\'.") io`;
		   alert2                  =: Alert "The Turing machine has not been saved."
		   					               "The file could not be opened." io`;
		   };


/*	Load a Turing machine from a file.
*/

DoOpen	:: Tm IOTm -> TmIOState;
DoOpen tm=:(tms, nam, delay, disk, saved) io
		| saved =  EvtOpenTuring tm io;
		| sure =  EvtOpenTuring tm` io`;
		=  (tm`,io`);
		where {
		(sure,tm`,io`)=: SaveBeforeClose "opening an other Turing machine" tm io;
		};

EvtOpenTuring	:: Tm IOTm -> TmIOState;
EvtOpenTuring (tms, nam, delay, disk, saved) io
		| result =  OpenTuring filename (tms,nam,delay,disk`,saved) io`;
		=  ((tms,nam,delay,disk`,saved), io`);
		where {
		(result,filename,disk`,io`)=: SelectInputFile disk io;
		};

OpenTuring	:: String Tm IOTm -> TmIOState;
OpenTuring name tm=:(tms, nam, delay, disk, saved) io
		| fname == HelpFile =  (tm,alert1);
		| status == 0 =  (newtm,update);
		| status > 0 =  (tm`,alert2);
		| status == (-1) =  (tm`,alert3);
		=  (tm`,alert4)  ;
		   where {
		   newtm=: ((newturing,0,None),name,delay,newd,True);
		   tm`  =: (tms, nam, delay, newd, saved);
		   (status, newturing, newd)=: ReadTuring name disk;
		   update=: ChangeIOState [EnableMenuItems [RunItemId,StepItemId,HaltItemId],
		   						  DrawInWindow TapeWdID [ShowTape tape],
		   						  DrawInWindow WindowID [ShowTransitions trs state],
		   						  ChangeWindowTitle WindowID (RemovePath name),
		   						  DisableMenuItems [SaveItemId] ] io;
		   (trs,tape,state)=: newturing;
		   alert1          =: Alert ("The help file" +++ fstring)
								   "cannot be opened as a T.M." io;
		   alert2          =: Alert ("Parse error in line " +++  toString status )
		                           ("of file" +++  fstring +++ "." ) io;
		   alert3          =: Alert "Unexpected end of file" (fstring +++ ".") io;
		   alert4          =: Alert ("The file" +++ fstring) "could not be opened." io;
		   fstring         =: " \'" +++  fname +++ "\'" ;
		   fname           =: RemovePath name;
		   };

/*	The Help command.
*/

Help	:: Tm IOTm -> TmIOState;
Help (tms, nam, delay, disk, saved) io =  ((tms, nam, delay, disk`, saved), io`);
		where {
		(disk`,io`)=: ShowHelp HelpFile disk io;
		};


/*	Let the Turing machine do one step (transition).
*/

DoStep	:: Tm IOTm -> TmIOState;
DoStep tmst=: (tm=: ((trs, (cont, pos), state), trn, com), nam, del, dsk, svd) iostate
		| state == "halt" || state == "error" =  (tmst, iostate);
		=  ((newtm, nam, del, dsk, svd), newio);
		   where {
		   newtm=: Step tm;
		   newio=: ChangeIOState [DrawInWindow WindowID [ShowTransition trn newtrn,
		   												ShowNextState newstate],
		   						 DrawInWindow TapeWdID [ShowNewTape newcom pos],
		   						 StepChangeMenus newstate] iostate;
		   (newtrn, newstate, newcom)=: GetChanges newtm;
		   };

StepChangeMenus	:: String IOTm -> IOTm;
StepChangeMenus state iostate
		| state <> "halt" && state <> "error" =  iostate;
		=  DisableMenuItems [StepItemId,HaltItemId] iostate;

GetChanges	:: TmState -> (Int, String, Comm);
GetChanges ((trs,tape,state),trn,command) =  (trn,state,command);


/*	Let the T.M. run until the haltstate is reached.
*/

DoRun	:: Tm IOTm -> TmIOState;
DoRun (((ts,tp,state),tn,cm),nm,dl,dk,sd) io
	= 	(tm, ChangeIOState [DisableMouse			TapeWdID,
							DisableMouse			WindowID,
							EnableMenuItems 		[HaltItemId],
							ChangeMenuItemTitles	[(HaltItemId,"Halt")],
							ChangeMenuItemFunctions	[(HaltItemId,DoHalt)],
							DisableMenuItems		[StepItemId, RunItemId],
							DisableMenus    		[FileMenuId],
							DrawInWindow    		TapeWdID [EraseError],
							EnableTimer     		TimerID] io);
		where {
		tm=: (((ts,tp,"S"),tn,cm),nm,dl,dk,sd);
		};
 

/*	Halt a running T.M.
*/
	
DoHalt	:: Tm IOTm -> TmIOState;
DoHalt tm io
	= 	(tm, ChangeIOState [EnableMouse				TapeWdID,
							EnableMouse				WindowID,
							ChangeMenuItemTitles	[(HaltItemId,"Continue")],
							ChangeMenuItemFunctions	[(HaltItemId,DoContinue)],
							EnableMenuItems			[StepItemId, RunItemId],
							EnableMenus				[FileMenuId],
							DisableTimer			TimerID] io);

DoContinue	:: Tm IOTm -> TmIOState;
DoContinue tm io
	= 	(tm, ChangeIOState [DisableMouse			TapeWdID,
							DisableMouse			WindowID,
							ChangeMenuItemTitles	[(HaltItemId,"Halt")],
							ChangeMenuItemFunctions	[(HaltItemId,DoHalt)],
							DisableMenuItems		[StepItemId, RunItemId],
							DisableMenus 			[FileMenuId],
							EnableTimer				TimerID] io);

/*	Set the speed (delay) of a (possibly running) T.M.
*/

SetDelay	:: Int Tm IOTm -> TmIOState;
SetDelay delay (tm,nm,dl,dk,sd) io =  ((tm,nm,delay,dk,sd), SetTimerInterval TimerID delay io);


/*	Quit the program.
*/

DoQuit	:: Tm IOTm -> TmIOState;
DoQuit tm=:(tms, nam, delay, disk, saved) io
	| saved = 	(tm , QuitIO io );
	| sure = 	(tm`, QuitIO io`);
	= 	(tm`, io`);
		where {
		(sure,tm`,io`)=: SaveBeforeClose "quitting" tm io;
		};


/*	When a mouseclick occurs the T.M. can be edited.
*/

EditTransitions	:: MouseState Tm IOTm -> TmIOState;
EditTransitions (mpos,ButtonDown,com) tm=:(((ts,tp,st),tn,cm),nm,dl,dk,sd) io
	| ontrans = 	AlterTransition transnr tm io;
	| onstate = 	AlterState tm io;
	= 	(tm, io);
		where {
		transnr	 =: if (nr > lasttrans) lasttrans nr;
		lasttrans=: NrOfTransitions ts;
		(nr, ontrans, onstate)=: ClickedInWindow mpos;
		};
EditTransitions (mpos, button, mod_new) tm io =  (tm, io);

EditTape	:: MouseState Tm IOTm -> TmIOState;
EditTape (mpos,ButtonDown,CommandOnly) tm=:(((ts,tp=:(ct,oldpos),st),tn,cm),nm,dl,dk,sd) io
	| ontape = 	((((ts, (cont,pos), st), tn, cm), nm, dl, dk, sd), io`);
	= 	(tm, io);
		where {
		io`			   =: DrawInWindow TapeWdID [ShowHeadMove cont oldpos newpos left right] io1;
		(cont,pos)	   =: MoveHead newpos tp;
		(newpos,ontape)=: ClickedInTapeWd mpos;
		(left,top)     =: tleft;
		(right,bottom) =: bright;
		(tleft,bright) =: frame;
		(frame,io1)    =: WindowGetFrame TapeWdID io;
		};
EditTape (mpos, ButtonDown, com) tm=:(((ts,(ct,ps),st),tn,cm),nm,dl,dk,sd) io
	| ontape = 	AlterCell tapepos tm io;
	= 	(tm, io);
		where {
		tapepos		=: if (nr > lastcell) lastcell nr;
		lastcell	=: NrOfCells ct;
		(nr, ontape)=: ClickedInTapeWd mpos;
		};
EditTape (mpos, button, mod_new) tm io =  (tm, io);


/*	The window update and activate functions.
*/

UpdateWindow	:: UpdateArea Tm -> (Tm, [DrawFunction]);
UpdateWindow update_area tm=:(((trs,tp,state),trn,cm),name,dl,dk,sd)
	= 	(tm, [SetTuringFont,
	          ShowTransitions trs state,
	          ShowTransition trn trn]);

UpdateTapeWd	:: UpdateArea Tm -> (Tm, [DrawFunction]);
UpdateTapeWd [((from_new,t),(to,b)):areas] tm=:(((trs,tape,state),tn,cm),nm,dl,dk,sd)
	= 	(tm`, [SetTuringFont, ShowTapePart tape from_new to : rest]);
		where {
		(tm`,rest)=: UpdateTapeWd areas tm;
		};
UpdateTapeWd areas tm =  (tm, []);


/*	The step function for the Timer-device (used by the Run-command).
*/

TimerStep	:: TimerState Tm IOTm -> TmIOState;
TimerStep times tm=:(((trs, tape, state), trn, com), nam, del, dsk,svd) io
	| state <> "halt" && state <> "error"	= 	DoStep tm io;
	= 	(tm, ChangeIOState [DisableTimer        	TimerID,
		                    EnableMouse				TapeWdID,
							EnableMouse				WindowID,
							ChangeMenuItemTitles	[(HaltItemId,"Continue")],
							ChangeMenuItemFunctions	[(HaltItemId,DoContinue)],
							DisableMenuItems		[HaltItemId],
							EnableMenuItems			[RunItemId],
							EnableMenus				[FileMenuId] ] io);
