/*
** Program: WAVUTIL.CMD
**
** Author: Rafal Szymczak
**
** Date: January 22, 2003
**
** Version: 1.00.000
**
** Last revised:
**
** Purpose: To perform simple WAV file manipulations using RxWav library.
*/

Parse Arg args

infile = ''
Call ParseArgs args

If function = '' | function = '?' | function = '/?' | function = '-?' | function = 'HELP' | function = '/HELP' | function = '-HELP' | infile = '' Then
Do
   Call Help
   Exit 1
End

newfile = ''
Call ParseParams

rc = RxFuncAdd('WavLoadFuncs', 'RxWav', 'WavLoadFuncs')
Call WavLoadFuncs

Parse Value WavQueryFile(infile) With channels samples bits chanlen

If channels = 'NOWAV' | channels > 2 | samples = '' | bits = '' | chanlen = '' Then
Do
   Say
   Say 'Unable to open input file or wrong file type ('infile').'
   Say
   Call WavDropFuncs
   Exit 2
End

If newfile <> '' Then
Do
   Parse Value WavQueryFile(newfile) With nfchannels nfsamples nfbits nfchanlen

   If nfchannels = 'NOWAV' | nfchannels > 2 | nfsamples = '' | nfbits = '' | nfchanlen = '' Then
   Do
      Say
      Say 'Unable to open second input file or wrong file type ('newfile').'
      Say
      Call WavDropFuncs
      Exit 2
   End
End
Else
   nfchanlen = 0

If channels = 1 Then
   type = 'Mono'
Else
   type = 'Stereo'

playlen = chanlen / samples
If POS('.', playlen) > 0 Then
Do
   frac = SUBSTR(playlen, POS('.', playlen) + 1)
   If LENGTH(frac) > 3 Then
      If SUBSTR(frac, 4, 1) > 4 Then
         playlen = playlen + 0.001
End
mins = TRUNC(playlen / 60)
secs = TRUNC(playlen - mins * 60)
msecs = TRUNC(playlen, 3) - TRUNC(playlen)
msecs = SUBSTR(msecs, POS('.', msecs) + 1)

Say
Say 'Input file info:'
Say type 'file,' samples / 1000 'kHz,' bits 'bits,' mins 'minutes,' secs 'seconds,' msecs 'miliseconds.'
Say

If outfile = '' Then
   outfile = infile

If channels = 1 Then
   Call MonoProc
Else
   Call StereoProc

Call WavDropFuncs

Exit

/************************************************************/

MonoProc:

stereo = 0

lchanmem = WavAllocTrac(chanlen + nfchanlen + 2 * samples)
If function <> 'PREPEND' Then
   lchanbeg = lchanmem + samples
Else
   lchanbeg = lchanmem + samples + nfchanlen

rc = WavReadMono(infile, lchanbeg)

Select
   When function = 'INFO' Then
   Do
      Call Info
   End
   When function = 'MONO2STEREO' Then
   Do
      Call Mono2Stereo
   End
   When function = 'FADEIN' Then
   Do
      Call FadeIn starttime endtime
   End
   When function = 'FADEOUT' Then
   Do
      Call FadeOut starttime endtime
   End
   When function = 'TRUNCATE' Then
   Do
      Call Truncate starttime endtime
   End
   When function = 'APPEND' Then
   Do
      Call Append newfile
   End
   When function = 'PREPEND' Then
   Do
      Call Prepend newfile
   End
   Otherwise
   Do
      Say
      Say 'Unknown function specified.'
      Say
      Call Help
      Call CleanupAndExit 2
   End
End

rc = WavWriteMono(outfile, lchanbeg, chanlen)

rc = WavDropTrac(lchanmem)

Return

/****************************************/

StereoProc:

stereo = 1

lchanmem = WavAllocTrac(chanlen + nfchanlen + 2 * samples)
rchanmem = WavAllocTrac(chanlen + nfchanlen + 2 * samples)
If function <> 'PREPEND' Then
Do
   lchanbeg = lchanmem + samples
   rchanbeg = rchanmem + samples
End
Else
Do
   lchanbeg = lchanmem + samples + nfchanlen
   rchanbeg = rchanmem + samples + nfchanlen
End

rc = WavReadStereo(infile, lchanbeg, rchanbeg)

Select
   When function = 'INFO' Then
   Do
      Call Info
   End
   When function = 'FADEIN' Then
   Do
      Call FadeIn starttime endtime
   End
   When function = 'FADEOUT' Then
   Do
      Call FadeOut starttime endtime
   End
   When function = 'TRUNCATE' Then
   Do
      Call Truncate starttime endtime
   End
   When function = 'APPEND' Then
   Do
      Call Append newfile
   End
   When function = 'PREPEND' Then
   Do
      Call Prepend newfile
   End
   Otherwise
   Do
      Say
      Say 'Unknown function specified.'
      Say
      Call Help
      Call CleanupAndExit 2
   End
End

rc = WavWriteStereo(outfile, lchanbeg, rchanbeg, chanlen)

If rc = 0 Then
Do
   Say
   Say 'Success.'
   Say
End

rc = WavDropTrac(lchanmem)
rc = WavDropTrac(rchanmem)

Return

/****************************************/

ParseArgs: Procedure Expose infile outfile function params

Arg args

args = STRIP(args)
function = TRANSLATE(WORD(args, 1))
If function <> '' Then
   args = STRIP(SUBSTR(args, LENGTH(function) + 1))

If args = '' Then
   Return

If SUBSTR(args, 1, 1) = '"' Then
Do
   endpos = POS('"', args, 2)
   If endpos > 0 Then
   Do
      infile = SUBSTR(args, 1, endpos)
   End
   Else
   Do
      Say
      Say 'Invalid input file name.'
      Say
      Call Help
      Exit 2
   End
End
Else
   infile = WORD(args, 1)

args = STRIP(SUBSTR(args, LENGTH(infile) + 1))

If SUBSTR(args, 1, 1) = '"' Then
Do
   endpos = POS('"', args, 2)
   If endpos > 0 Then
   Do
      outfile = SUBSTR(args, 1, endpos)
   End
   Else
   Do
      Say
      Say 'Invalid output file name.'
      Say
      Call Help
      Exit 2
   End
End
Else
Do
   If SUBSTR(args, 1, 6) <> 'START=' & SUBSTR(args, 1, 4) <> 'END=' & SUBSTR(args, 1, 5) <> 'FILE=' Then
      outfile = WORD(args, 1)
   Else
      outfile = ''
End

params = STRIP(SUBSTR(args, LENGTH(outfile) + 1))

Return

/****************************************/

ParseParams:

Select
   When function = 'INFO' | function = 'MONO2STEREO' Then
      Nop
   When function = 'FADEIN' | function = 'FADEOUT' Then
   Do
      startpos = POS('START=', params)
      If startpos > 0 Then
      Do
         starttime = WORD(SUBSTR(params, startpos + 6), 1)
      End
      Else
      Do
         Say
         Say 'Error in parameters.'
         Say
         Call Help
         Exit 2
      End
      endpos = POS('END=', params)
      If endpos > 0 Then
      Do
         endtime = WORD(SUBSTR(params, endpos + 4), 1)
      End
      Else
      Do
         Say
         Say 'Error in parameters.'
         Say
         Call Help
         Exit 2
      End
      starttime = ParseTime(starttime)
      endtime = ParseTime(endtime)
   End
   When function = 'TRUNCATE' Then
   Do
      starttime = 0
      endtime = 0
      startpos = POS('START=', params)
      If startpos > 0 Then
      Do
         starttime = WORD(SUBSTR(params, startpos + 6), 1)
      End
      endpos = POS('END=', params)
      If endpos > 0 Then
      Do
         endtime = WORD(SUBSTR(params, endpos + 4), 1)
      End
      starttime = ParseTime(starttime)
      endtime = ParseTime(endtime)
      If starttime = 0 & endtime = 0 Then
      Do
         Say
         Say 'Error in parameters.'
         Say
         Call Help
         Exit 2
      End
   End
   When function = 'APPEND' | function = 'PREPEND' Then
   Do
      startpos = POS('FILE=', params)
      If startpos > 0 Then
      Do
         If SUBSTR(params, startpos + 5, 1) = '"' Then
         Do
            endpos = POS('"', params, startpos + 6)
            If endpos > 0 Then
            Do
               newfile = SUBSTR(params, startpos + 5, endpos - startpos - 4)
            End
            Else
            Do
               Say
               Say 'Error in parameters.'
               Say
               Call Help
               Exit 2
            End
         End
         Else
            newfile = WORD(SUBSTR(params, startpos + 5), 1)
      End
      Else
      Do
         Say
         Say 'Error in parameters.'
         Say
         Call Help
         Exit 2
      End
   End
   Otherwise
   Do
      Say
      Say 'Unknown function specified.'
      Say
      Call Help
      Exit 2
   End
End

Return

/****************************************/

ParseTime: Procedure

Arg inptime

If POS(':', inptime) = 0 & POS('.', inptime) = 0 Then
   Return inptime

If POS(':', SUBSTR(inptime, POS(':', inptime) + 1)) > 0 Then
Do
   Say
   Say 'Error in input data.'
   Say
   Call WavDropFuncs
   Exit 2
End

If POS('.', SUBSTR(inptime, POS('.', inptime) + 1)) > 0 Then
Do
   Say
   Say 'Error in input data.'
   Say
   Call WavDropFuncs
   Exit 2
End

mins = 0
secs = 0
msecs = 0

If POS(':', inptime) > 0 & POS('.', inptime) > 0 Then
   Parse Value inptime With mins ':' secs '.' msecs
Else
If POS(':', inptime) > 0 Then
   Parse Value inptime With mins ':' secs
Else
If POS('.', inptime) > 0 Then
   Parse Value inptime With secs '.' msecs
Else
Do
   Say
   Say 'Error in input data.'
   Say
   Call WavDropFuncs
   Exit 2
End

Return mins * 60 * 1000 + secs * 1000 + msecs

/****************************************/

Info:

lpeak = WavPeak(lchanbeg, chanlen)
If stereo = 1 Then
   rpeak = WavPeak(rchanbeg, chanlen)
Parse Value WavAvg(lchanbeg, chanlen) With lavg lavgabs
If stereo = 1 Then
   Parse Value WavAvg(rchanbeg, chanlen) With ravg ravgabs

If stereo = 1 Then
Do
   Say 'Peak:   L:' lpeak '   R:' rpeak
   Say 'DC bias:   L:' lavg '   R:' ravg
   Say 'Power:   L:' lavgabs '   R:' ravgabs
   Say
End
Else
Do
   Say 'Peak: ' lpeak
   Say 'DC bias: ' lavg
   Say 'Power: ' lavgabs
   Say
End

rc = WavDropTrac(lchanmem)
If stereo = 1 Then
   rc = WavDropTrac(rchanmem)

Call WavDropFuncs

Exit

/****************************************/

Mono2Stereo:

rchanmem = WavAllocTrac(chanlen + 2 * samples)
rchanbeg = rchanmem + samples

rc = WavCopyTrac(lchanbeg, rchanbeg, chanlen, 1, 1, 'CONST', 'NOMIX')

rc = WavWriteStereo(outfile, lchanbeg, rchanbeg, chanlen)

If rc = 0 Then
Do
   Say
   Say 'Success.'
   Say
End

rc = WavDropTrac(lchanmem)
rc = WavDropTrac(rchanmem)

Call WavDropFuncs

Exit

/****************************************/

FadeIn:

Arg start end

stlchan = lchanbeg + TRUNC((samples / 1000) * start)
If stereo = 1 Then
   strchan = rchanbeg + TRUNC((samples / 1000) * start)
len = TRUNC(ABS(end - start) * (samples / 1000))

If TRUNC((samples / 1000) * start) + len > chanlen Then
Do
   Say
   Say 'Parameter error: Exceeded file size.'
   Say
   Call CleanupAndExit 2
End

rc = WavCopyTrac(stlchan, stlchan, len, 0, 1, 'LOG', 'NOMIX')
If stereo = 1 Then
   rc = WavCopyTrac(strchan, strchan, len, 0, 1, 'LOG', 'NOMIX')

Return

/****************************************/

FadeOut:

Arg start end

stlchan = lchanbeg + TRUNC((samples / 1000) * start)
If stereo = 1 Then
   strchan = rchanbeg + TRUNC((samples / 1000) * start)
len = TRUNC(ABS(end - start) * (samples / 1000))

If TRUNC((samples / 1000) * start) + len > chanlen Then
Do
   Say
   Say 'Parameter error: Exceeded file size.'
   Say
   Call CleanupAndExit 2
End

rc = WavCopyTrac(stlchan, stlchan, len, 1, 0, 'LOG', 'NOMIX')
If stereo = 1 Then
   rc = WavCopyTrac(strchan, strchan, len, 1, 0, 'LOG', 'NOMIX')

Return

/****************************************/

Truncate:

Arg start end

If start > 0 Then
Do
   newbeg = TRUNC(start * (samples / 1000))
   If newbeg >= chanlen Then
   Do
      Say
      Say 'Parameter error: Exceeded file size.'
      Say
   End
   Else
   Do
      lchanbeg = lchanbeg + newbeg
      If stereo = 1 Then
         rchanbeg = rchanbeg + newbeg
      chanlen = chanlen - newbeg
   End
End

If end > 0 Then
Do
   newend = TRUNC(end * (samples / 1000))
   If newend > chanlen Then
   Do
      Say
      Say 'Parameter error: Exceeded file size.'
      Say
   End
   Else
      chanlen = newend
End

Return

/****************************************/

Append:

Arg nfname

If nfchannels = 1 Then
   nftype = 'Mono'
Else
   nftype = 'Stereo'

nfplaylen = nfchanlen / nfsamples
If POS('.', nfplaylen) > 0 Then
Do
   nffrac = SUBSTR(nfplaylen, POS('.', nfplaylen) + 1)
   If LENGTH(nffrac) > 3 Then
      If SUBSTR(nffrac, 4, 1) > 4 Then
         nfplaylen = nfplaylen + 0.001
End
nfmins = TRUNC(nfplaylen / 60)
nfsecs = TRUNC(nfplaylen - nfmins * 60)
nfmsecs = TRUNC(nfplaylen, 3) - TRUNC(nfplaylen)
nfmsecs = SUBSTR(nfmsecs, POS('.', nfmsecs) + 1)

Say
Say 'New file info:'
Say nftype 'file,' nfsamples / 1000 'kHz,' nfbits 'bits,' nfmins 'minutes,' nfsecs 'seconds,' nfmsecs 'miliseconds.'
Say

If nfchannels <> channels | nfsamples <> samples | nfbits <> bits Then
Do
   Say
   Say 'The files are incompatible.'
   Say
   Call CleanupAndExit 2
End

rc = WavReadStereo(nfname, lchanbeg + chanlen, rchanbeg + chanlen)
chanlen = chanlen + nfchanlen

Return

/****************************************/

Prepend:

Arg nfname

If nfchannels = 1 Then
   nftype = 'Mono'
Else
   nftype = 'Stereo'

nfplaylen = nfchanlen / nfsamples
If POS('.', nfplaylen) > 0 Then
Do
   nffrac = SUBSTR(nfplaylen, POS('.', nfplaylen) + 1)
   If LENGTH(nffrac) > 3 Then
      If SUBSTR(nffrac, 4, 1) > 4 Then
         nfplaylen = nfplaylen + 0.001
End
nfmins = TRUNC(nfplaylen / 60)
nfsecs = TRUNC(nfplaylen - nfmins * 60)
nfmsecs = TRUNC(nfplaylen, 3) - TRUNC(nfplaylen)
nfmsecs = SUBSTR(nfmsecs, POS('.', nfmsecs) + 1)

Say
Say 'New file info:'
Say nftype 'file,' nfsamples / 1000 'kHz,' nfbits 'bits,' nfmins 'minutes,' nfsecs 'seconds,' nfmsecs 'miliseconds.'
Say

If nfchannels <> channels | nfsamples <> samples | nfbits <> bits Then
Do
   Say
   Say 'The files are incompatible.'
   Say
   Call CleanupAndExit 2
End

lchanbeg = lchanbeg - nfchanlen
rchanbeg = rchanbeg - nfchanlen
rc = WavReadStereo(nfname, lchanbeg, rchanbeg)
chanlen = chanlen + nfchanlen

Return

/****************************************/

Help:

Say
Say 'Usage: WAVUTIL function input_file [output_file] [function_parameters]'
Say
Say 'Functions and their parameters:'
Say
Say 'INFO'
Say 'FADEIN : START=time END=time'
Say 'FADEOUT : START=time END=time'
Say 'TRUNCATE : [START=time] [END=time]   (at least one parameter is required)'
Say 'APPEND : FILE=filename'
Say 'PREPEND : FILE=filename'
Say 'MONO2STEREO'
Say
Say 'time can be: n - single number representing the number of miliseconds from the'
Say '             start of the file'
Say '         or: mm:ss.ms - minutes, seconds and miliseconds from the start of the'
Say '             file'
Say
Say 'If no output file name is given the input file will also become the output file'
Say

Return

/****************************************/

CleanupAndExit:

Arg exitcode

rc = WavDropTrac(lchanmem)
If stereo = 1 Then
   rc = WavDropTrac(rchanmem)

Call WavDropFuncs

Exit exitcode
