/* service control manager functions:
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is w32funcs.
 *
 * The Initial Developer of the Original Code is Patrick TJ McPhee.
 * Portions created by Patrick McPhee are Copyright 1999
 * Patrick TJ McPhee. All Rights Reserved.
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/services.c 1.4 2002/01/01 08:28:52 pmcphee Rel $
 */

/* functions defined in this file:
 * w32svcstart(servicename)
 * w32svcstop(servicename)
 * w32svcremove(servicename)
 * w32svcinstall(servicename, displayname, programpath[, autostart, user, password])
 * all functions return 0 for success, or an error code. Note: start and stop return 1
 * to mean the function was called unneccesarily.
 * for start, stop, and remove, servicename can be either the registry key for
 * the service, or the display name as shown in the service control manager
 */ 
 

#include <windows.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include "rxproto.h"

static SC_HANDLE smHandle = NULL;

static int svcconnect()
{
   if (!smHandle) {
      smHandle = OpenSCManager(0, 0, SC_MANAGER_ALL_ACCESS);
   }

   return smHandle != NULL;
}

static void svcdisconnect()
{
   if (smHandle != NULL)
      CloseServiceHandle(smHandle);
   smHandle = NULL;
}



int Install(const char * const servName, const char * const dispName,
                          const char * const path, const char * const user,
                          const char * const pwd, BOOL autoStart)
{
   SC_HANDLE srv;

   if (!svcconnect())
      return -1;

   srv = CreateService(smHandle, servName, dispName, SERVICE_ALL_ACCESS,
                       SERVICE_WIN32_OWN_PROCESS, autoStart ? SERVICE_AUTO_START : SERVICE_DEMAND_START,
                       SERVICE_ERROR_NORMAL, path, 0, 0, 0, user, pwd);

   if (srv == NULL) {
      return -2;
   }


   CloseServiceHandle(srv);
   return 0;
}

int Remove(const char * const servName)
{
   SC_HANDLE      srv;
   SERVICE_STATUS stat;
   int rc = 0;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_ALL_ACCESS | DELETE);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_ALL_ACCESS | DELETE);
      }
   }

   if (!srv)
      return -2;

   if (!QueryServiceStatus(srv, &stat))
      rc = -3;

   else {
      if (stat.dwCurrentState != SERVICE_STOPPED) {
         if (!ControlService(srv, SERVICE_CONTROL_STOP, &stat))
            return -4;
      }

      if (!DeleteService(srv))
         rc = -5;
   }

   CloseServiceHandle(srv);

   return rc;
}

int Start(const char * const servName)
{
   SC_HANDLE srv;
   SERVICE_STATUS stat;
   int rc = 0;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_QUERY_STATUS|SERVICE_START);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_QUERY_STATUS|SERVICE_START);
      }
   }

   if (!srv)
      return -2;

   if (!QueryServiceStatus(srv, &stat)) {
      rc = -3;
   }

   else if (stat.dwCurrentState != SERVICE_STOPPED) {
      rc = 1;   /* not really an error */
   }

   else if (!StartService(srv, 0, NULL)) {
      rc = -4;
   }

   CloseServiceHandle(srv);
   return rc;
}

int Stop(const char * const servName)
{
   int rc = 0;
   SC_HANDLE srv;
   SERVICE_STATUS stat;
   char key[100];
   long len = sizeof(key);

   if (!svcconnect())
      return -1;
   
   srv = OpenService(smHandle, servName, SERVICE_QUERY_STATUS|SERVICE_STOP);

   if (!srv) {
      if (GetServiceKeyName(smHandle, servName, key, &len)) {
         srv = OpenService(smHandle, key, SERVICE_QUERY_STATUS|SERVICE_STOP);
      }
   }

   if (srv == NULL)
      return -2;

   if (!QueryServiceStatus(srv, &stat)) {
      rc = -3;
   }

   else if (stat.dwCurrentState == SERVICE_STOPPED) {
      rc = 1;   /* not really an error */
   }

   if (!ControlService(srv, SERVICE_CONTROL_STOP, &stat))
      return -4;

   CloseServiceHandle(srv);

   return rc;
}

/* w32svcstart servicename */
rxfunc(w32svcstart)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Start(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcstop servicename */
rxfunc(w32svcstop)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Stop(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcremove servicename */
rxfunc(w32svcremove)
{
   char * servName;
   int rc;

   checkparam(1,1);
   rxstrdup(servName, argv[0]);
   rc = Remove(servName);
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}

/* svcinstall servicename, display name, program path[, autostart, user, password] */
rxfunc(w32svcinstall)
{
   char * servName, *dispname, *path, *as = "0", *user = NULL, *pwd = NULL;
   int rc;

   checkparam(3,6);
   rxstrdup(servName, argv[0]);
   rxstrdup(dispname, argv[1]);
   rxstrdup(path, argv[2]);

   if (argc > 3 && argv[3].strlength)
      rxstrdup(as, argv[3]);

   if (argc > 4 && argv[4].strlength)
      rxstrdup(user, argv[4]);

   if (argc > 5 && argv[5].strlength)
      rxstrdup(pwd, argv[5]);

   rc = Install(servName, dispname, path, user, pwd, atoi(as));
   result->strlength = sprintf(result->strptr, "%d", rc);

   return 0;
}
