/* system information and control functions:
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is w32funcs.
 *
 * The Initial Developer of the Original Code is Patrick TJ McPhee.
 * Portions created by Patrick McPhee are Copyright 2001
 * Patrick TJ McPhee. All Rights Reserved.
 *
 * $Header: C:/ptjm/rexx/w32funcs/RCS/control.c 1.7 2003/12/14 23:57:55 ptjm Rel $
 */

/* functions defined in this file:
 * w32sysgetusername() -> value
 * w32sysgetuserdirectory() -> value
 * w32sysgetcomputername() -> value
 * w32syssetcomputername(newname) -> 0 or 1
 * w32sysgethardwareprofilename() -> value
 * w32sysshutdown(how, options) -> 0 or 1
 * w32syssetpowerstate(suspend, force) -> 0 or 1
 * w32sysgetpowerstatus() -> line battery batterypct batterytime batterylife
 * w32sysgetosversion() -> major '.' minor build platform CSDVersion
 * w32sysgetcolours(stem) -> 0 or 1
 * w32syssetcolours(stem) -> 0 or 1
 * w32sysgetparameters(parm) -> value(s)
 * w32syssetparameters(stem, permanent, value[, value2, ...]) -> 0 or 1
 * w32sysshortfilename(name[, name, ...]) -> shortname [shortname ...]
 * w32syslongfilename(name[, name, ...]) -> longname [longname ...]
 * w32sysfullfilename(name[, name, ...]) -> fullname [fullname ...]
 */ 
 
#define _WIN32_WINNT 0x400
#include <windows.h>
#include <malloc.h>
#include "rxproto.h"

rxfunc(w32sysgetusername)
{
   unsigned long len = DEFAULTSTRINGSIZE;
   int rc;

   rc = GetUserName(result->strptr, &len);
   result->strlength = len;
   return 0;
}

#if 0
rxfunc(w32sysgetuserdirectory)
{
   unsigned long len = DEFAULTSTRINGSIZE;
   int rc;
   HANDLE proc, tok;

   proc = GetCurrentProcess();
   OpenProcessToken(proc, TOKEN_QUERY, &tok);
   rc = GetUserProfileDirectory(tok, result->strptr, &len);
   CloseHandle(tok);
   CloseHandle(proc);

   result->strlength = len;
   return 0;
}
#endif

rxfunc(w32sysgetcomputername)
{
   unsigned long len = DEFAULTSTRINGSIZE;
   int rc;

   rc = GetComputerName(result->strptr, &len);
   result->strlength = len;

   return 0;
}

rxfunc(w32syssetcomputername)
{
   char * newname;
   int rc;

   checkparam(1,1);


   rxstrdup(newname, argv[0]);

   rc = SetComputerName(newname);

   if (rc)
      result_zero();
   else
      result_one();

   return 0;
}

rxfunc(w32sysgethardwareprofilename)
{
   HW_PROFILE_INFO hwp;
   int rc;

   /* the mingw compiler doesn't really support GetCurrentHwProfile */
#ifndef __MINGW32__
   memset(&hwp, 0, sizeof(hwp));

   rc = GetCurrentHwProfile(&hwp);

   if (rc) {
      result->strlength = strlen(hwp.szHwProfileName);
      memcpy(result->strptr, hwp.szHwProfileName, result->strlength);
   }

   else
#else
      result->strlength = 0;
#endif

   return 0;
}

/* NT security is very strong. You must first ask for permission before
 * doing something dangerous like shutting down the system -- and
 * see how many calls it takes! That should discourage the blighters! */ 
static int getshutdownpriv()
{
   HANDLE perms, proc;
   TOKEN_PRIVILEGES tp;
   int rc;
   
   proc = GetCurrentProcess();
   OpenProcessToken(proc, TOKEN_ADJUST_PRIVILEGES, &perms);
   tp.PrivilegeCount = 1;

   LookupPrivilegeValue(NULL, SE_SHUTDOWN_NAME, &tp.Privileges[0].Luid);
   tp.Privileges[0].Attributes = SE_PRIVILEGE_ENABLED;

   rc = AdjustTokenPrivileges(perms, FALSE, &tp, 0, NULL, NULL);
   CloseHandle(perms);
   CloseHandle(proc);

   return rc;
}


/* shut down the machine. How? shutdown, reboot, power-off, logoff.
 * HowHard? force, noforce */
rxfunc(w32sysshutdown)
{
   int flags, rc;
   char * how = "shutdown", * howhard = "noforce";

   checkparam(0, 2);

   if (argc > 0 && argv[0].strlength > 0) {
      rxstrdup(how, argv[0]);
      strlwr(how);
   }

   if (argc > 1 && argv[1].strlength > 0) {
      rxstrdup(howhard, argv[1]);
      strlwr(howhard);
   }

   switch (*how) {
      case 'r': flags = EWX_REBOOT; break;
      case 'p': flags = EWX_POWEROFF; break;
      case 'l': flags = EWX_LOGOFF; break;
      case 's': flags = EWX_SHUTDOWN; break;
      default: return BADARGS;
   }

   /* do this before checking next flag because it makes the test simpler.
    * Don't act on the value of rc until afterwards, because an invalid
    * parameter trumps insufficient privilege in my books. */
   if (flags != EWX_LOGOFF)
      rc = getshutdownpriv();

   switch (*howhard) {
      case 'f': flags |= EWX_FORCE; break;
      case 'n': ; break;
      default: return BADARGS;
   }

   if (!rc) {
      result_one();
      return 0;
   }

   rc = ExitWindowsEx(flags, 0);

   if (!rc) {
      result_one();
   }
   else {
      result_zero();
   }
   return 0;
}

rxfunc(w32syssetpowerstate)
{
   int suspend = 1, force = 0;
   char * how = "suspend", * howhard = "noforce";

   checkparam(0,2);
   if (argc > 0 && argv[0].strptr) {
      rxstrdup(how, argv[0]);
      strlwr(how);
   }
   if (argc > 1 && argv[1].strptr) {
      rxstrdup(howhard, argv[1]);
      strlwr(howhard);
   }

   if (how[0] == 's') suspend = 1;
   else if (how[0] == 'n') suspend = 0;
   else return BADARGS;

   if (howhard[0] == 'f') force = 1;
   else if (howhard[0] == 'n') force = 0;
   else return BADARGS;

   getshutdownpriv();

   result->strlength = sprintf(result->strptr, "%d", SetSystemPowerState(suspend,force));
   return 0;
}

rxfunc(w32sysgetpowerstatus)
{
   SYSTEM_POWER_STATUS sps;
   int rc;

   memset(&sps, 0, sizeof(sps));

   rc = GetSystemPowerStatus(&sps);

   if (rc) {
      char * line, * bat;

      switch (sps.ACLineStatus) {
         case AC_LINE_OFFLINE: line = "Offline"; break;
         case AC_LINE_ONLINE: line = "Online"; break;
         case AC_LINE_BACKUP_POWER: line = "Backup"; break;
         default:
         case AC_LINE_UNKNOWN: line = "Unknown"; break;
      }

      switch (sps.BatteryFlag) {
         case BATTERY_FLAG_HIGH: bat = "High"; break;
         case BATTERY_FLAG_LOW: bat = "Low"; break;
         case BATTERY_FLAG_CRITICAL: bat = "Critical"; break;
         case BATTERY_FLAG_CHARGING: bat = "Charging"; break;
         case BATTERY_FLAG_NO_BATTERY: bat = "NoBattery"; break;
         default:
         case BATTERY_FLAG_UNKNOWN: bat = "Unknown"; break;
      }

      result->strlength = sprintf(result->strptr, "%s %s %d %d %d", line, bat,
                                  (int)sps.BatteryLifePercent, sps.BatteryLifeTime,
                                  sps.BatteryFullLifeTime);
   }
   else {
      result->strlength = 0;
   }

   return 0;
}

rxfunc(w32sysgetosversion)
{
   OSVERSIONINFO osvi;
   int rc;

   memset(&osvi, 0, sizeof(osvi));
   osvi.dwOSVersionInfoSize = sizeof(osvi);

   rc = GetVersionEx(&osvi);

   if (rc) {
      result->strlength = sprintf(result->strptr, "%d.%d %d %d %s",
                                  osvi.dwMajorVersion, osvi.dwMinorVersion,
                                  osvi.dwBuildNumber, osvi.dwPlatformId,
                                  osvi.szCSDVersion);
   }
   else {
      result->strlength = 0;
   }

   return 0;
}

static struct syscolours_T {
  char * name;
  int namel;
  int colour;
} syscolours[] = {
   "ACTIVEBORDER", 12, COLOR_ACTIVEBORDER,
   "ACTIVETITLE", 11, COLOR_ACTIVECAPTION,
   "APPWORKSPACE", 12, COLOR_APPWORKSPACE,
   "BACKGROUND", 10, COLOR_BACKGROUND,
   "BUTTONDKSHADOW", 14, COLOR_3DDKSHADOW,
   "BUTTONFACE", 10, COLOR_BTNFACE,
   "BUTTONHIGHLIGHT", 15, COLOR_BTNHIGHLIGHT,
   "BUTTONHILIGHT", 13, COLOR_BTNHIGHLIGHT,
   "BUTTONLIGHT", 11, COLOR_3DLIGHT,
   "BUTTONSHADOW", 12, COLOR_BTNSHADOW,
   "BUTTONTEXT", 10, COLOR_BTNTEXT,
   "GRAYTEXT", 8, COLOR_GRAYTEXT,
   "GREYTEXT", 8, COLOR_GRAYTEXT,
   "HIGHLIGHT", 9, COLOR_HIGHLIGHT,
   "HIGHLIGHTTEXT", 13, COLOR_HIGHLIGHTTEXT,
   "HILIGHT", 7, COLOR_HIGHLIGHT,
   "HILIGHTTEXT", 11, COLOR_HIGHLIGHTTEXT,
   "INACTIVEBORDER", 14, COLOR_INACTIVEBORDER,
   "INACTIVETITLE", 13, COLOR_INACTIVECAPTION,
   "INACTIVETITLETEXT", 17, COLOR_INACTIVECAPTIONTEXT,
   "INFOTEXT", 8, COLOR_INFOTEXT,
   "INFOWINDOW", 10, COLOR_INFOBK,
   "MENU", 4, COLOR_MENU,
   "MENUTEXT", 8, COLOR_MENUTEXT,
   "SCROLLBAR", 9, COLOR_SCROLLBAR,
   "TITLETEXT", 9, COLOR_CAPTIONTEXT,
   "WINDOW", 6, COLOR_WINDOW,
   "WINDOWFRAME", 11, COLOR_WINDOWFRAME,
   "WINDOWTEXT", 10, COLOR_WINDOWTEXT
};

rxfunc(w32sysgetcolours)
{
   register int i;
   char name[1024], colour[13];
   SHVBLOCK svb;
   int nl;
   COLORREF syscol;

   checkparam(1, 1);

   memcpy(name, argv[0].strptr, argv[0].strlength);
   if (name[argv[0].strlength-1] == '.')
      nl = argv[0].strlength;
   else {
      nl = argv[0].strlength + 1;
      name[nl-1] = '.';
   }

   memset(&svb, 0, sizeof(svb));
   svb.shvname.strptr = name;
   svb.shvvalue.strptr = colour;
   svb.shvcode = RXSHV_SET;

   for (i = 0; i < DIM(syscolours); i++) {
      svb.shvname.strlength = nl + syscolours[i].namel;
      memcpy(name+nl, syscolours[i].name, syscolours[i].namel);
      syscol = GetSysColor(syscolours[i].colour);
      svb.shvvalue.strlength = sprintf(colour, "%u %u %u", GetRValue(syscol),
                                       GetGValue(syscol), GetBValue(syscol));
      RexxVariablePool(&svb);
   }

   result_zero();

   return 0;
}

rxfunc(w32syssetcolours)
{
   register int i, nc = 0;
   char name[1024], colour[13];
   int colours[30], r, g, b, rc = 0;
   COLORREF values[30]; 
   SHVBLOCK svb;
   int nl;

   checkparam(1, 1);

   memcpy(name, argv[0].strptr, argv[0].strlength);
   if (name[argv[0].strlength-1] == '.')
      nl = argv[0].strlength;
   else {
      nl = argv[0].strlength + 1;
      name[nl-1] = '.';
   }

   memset(&svb, 0, sizeof(svb));
   svb.shvname.strptr = name;
   svb.shvvalue.strptr = colour;
   svb.shvcode = RXSHV_FETCH;

   for (i = 0; i < DIM(syscolours); i++) {
      svb.shvname.strlength = nl + syscolours[i].namel;
      memcpy(name+nl, syscolours[i].name, syscolours[i].namel);
      svb.shvvalue.strlength = svb.shvvaluelen = sizeof(colour);
      RexxVariablePool(&svb);
      if (!(svb.shvret & RXSHV_NEWV)) {
         colour[svb.shvvalue.strlength] = 0;
         sscanf(colour, "%d %d %d", &r, &g, &b);
         values[nc] = RGB(r,g,b);
         colours[nc] = syscolours[i].colour;
         nc++;
         if (nc >= 30) {
            rc = SetSysColors(nc, colours, values);
            nc = 0;
         }
      }
   }

   if (nc) {
      rc = SetSysColors(nc, colours, values);
   }

   if (rc)
      result_zero();
   else
      result_one();

   return 0;
}

/* system information */
struct sysinfoparm {
   const char * name;
   int namel;
   const int getparam, setparam, parms, flags;
};

#define ARG_STRING 1
#define ARG_STRUCT 2

/* structure which can represent fairly arbitrary windows structures */
struct genericwindowsstructure {
   unsigned int size;
   unsigned int parms[20];
};

/* mingw compiler doesn't have all the needed definitions */
#ifndef SPI_SETCURSORS
# define SPI_SETCURSORS		  87
# define SPI_SETICONS		  88
# define SPI_GETSNAPTODEFBUTTON	  95
# define SPI_SETSNAPTODEFBUTTON	  96
# define SPI_GETMOUSEHOVERWIDTH	  98
# define SPI_SETMOUSEHOVERWIDTH	  99
# define SPI_GETMOUSEHOVERHEIGHT 100
# define SPI_SETMOUSEHOVERHEIGHT 101
# define SPI_GETMOUSEHOVERTIME	 102
# define SPI_SETMOUSEHOVERTIME	 103
# define SPI_GETWHEELSCROLLLINES 104
# define SPI_SETWHEELSCROLLLINES 105
#endif


/* list of system parameters, sorted by length and then alphabetically */
static struct sysinfoparm sysparms[] = {
   "BEEP", 4, SPI_GETBEEP, SPI_SETBEEP, 1, 0,
   "ICONS", 5, -1, SPI_SETICONS, 0, 0,
   "MOUSE", 5, SPI_GETMOUSE, SPI_SETMOUSE, 1, 0,
   "BORDER", 6, SPI_GETBORDER, SPI_SETBORDER, 1, 0,
   "CURSORS", 7, -1, SPI_SETCURSORS, 0, 0,
   "ANIMATION", 9, SPI_GETANIMATION, SPI_SETANIMATION, 1, ARG_STRUCT,
   "DRAGWIDTH", 9, -1, SPI_SETDRAGWIDTH, 1, 0,
   "MOUSEKEYS", 9, SPI_GETMOUSEKEYS, SPI_SETMOUSEKEYS, 6, ARG_STRUCT,
   "DRAGHEIGHT", 10, -1, SPI_SETDRAGHEIGHT, 1, 0,
   "FILTERKEYS", 10, SPI_GETFILTERKEYS, SPI_SETFILTERKEYS, 5, ARG_STRUCT,
   "LANGTOGGLE", 10, -1, SPI_SETLANGTOGGLE, 0, 0,
   "PENWINDOWS", 10, -1, SPI_SETPENWINDOWS, 1, 0,
   "SHOWSOUNDS", 10, SPI_GETSHOWSOUNDS, SPI_SETSHOWSOUNDS, 1, 0,
   "STICKYKEYS", 10, SPI_GETSTICKYKEYS, SPI_SETSTICKYKEYS, 1, ARG_STRUCT,
   "TOGGLEKEYS", 10, SPI_GETTOGGLEKEYS, SPI_SETTOGGLEKEYS, 1, ARG_STRUCT,
   "DESKPATTERN", 11, -1, SPI_SETDESKPATTERN, 0, 0,
   "MOUSETRAILS", 11, SPI_GETMOUSETRAILS, SPI_SETMOUSETRAILS, 1, 0,
   "KEYBOARDPREF", 12, SPI_GETKEYBOARDPREF, SPI_SETKEYBOARDPREF, 1, 0,
   "SCREENREADER", 12, SPI_GETSCREENREADER, SPI_SETSCREENREADER, 1, 0,
   "ACCESSTIMEOUT", 13, SPI_GETACCESSTIMEOUT, SPI_SETACCESSTIMEOUT, 2, ARG_STRUCT,
   "DESKWALLPAPER", 13, -1, SPI_SETDESKWALLPAPER, 1, ARG_STRING,
   "FONTSMOOTHING", 13, SPI_GETFONTSMOOTHING, SPI_SETFONTSMOOTHING, 1, 0,
   "ICONTITLEWRAP", 13, SPI_GETICONTITLEWRAP, SPI_SETICONTITLEWRAP, 1, 0,
   "KEYBOARDDELAY", 13, SPI_GETKEYBOARDDELAY, SPI_SETKEYBOARDDELAY, 1, 0,
   "KEYBOARDSPEED", 13, SPI_GETKEYBOARDSPEED, SPI_SETKEYBOARDSPEED, 1, 0,
   "DOUBLECLKWIDTH", 14, -1, SPI_SETDOUBLECLKWIDTH, 1, 0,
   "LOWPOWERACTIVE", 14, SPI_GETLOWPOWERACTIVE, SPI_SETLOWPOWERACTIVE, 1, 0,
   "MOUSEHOVERTIME", 14, SPI_GETMOUSEHOVERTIME, SPI_SETMOUSEHOVERTIME, 1, 0,
   "POWEROFFACTIVE", 14, SPI_GETPOWEROFFACTIVE, SPI_SETPOWEROFFACTIVE, 1, 0,
   "DOUBLECLICKTIME", 15, -1, SPI_SETDOUBLECLICKTIME, 1, 0,
   "DOUBLECLKHEIGHT", 15, -1, SPI_SETDOUBLECLKHEIGHT, 1, 0,
   "DRAGFULLWINDOWS", 15, SPI_GETDRAGFULLWINDOWS, SPI_SETDRAGFULLWINDOWS, 1, 0,
   "GRIDGRANULARITY", 15, SPI_GETGRIDGRANULARITY, SPI_SETGRIDGRANULARITY, 1, 0,
   "LOWPOWERTIMEOUT", 15, SPI_GETLOWPOWERTIMEOUT, SPI_SETLOWPOWERTIMEOUT, 1, 0,
   "MOUSEBUTTONSWAP", 15, -1, SPI_SETMOUSEBUTTONSWAP, 1, 0,
   "MOUSEHOVERWIDTH", 15, SPI_GETMOUSEHOVERWIDTH, SPI_SETMOUSEHOVERWIDTH, 1, 0,
   "POWEROFFTIMEOUT", 15, SPI_GETPOWEROFFTIMEOUT, SPI_SETPOWEROFFTIMEOUT, 1, 0,
   "SNAPTODEFBUTTON", 15, SPI_GETSNAPTODEFBUTTON, SPI_SETSNAPTODEFBUTTON, 1, 0,
   "MINIMIZEDMETRICS", 16, SPI_GETMINIMIZEDMETRICS, SPI_SETMINIMIZEDMETRICS, 4, ARG_STRUCT,
   "MOUSEHOVERHEIGHT", 16, SPI_GETMOUSEHOVERHEIGHT, SPI_SETMOUSEHOVERHEIGHT, 1, 0,
   "SCREENSAVEACTIVE", 16, SPI_GETSCREENSAVEACTIVE, SPI_SETSCREENSAVEACTIVE, 1, 0,
   "WHEELSCROLLLINES", 16, SPI_GETWHEELSCROLLLINES, SPI_SETWHEELSCROLLLINES, 1, 0,
   "WINDOWSEXTENSION", 16, SPI_GETWINDOWSEXTENSION, -1, 1, 0,
   "SCREENSAVETIMEOUT", 17, SPI_GETSCREENSAVETIMEOUT, SPI_SETSCREENSAVETIMEOUT, 1, 0,
   "SCREENSAVERRUNNING", 18, SPI_SCREENSAVERRUNNING, SPI_SCREENSAVERRUNNING, 1, 0,
   "ICONVERTICALSPACING", 19, SPI_ICONVERTICALSPACING, SPI_ICONVERTICALSPACING, 1, 0,
   "ICONHORIZONTALSPACING", 21, SPI_ICONHORIZONTALSPACING, SPI_ICONHORIZONTALSPACING, 1, 0,
/*   "DEFAULTINPUTLANG", 16, SPI_GETDEFAULTINPUTLANG, SPI_SETDEFAULTINPUTLANG, 3, 0, */
/*   "HIGHCONTRAST", 12, SPI_GETHIGHCONTRAST, SPI_SETHIGHCONTRAST, 3, 0, */
/*   "ICONTITLELOGFONT", 16, SPI_GETICONTITLELOGFONT, SPI_SETICONTITLELOGFONT, 3, 0, */
/*   "NONCLIENTMETRICS", 16, SPI_GETNONCLIENTMETRICS, SPI_SETNONCLIENTMETRICS, 3, 0, */
/*   "SERIALKEYS", 10, SPI_GETSERIALKEYS, SPI_SETSERIALKEYS, 3, 0, */
/*   "SOUNDSENTRY", 11, SPI_GETSOUNDSENTRY, SPI_SETSOUNDSENTRY, 3, 0, */
/*   "WORKAREA", 8, SPI_GETWORKAREA, SPI_SETWORKAREA */
/*?   "FASTTASKSWITCH", 14, SPI_GETFASTTASKSWITCH, SPI_SETFASTTASKSWITCH, 3, 0, */
/*?   "HANDHELD", 8, -1, SPI_SETHANDHELD,*/
/*?   "LANGDRIVER", 10, SPI_LANGDRIVER, SPI_LANGDRIVER, 3, 0, */
/*?   "MENUDROPALIGNMENT", 17, SPI_GETMENUDROPALIGNMENT, SPI_SETMENUDROPALIGNMENT, 3, 0, */
};

/* search for a parameter which matches the rxstring */
static const int parmcmp(const PRXSTRING l, const struct sysinfoparm * r)
{
   int c = l->strlength - r->namel;
   if (!c)
      c = strcasecmp(l->strptr, r->name);

   return c;
}



rxfunc(w32sysgetparameter)
{
   struct sysinfoparm *ssi;
   struct genericwindowsstructure gws;
   unsigned int parm1;
   char buf[11];
   void * parm2;
   int rc;
   register int i;

   checkparam(1,1);

   ssi =  bsearch(argv, sysparms, DIM(sysparms), sizeof(*sysparms), parmcmp);
   
   if (ssi == NULL || ssi->getparam == -1) {
      return BADARGS;
   }

   if (ssi->flags & ARG_STRUCT) {
      memset(&gws, 0, sizeof(gws));
      parm2 = &gws;
      parm1 = 0;
      gws.size = sizeof(int) * (1+ssi->parms);
   }
   else {
      parm1 = 0;
      parm2 = gws.parms;
   }

   rc = SystemParametersInfo(ssi->getparam, parm1, parm2, 0);

   if (rc)
      for (i = result->strlength = 0; i < ssi->parms; i++) {
         result->strlength += sprintf(result->strptr+result->strlength, "%u%s",
                                      gws.parms[i], i == (ssi->parms - 1) ? "" : " ");
      }

   else {
      result->strlength = 0;
   }

   return 0;
}

rxfunc(w32syssetparameter)
{
   struct sysinfoparm *ssi;
   struct genericwindowsstructure gws;
   unsigned int parm1, flags = 0;
   char buf[11];
   void * parm2;
   int rc;
   register int i;

   checkparam(1,6);

   ssi =  bsearch(argv, sysparms, DIM(sysparms), sizeof(*sysparms), parmcmp);
   
   if (ssi == NULL || ssi->setparam == -1 || (argc - 2) > ssi->parms) {
      return BADARGS;
   }

   if (argc > 1) {
      for (i = 0; i < argv[1].strlength; i++) {
         /* flags: n -- notify applications of change
          *        p -- make change permanent
          */
         switch (argv[1].strptr[i]) {
            case 'p':
            case 'P': flags |= SPIF_UPDATEINIFILE;
               break;
            case 'n':
            case 'N': flags |= SPIF_SENDCHANGE;
               break;

            /* there are several valid separators */
            case ' ':
            case ',':
            case ';':
            case ':':
            case '/':
               break;

            default:
               return BADARGS;
         }
      }
   }


   if (ssi->parms == 0) {
      parm1 = 0;
      parm2 = NULL;
   }

   if (ssi->flags & ARG_STRUCT) {
      memset(&gws, 0, sizeof(gws));
      parm2 = &gws;
      parm1 = gws.size = sizeof(int) * (1+ssi->parms);
      for (i = 2; i < argc; i++) {
         if (argv[i].strlength <= 0 || argv[i].strlength > 10)
            continue;
         memcpy(buf, argv[i].strptr, argv[i].strlength);
         buf[argv[i].strlength] = 0;
         gws.parms[i-2] = atoi(buf);
      }
   }
   else if (ssi->flags & ARG_STRING) {
      parm1 = argv[2].strlength;
      parm2 = argv[2].strptr;
   }
   else {
      parm2 = NULL;
      if (argc < 2 || argv[2].strlength <= 0 || argv[2].strlength > 10)
         parm1 = 0;
      else {
         memcpy(buf, argv[2].strptr, argv[2].strlength);
         buf[argv[2].strlength] = 0;
         parm1 = atoi(buf);
      }
   }

   rc = SystemParametersInfo(ssi->setparam, parm1, parm2, flags);

   if (rc)
      result_zero();
   else
      result_one();

   return 0;
}


/* given one or more file names, return the stupid versions of the names.
 * These are guaranteed not to have spaces, so they're space-delimited.
 * Of course, putting spaces in file names is even stupider than the whole
 * stupid `short' file name concept */
rxfunc(w32sysshortfilename)
{
   register int i, maxl, reslen, l;
   unsigned char s[_MAX_PATH], * t;

   checkparam(1,-1);

   result->strlength = 0;
   for (i = l = 0, reslen = DEFAULTSTRINGSIZE-1; i < argc; i++) {
      if (argv[i].strlength > _MAX_PATH)
         return BADARGS;
      memcpy(s, argv[i].strptr, argv[i].strlength);
      s[argv[i].strlength] = 0;

      if (i > 0)
         result->strptr[result->strlength++] = ' ';

      l = GetShortPathName(s, result->strptr+result->strlength, reslen - result->strlength);

      if ((l+result->strlength) > reslen) {
         t = REXXALLOCATEMEMORY(l+reslen*2);
         if (i > 0)
            result->strlength--;
         memcpy(t, result->strptr, result->strlength);
         if (reslen > DEFAULTSTRINGSIZE)
            REXXFREEMEMORY(result->strptr);
         result->strptr = t;
         reslen = l + reslen*2 - 1;
      }
      else {
         result->strlength += l;
      }
   }

   return 0;
}

/* the inverse of w32sysshortfilename.
 * They're space-delimited for compatibility with sysshortfilename, but this is not
 * safe if you can't depend on people being sensible. */

/* it turns out this needs to be done once for each component of the path */
static const unsigned char * componentLongname(PRXSTRING ln, const PRXSTRING sn, const unsigned char * snp)
{
   WIN32_FIND_DATA pwfd;
   HANDLE sh;
   char s[_MAX_PATH];
   unsigned char * sd1, *sd2;
   int ll, sl;

   if (!snp)
      snp = sn->strptr;

   sd1 = memchr(snp, '\\', sn->strlength - (snp - (const unsigned char *) sn->strptr));
   sd2 = memchr(snp, '/', sn->strlength - (snp - (const unsigned char *) sn->strptr));

   if (sd1 && (!sd2 || sd2 > sd1)) {
      sl = sd1 - snp;
      memcpy(s, sn->strptr, sd1 - (const unsigned char *)sn->strptr);
      s[sd1 - (const unsigned char *)sn->strptr] = 0;
   }
   else if (sd2) {
      sl = sd2 - snp;
      memcpy(s, sn->strptr, sd2 - (const unsigned char *)sn->strptr);
      s[sd2 - (const unsigned char *)sn->strptr] = 0;
   }
   else {
      sl = sn->strlength - (snp - (const unsigned char *)sn->strptr);
      memcpy(s, sn->strptr, sn->strlength);
      s[sn->strlength] = 0;
   }

   sh = FindFirstFile(s, &pwfd);
   if (sh == INVALID_HANDLE_VALUE)
      ll = 0;
   else {
      ll = strlen(pwfd.cFileName);
      FindClose(sh);
   }

   if (ln->strlength) {
      ln->strptr[ln->strlength++] = '\\';
   }


   if (ll) {
      /* need to special-case d:\ */
      if (sl == 2 && snp == sn->strptr && snp[1] == ':') {
         memcpy(ln->strptr+ln->strlength, snp, sl);
         ln->strlength += sl;
      }
      else {
         memcpy(ln->strptr+ln->strlength, pwfd.cFileName, ll);
         ln->strlength += ll;
      }
   }
   else {
      memcpy(ln->strptr+ln->strlength, snp, sl);
      ln->strlength += sl;
   }


   snp += sl + 1;



   if (snp >= (sn->strptr + sn->strlength))
      snp = NULL;


   return snp;
}




rxfunc(w32syslongfilename)
{
   register int i, reslen;
   unsigned char * t;
   const unsigned char * s;
   RXSTRING ln;

   checkparam(1,-1);

   result->strlength = 0;
   /* make sure we start with a lot of memory */
   reslen = 2*_MAX_PATH;
   result->strptr = REXXALLOCATEMEMORY(reslen);

   for (i = 0; i < argc; i++) {
      if (i > 0)
         result->strptr[result->strlength++] = ' ';

      /* always need to have _MAX_PATH bytes available */
      if ((reslen - result->strlength) < _MAX_PATH) {
         reslen = result->strlength + 2*_MAX_PATH;
         t = REXXALLOCATEMEMORY(reslen);
         memcpy(t, result->strptr, result->strlength);
         REXXFREEMEMORY(result->strptr);
         result->strptr = t;
      }


      ln.strptr = result->strptr+result->strlength;
      ln.strlength = 0;
      s = NULL;

      while (s = componentLongname(&ln, argv+i, s))
         ;

      result->strlength += ln.strlength;
   }

   return 0;
}

/* given a (presumably relative) path, return the fully qualified path */
rxfunc(w32sysfullfilename)
{
   register int i, maxl, reslen, l;
   unsigned char s[_MAX_PATH], * t;

   checkparam(1,-1);

   result->strlength = 0;
   for (i = l = 0, reslen = DEFAULTSTRINGSIZE-1; i < argc; i++) {
      if (argv[i].strlength > _MAX_PATH)
         return BADARGS;
      memcpy(s, argv[i].strptr, argv[i].strlength);
      s[argv[i].strlength] = 0;

      if (i > 0)
         result->strptr[result->strlength++] = ' ';

      /* this may seem similar to GetShortPathName, but that doesn't mean the
       * arguments are in the same order. prats */
      l = GetFullPathName(s, reslen - result->strlength, result->strptr+result->strlength, &t);

      if ((l+result->strlength) > reslen) {
         t = REXXALLOCATEMEMORY(l+reslen*2);
         if (i > 0)
            result->strlength--;
         memcpy(t, result->strptr, result->strlength);
         if (reslen > DEFAULTSTRINGSIZE)
            REXXFREEMEMORY(result->strptr);
         result->strptr = t;
         reslen = l + reslen*2 - 1;
      }
      else {
         result->strlength += l;
      }
   }

   return 0;
}

