/*******************************************************************
 *
 *  ttfile.h                                                     1.3
 *
 *    File I/O Component (specification).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Changes between 1.3 and 1.2 :
 *
 *  - all functions report error values now
 *
 *  - the stream semantics have also changed
 *
 *  Changes between 1.2 and 1.1 :
 *
 *  - added macros to support re-entrant builds
 *
 *  - added the TT_Duplicate_File function to duplicate streams
 *    ( re-entrant builds only )
 *
 ******************************************************************/

/* #define TT_CONFIG_REENTRANT */

#ifndef TTFILE_H
#define TTFILE_H

#include "freetype.h"
#include "tterror.h"

  #ifdef __cplusplus
  extern "C" {
  #endif
  
  TT_Error  TTFile_Init();
  /* Init file component */

  TT_Error  TTFile_Done();
  /* Done with file component */

  /**********************************************************************/
  /*                                                                    */
  /*  Stream functions.                                                 */
  /*                                                                    */
  /**********************************************************************/

  TT_Error  TT_Open_Stream( const char*  name,
                            TT_Stream*   stream );
  /* Open a file and return a stream handle for it           */
  /* should only be used for a new face object's main stream */

  TT_Error  TT_Close_Stream( TT_Stream*  stream );
  /* closes, then discards, a stream when it becomes unuseful   */
  /* should only be used for a stream opend with TT_Open_Stream */

  TT_Error  TT_Use_Stream( TT_Stream   org_stream,
                           TT_Stream*  stream );
  /* Notice the component that we're going to use the file     */
  /* opened in 'org_stream', and report errors to the 'error'  */
  /* variable.                                                 */

  /* in non re-entrant builds, 'org_stream' is simply copied   */
  /* to 'stream'. Otherwise, the latter is a duplicate handle  */
  /* for the file opened with 'org_stream'                     */

  TT_Error  TT_Done_Stream( TT_Stream*  stream );
  /* Notice the component that we don't need to perform file */
  /* operations on the stream 'stream' anymore. This must be */
  /* used with streams "opened" with TT_Use_Stream only !    */

  /* in re-entrant builds, this will really discard the stream */

#ifdef TT_CONFIG_REENTRANT
  #define STREAM_ARGS   TT_Stream  stream,
  #define STREAM_ARG    TT_Stream  stream

#else
  #define STREAM_ARGS   /* void */
  #define STREAM_ARG

#endif /* REENTRANT */

  /* The macro STREAM_OPS lets us build a thread-safe or re-entrant  */
  /* implementation depending on a single configuration define     */

  /****************************************************************/
  /*                                                              */
  /*  File Functions.                                             */
  /*                                                              */
  /*  the following functions perform file operations on the      */
  /*  currently 'used' stream. In thread-safe builds, only one    */
  /*  stream can be used at a time. Synchronisation is performed  */
  /*  through the Use_Stream/Done_Stream functions.               */
  /*                                                              */
  /****************************************************************/

  TT_Error  TT_Read_File( STREAM_ARGS void*  buffer, 
                                      long   count );
  /* Read 'count' bytes from file into 'buffer' */

  TT_Error  TT_Seek_File( STREAM_ARGS long  position );
  /* Seek file cursor to a given position */

  TT_Error  TT_Skip_File( STREAM_ARGS long  distance );
  /* Skip the next 'distance' bytes in file */

  TT_Error  TT_Read_At_File( STREAM_ARGS long   position,
                                         void*  buffer,
                                         long   count );
  /* Read the 'count' bytes at 'position' into 'buffer' */

  Long  TT_File_Pos( STREAM_ARG );
  /* Return current file position */

  /********************************************************************/
  /*                                                                  */
  /*  Frame operations.                                               */
  /*                                                                  */
  /*  For a comprehensive explanation of frames, please refer to the  */
  /*  documentation file 'macros.txt'                                 */
  /*                                                                  */
  /********************************************************************/

  /* Frame type declaration.*/

  typedef struct _TFileFrame
  {
    unsigned char*  address;  /* frame buffer                     */
    unsigned char*  cursor;   /* current cursor position in frame */
    int             size;     /* frame size                       */
  } TFileFrame;

  extern const TFileFrame  Null_FileFrame;

  #define ZERO_Frame(frame) \
      { \
        (frame).address = NULL; \
        (frame).cursor  = NULL; \
        (frame).size    = 0;    \
      }

  /* The macro CONSTRUCT_FRAME is used to define and init a frame  */
  /* It is important to have an default frame of { NULL, NULL, 0 } */
  /* before a call to TT_Access_Frame. Otherwise, the call will    */
  /* fail with a TT_Err_Nested_Frame_Accesses error                */

#ifdef TT_CONFIG_REENTRANT
  #define FRAME_ARGS   TFileFrame* frame,
  #define FRAME_ARG    TFileFrame* frame

#else
  #define FRAME_ARGS  /* void */
  #define FRAME_ARG   /* void */

#endif /* REENTRANT */

  /* The macro FRAME_OPS lets us build a thread-safe or re-entrant */
  /* implementation depending on a single configuration define     */

  TT_Error  TT_Access_Frame( STREAM_ARGS FRAME_ARGS int size );
  /* Access the next 'size' bytes from current position */
  /* Fails if all bytes cannot be read/accessed         */

  TT_Error  TT_Check_And_Access_Frame( STREAM_ARGS FRAME_ARGS int size );
  /* Access the bytes located in the next 'size' bytes of the file */
  /* Doesn't fail if less than 'size' bytes are accessible ( like  */
  /* at the end of the file )                                      */

  TT_Error  TT_Forget_Frame( FRAME_ARG );
  /* Forget frame */

  unsigned char   Get_Byte  ( FRAME_ARG );
  char            Get_Char  ( FRAME_ARG );
  short           Get_Short ( FRAME_ARG );
  unsigned short  Get_UShort( FRAME_ARG );
  long            Get_Long  ( FRAME_ARG );
  unsigned long   Get_ULong ( FRAME_ARG );

  #ifdef __cplusplus
  }
  #endif

#endif /* TTFILE_H */


/* End */
