/*******************************************************************
 *
 *  ttapi.c    
 *
 *    High-level interface implementation
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Notes:
 *
 *    This file implements most of the features of the high-level
 *    interface. Note however that the character mapping tables
 *    functions ( TT_Get_CharMap and TT_Char_Index ) are all located
 *    in the 'ttindex.c' source file.
 *
 ******************************************************************/

#include "freetype.h"
#include "ttcommon.h"
#include "ttengine.h"
#include "ttcalc.h"
#include "ttmemory.h"
#include "ttcache.h"
#include "ttfile.h"
#include "ttobjs.h"
#include "ttload.h"
#include "ttgload.h"
#include "ttraster.h"

#ifdef TT_EXTEND_ENGINE
#include "extend/ttextend.h"
#endif

  #ifndef TT_CONFIG_REENTRANT
    TEngine_Instance  engine;
  #endif

  #ifdef TT_STATIC_RASTER

    #define  RAS_OPS   /* void */
    #define  RAS_OP    /* void */

  #else

    #define  RAS_OPS   ((TRaster_Instance*)engine.raster_component),
    #define  RAS_OP    ((TRaster_Instance*)engine.raster_component)

  #endif /* TT_STATIC_RASTER */
  


  #define RENDER_Glyph(glyph,target) \
            Render_Glyph( RAS_OPS  glyph, target )

  #define RENDER_Gray_Glyph(glyph,target,palette) \
            Render_Gray_Glyph( RAS_OPS  glyph, target, palette )

  #define SET_High_Precision( high ) \
            Set_High_Precision( RAS_OPS  high )

  #define SET_Second_Pass( pass )  \
            Set_Second_Pass( RAS_OPS  pass )

/*******************************************************************
 *
 *  Function    : TT_Init_FreeType
 *
 *  Description : The library's engine initializer. This function
 *                must be called prior to any call.
 *
 *  Input  :  None
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Init_FreeType()
  {
    TT_Error  error;
    int       n;

    /* Initalize components */
    if ( (error = TTMemory_Init())  ||
         (error = TTLists_Init ())  ||
         (error = TTFile_Init  ())  ||
         (error = TTObjs_Init  ())  ||
         (error = TTRaster_Init())  )
      return error;

    /* set the gray palette defaults : 0 to 4 */
    for ( n = 0; n < 5; n++ )
       engine.raster_palette[n] = n;

    /* Initialize extensions component if needed */
    #ifdef TT_EXTEND_ENGINE
    if ( (error = TTExtension_Init()) )
      return error;
    #endif

    /* create the engine lock */
    MUTEX_Create( engine.lock );

    SET_Second_Pass(1);

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    : TT_Done_FreeType
 *
 *  Description : The library's engine finalizer. This function
 *                will discard all active face and glyph objects
 *                from the heap.
 *
 *  Input  :  None
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Done_FreeType()
  {
    MUTEX_Destroy( engine.lock );

    #ifdef TT_EXTEND_ENGINE
    TTExtension_Done();
    #endif

    TTRaster_Done();
    TTObjs_Done  ();
    TTFile_Done  ();
    TTLists_Done ();
    TTMemory_Done();

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Set_Raster_Gray_Palette
 *
 *  Description :  set the gray-levels palette used for font
 *                 smoothing.
 *
 *  Input  :  palette   address of palette ( a 5 bytes array )
 *
 *  Output :  invalid argument if 'palette' is NULL
 *
 ******************************************************************/

  TT_Error  TT_Set_Raster_Gray_Palette( char* palette )
  {  
    int i;

    if (!palette)
      return TT_Err_Invalid_Argument;

    for (i = 0; i < 5; i++)
      engine.raster_palette[i] = palette[i];

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Open_Face
 *
 *  Description :  Create a new face object from a given font file
 *
 *  Input  :  fontpathname    the font file's pathname
 *            face            adress of returned face handle
 *
 *  Output :  Error code.
 *
 *  Note : the face handle is set to NULL in case of failure.
 *
 ******************************************************************/

  TT_Error  TT_Open_Face( const char*  fontpathname,
                          TT_Face*     face )
  {
    TFont_Input  input;
    TT_Error     error;
    TT_Stream    stream;
    PFace        _face;

    /* open the file */
    error = TT_Open_Stream( fontpathname, &stream );
    if (error)
      return error;

    input.stream    = stream;
    input.fontIndex = 0;

    /* Create and load the new face object */
    error = CACHE_New( engine.objs_face_cache,
                       _face,
                       &input );

    /* Set the handle */
    HANDLE_Set( *face, _face );

    if (error)
      goto Fail;

    return TT_Err_Ok;

  Fail:
    TT_Close_Stream( &stream );
    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Open_Collection
 *
 *  Description :  Create a new face object from a given font file
 *
 *  Input  :  fontpathname    the font file's pathname
 *            face            adress of returned face handle
 *
 *  Output :  Error code.
 *
 *  Note : the face handle is set to NULL in case of failure.
 *
 ******************************************************************/

  TT_Error  TT_Open_Collection( const char*  collectionpathname,
                                int          fontIndex,
                                TT_Face*     face )
  {
    TFont_Input  input;
    TT_Error     error;
    TT_Stream    stream;
    PFace        _face;

    /* open the file */
    error = TT_Open_Stream( collectionpathname, &stream );
    if (error)
      return error;

    input.stream    = stream;
    input.fontIndex = fontIndex;

    /* Create and load the new face object */
    error = CACHE_New( engine.objs_face_cache,
                       _face,
                       &input );

    /* Set the handle */
    HANDLE_Set( *face, _face );

    if (error)
      goto Fail;

    return TT_Err_Ok;

  Fail:
    TT_Close_Stream( &stream );
    return error;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Face_Properties
 *
 *  Description :  Return face properties
 *
 *  Input  :  face        the face handle
 *            properties  address of target properties record
 *
 *  Output :  Error code.
 *
 *  Note : Currently, max_Faces is always set to 0
 *
 ******************************************************************/

  TT_Error  TT_Get_Face_Properties( TT_Face              face,
                                    TT_Face_Properties*  properties )
  {
    PFace _face = HANDLE_Face(face);

    if (!_face)
      return TT_Err_Invalid_Face_Handle;

    properties->num_Glyphs   = _face->numGlyphs;
    properties->max_Points   = _face->maxPoints;
    properties->max_Contours = _face->maxContours;

    properties->max_Faces    = 0;

    properties->header     = &_face->fontHeader;
    properties->horizontal = &_face->horizontalHeader;
    properties->os2        = &_face->os2;
    properties->postscript = &_face->postscript;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Close_Face
 *
 *  Description :  Closing an opened face object. This function
 *                 will destroyed all objects associated to the
 *                 face, except the glyphs.
 *
 *  Input  :  face    the given face handle
 *
 *  Output :  Error code.
 *
 *  NOTE   :  the handle is set to NULL on exit
 *
 ******************************************************************/

  TT_Error  TT_Close_Face( TT_Face  face )
  {
    PFace  _face = HANDLE_Face(face);

    if (!_face)
      return TT_Err_Invalid_Face_Handle;

    return CACHE_Done( engine.objs_face_cache, _face );
  }

/*******************************************************************
 *
 *  Function    :  TT_New_Instance
 *
 *  Description :  Create a new instance from a given face
 *
 *  Input  :  face        parent face handle
 *            instance    address of resimt instance handle
 *            resolution  device resolution in dpi
 *            pointsize   point size
 *
 *  Output :  Error code.
 *
 *  Note   :  the handle is set to NULL in case of failure
 *
 ******************************************************************/

  TT_Error  TT_New_Instance( TT_Face       face,
                             TT_Instance*  instance )
  {
    TT_Error   error;
    PFace      _face = HANDLE_Face(face);
    PInstance  _ins;

    if (!_face)
      return TT_Err_Invalid_Face_Handle;

    /* get a new instance from the face's cache */
    error = CACHE_New( &_face->instances, _ins, _face );

    HANDLE_Set( *instance, _ins );

    if (!error)
    {
      error = Instance_Init( _ins );
      if (error)
      {
        HANDLE_Set( *instance, NULL );
        CACHE_Done( &_face->instances, _ins );
      }
    }

    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_Resolution
 *
 *  Description :  Resets an instance to new device resolutions
 *
 *  Input  :  instance      the instance handle
 *            x_resolution  new horizontal device resolution in dpi
 *            y_resolution  new vertical device resolution in dpi
 *
 *  Output :  Error code.
 *
 *  NOTE : y_resolution is currently ignored
 *
 ******************************************************************/

  TT_Error  TT_Set_Instance_Resolution( TT_Instance  instance,
                                        int          x_resolution,
                                        int          y_resolution )
  {
    PInstance  ins = HANDLE_Instance(instance);

    if (!ins)
      return TT_Err_Invalid_Instance_Handle;

    ins->metrics.x_resolution = x_resolution;
    ins->metrics.y_resolution = y_resolution;
    ins->valid                = FALSE;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_PointSize
 *
 *  Description :  Resets an instance to new pointsize
 *
 *  Input  :  instance      the instance handle
 *            pointsize     new pointsize
 *
 *  Output :  Error code.
 *
 *  NOTE : y_resolution is currently ignored
 *
 ******************************************************************/

  TT_Error  TT_Set_Instance_PointSize( TT_Instance  instance,
                                       int          pointsize )
  {
    PInstance  ins = HANDLE_Instance(instance);

    if (!ins)
      return TT_Err_Invalid_Instance_Handle;

    ins->metrics.pointSize = pointsize * 64;
    ins->valid             = FALSE;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Set_Instance_Transforms
 *
 *  Description :  Notice the interpreter about the transforms
 *                 that will be applied to the rendered glyphs
 *
 *  Input  :  instance      the instance handle
 *            rotated       set to TRUE if the glyph are rotated
 *            stretched     set to TRUE if the glyph are stretched
 *
 *  Output :  Error code.
 *
 *  NOTE : y_resolution is currently ignored
 *
 ******************************************************************/

  TT_Error  TT_Set_Instance_Transforms( TT_Instance instance,
					int         rotated,
					int         stretched )
  {
    PInstance  ins = HANDLE_Instance(instance);

    if (!ins)
      return TT_Err_Invalid_Instance_Handle;

    ins->metrics.rotated   = rotated;
    ins->metrics.stretched = stretched;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_Instance_Metrics
 *
 *  Description :  returns instance metrics
 *
 *  Input  :  instance   the instance handle.
 *            metrics    address of target instance metrics record
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Get_Instance_Metrics( TT_Instance           instance,
                                     TT_Instance_Metrics*  metrics )
  {
    PInstance  ins = HANDLE_Instance(instance);

    if (!ins)
     return TT_Err_Invalid_Instance_Handle;

    if (!ins->valid)
      Instance_Reset( ins, FALSE );

    metrics->pointSize    = ins->metrics.pointSize;
    metrics->x_resolution = ins->metrics.x_resolution;
    metrics->y_resolution = ins->metrics.y_resolution;
    metrics->x_ppem       = ins->metrics.x_ppem;
    metrics->y_ppem       = ins->metrics.y_ppem;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Done_Instance
 *
 *  Description :  Closes a given instance
 *
 *  Input  :  instance   address of instance handle.
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Done_Instance( TT_Instance  instance )
  {
    PInstance  ins = HANDLE_Instance(instance);

    if (!ins)
      return TT_Err_Invalid_Instance_Handle;

    return CACHE_Done( &ins->owner->instances, ins );
  }

/*******************************************************************
 *
 *  Function    :  TT_New_Glyph
 *
 *  Description :  Creates a new glyph object related to a given
 *                 face.
 *
 *  Input  :  face       the face handle
 *            glyph      address of target glyph handle
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_New_Glyph( TT_Face   face,
                          TT_Glyph* glyph )
  {
    TT_Error  error;
    PFace     _face = HANDLE_Face(face);
    PGlyph    _glyph;

    if (!_face)
      return TT_Err_Invalid_Face_Handle;

    error = CACHE_New( engine.objs_glyph_cache, _glyph, _face );

    HANDLE_Set( *glyph, _glyph );

    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Done_Glyph
 *
 *  Description :  Destroys a given glyph object
 *
 *  Input  :  glyph  the glyph handle
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Done_Glyph( TT_Glyph  glyph )
  {
    PGlyph  _glyph = HANDLE_Glyph(glyph);

    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    return CACHE_Done( engine.objs_glyph_cache, _glyph );
  }

/*******************************************************************
 *
 *  Function    :  TT_Load_Glyph
 *
 *  Description :  Load a glyph
 *
 *  Input  :  None
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Load_Glyph( TT_Instance  instance,
                           TT_Glyph     glyph,
                           int          glyph_index,
                           int          load_flags   )
  {
    PInstance  _ins;
    PGlyph     _glyph;
    TT_Error   error;

    _ins = HANDLE_Instance(instance);
    if (!_ins)
      return TT_Err_Invalid_Instance_Handle;

    _glyph = HANDLE_Glyph(glyph);
    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    if ( _ins->owner != _glyph->face )
      return TT_Err_Invalid_Face_Handle;

    if ( !_ins->valid )
    {
      error = Instance_Reset( _ins, FALSE );
      if (error)
        return error;
    }

    return Load_TrueType_Glyph( _ins, _glyph, glyph_index, load_flags );
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Outline
 *
 *  Description :
 *
 *  Input  :  None
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  void  Glyph2Outline( PGlyph  glyph, TT_Glyph_Outline*  outline )
  {
    outline->points   = glyph->num_points;
    outline->contours = glyph->num_contours;

    outline->xCoord    = glyph->x_coord;
    outline->yCoord    = glyph->y_coord;
    outline->flag      = glyph->touch;
    outline->conStarts = glyph->endContours;
  }

  TT_Error  TT_Get_Glyph_Outline( TT_Glyph           glyph,
                                  TT_Glyph_Outline*  outline )
  {
    PGlyph  _glyph = HANDLE_Glyph(glyph);

    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    Glyph2Outline( _glyph, outline );

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Metrics
 *
 *  Description :
 *
 *  Input  :  None
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Get_Glyph_Metrics( TT_Glyph           glyph,
                                  TT_Glyph_Metrics*  metrics )
  {
    PGlyph  _glyph = HANDLE_Glyph(glyph);

    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    metrics->xMin = _glyph->xMin;
    metrics->xMax = _glyph->xMax;
    metrics->yMin = _glyph->yMin;
    metrics->yMax = _glyph->yMax;

    metrics->advanceWidth    = _glyph->advanceWidth;
    metrics->leftSideBearing = _glyph->leftSideBearing;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Bitmap
 *
 *  Description :  Produce a bitmap from a glyph outline.
 *
 *  Input  :  glyph      the glyph container's handle
 *            map        target pixmap description block
 *            x_offset   x_offset in fractional pixels ( 26.6 format )
 *            y_offset   y_offset in fractional pixels ( 26.6 format )
 *
 *  Output :  Error code.
 *
 *  Note : Only use integer pixel offsets if you want to preserve
 *         the fine hints applied to the outline. This means that
 *         x_offset and y_offset must be multiples of 64 !
 *
 ******************************************************************/

  TT_Error  TT_Get_Glyph_Bitmap( TT_Glyph        glyph,
                                 TT_Raster_Map*  map,
                                 TT_F26Dot6      x_offset,
                                 TT_F26Dot6      y_offset )
  {
    TT_Error  error;
    PGlyph    _glyph = HANDLE_Glyph(glyph);
    Int       i;

    TT_Glyph_Outline  outline;

    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    x_offset -= 32;
    y_offset -= 32;

    if (x_offset)
      for ( i = 0; i < _glyph->num_points; i++ )
        _glyph->x_coord[i] += x_offset;

    if (y_offset)
      for ( i = 0; i < _glyph->num_points; i++ )
        _glyph->y_coord[i] += y_offset;

    Glyph2Outline( _glyph, &outline );

    SET_High_Precision( _glyph->high_precision );

    /*  outline.dropout_mode = _glyph->scan_type; */
    outline.dropout_mode = 2;

    error = RENDER_Glyph( &outline, map );

    if (x_offset)
      for ( i = 0; i < _glyph->num_points; i++ )
        _glyph->x_coord[i] -= x_offset;

    if (y_offset)
      for ( i = 0; i < _glyph->num_points; i++ )
        _glyph->y_coord[i] -= y_offset;

    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_Glyph_Pixmap
 *
 *  Description :  Produce a grayscaled pixmap from a glyph
 *                 outline.
 *
 *  Input  :  glyph      the glyph container's handle
 *            map        target pixmap description block
 *            x_offset   x_offset in fractional pixels ( 26.6 format )
 *            y_offset   y_offset in fractional pixels ( 26.6 format )
 *
 *  Output :  Error code.
 *
 *  Note : Only use integer pixel offsets to preserve the fine
 *         hinting of the glyph, and the 'correct' anti-aliasing
 *         ( where vertical and horizontal stems aren't grayed ).
 *         This means that x_offset and y_offset must be multiples
 *         of 64 !
 *
 *         You can experiment with offsets of +32 to get 'blurred'
 *         versions of the glyphs (a nice effect at large sizes that
 *         some graphists may appreciate :)
 *
 ******************************************************************/

  TT_Error  TT_Get_Glyph_Pixmap( TT_Glyph        glyph,
                                 TT_Raster_Map*  map,
                                 TT_F26Dot6      x_offset,
                                 TT_F26Dot6      y_offset )
  {
    TT_Error  error;
    PGlyph    _glyph = HANDLE_Glyph(glyph);
    Int       i;

    TT_Glyph_Outline  outline;

    if (!_glyph)
      return TT_Err_Invalid_Glyph_Handle;

    x_offset -= 16;
    y_offset -= 16;

    for ( i = 0; i < _glyph->num_points; i++ )
      _glyph->x_coord[i] = (_glyph->x_coord[i] + x_offset) << 1;

    for ( i = 0; i < _glyph->num_points; i++ )
      _glyph->y_coord[i] = (_glyph->y_coord[i] + y_offset) << 1;

    Glyph2Outline( _glyph, &outline );

    SET_High_Precision( _glyph->high_precision );

    /* outline.dropout_mode = _glyph->scan_type; */
    outline.dropout_mode = 2;

    error = RENDER_Gray_Glyph( &outline, 
                               map, 
                               engine.raster_palette );

    for ( i = 0; i < _glyph->num_points; i++ )
      _glyph->x_coord[i] = (_glyph->x_coord[i] >> 1) - x_offset;

    for ( i = 0; i < _glyph->num_points; i++ )
      _glyph->y_coord[i] = (_glyph->y_coord[i] >> 1) - y_offset;

    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Apply_Outline_Matrix
 *
 *  Description :  Apply a simple transform to an outline
 *
 *  Input  :  outline     the glyph's outline. Can be extracted
 *                        from a glyph container through TT_Get_Glyph_Outline
 *
 *            matrix      simple matrix with 16.16 fixed floats
 *
 *  Output :  Error code. (always TT_Err_Ok)
 *
 ******************************************************************/

  TT_Error  TT_Apply_Outline_Matrix( TT_Glyph_Outline*  outline,
                                     TT_Matrix*         matrix )
  {
    Int n;
    TT_F26Dot6  x, y;

    for ( n = 0; n < outline->points; n++ )
    {
      x = MulDiv_Round( outline->xCoord[n], matrix->xx, 0x10000 ) +
	  MulDiv_Round( outline->yCoord[n], matrix->xy, 0x10000 );

      y = MulDiv_Round( outline->xCoord[n], matrix->yx, 0x10000 ) +
	  MulDiv_Round( outline->yCoord[n], matrix->yy, 0x10000 );

      outline->xCoord[n] = x;
      outline->yCoord[n] = y;
    }

    return TT_Err_Ok;
  }  


/*******************************************************************
 *
 *  Function    :  TT_Apply_Outline_Translation
 *
 *  Description :  Apply a simple translation
 *
 *  Input  :  outline   no comment :)
 *            x_offset
 *            y_offset
 *
 *  Output :  Error code.
 *
 ******************************************************************/

  TT_Error  TT_Apply_Outline_Translation( TT_Glyph_Outline*  outline,
                                          TT_F26Dot6         x_offset,
                                          TT_F26Dot6         y_offset )
  {
    Int n;

    for ( n = 0; n < outline->points; n++ )
    {
      outline->xCoord[n] += x_offset;
      outline->yCoord[n] += y_offset;
    }

    return TT_Err_Ok;
  }


  /* ----------------- character mappings support ------------- */

/*******************************************************************
 *
 *  Function    :  TT_Get_CgharMap_Count
 *
 *  Description :  returns the number of charmaps in a given face
 *
 *  Input  :  face   face object handle
 *
 *  Output :  number of tables. -1 in case of error (bad handle)
 *
 ******************************************************************/

  int  TT_Get_CharMap_Count( TT_Face  face )
  {
    PFace  faze = HANDLE_Face(face);

    if (!faze)
      return -1;

    return faze->numCMaps;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_CharMap_ID
 *
 *  Description :  Returns the ID of a given charmap
 *
 *  Input  :  face             face object handle
 *            charmapIndex     index of charmap in directory
 *            platformID       address of returned platform ID
 *            encodingID       address of returned encoding ID
 *
 *  Output :  error code
 *
 *  Notes  :
 *
 ******************************************************************/

  TT_Error   TT_Get_CharMap_ID( TT_Face  face,
                                int      charmapIndex,
                                short*   platformID,
                                short*   encodingID )
  {
    PCMapTable  cmap;
    PFace       faze = HANDLE_Face(face);

    if (!faze)
      return TT_Err_Invalid_Face_Handle;

    if (charmapIndex < 0 || charmapIndex >= faze->numCMaps )
      return TT_Err_Bad_Argument;

    cmap = faze->cMaps + charmapIndex;

    *platformID = cmap->platformID;
    *encodingID = cmap->platformEncodingID;

    return TT_Err_Ok;
  }

/*******************************************************************
 *
 *  Function    :  TT_Get_CharMap
 *
 *  Description :  Look-up for a charmap
 *
 *  Input  :  face          face object handle
 *            charmapIndex  index of charmap in directory
 *            charMap       address of returned charmap handle
 *
 *  Output :  error code
 *
 ******************************************************************/

  TT_Error  TT_Get_CharMap( TT_Face      face,
                            int          charmapIndex,
                            TT_CharMap*  charMap )
  {
    TT_Error    error;
    TT_Stream   stream;
    PCMapTable  cmap;
    PFace       faze = HANDLE_Face(face);

    if (!faze)
      return TT_Err_Invalid_Face_Handle;

    if (charmapIndex < 0 || charmapIndex >= faze->numCMaps)
      return TT_Err_Bad_Argument;

    cmap = faze->cMaps + charmapIndex;

    /* Load table if needed */

    error = TT_Err_Ok;

    if (!cmap->loaded)
    {
      USE_Stream( faze->stream, stream );
      if (!error)
      {
	error = CharMap_Load( cmap, stream );
	TT_Done_Stream( &stream );
      }

      if (error)
	cmap = NULL;
      else
	cmap->loaded = TRUE;
    }

    HANDLE_Set( *charMap, cmap );
    return error;
  }

/*******************************************************************
 *
 *  Function    :  TT_Char_Index
 *
 *  Description :  returns the glyph index corresponding to
 *                 a given charcode defined for the 'charmap'
 *
 *  Input  :  charMap    charmap handle
 *            charcode   character code in locale
 *
 *  Output :  glyph index.
 *
 *  Notes  :  a 0 is the unknown glyph, which should never be
 *            displayed.
 *
 ******************************************************************/

  int  TT_Char_Index( TT_CharMap      charMap,
                      unsigned short  charCode )
  {
    PCMapTable  cmap = HANDLE_CharMap(charMap);

    if (!cmap)
      return TT_Err_Invalid_CharMap_Handle;

    return CharMap_Index( cmap, charCode );
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Name_Count
 *
 *  Description :  Return the number of strings found in the
 *                 name table.
 *
 *  Input  :  face   face handle
 *
 *  Output :  number of strings
 *
 *  Notes  :  Returns -1 on error (invalid handle)
 *
 ******************************************************************/

  int  TT_Get_Name_Count( TT_Face  face )
  {
    PFace  faze = HANDLE_Face(face);

    if (!faze)
      return TT_Err_Invalid_Face_Handle;

    return faze->nameTable.numNameRecords;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Name_ID
 *
 *  Description :  Returns the IDs of the string number 'nameIndex'
 *                 in the name table of a given face
 *
 *  Input  :  face        face handle
 *            nameIndex   index of string. First is 0
 *            platformID  addresses of returned IDs
 *            encodingID
 *            languageID
 *            nameID
 *
 *  Output :  Error code
 *
 *  Notes  :  Some files have a corrupt name table, some of their
 *            entries having a platformID > 3. These should be
 *            ignored by a client application.
 *
 ******************************************************************/

  TT_Error  TT_Get_Name_ID( TT_Face  face,
                            int      nameIndex,
                            short*   platformID,
                            short*   encodingID,
                            short*   languageID,
                            short*   nameID )
  {
    TNameRec*  namerec;
    PFace      faze = HANDLE_Face(face);

    if (!faze)
      return TT_Err_Invalid_Face_Handle;

    if (nameIndex < 0 || nameIndex >= faze->nameTable.numNameRecords)
      return TT_Err_Bad_Argument;

    namerec = faze->nameTable.names + nameIndex;

    *platformID = namerec->platformID;
    *encodingID = namerec->encodingID;
    *languageID = namerec->languageID;
    *nameID     = namerec->nameID;

    return TT_Err_Ok;
  }


/*******************************************************************
 *
 *  Function    :  TT_Get_Name_String
 *
 *  Description :  Returns the address and length of a given
 *                 string found in the name table
 *
 *  Input  :  face        face handle
 *            nameIndex   string index
 *            stringPtr   address of returned pointer to string
 *            length      address of returned string length
 *
 *  Output :  Error code
 *
 *  Notes  :  If the string's platformID is > 3 (which denotes
 *            an invalid entry in the name table), the returned
 *            pointer is NULL, and the length is 0.
 *
 ******************************************************************/

  TT_Error  TT_Get_Name_String( TT_Face  face,
                                int      nameIndex,
                                char**   stringPtr,
                                int*     length )
  {
    TNameRec*  namerec;
    PFace      faze = HANDLE_Face(face);

    if (!faze)
      return TT_Err_Invalid_Face_Handle;

    if (nameIndex < 0 || nameIndex >= faze->nameTable.numNameRecords)
      return TT_Err_Bad_Argument;

    namerec = faze->nameTable.names + nameIndex;

    *stringPtr = (char*)namerec->string;
    *length    = namerec->stringLength;

    return TT_Err_Ok;
  }


/* END */
