(*******************************************************************
 *
 *  TTIndex.pas                                                 1.0
 *
 *    TrueType index and cmap handling
 *
 *  Copyright 1996 David Turner, Robert Wilhelm and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************)

unit TTIndex;

interface

uses FreeType,
     TTTypes,
     TTTables;  (* TTables must be used in the interface as long       *)
                (* as the PInstanceRecord is not moved to FreeType.pas *)

  function TT_Select_CMap( aPlatformID         : UShort;
                           aPlatformEncodingID : UShort;
                           aInstance           : PInstanceRecord

                         ) : TT_PCMapTable;

  function TT_Code_To_Index( charCode  : UShort;
                             cmapIndex : TT_PCMapTable;
                             aInstance : PInstanceRecord ) : Int;

implementation

(*******************************************************************
 *
 *  Function    : code_to_index0
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 0.
 *                charCode will be masked to get a value in the range
 *                0x00-0xFF.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap0         a pointer to a cmap table in format 0
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************)

 function Code_To_Index_0( charCode : UShort;
                           cmap0    : PCMap0Table ) : Int;
 begin
   Code_To_Index_0 := cmap0^.glyphIdArray^[charCode and $FF];
 end;

(*******************************************************************
 *
 *  Function    : code_to_index2
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 2.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap2         a pointer to a cmap table in format 2
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************)

 function Code_To_Index_2( charCode : UShort;
                           cmap2    : PCMap2Table ) : Int;
 var
   index1 : UShort;
 begin
   Code_To_Index_2 := 0;

   if charCode < 256 then index1 := charCode
                     else index1 := charCode shr 8;

   index1 := cmap2^.header.subHeaderKeys[index1];

   if index1 = 0 then

     if charCode < 256
       then Code_To_Index_2 := cmap2^.glyphIdArray^[charCode]
     else

   else

     if charCode > 255 then with cmap2^.subHeaders^[index1] do

       if ( (charCode and $FF) >= firstCode ) and
          ( (charCode and $FF) <  firstCode + entryCount ) then

       Code_To_Index_2 := cmap2^.glyphIdArray^[
                             idRangeOffset div 2 +
                             ( charCode and $FF ) -
                             firstCode + idDelta ] and $FFFF;
 end;

(*******************************************************************
 *
 *  Function    : code_to_index4
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 4.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap4         a pointer to a cmap table in format 4
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************)

 function Code_To_Index_4( charCode : UShort;
                           cmap4    : PCMap4Table ) : Int;
 var
   index1, segCount : UShort;
   i                : Int;
 label
   break;
 begin
   Code_To_Index_4 := 0;

   segCount := cmap4^.header.segCountX2 div 2;

   i := 0;
   while ( i < segCount ) and
         ( charCode > cmap4^.segments^[i].endCount ) do
     inc( i );

   with cmap4^.Segments^[i] do
   begin
     if charCode < startCount then exit;

     if idRangeOffset = 0 then
       Code_To_Index_4 := ( charCode + idDelta ) and $FFFF
     else
     begin
       index1 := idRangeOffset div 2 +
                ( charCode - startCount ) -
                ( segCount - i );

       if cmap4^.glyphIdArray^[index1] = 0 then exit;

       Code_To_Index_4 := ( cmap4^.glyphIdArray^[index1] + idDelta )
                          and $FFFF;
     end;

   end;
 end;

(*******************************************************************
 *
 *  Function    : code_to_index6
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 6.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap6         a pointer to a cmap table in format 6
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist (`missing character glyph')
 *
 ******************************************************************)

 function Code_To_Index_6( charCode : UShort;
                           cmap6    : PCMap6Table ) : Int;
 var
   firstCode : UShort;
 begin
   Code_To_Index_6 := 0;

   with cmap6^ do
   begin

     if ( charCode < header.firstCode ) or
        ( charCode >= header.firstCode + header.entryCount ) then exit;

     Code_To_Index_6 := glyphIdArray^[charCode-header.firstCode];
   end;
 end;

(*******************************************************************
 *
 *  Function    : TT_select_CMap
 *
 *  Description : Select the CMap to use.
 *
 *  Input  :  platformID            the target platform
 *            platformEncodingID    the platform specific target
 *                                  encoding
 *            aInstance             the target glyph instance
 *
 *  Output :  NULL if map not found, TT_PCMapTable if map available.
 *
 ******************************************************************)

 function TT_Select_CMap( aPlatformID         : UShort;
                          aPlatformEncodingID : UShort;
                          aInstance           : PInstanceRecord

                        ) : TT_PCMapTable;
 var
   i : Int;
 begin

   with aInstance^.fontRes^ do

     for i := 0 to numCMaps-1 do
       with CMaps^[i] do

         if ( platformID = aPlatformID ) and
            ( platformEncodingID = aPlatformEncodingID ) then
           begin
             TT_Select_CMap := @CMaps^[i];
             exit;
           end;

   TT_Select_CMap := nil;
 end;

(*******************************************************************
 *
 *  Function    : TT_code_to_index
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses the CMap index returned from TT_select_CMap().
 *
 *  Input  :  charCode      the wanted character code
 *            cmapIndex     the selected cmap table
 *            aInstance     the target glyph instance
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist (`missing character glyph').
 *            -1 on failure.
 *
 ******************************************************************)

 function TT_Code_To_Index( charCode  : UShort;
                            cmapIndex : TT_PCMapTable;
                            aInstance : PInstanceRecord ) : Int;
 begin
   TT_Code_To_Index := -1;

   if cmapIndex = nil then exit;

   case PCMapTable(cmapIndex)^.Format of

     0 : TT_Code_To_Index := Code_To_Index_0
                              ( charCode, PCMapTable(cmapIndex)^.cmap0 );

     2 : TT_Code_To_Index := Code_To_Index_2
                              ( charCode, PCMapTable(cmapIndex)^.cmap2 );

     4 : TT_Code_To_Index := Code_To_Index_4
                              ( charCode, PCMapTable(cmapIndex)^.cmap4 );

     6 : TT_Code_To_Index := Code_To_Index_6
                              ( charCode, PCMapTable(cmapIndex)^.cmap6 );
   end;

 end;

end.
