/*******************************************************************
 *
 *  tttables.h                                                  1.1
 *
 *    TrueType Tables structures and handling (specification).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#ifndef TTTABLES_H
#define TTTABLES_H

#include "tttypes.h"

#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************/
/*                                                                         */
/*                      TrueType Table Types                               */
/*                                                                         */
/***************************************************************************/

/* Graphics State                            */
/*                                           */
/* The Graphics State (GS) is managed by the */
/* instruction field, but does not come from */
/* the font file. Thus, we can use 'int's    */
/* where needed.                             */

  typedef struct  _TGraphicsState
  {
    Bool        auto_flip;
    TT_F26Dot6  control_value_cutin;
    Int         delta_base;
    Int         delta_shift;

    TT_UnitVector  dualVector;
    TT_UnitVector  projVector;
    TT_UnitVector  freeVector;

    Int         gep0;
    Int         gep1;
    Int         gep2;

    Byte        instruct_control;
    Long        loop;

    TT_F26Dot6  minimum_distance;
    Int         round_state;

    Int         rp0;
    Int         rp1;
    Int         rp2;

    Bool        scan_control;
    TT_F26Dot6  single_width_cutin;
    TT_F26Dot6  single_width_value;

    Int         scan_type;
  } TGraphicsState;

  /* TrueType Table Directory type */

  typedef struct  _TTableDir
  {
    TT_Fixed  version;      /* should be 0x10000 */
    UShort    numTables;    /* number of tables  */

    UShort  searchRange;    /* These parameters are only used  */
    UShort  entrySelector;  /* for a dichotomy search in the   */
    UShort  rangeShift;     /* directory. We ignore them.      */
  } TTableDir;

  typedef TTableDir*  PTableDir;

  /* The 'TableDir' is followed by 'numTables' TableDirEntries */

  typedef struct  _TTableDirEntry
  {
    Long  Tag;        /*        table type */
    Long  CheckSum;   /*    table checksum */
    Long  Offset;     /* table file offset */
    Long  Length;     /*      table length */
  } TTableDirEntry;

  typedef TTableDirEntry*  PTableDirEntry;

  /* 'cmap' tables */

  typedef struct  _TCMapDir
  {
    UShort  tableVersionNumber;
    UShort  numCMaps;
  } TCMapDir;

  typedef TCMapDir*  PCMapDir;

  typedef struct  _TCMapDirEntry
  {
    UShort  platformID;
    UShort  platformEncodingID;
    Long    offset;
  } TCMapDirEntry;

  typedef TCMapDirEntry*  PCMapDirEntries;


  /* NOTE : The following types are not defined by the TrueType */
  /*        spec. However, they represent the layout of the     */
  /*        character mapping tables in memory. This could      */
  /*        easily change in future versions of the library.    */

  /* Apple standard character to glyph index mapping table. */
  /* The glyphIdArray for this format has 256 entries.      */
  /* There is no map header for this encoding type.         */

#if 0
  typedef struct  _TCMap0
  {
    ;
  } TCMap0;
#endif

  /* Format 2 is used for mixed 8/16bit encodings (usually CJK fonts). */

  typedef struct  _TCMap2
  {
    UShort subHeaderKeys[256];
    /* high byte mapping table            */
    /* value = subHeader index * 8        */
  } TCMap2;

  /* The format 2 table contains a variable-length array of subHeaders    */
  /* (at most 256 entries) whose size must be determined algorithmically. */

  typedef struct  _TCMap2SubHeader
  {
    UShort  firstCode;      /* first valid low byte         */
    UShort  entryCount;     /* number of valid low bytes    */
    Short   idDelta;        /* delta value to glyphIndex    */
    UShort  idRangeOffset;  /* offset from here to 1st code */
  } TCMap2SubHeader;

  typedef TCMap2SubHeader*  PCMap2SubHeader;

  /* Microsoft standard character to glyph index mapping table. */

  typedef struct  _TCMap4
  {
    UShort  segCountX2;     /* segments number * 2          */
    UShort  searchRange;    /* these parameters can be used */
    UShort  entrySelector;  /* for a binary search          */
    UShort  rangeShift;
  } TCMap4;

  typedef TCMap4*  PCMap4;

  /* The format 4 table contains segCount segments. */

  typedef struct  _TCMap4Segment
  {
    UShort  endCount;
    UShort  startCount;
    Short   idDelta;        /* in the specs defined as UShort but the
                               example there gives negative values... */
    UShort  idRangeOffset;
  } TCMap4Segment;

  typedef TCMap4Segment*  PCMap4Segment;

  /* Trimmed table mapping (for representing one subrange). */
  typedef struct  _TCMap6
  {
    UShort  firstCode;      /* first character code of subrange      */
    UShort  entryCount;     /* number of character codes in subrange */
  } TCMap6;


  typedef struct  _TCMap0Table
  {
    PByte  glyphIdArray;
  } TCMap0Table;

  typedef TCMap0Table*  PCMap0Table;

  typedef struct  _TCMap2Table
  {
    TCMap2           Header;
    PCMap2SubHeader  subHeaders;
    PUShort          glyphIdArray;
  } TCMap2Table;

  typedef TCMap2Table*  PCMap2Table;

  typedef struct _TCMap4Table
  {
    TCMap4         Header;
    PCMap4Segment  Segments;
    PUShort        glyphIdArray;
  } TCMap4Table;

  typedef TCMap4Table*  PCMap4Table;

  typedef struct  _TCMap6Table
  {
    TCMap6   Header;
    PUShort  glyphIdArray;
  } TCMap6Table;

  typedef TCMap6Table*  PCMap6Table;


  typedef struct  _TCMapTable
  {
    UShort  platformID;
    UShort  platformEncodingID;
    UShort  Format;
    UShort  Length;
    UShort  Version;

    union
    {
      PCMap0Table cmap0;
      PCMap2Table cmap2;
      PCMap4Table cmap4;
      PCMap6Table cmap6;
    } uc;
  } TCMapTable;

  typedef TCMapTable*  PCMapTable;

  /* 'maxp' Maximum Profiles table */

  typedef struct  _TMaxProfile
  {
    TT_Fixed  version;
    UShort    numGlyphs,
              maxPoints,
              maxContours,
              maxCompositePoints,
              maxCompositeContours,
              maxZones,
              maxTwilightPoints,
              maxStorage,
              maxFunctionDefs,
              maxInstructionDefs,
              maxStackElements,
              maxSizeOfInstructions,
              maxComponentElements,
              maxComponentDepth;
  } TMaxProfile;

  typedef TMaxProfile*  PMaxProfile;

  /* table "head" */

  typedef struct  _TLongDateTime
  {
    Long  l1;
    Long  l2;
  } TLongDateTime;

  typedef struct  _THeader
  {
    TT_Fixed  Table_Version;
    TT_Fixed  Font_Revision;

    Long      CheckSum_Adjust;
    Long      Magic_Number;

    UShort    Flags;
    UShort    Units_Per_EM;

    TLongDateTime  Created, Modified;

    Short     xMin;
    Short     yMin;
    Short     xMax;
    Short     yMax;

    UShort    Mac_Style;
    UShort    Lowest_Rec_PPEM;

    Short     Font_Direction;
    Short     Index_To_Loc_Format;
    Short     Glyph_Data_Format;
  } THeader;

  /* table "hhea" */

  typedef struct _THorizontalHeader
  {
    Fixed   Version;
    Short   Ascender;
    Short   Descender;
    Short   Line_Gap;

    UShort  advance_Width_Max;

    Short   min_Left_Side_Bearing;
    Short   min_Right_Side_Bearing;
    Short   xMax_Extent;
    Short   caret_Slope_Rise;
    Short   caret_Slope_Run;

    Short   Reserved[5];

    Short   metric_Data_Format;
    UShort  number_Of_HMetrics;
  } THorizontalHeader;

  /* table "HMTX" */

  typedef struct  _TLongHorMetric
  {
    UShort  advance_Width;
    Short   lsb;
  } TLongHorMetric;

  typedef TLongHorMetric*  PTableHorMetrics;

  /* 'loca' location table type */

  typedef struct _TLoca
  {
    UShort    Size;
    PStorage  Table;
  } TLoca;

  /* composite additions */

  typedef struct  _TGlyphContour
  {
    UShort  start;
    UShort  finish;
  } TGlyphContour;

  typedef TGlyphContour   TGlyphContours[];
  typedef TGlyphContour*  PGlyphContours;

  typedef struct  _TGlyphInsInfo
  {
    UShort  numIns;   /* number of instructions for subglyph */
    ULong   offset;   /* file offset to instruction stream */
  } TGlyphInstruct;

  typedef TGlyphInstruct*  PGlyphInstruct;

  typedef struct  _TGlyphScale
  {
    UShort      flags;
    Int         glyphSize;   /* glyph instruction size for this subglyph */
    UShort      glyphOffset; /* offset in file to instruction stream     */

    UShort      start,
                finish;

    TT_F26Dot6  x_offset,
                y_offset;
    TT_Fixed    x_scale,    /* TT_Fixed allows multiplication of scaling */
                scale01,    /* values without overflow.                  */
                scale10,
                y_scale;
  } TGlyphScale;

  typedef TGlyphScale   TGlyphScales[];
  typedef TGlyphScale*  PGlyphScale;
  typedef TGlyphScale*  PGlyphScales;

  typedef struct  _TGlyph
  {
    Short           numberOfContours;
    Short           numberOfPoints;
    Short           numberOfSubglyphs;
    Short           xMin;
    Short           yMin;
    Short           xMax;
    Short           yMax;
    PGlyphInstruct  instruct;
    PGlyphContours  contours;
    PGlyphScales    scales;
  } TGlyph;

  typedef TGlyph   TGlyphs[];
  typedef TGlyph*  PGlyph;
  typedef TGlyph*  PGlyphs;


/***************************************************************************/
/*                                                                         */
/*                  FreeType Resident table definitions                    */
/*                                                                         */
/***************************************************************************/

  typedef struct  _TResident_Record
  {
    Long               totalSize;
    TT_Stream          stream;

    Int                numTables;
    PTableDirEntry     dirTables;
    Int                numCMaps;
    PCMapTable         cMaps;

    TMaxProfile        maxProfile;

    THeader            fontHeader;
    Int                numGlyphs;
    Int                numLocations;
    PStorage           glyphLocations;

    THorizontalHeader  horizontalHeader;
    PTableHorMetrics   LongHMetrics;
    PShort             ShortMetrics;

    Int                fontPgmSize;
    PByte              fontProgram;

    Int                cvtPgmSize;
    PByte              cvtProgram;
    Int                cvtSize;
    PShort             cvt;
  } TResident_Record;

  typedef TResident_Record*  PResident_Record;


/***************************************************************************/
/*                                                                         */
/*                  FreeType Instance Types                                */
/*                                                                         */
/***************************************************************************/

#define MaxCodeRanges   3

  /* There can only be 3 active code ranges at once:   */
  /*   - the Font Program                              */
  /*   - the CVT Program                               */
  /*   - a glyph's instructions set                    */

  typedef struct  _TCodeRange
  {
    PByte  Base;
    Int    Size;
  } TCodeRange;

  typedef TCodeRange*  PCodeRange;

  /* Defintion of a code range                                       */
  /*                                                                 */
  /* Code ranges can be resident to a glyph (i.e. the Font Program)  */
  /* while some others are volatile (Glyph instructions).            */
  /* Tracking the state and presence of code ranges allows function  */
  /* and instruction definitions within a code range to be forgotten */
  /* when the range is discarded.                                    */

  typedef TCodeRange  TCodeRangeTable[MaxCodeRanges];

  /* defines a function/instruction definition record */

  typedef struct  _TDefRecord
  {
    Int   Range;      /* in which code range is it located ? */
    Int   Start;      /* where does it start ?               */
    Byte  Opc;        /* function #, or instruction code     */
    Bool  Active;     /* is it active ?                      */
  } TDefRecord;

  typedef TDefRecord*  PDefArray;

  /* defines a call record, used to manage function calls. */
  typedef struct  _TCallRecord
  {
    Int  Caller_Range;
    Int  Caller_IP;
    Int  Cur_Count;
    Int  Cur_Restart;
  } TCallRecord;

  /* defines a simple call stack */
  typedef TCallRecord*  PCallStack;

  typedef struct  _TInstance_Record
  {
    Long              totalSize;  /* total size of instance data */

    int               error;
    TT_Stream         stream;

    PResident_Record  fontRes;

    PGlyph            glyph;      /* the current glyph we are operating on */

    Int               hint_glyph; /* if TRUE use instructions */

    PByte             glyphIns;   /* glyph instructions */
    Int               glyphSize;  /* glyph ins. size    */

    Int               numFDefs;
    PDefArray         FDefs;
    Int               numIDefs;
    PDefArray         IDefs;

    /* Following two values should be cached with */
    /* the bitmap, they are here in FUnits.       */

    Int              leftSideBearing;
    Int              advanceWidth;

    Int              callTop;
    Int              callSize;
    PCallStack       callStack;

    TCodeRangeTable  codeRangeTable;

    Int              storeSize;  /* size of current storage area */
    PStorage         storage;    /* storage area                 */

    Int              stackSize;  /* size of instance stack */
    Int              top;        /* top of instance stack  */
    PStorage         stack;      /* current instance stack */

    TT_VecRecord     twilight;
    TT_VecRecord     pts;

    /* smack these back in here from glyph, we must consolidate SOON */

    Int              numContours;
    PUShort          endContours;

    Int              cvtSize;
    PShort           cvt;

    void*            exec;       /* void* to avoid circular deps */
  } TInstance_Record;

  typedef TInstance_Record*  PInstance_Record;


/* composite font flags */

#define ARGS_ARE_WORDS       0x001
#define ARGS_ARE_XY_VALUES   0x002
#define ROUND_XY_TO_GRID     0x004
#define WE_HAVE_A_SCALE      0x008
/* reserved                  0x010 */
#define MORE_COMPONENTS      0x020
#define WE_HAVE_AN_XY_SCALE  0x040
#define WE_HAVE_A_2X2        0x080
#define WE_HAVE_INSTR        0x100
#define USE_MY_METRICS       0x200

/* end composite */

/* The following functions are used by the interpreter */

#define TT_CodeRange_Font   1
#define TT_CodeRange_Cvt    2
#define TT_CodeRange_Glyph  3

  typedef struct  _TGlyphTransform
  {
    TT_Fixed    xx, xy;    /* transformation matrix */
    TT_Fixed    yx, yy;
    TT_F26Dot6  ox, oy;    /* offsets */
  } TGlyphTransform;

  typedef TGlyphTransform*  PGlyphTransform;


  Bool  Load_TrueType_Glyph( PInstance_Record  instance, 
                             Int               glyphIndex,
                             Bool              allow_components );

  Bool  Load_Glyph_Instructions( PInstance_Record  instance,
                                 Int               component );

  Bool  TT_Load_Resident_Table( TT_Stream          stream,
                                PResident_Record*  resident );

  Bool  TT_Load_Instance_Data( PResident_Record   resident,
                               PInstance_Record*  instance );

#if 0

  Bool  TT_Reset_Glyph( PInstance_Record  instance );

  Bool  TT_Prepare_Glyph( PInstance_Record  instance );

  Bool  TT_Reset_Instance( PInstance_Record  instance,
                           Int               ptSize,
                           Int               resolution );

#endif

  Int  TT_Transform_Glyph( PInstance_Record  instance );

#ifdef __cplusplus
}
#endif

#endif /* TTTABLES_H */


/* End */
