/*******************************************************************
 *
 *  tttables.c                                                  1.2
 *
 *    TrueType Tables structures and handling (body).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  Differences between 1.2 and 1.1
 *
 *  - Moved execution context ( coderange and instance ) functions
 *    in the TTExec component. Only the resident and instance loader
 *    were kept there.
 *
 *  - Re-introduced the ALLOC and ALLOC_ARRAY macros for readability
 *
 *  - Fixed the composite loader (a really bad bug)
 *    The loader should probably be rewritten in the future,
 *    its current implementation is a bit too cryptic..
 *
 *  - Added support for re-entrancy through macros.
 *    ( Equivalent to Joerg's changes )
 *
 ******************************************************************/

#include "tttypes.h"
#include "tttables.h"
#include "ttcalc.h"
#include "ttmemory.h"
#include "tterror.h"
#include "ttfile.h"
#include "tttags.h"
#include "ttexec.h"

#ifdef TT_CONFIG_REENTRANT  /* re-entrant implementation */

#define LOCK_Access( _stream_ )    TRUE     /* there is no locking for the */
#define RELEASE_Access( _stream_ ) \        /* re-entrant build            */
          do {} while( 0 )

/* the next three macros must be the last statements of the variable */
/* definition block to make the thread-safe macros versions working. */
#define DEFINE_RES_STREAM  TT_Stream  stream = resident->stream
#define DEFINE_STREAM      TT_Stream  stream = instance->stream
#define DEFINE_FRAME       CONSTRUCT_FRAME( frame )  /* see ttfile.h */

#define ACCESS_Frame( _size_ )  TT_Access_Frame( stream, &frame, _size_ )

#define CHECK_ACCESS_Frame( _size_ ) \
          TT_Check_And_Access_Frame( stream, &frame, _size_ )
#define FORGET_Frame()  TT_Forget_Frame( &frame )

#define GET_Byte()    Get_Byte  ( &frame )
#define GET_Char()    Get_Char  ( &frame )
#define GET_UShort()  Get_UShort( &frame )
#define GET_Short()   Get_Short ( &frame )
#define GET_Long()    Get_Long  ( &frame )
#define GET_ULong()   Get_ULong ( &frame )
#define GET_Tag4()    Get_Long  ( &frame )

#define FILE_Pos()               TT_File_Pos ( stream )
#define FILE_Seek( _position_ )  TT_Seek_File( stream, _position_ )
#define FILE_Skip( _distance_ )  TT_Skip_File( stream, _distance_ )

#define FILE_Read( buffer, count ) \
          TT_Read_File ( stream, buffer, count )
#define FILE_Read_At( pos, buffer, count ) \
          TT_Read_At_File( stream, pos, buffer, count )

#define ERROR  instance->error

#else   /* thread-safe implementation */

#define LOCK_Access( _stream_ )     TT_Lock_Access( _stream_ )
#define RELEASE_Access( _stream_ )  TT_Release_Access( _stream_ )

#define ACCESS_Frame( _size_ )  TT_Access_Frame( _size_ )
#define FORGET_Frame()          TT_Forget_Frame()

#define CHECK_ACCESS_Frame( _size_ )  TT_Check_And_Access_Frame( _size_ )

#define DEFINE_RES_STREAM  do {} while ( 0 )    /* void */
#define DEFINE_STREAM      do {} while ( 0 )    /* void */
#define DEFINE_FRAME       do {} while ( 0 )    /* void */

#define GET_Byte()    Get_Byte  ()
#define GET_Char()    Get_Char  ()
#define GET_UShort()  Get_UShort()
#define GET_Short()   Get_Short ()
#define GET_Long()    Get_Long  ()
#define GET_ULong()   Get_ULong ()
#define GET_Tag4()    Get_Long  ()

#define FILE_Pos()                TT_File_Pos ()
#define FILE_Seek( _position_ )   TT_Seek_File( _position_ )
#define FILE_Skip( _distance_ )   TT_Skip_File( _distance_ )

#define FILE_Read( buffer, count ) \
          TT_Read_File( buffer, count )
#define FILE_Read_At( pos, buffer, count ) \
          TT_Read_At_File( pos, buffer, count )

#define ERROR  Error  /* global error variable */

#endif /* REENTRANT */


  Bool  Load_Glyph( PInstance_Record  instance,
                    Int               glyphIndex,
                    PInt              curr_contour,
                    PInt              curr_point,
                    PInt              curr_subglyph,
                    Bool              allow_components );
  /* Note that the 'allow_components' is a temporary parameter      */
  /* used only for the viewer, while component hinting doesn't work */

  Bool  Load_Simple_Glyph( PInstance_Record  instance,
                           Int               glyphIndex,
                           Int               subglyph,
                           Int               szc,
                           Int               curr_contour,
                           PInt              curr_point );


/*******************************************************************
 *
 *  Function    :  LookUp_TrueType_Table
 *
 *  Description :  Looks for a TrueType table by name.
 *
 *  Input  :  resident   resident table to look for
 *            tag        searched tag
 *
 *  Output :  index of table if found, -1 otherwise.
 *
 ******************************************************************/

  static Int LookUp_TrueType_Table( PResident_Record  resident,
                                    Long              tag )
  {
    Int  i, limit;

  
    limit = resident->numTables;

    for ( i = 0; i < limit; i++ )
      if ( resident->dirTables[i].Tag == tag )
        return i;

    return -1;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_Directory
 *
 *  Description :  Loads the table directory into resident table.
 *
 *  Input  :  resident    resident record to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool Load_TrueType_Directory( PResident_Record resident )
  {
    long  n;

    TTableDir  tableDir;
    int        limit;

    DEFINE_RES_STREAM;  /* current stream from resident record */
    DEFINE_FRAME;       /* define a local frame record         */


    DebugTrace1( "Directory " );

    if ( !FILE_Seek( 0L ) )
      return FAILURE;

    if ( !ACCESS_Frame(12L) )
      return FAILURE;

    tableDir.version   = GET_Long();
    tableDir.numTables = GET_UShort();

    tableDir.searchRange   = GET_UShort();
    tableDir.entrySelector = GET_UShort();
    tableDir.rangeShift    = GET_UShort();

    DebugTrace2( "Tables number : %12u\n", tableDir.numTables );

    FORGET_Frame();

    resident->numTables = tableDir.numTables;

    if ( !ALLOC_ARRAY( resident->dirTables,
                       resident->numTables, 
                       TTableDirEntry ) )
      return FAILURE;

    if ( !ACCESS_Frame( resident->numTables * 16L ) )
      return FAILURE;

    limit = resident->numTables;
    for ( n = 0; n < limit; n++ )
    {                       /* loop through the tables and get all entries */
      resident->dirTables[n].Tag      = GET_Tag4();
      resident->dirTables[n].CheckSum = GET_ULong();
      resident->dirTables[n].Offset   = GET_Long();
      resident->dirTables[n].Length   = GET_Long();
    }

    FORGET_Frame();

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_MaxProfile
 *
 *  Description :  Loads the maxp table into resident table.
 *
 *  Input  :  resident     resident table to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_MaxProfile( PResident_Record  resident )
  {
    int  i;

    PMaxProfile  maxProfile = &resident->maxProfile;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "MaxProfile " );

    if ( ( i = LookUp_TrueType_Table( resident, TTAG_maxp ) ) < 0 )
      return FAILURE;

    if ( !FILE_Seek( resident->dirTables[i].Offset ) )
                                                 /* seek to maxprofile */
      return FAILURE;

    if ( !ACCESS_Frame( 32L ) )  /* read into frame */
      return FAILURE;

    /* read frame data into resident table */
    maxProfile->version = GET_ULong();

    maxProfile->numGlyphs             = GET_UShort();

    maxProfile->maxPoints             = GET_UShort();
    maxProfile->maxContours           = GET_UShort();
    maxProfile->maxCompositePoints    = GET_UShort();
    maxProfile->maxCompositeContours  = GET_UShort();

    maxProfile->maxZones              = GET_UShort();
    maxProfile->maxTwilightPoints     = GET_UShort();

    maxProfile->maxStorage            = GET_UShort();
    maxProfile->maxFunctionDefs       = GET_UShort();
    maxProfile->maxInstructionDefs    = GET_UShort();
    maxProfile->maxStackElements      = GET_UShort();
    maxProfile->maxSizeOfInstructions = GET_UShort();
    maxProfile->maxComponentElements  = GET_UShort();
    maxProfile->maxComponentDepth     = GET_UShort();

    FORGET_Frame();

    resident->numGlyphs = maxProfile->numGlyphs;

    DebugTrace1( "loaded\n" );
  
    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_Header
 *
 *  Description :  Load the TrueType header table into the resident
 *                 table.
 *
 *  Input  :  resident     resident table to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_Header( PResident_Record  resident )
  {
    int  i;

    THeader*  header;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "Header " );

    if ( ( i = LookUp_TrueType_Table( resident, TTAG_head ) ) < 0 )
      return FAILURE;

    if ( !FILE_Seek( resident->dirTables[i].Offset ) )
      return FAILURE;

    if ( !ACCESS_Frame( 54L ) )
      return FAILURE;

    header = &resident->fontHeader;

    header->Table_Version = GET_ULong();
    header->Font_Revision = GET_ULong();

    header->CheckSum_Adjust = GET_Long();
    header->Magic_Number    = GET_Long();

    header->Flags        = GET_UShort();
    header->Units_Per_EM = GET_UShort();

    header->Created.l1  = GET_Long();
    header->Created.l2  = GET_Long();
    header->Modified.l1 = GET_Long();
    header->Modified.l2 = GET_Long();

    header->xMin = GET_Short();
    header->yMin = GET_Short();
    header->xMax = GET_Short();
    header->yMax = GET_Short();

    header->Mac_Style       = GET_UShort();
    header->Lowest_Rec_PPEM = GET_UShort();

    header->Font_Direction      = GET_Short();
    header->Index_To_Loc_Format = GET_Short();
    header->Glyph_Data_Format   = GET_Short();

    DebugTrace1( "loaded\n" );

    DebugTrace2( "Units per EM       : %12u\n", 
                 resident->fontHeader.Units_Per_EM );

    DebugTrace2( "IndexToLocFormat   : %12d\n",
                 resident->fontHeader.Index_To_Loc_Format );

    FORGET_Frame();

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    : Load_TrueType_Horizontal_Header
 *
 *  Description : Loads the hhea table into resident table.
 *
 *  Input  :  resident   resident table to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_Horizontal_Header( PResident_Record  resident )
  {
    int  i;

    THorizontalHeader*  hheader;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "Horizontal header " );

    if ( ( i = LookUp_TrueType_Table( resident, TTAG_hhea ) ) < 0 )
      return FAILURE;

    if ( !FILE_Seek( resident->dirTables[i].Offset ) )
      return FAILURE;

    if ( !ACCESS_Frame( 36L ) )
      return FAILURE;

    hheader = &resident->horizontalHeader;

    hheader->Version   = GET_ULong();
    hheader->Ascender  = GET_Short();
    hheader->Descender = GET_Short();
    hheader->Line_Gap  = GET_Short();

    hheader->advance_Width_Max = GET_UShort();

    hheader->min_Left_Side_Bearing  = GET_Short();
    hheader->min_Right_Side_Bearing = GET_Short();
    hheader->xMax_Extent            = GET_Short();
    hheader->caret_Slope_Rise       = GET_Short();
    hheader->caret_Slope_Run        = GET_Short();

    hheader->Reserved[0] = GET_Short();
    hheader->Reserved[1] = GET_Short();
    hheader->Reserved[2] = GET_Short();
    hheader->Reserved[3] = GET_Short();
    hheader->Reserved[4] = GET_Short();

    hheader->metric_Data_Format = GET_Short();
    hheader->number_Of_HMetrics = GET_UShort();

    FORGET_Frame();

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_Locations
 *
 *  Description :  Loads the location table into resident table.
 *
 *  Input  :  resident     resident table to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  NOTES :
 *
 *    The Font Header *must* be loaded in the leading segment
 *    calling this function.
 *
 ******************************************************************/

  static Bool  Load_TrueType_Locations( PResident_Record  resident )
  {
    int  n, limit, LongOffsets;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "Locations " );

    LongOffsets = resident->fontHeader.Index_To_Loc_Format;

    if ( ( n = LookUp_TrueType_Table( resident, TTAG_loca ) ) < 0 )
      return FAILURE;

    if ( !FILE_Seek( resident->dirTables[n].Offset ) )
      return FAILURE;

    if ( LongOffsets != 0 )
    {
      resident->numLocations =
        (unsigned long)resident->dirTables[n].Length >> 2;

      DebugTrace2( "( 32 bits offsets ) : %12d ",
                   resident->numLocations );

      if ( !ALLOC_ARRAY( resident->glyphLocations,
                         resident->numLocations,
                         Long ) )
        return FAILURE;

      if ( !ACCESS_Frame( resident->numLocations * 4L ) )
        return FAILURE;

      limit = resident->numLocations;

      for ( n = 0; n < limit; n++ )
        resident->glyphLocations[n] = GET_Long();

      FORGET_Frame();
    }
    else
    {
      resident->numLocations =
        (unsigned long)resident->dirTables[n].Length >> 1;

      DebugTrace2( "( 16 bits offsets ) : %12d ",
                   resident->numLocations );

      if ( !ALLOC_ARRAY( resident->glyphLocations,
                         resident->numLocations,
                         Long ) )
        return FAILURE;

      if ( !ACCESS_Frame( resident->numLocations * 2L ) )
        return FAILURE;   

      limit = resident->numLocations;

      for ( n = 0; n < limit; n++ )
        resident->glyphLocations[n] =
          (Long)((unsigned long)GET_UShort() * 2);

      FORGET_Frame();
    }

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_CVT
 *
 *  Description :  Loads cvt table into resdent table.
 *
 *  Input  :  resident     resident table to look for
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_CVT( PResident_Record  resident )
  {
    long  n;
    Int  limit;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "CVT " );

    if ( ( n = LookUp_TrueType_Table( resident, TTAG_cvt ) ) < 0 )
    {

      DebugTrace1( "is missing!\n" );

      resident->cvtSize = 0;
      resident->cvt     = NULL;
      return SUCCESS;
    }

    resident->cvtSize = resident->dirTables[n].Length / 2;

    if ( !ALLOC_ARRAY( resident->cvt,
                       resident->cvtSize,
                       Short ) )
      return FAILURE;

    if ( !FILE_Seek( resident->dirTables[n].Offset ) )
      return FAILURE;

    if ( !ACCESS_Frame( resident->cvtSize * 2 ) )
      return FAILURE;

    limit = resident->cvtSize;

    for ( n = 0; n < limit; n++ )
      resident->cvt[n] = GET_Short();

    FORGET_Frame();

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_Glyph
 *
 *  Description :  Loads a glyph from the TrueType file into memory.
 *
 *  Input  :  instance     the instance where the glyph data will be
 *                          stored
 *            glyphIndex    the index of glyph to be loaded
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  Bool  Load_TrueType_Glyph( PInstance_Record  instance, 
                             Int               glyphIndex,
                             Bool              allow_components )
  {
    Int   curr_contour  = 0;
    Int   curr_point    = 0;
    Int   curr_subglyph = 0;
    Bool  result;
    Int   i;

    /* verify sanity of user call */
    if ( instance == NULL )
    {
      ERROR = TT_Err_Invalid_Font_Handle;
      return FAILURE;
    }
    if ( glyphIndex >= instance->fontRes->numGlyphs )
    {
      ERROR = TT_Err_Invalid_Glyph_Index;
      return FAILURE;
    }

    /* 
     * Must lock stream here!
     * Load_Glyph() is reentrant so we cannot lock it there.
     */
    if ( !LOCK_Access( instance->stream ) )
      return FAILURE;

    instance->pts.n = 0;

    result = Load_Glyph( instance, 
                         glyphIndex,
                         &curr_contour, 
                         &curr_point, 
                         &curr_subglyph,
                         allow_components );

    RELEASE_Access( instance->stream );

    if ( result )
    {
      instance->numContours = instance->glyph->numberOfContours;

      for ( i = 0; i < instance->numContours; i++ )
        instance->endContours[i] = instance->glyph->contours[i].finish;

    }

    return result;
  }

/* internal glyph loader */

  Bool  Load_Glyph( PInstance_Record  instance,
                    Int               glyphIndex,
                    PInt              curr_contour,
                    PInt              curr_point,
                    PInt              curr_subglyph,
                    Bool              allow_components )
  {
    PResident_Record  fontRes = instance->fontRes;

    PGlyph        glyph = instance->glyph;                

    Int           i, old_subglyph;
    Short         szc;

    Long          offset, here;
    UShort        subglyphIndex, temp, flags;
    Short         arg1, arg2;

    PGlyphScales  gs;
    PGlyphScale   curr_gs;

    TT_Fixed      x_scale = INT_TO_FIXED( 1 ),
                  scale01 = 0,
                  scale10 = 0,
                  y_scale = INT_TO_FIXED( 1 );

    TT_Fixed      n1, n2, n3, n4;

    TT_F26Dot6    x_offset = 0,
                  y_offset = 0;

    /* used to track composite metrics */
    Int           cAdvanceWidth,
                  cLeftSideBearing;

    Short         cxMin,cxMax,cyMin,cyMax;

    DEFINE_STREAM;
    DEFINE_FRAME;


    if ( ( i = LookUp_TrueType_Table( fontRes, TTAG_glyf ) ) < 0 )
      return FAILURE;
    
    offset = fontRes->dirTables[i].Offset;
  
    /* prep and read in first 5 words of info about glyph */
    /* some fonts handle the end of the glyph index in a non-standard
       way */
    if ( fontRes->glyphLocations[glyphIndex] ==
         fontRes->glyphLocations[glyphIndex + 1 ] )
      glyphIndex = 0;

    if ( !FILE_Seek ( offset + fontRes->glyphLocations[glyphIndex] ) )
      return FAILURE;

    if ( !ACCESS_Frame( 5 * 2L ) )
      return FAILURE;

    /* please note that these extrema are not reliable in all cases! */

    glyph->numberOfContours = GET_UShort();
    glyph->xMin             = GET_UShort();
    glyph->yMin             = GET_UShort();
    glyph->xMax             = GET_UShort();
    glyph->yMax             = GET_UShort();

    FORGET_Frame();
    
    DebugTrace3( "Glyph %d offset %08lX\n", glyphIndex, offset );
    DebugTrace2( " # of Contours :  %12d\n", glyph->numberOfContours );
    DebugTrace3( " xMin : %4d  xMax : %12d\n", glyph->xMin, glyph->xMax );
    DebugTrace3( " yMin : %4d  yMax : %12d\n", glyph->yMin, glyph->yMax );
    DebugTrace1( "-\n" );
  
    szc = glyph->numberOfContours;
    gs  = glyph->scales;
    
    if ( szc > MAX( fontRes->maxProfile.maxContours,
                    fontRes->maxProfile.maxCompositeContours ) )
      return FAILURE;

    else if ( szc == 0 )                        /* empty glyph */
      return SUCCESS;

    else if ( szc > 0 )                         /* simple glyph */
    {
      gs[*curr_subglyph].start = *curr_point;

      if ( !Load_Simple_Glyph( instance,
                               glyphIndex,
                               *curr_subglyph,
                               szc,
                               *curr_contour,
                               curr_point ) )
        return FAILURE;

      gs[*curr_subglyph].flags  = 0;        /* set flags for simple glyph */
      gs[*curr_subglyph].finish = *curr_point - 1;

      /* default scaling values */
      gs[*curr_subglyph].x_offset = 0;
      gs[*curr_subglyph].y_offset = 0;
      gs[*curr_subglyph].x_scale  = INT_TO_FIXED( 1 );
      gs[*curr_subglyph].scale01  = 0;
      gs[*curr_subglyph].scale10  = 0;
      gs[*curr_subglyph].y_scale  = INT_TO_FIXED( 1 );

      *curr_contour += szc;
      (*curr_subglyph)++;

      glyph->numberOfContours  = *curr_contour;
      glyph->numberOfPoints    = *curr_point;
      glyph->numberOfSubglyphs = *curr_subglyph;
    }
    else
    {                                     /* composite glyph */
      if ( !allow_components )
        return FAILURE;

      /* snapshot of source metrics. We will end up using these */
      /* unless USE_MY_METRICS is set in a sub_glyph            */
      cxMin = glyph->xMin;
      cxMax = glyph->xMax;
      cyMin = glyph->yMin;
      cyMax = glyph->yMax;
                  
      i = fontRes->horizontalHeader.number_Of_HMetrics;
      if ( glyphIndex < i )
      {
        cLeftSideBearing = fontRes->LongHMetrics[glyphIndex].lsb;
        cAdvanceWidth = fontRes->LongHMetrics[glyphIndex].advance_Width;
      }
      else
      {
        cLeftSideBearing = fontRes->ShortMetrics[glyphIndex - i];
        cAdvanceWidth = fontRes->LongHMetrics[i - 1].advance_Width;
      }
      /* end snapshot of source metrics */    

      do
      {
        old_subglyph = *curr_subglyph;

        if ( !ACCESS_Frame( 4L ) )
          return FAILURE;

        flags         = GET_UShort();
        subglyphIndex = GET_UShort();

        FORGET_Frame();

        /* the only flag which is of relevance is ROUND_XY_TO_GRID. We   */
        /* ignore this flag from deeper levels---USE_MY_METRICS is not   */
        /* implemented yet. The grid rounding flag is the very reason    */
        /* that we can't apply the scaling values immediately but rather */
        /* have to collect them in a separate array.                     */

        here = FILE_Pos();   /* keep track of our position */

        Load_Glyph( instance, 
                    subglyphIndex,
                    curr_contour, 
                    curr_point, 
                    curr_subglyph,
                    TRUE );

        FILE_Seek( here );

        /* Access the entire frame for the composite in one block. */

        i = 2;  /* number of bytes needed if no flags are set */

        if ( flags & ARGS_ARE_WORDS )
          i += 2;

        /* ignore ARGS_ARE_XY_VALUES.               */
        /* It indicates a data transformation only. */

        if ( flags & WE_HAVE_A_SCALE )
          i += 2;

        else if ( flags & WE_HAVE_AN_XY_SCALE )
          i += 4;

        else if ( flags & WE_HAVE_A_2X2 )
          i += 8;

        if ( !ACCESS_Frame( i ) )
          return FAILURE;
       
        if ( flags & ARGS_ARE_WORDS )
        {
          arg1 = GET_Short();
          arg2 = GET_Short();
        }
        else
        {
          temp = GET_UShort();
          arg1 = (signed char)(temp >> 8);
          arg2 = (signed char)(temp & 0xFF);
        }

        if ( flags & ARGS_ARE_XY_VALUES )
        {
#if 0
          x_offset = INT_TO_F26DOT6( arg1 );
          y_offset = INT_TO_F26DOT6( arg2 );
#else
          x_offset = arg1;
          y_offset = arg2;
#endif
        }
        else
        {
          if ( arg1 > *curr_point || arg2 > *curr_point )
          {
            FORGET_Frame();
            return FAILURE;
          }
#if 0
          x_offset = glyph->points[arg2].V.x -
                     glyph->points[arg1].V.x;

          y_offset = glyph->points[arg2].V.y -
                     glyph->points[arg1].V.y;
#else
          /* Are these points based off of the    */
          /* start of the entire composite glyph? */
          /* this code presumes it                */
          x_offset = instance->pts.cur_x[arg2] -
                     instance->pts.cur_x[arg1];

          y_offset = instance->pts.cur_y[arg2] -
                     instance->pts.cur_y[arg1];
#endif
        }

        if ( flags & WE_HAVE_A_SCALE )
        {
          x_scale = F2DOT14_TO_FIXED( GET_Short() );
          scale01 = 0;
          scale10 = 0;
          y_scale = x_scale;
        }
        else if ( flags & WE_HAVE_AN_XY_SCALE )
        {
          x_scale = F2DOT14_TO_FIXED( GET_Short() );
          scale01 = 0;
          scale10 = 0;
          y_scale = F2DOT14_TO_FIXED( GET_Short() );
        }
        else if ( flags & WE_HAVE_A_2X2 )
        {
          x_scale = F2DOT14_TO_FIXED( GET_Short() );
          scale01 = F2DOT14_TO_FIXED( GET_Short() );
          scale10 = F2DOT14_TO_FIXED( GET_Short() );
          y_scale = F2DOT14_TO_FIXED( GET_Short() );
        }

        FORGET_Frame();

        /* now we apply the current scaling values to all the recursive */
        /* subglyphs called from this depth level.                      */

        for ( i = old_subglyph, curr_gs = glyph->scales + old_subglyph;
              i < *curr_subglyph; i++, curr_gs++ )
        {
          curr_gs->flags = flags | WE_HAVE_A_2X2;

          n1 = x_offset +
               MUL_FIXED( curr_gs->x_offset, x_scale ) +
               MUL_FIXED( curr_gs->y_offset, scale01 );
          n2 = y_offset +
               MUL_FIXED( curr_gs->x_offset, scale10 ) +
               MUL_FIXED( curr_gs->y_offset, y_scale );

          curr_gs->x_offset = n1;
          curr_gs->y_offset = n2;

          n1 = MUL_FIXED( curr_gs->x_scale, x_scale ) +
               MUL_FIXED( curr_gs->scale01, scale10 );

          n2 = MUL_FIXED( curr_gs->x_scale, scale01 ) +
               MUL_FIXED( curr_gs->scale01, y_scale );

          n3 = MUL_FIXED( curr_gs->scale10, x_scale ) +
               MUL_FIXED( curr_gs->y_scale, scale10 );

          n4 = MUL_FIXED( curr_gs->scale10, scale01 ) +
               MUL_FIXED( curr_gs->y_scale, y_scale );

          curr_gs->x_scale = n1;
          curr_gs->scale01 = n2;
          curr_gs->scale10 = n3;
          curr_gs->y_scale = n4;
        }

        /* Use my metrics */
        if( flags & USE_MY_METRICS )
        {
          cxMin = glyph->xMin;
          cxMax = glyph->xMax;
          cyMin = glyph->yMin;
          cyMax = glyph->yMax;

          i = instance->fontRes->horizontalHeader.number_Of_HMetrics;
          if ( glyphIndex < i )
          {
            cLeftSideBearing = 
              fontRes->LongHMetrics[ glyphIndex ].lsb;

            cAdvanceWidth = 
              fontRes->LongHMetrics[ glyphIndex ].advance_Width;
          }
          else
          {
            cLeftSideBearing =
              fontRes->ShortMetrics[ glyphIndex-i ];

            cAdvanceWidth =
              fontRes->LongHMetrics[ i-1 ].advance_Width;
          }
        }
        /* end Use my metrics */        
      } while ( flags & MORE_COMPONENTS );

      /* use the local metrics to set the phantom points;          */
      /* also reset the glyph Min and Max to the correct settings. */

      instance->advanceWidth    = cAdvanceWidth;
      instance->leftSideBearing = cLeftSideBearing;

      /* composite phantom points are based on the */
      /* original glyph metrics, FOR NOW.          */

      glyph->xMin = cxMin;
      glyph->xMax = cxMax;
      glyph->yMin = cyMin;
      glyph->yMax = cyMax;
        
      instance->pts.org_x[instance->pts.n - 2] = 
                 cxMin - cLeftSideBearing;
      instance->pts.cur_x[instance->pts.n - 2] = 
                 instance->pts.org_x[instance->pts.n - 2]; 
      instance->pts.cur_x[instance->pts.n - 1] =
                 cxMin - cLeftSideBearing + cAdvanceWidth;
      instance->pts.org_x[instance->pts.n - 1] =
                 instance->pts.org_x[instance->pts.n - 1];

      if ( flags & WE_HAVE_INSTR )
      {
        ; /* Need to add support for this */
      }
    }

    return SUCCESS;
  }

  /* this is a subroutine of Load_Glyph() */

  Bool  Load_Simple_Glyph( PInstance_Record  instance,
                           Int               glyphIndex,
                           Int               subglyph,
                           Int               num_contours,
                           Int               curr_contour,
                           PInt              curr_point )
  {
    PResident_Record  fontRes = instance->fontRes;

    PGlyph  glyph = instance->glyph;
    Int     szp;
    UShort  k;
    Byte    c, cnt;

    /* the following allows composite glyphs to accumulate the data
       for all the subglyphs */
    PGlyphContours   con = glyph->contours + curr_contour;
                                                    /* new contours follow */
    TT_VecRecord*    pts = &instance->pts; 

    TT_PCoordinates  pOrg_x;
    TT_PCoordinates  pOrg_y;
    TT_PCoordinates  pCur_x;
    TT_PCoordinates  pCur_y;
    TT_PTouchTable   pTouch;

    DEFINE_STREAM;
    DEFINE_FRAME;


    /* new points follow */
    pOrg_x = pts->org_x + *curr_point;
    pOrg_y = pts->org_y + *curr_point;
    pCur_x = pts->cur_x + *curr_point;
    pCur_y = pts->cur_y + *curr_point;
    pTouch = pts->touch + *curr_point;
 
    /* Reading the contours endpoints */
    szp = 0;

    /* also get instruction count follows contour information */
    if ( !ACCESS_Frame( ( num_contours + 1 ) * 2L ) )
      return FAILURE;

    /* load as start and end to support composite transformations */
    for ( k = 0; k < num_contours; k++ )
    {
      con[k].start = *curr_point + szp;
      szp = GET_Short();
      con[k].finish = *curr_point + szp;

      DebugTrace2( "%08d :", szp );

      szp++;
    }

    *curr_point += szp;
                    /* update total glyph point count, could be composite */

    /* Loading instructions info */
    k = glyph->instruct[subglyph].numIns = GET_Short();
    glyph->instruct[subglyph].offset = FILE_Pos();

    FORGET_Frame();

    DebugTrace2( "\nInstructions size : %12d\n", k );

    if ( glyph->instruct[subglyph].numIns >
         fontRes->maxProfile.maxSizeOfInstructions )
    {
      DebugTrace1( "Too many instructions\n" );

      ERROR = TT_Err_Too_Many_Ins;
      return FAILURE;
    }

    /* Ignore instructions here.  We do not want to waste space storing */
    /* them all; they will accessed when hinting is working.            */
    if ( !FILE_Skip( k ) )
      return FAILURE;

    /* prepare to access the flags and x,y points */
    if ( !CHECK_ACCESS_Frame( szp * ( 1L + 2L * 2 ) ) )
      return FAILURE;

    /* Reading Flags */
    k = 0;
    while ( k < szp )
    {
      c = GET_Byte();
      pTouch[k] = c;
      k++;
      if ( (c & 8) == 0 )
        continue;

      cnt = GET_Byte();
      while ( cnt > 0 )
      {
        pTouch[k] = c;
        k++;
        cnt--;
      }
    }

    /* Reading the Xs */
    for ( k = 0; k < szp; k++ )
    {
      if ( (pTouch[k] & 2) != 0 )
      {
        if ( (pTouch[k] & 16) != 0 )
          pOrg_x[k] = GET_Byte();
        else
          pOrg_x[k] = -GET_Byte();
      }
      else
      {
        if ( (pTouch[k] & 16) != 0 )
          pOrg_x[k] = 0;
        else
          pOrg_x[k] = GET_Short();
      }
    }

    /* Reading the Ys */
    for ( k = 0; k < szp; k++ )
    {
      if ( (pTouch[k] & 4) != 0 )
      {
        if ( (pTouch[k] & 32) != 0 )
          pOrg_y[k] = GET_Byte();
        else
          pOrg_y[k] = -GET_Byte();
      }
      else
      {
        if ( (pTouch[k] & 32) != 0 )
          pOrg_y[k] = 0;
        else
          pOrg_y[k] = GET_Short();
      }
    }

    FORGET_Frame();

    /* Convert relative coords to absolute ones. */
    for ( k = 1; k < szp; k++ )
    {
      pOrg_x[k] += pOrg_x[k - 1];
      pOrg_y[k] += pOrg_y[k - 1];
    }

    /* Now add the two shadow points at n and n+1.      */
    /* We need the left side bearing and advance width. */
    k = fontRes->horizontalHeader.number_Of_HMetrics;
    if ( glyphIndex < k )
    {
      instance->leftSideBearing =
        fontRes->LongHMetrics[glyphIndex].lsb;

      instance->advanceWidth =
        fontRes->LongHMetrics[glyphIndex].advance_Width;
    }
    else
    {
      instance->leftSideBearing =
        fontRes->ShortMetrics[glyphIndex - k];

      instance->advanceWidth =
        fontRes->LongHMetrics[k - 1].advance_Width;
    }

    pOrg_x[szp] = glyph->xMin - instance->leftSideBearing;
    pOrg_y[szp] = 0;            /* pp1 = xMin - lsb */

    pOrg_x[szp + 1] =
      glyph->xMin - instance->leftSideBearing + instance->advanceWidth;

    pOrg_y[szp + 1] = 0;        /* pp2 = pp1 + aw   */

    for ( k = 0; k < szp; k++ ) pTouch[k] &= TT_Flag_On_Curve;

    szp += 2;

    /* Note that we now return two more points */
    /* not part of the glyph outline.          */

    instance->pts.n += szp;  

    for ( k = 0; k < szp; k++ )
    {
      pCur_x[k] = pOrg_x[k];
      pCur_y[k] = pOrg_y[k];
    }

    return SUCCESS;
  }


  Bool  Load_Glyph_Instructions( PInstance_Record  instance,
                                 Int               component )
  {
    PGlyph  glyph;
    Bool    result;

    DEFINE_STREAM;
    /* No frame used */


    glyph = instance->glyph;

    if ( component >= glyph->numberOfSubglyphs )
      return FAILURE;

    if ( glyph->instruct[component].numIns > 
         instance->fontRes->maxProfile.maxSizeOfInstructions )
      return FAILURE;

    if ( !LOCK_Access( instance->stream ) )
      return FAILURE;

    result = ( FILE_Read_At( glyph->instruct[component].offset,
                             instance->glyphIns,
                             glyph->instruct[component].numIns ) );

    RELEASE_Access( instance );

    if (result) Set_CodeRange( instance->exec, 
                               TT_CodeRange_Glyph,
                               instance->glyphIns,
                               glyph->instruct[component].numIns );

    return result;
  }
                                 

/*******************************************************************
 *
 *  Function    :  Load_TrueType_CMap
 *
 *  Description :  Loads all cmap tables into resident table.
 *
 *  Input  :  resident
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_CMap( PResident_Record  resident )
  {
    long             n, num_SH;
    unsigned short   u;
    long             off;
    unsigned short   l;
    long             num_Seg;
    unsigned short*  glArray;
    long             table_start;
    int              limit, i;

    TCMapDir         cmap_dir;
    TCMapDirEntry    entry_;
    PCMapTable       Plcmt;
    PCMap2SubHeader  Plcmsub;
    PCMap4           Plcm4;
    PCMap4Segment    segments;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "CMaps " );

    if ( ( n = LookUp_TrueType_Table( resident, TTAG_cmap ) ) < 0 )
      return FAILURE;

    table_start = resident->dirTables[n].Offset;

    if ( ( !FILE_Seek( resident->dirTables[n].Offset ) ) ||
         ( !ACCESS_Frame ( 4L ) ) )           /* 4 bytes cmap header */
      return FAILURE;

    cmap_dir.tableVersionNumber = GET_UShort();
    cmap_dir.numCMaps           = GET_UShort();

    FORGET_Frame();

    off = FILE_Pos();  /* save offset to cmapdir[] which follows */

    resident->numCMaps = cmap_dir.numCMaps;

    /* save space in resident table for cmap tables */
    if ( !ALLOC_ARRAY( resident->cMaps,
                       resident->numCMaps,
                       TCMapTable ) )
      return FAILURE;

    limit = resident->numCMaps;

    for ( n = 0; n < limit; n++ )
    {
      Plcmt = &resident->cMaps[n];

      if ( !FILE_Seek( off ) )
        return FAILURE;

      if ( !ACCESS_Frame( 8L ) )
        return FAILURE;

      /* extra code using entry_ for platxxx could be cleaned up later */
      Plcmt->platformID         = entry_.platformID         = GET_UShort();
      Plcmt->platformEncodingID = entry_.platformEncodingID = GET_UShort();

      entry_.offset = GET_Long();

      FORGET_Frame();

      off = FILE_Pos();

      if ( ( !FILE_Seek( table_start + entry_.offset ) ) ||
           ( !ACCESS_Frame( 6L ) ) )
        return FAILURE;
      
      Plcmt->Format  = GET_UShort();
      Plcmt->Length  = GET_UShort();
      Plcmt->Version = GET_UShort();

      FORGET_Frame();

      switch ( Plcmt->Format )
      {
      case 0:
        if ( !ALLOC( Plcmt->uc.cmap0, sizeof( TCMap0Table ) ) )
          return FAILURE;

        if ( !ALLOC( Plcmt->uc.cmap0->glyphIdArray, 256L ) ||
             !FILE_Read( (void *)Plcmt->uc.cmap0->glyphIdArray, 256L ) )
          return FAILURE;
        break;

      case 2:
        if ( !ALLOC( Plcmt->uc.cmap2, sizeof( TCMap2Table ) ) )
          return FAILURE;

        num_SH = 0;
  
        if ( !ACCESS_Frame( 512L ) )
          return FAILURE;
  
        for ( i = 0; i < 256; i++ )
        {
          u = GET_UShort() / 8;
          Plcmt->uc.cmap2->Header.subHeaderKeys[i] = u;

          if ( num_SH < u )
            num_SH = u;
        }

        FORGET_Frame();

        l = Plcmt->Length -
                   sizeof ( UShort ) * (256 + 3) - num_SH * 8;

        if ( !ALLOC_ARRAY( Plcmt->uc.cmap2->subHeaders,
                           num_SH, 
                           TCMap2SubHeader )  ||
             !ACCESS_Frame( (num_SH+1)*8L ) )

          return FAILURE;

        for ( i = 0; i <= num_SH; i++ )
        {
          Plcmsub = &Plcmt->uc.cmap2->subHeaders[i];

          Plcmsub->firstCode     = GET_UShort();
          Plcmsub->entryCount    = GET_UShort();
          Plcmsub->idDelta       = GET_Short();
          /* we apply the location offset immediately */
          Plcmsub->idRangeOffset = GET_UShort() - ( num_SH - i ) * 8 - 2;
        }

        FORGET_Frame();

        l /= 2;

        if ( !ALLOC_ARRAY( Plcmt->uc.cmap2->glyphIdArray,
                           l,
                           UShort ) ||
             !ACCESS_Frame( l*2L ) )

          return FAILURE;

        glArray = Plcmt->uc.cmap2->glyphIdArray;
        for ( i = 0; i < l; i++ )
          glArray[i] = GET_UShort();

        FORGET_Frame();
        break;

      case 4:
        if ( !ALLOC( Plcmt->uc.cmap4, sizeof( TCMap4Table ) ) ||
             !ACCESS_Frame( 4*2L ) )
          return FAILURE;
  
        Plcm4 = &Plcmt->uc.cmap4->Header;
  
        Plcm4->segCountX2    = GET_UShort();
        Plcm4->searchRange   = GET_UShort();
        Plcm4->entrySelector = GET_UShort();
        Plcm4->rangeShift    = GET_UShort();

        num_Seg = Plcm4->segCountX2 / 2;

        FORGET_Frame();

        if ( !ALLOC_ARRAY( Plcmt->uc.cmap4->Segments,
                           num_Seg,
                           TCMap4Segment )           ||
             !ACCESS_Frame( (num_Seg * 4 + 1) * 2L ) )

          return FAILURE;

        segments = Plcmt->uc.cmap4->Segments;

        for ( i = 0; i < num_Seg; i++ )
          segments[i].endCount      = GET_UShort();

        GET_UShort();

        for ( i = 0; i < num_Seg; i++ )
          segments[i].startCount    = GET_UShort();

        for ( i = 0; i < num_Seg; i++ )
          segments[i].idDelta       = GET_UShort();

        for ( i = 0; i < num_Seg; i++ )
          segments[i].idRangeOffset = GET_UShort();

        FORGET_Frame();

        l = ( resident->cMaps[n].Length - ( 16L + 8L * num_Seg ) ) & 0xffff;

        if ( !ALLOC( Plcmt->uc.cmap4->glyphIdArray, l ) ||
             !ACCESS_Frame( l ) )
          return FAILURE;

        glArray = Plcmt->uc.cmap4->glyphIdArray;
        for ( i = 0; i < l / 2; i++ )
          glArray[i] = GET_UShort();

        FORGET_Frame();
        break;

      case 6:
        if ( !ALLOC( Plcmt->uc.cmap6, sizeof( TCMap6Table ) ) ||
             !ACCESS_Frame( 2 * 2L ) )
          return FAILURE;
  
        Plcmt->uc.cmap6->Header.firstCode  = GET_UShort();
        Plcmt->uc.cmap6->Header.entryCount = GET_UShort();

        FORGET_Frame();

        l = Plcmt->uc.cmap6->Header.entryCount;

        if ( !ALLOC_ARRAY( Plcmt->uc.cmap6->glyphIdArray,
                           l,
                           Short ) ||
             !ACCESS_Frame( l*2L ) )

          return FAILURE;

        glArray = Plcmt->uc.cmap6->glyphIdArray;

        for ( i = 0; i < l; i++ )
          glArray[i] = GET_UShort();

        FORGET_Frame();
        break;

      default:   /* corrupt character mapping table */
        return FAILURE;
        break;
      }
    }

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_HMTX
 *
 *  Description :  Load the horizontal metrics table into resident
 *                 table.
 *
 *  Input  :  resident
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_HMTX( PResident_Record  resident )
  {
    long              n, nsmetrics, nhmetrics;
    PTableHorMetrics  plhm;

    DEFINE_RES_STREAM;
    DEFINE_FRAME;


    DebugTrace1( "HMTX " );

    if ( ( n = LookUp_TrueType_Table( resident, TTAG_hmtx ) ) < 0 )
      return FAILURE;

    nhmetrics = resident->horizontalHeader.number_Of_HMetrics;
    nsmetrics = resident->maxProfile.numGlyphs - nhmetrics;

    if ( !ALLOC_ARRAY( resident->LongHMetrics,
                       nhmetrics,
                       TLongHorMetric )                 ||
         !ALLOC_ARRAY( resident->ShortMetrics,
                       nsmetrics,
                       Long )                           ||
         !FILE_Seek( resident->dirTables[n].Offset )    ||
         !ACCESS_Frame( resident->dirTables[n].Length ) )

      return FAILURE;

    for ( n = 0; n < nhmetrics; n++ )
    {
      plhm = &resident->LongHMetrics[n];

      plhm->advance_Width = GET_Short();
      plhm->lsb           = GET_Short();
    }

    for ( n = 0; n < nsmetrics; n++ )
      resident->ShortMetrics[n] = GET_Short();

    FORGET_Frame();

    DebugTrace1( "loaded\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  Load_TrueType_Programs
 *
 *  Description :  Load the font (fpgm) and cvt programs into the
 *                 resident table.
 *
 *  Input  :  resident
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  static Bool  Load_TrueType_Programs( PResident_Record  resident )
  {
    int  n;

    DEFINE_RES_STREAM;
    /* No Frame used */


    DebugTrace1( "Font program " );

    /* The font program is optional */
    if ( ( n = LookUp_TrueType_Table( resident, TTAG_fpgm ) ) < 0 )
    {
      resident->fontProgram = NULL;
      resident->fontPgmSize = 0;

      DebugTrace1( "is missing!\n" );
    }
    else
    {
      resident->fontPgmSize = resident->dirTables[n].Length;

      if ( !ALLOC( resident->fontProgram,
                   resident->fontPgmSize )              ||
           !FILE_Read_At( resident->dirTables[n].Offset,
                          (void*)resident->fontProgram,
                          resident->fontPgmSize )       )
        return FAILURE;

      DebugTrace2( "loaded, %12d bytes\n", resident->fontPgmSize );
    }

   DebugTrace1( "Prep program " );

    if ( ( n = LookUp_TrueType_Table( resident, TTAG_prep ) ) < 0 )
    {
      resident->cvtProgram = NULL;
      resident->cvtPgmSize = 0;

      DebugTrace1( "is missing!\n" );

      return SUCCESS;
    }

    resident->cvtPgmSize = resident->dirTables[n].Length;

    if ( !ALLOC( resident->cvtProgram,
                 resident->cvtPgmSize )               ||
         !FILE_Read_At( resident->dirTables[n].Offset,
                        (void*)resident->cvtProgram,
                        resident->cvtPgmSize )        )
      return FAILURE;

    DebugTrace2( "loaded, %12d bytes\n", resident->cvtPgmSize );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  TT_Load_Resident_Table
 *
 *  Description :  Loads one font file and builds its resident tables.
 *
 *  Input  :  stream      file handle of the opened file.
 *            President  pointer to a resident table pointer.
 *
 *  Output :  SUCCESS on success, FAILURE on error.
 *
 *****************************************************************/

  Bool  TT_Load_Resident_Table( TT_Stream          stream,
                                PResident_Record*  resident )
  {
    TMarkRecord  start;
    long         size;


    Mark( &start );

    size = Mem_Free();

    if ( LOCK_Access( stream ) )
    {
      if ( ALLOC( (*resident), sizeof ( TResident_Record ) ) )
      {
        (*resident)->stream = stream;

        if ( ( Load_TrueType_Directory( *resident ) )         &&
             ( Load_TrueType_Header( *resident ) )            &&
             ( Load_TrueType_MaxProfile( *resident ) )        &&
             ( Load_TrueType_Locations( *resident ) )         &&
             ( Load_TrueType_CMap( *resident ) )              &&
             ( Load_TrueType_CVT( *resident ) )               &&
             ( Load_TrueType_Horizontal_Header( *resident ) ) &&
             ( Load_TrueType_Programs( *resident ) )          && 
             ( Load_TrueType_HMTX( *resident ) )              )
        {
          RELEASE_Access( stream );

          size -= Mem_Free();

          (*resident)->stream    = stream;
          (*resident)->totalSize = size;

          return SUCCESS;
        }
      }

      RELEASE_Access( stream );
    }

    Release( &start );

    *resident = NULL;

    return FAILURE;
  }


/*******************************************************************
 *
 *  Function    :  TT_Reset_Glyph
 *
 *  Description :  Resets instance data to default values.
 *
 *  Input  :  instance     the instance where the glyph data will be
 *                          stored
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  :  Must be used before loading a glyph.
 *
 ******************************************************************/

  Bool  TT_Reset_Glyph( PInstance_Record  instance )
  {
    PExecution_Context exec;


    exec = (PExecution_Context)instance->exec;

    /* Clear current execution state */
    exec->codeSize = 0;
    exec->IP       = 0;
    exec->curRange = -1;
    exec->callTop  = 0;

    if ( !Clear_CodeRange( exec, TT_CodeRange_Glyph ) )
      return FAILURE;

    exec->top              = 0;
    exec->instruction_trap = FALSE;
    exec->pts.n            = 0;
 /* exec->GS               = Default_GraphicsState; */

    instance->glyph->numberOfContours = 0;

    return SUCCESS;
  }


#if 0

/*******************************************************************
 *
 *  Function    :  TT_Prepare_Glyph
 *
 *  Description :  Prepare instance data for instruction decoding.
 *
 *  Input  :  instance     the instance where the glyph data will be
 *                          stored
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  :  Must be used before calling run(), once the glyph
 *            has been loaded.
 *
 *  DEBUG  :  Currently cannot fail.
 *
 ******************************************************************/

  Bool  TT_Prepare_Glyph( PInstance_Record  instance )
  {
    instance->zp0 = instance->pts;
    instance->zp1 = instance->pts;
    instance->zp2 = instance->pts;

    return SUCCESS;
  }

#else

/*******************************************************************
 *
 *  Function    :  TT_Set_Glyph_Defaults
 *
 *  Description :  The values computed by the CVT program will be
 *                 set as the new glyph defaults.
 *
 *  Input  :  instance
 *
 *  Output :  none.
 *
 *  Notes  :  Must be used after executing the CVT program
 *
 ******************************************************************/

  void  TT_Set_Glyph_Defaults( PInstance_Record  instance )
  {
    PExecution_Context  exec;


    exec = (PExecution_Context)instance->exec;

    /* BULLSHIT : The specs stipulates that all values that are   */
    /*            set within the CVT program should be kept as    */
    /*            new glyph defaults. Unfortunately, all glyph    */
    /*            streams expect zp0..2 set to zone 1, while the  */
    /*            CVT program sets them to zone 0!                */
    /*            Moreover, the vectors must be set to the x-axis */

    exec->GS.gep0 = 1;
    exec->GS.gep1 = 1;
    exec->GS.gep2 = 1;

    exec->GS.projVector.x = 0x4000;
    exec->GS.projVector.y = 0;

    exec->GS.dualVector.x = 0x4000;
    exec->GS.dualVector.y = 0;

    exec->GS.freeVector.x = 0x4000;
    exec->GS.freeVector.y = 0;

    exec->GS.round_state = 1;   /* even rounding! */

    exec->zp0 = exec->pts;
    exec->zp1 = exec->pts;
    exec->zp2 = exec->pts;

 /* exec->default_GS = exec->GS;    Save the graphics state */

    /* that's all for now */
  }


/*******************************************************************
 *
 *  Function    :  TT_Reset_Glyph_Defaults
 *
 *  Description :  Resets glyph defaults generated by the
 *                 CVT program.
 *
 *  Input  :  instance
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  :  Must be used before executing a glyph's instructions.
 *
 ******************************************************************/

   Bool  TT_Glyph_Defaults( PInstance_Record  instance )
   {
     PExecution_Context  exec;


     exec = (PExecution_Context)instance->exec;

  /* exec->GS = exec->default_GS;    restore the graphics state */
 
     exec->zp0 = exec->pts;
     exec->zp1 = exec->pts;
     exec->zp2 = exec->pts;
 
     return SUCCESS;
   }
 
#endif


#if 0

/*******************************************************************
 *
 *  Function    :  TT_Reset_Instance
 *
 *  Description :  Resets instance data to load defaults.
 *
 *  Input  :  instance     the instance where the glyph data will be
 *                         stored
 *            ptSize       desired point size
 *            resolution   physical screen resolution, e.g. 96 dpi
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *  Notes  :  Must be called before executing the Font Program.
 *
 ******************************************************************/

  Bool  TT_Reset_Instance( PInstance_Record  instance,
                           Int ptSize,
                           Int resolution )
  {
    int i;


    TT_Reset_Glyph( instance );

    instance->numFDefs = instance->fontRes->maxProfile.maxFunctionDefs;
    instance->numIDefs = instance->fontRes->maxProfile.maxInstructionDefs;

    if( !Set_CodeRange( instance, (Int)TT_CodeRange_Font,
                        (void*)instance->fontRes->fontProgram,
                        instance->fontRes->fontPgmSize ) )
      return FAILURE;

    if( !Set_CodeRange( instance, (Int)TT_CodeRange_Cvt,
                        (void*)instance->fontRes->cvtProgram,
                        instance->fontRes->cvtPgmSize ) )
      return FAILURE;

    if ( !Clear_CodeRange( instance, (Int)TT_CodeRange_Glyph ) )
      return FAILURE;

    for ( i = 0; i < instance->storeSize; i++ )
      instance->storage[i] = 0;

    instance->period    = 0;
    instance->phase     = 0;
    instance->threshold = 0;
    
    /* Important note :                                     */
    /*                                                      */
    /*   Respecting the order of these computations is      */
    /*   important. The ppem must be computed first, and    */
    /*   truncated to an integer. It will then be used to   */
    /*   scale the rest of the glyphs.                      */
    /*                                                      */
    /*   I started with                                     */
    /*                                                      */
    /*     scale1  := ptSize * 64 * resolution              */
    /*     scale2  := 72 * fontHeader.Units_Per_EM          */
    /*                                                      */
    /*   which used to generate glyphs that were too high   */
    /*   by one pixel.                                      */

    instance->pointSize = ptSize * 64;
    instance->ppem = ptSize * resolution / 72;
    instance->scale1 = instance->ppem * 64;
    instance->scale2 = 
      (Long)instance->fontRes->fontHeader.Units_Per_EM;

    instance->Instruction_Trap = 0; /* was false DaveP */

    if ( instance->cvtSize > 0 )
      for ( i = 0; i < instance->cvtSize; i++ )
        instance->cvt[i] = instance->fontRes->cvt[i];

    /* Note that, apparently, rounding these values does not */
    /* work (it breaks a lot of symmetries in Arial).        */

    /* All twilight points are originally zero */

    for ( i = 0; i < instance->twilight.n; i++ )
    {
      instance->twilight.org_x[i] = 0;
      instance->twilight.org_y[i] = 0;
      instance->twilight.cur_x[i] = 0;
      instance->twilight.cur_y[i] = 0;
    }

    return SUCCESS;
  }

#endif /* 0 */

/*******************************************************************
 *
 *  Function    :  TT_Load_Instance_Data
 *
 *  Description :  Allocates memory for the instance tables.
 *
 *  Input  :  resident     resident table to look for
 *            Pinstance    pointer the instance which should be
 *                         initialized
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 ******************************************************************/

  Bool TT_Load_Instance_Data( PResident_Record  resident,
                              PInstance_Record* instance)
  {
    long              Size, i, l;
    PInstance_Record  pir;
    UShort            maxp, maxc;


    Size = Mem_Free();

    if ( !ALLOC( *instance, sizeof( TInstance_Record ) ) )
      return FAILURE;

    pir = *instance;

    pir->fontRes    = resident;
    pir->hint_glyph = 0;        /* default to no hinting */

    /* reserve glyph contour / points + phantom / data */

    maxp = MAX( resident->maxProfile.maxCompositePoints,
                resident->maxProfile.maxPoints );

    maxc = MAX( resident->maxProfile.maxCompositeContours,
                resident->maxProfile.maxContours );
                       
    if ( !ALLOC( pir->glyph, sizeof( TGlyph ) )                   ||
         !ALLOC_ARRAY( pir->endContours, maxc, UShort )           ||
         !ALLOC_ARRAY( pir->glyph->contours,
                       maxc,
                       TGlyphContour )                            ||
         !ALLOC_ARRAY( pir->glyph->instruct, 
                       resident->maxProfile.maxComponentElements, 
                       TGlyphInstruct )                           ||
         !ALLOC_ARRAY( pir->glyph->scales, 
                       resident->maxProfile.maxComponentElements,
                       TGlyphScale )                              )
      return FAILURE;

    /**** The following tables concern instruction decoding. Some of  ****/
    /**   these could be allocated on demand. However, we're still      **/
    /**   debugging and locate the allocation here.                     **/
    /**   See ttexec.h and ttexec.c for more details on execution       **/
    /**** contexts.                                                   ****/

    /* Reserve execution context */
    if ( !ALLOC( pir->exec, sizeof( TExecution_Context ) ) )
      return FAILURE;

    DebugTrace2( "exec allocated = %lx\n", (Long)pir->exec );

    /* Reserve glyph code range */
    if ( !ALLOC( pir->glyphIns, resident->maxProfile.maxSizeOfInstructions ) )
      return FAILURE;

    /* Reserve function and instruction definitions */
    if ( !ALLOC_ARRAY( pir->FDefs, 
                       resident->maxProfile.maxFunctionDefs,
                       TDefRecord )                             ||
         !ALLOC_ARRAY( pir->IDefs,
                       resident->maxProfile.maxInstructionDefs,
                       TDefRecord )                             )
      return FAILURE;

    pir->numFDefs = resident->maxProfile.maxFunctionDefs;
    pir->numIDefs = resident->maxProfile.maxInstructionDefs;

    /* Reserve call stack */
    if ( !ALLOC_ARRAY( pir->callStack, 32, TCallRecord ) )
      return FAILURE;

    pir->callSize = 32;

    /* Reserve storage area */
    if ( !ALLOC_ARRAY( pir->storage, 
                       resident->maxProfile.maxStorage,
                       Long ) )
      return FAILURE;

    pir->storeSize = resident->maxProfile.maxStorage;
    pir->stackSize = resident->maxProfile.maxStackElements;

    /* Reserve stack */
    if ( !ALLOC_ARRAY( pir->stack, pir->stackSize, Long ) )
      return FAILURE;

    /* Reserve point arrays */
    pir->pts.n = maxp;

    /* We should not forget to allocate the phantom points. */
    l = (maxp + 2) * sizeof ( TT_F26Dot6 );

    if ( !ALLOC( pir->pts.org_x, l ) ||
         !ALLOC( pir->pts.org_y, l ) ||
         !ALLOC( pir->pts.cur_x, l ) ||
         !ALLOC( pir->pts.cur_y, l ) ||
         !ALLOC( pir->pts.touch, pir->pts.n + 2 ) )
      return FAILURE;

    pir->twilight.n = resident->maxProfile.maxTwilightPoints;
    l = pir->twilight.n * sizeof ( TT_F26Dot6 );

    if( !ALLOC( pir->twilight.org_x, l ) ||
        !ALLOC( pir->twilight.org_y, l ) ||
        !ALLOC( pir->twilight.cur_x, l ) ||
        !ALLOC( pir->twilight.cur_y, l ) ||
        !ALLOC( pir->twilight.touch, pir->twilight.n ) )
      return FAILURE;

    /* All twilight points are originally zero */
    for ( i = 0; i < pir->twilight.n; i++ )
    {
      pir->twilight.org_x[i] = 0;
      pir->twilight.org_y[i] = 0;
      pir->twilight.cur_x[i] = 0;
      pir->twilight.cur_y[i] = 0;
    }

    /* Reserve Control Value Table */
    pir->cvtSize = resident->cvtSize;

    if ( !ALLOC_ARRAY( pir->cvt, pir->cvtSize, UShort ) )
      return FAILURE;

    Size -= Mem_Free();

    pir->totalSize = Size;

#ifdef TT_CONFIG_REENTRANT

    pir->stream = TT_Duplicate_File( stream, &pir->stream, &pir->error );

#else

    pir->stream = pir->fontRes->stream;

#endif /* REENTRANT */

    DebugTrace1( "instance loaded OK\n" );

    return SUCCESS;
  }


/*******************************************************************
 *
 *  Function    :  TT_Transform_Glyph
 *
 *  Description :  Transform glyph outline in a proper way.
 *
 *  Input  :  instance     target font instance
 *            aTransf       user supplied transformation (not used yet)
 *
 *  Output :  SUCCESS on success. FAILURE on error.
 *
 *****************************************************************/

/* #undef MulDiv */

  Int  TT_Transform_Glyph( PInstance_Record  instance )
  {
    UShort      i, j;
    TT_F26Dot6  x, y, nx, ny;
    TT_F26Dot6  x_offset, y_offset;

    PGlyphScales    glyScales;
    PGlyphContours  glyContours;

    PExecution_Context  exec;


    exec = (PExecution_Context)instance->exec;

    glyContours = instance->glyph->contours;
    glyScales   = instance->glyph->scales;

    for ( i = 0; i < instance->glyph->numberOfSubglyphs; i++ )
    {
      x_offset = glyScales[i].x_offset;
      y_offset = glyScales[i].y_offset;

      for ( j = glyScales[i].start; j <= glyScales[i].finish; j++ )
      {
        x = instance->pts.org_x[j];
        y = instance->pts.org_y[j];

        if ( glyScales[i].flags & WE_HAVE_A_2X2 )
        {
          nx = ( MUL_FIXED( x, glyScales[i].x_scale ) +
                 MUL_FIXED( y, glyScales[i].scale10 ) );

          ny = ( MUL_FIXED( x, glyScales[i].scale01 ) +
                 MUL_FIXED( y, glyScales[i].y_scale ) );
          x = nx;
          y = ny;
        }

        x += x_offset;
        y += y_offset;

        nx = MulDiv_Round( x, exec->scale1, exec->scale2 );
        ny = MulDiv_Round( y, exec->scale1, exec->scale2 );

        instance->pts.org_x[j] = instance->pts.cur_x[j] = nx;
        instance->pts.org_y[j] = instance->pts.cur_y[j] = ny;
      }
    }

    /* ugh... phantom points */
    for ( j = instance->pts.n - 2; j < instance->pts.n; j++ )
    {
      x = instance->pts.org_x[j];
      y = instance->pts.org_y[j];

      nx = MulDiv_Round( x, exec->scale1, exec->scale2 );
      ny = MulDiv_Round( y, exec->scale1, exec->scale2 );

      instance->pts.org_x[j] = instance->pts.cur_x[j] = nx;
      instance->pts.org_y[j] = instance->pts.cur_y[j] = ny;
    }

    j = instance->pts.n;

    /* These two lines are _very_ important and should not be changed! */
    instance->pts.cur_x[j - 2] = instance->pts.org_x[j - 2] & -64;
    instance->pts.cur_x[j - 1] = (instance->pts.org_x[j - 1] + 32) & -64;

    return SUCCESS;
  }


/* End */
