/*******************************************************************
 *
 *  ttraster.c                                                  1.2
 *
 *  The FreeType glyph rasterizer (body).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 *  NOTES:
 *
 *  This version supports the following:
 *
 *    - direct grayscaling
 *    - sub-banding
 *    - drop-out modes 4 and 5
 *    - second pass for complete drop-out control (bitmap only)
 *    - variable precision
 *
 *   Re-entrancy is _not_ planned.
 *
 *   Changes between 1.1 and 1.2:
 *
 *     - no more trace tables, now uses linked list to sort
 *       coordinates.
 *
 *     - reduced code size using function dispatch within a generic
 *       draw_sweep function.
 *
 *     - added variable precision for finer rendering at small ppems
 *
 *     - sorry, had to change the interface
 *
 *
 *   Note that its interface may change in the future.
 *
 ******************************************************************/

#include "ttraster.h"
#include "tttypes.h"

#include <string.h>

/* Bug fixes :                                                            */
/*                                                                        */
/*   11/16/96 : David : Added first grayscale support.                    */
/*   11/ 8/96 : David : Fixed 'Curve_To' from access bug                  */
/*   03/10/97 : David : Synchronized to Pascal version                    */
/*   03/13/97 : David : bug fix in gray rendering                         */

/* Define SECURE if you want to use the 'MulDiv' function from 'ttcalc'   */
/* (it computes (A*B)/C with 64 bits intermediate accuracy). However, for */
/* 99.9% of screen display, this operation can be done directly with the  */
/* good accuracy (because 'B' is only a 6 bits integer).                  */

/* Note that some compilers can manage directly 'a*b/c' with intermediate */
/* accuracy (GCC can use long longs, for example). Using the MACRO        */
/* definition of Muldiv would then be sufficient.                         */

/* The definition of SECURE should appear in the file "ttconfig.h".       */

#ifdef SECURE
#include "ttcalc.h"
#else
#define MulDiv( a, b, c )  ( (a) * (b) / (c) )
#endif

/* Define DEBUG_RASTER if you want to generate a debug version of the  */
/* rasterizer. This will progressively draw the glyphs while all the   */
/* computation are done directly on the graphics screen (the glyphs    */
/* will be inverted).                                                  */

/* Note that DEBUG_RASTER should only be used for debugging with b/w   */
/* rendering, not with gray levels.                                    */

/* The definition of DEBUG_RASTER should appear in the file "ttconfig.h". */

#ifdef DEBUG_RASTER
  extern char*  Vio;  /* A pointer to VRAM or display buffer */
#endif

/* The rasterizer is a very general purpose component, please leave */
/* the following redefinitions there (you never know your target    */
/* environment).                                                    */

#ifndef TRUE
#define TRUE   1
#endif

#ifndef FALSE
#define FALSE  0
#endif

#ifndef NULL
#define NULL  (void*)0
#endif

#define MaxBezier  32   /* The maximum number of stacked Bezier curves. */
                        /* Setting this constant to more than 32 is a   */
                        /* pure waste of space.                         */

#define Pixel_Bits  6   /* fractional bits of input coordinates */

  /* States of each line, arc and profile */
  typedef enum _TStates
  {
    Unknown,
    Ascending,
    Descending,
    Flat
  } TStates;

  struct _TProfile;
  typedef struct _TProfile  TProfile;
  typedef TProfile*         PProfile;

  struct _TProfile
  {                                                                     
    Int         flow;        /* Profile orientation: Asc/Descending      */
    Int         height;      /* profile's height in scanlines            */
    Int         start;       /* profile's starting scanline              */
    ULong       offset;      /* offset of profile's data in render pool  */

    PProfile    link;        /* link to next profile                     */

    TT_F26Dot6  X;           /* current coordinate during sweep          */
    Int         countL;      /* number of lines to step before this      */
                             /* profile becomes drawable                 */

    PProfile    next;        /* next profile in same contour, used       */
                             /* during drop-out control                  */
  };

  typedef struct _TProfileList /* simple structure to manage linked lists */
  {
    TProfile  head;
    TProfile  tail;
  } TProfileList;
  typedef TProfileList*  PProfileList;

  /* I use the classic trick of two dummy records for the head and tail  */
  /* of a linked list, this reduces tests in insertion/deletion/sorting. */
  /* NOTE: used during sweeps only.                                      */

  /* Simple record used to implement a stack of bands, required */
  /* by the sub-banding mechanism                               */
  typedef struct _TBand
  {
    Int  y_min;   /* band's minimum */
    Int  y_max;   /* band's maximum */
  } TBand;

#define AlignProfileSize  ( (sizeof ( TProfile ) + 3) / 4 )

  /* Left fill bitmask */
  static const unsigned char  LMask[8] = 
    { 0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01 };

  /* Right fill bitmask */
  static const unsigned char  RMask[8] =
    { 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF };

  /* prototypes used for sweep function dispatch */

  typedef void  Function_Sweep_Init( int*  min, int*  max );

  typedef void  Function_Sweep_Span( int         y,
                                     TT_F26Dot6  x1,
                                     TT_F26Dot6  x2,
                                     PProfile    left,
                                     PProfile    right );
                    
  typedef void  Function_Sweep_Step();


#ifndef CONST_PREC

  static Int   precision_bits;
  static Int   precision;
  static Int   precision_half;
  static Long  precision_mask;
  static Int   precision_shift;
  static Int   precision_step;

#else

  #define precision_bits   6
  #define precision        (1 << precision_bits)
  #define precision_half   (precision / 2)   
  #define precision_mask   (-precision)
  #define precision_shift  0
  #define precision_step   precision_half

#endif

/* NOTE : These operations are only valid on 2's complement processors */
#define FLOOR( x )    ( (x) & precision_mask )
#define CEILING( x )  ( ((x) + precision - 1) & precision_mask )
#define TRUNC( x )    ( (signed long)(x) >> precision_bits )
#define FRAC( x )     ( (x) & (precision - 1) )
#define SCALED( x )   ( (x) << precision_shift )

#ifdef DEBUG_RASTER
#define DEBUG_PSET  Pset()
#else
#define DEBUG_PSET  
#endif

  static PProfile  cProfile; /* current profile                           */
  static PProfile  fProfile; /* head of linked list of profiles           */
  static PProfile  gProfile; /* contour's first profile in case of impact */

  static Int       nProfs;   /* current number of profiles */

  static TStates   state;    /* rendering state */

  static Bool      fresh;    /* signals a fresh new profile which 'start'*/ 
                             /* field must be completed                  */

  static Bool      joint;    /* signals that le last arc ended exactly   */
                             /* on a scanline. Allows removal of doublets*/

  static PStorage  buff    = (PStorage)0;  /* The profiles buffer          */
  static Int       maxBuff = 0;            /* Profiles buffer size         */
  static Int       profCur = 0;            /* Current cursor in buffer     */

  static TRasterBlock  target;  /* description of target bit/pixmap */

  static Int       bWidth;  /* target bitmap width  */
  static PByte     bTarget; /* target bitmap buffer */
  static PByte     gTarget; /* target pixmap buffer */

  static TBand     band_stack[16];  /* band stack used for sub-banding */
  static Int       band_top;        /* band stack top                  */

  static Int       traceOfs;    /* current offset in target bitmap */
  static Int       traceG;      /* current offset in target pixmap */

  static Int       gray_min_x;  /* current min x during gray rendering */
  static Int       gray_max_x;  /* current max x during gray rendering */

  /* dispatch variables */

  static Function_Sweep_Init*  Proc_Sweep_Init;
  static Function_Sweep_Span*  Proc_Sweep_Span;
  static Function_Sweep_Span*  Proc_Sweep_Drop;
  static Function_Sweep_Step*  Proc_Sweep_Step;

  static struct { Long  x, y; }
    Arcs[2 * MaxBezier + 1];    /* The Bezier stack */

  static Int       curArc;      /* Bezier stack top */

  static PStorage  xCoord;  /* current xCoord table */
  static PStorage  yCoord;  /* current yCoord table */

  static PByte     flags;   /* current flags table    */
  static PShort    outs;    /* current outlines table */

  static Int       nPoints;   /* number of points in current glyph   */
  static Int       nContours; /* number of contours in current glyph */

  static Long      LastX, LastY, MinY, MaxY;

  static Byte      dropOutControl;   /* current drop_out control method */

  static Short     count_table[256]; /* Look-up table used to quickly count */
                                     /* set bits in a gray 2x2 cell         */

  static char      grays[5]; /* Palette of gray levels used for render */

  static Byte*     gray_lines; /* Intermediate table used to render the  */
                               /* graylevels pixmaps                     */
                               /* gray_lines is a buffer holding two     */
                               /* monochrome scanlines                   */

  static Int       gray_width; /* width in bytes of one monochrome       */
                               /* intermediate scanline of gray_lines.   */
                               /* Each gray pixel takes 2 bits long there*/

                     /* The gray_lines must hold 2 lines, thus with size */
                     /* in bytes of at least 'gray_width*2'              */

  static Bool      second_pass; /* indicates wether a horizontal pass      */
                                /* should be performed to control drop-out */
                                /* accurately when calling Render_Glyph.   */
                                /* Note that there is no horizontal pass   */
                                /* during gray rendering.                  */

  Int  Raster_Error = 0;


#ifdef DEBUG_RASTER

  /************************************************/
  /*                                              */
  /* Pset :                                       */
  /*                                              */
  /*  Used for debugging only. Plots a point      */
  /*  in VRAM during rendering (not afterwards).  */
  /*                                              */
  /* NOTE : This procedure relies on the value    */
  /*        of cProfile->start, which may not     */
  /*        be set when Pset is called sometimes  */
  /*        This will usually result in a dot     */
  /*        plotted on the first screen scanline  */
  /*        (far away its original position).     */
  /*                                              */
  /*        This "bug" reflects nothing wrong     */
  /*        in the current implementation, and    */
  /*        the bitmap is rendered correctly,     */
  /*        so don't panic if you see 'flying'    */
  /*        dots in debugging mode.               */
  /*                                              */
  /*  - David                                     */
  /*                                              */
  /************************************************/

  void  Pset()
  {
    Long  o;
    Long  x;

    x = buff[profCur];

    switch ( cProfile->flow )
    {
    case TT_Flow_Up:
      o = Vio_ScanLineWidth *
         ( profCur-cProfile->offset + cProfile->start ) +
         ( x / (precision*8) );
      break;

    case TT_Flow_Down:
      o = Vio_ScanLineWidth *
         ( cProfile->start-profCur + cProfile->offset ) +
         ( x / (precision*8) );
      break;
    }

    if ( o > 0 )
      Vio[o] |= (unsigned)0x80 >> ( (x/precision) & 7 );
  }


  void  Clear_Band( Int  y1, Int  y2 )
  {
    memset( Vio + y1*Vio_ScanLineWidth, (y2-y1+1)*Vio_ScanLineWidth, 0 );
  }

#endif /* DEBUG_RASTER */


#ifndef CONST_PREC

  /************************************************************************/
  /*                                                                      */
  /* Function:    Set_High_Precision                                      */
  /*                                                                      */
  /* Description: Sets precision variables according to param flag        */
  /*                                                                      */
  /* Input:       High     set to True for high precision (typically for  */
  /*                       ppem < 18), false otherwise.                   */
  /*                                                                      */
  /************************************************************************/

  void  Set_High_Precision( int  High )
  {
    second_pass = High;

    if ( High )
    {
      precision_bits = 10;
      precision_step = 4;
    }
    else
    {
      precision_bits = 6;
      precision_step = 32;
    }

    precision       = 1 << precision_bits;
    precision_half  = precision / 2;
    precision_shift = precision_bits - Pixel_Bits;
    precision_mask  = -precision;
  }

#endif


/****************************************************************************/
/*                                                                          */
/* Function:    New_Profile                                                 */
/*                                                                          */
/* Description: Creates a new Profile in the render pool                    */
/*                                                                          */
/* Input:       aState   state/orientation of the new Profile               */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE in case of overflow or of incoherent Profile.       */
/*                                                                          */
/****************************************************************************/

  Bool  New_Profile( TStates  aState )
  {
    if ( !fProfile )
    {
      cProfile  = (PProfile)( buff + profCur );
      fProfile  = cProfile;
      profCur  += AlignProfileSize;
    }

    if ( profCur >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    switch ( aState )
    {
    case Ascending:
      cProfile->flow = TT_Flow_Up;
      break;

    case Descending:
      cProfile->flow = TT_Flow_Down;
      break;

    default:
      Raster_Error = Raster_Err_Invalid;
      return FAILURE;
    }

    cProfile->start   = 0;
    cProfile->height  = 0;
    cProfile->offset  = profCur;
    cProfile->link    = (PProfile)0;
    cProfile->next    = (PProfile)0;

    if ( !gProfile ) gProfile = cProfile;

    state = aState;
    fresh = TRUE;
    joint = FALSE;

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    End_Profile                                                 */
/*                                                                          */
/* Description: Finalizes the current Profile.                              */
/*                                                                          */
/* Input:       None                                                        */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE in case of overflow or incoherency.                 */
/*                                                                          */
/****************************************************************************/

  Bool  End_Profile()
  {
    Int       h;
    PProfile  oldProfile;

    h = profCur - cProfile->offset;

    if ( h < 0 )
    {
      Raster_Error = Raster_Err_Neg_Height;
      return FAILURE;
    }

    if ( h > 0 )
    {
      oldProfile       = cProfile;
      cProfile->height = h;
      cProfile         = (PProfile)(buff + profCur );

      profCur         += AlignProfileSize;

      cProfile->height = 0;
      cProfile->offset = profCur;
      oldProfile->next = cProfile;
      nProfs++;
    }

    if ( profCur >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    joint = FALSE;

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Finalize_Profile_Table                                      */
/*                                                                          */
/* Description: Adjusts all links in the Profiles list                      */
/*                                                                          */
/* Input:       None                                                        */
/*                                                                          */
/* Returns:     None.                                                       */
/*                                                                          */
/****************************************************************************/

  void  Finalize_Profile_Table()
  {
    Int       n;
    PProfile  p;

    n = nProfs;

    if ( n > 1 )
    {
      p = fProfile;
      while ( n > 1 )
      {
        p->link = (PProfile)( buff + p->offset + p->height );
        p       = p->link;
        n--;
      }
      p->link = NULL;
    }
    else
      fProfile = NULL;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Split_Bezier                                                */
/*                                                                          */
/* Description: Subdivides one Bezier arc into two joint                    */
/*              sub-arcs in the Bezier stack.                               */
/*                                                                          */
/* Input:       None (subdivided bezier is taken from the top of the        */
/*              stack)                                                      */
/*                                                                          */
/* Returns:     None.                                                       */
/*                                                                          */
/****************************************************************************/

  void  Split_Bezier()
  {

#if 1

    /* This is the previous version, untouched */

    Arcs[curArc+4].x = Arcs[curArc+2].x;
    Arcs[curArc+4].y = Arcs[curArc+2].y;

    Arcs[curArc+3].x = ( Arcs[curArc+2].x + Arcs[curArc+1].x ) / 2;
    Arcs[curArc+3].y = ( Arcs[curArc+2].y + Arcs[curArc+1].y ) / 2;

    Arcs[curArc+1].x = ( Arcs[ curArc ].x + Arcs[curArc+1].x ) / 2;
    Arcs[curArc+1].y = ( Arcs[ curArc ].y + Arcs[curArc+1].y ) / 2;

    Arcs[curArc+2].x = ( Arcs[curArc+3].x + Arcs[curArc+1].x ) / 2;
    Arcs[curArc+2].y = ( Arcs[curArc+3].y + Arcs[curArc+1].y ) / 2;

#else

    /*** better rounding                                      ****/
    /***                                                      ****/
    /*** I have changed the version we kept in comments there ****/
    /*** before (as it was miscompiled under gcc 2.7.2 with   ****/
    /*** -O2). Please check it now                            ****/
    /***                                         - David      ****/

    Long  x1, y1, x2, y2;

    x1 = Arcs[curArc+2].x;    y1 = Arcs[curArc+2].y;
    x2 = Arcs[ curArc ].x;    y2 = Arcs[ curArc ].y;

    Arcs[curArc+4].x = x1;    Arcs[curArc+4].y = y1;

    x1 += Arcs[curArc+1].x;   y1 += Arcs[curArc+1].y;
    x2 += Arcs[curArc+1].x;   y2 += Arcs[curArc+1].y;

    Arcs[curArc+3].x = (x1+1) / 2;    Arcs[curArc+3].y = (y1+1) / 2;
    Arcs[curArc+1].x = (x2+2) / 2;    Arcs[curArc+1].y = (y2+1) / 2;

    Arcs[curArc+2].x = (x1+x2+2) / 4;
    Arcs[curArc+2].y = (y1+y2+2) / 4;

#endif

    curArc += 2;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Push_Bezier                                                 */
/*                                                                          */
/* Description: Clears the Bezier stack and pushes a new Arc on top of it.  */
/*                                                                          */
/* Input:       x1,y1 x2,y2 x3,y3  new Bezier arc                           */
/*                                                                          */
/* Returns:     None.                                                       */
/*                                                                          */
/****************************************************************************/

  void  Push_Bezier( Long  x1, Long  y1,
                     Long  x2, Long  y2,
                     Long  x3, Long  y3 ) 
  {
    curArc = 0;
    Arcs[curArc+2].x = x1;  Arcs[curArc+2].y = y1;
    Arcs[curArc+1].x = x2;  Arcs[curArc+1].y = y2;
    Arcs[ curArc ].x = x3;  Arcs[ curArc ].y = y3;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Line_Up                                                     */
/*                                                                          */
/* Description: Compute the x-coordinates of an ascending line segment      */
/*              and stores them in the render pool.                         */
/*                                                                          */
/* Input:       x1,y1,x2,y2  Segment start (x1,y1) and end (x2,y2) points   */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow.                            */
/*                                                                          */
/****************************************************************************/

  Bool  Line_Up( Long  x1, Long  y1, Long  x2, Long  y2 )
  {
    Long  Dx, Dy;
    Int   e1, e2, f1, f2, size;
    Long  Ix, Rx, Ax;

    Dx = x2 - x1; Dy = y2 - y1;

    if ( Dy <= 0 || y2 < MinY || y1 > MaxY ) return SUCCESS;

    if ( y1 < MinY )
    {
      x1 += MulDiv( Dx, MinY - y1, Dy );
      e1  = TRUNC( MinY );
      f1  = 0;
    }
    else
    {
      e1 = TRUNC( y1 );
      f1 = FRAC( y1 );
    }

    if ( y2 > MaxY )
    {
      /* x2 += MulDiv( Dx, MaxY-y2, Dy );  UNNECESSARY */
      e2  = TRUNC( MaxY );
      f2  = 0;
    }
    else
    {
      e2 = TRUNC( y2 );
      f2 = FRAC( y2 );
    }

    if ( f1 > 0 )
      if ( e1 == e2 ) return SUCCESS;
      else
      {
        x1 += MulDiv( Dx, precision - f1, Dy );
        e1 += 1;
      }
    else
      if ( joint )
      {
        profCur--;
        joint = FALSE;
      }

    joint = ( f2 == 0 );

    if ( fresh )
    {
      cProfile->start = e1;
      fresh           = FALSE;
    }

    size = e2 - e1 + 1;
    if ( profCur + size >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    if ( Dx > 0 )
    {
      Ix = (precision*Dx) / Dy;
      Rx = (precision*Dx) % Dy;
      Ax = 0;
      Dx = 1;
    }
    else
    {
      Ix = -( (precision*-Dx) / Dy );
      Rx =    (precision*-Dx) % Dy;
      Ax =  0;
      Dx = -1;
    }

    while ( size > 0 ) 
    {
      buff[profCur] = x1;

      DEBUG_PSET;

      profCur++;

      x1 += Ix;
      Ax += Rx;
      if ( Ax >= Dy )
      {
        Ax -= Dy;
        x1 += Dx;
      }
      size--;
    }

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Line_Down                                                   */
/*                                                                          */
/* Description: Compute the x-coordinates of a descending line segment      */
/*              and stores them in the render pool.                         */
/*                                                                          */
/* Input:       x1,y1,x2,y2  Segment start (x1,y1) and end (x2,y2) points   */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow.                            */
/*                                                                          */
/****************************************************************************/

  Bool  Line_Down( Long  x1, Long  y1, Long  x2, Long  y2 )
  {
    Long  Dx, Dy;
    Int   e1, e2, f1, f2, size;
    Long  Ix, Rx, Ax;

    Dx = x2 - x1; Dy = y2 - y1;

    if ( Dy >= 0 || y1 < MinY || y2 > MaxY ) return SUCCESS;

    if ( y1 > MaxY )
    {
      x1 += MulDiv( Dx, MaxY - y1, Dy );
      e1  = TRUNC( MaxY );
      f1  = 0;
    }
    else
    {
      e1 = TRUNC( y1 );
      f1 = FRAC( y1 );
    }

    if ( y2 < MinY )
    {
      /* x2 += MulDiv( Dx, MinY - y2, Dy ); UNNECESSARY */
      e2  = TRUNC( MinY );
      f2  = 0;
    }
    else
    {
      e2 = TRUNC( y2 );
      f2 = FRAC( y2 );
    }

    if ( f1 > 0 )
      x1 += MulDiv( Dx, -f1, Dy );
    else
      if ( joint )
      {
        profCur--;
        joint = FALSE;
      }

    if ( f2 > 0 )
      if ( e1 == e2 )
        return SUCCESS;
      else
      {
        /* x2 += MulDiv( Dx, precision - f2, Dy ); unnecessary */
        e2 += 1;
      }
    else
      joint = TRUE;

    if ( fresh )
    {
      cProfile->start = e1;
      fresh           = FALSE;
    }

    size = e1 - e2 + 1;
    if ( profCur + size >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    if ( Dx > 0 )
    {
      Ix = (precision*Dx) / -Dy;
      Rx = (precision*Dx) % -Dy;
      Ax = 0;
      Dx = 1;
    }
    else
    {
      Ix = -( (precision*-Dx) / -Dy );
      Rx =    (precision*-Dx) % -Dy;
      Ax = 0;
      Dx = -1;
    }

    Dy = -Dy;

    while (size > 0)
    {
      buff[profCur] = x1;

      DEBUG_PSET;

      profCur++;

      x1 += Ix;
      Ax += Rx;
      if ( Ax >= Dy )
      {
        Ax -= Dy;
        x1 += Dx;
      }
      size--;
    }

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Bezier_Up                                                   */
/*                                                                          */
/* Description: Compute the x-coordinates of an ascending bezier arc        */
/*              and store them in the render pool.                          */
/*                                                                          */
/* Input:       None. The arc is taken from the top of the Bezier stack.    */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow.                            */
/*                                                                          */
/****************************************************************************/

  Bool  Bezier_Up()
  {
    Long  x1, y1, x2, y2, e, e2, e0;
    Int   debArc, f1;

    y1 = Arcs[curArc+2].y;
    y2 = Arcs[ curArc ].y;

    if ( y2 < MinY || y1 > MaxY )
    {
      curArc -= 2;
      return SUCCESS;
    }
    
    e2 = FLOOR( y2 );

    if ( e2 > MaxY ) e2 = MaxY;

    e0 = MinY;

    if ( y1 < MinY )
      e = MinY;
    else
    {
      e  = CEILING( y1 );
      f1 = FRAC   ( y1 );
      e0 = e;

      if ( f1 == 0 )
      {
        if ( joint )
        {
          profCur--;
          joint = FALSE;
        }

        buff[profCur] = Arcs[curArc+2].x;

        DEBUG_PSET;

        profCur++;

        e += precision;
      }
    }

    if ( fresh )
    {
      cProfile->start = TRUNC( e0 );
      fresh = FALSE;
    }

    /* Table overflow ? */
    if ( ( profCur + TRUNC( e2 - e ) + 1 ) >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    debArc = curArc;

    while ( curArc >= debArc && e <= e2 )
    {
      joint = FALSE;

      y2 = Arcs[curArc].y;

      if ( y2 == e )
      {
        joint = TRUE;
        buff[profCur] = Arcs[curArc].x;

        DEBUG_PSET;

        profCur++;

      /* Note on table overflow:                         */
      /*                                                 */
      /* We already know that profCur < MaxBuff here,    */
      /* so there is room for at least 1 coordinate      */
      /* and we don't need to test overflow there!       */
      /*                                                 */

        e += precision;
        curArc -= 2;
      }
      else
        if ( y2 < e )
          curArc -= 2;
        else
        {
          y1 = Arcs[curArc+2].y;
          if ( ( y2 - y1 ) < precision_step )
          {
            x1 = Arcs[curArc+2].x;
            x2 = Arcs[ curArc ].x;
            buff[profCur] = x1 + MulDiv( x2 - x1, e - y1, y2 - y1 );

            DEBUG_PSET;

            profCur++;

            curArc -= 2;
            e += precision;
          }
          else
            Split_Bezier();
        }
    }

    curArc = debArc - 2;
    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Bezier_Down                                                 */
/*                                                                          */
/* Description: Compute the x-coordinates of a descending bezier arc        */
/*              and store them in the render pool.                          */
/*                                                                          */
/* Input:       None. Arc is taken from the top of the Bezier stack.        */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow.                            */
/*                                                                          */
/****************************************************************************/

  Bool  Bezier_Down()
  {
    Long  x1, y1, x2, y2, e, e2, e0;
    Int   debArc, f1;

    y1 = Arcs[curArc+2].y;
    y2 = Arcs[ curArc ].y;

    if ( y1 < MinY || y2 > MaxY )
    {
      curArc -= 2;
      return SUCCESS;
    }
    
    e2 = CEILING( y2 );

    if ( e2 < MinY ) e2 = MinY;

    e0 = MaxY;

    if ( y1 > MaxY ) e = MaxY;
    else
    {
      e  = FLOOR( y1 );
      f1 = FRAC ( y1 );
      e0 = e;

      if ( f1 == 0 )
      {
        if ( joint )
        {
          profCur--;
          joint = FALSE;
        }

        buff[profCur] = Arcs[curArc+2].x;

        DEBUG_PSET;

        profCur++;

      /* Note on table overflow:                         */
      /*                                                 */
      /* We already know that profCur < MaxBuff here,    */
      /* so there is room for at least 1 coordinate      */
      /* and we don't need to test overflow there!       */
      /*                                                 */

        e -= precision;
      }
    }

    if ( fresh )
    {
      cProfile->start = TRUNC( e0 );
      fresh = FALSE;
    }

    /* Table overflow ? */
    if ( ( profCur + TRUNC( e - e2 ) + 1 ) >= maxBuff )
    {
      Raster_Error = Raster_Err_Overflow;
      return FAILURE;
    }

    debArc = curArc;

    while ( curArc >= debArc && e >= e2 )
    {
      joint = FALSE;

      y2 = Arcs[curArc].y;

      if ( y2 == e )
      {
        joint = TRUE;
        buff[profCur] = Arcs[curArc].x;

        DEBUG_PSET;

        profCur++;

        e -= precision;
        curArc -= 2;
      }
      else
        if ( y2 > e )
          curArc -= 2;
        else
        {
          y1 = Arcs[curArc+2].y;
          if ( y1 - y2 < precision_step )
          {
            x1 = Arcs[curArc+2].x;
            x2 = Arcs[ curArc ].x;

            buff[profCur] = x1 + MulDiv( x2 - x1, e - y1, y2 - y1 );

            DEBUG_PSET;

            profCur++;

            curArc -= 2;
            e      -= precision;
          }
          else
            Split_Bezier();
       }
    }

    curArc = debArc - 2;
    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Line_To                                                     */
/*                                                                          */
/* Description: Inject a new line segment and adjust Profiles list.         */
/*                                                                          */
/* Input:       x, y : segment endpoint (start point in LastX,LastY)        */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow or Incorrect Profile.       */
/*                                                                          */
/****************************************************************************/
  
  Bool  Line_To( Long  x, Long  y )
  {
    switch ( state )
    {
    case Unknown:
      if ( y > LastY )
      {
        if ( !New_Profile( Ascending ) ) return FAILURE;
      }
      else
      {
        if ( y < LastY )
          if ( !New_Profile( Descending ) ) return FAILURE;
      }
      break;

    case Ascending:
      if ( y < LastY )
      {
        if ( !End_Profile() ||
             !New_Profile( Descending ) ) return FAILURE;
      }
      break;

    case Descending:
      if ( y > LastY )
      {
        if ( !End_Profile() ||
             !New_Profile( Ascending ) ) return FAILURE;
      }
      break;

    default:
      ;
    }

    switch ( state )
    {
    case Ascending:
      if ( !Line_Up ( LastX, LastY, x, y ) ) return FAILURE;
        break;

    case Descending:
      if ( !Line_Down( LastX, LastY, x, y ) ) return FAILURE;
        break;

    default:
      ;
    }

    LastX = x;
    LastY = y;

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Bezier_To                                                   */
/*                                                                          */
/* Description: Inject a new bezier arc and adjust Profiles list.           */
/*                                                                          */
/* Input:       x,   y : arc endpoint (start point in LastX, LastY)         */
/*              Cx, Cy : control point                                      */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow or Incorrect Profile.       */
/*                                                                          */
/****************************************************************************/

  Bool  Bezier_To( Long  x, Long  y, Long  cx, Long  cy )
  {
    Long     y1, y2, y3, x3;
    TStates  state_bez;

    Push_Bezier( LastX, LastY, cx, cy, x, y );

    do
    {
      y1 = Arcs[curArc+2].y;
      y2 = Arcs[curArc+1].y;
      y3 = Arcs[ curArc ].y;
      x3 = Arcs[ curArc ].x;

      if ( y1 == y2 )
      {
        if ( y2 == y3 )
          state_bez = Flat;
        else if ( y2 > y3 )
          state_bez = Descending;
        else
          state_bez = Ascending;
      }
      else
      if ( y1 > y2 )
      {
        if ( y2 >= y3 )
          state_bez = Descending;
        else
          state_bez = Unknown;
      }
      else
        if ( y2 <= y3 )
          state_bez = Ascending;
        else
          state_bez = Unknown;

      switch ( state_bez )
      {
      case Flat:
        curArc -= 2;
        break;

      case Unknown:
        Split_Bezier();
        break;

      default:
        if ( state != state_bez )
        {
          if ( state != Unknown )
            if ( !End_Profile() ) return FAILURE;

          if ( !New_Profile( state_bez ) ) return FAILURE;
        }

        switch ( state )
        {
        case Ascending:
          if ( !Bezier_Up  () ) return FAILURE;
          break;

        case Descending:
          if ( !Bezier_Down() ) return FAILURE;
          break;

        default:
          ;
        }
      }
    } while ( curArc >= 0 );

    LastX = x3;
    LastY = y3;

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Curve_To                                                    */
/*                                                                          */
/* Description: Injects several following Bezier arcs.                      */
/*                                                                          */
/* Input:       x, y : arc endpoint (start point in LastX, LastY)           */
/*                                                                          */
/*              firstCtrl, lastCtrlint : first and last control point       */
/*                                       index.                             */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE on Render Pool overflow or Incorrect Profile.       */
/*                                                                          */
/****************************************************************************/

  Bool  Curve_To( Long  x, Long  y, Int  firstCtrl, Int  lastCtrl )
  {
    Long  xz, yz, cx, cy;

    xz = SCALED( xCoord[firstCtrl] );
    yz = SCALED( yCoord[firstCtrl] );

    firstCtrl++;

    while ( firstCtrl <= lastCtrl )
    {
      cx = ( xz + SCALED( xCoord[firstCtrl] ) ) / 2;
      cy = ( yz + SCALED( yCoord[firstCtrl] ) ) / 2;

      if ( !Bezier_To( cx, cy, xz, yz ) ) return FAILURE;

      xz = SCALED( xCoord[firstCtrl] );
      yz = SCALED( yCoord[firstCtrl] );

      firstCtrl++;
    }

    return Bezier_To( x, y, xz, yz );
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Convert_Glyph                                               */
/*                                                                          */
/* Description: Convert a glyph into a series of segments and arcs          */
/*              and make a Profiles list with them.                         */
/*                                                                          */
/* Input:       _xCoord, _yCoord : coordinates tables.                      */
/*                                                                          */
/*              Uses the 'Flag' table too.                                  */
/*                                                                          */
/* Returns:     SUCCESS on success                                          */
/*              FAILURE if any error was encountered during rendering.      */
/*                                                                          */
/****************************************************************************/

 Bool  Convert_Glyph( PStorage  _xCoord, PStorage  _yCoord )
 {
   Int       i, j, first, last, start;

   PProfile  lastProfile;

   j        = 0;
   fProfile = NULL;
   joint    = FALSE;
   fresh    = FALSE;
   last     = 0;

   xCoord = _xCoord;
   yCoord = _yCoord;

   cProfile         = (PProfile)( buff + profCur );
   cProfile->offset = profCur;
   nProfs           = 0;

   for ( i = 0; i < nContours; i++ )
   {
     state    = Unknown;
     first    = j;
     LastX    = SCALED( xCoord[j] );
     LastY    = SCALED( yCoord[j] );
     start    = 0;
     gProfile = NULL;

     j++;

     while ( j <= outs[i] )
     {
       if ( (flags[j] & 1) == 0 )   /* OFF Curve */
         if ( start == 0 )
         {
           start = j;
           last  = j;
         }
         else
           last++;
       else                         /* ON Curve */
         if ( start != 0 )
         {
           if ( !Curve_To( SCALED( xCoord[j] ), 
                           SCALED( yCoord[j] ),
                           start,
                           last ) )
             return FAILURE;
           start = 0;
         }
         else
           if ( !Line_To( SCALED( xCoord[j] ),
                          SCALED( yCoord[j] ) ) )
             return FAILURE;

       j++;
     }

     if ( start != 0 )
     {
       if ( !Curve_To( SCALED( xCoord[first] ),
                       SCALED( yCoord[first] ),
                       start,
                       last ) )
         return FAILURE;
     }
     else
       if ( !Line_To( SCALED( xCoord[first] ),
                      SCALED( yCoord[first] ) ) )
         return FAILURE;

     /* We must now see if the extreme arcs join or not */

     if ( ( FRAC( LastY ) == 0 &&
            LastY >= MinY      &&
            LastY <= MaxY ) )
       if ( gProfile && gProfile->flow == cProfile->flow )
         profCur--;
       /* Note that gProfile can be nil if the contour was too small */
       /* to be drawn.                                               */

     lastProfile = cProfile;

     if ( !End_Profile() ) return FAILURE;

     if ( gProfile ) lastProfile->next = gProfile;
   }

   Finalize_Profile_Table();

   return SUCCESS;
 }


/************************************************/
/*                                              */
/*  Init_Linked                                 */
/*                                              */
/*    Init an empty linked list.                */
/*                                              */
/************************************************/

  void  Init_Linked( TProfileList*  l )
  {
    l->head.link = &l->tail;
    l->tail.link = &l->tail;
    l->tail.X    = 0x7FFFFFFF;
  }


/************************************************/
/*                                              */
/*  InsNew :                                    */
/*                                              */
/*    Inserts a new Profile in a linked list.   */
/*                                              */
/************************************************/

  void  InsNew( PProfileList  list,
                PProfile      profile )
  {
    PProfile  old, current;

    old     = &list->head;
    current =  old->link;

    for ( ;; )
    {
      if ( profile->X < current->X )
      {
        profile->link = current;
        old    ->link = profile;
        return;
      }

      old     = current;
      current = current->link;
    }
  }


/************************************************/
/*                                              */
/*  DelOld :                                    */
/*                                              */
/*    Removes an old Profile from a linked list */
/*                                              */
/************************************************/

  void  DelOld( PProfileList  list,
                PProfile      profile )
  {
    PProfile  old, current, next;

    old     = &list->head;
    current =  old->link;
    next    =  current->link;

    while ( next != current )
    {
      if ( current == profile )
      {
        old->link = next;
        return;
      }

      old     = current;
      current = next;
      next    = current->link;
    }

    /* we should never get there, unless the Profile was not part of */
    /* the list.                                                     */
  }


/************************************************/
/*                                              */
/*  Sort :                                      */
/*                                              */
/*    Sorts 'quickly' (?) a trace list.         */
/*                                              */
/************************************************/

  void  Sort( PProfileList  list )
  {
    PProfile  old, current, next;

    /* First, set the new X coordinate of each profile */

    current = list->head.link;
    next    = current->link;

    while ( current != next )
    {
      current->X       = buff[ current->offset ];
      current->offset += current->flow;
      current->height--;

      current = next;
      next    = current->link;
    }

    /* Then sort them */

    old     = &list->head;
    current =  old->link;
    next    =  current->link;

    while ( current != next )
    {
      if ( current->X > next->X )
      {
        old->link     = next;
        current->link = next->link;
        next->link    = current;

        old     = &list->head;
        current =  old->link;
      }
      else
      {
        old     = current;
        current = next;
      }

      next = current->link;
    }
  }


/***********************************************************************/
/*                                                                     */
/*  Vertical Sweep Procedure Set :                                     */
/*                                                                     */
/*  These three routines are used during the vertical black/white      */
/*  sweep phase by the generic Draw_Sweep function.                    */
/*                                                                     */
/***********************************************************************/

  void  Vertical_Sweep_Init( int*  min, int*  max )
  {
    traceOfs   = *min * target.cols;
    gray_min_x = 0;
    gray_max_x = 0;
  }


  void Vertical_Sweep_Span( int         y,
                            TT_F26Dot6  x1,
                            TT_F26Dot6  x2,
                            PProfile    left,
                            PProfile    right )
  {
    Long  e1, e2;
    Int   c1, c2;
    Int   f1, f2;
    Int   j;

    /* Drop-out control */

    e1 = TRUNC( CEILING( x1 ) );
    e2 = TRUNC( FLOOR  ( x2 ) );
 
    if ( e2 >= 0 && e1 < bWidth )
    {
      if ( e1 < 0 )       e1 = 0;
      if ( e2 >= bWidth ) e2 = bWidth-1;
 
      c1 = e1 >> 3;
      c2 = e2 >> 3;
 
      f1 = e1 & 7;
      f2 = e2 & 7;
 
      if ( gray_min_x > c1 ) gray_min_x = c1;
      if ( gray_max_x < c2 ) gray_max_x = c2;

      j = traceOfs + c1;
 
      if ( c1 == c2 )
        bTarget[j] |= ( LMask[f1] & RMask[f2] );
      else
      {
        bTarget[j] |= LMask[f1];
        
        if ( c2 > c1+1 )
          memset( bTarget + j + 1, 0xFF, c2 - c1 - 1 );
        
        j += c2-c1;
 
        bTarget[j] |= RMask[f2];
      }
    }
  }


  void Vertical_Sweep_Drop( int         y,
                            TT_F26Dot6  x1,
                            TT_F26Dot6  x2,
                            PProfile    left,
                            PProfile    right )
  {
    Long  e1, e2;
    Int   c1, f1;
    Int   j;

    /* Drop-out control */

    e1 = CEILING( x1 );
    e2 = FLOOR  ( x2 );

    if ( e1 > e2 )
    {
      if ( e1 == e2 + precision )
      {
        switch ( dropOutControl )
        {
        case 1:
          e1 = e2;
          break;

        case 4:
          e1 = CEILING( (x1+x2+1) / 2 );
          e2 = e1;
          break;

        case 2:
        case 5:
          /* Drop-out Control Rule #4 */
  
          /* The spec is not very clear regarding rule #4. It       */
          /* presents a method that is way too costly to implement  */
          /* while the general idea seems to get rid of 'stubs'.    */
          /*                                                        */
          /* Here, we only get rid of stubs recognized when :       */
          /*                                                        */
          /*  upper stub :                                          */
          /*                                                        */
          /*   - P_Left and P_Right are in the same contour         */
          /*   - P_Right is the successor of P_Left in that contour */
          /*   - y is the top of P_Left and P_Right                 */
          /*                                                        */
          /*  lower stub :                                          */
          /*                                                        */
          /*   - P_Left and P_Right are in the same contour         */
          /*   - P_Left is the successor of P_Right in that contour */
          /*   - y is the bottom of P_Left                          */
          /*                                                        */

          /* upper stub test */

          if ( left->next == right && left->height <= 0 ) return;

          /* lower stub test */

          if ( right->next == left && left->start == y ) return;

          /* check that the rightmost pixel isn't set */

          e1 = TRUNC( e1 );

          c1 = e1 >> 3;
          f1 = e1 &  7;

          if ( e1 >= 0 && e1 < bWidth &&
               bTarget[traceOfs + c1] & (0x80 >> f1) ) return;

		  if ( dropOutControl == 2 )
		    e1 = e2;
		  else
		    e1 = CEILING( (x1+x2+1)/2 );
		    
          break;

        default:
          return;  /* unsupported mode */
        }
      }
      else
        return;
    }
    else
      e2 = e1;

    e1 = TRUNC( e1 );
 
    if ( e1 >= 0 && e1 < bWidth )
    {
      c1 = e1 >> 3;
      f1 = e1 & 7;

      if ( gray_min_x > c1 ) gray_min_x = c1;
      if ( gray_max_x < c1 ) gray_max_x = c1;

      j = traceOfs + c1;

      bTarget[j] |= (0x80 >> f1);
    }
  }


  void Vertical_Sweep_Step()
  {
    traceOfs += target.cols;
  }


/***********************************************************************/
/*                                                                     */
/*  Horizontal Sweep Procedure Set :                                   */
/*                                                                     */
/*  These three routines are used during the horizontal black/white    */
/*  sweep phase by the generic Draw_Sweep function.                    */
/*                                                                     */
/***********************************************************************/

  void  Horizontal_Sweep_Init( int*  min, int*  max )
  {
    /* nothing, really */
  }


  void Horizontal_Sweep_Span( int         y,
                              TT_F26Dot6  x1,
                              TT_F26Dot6  x2,
                              PProfile    left,
                              PProfile    right )
  {
    Long  e1, e2;
    Int   c1;
    Int   f1;

    /* During the horizontal sweep, we only take care of drop-outs */

    e1 = CEILING( x1 );
    e2 = FLOOR  ( x2 );

    c1 = y >> 3;
    f1 = y  & 7;

    e1 = TRUNC( e1 );

    if ( e1 >= 0 && e1 < target.rows )
      bTarget[c1 + e1*target.cols] |= (0x80 >> f1);

    e2 = TRUNC( e2 );

    if ( e2 >= 0 && e2 < target.rows )
      bTarget[c1 + e2*target.cols] |= (0x80 >> f1);
  }


  void Horizontal_Sweep_Drop( int         y,
                              TT_F26Dot6  x1,
                              TT_F26Dot6  x2,
                              PProfile    left,
                              PProfile    right )
  {
    Long  e1, e2;
    Int   c1;
    Int   f1;

    Int   j;

    /* During the horizontal sweep, we only take care of drop-outs */

    e1 = CEILING( x1 );
    e2 = FLOOR  ( x2 );

    if ( e1 > e2 )
    {
      if ( e1 == e2 + precision )
      {
        switch ( dropOutControl )
        {
        case 1:
          e1 = e2;
          break;

        case 4:
          e1 = CEILING( (x1+x2+1) / 2 );
          break;

        case 2:
        case 5:

          /* Drop-out Control Rule #4 */
  
          /* The spec is not very clear regarding rule #4. It       */
          /* presents a method that is way too costly to implement  */
          /* while the general idea seems to get rid of 'stubs'.    */
          /*                                                        */

          /* rightmost stub test */

          if ( left->next == right && left->height <= 0 ) return;

          /* leftmost stub test */

          if ( right->next == left && left->start == y ) return;

          /* check that the rightmost pixel isn't set */

          e1 = TRUNC( e1 );

          c1 = y >> 3;
          f1 = y &  7;

          j = c1 + e1 * target.cols;

          if ( e1 >= 0 && e1 < target.rows &&
               bTarget[j] & (0x80 >> f1) ) return;

          if ( dropOutControl == 2 )
            e1 = e2;
          else
            e1 = CEILING( (x1+x2+1) / 2 );

          break;

        default:
          return;  /* unsupported mode */
        }
      }
      else
        return;
    }

    c1 = y >> 3;
    f1 = y  & 7;

    e1 = TRUNC( e1 );

    if ( e1 >= 0 && e1 < target.rows )
      bTarget[c1 + e1*target.cols] |= (0x80 >> f1);
  }


  void Horizontal_Sweep_Step()
  {
    /* Nothing, really */
  }


/***********************************************************************/
/*                                                                     */
/*  Vertical Gray Sweep Procedure Set :                                */
/*                                                                     */
/*  These two routines are used during the vertical gray-levels        */
/*  sweep phase by the generic Draw_Sweep function.                    */
/*                                                                     */
/*                                                                     */
/*  NOTES :                                                            */
/*                                                                     */
/*  - The target pixmap's width *must* be a multiple of 4.             */
/*                                                                     */
/*  - you have to use the function Vertical_Sweep_Span for             */
/*    the gray span call.                                              */
/*                                                                     */
/***********************************************************************/

  void  Gray_Sweep_Init( int*  min, int*  max )
  {
    *min = *min & -2;
    *max = ( *max+3 ) & -2;

    traceOfs = 0;
    traceG   = target.cols * ( *min / 2 );

    gray_min_x =  target.cols;
    gray_max_x = -target.cols;
  }


  void Gray_Sweep_Step()
  {
    Int j, c1, c2;

    traceOfs += gray_width;

    if ( traceOfs > gray_width )
    {
      j = traceG + gray_min_x * 4;

      if ( gray_max_x >= 0 )
      {
        if ( gray_max_x >= target.width ) gray_max_x = target.width - 1;

        if ( gray_min_x < 0 ) gray_min_x = 0;
 
        for ( c1 = gray_min_x; c1 <= gray_max_x; c1++ )
        {
          c2 = count_table[ bTarget[c1           ] ] + 
               count_table[ bTarget[c1+gray_width] ];
 
          if (c2)
          {
            gTarget[j++] = grays[(unsigned)(c2 & 0xF000) >> 12];
            gTarget[j++] = grays[(unsigned)(c2 & 0x0F00) >>  8];
            gTarget[j++] = grays[(unsigned)(c2 & 0x00F0) >>  4];
            gTarget[j++] = grays[(unsigned)(c2 & 0x000F)      ];
 
            bTarget[c1           ] = 0;
            bTarget[c1+gray_width] = 0;
          }
          else
            j += 4;
        }
      }

      traceOfs = 0;
      traceG  += target.cols;

      gray_min_x =  target.cols;
      gray_max_x = -target.cols;
    }
  }


/********************************************************************/
/*                                                                  */
/*  Generic Sweep Drawing routine                                   */
/*                                                                  */
/********************************************************************/

  Bool  Draw_Sweep()
  {
    Int  y;

    PProfile  P, Q, P_Left, Q_Left, P_Right, Q_Right;

    Int  min_Y, max_Y, top, bottom, dropouts;

    Long x1, x2, xs, e1, e2;

    TProfileList  wait;
    TProfileList  draw_left, draw_right;
    TProfileList  drop_left, drop_right;

    /* Init empty linked lists */

    Init_Linked( &wait );

    Init_Linked( &draw_left  );
    Init_Linked( &draw_right );

    Init_Linked( &drop_left  );
    Init_Linked( &drop_right );

    /* first, compute min and max Y */

    P     = fProfile;
    max_Y = TRUNC( MinY );
    min_Y = TRUNC( MaxY );

    while ( P )
    {
      Q = P->link;

      switch ( P->flow )
      {
      case TT_Flow_Up:
        bottom = P->start;
        top    = P->start + P->height - 1;
        break;

      case TT_Flow_Down:
        bottom = P->start - P->height + 1;
        top    = P->start;

        P->start   = bottom;
        P->offset += P->height - 1;
        break;

      default:
        Raster_Error = Raster_Err_Invalid;
        return FAILURE;
      }

      if ( min_Y > bottom ) min_Y = bottom;
      if ( max_Y < top    ) max_Y = top;

      P->X = 0;
      InsNew( &wait, P );

      P = Q;
    }

    /* Now inits the sweep */

    Proc_Sweep_Init( &min_Y, &max_Y );

    /* Then compute the distance of each profile from min_Y */

    P = wait.head.link;
    Q = P->link;

    while ( P != Q )
    {
      P->countL = P->start - min_Y;
      P = Q;
      Q = P->link;
    }

    /* Let's go */

    for ( y = min_Y; y <= max_Y; y++ )
    {
      /* look in the wait list for new activations */

      P = wait.head.link;
      Q = P->link;

      while ( P != Q )
      {
        if ( P->countL == 0 )
        {
          DelOld( &wait, P );
          switch ( P->flow )
          {
          case TT_Flow_Up:
            InsNew( &draw_left,  P );
            break;
          case TT_Flow_Down:
            InsNew( &draw_right, P );
            break;
          }
        }
        else
          P->countL--;

        P = Q;
        Q = P->link;
      }

      /* Sort the drawing lists */

      Sort( &draw_left );
      Sort( &draw_right );

      /* Let's trace */

      dropouts = 0;

      P_Left  = draw_left .head.link;
      P_Right = draw_right.head.link;

      Q_Left  = P_Left ->link;
      Q_Right = P_Right->link;

      while ( P_Left != Q_Left )
      {
        x1 = P_Left ->X;
        x2 = P_Right->X;

#ifdef REVERSE
        if (x1 > x2)
        {
          xs = x1;
          x1 = x2;
          x2 = xs;
        }
#endif

        if ( x2-x1 <= precision )
        {
          e1 = FLOOR(x1);
          e2 = CEILING(x2);
 
          if ( e1 > e2 || e2 == e1 + precision )
          {
            /* a drop out was detected */
 
            P_Left ->X = x1;
            P_Right->X = x2;
 
            dropouts++;
 
            DelOld( &draw_left,  P_Left  );
            DelOld( &draw_right, P_Right );
 
            InsNew( &drop_left,  P_Left  );
            InsNew( &drop_right, P_Right );
 
            goto Skip_To_Next;
          }
        }
 
        Proc_Sweep_Span( y, x1, x2, P_Left, P_Right );
 
        /* We finalize the profile if needed */
 
        if (P_Left->height <= 0)
          DelOld( &draw_left, P_Left );
 
        if (P_Right->height <= 0)
          DelOld( &draw_right, P_Right );

   Skip_To_Next:
 
        P_Left  = Q_Left;
        P_Right = Q_Right;

        Q_Left  = P_Left ->link;
        Q_Right = P_Right->link;
      }

      P_Left  = drop_left. head.link;
      P_Right = drop_right.head.link;

      while (dropouts > 0)
      {
        Q_Left  = P_Left->link;
        Q_Right = P_Right->link;

        DelOld( &drop_left,  P_Left  );
        DelOld( &drop_right, P_Right );

        Proc_Sweep_Drop( y, P_Left->X, P_Right->X, P_Left, P_Right );

        if (P_Left->height > 0)
          InsNew( &draw_left, P_Left );
 
        if (P_Right->height > 0)
          InsNew( &draw_right, P_Right );

        P_Left  = Q_Left;
        P_Right = Q_Right;

        dropouts--;
      }

      /* Step to next line */

      Proc_Sweep_Step();
    }

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Render_Single_Pass                                          */
/*                                                                          */
/* Description: Performs one sweep with sub-banding.                        */
/*                                                                          */
/* Input:       _XCoord, _YCoord : x and y coordinates arrays               */
/*                                                                          */
/* Returns:     SUCCESS on success                                          */
/*              FAILURE if any error was encountered during render.         */
/*                                                                          */
/****************************************************************************/

  Bool  Render_Single_Pass( TT_F26Dot6*  _xcoord,
                            TT_F26Dot6*  _ycoord )
  {
    Int  i, j, k;

    while ( band_top >= 0 )
    {
      MaxY = band_stack[band_top].y_max * precision;
      MinY = band_stack[band_top].y_min * precision;

      profCur = 0;

      Raster_Error = Raster_Err_None;

      if ( !Convert_Glyph( _xcoord, _ycoord ) )
      {
        if ( Raster_Error != Raster_Err_Overflow ) return FAILURE;

        Raster_Error = Raster_Err_None;

        /* sub-banding */

#ifdef DEBUG_RASTER
        ClearBand( TRUNC( MinY ), TRUNC( MaxY ) );
#endif

        i = band_stack[band_top].y_min;
        j = band_stack[band_top].y_max;

        k = ( i + j )/2;

        if ( band_top >= 7 || k < i )
        {
          band_top     = 0;
          Raster_Error = Raster_Err_Invalid;
          return FAILURE;
        }

        band_stack[band_top+1].y_min = k;
        band_stack[band_top+1].y_max = j;

        band_stack[band_top].y_max = k - 1;

        band_top++;
      }
      else
      {
        if ( fProfile )
          if ( !Draw_Sweep() ) return FAILURE;
        band_top--;
      }
    }

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Render_Glyph                                                */
/*                                                                          */
/* Description: Renders a glyph in a bitmap.      Sub-banding if needed.    */
/*                                                                          */
/* Input:       AGlyph   Glyph record                                       */
/*                                                                          */
/* Returns:     SUCCESS on success.                                         */
/*              FAILURE if any error was encountered during rendering.      */
/*                                                                          */
/****************************************************************************/

  int  Render_Glyph( TGlyphRecord*  glyph,
                     TRasterBlock*  target_map,
                     char           scan )
  {
    if ( !buff )
    {
      Raster_Error = Raster_Err_Not_Ini;
      return FAILURE;
    }

    if ( target_map )
      memcpy( &target, target_map, sizeof ( target ) );

    outs      = (PShort) glyph->outStarts;
    flags     = (PByte)  glyph->flag;
    nPoints   = glyph->points;
    nContours = glyph->outlines;

    dropOutControl = scan;

    /* Vertical Sweep */

    Proc_Sweep_Init = Vertical_Sweep_Init;
    Proc_Sweep_Span = Vertical_Sweep_Span;
    Proc_Sweep_Drop = Vertical_Sweep_Drop;
    Proc_Sweep_Step = Vertical_Sweep_Step;

    band_top            = 0;
    band_stack[0].y_min = 0;
    band_stack[0].y_max = target.rows - 1;

    bWidth  = target.width;
    bTarget = (unsigned char*)target.bitmap;

    if ( !Render_Single_Pass( glyph->xCoord, glyph->yCoord ) )
      return FAILURE;

    /* Horizontal Sweep */

    if ( second_pass )
    {
      Proc_Sweep_Init = Horizontal_Sweep_Init;
      Proc_Sweep_Span = Horizontal_Sweep_Span;
      Proc_Sweep_Drop = Horizontal_Sweep_Drop;
      Proc_Sweep_Step = Horizontal_Sweep_Step;

      band_top            = 0;
      band_stack[0].y_min = 0;
      band_stack[0].y_max = target.width - 1;

      if ( !Render_Single_Pass( glyph->yCoord, glyph->xCoord ) )
        return FAILURE;
    }

    return SUCCESS;
  }


/****************************************************************************/
/*                                                                          */
/* Function:    Render_Gray_Glyph                                           */
/*                                                                          */
/* Description: Renders a glyph with grayscaling. Sub-banding if needed.    */
/*                                                                          */
/* Input:       AGlyph   Glyph record                                       */
/*                                                                          */
/* Returns:     SUCCESS on success                                          */
/*              FAILURE if any error was encountered during rendering.      */
/*                                                                          */
/****************************************************************************/

  int  Render_Gray_Glyph( TGlyphRecord*  glyph,
                          TRasterBlock*  target_map,
                          char           scan,
                          char*          palette )
  {
    Int i;

    if ( !buff )
    {
      Raster_Error = Raster_Err_Not_Ini;
      return FAILURE;
    }

    if ( palette )
    {
      for ( i = 0; i < 5; i++ ) grays[i] = palette[i];
    }

    if ( target_map )
      memcpy( &target, target_map, sizeof ( target ) );

    outs      = (PShort) glyph->outStarts;
    flags     = (PByte)  glyph->flag;
    nPoints   = glyph->points;
    nContours = glyph->outlines;

    dropOutControl = scan;

    /* Vertical Sweep */

    band_top            = 0;
    band_stack[0].y_min = 0;
    band_stack[0].y_max = 2 * target.rows - 1;

    bWidth  = gray_width;
    if ( bWidth > target.cols/4 ) bWidth = target.cols/4;

    bWidth  = bWidth * 8;
    bTarget = (unsigned char*)gray_lines;
    gTarget = (unsigned char*)target.bitmap;

    Proc_Sweep_Init = Gray_Sweep_Init;
    Proc_Sweep_Span = Vertical_Sweep_Span;
    Proc_Sweep_Drop = Vertical_Sweep_Drop;
    Proc_Sweep_Step = Gray_Sweep_Step;

    return Render_Single_Pass( glyph->xCoord, glyph->yCoord );
  }


/************************************************/
/*                                              */
/* InitRasterizer                               */
/*                                              */
/*  Raster Initialization.                      */
/*  Get the bitmap description and render pool  */
/*  addresses.                                  */
/*                                              */
/************************************************/

  int  InitRasterizer( void*  profBuffer,
                       long   profSize,
                       void*  grayBuffer,
                       int    grayLength )
  {
    int  i, l, j, c;

    if ( !profBuffer || profSize < 0 )
      return FAILURE;

    /* XXX We should also check the contents of the passed raster block */

    buff    = (PStorage)profBuffer;
    maxBuff = ( profSize / 4 ) - AlignProfileSize;

    gray_lines = (Byte*)grayBuffer;
    gray_width = grayLength / 2;

    /* Initialization of Count_Table */

    for ( i = 0; i < 256; i++ )
    {
      l = 0;
      j = i;
      for ( c = 0; c < 4; c++ )
      {
        l <<= 4;

        if ( j & 0x80 ) l++;
        if ( j & 0x40 ) l++;

        j = ( j << 2 ) & 0xFF;
      }

      count_table[i] = l;
    }

    dropOutControl = 2;
    Raster_Error   = Raster_Err_None;

#ifndef CONST_PREC
    Set_High_Precision( FALSE );
#endif

    return SUCCESS;
  }


/* End */
