/*******************************************************************
 *
 *  ttindex.c
 *
 *    TrueType index and cmap handling (body).
 *
 *  Copyright 1996, 1997 by
 *  David Turner, Robert Wilhelm, and Werner Lemberg.
 *
 *  This file is part of the FreeType project, and may only be used
 *  modified and distributed under the terms of the FreeType project
 *  license, LICENSE.TXT. By continuing to use, modify or distribute 
 *  this file you indicate that you have read the license and
 *  understand and accept it fully.
 *
 ******************************************************************/

#include "freetype.h"
#include "tttables.h"
#include "ttindex.h"

  Int  code_to_index0( UShort  charCode, PCMap0Table  cmap0 );
  Int  code_to_index2( UShort  charCode, PCMap2Table  cmap2 );
  Int  code_to_index4( UShort  charCode, PCMap4Table  cmap4 );
  Int  code_to_index6( UShort  charCode, PCMap6Table  cmap6 );


/*******************************************************************
 *
 *  Function    : TT_select_CMap
 *
 *  Description : Select the CMap to use.
 *
 *  Input  :  platformID            the target platform
 *            platformEncodingID    the platform specific target
 *                                  encoding
 *            instance             the target glyph instance
 *
 *  Output :  NULL if map not found, TT_PCMapTable if map available.
 *
 ******************************************************************/

  TT_PCMapTable  TT_select_CMap( UShort            platformID,
                                 UShort            platformEncodingID,
                                 PInstance_Record  instance )
  {
    Int         i, num_CME;
    PCMapTable  pcmt;

    num_CME = instance->fontRes->numCMaps;
    pcmt    = instance->fontRes->cMaps;

    for ( i = 0; i < num_CME; i++, pcmt++ )
    {
      if ( ( pcmt->platformID         == platformID ) &&
           ( pcmt->platformEncodingID == platformEncodingID ) )

        return (TT_PCMapTable)pcmt;
    }

    return NULL;
  }


/*******************************************************************
 *
 *  Function    : TT_code_to_index
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses the CMap index returned from TT_select_CMap().
 *
 *  Input  :  charCode      the wanted character code
 *            cmapIndex     the selected cmap table
 *            instance     the target glyph instance
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist (`missing character glyph').
 *            -1 on failure.
 *
 ******************************************************************/

  Int  TT_code_to_index( UShort            charCode,
                         TT_PCMapTable     cmapIndex,
                         PInstance_Record  instance )
  {
    switch ( ((PCMapTable)cmapIndex)->Format )
    {
    case 0:
      return code_to_index0( charCode,
                             ((PCMapTable)cmapIndex)->uc.cmap0 );
    case 2:
      return code_to_index2( charCode,
                             ((PCMapTable)cmapIndex)->uc.cmap2 );
    case 4:
      return code_to_index4( charCode,
                             ((PCMapTable)cmapIndex)->uc.cmap4 );
    case 6:
      return code_to_index6( charCode,
                             ((PCMapTable)cmapIndex)->uc.cmap6 );
    default:
      return -1;
    }
  }


/*******************************************************************
 *
 *  Function    : code_to_index0
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 0.
 *                charCode will be masked to get a value in the range
 *                0x00-0xFF.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap0         a pointer to a cmap table in format 0
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************/

  Int  code_to_index0( UShort       charCode,
                       PCMap0Table  cmap0 )
  {
    return cmap0->glyphIdArray[charCode & 0xFF];
  }


/*******************************************************************
 *
 *  Function    : code_to_index2
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 2.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap2         a pointer to a cmap table in format 2
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************/

  Int  code_to_index2( UShort       charCode,
                       PCMap2Table  cmap2 )
  {
    UShort           index1;
    TCMap2SubHeader  sh2;

    index1 = cmap2->Header.subHeaderKeys[charCode <= 0xFF ?
                                         charCode : (charCode >> 8)];

    if ( index1 == 0 )
    {
      if ( charCode <= 0xFF )
        return cmap2->glyphIdArray[charCode];   /* 8bit character code */
      else
        return 0;
    }
    else                                        /* 16bit character code */
    {
      if ( charCode <= 0xFF )
        return 0;

      sh2 = cmap2->subHeaders[index1];

      if ( (charCode & 0xFF) < sh2.firstCode )
        return 0;

      if ( (charCode & 0xFF) >= (sh2.firstCode + sh2.entryCount) )
        return 0;

      return ( cmap2->glyphIdArray[sh2.idRangeOffset / 2 + (charCode & 0xFF) -
                                   sh2.firstCode] + sh2.idDelta ) & 0xFFFF;
    }
  }


/*******************************************************************
 *
 *  Function    : code_to_index4
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 4.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap4         a pointer to a cmap table in format 4
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist.
 *
 ******************************************************************/

  Int  code_to_index4( UShort       charCode,
                       PCMap4Table  cmap4 )
  {
    UShort         index1, segCount;
    Int            i;
    TCMap4Segment  seg4;

    segCount = cmap4->Header.segCountX2 / 2;

    for ( i = 0; i < segCount; i++ )
      if ( charCode <= cmap4->Segments[i].endCount )
        break;

    seg4 = cmap4->Segments[i];

    if ( charCode < seg4.startCount )
      return 0;

    if ( seg4.idRangeOffset == 0 )
      return ( charCode + seg4.idDelta ) & 0xFFFF;
    else
    {
      index1 = seg4.idRangeOffset / 2 + (charCode - seg4.startCount) -
               (segCount - i);

      if ( cmap4->glyphIdArray[index1] == 0 )
        return 0;
      else
        return ( cmap4->glyphIdArray[index1] + seg4.idDelta ) & 0xFFFF;
    }
  }


/*******************************************************************
 *
 *  Function    : code_to_index6
 *
 *  Description : Convert the character code into a glyph index.
 *                Uses format 6.
 *
 *  Input  :  charCode      the wanted character code
 *            cmap6         a pointer to a cmap table in format 6
 *
 *  Output :  Glyph index into the glyphs array.
 *            0 if the glyph does not exist (`missing character glyph')
 *
 ******************************************************************/

  Int  code_to_index6( UShort       charCode,
                       PCMap6Table  cmap6 )
  {
    UShort firstCode;

    firstCode = cmap6->Header.firstCode;

    if ( charCode < firstCode )
      return 0;

    if ( charCode >= (firstCode + cmap6->Header.entryCount) )
      return 0;

    return cmap6->glyphIdArray[charCode - firstCode];
  }


/* End */
